!> @file
!!  Wrapper around XC library routines (both ABINIT and LibXC).
!! @author
!!    Copyright (C) 2008-2010 ABINIT group (MOliveira)
!!    Copyright (C) 2008-2013 BigDFT group (DC)
!!    This file is distributed under the terms of the
!!    GNU General Public License, see ~/COPYING file
!!    or http://www.gnu.org/copyleft/gpl.txt .
!!    For the list of contributors, see ~/AUTHORS 


!> Module handling the eXchange-Correlation functionals
!! using ABINIT (old) and libxc library

module module_xc

  use iso_c_binding

  use module_base
  use yaml_output
  use dictionaries, only: f_err_raise

  use abi_libxc_functionals
  use abi_interfaces_xc_lowlevel, only: abi_drivexc,abi_size_dvxc

  implicit none

  integer, public, parameter :: XC_ABINIT = 1  !< xc functionals from ABINIT
  integer, public, parameter :: XC_LIBXC  = 2  !< xc from libxc
  integer, public, parameter :: XC_MIXED  = 3  !> xc mixing the origin of the xc functionals

  integer, public, parameter :: XC_HARTREE = 0        !< IXC code for Hartree
  integer, public, parameter :: XC_HARTREE_FOCK = 100 !< IXC code for Hartree-Fock
  integer, public, parameter :: XC_NO_HARTREE = 1000  !< IXC code for no Hartree and no functional (ex. hydrogen atom)

  integer,private,save :: XC_FAMILY_UNKNOWN_       = -1
  integer,private,save :: XC_FAMILY_LDA_           =  1
  integer,private,save :: XC_FAMILY_GGA_           =  2
  integer,private,save :: XC_FAMILY_HYB_GGA_       = 32
  integer,private,save :: XC_EXCHANGE_             =  0
  integer,private,save :: XC_CORRELATION_          =  1
  integer,private,save :: XC_EXCHANGE_CORRELATION_ =  2

  integer, parameter :: XC_HYB_GGA_XC_PBEH_        = 0
  integer, parameter :: XC_HYB_GGA_XC_B3LYP_       = 0


  !> Structure containing the information about the xc functionals
  type xc_info
     integer :: ixc        !< input XC code
     integer :: kind       !< ABINIT or LibXC
     integer :: family(2)  !< LDA, GGA, etc.
     integer :: id(2)      !< Identifier

     type(abi_libxc_functional_type) :: funcs(2)
  end type xc_info

  !type(xc_info) :: xc      !< Global structure about the used xc functionals

  logical :: abinit_init = .false.                                   !< .True. if already ABINIT_XC_NAMES intialized
  !logical :: libxc_init = .false.                                   !< .True. if the libXC library has been initialized
  integer, parameter :: ABINIT_N_NAMES = 29                          !< Number of names for ABINIT_XC_NAMES
  character(len=500), dimension(0:ABINIT_N_NAMES) :: ABINIT_XC_NAMES !< Names of the xc functionals used by ABINIT

  private

  !> Public routines
  public :: xc_info, &
       &    xc_init, &
       &    xc_dump, &
       &    xc_init_rho, &
       &    xc_clean_rho, &
       &    xc_getvxc, &
       &    xc_isgga, &
       &    xc_exctXfac, &
       &    xc_end, &
       &    xc_get_name

contains

  subroutine obj_init_(xcObj, ixc, kind, nspden)
    implicit none

    !Arguments
    !scalars
    type(xc_info), intent(out) :: xcObj
    integer, intent(in) :: nspden
    integer, intent(in) :: kind
    integer, intent(in) :: ixc

    !Local variables
    !scalars
    integer :: i, ierr

    ! **

    xcObj%ixc  = ixc
    xcObj%kind = kind
    xcObj%family = 0
    xcObj%id = 0

    if (xcObj%kind == XC_LIBXC .or. xcObj%kind == XC_MIXED) then
       ! LibXC case.

       call init_xc_constants_()
       call abi_libxc_functionals_init(xcObj%ixc,nspden,xc_functionals=xcObj%funcs)

       xcObj%id(1)=xcObj%funcs(1)%id
       xcObj%id(2)=xcObj%funcs(2)%id
       xcObj%family(1) = xcObj%funcs(1)%family
       xcObj%family(2) = xcObj%funcs(2)%family
       if (any(xcObj%family(:)/=XC_FAMILY_LDA_.and.xcObj%family(:)/=XC_FAMILY_GGA_.and. &
&              xcObj%family(:)/=XC_FAMILY_HYB_GGA_)) then
          write(*,*) "Error: unsupported functional, change ixc."
          call MPI_ABORT(bigdft_mpi%mpi_comm,0,ierr)
       end if

    else if (xcObj%kind == XC_ABINIT) then
       ! ABINIT case

       xcObj%id(1) = abs(ixc)
       xcObj%id(2) = 0

       ! Get XC functional family
       if ((xcObj%id(1) >= 1 .and. xcObj%id(1) < 11) .or. xcObj%id(1) == 24) then
          xcObj%family(1) = XC_FAMILY_LDA_
       else if (xcObj%id(1) >= 11 .and. xcObj%id(1) < 18) then
          xcObj%family(1) = XC_FAMILY_GGA_
       else if (xcObj%id(1) >= 23 .and. xcObj%id(1) < 28) then
          xcObj%family(1) = XC_FAMILY_GGA_
       else if (xcObj%id(1) >= 31 .and. xcObj%id(1) < 35) then
          xcObj%family(1) = XC_FAMILY_LDA_
       else if (xcObj%id(1) == XC_HARTREE .or. xcObj%id(1) == XC_HARTREE_FOCK .or. xcObj%id(1) == XC_NO_HARTREE) then
          xcObj%family(1) = 0
       else
          write(*,*) "Error: unsupported functional, change ixc."
          call MPI_ABORT(bigdft_mpi%mpi_comm,0,ierr)
       end if
    end if
  end subroutine obj_init_

  subroutine obj_free_(xcObj)
    implicit none

    type(xc_info), intent(inout) :: xcObj

    if (xcObj%kind == XC_LIBXC .or. xcObj%kind == XC_MIXED) then
       call abi_libxc_functionals_end(xc_functionals=xcObj%funcs)
    end if

  end subroutine obj_free_


  !> Give the name of the XC functional
  subroutine obj_get_name_(xcObj, name)
    implicit none
    !Arguments
    type(xc_info), intent(in) :: xcObj      !< XC objects
    character(len=500), intent(out) :: name !< XC functional name
    !Local variables
    integer :: i
    character(len = 500) :: messX, messC, messXC

    if (xcObj%kind == XC_LIBXC .or. xcObj%kind == XC_MIXED) then
       write(messX, "(A)") ""
       write(messC, "(A)") ""
       write(messXC, "(A)") ""
       do i = 1, 2
          if (xcObj%family(i) == 0) cycle
          if (xcObj%funcs(i)%kind==XC_EXCHANGE_) then
             messX=abi_libxc_functionals_fullname(xcObj%funcs(i:i))
          else if (xcObj%funcs(i)%kind==XC_CORRELATION_) then
             messC=abi_libxc_functionals_fullname(xcObj%funcs(i:i))
          else if (xcObj%funcs(i)%kind==XC_EXCHANGE_CORRELATION_) then
             messXC=abi_libxc_functionals_fullname(xcObj%funcs(i:i))
          end if
       end do
       if (len(trim(messXC)) > 0) then
          write(name, "(2A)")    "XC: ", trim(messXC)
       else
          if (len(trim(messC)) == 0) then
             write(name, "(2A)") "X-: ", trim(messX)
          else if (len(trim(messX)) == 0) then
             write(name, "(2A)") "-C: ", trim(messC)
          else
             if (trim(messX) == trim(messC)) then
                write(name, "(2A)") "XC: ", trim(messX)
             else
                write(name, "(4A)") "XC: ", trim(messX), ", ", trim(messC)
             end if
          end if
       end if
    else if (xcObj%kind == XC_ABINIT) then
       call obj_init_abinit_xc_names_()
       if (xcObj%id(1) == 1000) then
          !No Hartree and No functional
          write(name,"(A)") ABINIT_XC_NAMES(ABINIT_N_NAMES-1)
       else
          write(name, "(A)") ABINIT_XC_NAMES(min(xcObj%id(1),ABINIT_N_NAMES))
       end if
    end if
  end subroutine obj_get_name_


  !>  Dump XC info on screen.
  subroutine obj_dump_(xcObj)
    implicit none
    type(xc_info), intent(in) :: xcObj

    integer :: i, ii
    integer(C_INT) :: iref_c
    character(len=500) :: message
    character(kind=C_CHAR,len=1),pointer :: strg_c
    character(len = 500) :: xc_refs(10)

    ! Dump functional information
    call obj_get_name_(xcObj, message)
    !write(*,"(1x,a19,a65)") "Exchange-corr. ref.", "("//trim(message)//")"
    call yaml_map('Exchange-Correlation reference','"'//trim(message)//'"')
    if (xcObj%kind == XC_ABINIT) then
       call yaml_map('XC functional implementation','ABINIT')
       !write(*,"(1x,A84)") "XC functional provided by ABINIT."
    else
       call yaml_map('XC functional implementation','libXC')
       call yaml_sequence_open('Reference Papers')
       call yaml_sequence('"Comput. Phys. Commun. 183, 2272 (2012)"')
       do i = 1, 2
          if (xcObj%family(i) <= 0) cycle

          call abi_libxc_functionals_getrefs(xc_refs,xcObj%funcs(i))
          ii=1
          do while (trim(xc_refs(ii))/=''.and.ii<=10)
            message=xc_refs(ii)
            call yaml_sequence('"'//trim(message)//'"')
          end do

       end do
       call yaml_sequence_close()
    end if
  end subroutine obj_dump_

  !>  Get a name, corresponding to the given XC id.
  subroutine xc_get_name(name,ixc,kind)
    implicit none

    character(len=500), intent(out) :: name
    integer, intent(in) :: kind
    integer, intent(in) :: ixc

    type(xc_info) :: xcObj

    call obj_init_(xcObj, ixc, kind, 1)
    call obj_get_name_(xcObj, name)
    call obj_free_(xcObj)
  end subroutine xc_get_name

  !>  Dump XC info on screen.
  subroutine xc_dump(ixc,kind,nspden)
    implicit none
    integer, intent(in) :: kind
    integer, intent(in) :: ixc
    integer, intent(in) :: nspden

    type(xc_info) :: xcObj

    call obj_init_(xcObj, ixc, kind, nspden)
    call obj_dump_(xcObj)
    call obj_free_(xcObj)
  end subroutine xc_dump

  !>  Initialize the desired XC functional, from LibXC.
  subroutine xc_init(xcObj,ixc,kind,nspden)
    implicit none

    !Arguments
    !scalars
    type(xc_info), intent(out) :: xcObj
    integer, intent(in) :: nspden
    integer, intent(in) :: kind
    integer, intent(in) :: ixc
    !local variables
!!$    integer :: ixc_prev

    !check if we are trying to initialize the libXC to a different functional
!!$    if (libxc_init) then
!!$       ixc_prev=xc%id(1)+1000*xc%id(2)
!!$       if (f_err_raise(((xc%kind== XC_LIBXC .or. kind == XC_MIXED) .and. ixc/=-ixc_prev),&
!!$            'LibXC library has been already initialized with ixc='//trim(yaml_toa(ixc_prev))//&
!!$            ', finalize it first to reinitialize it with ixc='//trim(yaml_toa(ixc)),&
!!$            err_name='BIGDFT_RUNTIME_ERROR')) return
!!$    end if
!!$    libxc_init=.true.
    call obj_init_(xcObj, ixc, kind, nspden)
  end subroutine xc_init

  !> End usage of LibXC functional. Call LibXC end function,
  !! and deallocate module contents.
  subroutine xc_end(xcObj)
    implicit none
    type(xc_info), intent(inout) :: xcObj

    !here no exception is needed if finalization has already been done
    !if (libxc_init) then
    call obj_free_(xcObj)
    !   libxc_init=.false.
    !end if

  end subroutine xc_end

  !> Test function to identify whether the presently used functional
  !! is a GGA or not
  function xc_isgga(xcObj)
    implicit none
    type(xc_info), intent(in) :: xcObj

    logical :: xc_isgga

    xc_isgga = .false.
    if (xcObj%kind== XC_LIBXC .or. xcObj%kind == XC_MIXED) then
      xc_isgga=abi_libxc_functionals_isgga(xcObj%funcs)
    else if (any(xcObj%family == XC_FAMILY_GGA_) .or. &
           & any(xcObj%family == XC_FAMILY_HYB_GGA_)) then
      xc_isgga = .true.
    end if
  end function xc_isgga

  !> Calculate the exchange-correlation factor (percentage) to add in the functional
  real(kind=8) function xc_exctXfac(xcObj)
    implicit none
    type(xc_info), intent(in) :: xcObj

    xc_exctXfac = 0.d0
    if (any(xcObj%family == XC_FAMILY_HYB_GGA_)) then
       !factors for the exact exchange contribution of different hybrid functionals
      if (xcObj%kind== XC_LIBXC .or. xcObj%kind == XC_MIXED) then
        if (any(xcObj%id == abi_libxc_functionals_getid('XC_HYB_GGA_XC_PBEH'))) then
          xc_exctXfac = 0.25d0 !PBE0
        else if (any(xcObj%id == abi_libxc_functionals_getid('XC_HYB_GGA_XC_B3LYP'))) then
          xc_exctXfac = 0.2d0  !B3LYP
        end if
      else
        if (any(xcObj%id == XC_HYB_GGA_XC_PBEH_)) then
          xc_exctXfac = 0.25d0 !PBE0
        else if (any(xcObj%id == XC_HYB_GGA_XC_B3LYP_)) then
          xc_exctXfac = 0.2d0  !B3LYP
        end if
      end if
    end if

    !Hartree-Fock value
    if (xcObj%id(1) == XC_HARTREE_FOCK .and. xcObj%id(2) == 0) then
       xc_exctXfac = 1.d0 
    end if

  end function xc_exctXfac

  subroutine xc_init_rho(xcObj, n, rho, nproc)
    implicit none
    ! Arguments
    type(xc_info), intent(in) :: xcObj
    integer :: n,nproc
    real(dp) :: rho(n)

    if (xcObj%kind == XC_ABINIT) then
       call tenminustwenty(n,rho,nproc)
    else
       call to_zero(n,rho)
    end if
  end subroutine xc_init_rho

  subroutine xc_clean_rho(xcObj, n, rho, nproc)
    implicit none
    ! Arguments
    type(xc_info), intent(in) :: xcObj
    integer :: n,nproc
    real(dp) :: rho(n)

    integer :: i

    if (xcObj%kind == XC_ABINIT) then
       do i = 1, n, 1
          if (rho(i) < 1d-20) then
             rho(i) = 1d-20 / real(nproc,kind=dp)
          end if
       end do
    else
       do i = 1, n, 1
          if (rho(i) < 0.) then
             rho(i) = 0.0_dp
          end if
       end do
    end if
  end subroutine xc_clean_rho

  !> Return XC potential and energy, from input density (even gradient etc...)
  subroutine xc_getvxc(xcObj, npts,exc,nspden,rho,vxc,grho2,vxcgr,dvxci)
    implicit none

    !Arguments ------------------------------------
    type(xc_info), intent(inout) :: xcObj
    integer, intent(in) :: npts,nspden
    real(dp),intent(out), dimension(npts) :: exc
    real(dp),intent(in), dimension(npts,nspden)  :: rho
    real(dp),intent(out), dimension(npts,nspden) :: vxc
    real(dp),intent(in) :: grho2(*)
    real(dp),intent(out) :: vxcgr(*)
    real(dp),intent(out), optional :: dvxci(npts,nspden + 1)

    !Local variables-------------------------------
    integer :: i, j, ipts, ixc, ndvxc, ngr2, nd2vxc, nvxcdgr, order

    if (xcObj%kind == XC_ABINIT) then
       ! ABINIT case, call drivexc
       ixc = xcObj%id(1)
       !Allocations of the exchange-correlation terms, depending on the ixc value
       order = 1
       if (present(dvxci) .and. nspden == 1) order = -2
       if (present(dvxci) .and. nspden == 2) order = +2

       call abi_size_dvxc(ixc,ndvxc,ngr2,nd2vxc,nspden,nvxcdgr,order)

       !let us apply ABINIT routines
       if (xcObj%family(1)==XC_FAMILY_LDA_) then
          if (order**2 <=1 .or. ixc >= 31 .and. ixc <= 34) then
             call abi_drivexc(exc,ixc,npts,nspden,order,rho,vxc,&
                  ndvxc,ngr2,nd2vxc,nvxcdgr)
          else
             call abi_drivexc(exc,ixc,npts,nspden,order,rho,vxc,&
                  ndvxc,ngr2,nd2vxc,nvxcdgr,&
                  dvxc=dvxci)
          end if
       else if (xcObj%family(1)==XC_FAMILY_GGA_) then
          !case with gradient, no big order
          if (ixc /= 13) then
             call abi_drivexc(exc,ixc,npts,nspden,order,rho,&
                  vxc,ndvxc,ngr2,nd2vxc,nvxcdgr,&
                  grho2_updn=grho2,vxcgr=vxcgr) 
          else
             call abi_drivexc(exc,ixc,npts,nspden,order,rho,&
                  vxc,ndvxc,ngr2,nd2vxc,nvxcdgr,&
                  grho2_updn=grho2) 
          end if
       end if

    else if (xcObj%kind == XC_MIXED) then
       ! LibXC case with ABINIT rho distribution.

       if (xc_isgga(xcObj)) then
          if (present(dvxci)) then
             call abi_libxc_functionals_getvxc(nspden+1,0,npts,nspden,order,rho,exc,vxc, &
&                 grho2=grho2,vxcgr=vxcgr,dvxc=dvxci,xc_functionals=xcObj%funcs)
          else
             call abi_libxc_functionals_getvxc(0,0,npts,nspden,order,rho,exc,vxc, &
&                 grho2=grho2,vxcgr=vxcgr,xc_functionals=xcObj%funcs)
          end if
       else
          if (present(dvxci)) then
             call abi_libxc_functionals_getvxc(nspden+1,0,npts,nspden,order,rho,exc,vxc, &
&                 dvxc=dvxci,xc_functionals=xcObj%funcs)
          else
             call abi_libxc_functionals_getvxc(0,0,npts,nspden,order,rho,exc,vxc, &
&                 xc_functionals=xcObj%funcs)
          end if
       end if

    else if (xcObj%kind == XC_LIBXC) then
       ! Pure LibXC case.
       ! WARNING: LDA implementation only, first derivative, no fxc

       call abi_libxc_functionals_getvxc(0,0,npts,nspden,order,rho,exc,vxc, &
&                                        xc_functionals=xcObj%funcs)

    else
       write(0,*) "ERROR: XC module not initialised."
    end if

  end subroutine xc_getvxc


  !> Initialize the names of the xc functionals used by ABINIT
  subroutine obj_init_abinit_xc_names_()
    if (abinit_init) return

    write(ABINIT_XC_NAMES( 0), "(A)") "XC: NO Semilocal XC (Hartree only)"
    write(ABINIT_XC_NAMES( 1), "(A)") "XC: Teter 93"
    write(ABINIT_XC_NAMES( 2), "(A)") "XC: Slater exchange, Perdew & Zunger"
    write(ABINIT_XC_NAMES( 3), "(A)") "XC: Teter 91"
    write(ABINIT_XC_NAMES( 4), "(A)") "XC: Slater exchange, Wigner"
    write(ABINIT_XC_NAMES( 5), "(A)") "XC: Slater exchange, Hedin & Lundqvist"
    write(ABINIT_XC_NAMES( 6), "(A)") "-C: Slater's Xalpha"
    write(ABINIT_XC_NAMES( 7), "(A)") "XC: Slater exchange, Perdew & Wang"
    write(ABINIT_XC_NAMES( 8), "(A)") "X-: Slater exchange"
    write(ABINIT_XC_NAMES( 9), "(A)") "XC: Slater exchange, Random Phase Approximation (RPA)"
    write(ABINIT_XC_NAMES(11), "(A)") "XC: Perdew, Burke & Ernzerhof"
    write(ABINIT_XC_NAMES(12), "(A)") "X-: Perdew, Burke & Ernzerhof"
    write(ABINIT_XC_NAMES(13), "(A)") "XC: van Leeuwen & Baerends"
    write(ABINIT_XC_NAMES(14), "(A)") "XC: Revised PBE from Zhang & Yang, Perdew, Burke & Ernzerhof"
    write(ABINIT_XC_NAMES(15), "(A)") "XC: Hammer, Hansen, and Nørskov, Perdew, Burke & Ernzerhof"
    write(ABINIT_XC_NAMES(16), "(A)") "XC: HCTH/93"
    write(ABINIT_XC_NAMES(17), "(A)") "XC: HCTH/120"
    write(ABINIT_XC_NAMES(23), "(A)") "X-: Wu & Cohen"
    write(ABINIT_XC_NAMES(26), "(A)") "XC: HCTH/147"
    write(ABINIT_XC_NAMES(27), "(A)") "XC: HCTH/407"
    write(ABINIT_XC_NAMES(28), "(A)") "No Hartree and XC terms"
    !Hartree-Fock (always the last - ixc=100)
    write(ABINIT_XC_NAMES(29), "(A)") "Hartree-Fock Exchange only"

    abinit_init = .true.
  end subroutine obj_init_abinit_xc_names_

  !> Initialize some libXC constants
  subroutine init_xc_constants_()

    call abi_libxc_functionals_constants_load()

    XC_FAMILY_UNKNOWN_       = ABI_XC_FAMILY_UNKNOWN
    XC_FAMILY_LDA_           = ABI_XC_FAMILY_LDA
    XC_FAMILY_GGA_           = ABI_XC_FAMILY_GGA
    XC_FAMILY_HYB_GGA_       = ABI_XC_FAMILY_HYB_GGA
    XC_EXCHANGE_             = ABI_XC_EXCHANGE
    XC_CORRELATION_          = ABI_XC_CORRELATION
    XC_EXCHANGE_CORRELATION_ = ABI_XC_EXCHANGE_CORRELATION

  end subroutine init_xc_constants_

end module module_xc
