!> @file
!!  Modules to handle input variables
!! @author
!!    Copyright (C) 2010-2013 BigDFT group
!!    This file is distributed under the terms of the
!!    GNU General Public License, see ~/COPYING file
!!    or http://www.gnu.org/copyleft/gpl.txt .
!!    For the list of contributors, see ~/AUTHORS 
!! @todo
!! Add a mechanism to give a warning for unused keywords (TD)
!!   - add a dictionary of keywords used in the file (should be accelerated process_line)
!!   - fo a new list with all keywords
!!   - for each call to input_var remove keywords inthe previous line
!!   - Check if all keywords were used.


!> Module which defines all basic operations to handle input variables
module module_input

   use module_base
   use yaml_strings, only: read_fraction_string
   implicit none
   private

   !> Error ids for this module.
   integer, parameter, public :: INPUT_VAR_NOT_IN_LIST      = 1
   integer, parameter, public :: INPUT_VAR_NOT_IN_RANGE     = 2
   integer, parameter, public :: INPUT_VAR_CONVERSION_ERROR = 3
   integer, parameter, public :: INPUT_VAR_ARG_ERROR        = 99

   !> Max line of a file
   integer, parameter :: nmax_lines=500
   !> Max length of a line
   integer, parameter :: max_length=92
   character(len=max_length) :: input_file,input_radical,input_type,line_being_processed
   logical :: output,lmpinit
   integer :: iline_parsed,iline_written,iargument,ipos,nlines_total
   integer :: stdout=6
   integer, dimension(:), allocatable :: parsed_lines
   character(len=max_length), dimension(:), allocatable :: inout_lines


   interface input_var
      module procedure var_character, var_logical, var_integer, &
         &   var_integer_array, var_double, var_keyword, var_ids,&
         & var_double_compulsory,var_real_compulsory,var_int_compulsory,var_logical_compulsory,&
         &   var_char_compulsory
   end interface

   public :: input_set_file,input_set_stdout
   public :: input_var
   public :: input_free


contains


   subroutine input_set_stdout(unit)
     implicit none
     integer, intent(in) :: unit
     
     stdout=unit
   end subroutine input_set_stdout


   subroutine input_set_file(iproc, dump, filename, exists,comment_file_usage)
      integer, intent(in) :: iproc
      logical, intent(in) :: dump
      character(len = *), intent(in) :: filename,comment_file_usage
      logical, intent(out) :: exists

      character(len=max_length), dimension(nmax_lines) :: lines
      integer :: i,nlines,ierror,ierr=0 !for MPIfake BCAST

      !no line parsed if the file not exists
      iline_parsed=0
      !no line has been written in the output
      iline_written=0
      !no argument has been read yet
      iargument=1
      !the present line is empty
      line_being_processed=repeat(' ',max_length)
      !the starting position for the lines is zero
      ipos=0
      !there are no lines for the moment
      nlines_total=0

      !verify whether MPI has been initialized
      lmpinit=.false.
      call MPI_INITIALIZED(lmpinit,ierr)

      write(input_file, "(A)") trim(filename)
      i = index(input_file, ".", back = .true.)
      write(input_radical, "(A)") input_file(1:i - 1)
      write(input_type, "(A)") trim(input_file(i + 1:))

      !check if the file is present, only the proc 0 to avoid race conditions
      !when the inquired file is indeed generated by the proc 0 on free.
      exists = .false.
      if (iproc == 0) inquire(file=trim(input_file),exist=exists)
      if (lmpinit) call MPI_BCAST(exists,1,MPI_LOGICAL,0,bigdft_mpi%mpi_comm,ierr)
      if (ierr /=0) stop 'input_file BCAST (0) '
      if (exists) then
         !only the root processor parse the file
         nlines = 0
         if (iproc==0) then
            open(unit = 1, file = trim(filename), status = 'old')
            i=0
            parse_file: do 
               i=i+1
               lines(i)=repeat(' ',max_length) !initialize lines
               read(1, fmt = '(a)', iostat = ierror) lines(i)
               !eliminate leading blanks from the line
               !print *,'here',i,lines(i),ierror,trim(lines(i)),'len',len(trim(lines(i)))
               lines(i)=adjustl(lines(i))
               if (ierror /= 0) exit parse_file
            end do parse_file
            close(1)
            !check if the last line has 
            if(len(trim(lines(i))) > 0) then
               nlines=i
            else
               nlines=i-1
            end if
         end if
         !broadcast the number of lines
         if (lmpinit) call MPI_BCAST(nlines,1,MPI_INTEGER,0,bigdft_mpi%mpi_comm,ierr)
         if (ierr /=0) stop 'input_file BCAST (1) '
         nlines_total=nlines

         !broadcast all the lines
         if (lmpinit) call MPI_BCAST(lines,max_length*nlines,MPI_CHARACTER,0,bigdft_mpi%mpi_comm,ierr)
         if (ierr /=0) stop 'input_file BCAST (2) '

         !!$    write(0,*) "Setup input file '", trim(filename), "' with ", i - 1, "lines."

!!$         allocate(inout_lines(0:nlines)) !the 0-th line is for the description of the file
!!$         do i=1,nlines
!!$            inout_lines(i)=lines(i)
!!$         end do
         !start parsing from the first line
         iline_parsed=1
      else
         !in this case the array constitute the output results
!!$         allocate(inout_lines(0:nmax_lines))
!!$         do i=1,nmax_lines
!!$            inout_lines(i)=repeat(' ',max_length) !initialize lines
!!$         end do
         nlines_total=nmax_lines
         nlines=0
      end if

      !lines which are parsed by the input
      allocate(inout_lines(0:nmax_lines))
      allocate(parsed_lines(nmax_lines))
      do i=1,nlines
         inout_lines(i)=lines(i)
         parsed_lines(i)=i
      end do
      do i=nlines+1,nmax_lines
         inout_lines(i)=repeat(' ',max_length) !initialize lines
         parsed_lines(i)=i
      end do


      !write the first line in the output
      if (exists) then
         write(inout_lines(iline_written),'(1x,5a)')&
            &   '#... (file:', trim(filename),')',&
            &    repeat('.',max_length-2-(len(trim(filename)//trim(comment_file_usage))+11)),&
            &   trim(comment_file_usage)
      else
         write(inout_lines(iline_written),'(1x,5a)')&
            &   '#... (file:',trim(filename),'.. not present)',&
            &     repeat('.',max_length-2-(len(trim(filename)//trim(comment_file_usage))+26)),&
            &   trim(comment_file_usage)
      end if
      iline_written=iline_written+1

      output = (iproc == 0) .and. dump
      !dump the 0-th line on the screen
      !if (iproc == 0 .and. dump) then
      !   write(stdout,'(a)') inout_lines(0)
      !end if
      !!$    output = (iproc == 0)
      !!$    ! Output
      !!$    if (iproc == 0) then
      !!$       write(stdout,*)
      !!$       if (exists) then
      !!$          write(stdout,'(1x,3a)') '--- (file: ', trim(filename), &
      !!$               & ') -----------------------------------------'//&
      !!$               trim(comment_file_usage)
      !!$       else
      !!$          write(stdout,'(1x,a)')&
      !!$               '--- (file:'//trim(filename)//'-- not present) --------------------------'//&
      !!$               trim(comment_file_usage)
      !!$       end if
      !!$    end if
   END SUBROUTINE input_set_file


   subroutine input_free(dump)
      implicit none
      logical, intent(in), optional :: dump
      !Local variables
      integer, parameter :: iunit=11
      logical :: warn
      integer :: ierr,iline,jline

      if (present(dump)) then !case for compulsory variables
         !if (iline_written==1) iline_written=2
         if (dump) then
            !dump the file on the screen
            write(stdout,'(a)') inout_lines(0)
            do iline=1,iline_written-1
               !print *,'end',iline,parsed_lines(iline)
               write(stdout,fmt='(1x,a,a)') '#|',&
                    inout_lines(parsed_lines(iline))(1:max_length-2)
            end do
            if (iline_parsed==0) then !the file does not exist
               !add the writing of the file in the given unit
               open(unit=iunit,file='default.' // trim(input_type), status ='unknown')
               do iline=1,iline_written-1
                  write(iunit,'(a)')inout_lines(iline)
               end do
               close(unit=iunit)
            else if (nlines_total /= nmax_lines) then !case with existing file
               !search for lines which have not been processed
               warn=.false.
               line_done: do iline=1,iline_parsed
                  do jline=1,iline_parsed
                     if (parsed_lines(jline)==iline) cycle line_done
                  end do
                  if (len_trim(inout_lines(iline))==0) cycle line_done
                  if (.not. warn) then
                     write(stdout,*)&
                          '# ==== WARNING: the following lines have not been processed by the parser ===='
                     warn=.true.
                  end if
                  write(stdout,fmt='(1x,a,a)') '#|',inout_lines(iline)(1:max_length-2)
               end do line_done
               !put the rest of the lines
               do iline=iline_parsed,nlines_total
                  if (.not. warn) then
                     write(stdout,*)&
                          '# ==== WARNING: the following lines have not been processed by the parser ===='
                     warn=.true.
                  end if
                  write(stdout,fmt='(1x,a,a)') '#|',inout_lines(iline)(1:max_length-2)
               end do
            end if
         end if
      end if


      if (allocated(inout_lines)) deallocate(inout_lines)
      deallocate(parsed_lines)
      if (lmpinit) call MPI_BARRIER(bigdft_mpi%mpi_comm,ierr)
   END SUBROUTINE input_free


   subroutine leave()
      implicit none
      !local variables
      integer :: ierr
      if (output) then
         write(stdout,'(1x,a,a,2(a,i3))')'Error while reading the file "', &
              &   trim(input_file), '", line=', iline_written,' argument=', iargument
         if (iline_written <= nlines_total) then
            write(stdout,*)inout_lines(iline_written),line_being_processed
         end if
         !to be called only if mpi is initialized
      end if
      if (lmpinit) call MPI_BARRIER(bigdft_mpi%mpi_comm,ierr)
      stop
   END SUBROUTINE leave


   subroutine check(ierror)
      implicit none
      !Arguments
      integer, intent(in) :: ierror

      if (ierror/=0) then
         call leave()
      end if
      !increment the argument at each passed check
      iargument=iargument+1

   END SUBROUTINE check


   !> Process the line needed with the default value in mind
   subroutine process_line(default,line_comment)
      implicit none
      character(len=*), intent(in), optional :: default,line_comment
      !Local variables
      integer :: i,iblank,istart,nchars

      if (iargument==1) then
         ipos=0
         line_being_processed=repeat(' ',max_length)
      end if

      if (present(default) .and. iline_parsed==0) then
         !case without file, write default and continue the line
         do i=1,len(default)
            inout_lines(iline_written)(i+ipos:i+ipos)=default(i:i)
         end do
         ipos=ipos+len(default)+1
         inout_lines(iline_written)(ipos:ipos)=' '
      end if
      if (present(line_comment) .and. iline_parsed==0) then
         !case without file, close the line. Start the comment at column 16 if possible
         istart=max(ipos+1,16)
         nchars=min(len(line_comment),max_length-istart)
         inout_lines(iline_written)(istart:istart+nchars)=line_comment(1:nchars)
         iline_written=iline_written+1
         iargument=0 !the argument should become 0 for the default case and one for the other one
      else if (.not. present(default) .and. .not. present(line_comment).and. iline_parsed/=0) then
         !traditional case, the argument should be parsed one after another
         !start with the entire line
         if (iargument==1) then 
            !print *,'prsed',iline_parsed,inout_lines(iline_parsed)
            line_being_processed=inout_lines(iline_parsed)
         else
            !search in the line the first blank
            iblank=scan(line_being_processed,' ')
            do i=1,max_length-iblank
               line_being_processed(i:i)=line_being_processed(i+iblank:i+iblank)
            end do
            do i=max_length-iblank+1,max_length
               line_being_processed(i:i)=' '
            end do
            ipos=ipos+iblank
         end if
         !adjust the line to eliminate further blanks
         line_being_processed=adjustl(line_being_processed)
      else if (.not. present(default) .and. present(line_comment) .and. iline_parsed/=0) then
         !traditional case, close the line and skip to the next one
         !case without file, close the line. Write also the predefined comment at the end
         !Start the comment at column 16 if possible
         iblank=scan(line_being_processed,' ')
         ipos=ipos+iblank
         !+2 in order to add a space before the comment 
         istart=max(ipos+2,16)
         do i=ipos+1,15
            inout_lines(iline_written)(i:i)=' '
         end do
         nchars=min(len(line_comment),max_length-istart)
         inout_lines(iline_written)(istart:istart+nchars)=line_comment(1:nchars)
         !clean the rest of the line
         do i=istart+nchars+1,max_length
            inout_lines(iline_written)(i:i)=' '
         end do
         iargument=1
         iline_parsed=iline_parsed+1
         iline_written=iline_written+1
      end if

   END SUBROUTINE process_line


   subroutine find(name, iline, ii)
     implicit none
     character(len = *), intent(in) :: name
     integer, intent(out) :: iline, ii
     !local variables
     logical :: line_found
      integer :: k
      line_found=.false.
      if (iline_parsed /= 0) then
         iline_parsed=iline_parsed+1
         search_line: do iline = 1, size(inout_lines)-1 !there is also the zero now
            k = 1
            do ii = 1, len(inout_lines(iline)), 1
               if (ichar(inout_lines(iline)(ii:ii)) == ichar(name(k:k)) .or. &
                  &   ichar(inout_lines(iline)(ii:ii)) == ichar(name(k:k)) + 32 .or. &
                  & ichar(inout_lines(iline)(ii:ii)) == ichar(name(k:k)) - 32) then
                  k = k + 1
               else
                  k = 1
                  cycle search_line
               end if
               if (k == len(name) + 1) then !the name has been found
                  line_found=.true.
                  exit search_line   !return
               end if
            end do
         end do search_line
      else
         iline_parsed=2
      end if
      if (.not. line_found) iline = 0
      !swap the lines found
      !print *,'iline',iline,iline_parsed,nlines_total+iline_parsed-1,line_found,name
      if (nlines_total /= nmax_lines ) then !the file exists
         if (line_found) then
            parsed_lines(iline_parsed-1)=iline
            !parsed_lines(iline)=iline_parsed-1
         else
            parsed_lines(iline_parsed-1)=nlines_total+iline_parsed-1
         end if
      end if

   END SUBROUTINE find


   !> Compare two strings (case-insensitive). Blanks are relevant!
   function case_insensitive_equiv(stra,strb)
      implicit none
      character(len=*), intent(in) :: stra,strb
      logical :: case_insensitive_equiv
      !Local variables
      integer :: i,ica,icb,ila,ilb,ilength
      ila=len(stra)
      ilb=len(strb)
      ilength=min(ila,ilb)
      ica=ichar(stra(1:1))
      icb=ichar(strb(1:1))
      case_insensitive_equiv=(modulo(ica-icb,32) == 0) .and. (ila==ilb)
      do i=2,ilength
         ica=ichar(stra(i:i))
         icb=ichar(strb(i:i))
         case_insensitive_equiv=case_insensitive_equiv .and. &
            &   (modulo(ica-icb,32) == 0)
         if (.not. case_insensitive_equiv) exit
      end do

   END FUNCTION case_insensitive_equiv


   !> Routines for compulsory file
   subroutine var_double_compulsory(var,default,dict,ranges,exclusive,comment,input_iostat)
      implicit none
      character(len=*), intent(in) :: default
      real(kind=8), intent(out) :: var
      type(dictionary), pointer, optional :: dict
      character(len=*), intent(in), optional :: comment
      integer, intent(out), optional :: input_iostat
      real(kind=8), dimension(2), intent(in), optional :: ranges
      real(kind=8), dimension(:), intent(in), optional :: exclusive
      !Local variables
      logical :: found
      integer :: ierror,ilist

      if (present(input_iostat)) then
         !first, check if the line is correct
         if (iline_written>nlines_total ) then
            input_iostat=-1
            return
         else
            input_iostat=0 !no error for the moment
         end if
      else
         if (iline_written>nlines_total) then
            call leave()
         end if
      end if

      !if the file has not been opened, use the default variable 
      !then write in the output lines the default
      if (iline_parsed==0) then
         !finalize the line if the comment is present
         if (present(comment)) then
            call process_line(default=default,line_comment=comment)
         else
            call process_line(default=default)
         end if
         call read_fraction_string(default,var,ierror)
         if (present(input_iostat) .and. ierror /= 0) then
            input_iostat=-2
            return
         else
            call check(ierror)
         end if
         !otherwise read the corresponding argument and check its validity
      else
         !read the argument
         call process_line()
         !print *,line_being_processed
         call read_fraction_string(line_being_processed,var,ierror)
         if (present(input_iostat) .and. ierror /= 0) then
            input_iostat=-2
            return
         else
            call check(ierror)
         end if

         !check the validity of the variable
         if (present(ranges)) then
            if (var < ranges(1) .or. var > ranges(2)) then
               if (output) then
                  write(stdout,'(1x,a,i0,a,i0)') &
                     &   'ERROR in parsing file '//trim(input_file)//', line=', iline_written,' argument=', iargument-1
                  write(stdout,*)'      values should be in range: [',ranges(1),'-',ranges(2),']'
               end if
               if (present(input_iostat)) then
                  input_iostat=1
                  return
               else
                  call leave()
               end if
            end if
         else if (present(exclusive)) then
            found=.false.
            found_loop: do ilist=1,size(exclusive)
               if (var == exclusive(ilist)) then 
                  found=.true.
                  exit found_loop
               end if
            end do found_loop
            if (.not. found) then
               if (output) then
                  write(stdout,'(1x,a,i0,a,i0)') &
                     &   'ERROR in parsing file '//trim(input_file)//', line=', iline_written,' argument=', iargument-1
                  write(stdout,*)'      values should be in list: ',exclusive(:)
               end if
               if (present(input_iostat)) then
                  input_iostat=1
                  return
               else
                  call leave()
               end if
            end if
         end if

         !increment the line if comment is present, do not touch the input file
         if (present(comment)) then
            call process_line(line_comment=comment)
         end if
      end if

      if (present(dict)) then
         if (associated(dict)) call dict_set(dict,var) !for the moment the format is the default one
      end if

    END SUBROUTINE var_double_compulsory


   subroutine var_real_compulsory(var,default,dict,ranges,exclusive,comment,input_iostat)
      implicit none
      character(len=*), intent(in) :: default
      real(kind=4), intent(out) :: var
      type(dictionary), pointer, optional :: dict
      character(len=*), intent(in), optional :: comment
      integer, intent(out), optional :: input_iostat
      real(kind=4), dimension(2), intent(in), optional :: ranges
      real(kind=4), dimension(:), intent(in), optional :: exclusive
      !Local variables
      real(gp) :: double_var
      logical :: found
      integer :: ierror,ilist

      if (present(input_iostat)) then
         !first, check if the line is correct
         if (iline_written>nlines_total ) then
            input_iostat=-1
            return
         else
            input_iostat=0 !no error for the moment
         end if
      else
         if (iline_written>nlines_total) then
            call leave()
         end if
      end if

      !if the file has not been opened, use the default variable 
      !then write in the output lines the default
      if (iline_parsed==0) then
         !finalize the line if the comment is present
         if (present(comment)) then
            call process_line(default=default,line_comment=comment)
         else
            call process_line(default=default)
         end if
         call read_fraction_string(default,double_var,ierror)
         if (present(input_iostat) .and. ierror /= 0) then
            input_iostat=-2
            return
         else
            call check(ierror)
         end if
         var=real(double_var,kind=4)
         !otherwise read the corresponding argument and check its validity
      else
         !read the argument
         call process_line()
         call read_fraction_string(line_being_processed,double_var,ierror)
         if (present(input_iostat) .and. ierror /= 0) then
            input_iostat=-2
            return
         else
            call check(ierror)
         end if
         var=real(double_var,kind=4)

         !check the validity of the variable
         if (present(ranges)) then
            if (var < ranges(1) .or. var > ranges(2)) then
               if (output) then
                  write(stdout,'(1x,a,i0,a,i0)') &
                     &   'ERROR in parsing file '//trim(input_file)//', line=', iline_written,' argument=', iargument-1
                  write(stdout,*)'      values should be in range: [',ranges(1),'-',ranges(2),']'
               end if
               if (present(input_iostat)) then
                  input_iostat=1
                  return
               else
                  call leave()
               end if
            end if
         else if (present(exclusive)) then
            found=.false.
            found_loop: do ilist=1,size(exclusive)
               if (var == exclusive(ilist)) then 
                  found=.true.
                  exit found_loop
               end if
            end do found_loop
            if (.not. found) then
               if (output) then 
                  write(stdout,'(1x,a,i0,a,i0)') &
                     &   'ERROR in parsing file '//trim(input_file)//', line=', iline_written,' argument=', iargument-1
                  write(stdout,*)'      values should be in list: ',exclusive(:)
               end if
               if (present(input_iostat)) then
                  input_iostat=1
                  return
               else
                  call leave()
               end if
            end if
         end if 

         !increment the line if comment is present, do not touch the input file
         if (present(comment)) then
            call process_line(line_comment=comment)
         end if
      end if   

      if (present(dict)) then
         if (associated(dict)) call dict_set(dict,var) !for the moment the format is the default one
      end if

   END SUBROUTINE var_real_compulsory


   subroutine var_int_compulsory(var,default,dict,ranges,exclusive,comment,input_iostat)
      implicit none
      character(len=*), intent(in) :: default
      integer, intent(out) :: var
      type(dictionary), pointer, optional :: dict
      character(len=*), intent(in), optional :: comment
      integer, intent(out), optional :: input_iostat
      integer, dimension(2), intent(in), optional :: ranges
      integer, dimension(:), intent(in), optional :: exclusive
      !Local variables
      logical :: found
      integer :: ierror,ilist

      if (present(input_iostat)) then
         !first, check if the line is correct
         if (iline_written>nlines_total) then
            input_iostat=-1
            return
         else
            input_iostat=0 !no error for the moment
         end if
      else
         if (iline_written>nlines_total) then
            call leave()
         end if
      end if

      !if the file has not been opened, use the default variable 
      !then write in the output lines the default
      if (iline_parsed==0) then
         !finalize the line if the comment is present
         if (present(comment)) then
            call process_line(default=default,line_comment=comment)
         else
            call process_line(default=default)
         end if
         read(default,*,iostat=ierror) var
         if (present(input_iostat) .and. ierror /= 0) then
            input_iostat=-2
            return
         else
            call check(ierror)
         end if
         !otherwise read the corresponding argument and check its validity
      else
         !read the argument
         call process_line()

         read(line_being_processed,fmt=*,iostat=ierror) var
         if (present(input_iostat) .and. ierror /= 0) then
            input_iostat=-2
            return
         else
            call check(ierror)
         end if

         !check the validity of the variable
         if (present(ranges)) then
            if (var < ranges(1) .or. var > ranges(2)) then
               if (output) then
                  write(stdout,'(1x,a,i0,a,i0)') &
                     &   'ERROR in parsing file '//trim(input_file)//', line=', iline_written,' argument=', iargument-1
                  write(stdout,*)'      values should be in range: [',ranges(1),'-',ranges(2),']'
               end if
               if (present(input_iostat)) then
                  input_iostat=1
                  return
               else
                  call leave()
               end if
            end if
         else if (present(exclusive)) then
            found=.false.
            found_loop: do ilist=1,size(exclusive)
               if (var == exclusive(ilist)) then
                  found=.true.
                  exit found_loop
               end if
            end do found_loop
            if (.not. found) then
               if (output) then
                  write(stdout,'(1x,a,i0,a,i0)') &
                     &   'ERROR in parsing file '//trim(input_file)//', line=', iline_written,' argument=', iargument-1
                  write(stdout,*)'      values should be in list: ',exclusive(:)
               end if
               if (present(input_iostat)) then
                  input_iostat=1
                  return
               else
                  call leave()
               end if
            end if
         end if 

         !increment the line if comment is present, do not touch the input file
         if (present(comment)) then
            call process_line(line_comment=comment)
         end if
      end if   

      if (present(dict)) then
         if (associated(dict)) call dict_set(dict,var) !for the moment the format is the default one
      end if
      
   END SUBROUTINE var_int_compulsory


   subroutine var_char_compulsory(var,default,dict,exclusive,comment,input_iostat)
      implicit none
      character(len=*), intent(in) :: default
      character(len=*), intent(out) :: var
      type(dictionary), pointer, optional :: dict
      character(len=*), intent(in), optional :: comment
      integer, intent(out), optional :: input_iostat
      character(len=*), dimension(:), intent(in), optional :: exclusive
      !Local variables
      logical :: found
      integer :: ierror,ilist

      if (present(input_iostat)) then
         !first, check if the line is correct (or if it is an optional line)
         if (iline_written>nlines_total) then
            input_iostat=-1
            return
         else
            input_iostat=0 !no error for the moment
         end if
      else
         if (iline_written>nlines_total) then
            call leave()
         end if
      end if

      !if the file has not been opened, use the default variable 
      !then write in the output lines the default
      if (iline_parsed==0) then
         !finalize the line if the comment is present
         if (present(comment)) then
            call process_line(default=default,line_comment=comment)
         else
            call process_line(default=default)
         end if
         read(default,'(a)',iostat=ierror) var
         if (present(input_iostat) .and. ierror /= 0) then
            input_iostat=-2
            return
         else
            call check(ierror)
         end if
         !otherwise read the corresponding argument and check its validity
      else
         !read the argument
         call process_line()
         read(line_being_processed,fmt=*,iostat=ierror) var
         if (present(input_iostat) .and. ierror /= 0) then
            input_iostat=-2
            return
         else
            call check(ierror)
         end if
         if (present(exclusive)) then
            found=.false.
            found_loop: do ilist=1,size(exclusive)
               if (case_insensitive_equiv(trim(var),trim(exclusive(ilist)))) then
                  found=.true.
                  exit found_loop
               end if
            end do found_loop
            if (.not. found) then
               if (output) then 
                  write(stdout,'(1x,a,i0,a,i0)') &
                     &   'ERROR in parsing file '//trim(input_file)//', line=', iline_written,' argument=', iargument-1
                  write(stdout,'(6x,a,30(1x,a))')&
                     &   'values should be in list: ',exclusive(:)
               end if
               if (present(input_iostat)) then
                  input_iostat=1
                  return
               else
                  call leave()
               end if
            end if
         end if 

         !increment the line if comment is present, do not touch the input file
         if (present(comment)) then
            call process_line(line_comment=comment)
         end if
      end if   

      if (present(dict)) then
         if (associated(dict)) call dict_set(dict,var) !for the moment the format is the default one
      end if

   END SUBROUTINE var_char_compulsory


   subroutine var_logical_compulsory(var,default,dict,comment,input_iostat)
      implicit none
      character(len=*), intent(in) :: default
      logical, intent(out) :: var
      type(dictionary), pointer, optional :: dict
      character(len=*), intent(in), optional :: comment
      integer, intent(out), optional :: input_iostat
      !Local variables
      integer :: ierror

      !if the file has not been opened, use the default variable 
      !then write in the output lines the default
      if (iline_parsed==0) then
         !finalize the line if the comment is present
         if (present(comment)) then
            call process_line(default=default,line_comment=comment)
         else
            call process_line(default=default)
         end if
         read(default,*,iostat=ierror) var
         if (present(input_iostat) .and. ierror /= 0) then
            input_iostat=-2
            return
         else
            call check(ierror)
         end if
         !otherwise read the corresponding argument and check its validity
      else
         !read the argument
         call process_line()

         read(line_being_processed,fmt=*,iostat=ierror) var
         if (present(input_iostat) .and. ierror /= 0) then
            input_iostat=-2
            return
         else
            call check(ierror)
         end if

         !increment the line if comment is present, do not touch the input file
         if (present(comment)) then
            call process_line(line_comment=comment)
         end if
      end if   
      
      if (present(dict)) then
         if (associated(dict)) call dict_set(dict,var) !for the moment the format is the default one
      end if

   END SUBROUTINE var_logical_compulsory


   !> Routines for non-compulsory file (input.perf)
   subroutine var_character(name, default, description, var)
     use yaml_strings, only : buffer_string
      character(len = *), intent(in) :: name
      character(len = *), intent(in) :: default
      character(len = *), intent(in) :: description
      character(len = *), intent(out) :: var
      !local variables
      character(len=max_length) :: line
      character(len=2) :: frmt
      integer :: i, j, ierror, ierr,iblk,lgt,tab
      
      var=repeat(' ',len(var))

      write(var, "(A)") default

      call find(name, i, j)
      if (i > 0) then
         !read all the line
         write(frmt,'(i2)')max(max_length-j-1,0)
         read(inout_lines(i)(j + 2:), fmt = '(a'//trim(adjustl(frmt))//')', iostat = ierror) line
         if (ierror/=0) then
            if (output) write(stdout,'(1x,a,a,a,i3)')  'Error while reading the file "', &
               &   trim(input_file), '", line=', i
            call MPI_ABORT(bigdft_mpi%mpi_comm,ierror,ierr)
         end if
         !now line has potentially the slashes
         if (scan(line,'/') > 0) then
            line=adjustl(line)
            iblk=scan(line,' ')
            if (iblk >0 ) then
               var(1:iblk)=line(1:iblk)
            else
               read(inout_lines(i)(j + 2:), fmt = *, iostat = ierror) var
            end if
         else
            read(inout_lines(i)(j + 2:), fmt = *, iostat = ierror) var
         end if
         if (ierror/=0) then
            if (output) write(stdout,'(1x,a,a,a,i3)')  'Error while reading the file "', &
                 &   trim(input_file), '", line=', i
            call MPI_ABORT(bigdft_mpi%mpi_comm,ierror,ierr)
         end if
      end if
      if (output) then
         lgt=0
         call buffer_string(inout_lines(parsed_lines(iline_parsed-1)),max_length,trim(name)//' ',lgt)
         call buffer_string(inout_lines(parsed_lines(iline_parsed-1)),max_length,trim(var),lgt)
         tab=max(29-lgt,1)
         call buffer_string(inout_lines(parsed_lines(iline_parsed-1)),max_length,&
              repeat(' ',tab),lgt)
         call buffer_string(inout_lines(parsed_lines(iline_parsed-1)),max_length,&
              trim(description),lgt)

         !write(inout_lines(parsed_lines(iline_parsed-1)),"(a,1x,a,t30,a)") name, trim(var), description
         iline_written=iline_written+1
      end if
   END SUBROUTINE var_character


   subroutine var_logical(name, default, description, var)
      character(len = *), intent(in) :: name
      logical, intent(in) :: default
      character(len = *), intent(in) :: description
      logical, intent(out) :: var

      integer :: i, j, ierror

      var = default
      call find(name, i, j)
      if (i > 0) then
         read(inout_lines(i)(j + 2:), fmt = *, iostat = ierror) var
         if (ierror /= 0) then
            var = .true.
         end if
      end if
      if (output) then
         write(inout_lines(parsed_lines(iline_parsed-1)),"(a,1x,l1,t30,a)") name, var, description
         iline_written=iline_written+1
      end if
   END SUBROUTINE var_logical


   subroutine var_integer(name, default, description, var)
      character(len = *), intent(in) :: name
      integer, intent(in) :: default
      character(len = *), intent(in) :: description
      integer, intent(out) :: var

      integer :: i, j, ierror, ierr
      var = default
      call find(name, i, j)

      if (i > 0) then
         read(inout_lines(i)(j + 2:), fmt = *, iostat = ierror) var
         if (ierror/=0) then
            if (output) write(stdout,'(1x,a,a,a,i3)')  'Error while reading the file "', &
               &   trim(input_file), '", line=', i
            call MPI_ABORT(bigdft_mpi%mpi_comm,ierror,ierr)
         end if
      end if
      if (output) then
         write(inout_lines(parsed_lines(iline_parsed-1)),"(a,1x,I0,t30,a)") name, var, description
         iline_written=iline_written+1
      end if
   END SUBROUTINE var_integer


   subroutine var_integer_array(name, default, description, var)
     use yaml_strings, only: buffer_string,yaml_toa
      character(len = *), intent(in) :: name
      integer, intent(in) :: default(:)
      character(len = *), intent(in) :: description
      integer, intent(out) :: var(:)

      integer :: i, j, ierror, ierr,lgt,tab

      var = default
      call find(name, i, j)
      if (i > 0) then
         read(inout_lines(i)(j + 2:), fmt = *, iostat = ierror) var
         if (ierror/=0) then
            if (output) write(stdout,'(1x,a,a,a,i3)')  'Error while reading the file "', &
               &   trim(input_file), '", line=', i
            call MPI_ABORT(bigdft_mpi%mpi_comm,ierror,ierr)
         end if
      end if
      if (output) then
         lgt=0
         call buffer_string(inout_lines(parsed_lines(iline_parsed-1)),max_length,trim(name),lgt)
         do j = 1, size(var), 1
            call buffer_string(inout_lines(parsed_lines(iline_parsed-1)),max_length,&
                 trim(yaml_toa(var(j),fmt='(i0)')),lgt)
         end do
         tab=max(29-lgt,1)
         call buffer_string(inout_lines(parsed_lines(iline_parsed-1)),max_length,&
              repeat(' ',tab),lgt)
         call buffer_string(inout_lines(parsed_lines(iline_parsed-1)),max_length,&
              trim(description),lgt)
!!$         write(inout_lines(iline_written),"(1x,a,a,1x)", advance = "NO") "#|", name
!!$         do i = 1, size(var), 1
!!$            write(inout_lines(iline_written),"(1x,I0)", advance = "NO") var(i)
!!$         end do
!!$         write(inout_lines(iline_written),"(t10,a)")description
         iline_written=iline_written+1
      end if
   END SUBROUTINE var_integer_array


   subroutine var_double(name, default, description, var)
      character(len = *), intent(in) :: name
      double precision, intent(in) :: default
      character(len = *), intent(in) :: description
      double precision, intent(out) :: var

      integer :: i, j, ierror, ierr

      var = default
      call find(name, i, j)
      if (i > 0) then
         read(inout_lines(i)(j + 2:), fmt = *, iostat = ierror) var
         if (ierror/=0) then
            if (output) write(stdout,'(1x,a,a,a,i3)')  'Error while reading the file "', &
               &   trim(input_file), '", line=', i
            call MPI_ABORT(bigdft_mpi%mpi_comm,ierror,ierr)
         end if
      end if
      if (output) then
         write(inout_lines(parsed_lines(iline_parsed-1)),"(a,1x,es9.2,t30,a)") name, var, description
         iline_written=iline_written+1
      end if
   END SUBROUTINE var_double


   subroutine var_keyword(name, length, default, list, description, var)
     use yaml_strings, only: buffer_string
      character(len = *), intent(in) :: name
      integer, intent(in) :: length
      character(len = length), intent(in) :: default
      character(len = length), intent(in) :: list(:)
      character(len = *), intent(in) :: description
      integer, intent(out) :: var

      integer :: i, j, ierror, ierr,lgt,tab
      character(len = length) :: buf

      ! Set the default value to var.
      do i = 1, size(list), 1
         if (trim(default) == trim(list(i))) exit
      end do
      var = i - 1
      ! Find the keyword name in the file.
      call find(name, i, j)
      if (i > 0) then
         read(inout_lines(i)(j + 2:), fmt = *, iostat = ierror) buf
         if (ierror/=0) then
            if (output) write(stdout,'(1x,a,a,a,i3)')  'Error while reading the file "', &
               &   trim(input_file), '", line=', i
            call MPI_ABORT(bigdft_mpi%mpi_comm,ierror,ierr)
         end if
         ! Look for buf in list.
         do j = 1, size(list), 1
            if (trim(buf) == trim(list(j))) exit
         end do
         if (j > size(list)) then
            if (output) write(stdout,'(1x,a,a,a,i3)')  'Error while reading the file "', &
               &   trim(input_file), '", line=', i
            call MPI_ABORT(bigdft_mpi%mpi_comm,ierror,ierr)
         end if
         var = j - 1
      end if
      if (output) then
         lgt=0
         call buffer_string(inout_lines(parsed_lines(iline_parsed-1)),max_length,&
              trim(name)//' '//trim(list(var+1)),lgt)
         tab=max(29-lgt,1)
         call buffer_string(inout_lines(parsed_lines(iline_parsed-1)),max_length,&
              repeat(' ',tab),lgt)
         call buffer_string(inout_lines(parsed_lines(iline_parsed-1)),max_length,&
              trim(description)//' ('//trim(list(1)),lgt)
         do j = 2, size(list)
            call buffer_string(inout_lines(parsed_lines(iline_parsed-1)),max_length,&
                 ', '//trim(list(j)),lgt)
         end do
         call buffer_string(inout_lines(parsed_lines(iline_parsed-1)),max_length,') ',lgt)

!!$         write(inout_lines(iline_written),"(1x,a,a,1x,a,t30,2a)", advance = "NO") &
!!$            &   "#|", name, list(var + 1), description, " ("
!!$         write(inout_lines(iline_written),"(A)", advance = "NO") trim(list(1))
!!$         do i = 2, size(list), 1
!!$            write(inout_lines(iline_written),"(2A)", advance = "NO") ", ", trim(list(i))
!!$         end do
!!$         write(inout_lines(iline_written),"(A)") ")"
         iline_written=iline_written+1
      end if
   END SUBROUTINE var_keyword


   subroutine var_ids(name, default, list, description, var)
      character(len = *), intent(in) :: name
      integer, intent(in) :: default
      integer, intent(in) :: list(:)
      character(len = *), intent(in) :: description
      integer, intent(out) :: var

      integer :: i, j, ierror, ierr

      var = default
      call find(name, i, j)
      if (i > 0) then
         read(inout_lines(i)(j + 2:), fmt = *, iostat = ierror) var
         if (ierror/=0) then
            if (output) write(stdout,'(1x,a,a,a,i3)')  'Error while reading the file "', &
               &   trim(input_file), '", line=', i
            call MPI_ABORT(bigdft_mpi%mpi_comm,ierror,ierr)
         end if
         do j = 1, size(list), 1
            if (var == list(j)) exit
         end do
         if (j > size(list)) then
            if (output) write(stdout,'(1x,a,a,a,i3)')  'Error while reading the file "', &
               &   trim(input_file), '", line=', i
            call MPI_ABORT(bigdft_mpi%mpi_comm,ierror,ierr)
         end if
      end if
      if (output) then
         write(inout_lines(parsed_lines(iline_parsed-1)),"(a,1x,I0,t30,a)") name, var, description
         iline_written=iline_written+1
      end if
   END SUBROUTINE var_ids

END MODULE module_input


!> Module reading the old format (before 1.7) for the input
module input_old_text_format
  implicit none
  public
contains
  subroutine read_dft_from_text_format(iproc,dict,filename)
    use module_base
    use module_types
    use module_input
    use module_input_keys
    use dictionaries
    !  use yaml_output
    implicit none
    type(dictionary), pointer :: dict
    character(len=*), intent(in) :: filename
    integer, intent(in) :: iproc
    !local variables
    logical :: exists
    integer :: ierror
    real(gp), dimension(2), parameter :: hgrid_rng=(/0.0_gp,2.0_gp/)
    real(gp), dimension(2), parameter :: xrmult_rng=(/0.0_gp,100.0_gp/)

    logical :: dummy_bool
    integer :: dummy_int
    real(gp) :: dummy_real
    real(gp), dimension(3) :: dummy_real3

    !dft parameters, needed for the SCF part
    call input_set_file(iproc,(iproc == 0),trim(filename),exists, DFT_VARIABLES)
    !if (exists) in%files = in%files + INPUTS_DFT
    !call the variable, its default value, the line ends if there is a comment
    if (.not. exists) then
       call input_free(.false.)
       return
    end if

    if (.not. associated(dict)) call dict_init(dict)

    !grid spacings
    call input_var(dummy_real3(1),'0.45',dict//HGRIDS//0,ranges=hgrid_rng)
    call input_var(dummy_real3(2),'0.45',dict//HGRIDS//1,ranges=hgrid_rng)
    call input_var(dummy_real3(3),'0.45',dict//HGRIDS//2,ranges=hgrid_rng,&
         comment='hx,hy,hz: grid spacing in the three directions')

    !coarse and fine radii around atoms
    call input_var(dummy_real,'5.0',dict//RMULT//0,ranges=xrmult_rng)
    call input_var(dummy_real,'8.0',dict//RMULT//1,ranges=xrmult_rng,&
         comment='c(f)rmult: c(f)rmult*radii_cf(:,1(2))=coarse(fine) atom-based radius')

    !XC functional (ABINIT XC codes)
    call input_var(dummy_int,'1',dict//IXC,comment='ixc: exchange-correlation parameter (LDA=1,PBE=11)')

    !charge and electric field
    call input_var(dummy_int,'0',dict//NCHARGE,ranges=(/-500,500/))

    call input_var(dummy_real3(1),'0.',dict//ELECFIELD//0)
    call input_var(dummy_real3(2),'0.',dict//ELECFIELD//1)
    call input_var(dummy_real3(3),'0.',dict//ELECFIELD//2,&
         comment='charge of the system, Electric field (Ex,Ey,Ez)')

    !spin and polarization
    call input_var(dummy_int,'1',dict//NSPIN,exclusive=(/1,2,4/))
    call input_var(dummy_int,'0',dict//MPOL,comment='nspin=1 non-spin polarization, mpol=total magnetic moment')

    !convergence parameters
    call input_var(dummy_real,'1.e-4',dict//GNRM_CV,ranges=(/1.e-20_gp,1.0_gp/),&
         comment='gnrm_cv: convergence criterion gradient')
    call input_var(dummy_int,'50',dict//ITERMAX,ranges=(/0,10000/))
    call input_var(dummy_int,'1',dict//NREPMAX,ranges=(/0,1000/),&
         comment='itermax,nrepmax: max. # of wfn. opt. steps and of re-diag. runs')

    !convergence parameters
    call input_var(dummy_int,'6',dict//NCONG,ranges=(/0,20/))
    call input_var(dummy_int,'6',dict//IDSX,ranges=(/0,15/),&
         comment='ncong, idsx: # of CG it. for preconditioning eq., wfn. diis history')

    !dispersion parameter
    call input_var(dummy_int,'0',dict//DISPERSION,ranges=(/0,5/),&
         comment='dispersion correction potential (values 1,2,3,4,5), 0=none')

    ! Now the variables which are to be used only for the last run
    call input_var(dummy_int,'0',dict//INPUTPSIID,&
         exclusive=(/-2,-1,0,2,10,12,13,100,101,102/),input_iostat=ierror)
    ! Validate inputPsiId value (Can be added via error handling exception)
    if (ierror /=0 .and. iproc == 0) then
       write( *,'(1x,a,I0,a)')'ERROR: illegal value of inputPsiId (', dummy_int, ').'
       call input_psi_help()
       call MPI_ABORT(bigdft_mpi%mpi_comm,0,ierror)
    end if

    call input_var(dummy_int,'0',dict//OUTPUT_WF,exclusive=(/0,1,2,3/),input_iostat=ierror)
    ! Validate output_wf value.
    if (ierror /=0 .and. iproc == 0) then
       write( *,'(1x,a,I0,a)')'ERROR: illegal value of output_wf (', dummy_int, ').'
       call output_wf_format_help()
       call MPI_ABORT(bigdft_mpi%mpi_comm,0,ierror)
    end if

    call input_var(dummy_int,'0',dict//OUTPUT_DENSPOT,exclusive=(/0,1,2,10,11,12,20,21,22/),&
         comment='InputPsiId, output_wf, output_denspot')

    ! Tail treatment.
    call input_var(dummy_real,'0.0',dict//RBUF,ranges=(/0.0_gp,10.0_gp/))
    call input_var(dummy_int,'30',dict//NCONGT,ranges=(/1,50/),&
         comment='rbuf, ncongt: length of the tail (AU),# tail CG iterations')

    !davidson treatment
    call input_var(dummy_int,'0',dict//NORBV,ranges=(/-9999,9999/))
    call input_var(dummy_int,'0',dict//NVIRT)
    call input_var(dummy_int,'0',dict//NPLOT,&
         comment='Davidson subspace dim., # of opt. orbs, # of plotted orbs')

    ! Line to disable automatic behaviours (currently only symmetries).
    call input_var(dummy_bool,'F',dict//DISABLE_SYM,comment='disable the symmetry detection')

    call input_free(.false.)

  end subroutine read_dft_from_text_format


  !> Read the input variables needed for the geometry optmization
  !! Every argument should be considered as mandatory
  subroutine read_geopt_from_text_format(iproc,dict,filename)
    use module_base
    use module_types
    use module_input
    use module_input_keys
    use dictionaries
    implicit none
    integer, intent(in) :: iproc
    character(len=*), intent(in) :: filename
    type(dictionary), pointer :: dict
    !local variables
    character(len=*), parameter :: subname='read_geopt_from_text_format'
    integer :: i
    logical :: exists

    character(len = 5) :: dummy_str
    integer :: dummy_int, ionmov_
    real(gp) :: dummy_real

    !geometry input parameters
    call input_set_file(iproc,(iproc == 0),trim(filename),exists,GEOPT_VARIABLES)  
    !if (exists) in%files = in%files + INPUTS_GEOPT
    !call the variable, its default value, the line ends if there is a comment
    if (.not. exists) then
       call input_free(.false.)
       return
    end if

    call input_var(dummy_str,"BFGS",dict // GEOPT_METHOD, comment = "")
    !call set(dict // GEOPT_METHOD, dummy_str)
    call input_var(dummy_int,'1',dict // NCOUNT_CLUSTER_X,comment="")
    !call set(dict // NCOUNT_CLUSTER_X, dummy_int)

    call input_var(dummy_real,'1.0',dict // FRAC_FLUCT)
    !call set(dict // FRAC_FLUCT, dummy_real, fmt = "(E8.2)")
    call input_var(dummy_real,'0.0',dict // FORCEMAX,comment="")
    !call set(dict // FORCEMAX, dummy_real, fmt = "(E8.2)")
    call input_var(dummy_real,'0.0',dict // RANDDIS,comment="")
    !call set(dict // RANDDIS, dummy_real, fmt = "(E8.2)")

    if (input_keys_equal(trim(dummy_str),"AB6MD")) then
       call input_var(ionmov_,'6',dict // IONMOV,comment="")
       !call set(dict // IONMOV, ionmov_)
       call input_var(dummy_real,'20.670689',dict // DTION,comment="")
       !call set(dict // DTION, dummy_real)
       if (ionmov_ == 6) then
          call input_var(dummy_real,'300',dict // MDITEMP,comment="")
          !call set(dict // MDITEMP, dummy_real)
       elseif (ionmov_ > 7) then
          call input_var(dummy_real,'300',dict // MDITEMP)
          !call set(dict // MDITEMP, dummy_real)
          call input_var(dummy_real,'300',dict // MDFTEMP,comment="")
          !call set(dict // MDFTEMP, dummy_real)
       end if

       if (ionmov_ == 8) then
          call input_var(dummy_real,'1.e5',dict // NOSEINERT,comment="")
          !call set(dict // NOSEINERT, dummy_real)
       else if (ionmov_ == 9) then
          call input_var(dummy_real,'1.e-3',dict // FRICTION,comment="")
          !call set(dict // FRICTION, dummy_real)
          call input_var(dummy_real,'1.e4',dict // MDWALL,comment="")
          !call set(dict // MDWALL, dummy_real)
       else if (ionmov_ == 13) then
          !here no dictionary
          call input_var(dummy_int,'0',ranges=(/0,100/),comment="")
          do i=1,dummy_int-1
             call input_var(dummy_real,'0.0',dict // QMASS // (i-1))
             !call set(dict // QMASS // (i-1), dummy_real)
          end do
          if (dummy_int > 0) then
             call input_var(dummy_real,'0.0',dict // QMASS // (dummy_int-1),comment="")
             !call set(dict // QMASS // (dummy_int-1), dummy_real)
          end if
          call input_var(dummy_real,'10',dict // BMASS)
          !call set(dict // BMASS, dummy_real)
          call input_var(dummy_real,'1.0',dict // VMASS,comment="")
          !call set(dict // VMASS, dummy_real)
       end if
    else if (input_keys_equal(trim(dummy_str),"DIIS")) then
       call input_var(dummy_real,'2.0',dict // BETAX)
       !call set(dict // BETAX, dummy_real, fmt = "(F6.3)")
       call input_var(dummy_int,'4',dict // HISTORY,comment="")
       !call set(dict // HISTORY, dummy_int)
    else
       call input_var(dummy_real,'4.0',dict // BETAX,comment="")
       !call set(dict // BETAX, dummy_real, fmt = "(F6.3)")
    end if
    if (input_keys_equal(trim(dummy_str),"FIRE")) then
       call input_var(dummy_real,'0.75',dict // DTINIT)
       !call set(dict // DTINIT, dummy_real, fmt = "(F6.3)")
       call input_var(dummy_real, '1.5',dict // DTMAX,comment="")
       !call set(dict // DTMAX, dummy_real, fmt = "(F6.3)")
    endif

    call input_free(.false.)

  END SUBROUTINE read_geopt_from_text_format

  !> Read the input variables needed for the geometry optmization
  !!    Every argument should be considered as mandatory
  subroutine read_mix_from_text_format(iproc,dict,filename)
    use module_base
    use module_types
    use module_input
    use module_input_keys
    use dictionaries
    implicit none
    !Arguments
    integer, intent(in) :: iproc
    type(dictionary), pointer :: dict
    character(len=*), intent(in) :: filename
    !local variables
    !n(c) character(len=*), parameter :: subname='mix_input_variables'
    logical :: exists
    integer :: dummy_int
    real(gp) :: dummy_real

    !Mix parameters, needed for the SCF poart with Davidson
    call input_set_file(iproc,(iproc == 0),trim(filename),exists,MIX_VARIABLES)
    !if (exists) in%files = in%files + INPUTS_MIX
    !call the variable, its default value, the line ends if there is a comment
    if (.not.exists) then
       call input_free(.false.)
       return
    end if

    if (.not. associated(dict)) call dict_init(dict)

    !Controls the self-consistency: 0 direct minimisation otherwise ABINIT convention
    call input_var(dummy_int,'0',dict // ISCF,comment="")
    !call set(dict // ISCF, dummy_int)
    call input_var(dummy_int,'1',dict // ITRPMAX,comment="")
    !call set(dict // ITRPMAX, dummy_int)
    call input_var(dummy_real,'1.e-4',dict // RPNRM_CV,comment="")
    !call set(dict // RPNRM_CV, dummy_real, fmt = "(E8.1)")
    call input_var(dummy_int,'0',dict // NORBSEMPTY)
    !call set(dict // NORBSEMPTY, dummy_int)
    call input_var(dummy_real,'0.0',dict // TEL) 
    !call set(dict // TEL, dummy_real, fmt = "(E9.2)")
    call input_var(dummy_int,'1',dict // OCCOPT,comment="")
    !call set(dict // OCCOPT, dummy_int)
    call input_var(dummy_real,'0.0',dict // ALPHAMIX)
    !call set(dict // ALPHAMIX, dummy_real, fmt = "(F6.3)")
    call input_var(dummy_real,'2.0',dict // ALPHADIIS,comment="")
    !call set(dict // ALPHADIIS, dummy_real, fmt = "(F6.3)")

    call input_free(.false.)
  END SUBROUTINE read_mix_from_text_format

  !> Read Self-Interaction Correction (SIC) input parameters
  subroutine read_sic_from_text_format(iproc,dict,filename)
    use module_input
    use module_input_keys
    use dictionaries
    implicit none
    integer, intent(in) :: iproc
    type(dictionary), pointer :: dict
    character(len=*), intent(in) :: filename
    !local variables
    logical :: exists
    !n(c) character(len=*), parameter :: subname='sic_input_variables'
    double precision :: dummy_real
    character(len = 4) :: dummy_str

    !Self-Interaction Correction input parameters
    call input_set_file(iproc,(iproc == 0),trim(filename),exists,'SIC Parameters')  
    !if (exists) in%files = in%files + INPUTS_SIC
    if (.not.exists) then
       call input_free(.false.)
       return
    end if

    call input_var(dummy_str,'NONE',dict // SIC_APPROACH,comment='')
    !call set(dict // SIC_APPROACH, dummy_str)
    call input_var(dummy_real,'0.0',dict // SIC_ALPHA,comment='')
    !call set(dict // SIC_ALPHA, dummy_real, fmt = "(E8.2)")
    
    if (input_keys_equal(trim(dummy_str),'NK')) then
       call input_var(dummy_real,'0.0',dict // SIC_FREF,comment='')
       !call set(dict // SIC_FREF, dummy_real, fmt = "(E8.2)")
    end if

    call input_free(.false.)
  END SUBROUTINE read_sic_from_text_format

  subroutine read_tddft_from_text_format(iproc,dict,filename)
    use module_input
    use module_input_keys
    use dictionaries
    implicit none
    integer, intent(in) :: iproc
    type(dictionary), pointer :: dict
    character(len=*), intent(in) :: filename
    !local variables
    logical :: exists
    !n(c) character(len=*), parameter :: subname='tddft_input_variables'
    character(len = 4) :: dummy_str

    !TD-DFT parameters
    call input_set_file(iproc,(iproc == 0),trim(filename),exists,'TD-DFT Parameters')  
    !if (exists) in%files = in%files + INPUTS_TDDFT
    !call the variable, its default value, the line ends if there is a comment
    if (.not. exists) then
       call input_free(.false.)
       return
    end if

    if (.not. associated(dict)) call dict_init(dict)

    call input_var(dummy_str,"NONE",dict // TDDFT_APPROACH,comment="")
    !call set(dict // TDDFT_APPROACH, dummy_str)

    call input_free(.false.)

  END SUBROUTINE read_tddft_from_text_format

  subroutine read_kpt_from_text_format(iproc,dict,filename)
    use module_base
    use module_types
    use dictionaries
    use module_input
    use module_input_keys
    implicit none
    character(len=*), intent(in) :: filename
    integer, intent(in) :: iproc
    type(dictionary), pointer :: dict
    !local variables
    logical :: exists
    character(len=*), parameter :: subname='read_kpt_from_text_format'

    integer :: dummy_int, nseg, i, ierror
    integer, dimension(3) :: dummy_int3
    real(gp) :: dummy_real
    real(gp), dimension(3) :: dummy_real3
    character(len = max_field_length) :: dummy_str

    !kpt parameters, needed for the SCF part
    call input_set_file(iproc,(iproc == 0),trim(filename),exists, KPT_VARIABLES)
    !if (exists) in%files = in%files + INPUTS_KPT
    !call the variable, its default value, the line ends if there is a comment
    if (.not. exists) then
       call input_free(.false.)
       return
    end if

    if (.not. associated(dict)) call dict_init(dict)

    !if the file does exist, we fill up the dictionary.
    call input_var(dummy_str, 'manual',dict//KPT_METHOD, comment='K-point sampling method')
    !call set(dict//KPT_METHOD, trim(dummy_str))

    if (input_keys_equal(trim(dummy_str),'auto')) then
       call input_var(dummy_real,'0.0',dict//KPTRLEN, comment='Equivalent length of K-space resolution (Bohr)')
       !call set(dict//KPTRLEN, dummy_real)
    else if (input_keys_equal(trim(dummy_str),'mpgrid')) then
       !take the points of Monckorst-pack grid
       call input_var(dummy_int3(1),'1',dict//NGKPT//0)
       call input_var(dummy_int3(2),'1',dict//NGKPT//1)
       call input_var(dummy_int3(3),'1',dict//NGKPT//2, comment='No. of Monkhorst-Pack grid points')
       !call set(dict//NGKPT//0, dummy_int3(1))
       !call set(dict//NGKPT//1, dummy_int3(2))
       !call set(dict//NGKPT//2, dummy_int3(3))
       !shift
       !no dict here
       call input_var(dummy_int,'1',ranges=(/1,8/),comment='No. of different shifts')
       !read the shifts
       do i=1,dummy_int
          call input_var(dummy_real3(1),'0.',dict//SHIFTK//(i-1)//0)
          call input_var(dummy_real3(2),'0.',dict//SHIFTK//(i-1)//1)
          call input_var(dummy_real3(3),'0.',dict//SHIFTK//(i-1)//2,comment=' ')
          !call set(dict//SHIFTK//(i-1)//0, dummy_real3(1), fmt = "(F6.4)")
          !call set(dict//SHIFTK//(i-1)//1, dummy_real3(2), fmt = "(F6.4)")
          !call set(dict//SHIFTK//(i-1)//2, dummy_real3(3), fmt = "(F6.4)")
       end do
    else if (input_keys_equal(trim(dummy_str),'manual')) then
       call input_var(dummy_int,'1',ranges=(/1,10000/),&
            comment='Number of K-points')
       do i=1,dummy_int
          call input_var(dummy_real3(1),'0.',dict//KPT//(i-1)//0)
          call input_var(dummy_real3(2),'0.',dict//KPT//(i-1)//1)
          call input_var(dummy_real3(3),'0.',dict//KPT//(i-1)//2)
          !call set(dict//KPT//(i-1)//0, dummy_real3(1), fmt = "(F6.4)")
          !call set(dict//KPT//(i-1)//1, dummy_real3(2), fmt = "(F6.4)")
          !call set(dict//KPT//(i-1)//2, dummy_real3(3), fmt = "(F6.4)")
          call input_var(dummy_real,'1.',dict//WKPT//(i-1),comment='K-pt coords, K-pt weigth')
          !call set(dict//WKPT//(i-1), dummy_real, fmt = "(F6.4)")
       end do
    end if

    ! Now read the band structure definition. do it only if the file exists
    !no dictionary here
    call input_var(dummy_str,'bands',comment='For doing band structure calculation',&
         input_iostat=ierror)
    call set(dict//BANDS, (ierror==0))
    if (ierror==0) then
       call input_var(nseg,'1',ranges=(/1,1000/),&
            comment='# of segments of the BZ path')
       !number of points for each segment, parallel granularity
       do i=1,nseg
          call input_var(dummy_int,'1',dict//ISEG)
          !call set(dict//ISEG, dummy_int)
       end do
       call input_var(dummy_int,'1',dict//NGRANULARITY,&
            comment='points for each segment, # of points done for each group')
       !call set(dict//NGRANULARITY, dummy_int)

       call input_var(dummy_real3(1),'0.',dict//KPTV//0//0)
       call input_var(dummy_real3(2),'0.',dict//KPTV//0//1)
       call input_var(dummy_real3(3),'0.',dict//KPTV//0//2,comment=' ')
!       call set(dict//KPTV//0//0, dummy_real3(1))
!       call set(dict//KPTV//0//1, dummy_real3(2))
!       call set(dict//KPTV//0//2, dummy_real3(3))
       do i=1,nseg
          call input_var(dummy_real3(1),'0.5',dict//KPTV//(i-1)//0)
          call input_var(dummy_real3(2),'0.5',dict//KPTV//(i-1)//1)
          call input_var(dummy_real3(3),'0.5',dict//KPTV//(i-1)//2,comment=' ')
          !call set(dict//KPTV//(i-1)//0, dummy_real3(1))
          !call set(dict//KPTV//(i-1)//1, dummy_real3(2))
          !call set(dict//KPTV//(i-1)//2, dummy_real3(3))
       end do

       !read an optional line to see if there is a file associated
       !no dict for the moment
       call input_var(dummy_str,' ',&
            comment=' ',input_iostat=ierror)
       if (ierror == 0) then
          !since a file for the local potential is already given, do not perform ground state calculation
          call set(dict//BAND_STRUCTURE_FILENAME, dummy_str)
       end if
    end if

    call input_free(.false.)

  end subroutine read_kpt_from_text_format


  !> Read the input variables which can be used for performances
  subroutine read_perf_from_text_format(iproc,dict,filename)
    use module_input
    use module_input_keys
    use dictionaries
    implicit none
    character(len=*), intent(in) :: filename
    type(dictionary), pointer :: dict
    integer, intent(in) :: iproc
    !local variables
    !n(c) character(len=*), parameter :: subname='perf_input_variables'
    logical :: exists, dummy_bool
    integer :: dummy_int, blocks(2)
    double precision :: dummy_real
    character(len = 7) :: dummy_str
    character(len = max_field_length) :: dummy_path

    call input_set_file(iproc, (iproc == 0), filename, exists, PERF_VARIABLES)
    !if (exists) in%files = in%files + INPUTS_PERF
    if (.not. exists) then
       call input_free(.false.)
       return
    end if

    if (.not. associated(dict)) call dict_init(dict)

    call input_var("debug", .false., "Debug option", dummy_bool)
    call set(dict // DEBUG, dummy_bool)
    call input_var("fftcache", 8*1024, "Cache size for the FFT", dummy_int)
    call set(dict // FFTCACHE, dummy_int)
    call input_var("accel", "NO", "Acceleration", dummy_str)
    call set(dict // ACCEL, dummy_str)

    !determine desired OCL platform which is used for acceleration
    call input_var("OCL_platform"," ", "Chosen OCL platform", dummy_str)
    call set(dict // OCL_PLATFORM, dummy_str)
    call input_var("OCL_devices"," ", "Chosen OCL devices", dummy_str)
    call set(dict // OCL_DEVICES, dummy_str)

    !!@TODO to relocate
    call input_var("blas", .false., "CUBLAS acceleration", dummy_bool)
    call set(dict // BLAS, dummy_bool)
    call input_var("projrad", 15.0d0, "Radius ", dummy_real)
    call set(dict // PROJRAD, dummy_real, fmt = "(F6.3)")
    call input_var("exctxpar", "OP2P", "Exact exchange parallelisation scheme", dummy_str)
    call set(dict // EXCTXPAR, dummy_str)
    call input_var("ig_diag", .true.,"Input guess", dummy_bool)
    call set(dict // IG_DIAG, dummy_bool)
    call input_var("ig_norbp", 5, "Input guess: ", dummy_int)
    call set(dict // IG_NORBP, dummy_int)
    call input_var("ig_blocks", (/ 300, 800 /), "Input guess: ", blocks)
    call set(dict // IG_BLOCKS // 0, blocks(1))
    call set(dict // IG_BLOCKS // 1, blocks(2))
    call input_var("ig_tol", 1d-4, "Input guess: Tolerance criterion", dummy_real)
    call set(dict // IG_TOL, dummy_real, fmt = "(E8.1)")
    call input_var("methortho", 0, "Orthogonalisation ", dummy_int)
    call set(dict // METHORTHO, dummy_int)
    call input_var("rho_commun", "DEF","Density communication scheme (DBL, RSC, MIX)",dummy_str)
    call set(dict // RHO_COMMUN, dummy_str)
    call input_var("psolver_groupsize",0, "Size of ", dummy_int)
    call set(dict // PSOLVER_GROUPSIZE, dummy_int)
    call input_var("psolver_accel",0, "Acceleration ", dummy_int)
    call set(dict // PSOLVER_ACCEL, dummy_int)
    call input_var("unblock_comms", "OFF", "Overlap Com)",dummy_str)
    call set(dict // UNBLOCK_COMMS, dummy_str)
    call input_var("linear", 'OFF', "Linear Input Guess approach",dummy_str)
    call set(dict // LINEAR, dummy_str)
    call input_var("tolsym", 1d-8, "Tolerance for symmetry detection",dummy_real)
    call set(dict // TOLSYM, dummy_real, fmt = "(E8.1)")
    call input_var("signaling", .false., "Expose calculation results on Network",dummy_bool)
    call set(dict // SIGNALING, dummy_bool)
    call input_var("signalTimeout", 0, "Time out on startup for signal connection",dummy_int)  
    call set(dict // SIGNALTIMEOUT, dummy_int)
    call input_var("domain", "", "Domain to add to the hostname to find the IP", dummy_str)
    call set(dict // DOMAIN, dummy_str)
    call input_var("inguess_geopt", 0,"0= wavelet input ",dummy_int)
    call set(dict // INGUESS_GEOPT, dummy_int)
    call input_var("store_index", .true., "Linear scaling: store ", dummy_bool)
    call set(dict // STORE_INDEX, dummy_bool)
    !verbosity of the output
    call input_var("verbosity", 2, "Verbosity of the output 0=low, 2=high",dummy_int)
    call set(dict // VERBOSITY, dummy_int)
    call input_var("outdir", ".","Writing directory", dummy_path)
    call set(dict // OUTDIR, dummy_path)

    !If false, apply the projectors in the once-and-for-all scheme, otherwise on-the-fly
    call input_var("psp_onfly", .true., "Calculate the PSP projectors on the fly (less memory)",dummy_bool)
    call set(dict // PSP_ONFLY, dummy_bool)

    !If true, preserve the multipole of the ionic part (local potential) projecting on delta instead of ISF
    call input_var("multipole_preserving", .false., "Preserve multipole moment of the ionic charge",dummy_bool)
    call set(dict // MULTIPOLE_PRESERVING, dummy_bool)

    !block size for pdsyev/pdsygv, pdgemm (negative -> sequential)
    call input_var("pdsyev_blocksize",-8,"SCALAPACK linear scaling blocksize",dummy_int) !ranges=(/-100,1000/)
    call set(dict // PDSYEV_BLOCKSIZE, dummy_int)
    call input_var("pdgemm_blocksize",-8,"SCALAPACK linear scaling blocksize",dummy_int) !ranges=(/-100,1000/)
    call set(dict // PDGEMM_BLOCKSIZE, dummy_int)

    !max number of process uses for pdsyev/pdsygv, pdgemm
    call input_var("maxproc_pdsyev",4,"SCALAPACK linear scaling max num procs",dummy_int) !ranges=(/1,100000/)
    call set(dict // MAXPROC_PDSYEV, dummy_int)
    call input_var("maxproc_pdgemm",4,"SCALAPACK linear scaling max num procs",dummy_int) !ranges=(/1,100000/)
    call set(dict // MAXPROC_PDGEMM, dummy_int)

    !FOE: if the determinant of the interpolation matrix to find the Fermi energy
    !is smaller than this value, switch from cubic to linear interpolation.
    call input_var("ef_interpol_det",1.d-20,"FOE: max ",dummy_real)
    call set(dict // EF_INTERPOL_DET, dummy_real, fmt = "(E9.2)")
    call input_var("ef_interpol_chargediff",10.d0,"FOE: max ",dummy_real)
    call set(dict // EF_INTERPOL_CHARGEDIFF, dummy_real, fmt = "(E9.2)")

    !determines whether a mixing step shall be preformed after the input guess !(linear version)
    call input_var("mixing_after_inputguess",.true.,"mixing  (T/F)",dummy_bool)
    call set(dict // MIXING_AFTER_INPUTGUESS, dummy_bool)

    !determines whether the input guess support functions are orthogonalized iteratively (T) or in the standard way (F)
    call input_var("iterative_orthogonalization",.false.," orbitals",dummy_bool)
    call set(dict // ITERATIVE_ORTHOGONALIZATION, dummy_bool)

    call input_var("check_sumrho", 2, (/0,1,2/), "linear sumrho: 0=no check, 1=light check, 2=full check", dummy_int)
    call set(dict // CHECK_SUMRHO, dummy_int)

    call input_var("experimental_mode", .false., "linear scaling: activate the experimental mode", dummy_bool)
    call set(dict // EXPERIMENTAL_MODE, dummy_bool)

    call input_var("write_orbitals", .false., "linear scaling: write KS orbitals for cubic restart", dummy_bool)
    call set(dict // WRITE_ORBITALS, dummy_bool)

    call input_var("explicit_locregcenters", .false., "linear scaling: explicitely specify localization centers", dummy_bool)
    call set(dict // EXPLICIT_LOCREGCENTERS, dummy_bool)

    call input_var("calculate_KS_residue", .true., "linear scaling: calculate Kohn-Sham residue", dummy_bool)
    call set(dict // CALCULATE_KS_RESIDUE, dummy_bool)

    call input_var("intermediate_forces", .false., "linear scaling: calculate intermediate forces", dummy_bool)
    call set(dict // INTERMEDIATE_FORCES, dummy_bool)

    call input_var("kappa_conv", 0.1d0, "exit kappa for extended input guess (experimental mode)", dummy_real)
    call set(dict // KAPPA_CONV, dummy_real)

    call input_var("evbounds_nsatur", 3, "number of FOE cycles before the eigenvalue bounds are shrinked", dummy_int)
    call set(dict // EVBOUNDS_NSATUR, dummy_int)

    call input_var("evboundsshrink_nsatur", 4, "maximal number of unsuccessful eigenvalue bounds shrinkings", dummy_int)
    call set(dict // EVBOUNDSSHRINK_NSATUR, dummy_int)

    call input_var("method_updatekernel", 0, (/0,1,2/), "K update (sup fun opt) (0: purific., 1: FOE, 2: renorm.)", dummy_int)
    call set(dict // METHOD_UPDATEKERNEL, dummy_int)

    call input_var("purification_quickreturn", .false., "linear scaling: quick return in purification", dummy_bool)
    call set(dict // PURIFICATION_QUICKRETURN, dummy_bool)

    call input_var("adjust_FOE_temperature", .true., "dynamic adjustment of FOE error function decay length", dummy_bool)
    call set(dict // ADJUST_FOE_TEMPERATURE, dummy_bool)

    call input_var("calculate_gap", .false., "calculate the HOMO LUMO gap", dummy_bool)
    call set(dict // CALCULATE_GAP, dummy_bool)

    call input_var("loewdin_charge_analysis", .false., "perform a Loewdin charge analysis at the end", dummy_bool)
    call set(dict // LOEWDIN_CHARGE_ANALYSIS, dummy_bool)

    call input_var("check_matrix_compression", .true., "perform a check of the matrix compression routines", dummy_bool)
    call set(dict // CHECK_MATRIX_COMPRESSION, dummy_bool)

    call input_var("correction_co_contra", .false., "correction covariant / contravariant gradient", dummy_bool)
    call set(dict // CORRECTION_CO_CONTRA, dummy_bool)

    call input_var("fscale_lowerbound", 5.d-3, "lower bound for the error function decay length", dummy_real)
    call set(dict // FSCALE_LOWERBOUND, dummy_real)

    call input_var("fscale_upperbound", 5.d-2, "upper bound for the error function decay length", dummy_real)
    call set(dict // FSCALE_UPPERBOUND, dummy_real)


    call input_free(.false.)

  END SUBROUTINE read_perf_from_text_format


  !> Read the linear input variables
  subroutine read_lin_and_frag_from_text_format(iproc,dict,run_name)
    use module_base
    use module_input
    use module_input_keys
    implicit none
    character(len=*), intent(in) :: run_name
    type(dictionary), pointer :: dict
    integer, intent(in) :: iproc
    !local variables
    !n(c) character(len=*), parameter :: subname='perf_input_variables'
    logical :: exists, dummy_bool,frag_bool
    integer :: dummy_int,ios
    double precision :: dummy_real
    character(len=256) :: comments,dummy_char,filename
    type(dictionary), pointer :: dict_basis

    filename=repeat(' ',len(filename))
    call set_inputfile(filename, trim(run_name),    "lin")
    ! This name seems to be too long..
    !call input_set_file(iproc,(iproc == 0),trim(filename),exists,'Parameters for Localized basis generation (O(N) approach)')
    call input_set_file(iproc,(iproc == 0),trim(filename),exists,'Parameters for O(N) approach')
!    call input_set_file(iproc, (iproc == 0), filename, exists, LIN_GENERAL)
!    call input_set_file(iproc, (iproc == 0), filename, exists, LIN_BASIS)
!    call input_set_file(iproc, (iproc == 0), filename, exists, LIN_KERNEL)
    !if (exists) in%files = in%files + INPUTS_PERF
    if (.not. exists) then
       call input_free(.false.)
       return
    end if

    if (.not. associated(dict)) call dict_init(dict)

    ! General variables #######################################################

    comments='number of accuracy levels: either 2 (for low/high accuracy) or 1 (for hybrid mode)'
    call input_var(dummy_int,'2',ranges=(/1,2/),comment=comments)
    call dict_set(dict//LIN_GENERAL//HYBRID,dummy_int==1)

    ! number of iterations
    comments = 'outer loop iterations (low, high)'
    call input_var(dummy_int,'15',dict//LIN_GENERAL//NIT//0,ranges=(/0,100000/))
    call input_var(dummy_int,'1',dict//LIN_GENERAL//NIT//1,ranges=(/0,100000/),comment=comments)

    comments = 'basis iterations (low, high)'
    call input_var(dummy_int,'12',dict//LIN_BASIS//NIT//0,ranges=(/0,100000/))
    call input_var(dummy_int,'50',dict//LIN_BASIS//NIT//1,ranges=(/0,100000/),comment=comments)

    comments = 'kernel iterations (low, high) - directmin only'
    call input_var(dummy_int,'1',dict//LIN_KERNEL//NSTEP//0,ranges=(/0,1000/))
    call input_var(dummy_int,'1',dict//LIN_KERNEL//NSTEP//1,ranges=(/0,1000/),comment=comments)

    comments = 'density iterations (low, high)'
    call input_var(dummy_int,'15',dict//LIN_KERNEL//NIT//0,ranges=(/0,1000/))
    call input_var(dummy_int,'15',dict//LIN_KERNEL//NIT//1,ranges=(/0,1000/),comment=comments)

    ! DIIS history lengths
    comments = 'DIIS history for basis (low, high)'
    call input_var(dummy_int,'5',dict//LIN_BASIS//IDSX//0,ranges=(/0,100/))
    call input_var(dummy_int,'0',dict//LIN_BASIS//IDSX//1,ranges=(/0,100/),comment=comments)

    comments = 'DIIS history for kernel (low, high) - directmin only'
    call input_var(dummy_int,'0',dict//LIN_KERNEL//IDSX_COEFF//0,ranges=(/0,100/))
    call input_var(dummy_int,'0',dict//LIN_KERNEL//IDSX_COEFF//1,ranges=(/0,100/),comment=comments)

    comments = 'DIIS history for density mixing (low, high)'
    call input_var(dummy_int,'0',dict//LIN_KERNEL//IDSX//0,ranges=(/0,100/))
    call input_var(dummy_int,'0',dict//LIN_KERNEL//IDSX//1,ranges=(/0,100/),comment=comments)

    ! mixing parameters
    comments = 'density mixing parameter (low, high)'
    call input_var(dummy_real,'.5d0',dict//LIN_KERNEL//ALPHAMIX//0,ranges=(/0.d0,1.d0/))
    call input_var(dummy_real,'.5d0',dict//LIN_KERNEL//ALPHAMIX//1,ranges=(/0.d0,1.d0/),comment=comments)

    ! Convergence criteria
    comments = 'outer loop convergence (low, high)'
    call input_var(dummy_real,'1.d-8' ,dict//LIN_GENERAL//RPNRM_CV//0,ranges=(/0.d0,1.d0/))
    call input_var(dummy_real,'1.d-12',dict//LIN_GENERAL//RPNRM_CV//1,ranges=(/0.d0,1.d0/),comment=comments)

    comments = 'basis convergence (low, high) ; early stop TMB optimization, dynamic gnrm, activate dyn (exp. mode only)'
    call input_var(dummy_real,'1.d-3',dict//LIN_BASIS//GNRM_CV//0,ranges=(/0.0_gp,1.0_gp/))
    call input_var(dummy_real,'1.d-5',dict//LIN_BASIS//GNRM_CV//1,ranges=(/0.0_gp,1.0_gp/))
    call input_var(dummy_real,'1.d-4',dict//LIN_BASIS//DELTAE_CV,ranges=(/0.0_gp,1.0_gp/))
    call input_var(dummy_real,'1.d-4',dict//LIN_BASIS//GNRM_DYN,ranges=(/0.0_gp,1.0_gp/))
    call input_var(dummy_real,'1.d-3',dict//LIN_BASIS//MIN_GNRM_FOR_DYNAMIC,ranges=(/1.d-7,1.0_gp/),comment=comments)

    comments = 'factor to reduce the confinement. Only used for hybrid mode.'
    call input_var(dummy_real,'0.5d0',dict//LIN_GENERAL//CONF_DAMPING,ranges=(/-1.d100,1.d0/),comment=comments)

    comments = 'kernel convergence (low, high) - directmin only'
    call input_var(dummy_real,'75.d-5',dict//LIN_KERNEL//GNRM_CV_COEFF//0,ranges=(/0.d0,1.d0/))
    call input_var(dummy_real,'1.d-5',dict//LIN_KERNEL//GNRM_CV_COEFF//1,ranges=(/0.d0,1.d0/),comment=comments)

    comments = 'density convergence (low, high)'
    call input_var(dummy_real,'1.d-13',dict//LIN_KERNEL//RPNRM_CV//0,ranges=(/0.d0,1.d0/))
    call input_var(dummy_real,'1.d-13',dict//LIN_KERNEL//RPNRM_CV//1,ranges=(/0.d0,1.d0/),comment=comments)

    comments = 'convergence criterion on density to fix TMBS'
    call input_var(dummy_real,'1.d-10',dict//LIN_BASIS//FIX_BASIS,ranges=(/1.d-14,1.d-6/),comment=comments)
    !call input_var(in%lin%support_functions_converged,'1.d-10',ranges=(/0.d0,1.d0/),comment=comments)

    comments='mixing method: 100 (direct minimization), 101 (simple dens mixing), 102 (simple pot mixing), 103 (FOE)'
    call input_var(dummy_int,'100',ranges=(/100,103/),comment=comments)
    select case(dummy_int)
    case(100)
       call dict_set(dict//LIN_KERNEL//LINEAR_METHOD,'DIRMIN')
    case(101) 
       call dict_set(dict//LIN_KERNEL//LINEAR_METHOD,'DIAG')
       call dict_set(dict//LIN_KERNEL//MIXING_METHOD,'DEN')
    case(102)      
       call dict_set(dict//LIN_KERNEL//LINEAR_METHOD,'DIAG')
       call dict_set(dict//LIN_KERNEL//MIXING_METHOD,'POT')
    case(103)
       call dict_set(dict//LIN_KERNEL//LINEAR_METHOD,'FOE')
    end select

    comments = 'initial step size for basis optimization (DIIS, SD)' ! DELETE ONE
    call input_var(dummy_real,'1.d0',dict//LIN_BASIS//ALPHA_DIIS,ranges=(/0.0_gp,10.0_gp/))
    call input_var(dummy_real,'1.d0',dict//LIN_BASIS//ALPHA_SD,ranges=(/0.0_gp,10.0_gp/),comment=comments)

    comments = 'initial step size for kernel update (SD), curve fitting for alpha update - directmin only'
    call input_var(dummy_real,'1.d0',dict//LIN_KERNEL//ALPHA_SD_COEFF,ranges=(/0.0_gp,10.0_gp/))
    call input_var(dummy_bool,'F',dict//LIN_KERNEL//ALPHA_FIT_COEFF,comment=comments)

    comments = 'lower and upper bound for the eigenvalue spectrum (FOE). Will be adjusted automatically if chosen too small'
    call input_var(dummy_real,'-.5d0',dict//LIN_KERNEL//EVAL_RANGE_FOE//0,ranges=(/-10.d0,-1.d-10/))
    call input_var(dummy_real,'-.5d0',dict//LIN_KERNEL//EVAL_RANGE_FOE//1,ranges=(/1.d-10,10.d0/),comment=comments)

    !comments='number of iterations in the preconditioner, order of Taylor approximations'
    comments='number of iterations in the preconditioner'
    call input_var(dummy_int,'5',dict//LIN_BASIS//NSTEP_PREC,ranges=(/1,100/),comment=comments)
    !!call input_var(dummy_int,'1',dict//LIN_GENERAL//TAYLOR_ORDER,ranges=(/-100,100/),comment=comments)
    !call input_var(in%lin%order_taylor,'1',ranges=(/1,100/),comment=comments)

    comments = '0-> exact Loewdin, 1-> taylor expansion; &
               &in orthoconstraint: correction for non-orthogonality (0) or no correction (1)'
    call input_var(dummy_int,'1',dict//LIN_GENERAL//TAYLOR_ORDER,ranges=(/-100,100/))
    call input_var(dummy_int,'1',dict//LIN_BASIS//CORRECTION_ORTHOCONSTRAINT,comment=comments)
    !call input_var(in%lin%correctionOrthoconstraint,'1',ranges=(/0,1/),comment=comments)

    comments='fscale: length scale over which complementary error function decays from 1 to 0'
    call input_var(dummy_real,'1.d-2',dict//LIN_KERNEL//FSCALE_FOE,ranges=(/0.d0,1.d0/),comment=comments)

    !plot basis functions: true or false
    comments='Output basis functions: 0 no output, 1 formatted output, 2 Fortran bin, 3 ETSF ;'//&
             'calculate dipole ; pulay correction (old and new); diagonalization at the end (dmin, FOE)'
    call input_var(dummy_int,'0',dict//LIN_GENERAL//OUTPUT_WF,ranges=(/0,3/))
    call input_var(dummy_bool,'F',dict//LIN_GENERAL//CALC_DIPOLE)
    call input_var(dummy_bool,'T',dict//LIN_GENERAL//CALC_PULAY//0)
    call input_var(dummy_bool,'F',dict//LIN_GENERAL//CALC_PULAY//1)

!    in%lin%pulay_correction=dummy_bool
!    call input_var(in%lin%new_pulay_correction,'F')
    call input_var(dummy_bool,'F',dict//LIN_GENERAL//SUBSPACE_DIAG,comment=comments)

  !fragment calculation and transfer integrals: true or false
  comments='fragment calculation; calculate transfer_integrals; constrained DFT calculation; extra states to optimize (dmin only)'
  !these should becode dummy variables to build dictionary
  !!call input_var(in%lin%fragment_calculation,'F')
  !!call input_var(in%lin%calc_transfer_integrals,'F')
  !!call input_var(in%lin%constrained_dft,'F')
  !!call input_var(in%lin%extra_states,'0',ranges=(/0,10000/),comment=comments)
  call input_var(frag_bool,'F')
  !this variable makes sense only if fragments are specified
  call input_var(dummy_bool,'F')
  if (frag_bool) call dict_set(dict//FRAG_VARIABLES//TRANSFER_INTEGRALS,dummy_bool)
  
  call input_var(dummy_bool,'F') !constrained DFT, obtained via the charges
  call input_var(dummy_int,'0',dict//LIN_GENERAL//EXTRA_STATES,&
       ranges=(/0,10000/),comment=comments)

  ! Now read in the parameters specific for each atom type.
  comments = 'Atom name, number of basis functions per atom, prefactor for confinement potential,'//&
       'localization radius, kernel cutoff, kernel cutoff FOE'
  read_basis: do !while(itype <= atoms%astruct%ntypes) 
  !!   if (exists) then
        call input_var(dummy_char,'C',input_iostat=ios)
        if (ios /= 0) exit read_basis
     dict_basis=>dict//LIN_BASIS_PARAMS//trim(dummy_char)
     call input_var(dummy_int,'1',dict_basis//NBASIS,ranges=(/1,100/))
     call input_var(dummy_real,'1.2d-2',dict_basis//AO_CONFINEMENT,&
          ranges=(/0.0_gp,1.0_gp/))
     call input_var(dummy_real,'1.2d-2',dict_basis//CONFINEMENT//0,&
          ranges=(/0.0_gp,1.0_gp/))
     call input_var(dummy_real,'5.d-5',dict_basis//CONFINEMENT//1,&
          ranges=(/0.0_gp,1.0_gp/))
     call input_var(dummy_real,'10.d0',dict_basis//RLOC//0,&
          ranges=(/1.0_gp,10000.0_gp/))
     call input_var(dummy_real,'10.d0',dict_basis//RLOC//1,&
          ranges=(/1.0_gp,10000.0_gp/))
     call input_var(dummy_real,'12.d0',dict_basis//RLOC_KERNEL,&
          ranges=(/1.0_gp,10000.0_gp/))
     call input_var(dummy_real,'20.d0',dict_basis//RLOC_KERNEL_FOE,&
          ranges=(/1.0_gp,10000.0_gp/),comment=comments)

  !!   if (.not. exists) exit read_basis !default has been filled
  end do read_basis

    call input_free(.false.)

    !read extensively the file and build the temporary variable
    !from which the input dictionary is updated
    if (frag_bool) then
       filename=repeat(' ',len(filename))
       call set_inputfile(filename, run_name,   "frag")
       call fragment_input_variables_from_text_format(iproc,.false.,&
            trim(filename),frag_bool,dict//FRAG_VARIABLES)
    end if

  END SUBROUTINE read_lin_and_frag_from_text_format


  subroutine read_neb_from_text_format(iproc,dict,filename)
    use module_base
    use module_input
    use module_input_keys
    use dictionaries
    implicit none
    character(len=*), intent(in) :: filename
    type(dictionary), pointer :: dict
    integer, intent(in) :: iproc

    INTEGER :: num_of_images
    CHARACTER (LEN=20) :: minimization_scheme
    logical :: climbing, optimization, restart, exists
    integer :: max_iterations
    real(gp) :: convergence, damp, k_min, k_max, ds, temp_req, tolerance
    CHARACTER (LEN=80) :: first_config, last_config, job_name, scratch_dir

    NAMELIST /NEB/ scratch_dir,         &
         climbing,            &
         optimization,        &
         minimization_scheme, &
         damp,                &
         temp_req,            &
         k_max, k_min,        &
         ds,                  &
         max_iterations,      &
         tolerance,           &
         convergence,         &
         num_of_images,       &
         restart,             & ! not used
         job_name,            & ! not used
         first_config,        & ! not used
         last_config            ! not used

    inquire(file=trim(filename),exist=exists)
    if (.not. exists) return

    open(unit = 123, file = trim(filename), action = "read")
    READ(123 , NML=NEB )
    close(123)

    call set(dict // GEOPT_METHOD, "NEB")
    call set(dict // NEB_CLIMBING, climbing)
    call set(dict // EXTREMA_OPT, optimization)
    call set(dict // NEB_METHOD, minimization_scheme)
    if (trim(minimization_scheme) == 'damped-verlet') call set(dict // NEB_DAMP, damp)
    call set(dict // SPRINGS_K // 0, k_min)
    call set(dict // SPRINGS_K // 1, k_max)
    if (trim(minimization_scheme) == 'sim-annealing') call set(dict // TEMP, temp_req)
    call set(dict // BETAX, ds)
    call set(dict // NCOUNT_CLUSTER_X, max_iterations)
    call set(dict // FIX_TOL, tolerance)
    call set(dict // FORCEMAX, convergence)
    call set(dict // NIMG, num_of_images)

  end subroutine read_neb_from_text_format


  !> Read fragment input parameters
  subroutine fragment_input_variables_from_text_format(iproc,dump,filename,shouldexist,dict)
    use module_base
    use module_types
    use module_input
    use module_input_keys
    use yaml_output, only: yaml_toa,yaml_map
    implicit none
    logical, intent(in) :: shouldexist
    integer, intent(in) :: iproc
    character(len=*), intent(in) :: filename
    type(dictionary), pointer :: dict
    logical, intent(in) :: dump
    !local variables
    !character(len=*), parameter :: subname='fragment_input_variables'
    logical :: exists
    character(len=256) :: comments
    integer :: ifrag, frag_num
    real(gp) :: charge
    type(fragmentInputParameters) :: frag
    type(dictionary), pointer :: dict_frag

    !Linear input parameters
    call input_set_file(iproc,dump,trim(filename),exists,'Fragment Parameters') 

    if (.not. exists .and. shouldexist) then ! we should be doing a fragment calculation, so this is a problem
       call f_err_throw("The file 'input.frag' is missing and fragment calculation was specified",&
            err_name='BIGDFT_INPUT_VARIABLES_ERROR')
       call input_free(.false.)
       return
    end if

    call nullifyInputFragParameters(frag)

    !example of interpreted fragment yaml file
    !!frag:
    !!  transfer_integrals: Yes
    !!  frag_name1: [1, ... , 3, 5, ... , 8]
    !!  frag_name2: [9, 10, 13, 16, ... ,18]
    !!  frag_name3: [11, 12, 15]
    !!  constrained_dft:
    !!    Fragment No. 9: +1
    !!    Fragment No. 15: -1
    !!

    ! number of reference fragments
    comments='# number of fragments in reference system, number of fragments in current system'
    call input_var(frag%nfrag_ref,'1',ranges=(/1,100000/))
    call input_var(frag%nfrag,'1',ranges=(/1,100000/),comment=comments)

    ! Allocate fragment pointers
    call allocateInputFragArrays(frag)

    ! ADD A SENSIBLE DEFAULT AND ALLOW FOR USER NOT TO SPECIFY FRAGMENT NAMES
    comments = '#  reference fragment number i, fragment label'
    do ifrag=1,frag%nfrag_ref
       call input_var(frag_num,'1',ranges=(/1,frag%nfrag_ref/))
       if (frag_num/=ifrag) then
          call f_err_throw("The file 'input.frag'  has an error when specifying"//&
               " the reference fragments",err_name='BIGDFT_INPUT_VARIABLES_ERROR')
       end if
       call input_var(frag%label(frag_num),' ',comment=comments)
       frag%label(frag_num)=trim(frag%label(frag_num))
       ! keep dirname blank if this isn't a fragment calculation
       if (len(trim(frag%label(frag_num)))>=1) then
          frag%dirname(frag_num)='data-'//trim(frag%label(frag_num))//'/'
       else
          frag%dirname(frag_num)=''
       end if
    end do

    comments = '# fragment number j, reference fragment i this corresponds to, charge on this fragment'
    do ifrag=1,frag%nfrag
       call input_var(frag_num,'1',ranges=(/1,frag%nfrag/))
       if (frag_num/=ifrag) then
          call f_err_throw("The file 'input.frag'  has an error when specifying"//&
               " the system fragments",err_name='BIGDFT_INPUT_VARIABLES_ERROR')
       end if
       call input_var(frag%frag_index(frag_num),'1',ranges=(/0,100000/))
       call input_var(charge,'0.d0',ranges=(/-500.d0,500.d0/),comment=comments)
       frag%charge(frag_num)=charge
       !call input_var(frag%charge(frag_num),'1',ranges=(/-500,500/),comment=comments)
    end do

    call input_free(.false.)

    call dict_from_frag(frag,dict_frag)

    !call yaml_map('Fragment dictionary',dict_frag)

    call dict_update(dict,dict_frag)
    call dict_free(dict_frag)
    call deallocateInputFragArrays(frag)

  END SUBROUTINE fragment_input_variables_from_text_format

  !> routine to build dictionary of fragment for purposes of backward compatibility with the old format
  subroutine dict_from_frag(frag,dict_frag)
    use module_base
    use module_types, only: fragmentInputParameters
    use yaml_output, only: yaml_toa
    use module_input_keys
    implicit none
    type(fragmentInputParameters), intent(in) :: frag
    type(dictionary), pointer :: dict_frag
    !local variables
    integer :: ifrag
    type(dictionary), pointer :: frag_list

    !create a dictionary with the given information
    call dict_init(dict_frag)
    do ifrag=1,frag%nfrag_ref
       !build the list of fragments associated to the reference
       call build_frag_list(ifrag,frag%nfrag,frag%frag_index,frag_list)
       call dict_set(dict_frag//trim(frag%label(ifrag)),frag_list)
    end do
    !then set the constrained DFT elements in case there is a charge
    do ifrag=1,frag%nfrag
       if (frag%charge(ifrag) /= 0.d0) &
            call dict_set(dict_frag//CONSTRAINED_DFT//(FRAGMENT_NO//&
            trim(adjustl(yaml_toa(ifrag)))),frag%charge(ifrag),fmt='(f7.2)')
    end do

  contains

    subroutine build_frag_list(ifrag_ref,nfrag,frag_index,frag_list)
      implicit none
      integer, intent(in) :: nfrag,ifrag_ref
      integer, dimension(nfrag), intent(in) :: frag_index
      type(dictionary), pointer :: frag_list
      !local variables
      logical :: agree,willagree
      integer :: ifrag,istart,iend

      call dict_init(frag_list)

      !for each segment find the starting and ending point 
      ifrag=1
      istart=0
      iend=0
      find_segment: do while(ifrag <= nfrag)
         agree=frag_index(ifrag)== ifrag_ref
         willagree= ifrag < nfrag
         if (willagree) willagree=frag_index(ifrag+1) == ifrag_ref
         if (agree) then
            !starting point if it has not been found yet
            if (istart==0) then
               istart=ifrag
            end if
            if (willagree) then
               iend=0
            else
               iend=ifrag
            end if
         end if
         if (istart*iend /= 0) then
            if (istart==iend) then
               call add(frag_list,istart)
            else if (iend == istart+1) then
               call add(frag_list,istart)
               call add(frag_list,iend)
            else
               call add(frag_list,istart)
               call add(frag_list,'...')
               call add(frag_list,iend)
            end if
            istart=0
            iend=0
         end if
         ifrag=ifrag+1
      end do find_segment
    end subroutine build_frag_list
  end subroutine dict_from_frag


end module input_old_text_format
