!> @file
!!   Contains filter for Daubechies wavelets
!! @author
!!    Copyright (C) 2011-2013 BigDFT group
!!    This file is distributed under the terms of the
!!    GNU General Public License, see ~/COPYING file
!!    or http://www.gnu.org/copyleft/gpl.txt .
!!    For the list of contributors, see ~/AUTHORS




!> @brief Module which contains various filters for the Daubechies 16 wavelet family.
!! @details Definition of the filters:
!!     @f$ a_i=<\phi|Op|\phi_i> @f$ @n
!!     @f$ b_i=<\psi|Op|\phi_i> @f$ @n
!!     @f$ c_i=<\phi|Op|\psi_i> @f$ @n
!!     @f$ e_i=<\psi|Op|\psi_i> @f$ 
!!
!!   There are five different operators:
!!     Op = second derivative -> a, b, c, e        
!!     Op = first derivative -> ad1, bd1, cd1, ed1
!!     Op = @f$ x   @f$  -> a1, b1, c1, e1 @n
!!     Op = @f$ x^2 @f$  -> a2, b2, c2, e2 @n
!!     Op = @f$ x^3 @f$  -> a3, b3, c3, e3 @n
!!     Op = @f$ x^4 @f$  -> a4, b4, c4, e4 @n
!!     Op = @f$ x^5 @f$  -> a5, b5, c5, e5 @n
!!     Op = @f$ x^6 @f$  -> a6, b6, c6, e6
module filterModule

implicit none

integer,parameter:: lb=-14 !< The lower bound for the filters.
integer,parameter:: ub=14  !< The upper bound for the filters.

real(8),dimension(lb:ub),parameter:: a = [ -6.92447490505951028E-18, &
                                            2.70800498995346639E-13, &
                                           -5.81387993303650319E-11, &
                                           -1.05857056453828591E-08, &
                                           -3.72307624729728559E-07, &
                                            2.09042354981647804E-06, &
                                           -2.39822857110993937E-05, &
                                            4.51679195975884795E-04, &
                                           -4.09765681251883507E-03, &
                                            2.20702923834323883E-02, &
                                           -8.22663977742195129E-02, &
                                            2.37178057432174683E-01, &
                                           -6.15614175796508789E-01, &
                                            2.21914649009704590E+00, &
                                           -3.55369234085083008E+00, &
                                            2.21914649009704590E+00, &
                                           -6.15614175796508789E-01, &
                                            2.37178057432174683E-01, &
                                           -8.22663977742195129E-02, &
                                            2.20702923834323883E-02, &
                                           -4.09765681251883507E-03, &
                                            4.51679195975884795E-04, &
                                           -2.39822857110993937E-05, &
                                            2.09042354981647804E-06, &
                                           -3.72307624729728559E-07, &
                                           -1.05857056453828591E-08, &
                                           -5.81387993303650319E-11, &
                                            2.70800498995346639E-13, &
                                           -6.92447490505951028E-18  ]

real(8),dimension(lb:ub),parameter:: b = [  1.23926298748915144E-17, &
                                           -4.84665694980683149E-13, &
                                            1.04757348540220618E-10, &
                                            1.87909231522041173E-08, &
                                            6.39191314318015945E-07, &
                                           -4.62561549613067677E-06, &
                                            5.07580797039626444E-05, &
                                           -9.34236181144083580E-04, &
                                            8.08720235045630956E-03, &
                                           -5.65773658045679495E-02, &
                                            2.33704891436108609E-01, &
                                           -5.32982207596253462E-01, &
                                            6.34570315769713792E-01, &
                                           -2.17161162613989146E-01, &
                                           -4.20816528718038670E-01, &
                                            6.57030701554058305E-01, &
                                           -4.24980483446119562E-01, &
                                            1.43273284054659191E-01, &
                                           -2.84696145431842073E-02, &
                                            7.13761173990859820E-03, &
                                           -2.13562905732165138E-03, &
                                            2.17107937709352954E-04, &
                                           -1.13345668089136977E-05, &
                                            8.79697176379788384E-07, &
                                           -2.16568315068759430E-07, &
                                           -5.96264927673747788E-09, &
                                           -3.22647031147176524E-11, &
                                            1.51306166977329773E-13, &
                                           -3.86910248985843272E-18  ]

real(8),dimension(lb:ub),parameter:: c = [ -3.86910248985843272E-18, &
                                            1.51306166977329773E-13, &
                                           -3.22647031147176524E-11, &
                                           -5.96264927673747788E-09, &
                                           -2.16568315068759430E-07, &
                                            8.79697176379788384E-07, &
                                           -1.13345668089136977E-05, &
                                            2.17107937709352954E-04, &
                                           -2.13562905732165138E-03, &
                                            7.13761173990859820E-03, &
                                           -2.84696145431842073E-02, &
                                            1.43273284054659191E-01, &
                                           -4.24980483446119562E-01, &
                                            6.57030701554058305E-01, &
                                           -4.20816528718038670E-01, &
                                           -2.17161162613989146E-01, &
                                            6.34570315769713792E-01, &
                                           -5.32982207596253462E-01, &
                                            2.33704891436108609E-01, &
                                           -5.65773658045679495E-02, &
                                            8.08720235045630956E-03, &
                                           -9.34236181144083580E-04, &
                                            5.07580797039626444E-05, &
                                           -4.62561549613067677E-06, &
                                            6.39191314318015945E-07, &
                                            1.87909231522041173E-08, &
                                            1.04757348540220618E-10, &
                                           -4.84665694980683149E-13, &
                                            1.23926298748915144E-17  ]

real(8),dimension(lb:ub),parameter:: e = [  6.92447507792733360E-18, &
                                           -2.70800496076004922E-13, &
                                            5.81387997394137139E-11, &
                                            1.05857053487793953E-08, &
                                            3.72307646478137234E-07, &
                                           -2.09042376230958760E-06, &
                                            2.39822849010384008E-05, &
                                           -4.51679172658164203E-04, &
                                            4.09765614832326392E-03, &
                                           -2.20732688908272731E-02, &
                                            8.20745364532223690E-02, &
                                           -2.69959298589643554E-01, &
                                           -4.25170863143320842E-02, &
                                           -7.14405969106863975E+00, &
                                           -2.48758457220106488E+01, &
                                           -7.14405969106863975E+00, &
                                           -4.25170863143320842E-02, &
                                           -2.69959298589643554E-01, &
                                            8.20745364532223690E-02, &
                                           -2.20732688908272731E-02, &
                                            4.09765614832326392E-03, &
                                           -4.51679172658164203E-04, &
                                            2.39822849010384008E-05, &
                                           -2.09042376230958760E-06, &
                                            3.72307646478137234E-07, &
                                            1.05857053487793953E-08, &
                                            5.81387997394137139E-11, &
                                           -2.70800496076004922E-13, &
                                            6.92447507792733360E-18  ]

real(8),dimension(lb:ub),parameter:: ad1 = [  1.58546470565166012E-19, &
                                             -1.24007851174680848E-14, &
                                              7.25220664771208945E-13, &
                                              9.69718527699114929E-10, &
                                              7.20794801623014791E-08, &
                                             -3.99381043791890988E-08, &
                                             -2.45199203163792845E-07, &
                                             -7.66770681366324425E-05, &
                                              1.03153020609170198E-03, &
                                             -6.95837894454598427E-03, &
                                              3.12901474535465240E-02, &
                                             -1.06364071369171143E-01, &
                                              3.03259342908859253E-01, &
                                             -8.83446037769317627E-01, &
                                              0.00000000000000000E+00, &
                                              8.83446037769317627E-01, &
                                             -3.03259342908859253E-01, &
                                              1.06364071369171143E-01, &
                                             -3.12901474535465240E-02, &
                                              6.95837894454598427E-03, &
                                             -1.03153020609170198E-03, &
                                              7.66770681366324425E-05, &
                                              2.45199203163792845E-07, &
                                              3.99381043791890988E-08, &
                                             -7.20794801623014791E-08, &
                                             -9.69718527699114929E-10, &
                                             -7.25220664771208945E-13, &
                                              1.24007851174680848E-14, &
                                             -1.58546470565166012E-19  ]

real(8),dimension(lb:ub),parameter:: bd1 = [ -2.83748258753182423E-19, &
                                              2.21939156119648851E-14, &
                                             -1.33030039036001623E-12, &
                                             -1.73348991478733391E-09, &
                                             -1.26474450439228448E-07, &
                                              2.51358872709139158E-07, &
                                             -2.19068142292385566E-07, &
                                              1.43657323061649172E-04, &
                                             -1.90861407379624483E-03, &
                                              1.65466900921113713E-02, &
                                             -8.59045126449666543E-02, &
                                              2.72920198139993673E-01, &
                                             -5.60230305427086194E-01, &
                                              7.70607445629750450E-01, &
                                             -7.20284404589714662E-01, &
                                              4.54290602871777051E-01, &
                                             -1.88612304862721248E-01, &
                                              5.07964027800388478E-02, &
                                             -1.04243752554358699E-02, &
                                              2.58094249119395846E-03, &
                                             -5.62777489321193220E-04, &
                                              4.12649516250618737E-05, &
                                              2.62582836392995181E-07, &
                                             -3.49937345599639389E-08, &
                                             -4.10642581951833548E-08, &
                                             -5.42447406290619399E-10, &
                                             -3.95112185219095204E-13, &
                                              6.92890300854295999E-15, &
                                             -8.85890327975718372E-20  ]

real(8),dimension(lb:ub),parameter:: cd1 = [  8.85890327975718372E-20, &
                                             -6.92890300854295999E-15, &
                                              3.95112185219095204E-13, &
                                              5.42447406290619399E-10, &
                                              4.10642581951833548E-08, &
                                              3.49937345599639389E-08, &
                                             -2.62582836392995181E-07, &
                                             -4.12649516250618737E-05, &
                                              5.62777489321193220E-04, &
                                             -2.58094249119395846E-03, &
                                              1.04243752554358699E-02, &
                                             -5.07964027800388478E-02, &
                                              1.88612304862721248E-01, &
                                             -4.54290602871777051E-01, &
                                              7.20284404589714662E-01, &
                                             -7.70607445629750450E-01, &
                                              5.60230305427086194E-01, &
                                             -2.72920198139993673E-01, &
                                              8.59045126449666543E-02, &
                                             -1.65466900921113713E-02, &
                                              1.90861407379624483E-03, &
                                             -1.43657323061649172E-04, &
                                              2.19068142292385566E-07, &
                                             -2.51358872709139158E-07, &
                                              1.26474450439228448E-07, &
                                              1.73348991478733391E-09, &
                                              1.33030039036001623E-12, &
                                             -2.21939156119648851E-14, &
                                              2.83748258753182423E-19  ]

real(8),dimension(lb:ub),parameter:: ed1 = [ -1.58546472054535401E-19, &
                                              1.24007858101253958E-14, &
                                             -7.25220697077032673E-13, &
                                             -9.69718480430558393E-10, &
                                             -7.20794803976435051E-08, &
                                              3.99380288831250122E-08, &
                                              2.45200304496097431E-07, &
                                              7.66770598669571502E-05, &
                                             -1.03153015718146970E-03, &
                                              6.95866720310885251E-03, &
                                             -3.12911278301949776E-02, &
                                              1.10490186629848957E-01, &
                                             -1.78098753563933575E-01, &
                                              2.09648342542417660E+00, &
                                              9.02779661431516806E-36, &
                                             -2.09648342542417660E+00, &
                                              1.78098753563933575E-01, &
                                             -1.10490186629848957E-01, &
                                              3.12911278301949776E-02, &
                                             -6.95866720310885251E-03, &
                                              1.03153015718146970E-03, &
                                             -7.66770598669571502E-05, &
                                             -2.45200304496097431E-07, &
                                             -3.99380288831250122E-08, &
                                              7.20794803976435051E-08, &
                                              9.69718480430558393E-10, &
                                              7.25220697077032673E-13, &
                                             -1.24007858101253958E-14, &
                                              1.58546472054535401E-19  ]

real(8),dimension(lb:ub),parameter:: a1 = [ -4.27782380967095731E-22, &
                                             1.33836790789675412E-16, &
                                            -2.23077388809486687E-13, &
                                            -4.18701080751038290E-11, &
                                             3.84061421554449112E-10, &
                                             6.90677026682351425E-08, &
                                            -1.50667722209618660E-06, &
                                             1.43009565363172442E-05, &
                                            -7.38484450266696513E-05, &
                                             1.17437397420872003E-04, &
                                             4.36133099719882011E-03, &
                                            -2.15712171047925949E-02, &
                                             2.26678475737571716E-02, &
                                             9.18613970279693604E-02, &
                                            -3.43336127698421478E-02, &
                                             9.18613970279693604E-02, &
                                             2.26678475737571716E-02, &
                                            -2.15712171047925949E-02, &
                                             4.36133099719882011E-03, &
                                             1.17437397420872003E-04, &
                                            -7.38484450266696513E-05, &
                                             1.43009565363172442E-05, &
                                            -1.50667722209618660E-06, &
                                             6.90677026682351425E-08, &
                                             3.84061421554449112E-10, &
                                            -4.18701080751038290E-11, &
                                            -2.23077388809486687E-13, &
                                             1.33836790789675412E-16, &
                                            -4.27782380967095731E-22  ]

real(8),dimension(lb:ub),parameter:: b1 = [  7.65595806716108416E-22, &
                                            -2.39526829029937495E-16, &
                                             3.99587744885038026E-13, &
                                             7.43506281339207227E-11, &
                                            -7.94750732307692066E-10, &
                                            -1.22260456570109001E-07, &
                                             2.87022347262966861E-06, &
                                            -3.01746671145040958E-05, &
                                             1.87517678388117517E-04, &
                                            -5.06159000963081146E-04, &
                                            -7.78069985307987407E-03, &
                                             5.60208708828413971E-02, &
                                            -1.59595684240634494E-01, &
                                             2.45149541954812523E-01, &
                                            -2.14819128613303770E-01, &
                                             9.90323037445533675E-02, &
                                            -1.30155486978821159E-02, &
                                            -6.91564874276195792E-03, &
                                             2.28647192633767860E-03, &
                                             3.94195663735359520E-06, &
                                            -2.62413561954799163E-05, &
                                             6.63758559364207982E-06, &
                                            -7.86922176158694678E-07, &
                                             3.89648346451391969E-08, &
                                             1.80799149218002746E-10, &
                                            -2.35773089429422988E-11, &
                                            -1.24537043995966775E-13, &
                                             7.47819644251633595E-17, &
                                            -2.39026636952344337E-22  ]

real(8),dimension(lb:ub),parameter:: c1 = [ -2.39026636952344337E-22, &
                                             7.47819644251633595E-17, &
                                            -1.24537043995966775E-13, &
                                            -2.35773089429422988E-11, &
                                             1.80799149218002746E-10, &
                                             3.89648346451391969E-08, &
                                            -7.86922176158694678E-07, &
                                             6.63758559364207982E-06, &
                                            -2.62413561954799146E-05, &
                                             3.94195663735364602E-06, &
                                             2.28647192633767859E-03, &
                                            -6.91564874276195653E-03, &
                                            -1.30155486978821149E-02, &
                                             9.90323037445533658E-02, &
                                            -2.14819128613303770E-01, &
                                             2.45149541954812528E-01, &
                                            -1.59595684240634471E-01, &
                                             5.60208708828414089E-02, &
                                            -7.78069985307987367E-03, &
                                            -5.06159000963081133E-04, &
                                             1.87517678388117524E-04, &
                                            -3.01746671145040958E-05, &
                                             2.87022347262966861E-06, &
                                            -1.22260456570109001E-07, &
                                            -7.94750732307692066E-10, &
                                             7.43506281339207227E-11, &
                                             3.99587744885038026E-13, &
                                            -2.39526829029937495E-16, &
                                             7.65595806716108416E-22  ]

real(8),dimension(lb:ub),parameter:: e1 = [  4.27782410746594903E-22, &
                                            -1.33836791682177454E-16, &
                                             2.23077373341138633E-13, &
                                             4.18701078507265792E-11, &
                                            -3.84061437780811023E-10, &
                                            -6.90677057885295201E-08, &
                                             1.50667719157836429E-06, &
                                            -1.43009565685797134E-05, &
                                             7.38484438588798723E-05, &
                                            -1.17437022042463083E-04, &
                                            -4.36283743232178177E-03, &
                                             2.14973685507261842E-02, &
                                            -1.83065170245152523E-02, &
                                            -6.91935470160085394E-02, &
                                            -1.11786975230691315E-09, &
                                            -6.91935470160085264E-02, &
                                            -1.83065170245152626E-02, &
                                             2.14973685507261885E-02, &
                                            -4.36283743232178238E-03, &
                                            -1.17437022042463063E-04, &
                                             7.38484438588798689E-05, &
                                            -1.43009565685797134E-05, &
                                             1.50667719157836429E-06, &
                                            -6.90677057885295201E-08, &
                                            -3.84061437780811023E-10, &
                                             4.18701078507265792E-11, &
                                             2.23077373341138633E-13, &
                                            -1.33836791682177454E-16, &
                                             4.27782410746594903E-22  ]

real(8),dimension(lb:ub),parameter:: a2 = [  5.57613113797079647E-21, &
                                            -1.61540205057621221E-15, &
                                             2.44941827237665777E-12, &
                                             4.24629970074974494E-10, &
                                            -3.50622375577813727E-09, &
                                            -5.49912783753825352E-07, &
                                             1.07758069134433754E-05, &
                                            -8.87797723407857120E-05, &
                                             3.82019672542810440E-04, &
                                            -4.44838660769164562E-04, &
                                            -1.42199024558067322E-02, &
                                             3.99360917508602142E-02, &
                                             3.36528867483139038E-02, &
                                            -2.19314932823181152E-01, &
                                             1.65585339069366455E-01, &
                                            -3.55921462178230286E-02, &
                                             1.24324277043342590E-01, &
                                            -8.94912108778953552E-02, &
                                             2.06707436591386795E-02, &
                                             7.29535357095301151E-04, &
                                            -5.04161638673394918E-04, &
                                             1.11433619167655706E-04, &
                                            -1.33310277306009084E-05, &
                                             6.93305878485261928E-07, &
                                             4.17500478633314742E-09, &
                                            -4.96512386760628033E-10, &
                                            -2.90443884221058823E-12, &
                                             1.86435445701578929E-15, &
                                            -6.40177613495305920E-21  ]

real(8),dimension(lb:ub),parameter:: b2 = [ -1.07451146172287577E-20, &
                                             3.13060246628943970E-15, &
                                            -4.78749340108647095E-12, &
                                            -8.27894411800580670E-10, &
                                             8.15738084296199976E-09, &
                                             1.09377804413390625E-06, &
                                            -2.35274921407857363E-05, &
                                             2.22483374802824865E-04, &
                                            -1.23566737058074802E-03, &
                                             2.81931962789967665E-03, &
                                             3.56570274706771702E-02, &
                                            -1.99870719176705833E-01, &
                                             4.09070789978144124E-01, &
                                            -3.83487854930014397E-01, &
                                             1.22706132164835730E-01, &
                                             4.07716120232467710E-02, &
                                            -1.73643908349863515E-02, &
                                            -1.68596657049951048E-02, &
                                             7.63552926096439174E-03, &
                                             5.38805252160585462E-05, &
                                            -1.33751751720543844E-04, &
                                             4.34601260064193826E-05, &
                                            -6.11241925587226617E-06, &
                                             3.52531960342273632E-07, &
                                             1.75161930985329523E-09, &
                                            -2.56223460315090577E-10, &
                                            -1.49681996733928602E-12, &
                                             9.66934874772265558E-16, &
                                            -3.33801397186598040E-21  ]

real(8),dimension(lb:ub),parameter:: c2 = [  3.11570517857345129E-21, &
                                            -9.02614217267229018E-16, &
                                             1.36753205696331864E-12, &
                                             2.38900027287721328E-10, &
                                            -1.68356448485162769E-09, &
                                            -3.09870229248918553E-07, &
                                             5.69141338437313638E-06, &
                                            -4.28284866174613940E-05, &
                                             1.54903165637988905E-04, &
                                             1.84029190592224884E-05, &
                                            -8.36977423915950192E-03, &
                                             1.77185780796068396E-02, &
                                             2.16822550214745961E-02, &
                                            -5.82606911845323180E-02, &
                                            -9.21129972751010266E-02, &
                                             3.51960771815418264E-01, &
                                            -3.88907631878252118E-01, &
                                             1.92275377336910452E-01, &
                                            -3.43692713216321939E-02, &
                                            -2.74842935720907677E-03, &
                                             1.20206244492311618E-03, &
                                            -2.30136631629999615E-04, &
                                             2.52663068944202945E-05, &
                                            -1.22917063308546998E-06, &
                                            -8.53238440365876605E-09, &
                                             8.82170034527123439E-10, &
                                             5.20220026056775308E-12, &
                                            -3.33662185797648242E-15, &
                                             1.14571636261214844E-20  ]

real(8),dimension(lb:ub),parameter:: e2 = [ -6.00391354600385591E-21, &
                                             1.74924014379757951E-15, &
                                            -2.67288615604453156E-12, &
                                            -4.65847082273536233E-10, &
                                             4.00909115868514137E-09, &
                                             6.17273186723220065E-07, &
                                            -1.24681741129386372E-05, &
                                             1.08808259548054617E-04, &
                                            -5.33848387091264561E-04, &
                                             6.96357808776253802E-04, &
                                             2.17704467254226292E-02, &
                                            -9.27586349639902768E-02, &
                                             1.17357770175067983E-01, &
                                            -1.44609721723709135E-02, &
                                             2.28332297558082387E-01, &
                                            -1.52848064755532954E-01, &
                                             4.41317011994163056E-02, &
                                             3.62255764829988165E-02, &
                                            -1.31322527750372515E-02, &
                                            -4.78012401544689226E-04, &
                                             3.52332937508960380E-04, &
                                            -9.14051322471805596E-05, &
                                             1.16386609504637462E-05, &
                                            -6.25945518694966712E-07, &
                                            -3.67213752393608442E-09, &
                                             4.55295290054135748E-10, &
                                             2.68097082627805916E-12, &
                                            -1.73051640666939972E-15, &
                                             5.97399387029546928E-21  ]

real(8),dimension(lb:ub),parameter:: a3 = [ -5.45235389905989279E-20, &
                                             1.46358789074752665E-14, &
                                            -2.01811241329341584E-11, &
                                            -3.24547011487652526E-09, &
                                             2.42857414178843101E-08, &
                                             3.29380782204680145E-06, &
                                            -5.91745010751765221E-05, &
                                             4.36957285273820162E-04, &
                                            -1.67352124117314816E-03, &
                                             1.98640045709908009E-03, &
                                             4.50586304068565369E-02, &
                                            -6.27721697092056274E-02, &
                                            -1.87771692872047424E-01, &
                                             3.47782939672470093E-01, &
                                            -7.22572430968284607E-02, &
                                            -3.45776788890361786E-02, &
                                             2.86159813404083252E-01, &
                                            -2.85770207643508911E-01, &
                                             8.37636739015579224E-02, &
                                             4.12162579596042633E-03, &
                                            -2.77279899455606937E-03, &
                                             6.74822716973721981E-04, &
                                            -8.98371581570245326E-05, &
                                             5.22961454407777637E-06, &
                                             3.43174519912281539E-08, &
                                            -4.43153069795698684E-09, &
                                            -2.83714943899449068E-11, &
                                             1.94904500389536314E-14, &
                                            -7.18620847350200121E-20  ]

real(8),dimension(lb:ub),parameter:: b3 = [  1.13123436610365484E-19, &
                                            -3.07100166700629325E-14, &
                                             4.30376242080603376E-11, &
                                             6.94162274675861743E-09, &
                                            -6.34036165941134668E-08, &
                                            -7.35379995218091881E-06, &
                                             1.47173214489243831E-04, &
                                            -1.28034683673851311E-03, &
                                             6.59497106305552553E-03, &
                                            -1.36921541340832689E-02, &
                                            -1.46897049595819335E-01, &
                                             6.37478980242085363E-01, &
                                            -9.30556215252272467E-01, &
                                             5.59458823389911942E-01, &
                                            -1.72942680845484709E-01, &
                                             1.62694799746086264E-01, &
                                            -1.03342326611766857E-01, &
                                            -2.08251898321085647E-02, &
                                             2.27089101547702390E-02, &
                                             8.34163098143173926E-04, &
                                            -5.63993287410049933E-04, &
                                             2.23449248084776174E-04, &
                                            -3.63125559271239384E-05, &
                                             2.39828253857973364E-06, &
                                             1.28421664625344509E-08, &
                                            -2.09729277533997787E-09, &
                                            -1.34982620213705742E-11, &
                                             9.38388852480617887E-15, &
                                            -3.49671886957245740E-20  ]

real(8),dimension(lb:ub),parameter:: c3 = [ -3.04654359861049767E-20, &
                                             8.17787588062863309E-15, &
                                            -1.12680455877343732E-11, &
                                            -1.82444868673102661E-09, &
                                             1.18723849727281025E-08, &
                                             1.85434546705993772E-06, &
                                            -3.15762398492331581E-05, &
                                             2.17290763713708732E-04, &
                                            -7.38492447647641646E-04, &
                                             3.46249835019798076E-04, &
                                             2.65636962851717066E-02, &
                                            -2.40461111279288945E-02, &
                                            -1.13057521285956212E-01, &
                                             1.75811609409945607E-01, &
                                            -1.49997829824147128E-01, &
                                             4.88522886262723873E-01, &
                                            -8.54944372023607445E-01, &
                                             5.97603435176744222E-01, &
                                            -1.38204695319924360E-01, &
                                            -1.31073095034589249E-02, &
                                             6.26732305736694859E-03, &
                                            -1.36644597735980215E-03, &
                                             1.69343102660099443E-04, &
                                            -9.28314434210991049E-06, &
                                            -6.93099235960035602E-08, &
                                             7.87787729799378351E-09, &
                                             5.08133777611095321E-11, &
                                            -3.48819103637274518E-14, &
                                             1.28610501418155119E-19  ]

real(8),dimension(lb:ub),parameter:: e3 = [  6.32085703717368256E-20, &
                                            -1.71593847930409723E-14, &
                                             2.40296297678319361E-11, &
                                             3.90306169567011521E-09, &
                                            -3.16398262659731984E-08, &
                                            -4.14682388687797182E-06, &
                                             7.86772014015926421E-05, &
                                            -6.42360316804161536E-04, &
                                             3.03629694962606823E-03, &
                                            -4.23534856061406406E-03, &
                                            -9.01621008293589939E-02, &
                                             3.08186486078434389E-01, &
                                            -3.02648997344974718E-01, &
                                            -1.63291833641518359E-03, &
                                             2.40612326154245674E-02, &
                                            -2.52596470969653632E-01, &
                                             1.81819415713290495E-01, &
                                             5.37877230562302454E-02, &
                                            -3.83329370344230677E-02, &
                                            -2.59775806368091250E-03, &
                                             1.40265791479232433E-03, &
                                            -4.59627480916031660E-04, &
                                             6.87230434889209774E-05, &
                                            -4.26390036678194569E-06, &
                                            -2.65855222668025459E-08, &
                                             3.72895715741983696E-09, &
                                             2.41751537799131406E-11, &
                                            -1.67942725094584736E-14, &
                                             6.25802566107744820E-20  ]

real(8),dimension(lb:ub),parameter:: a4 = [  4.73982297509915827E-19, &
                                            -1.17970778003122223E-13, &
                                             1.47867218469599493E-10, &
                                             2.21538325462233843E-08, &
                                            -1.50909258422871062E-07, &
                                            -1.75753048097249120E-05, &
                                             2.94731522444635630E-04, &
                                            -2.00139172375202179E-03, &
                                             7.15284887701272964E-03, &
                                            -9.75563097745180130E-03, &
                                            -1.39017373323440552E-01, &
                                             3.75488102436065674E-02, &
                                             6.29527032375335693E-01, &
                                            -6.22455954551696777E-01, &
                                             2.39198490977287292E-01, &
                                            -1.79768219590187073E-01, &
                                             6.60393893718719482E-01, &
                                            -8.88859748840332031E-01, &
                                             3.33310693502426147E-01, &
                                             2.19652801752090454E-02, &
                                            -1.43004665151238441E-02, &
                                             3.75307234935462475E-03, &
                                            -5.46617549844086170E-04, &
                                             3.51455855707172304E-05, &
                                             2.53031799957170733E-07, &
                                            -3.52819533588899503E-08, &
                                            -2.46440173823359032E-10, &
                                             1.81234964102133800E-13, &
                                            -7.17145273459679160E-19  ]

real(8),dimension(lb:ub),parameter:: b4 = [ -1.05879107559698705E-18, &
                                             2.67975668991362477E-13, &
                                            -3.44037604990392807E-10, &
                                            -5.19398195117999355E-08, &
                                             4.41615034358723889E-07, &
                                             4.40071230383561964E-05, &
                                            -8.30762190462945196E-04, &
                                             6.77564167923146284E-03, &
                                            -3.32405637216197889E-02, &
                                             6.54637771233801294E-02, &
                                             5.95097451648797403E-01, &
                                            -2.00500735905367301E+00, &
                                             2.09342664719743266E+00, &
                                            -8.21319709469621505E-01, &
                                             1.49823989402031084E-01, &
                                             1.42193968646241480E-01, &
                                            -2.58174510470954888E-01, &
                                            -6.34201291534517743E-03, &
                                             6.66580611749571090E-02, &
                                             6.79364923822744610E-03, &
                                            -2.24574587498619634E-03, &
                                             1.06412843446368771E-03, &
                                            -1.95665138116746604E-04, &
                                             1.45486779922684776E-05, &
                                             8.45902475713471526E-08, &
                                            -1.53245377007307429E-08, &
                                            -1.08248520437343012E-10, &
                                             8.10102415572423511E-14, &
                                            -3.25649385965431673E-19  ]

real(8),dimension(lb:ub),parameter:: c4 = [  2.64841177354852000E-19, &
                                            -6.59168226006457738E-14, &
                                             8.25659985103387980E-11, &
                                             1.24446685547390328E-08, &
                                            -7.49608949097184224E-08, &
                                            -9.88743988180540206E-06, &
                                             1.58701212643154945E-04, &
                                            -1.01979783801389721E-03, &
                                             3.34978597853062363E-03, &
                                            -3.11164570844213000E-03, &
                                            -8.21852160064285005E-02, &
                                             1.90046894767111568E-03, &
                                             3.03170077824383044E-01, &
                                            -1.71554362097906985E-01, &
                                            -1.19411738947435464E-01, &
                                             6.67866005617359376E-01, &
                                            -1.84671114749237439E+00, &
                                             1.83677986289889223E+00, &
                                            -5.52785700365708630E-01, &
                                            -6.09059165980459183E-02, &
                                             3.09751752376641818E-02, &
                                            -7.46637518078680060E-03, &
                                             1.02466322093907858E-03, &
                                            -6.24487003129975085E-05, &
                                            -5.05322701667792061E-07, &
                                             6.27526586982342632E-08, &
                                             4.41347806410255367E-10, &
                                            -3.24354727486844695E-13, &
                                             1.28346416737257737E-18  ]

real(8),dimension(lb:ub),parameter:: e4 = [ -5.91607470707858413E-19, &
                                             1.49732874622216788E-13, &
                                            -1.92100738129978935E-10, &
                                            -2.91841511215122459E-08, &
                                             2.23394316572768974E-07, &
                                             2.48001476795504746E-05, &
                                            -4.47594374637332827E-04, &
                                             3.47128223750824498E-03, &
                                            -1.60161771714504936E-02, &
                                             2.38282684039843004E-02, &
                                             3.61599000680735382E-01, &
                                            -9.82482562143621237E-01, &
                                             7.52753810635772963E-01, &
                                            -1.14204696883252737E-01, &
                                             2.51501726273655956E-01, &
                                            -4.84276384425144080E-01, &
                                             5.62339757401709483E-01, &
                                             2.85047897727701947E-02, &
                                            -1.07918110020031040E-01, &
                                            -1.51435426900326290E-02, &
                                             5.34875354394726576E-03, &
                                            -2.14609073676606404E-03, &
                                             3.67972099703273657E-04, &
                                            -2.58921736961675653E-05, &
                                            -1.72989790542752860E-07, &
                                             2.72620366731214978E-08, &
                                             1.93858659954877273E-10, &
                                            -1.44983357227351254E-13, &
                                             5.82809854008927484E-19  ]

real(8),dimension(lb:ub),parameter:: a5 = [ -3.86358941484839379E-18, &
                                             8.92214362287202523E-13, &
                                            -1.01612784852989080E-09, &
                                            -1.42429854577130754E-07, &
                                             8.85454539911734173E-07, &
                                             8.80204461282119155E-05, &
                                            -1.39947456773370504E-03, &
                                             8.92222207039594650E-03, &
                                            -3.07686757296323776E-02, &
                                             4.86353039741516113E-02, &
                                             4.20306175947189331E-01, &
                                             2.88643628358840942E-01, &
                                            -1.81258690357208252E+00, &
                                             1.09223806858062744E+00, &
                                            -1.62169530987739563E-01, &
                                            -2.76054739952087402E-01, &
                                             1.47747445106506348E+00, &
                                            -2.75121784210205078E+00, &
                                             1.33110487461090088E+00, &
                                             1.12288206815719604E-01, &
                                            -7.20262676477432251E-02, &
                                             2.01509576290845871E-02, &
                                            -3.16651212051510811E-03, &
                                             2.22017857595346868E-04, &
                                             1.76656590156198945E-06, &
                                            -2.64264343741160701E-07, &
                                            -2.00762828583833652E-09, &
                                             1.58092939780973829E-12, &
                                            -6.71036706171597832E-18  ]

real(8),dimension(lb:ub),parameter:: b5 = [  9.29196968623255279E-18, &
                                            -2.19379535850976880E-12, &
                                             2.57923822182327832E-09, &
                                             3.65750580071213916E-07, &
                                            -2.90326042613796955E-06, &
                                            -2.47018794669771626E-04, &
                                             4.45264311080060912E-03, &
                                            -3.45910916424318560E-02, &
                                             1.64624847536219517E-01, &
                                            -3.15050164306190143E-01, &
                                            -2.40914605904118140E+00, &
                                             6.32491858596764921E+00, &
                                            -4.79473286487254831E+00, &
                                             1.41384652914718730E+00, &
                                            -4.73090756055158115E-01, &
                                             5.13713993337550274E-01, &
                                            -7.64784124039140572E-01, &
                                             1.36660807172358246E-01, &
                                             1.93459355162475246E-01, &
                                             4.46439858706027114E-02, &
                                            -8.67457807144757844E-03, &
                                             4.92344308530395023E-03, &
                                            -1.00846282411316514E-03, &
                                             8.30415685764317400E-05, &
                                             5.28851130487360886E-07, &
                                            -1.05416616842353047E-07, &
                                            -8.14211505165814048E-10, &
                                             6.56133379854319323E-13, &
                                            -2.84367931300219267E-18  ]

real(8),dimension(lb:ub),parameter:: c5 = [ -2.15880963265054293E-18, &
                                             4.98529857693309817E-13, &
                                            -5.67415869760897571E-10, &
                                            -7.99551543709831178E-08, &
                                             4.46025316335564118E-07, &
                                             4.94893477842650041E-05, &
                                            -7.59418854766930668E-04, &
                                             4.63673084223971601E-03, &
                                            -1.50574635388695922E-02, &
                                             1.96885189352911015E-02, &
                                             2.50620075948119045E-01, &
                                             1.97971721608719982E-01, &
                                            -8.97085523007260830E-01, &
                                             5.44949148586890800E-01, &
                                            -4.44635797887783874E-01, &
                                             1.10440490509906194E+00, &
                                            -4.04038436141621878E+00, &
                                             5.66705370983819462E+00, &
                                            -2.22650581667682930E+00, &
                                            -2.81865497535629677E-01, &
                                             1.50884164714579146E-01, &
                                            -3.94705503175446545E-02, &
                                             5.90341597049702970E-03, &
                                            -3.94800703013446613E-04, &
                                            -3.49030271694206535E-06, &
                                             4.70253593987672628E-07, &
                                             3.59524620335097006E-09, &
                                            -2.82937576334365206E-12, &
                                             1.20094437442074945E-17  ]

real(8),dimension(lb:ub),parameter:: e5 = [  5.19195789487232040E-18, &
                                            -1.22579554649148638E-12, &
                                             1.44024673139778572E-09, &
                                             2.05380493296490752E-07, &
                                            -1.48637943367662375E-06, &
                                            -1.39137450504676184E-04, &
                                             2.41536612790836166E-03, &
                                            -1.80292821174926513E-02, &
                                             8.20332981200940183E-02, &
                                            -1.28107205555085611E-01, &
                                            -1.45022549048296298E+00, &
                                             3.11099303342779457E+00, &
                                            -1.79613258515317665E+00, &
                                             2.09951980336226873E-01, &
                                             5.13235851657550650E-02, &
                                            -8.67978126996153153E-01, &
                                             1.54954583065062825E+00, &
                                            -2.27858810680987275E-01, &
                                            -2.95527620310386835E-01, &
                                            -8.77810707143849591E-02, &
                                             2.00402856015326302E-02, &
                                            -9.76161242669898318E-03, &
                                             1.88469747124616377E-03, &
                                            -1.47902711159590021E-04, &
                                            -1.06865024749219957E-06, &
                                             1.87633425043387568E-07, &
                                             1.45805862960930634E-09, &
                                            -1.17427613671822700E-12, &
                                             5.08929049672749320E-18  ]

real(8),dimension(lb:ub),parameter:: a6 = [  3.02392019484316302E-17, &
                                            -6.48337113162700263E-12, &
                                             6.70588029549890052E-09, &
                                             8.83043810517847305E-07, &
                                            -5.01435306432540528E-06, &
                                            -4.23169927671551704E-04, &
                                             6.46543456241488457E-03, &
                                            -3.93629930913448334E-02, &
                                             1.34289249777793884E-01, &
                                            -2.40211158990859985E-01, &
                                            -1.23696458339691162E+00, &
                                            -1.97315347194671631E+00, &
                                             5.02064752578735352E+00, &
                                            -2.34011125564575195E+00, &
                                             8.63172233104705811E-01, &
                                            -9.06419396400451660E-01, &
                                             3.42667508125305176E+00, &
                                            -8.53393363952636719E+00, &
                                             5.35284805297851563E+00, &
                                             5.58076441287994385E-01, &
                                            -3.59466850757598877E-01, &
                                             1.06608085334300995E-01, &
                                            -1.78731903433799744E-02, &
                                             1.35026266798377037E-03, &
                                             1.19671403808752075E-05, &
                                            -1.90676132660883013E-06, &
                                            -1.57074673268198239E-08, &
                                             1.32474777675417599E-11, &
                                            -6.02861075655018329E-17  ]

real(8),dimension(lb:ub),parameter:: b6 = [ -7.82969715005624735E-17, &
                                             1.72536336649682418E-11, &
                                            -1.85690988559713900E-08, &
                                            -2.48187253894392533E-06, &
                                             1.84264363133639033E-05, &
                                             1.33049189283782274E-03, &
                                            -2.31487821278390776E-02, &
                                             1.73635719041683294E-01, &
                                            -8.11860904793130117E-01, &
                                             1.53562400284481381E+00, &
                                             9.79292432547635063E+00, &
                                            -2.01026968440632469E+01, &
                                             1.12467356733846490E+01, &
                                            -2.71935420030065834E+00, &
                                             6.96800924363620245E-01, &
                                             6.07111353659967628E-01, &
                                            -2.01318185345508185E+00, &
                                             8.07602529439853924E-01, &
                                             5.57902802044728127E-01, &
                                             2.65193732791625130E-01, &
                                            -3.25454441497613145E-02, &
                                             2.25385394926560301E-02, &
                                            -5.08734857666342540E-03, &
                                             4.56843457940716371E-04, &
                                             3.21847914332802395E-06, &
                                            -6.99038397189051861E-07, &
                                            -5.88212560489345276E-09, &
                                             5.10550969914082229E-12, &
                                            -2.38424984280960258E-17  ]

real(8),dimension(lb:ub),parameter:: c6 = [  1.68963817240765969E-17, &
                                            -3.62262151011245305E-12, &
                                             3.74481655502828220E-09, &
                                             4.95413382154708831E-07, &
                                            -2.55666617682575805E-06, &
                                            -2.37811534683273654E-04, &
                                             3.53125322174602688E-03, &
                                            -2.07854227542615483E-02, &
                                             6.79397907147565274E-02, &
                                            -1.09113542314654879E-01, &
                                            -7.47503790784158611E-01, &
                                            -1.15557388619038603E+00, &
                                             2.40650505628253946E+00, &
                                            -7.87885393903565882E-01, &
                                            -5.18229681536211417E-01, &
                                             2.09771194973500406E+00, &
                                            -9.03750655928245488E+00, &
                                             1.75814904303131962E+01, &
                                            -9.03437102753423600E+00, &
                                            -1.30419547029994320E+00, &
                                             7.33527442380262403E-01, &
                                            -2.06026598132080995E-01, &
                                             3.31478497846707398E-02, &
                                            -2.40254973758632835E-03, &
                                            -2.34036578840579865E-05, &
                                             3.39463784757005428E-06, &
                                             2.81273938722896216E-08, &
                                            -2.37088900523280655E-11, &
                                             1.07893142276793146E-16  ]

real(8),dimension(lb:ub),parameter:: e6 = [ -4.37490212574897634E-17, &
                                             9.64056786609616252E-12, &
                                            -1.03694748372725480E-08, &
                                            -1.39285348074671248E-06, &
                                             9.53356330616977332E-06, &
                                             7.49122746576944068E-04, &
                                            -1.26307207351533054E-02, &
                                             9.17916374423757699E-02, &
                                            -4.14966697282052081E-01, &
                                             6.75119897872238366E-01, &
                                             5.85724411628382217E+00, &
                                            -9.90491612378112889E+00, &
                                             4.36866277864599885E+00, &
                                            -7.96173999515149302E-01, &
                                             8.17501943413809893E-01, &
                                            -1.91426676858868945E+00, &
                                             4.20747421703724013E+00, &
                                            -1.48006296936712585E+00, &
                                            -7.78654324425616538E-01, &
                                            -4.94456741198658739E-01, &
                                             7.37595427453384001E-02, &
                                            -4.40451352977646537E-02, &
                                             9.45096795045409110E-03, &
                                            -8.14145506700585199E-04, &
                                            -6.42744328481083985E-06, &
                                             1.24485674087466370E-06, &
                                             1.05329242240739103E-08, &
                                            -9.13728340058015045E-12, &
                                             4.26705641925017398E-17  ]

end module filterModule
