#!/usr/bin/env python
# ======================================================================
# == Python script checking if some Fortran modules are judiciously   ==
# == used in ABINIT src files.                                        ==
# ==                                                                  ==
# == At present, the following modules are tested:                    ==
# ==   defs_datatypes                                                 ==
# ==   defs_abitypes                                                  ==
# ==   m_pawrhoij                                                     ==
# ==   m_pawcprj                                                      ==
# ==                                                                  ==
# ==   Usage:                                                         ==
# ==     check_datatypes [--all] [--unused] [--suggest] [--ok]        ==
# ==                     [--only module_name] [--autofix]             ==
# ==     --unused  : list unused datatypes                            ==
# ==     --suggest : list suggested changes in use statements         ==
# ==     --ok      : list correctly used datatypes                    ==
# ==     --all     : list all (default)                               ==
# ==     --only    : treat only module named "module_name"            ==
# ==     --autofix : automatically repair errors in src files         ==
# ==     Only one option in (unused, suggest, ok, all) allowed !      ==
# ==                                                                  ==
# ==                                          M. Torrent - Sept. 2012 ==
# ======================================================================

import os
import re
import sys

# ---------------------------------------------------------------------------
MODULES_LIST = ["defs_datatypes","defs_abitypes","m_pawrhoij","m_pawcprj"]

#Content of module "defs_datatypes"
DEFS_DATATYPES_TYPES = [  # Note: use only lowercases
"bandstructure_type",
"bcp_type",
"macro_uj_type",
"pseudopotential_gth_type",
"pseudopotential_type",
"pspheader_paw_type",
"pspheader_type",
"pawang_type",
"pawfgr_type",
"pawfgrtab_type",
"pawrad_type",
"pawtab_type",
"paw_an_type",
"paw_an_flags_type",
"paw_ij_type",
"paw_ij_flags_type"]
DEFS_DATATYPES_ROUTINES = []
DEFS_DATATYPES_FUNCTIONS= []

#Content of module "defs_abitypes"
DEFS_ABITYPES_TYPES = [  # Note: use only lowercases
"aim_dataset_type",
"anaddb_dataset_type",
"dataset_type",
"bandfft_kpt_type",
"mpi_type",
"datafiles_type",
"hdr_type",
"ab_dimensions"]
DEFS_ABITYPES_ROUTINES = []
DEFS_ABITYPES_FUNCTIONS= []

#Content of module "m_pawrhoij"
M_PAWRHOIJ_TYPES = [  # Note: use only lowercases
"pawrhoij_type"]
M_PAWRHOIJ_ROUTINES = [  # Note: use only lowercases
"rhoij_alloc",
"rhoij_free",
"rhoij_nullify",
"rhoij_copy",
"rhoij_gather",
"rhoij_bcast",
"rhoij_io",
"rhoij_unpack",
"rhoij_init_unpacked",
"rhoij_destroy_unpacked",
"rhoij_mpi_sum"]
M_PAWRHOIJ_FUNCTIONS = []

#Content of module "m_pawcprj"
M_PAWCPRJ_TYPES = [  # Note: use only lowercases
"cprj_type"]
M_PAWCPRJ_ROUTINES = [  # Note: use only lowercases
"cprj_alloc",
"cprj_free",
"cprj_nullify",
"cprj_set_zero",
"cprj_copy",
"cprj_axpby",
"cprj_zaxpby",
"cprj_lincom",
"cprj_output",
"cprj_diskinit_r",
"cprj_diskinit_w",
"cprj_diskskip",
"cprj_get",
"cprj_put",
"cprj_exch",
"cprj_mpi_send",
"cprj_mpi_recv",
"cprj_mpi_allgather",
"cprj_bcast",
"cprj_transpose",
"cprj_gather_spin"]
M_PAWCPRJ_FUNCTIONS = [  # Note: use only lowercases
"paw_overlap"]

#Global lists
TYPES_LIST    = [DEFS_DATATYPES_TYPES,DEFS_ABITYPES_TYPES, \
                 M_PAWRHOIJ_TYPES,M_PAWCPRJ_TYPES]
ROUTINES_LIST = [DEFS_DATATYPES_ROUTINES,DEFS_ABITYPES_ROUTINES, \
                 M_PAWRHOIJ_ROUTINES,M_PAWCPRJ_ROUTINES]
FUNCTIONS_LIST= [DEFS_DATATYPES_FUNCTIONS,DEFS_ABITYPES_FUNCTIONS, \
                 M_PAWRHOIJ_FUNCTIONS,M_PAWCPRJ_FUNCTIONS]

#Lines beginning with the following statement will not be considered
COMMENT = "!"

#Files beginning with the following statement will not be considered
IGNORES_FILES = ["interfaces_"]

# ---------------------------------------------------------------------------

if (len(sys.argv)==1) or ("--help" in sys.argv):
  print "Usage:"
  print "  check_datatypes [--all] [--unused] [--suggest] [--ok]"
  print "    --unused : list unused datatypes"
  print "    --suggest: list suggested changes in use statements"
  print "    --ok     : list correctly used datatypes"
  print "    --all    : list all (default)"
  print "    Only one option allowed"

noselection=((not "--unused" in sys.argv) and (not "--suggest" in sys.argv) and \
             (not "--ok" in sys.argv))
do_unused =(("--all" in sys.argv) or ("--unused"  in sys.argv) or (noselection))
do_suggest=(("--all" in sys.argv) or ("--suggest" in sys.argv) or (noselection))
do_ok     =(("--all" in sys.argv) or ("--ok"      in sys.argv) or (noselection))
autofix   =("--autofix" in sys.argv)
if "--only" in sys.argv:
  if len(sys.argv)>sys.argv.index("--only"):
    modules_restricted_list=[sys.argv[sys.argv.index("--only")+1]]
  else:
    modules_restricted_list=[]
else:
  modules_restricted_list=MODULES_LIST

# ---------------------------------------------------------------------------

print
print '---------------------------------------------------------------------'
print ' Looking for "use module" statements in ABINIT Fortran source files. '
print '  - check if they are useful                                         '
print '  - list missing "only" statements                                   '
print '---------------------------------------------------------------------\n'

re_srcfile = re.compile("\.([Ff]|[Ff]90)$")
file_total_count=0

#Loop over files in src folder
for (root, dirs, files) in os.walk("./src"):
  for src in files:
    if (re_srcfile.search(src)):
      file_total_count+=1
      filename=os.path.join(root,src)
      src_data=file(filename).readlines()

#     Ignore some files
      ignore=False
      for item in IGNORES_FILES:
        ignore=(src.find(item)==0)
      if not ignore:

#       Loop over modules
        for module in MODULES_LIST:
          if module in modules_restricted_list:
            mod_index=MODULES_LIST.index(module)
            module_found=False

#           Loop over lines in file (searching use of module)
            for line in src_data:
              if line.find(COMMENT) != 0:
                line_lower=re.sub(" ","",line.lower()) # Lower case, no space

#               Check use of module
                if line_lower.find("use"+module) != -1:
                  module_found=True
                  line_use_module=line_lower
                  break

#           Continue if module has been found
            if module_found:
              types_list_mod=[];routines_list_mod=[];functions_list_mod=[]
              sugg_only_string="";missing_count=0;module_unused=False
              module_stat_found=False;contains_stat_found=False
              subroutine_count=0;function_count=0

#             Loop over lines in file (again)
              for line in src_data:
                if line.find(COMMENT) != 0:
                  line_lower=re.sub(" ","",line.lower()) # Lower case, no space

#                 Looking for datatypes in line
                  for type in TYPES_LIST[mod_index]:
                    if line_lower.find("type("+type+")") != -1:
                      if not type in types_list_mod: types_list_mod.append(type)

#                 Looking for routines in line
                  for routine in ROUTINES_LIST[mod_index]:
                    if line_lower.find("call"+routine+"(") != -1:
                      if not routine in routines_list_mod: routines_list_mod.append(routine)

#                 Looking for functions in line
                  for function in FUNCTIONS_LIST[mod_index]:
                    if line_lower.find(function+"(") != -1:
                      if not function in functions_list_mod: functions_list_mod.append(function)

#                 Looking for multiple "subroutines"
                  if not module_stat_found: module_stat_found=(line_lower.find("module") != -1)
                  if not contains_stat_found: contains_stat_found=(line_lower.find("contains") != -1)
                  if line_lower.find("endsubroutine")!=-1:subroutine_count+=1
                  if line_lower.find("endfunction")!=-1:function_count+=1

              multiple_sub=((not module_stat_found and not contains_stat_found) and \
                            (subroutine_count>1 or function_count>1))
              list_all=types_list_mod+routines_list_mod+functions_list_mod

#             First case: module not used - print a warning
              if (len(list_all))==0:
                module_unused=True
                if do_unused and not autofix:
                  print "File %s: module %s present but not used !" % (filename,module)

#             Second case: module used - suggest "only" statement
              else:
                len_cur=0
                for item in list_all:
                  if line_use_module.find(item) == -1: missing_count+=1
                  if sugg_only_string != "":
                    sugg_only_string+=", ";len_cur=len_cur+1
                  if len_cur>75:
                    if list_all.index(item)!=-1:
                      sugg_only_string+=" &\n&"+' '.ljust(13+len(module))
                      len_cur=0
                  sugg_only_string+=item;len_cur=len_cur+len(item)
                if missing_count==0:
                  if do_ok:
                    print "File %s: module %s correctly used !" % (filename,module)
                else:
                  if do_suggest and not autofix:
                    print "File %s: module %s, suggested use:" % (filename,module)
                    print "  * use "+module+", only : "+sugg_only_string
                    if multiple_sub:
                      print "  * WARNING: several subroutines/functions in file !"

#             === AUTOFIX ===
              if autofix:
                if (module_unused and do_unused) or (missing_count>0 and do_suggest):
                  print "==> FIXING file %s" %(filename)
                  if module_unused and do_unused:
                    print '    ELIMINATING module %s !' % (module)
                  if missing_count>0 and do_suggest:
                    print '    REWRITING "use module" line for %s !' % (module)
                    if multiple_sub:
                      print '    WARNING: several subroutines/functions in file (check it MANUALLY) !'
#                 Open a temporary file for writing
                  ErrorEncountered=False
                  filenametmp=filename+'.tmp'
                  try: filout=open(filenametmp,'w')
                  except:
                    print 'File %s, error: could not open tmp file !' % (filename)
                    ErrorEncountered=True
                  if not ErrorEncountered:
#                   Loop over lines in the file
                    for line in src_data:
                      line_lower=re.sub(" ","",line.lower()) # Lower case, no space
                      do_write_line=1
                      if line_lower.find("use"+module) != -1 and line.find(COMMENT) != 0:
                        if module_unused and do_unused: do_write_line=0
                        if missing_count>0 and do_suggest: do_write_line=2
                      if do_write_line>0:
                        newline=line
                        if do_write_line==2: newline=" use "+module+", only : "+sugg_only_string+"\n"
                        try: filout.write(newline)
                        except:
                          print 'File %s, error: could not write into tmp file !' % (filename)
                          ErrorEncountered=True;break
                  filout.close()
#                 Replace current file by temporary file
                  if not ErrorEncountered:
                    try: os.system('mv -f '+filenametmp+' '+filename)
                    except: print 'File %s, error: could not move tmp file !' % (filename)

#Final printing
print "--------------------------------------"
print "Done !, %s files have been explored." % (file_total_count)
