#!/usr/bin/env python

import sys
import os
import platform
from socket import gethostname

from os.path import join as pj, abspath as absp, exists as pexists, isfile, basename
from warnings import warn

try: 
  from ConfigParser import SafeConfigParser, NoOptionError
except ImportError: # The ConfigParser module has been renamed to configparser in Python 3
  from configparser import SafeConfigParser, NoOptionError

try:
  import tests 
except ImportError:
  # Add the directory [...]/abinit/tests to $PYTHONPATH
  pack_dir, x = os.path.split(absp(__file__))
  pack_dir, x = os.path.split(pack_dir)
  sys.path.insert(0,pack_dir)
  import tests 

__version__ = "0.3"
__author__  = "Matteo Giantomassi"

_my_name = basename(__file__) + "-" + __version__

abitests = tests.abitests
abenv = tests.abenv

#from pymods.devtools import number_of_cpus
from pymods.testsuite import BuildEnvironment
from pymods.jobrunner import TimeBomb, JobRunner, OMPEnvironment

def _yesno2bool(string):
  string = string.lower().strip().replace('"',"").replace("'","")
  if string == "yes": 
    return True
  elif string == "no": 
    return False
  else:
    raise ValueError("Cannot interpret string: %s" % string)

def _str2list(string):    return [s.strip() for s in string.split(",") if s]

class TestBot(object):

  _attrbs = { 
    # name         -->   (default, parser)
    "slavename"        : (None, str),
    "type"             : ("",   str),  # "ref" if this slave is the reference slave e.g testf
    "ncpus"            : (None, int),
    "mpi_prefix"       : ("",   str),
    "mpirun_np"        : ("",   str),
    "omp_num_threads"  : (0,    int),
    "enable_mpi"       : (None, _yesno2bool),
    "enable_openmp"    : (None, _yesno2bool),
    "poe"              : ("", str),
    "poe_args"         : ("", str),
    "with_tdirs"       : ("", _str2list),
    "without_tdirs"    : ("", _str2list),
    "timeout_time"     : (900, float),
  }

  def __init__(self):

    basedir, x = os.path.split(absp(__file__))
    testbot_cfg = pj(basedir, "testbot.cfg")

    parser = SafeConfigParser() 
    parser.read(testbot_cfg)

    attrs2read = [
      "slavename", 
      "type", 
      "ncpus", 
      "omp_num_threads", 
      "mpi_prefix",
      "mpirun_np", 
      "poe", 
      "poe_args",
      "with_tdirs",
      "without_tdirs",
      "timeout_time",
    ]

    for attr in attrs2read:
      default, parse = TestBot._attrbs[attr]
      try:
        value = parser.get("testbot", attr)
      except NoOptionError:
        value = default

      if value is None: # Dump cfg file and raise
        for section in parser.sections():
          print "[" + section + "]"
          for opt in parser.options(section):
             print opt + " = " + parser.get(section, opt)

        err_msg = "Mandatory option %s is not declared" % attr
        raise ValueError(err_msg)

      self.__dict__[attr] = parse(value)


    if self.with_tdirs and self.without_tdirs:
      err_msg = "with_tdirs and without_tdirs attribute are mutually exclusive"
      raise ValueError(err_msg)

    (system, node, release, version, machine, processor) = platform.uname()
    print "Running on %s -- slave %s -- system %s -- ncpus %s -- Python %s -- %s" % (
      gethostname(), self.slavename, system, self.ncpus, platform.python_version(), _my_name)
                                                                                                   
    build_examples = abenv.apath_of(pj("config", "specs", "build-examples.conf"))

    parser = SafeConfigParser()
    parser.read(build_examples)

    if self.slavename not in parser.sections():
      err_msg = "%s is not a valid buildbot slave." % self.slavename
      raise ValueError(err_msg)

    #print self
    # TODO
    # Consistency check
    #d = self.__dict__
    #for attr, (default, parse) in TestBot._attrbs.items():
    #  try:
    #    d[attr] = parser.get(self.slavename, attr)
    #  except NoOptionError:
    #    print "option %s is not declared" % attr
    #    #raise ValueError(err_msg)
    #    d[attr] = default
    #    if default is None:
    #      err_msg = "option %s is not declared" % attr
    #      raise ValueError(err_msg)
    #  d[attr] = parse(d[attr])

    self.build_env = build_env = BuildEnvironment(os.curdir)

    if build_env.has_bin("timeout") and self.timeout_time > 0:
      #
      # Run F90 executables under the control of timeout.c
      timeout_path = build_env.path_of_bin("timeout")
      timebomb = TimeBomb(self.timeout_time, exec_path=timeout_path)
    else:
      warn("Cannot find timeout executable at: %s" % build_env.path_of_bin("timeout"))
      timebomb = TimeBomb(self.timeout_time)

    print "Initalizing JobRunner for sequential runs."
    self.seq_runner = JobRunner.sequential(timebomb=timebomb)
    print self.seq_runner

    if self.has_mpi:
      print "Initalizing MPI JobRunner from self.__dict__"
      self.mpi_runner = JobRunner.fromdict(self.__dict__, timebomb=timebomb)
      print self.mpi_runner

    if self.omp_num_threads > 0: 
      omp_env = OMPEnvironment(OMP_NUM_THREADS = self.omp_num_threads)
      self.seq_runner.set_ompenv(omp_env)
      if self.has_mpi: self.mpi_runner.set_ompenv(omp_env)

    self.targz_fnames = []

    print self

  def __str__(self):
    return "\n".join([str(k) + " : " + str(v) for (k,v) in self.__dict__.items()])

  @property
  def has_mpi(self):
    "True if we have a MPI runner"
    return (bool(self.mpirun_np) or bool(self.poe))

  @property
  def has_openmp(self):
    "True if the tests must be executed with OpenMP threads."
    return self.omp_num_threads > 0

  def run_tests_with_np(self, mpi_nprocs, suite_args=None):
    "Run tests specified by suite_args using mpi_nprocs MPI processors"

    omp_nthreads = max(self.omp_num_threads, 1)
    #                                                                  
    py_nthreads = self.ncpus // (mpi_nprocs * omp_nthreads)
    assert py_nthreads > 0
    #                                                                  
    test_suite = abitests.select_tests(suite_args, regenerate=False)

    # Create workdir.
    workdir = "TestBot_MPI" + str(mpi_nprocs) 

    if self.has_openmp: workdir += "_OMP" + str(omp_nthreads)

    if os.path.exists(workdir):
      raise RuntimeError("%s already exists!" % workdir)
    else:
      os.mkdir(workdir)

    # Run the tests.
    if self.has_openmp: 
      msg = "Running ntests = %s, MPI_nprocs = %s, OMP_nthreads %s, py_nthreads = %s..." % (
               test_suite.full_lenght, mpi_nprocs, omp_nthreads, py_nthreads)
    else:
      msg = "Running ntests = %s, MPI_nprocs = %s, py_nthreads = %s..." % (
              test_suite.full_lenght, mpi_nprocs, py_nthreads)

    print msg

    runner = self.seq_runner
    if mpi_nprocs > 1: runner = self.mpi_runner

    results = test_suite.run_tests(self.build_env, workdir, runner, mpi_nprocs, py_nthreads)

    # Push the location of the tarball file
    self.targz_fnames.append(results.targz_fname)

    return (results.nfailed, results.npassed)

  def run(self):
    """
    Run all the automatic tests depending on the environment and the options 
    specified in the testbot configuration file.
    """

    # Run tests with 1 processor 
    # (execute all tests if with_tdirs and without_tdirs are not given)
    suite_args = None

    if self.with_tdirs:
      suite_args = " ".join([s for s in self.with_tdirs])
    elif self.without_tdirs:
      suite_args = "- ".join([s for s in self.without_tdirs]) + "-"

    (nfailed, npassed) = self.run_tests_with_np(1, suite_args=suite_args)

    if self.has_mpi:
      #
      # Run the parallel tests in the multi-parallel suites.
      mp_suites = abitests.multi_parallel_suites()
      mp_names = [suite.name for suite in mp_suites]

      suite_args = mp_names

      # Prune dirs.
      if self.with_tdirs:
        suite_list = [s for s in self.with_tdirs if s in mp_names]
        suite_args = " ".join([s for s in suite_list])

      elif self.without_tdirs:
        suite_list = [s for s in mp_names if s not in self.without_tdirs]
        suite_args = " ".join([s for s in suite_list])

      for np in [2, 4, 10]:
        if np > self.ncpus: continue
        print "Running multi-parallel tests with %s MPI processors, suite_args %s" % (np, suite_args)
        (para_nfailed, para_npassed) = self.run_tests_with_np(np, suite_args=suite_args)
        #
        # Accumulate the counters.
        nfailed += para_nfailed
        npassed += para_npassed

    # TODO Collect tarball files.
    for fname in self.targz_fnames:
      print "got targz file: ",fname

    # Status error depends on the builder type.
    if self.type == "ref":
      return nfailed + npassed
    else:
      return nfailed

#############################################################################################################

if __name__ == "__main__":
  sys.exit(TestBot().run())
