#!/usr/bin/env python
"""Script for running the ABINIT automatic tests."""
import sys
import os
import platform

from os.path import join as pj, abspath as absp, exists as pexists, isfile, basename
from warnings import warn

from optparse import OptionParser
from socket import gethostname

try:
  import tests 
except ImportError:
  # Add the directory [...]/abinit/tests to $PYTHONPATH
  pack_dir, x = os.path.split(absp(__file__))
  pack_dir, x = os.path.split(pack_dir)
  sys.path.insert(0,pack_dir)
  import tests 

abenv = tests.abenv
abitests = tests.abitests

from tests.pymods.devtools import number_of_cpus
JobRunner = tests.pymods.JobRunner
OMPEnvironment = tests.pymods.OMPEnvironment
TimeBomb = tests.pymods.TimeBomb
BuildEnvironment = tests.pymods.BuildEnvironment

__version__ = "0.4"
__author__  = "Matteo Giantomassi"

_my_name = basename(__file__) + "-" + __version__

#############################################################################################################
### Helper functions and tools

def asctime_dirname(prefix=""):
  from time import asctime
  if not prefix: prefix = "tsuite"
  s = asctime()
  # Remove day name, replace whitespaces with underscores and remove : chars. 
  s = "_".join( [ tok for tok in s.split()[1:] ] )
  s = s.replace(":","")
  return (prefix + "_" + s)

###############################################################################
def str_examples():
  examples = """
    Typical examples (assuming the script is exexcuted within a build tree):
    \n
    runtests.py                      => Run the entire test suite with one thread.
    runtests.py -j2                  => Run the entire test suite with two threads.
    runtests.py v1 v2 -k abinit      => Run only the tests in v1,v2 that contain abinit as keyword.
    runtests.py v3[:4] v4[45:] v5[3] => Run the tests in v3 from t1.in to t3.in, all the
                                        test in v4 starting from t45.in, and test t3 in v5
                                        (Note Python indexing) 
    runtests.py v3- v4- -k anaddb    => Run the anaddb tests, except those in v3 and v4
    runtests.py paral -n4 -c mpi.cfg => Run the paral test with 4 MPI processes. The MPI
                                        environment is read from from file mpi.cfg
                                        (default runner is 'mpirun -n' if c option is not used)
    runtest.py -k GW perl-           => Run the tests that contain the keyword 'GW', and exclude those with  
                                        the keyword 'perl'.
    runtest.py v1 -a Wall-           => Run the tests in v1 but exclude those constributed by the author 'Wall'
  """
  return examples

def show_examples_and_exit(err_msg=None, error_code=1):
  """Display the usage of the script."""
  sys.stderr.write(str_examples())
  if err_msg: sys.stderr.write("Fatal Error\n" + err_msg + "\n")
  sys.exit(error_code)

def vararg_callback(option, opt_str, value, parser):
  """callback for an option with variable arguments"""
  assert value is None
  value = []

  def floatable(str):
    try:
      float(str)
      return True
    except ValueError:
      return False

  for arg in parser.rargs:
    # stop on --foo like options
    if arg[:2] == "--" and len(arg) > 2: break
    # stop on -a, but not on -3 or -3.0
    if arg[:1] == "-" and len(arg) > 1 and not floatable(arg): break
    value.append(arg)

  del parser.rargs[:len(value)]
  setattr(parser.values, option.dest, value)

###############################################################################

def main():

  usage = "usage: %prog [suite_args] [options]. Use [-h|--help] for help."
  version="%prog "+ str(__version__)

  class MyOptionParser(OptionParser):
    def print_help(self):
      OptionParser.print_help(self)
      print "\n" + str_examples()

  parser = MyOptionParser(usage=usage, version=version)

  parser.add_option("-c", "--cfg_file", dest="cfg_fname", type="string",
                    help="Read options from cfg FILE.", metavar="FILE")

  parser.add_option("-n", "--num-mpi-processors", dest="mpi_nprocs", type="int", default=1,
                    help="maximum number of MPI processes.")

  parser.add_option("-j", "--jobs", dest="py_nthreads", type="int", default=1,
                    help="number of python threads.")

  parser.add_option("-r", "--regenerate", dest="regenerate", default=False, action="store_true",
                    help="regenerate the test suite database" +
                    "(use this option after any change of the input files of the test suite or any change of the python scripts.")

  parser.add_option("-k", "--keywords", dest="keys", default=[], action="callback", callback=vararg_callback,
                    help="Run the tests containing these keywords.")

  parser.add_option("-a", "--authors", dest="authors", default=[], action="callback", callback=vararg_callback,
                    help="Run the tests contributed by these developers.")

  parser.add_option("-t", "--timeout", dest="timeout_time", type="int", default=900, 
                     help="timeout value for Fortran executables (in seconds).")

  parser.add_option("-b", "--build-tree", dest="build_dir_path", default="", 
                     help="path to the top level directory of the build tree.")

  parser.add_option("-s", "--show-info", dest="show_info", default=False, action="store_true",
                    help="Show information on the test suite (keywords, authors...) and exit")

  parser.add_option("-w", "--workdir", dest="workdir", type="string", default="",
                    help="Directory where the test suite results will be produced")

  parser.add_option("-o", "--omp_num-threads", dest="omp_nthreads", type="int", default=0,
                    help="Number of OMP threads to use (set the value of the env variable OMP_NUM_THREADS.\n" + 
                         "Not compatible with -c. Use the cfg file to specify the OpenMP runtime variables.\n")

  parser.add_option("-p", "--patch", dest="patch", action="store_true", default=False, 
                    help="Patch reference files")

  parser.add_option("-v", "--verbose", dest="verbose", action="store_true", default=False, 
                    help="verbose mode")

  parser.add_option("--erase-files", dest="erase_files", type="int", default=2, 
                    help= ( "0 => Keep all files.\n" +
                            "1 => Remove files but only if the test passed or succeeded.\n"+
                            "2 => Remove files even when the test fail.\n" +  
                            "default=2\n") )

  parser.add_option("--make-html-diff", dest="make_html_diff", type="int", default=0, 
                    help= ("0 => Do not produce diff files in HTML format\n" + 
                           "1 => Produce HTML diff but only if test fails\n" +
                           "2 => Produce HTML diff independently of the final status.\n" +
                           "default=0\n") )

  parser.add_option("--sub-timeout", dest="sub_timeout", type="int", default=30, 
                    help="Timeout (s) for small subprocesses (fldiff.pl, python functions)")

  #parser.add_option("--dump-cpickle", dest=dump_cpkl"", action="store_true", default=False, 
  #                  help="Save python object in Cpickle format (option for developers)")

  (options, suite_args) = parser.parse_args()

  if options.show_info:
     abitests.show_info()
     sys.exit(0) 

  if options.verbose:
    def vrb_print(*args): print args
  else:
    def vrb_print(*args): pass

  ncpus_detected = max(1, number_of_cpus())

  (system, node, release, version, machine, processor) = platform.uname()
  print "Running on %s -- system %s -- ncpus %s -- Python %s -- %s" % (
    gethostname(), system, ncpus_detected, platform.python_version(), _my_name)

  mpi_nprocs  = options.mpi_nprocs   
  py_nthreads = options.py_nthreads 
  omp_nthreads = options.omp_nthreads

  # Initialize info on the build. User's option has the precedence.
  build_dir_path = os.path.curdir
  if options.build_dir_path: build_dir_path = absp(options.build_dir_path) 
    
  build_env = BuildEnvironment(build_dir_path)

  timeout_time = options.timeout_time
                                                                                           
  if timeout_time > 0 and build_env.has_bin("timeout"):  
    # Run executables under the control of timeout.
    timeout_path = build_env.path_of_bin("timeout")
    #timeout_signal = ""
    timebomb = TimeBomb(timeout_time, exec_path=timeout_path)
  else:
    err_msg = "Cannot find timeout executable at: %s" % build_env.path_of_bin("timeout")
    warn(err_msg)
    timebomb = TimeBomb(timeout_time)
  #
  # ------------------------------------------------
  # Initialize the jobrunner for the (MPI|seq) mode
  # ------------------------------------------------
  if options.cfg_fname: 
    # read the [mpi] and the [openmp] sections from the external cfg file.
    assert omp_nthreads == 0
    cfg_fname = options.cfg_fname
    vrb_print("Initalizing JobRunner from cnf file: ",cfg_fname)
    runner = JobRunner.fromfile(cfg_fname, timebomb=timebomb)
                                                                                           
  else:
    if mpi_nprocs == 1: 
      vrb_print("Initalizing JobRunner for sequential runs.")
      runner = JobRunner.sequential(timebomb=timebomb)
    else:
      vrb_print("Initalizing JobRunner assuming generic_mpi. [-c option not provided]")
      runner = JobRunner.generic_mpi(timebomb=timebomb)
                                                                                           
    if omp_nthreads > 0: 
      omp_env = OMPEnvironment(OMP_NUM_THREADS = omp_nthreads)
      runner.set_ompenv(omp_env)

  #print "runner",runner
  #print options.authors
  #sys.exit(1)

  try:
    test_suite = abitests.select_tests(suite_args, regenerate=options.regenerate, 
                                       keys=options.keys, authors=options.authors)
  except Exception, exc:
    raise
    show_examples_and_exit(str(exc))

  workdir = options.workdir
  if not workdir: 
    workdir = asctime_dirname()
    workdir = "Test_suite"

  # Create workdir.
  if os.path.exists(workdir):
    raise RuntimeError("%s already exists!" % workdir)
  else:
    os.mkdir(workdir)

  # Run the final test suites.
  if omp_nthreads == 0: 
    msg = "Running ntests = %s, MPI_nprocs = %s, py_nthreads = %s..." % (
      test_suite.full_lenght, mpi_nprocs, py_nthreads)
  else:
    msg = "Running ntests = %s, MPI_nprocs = %s, OMP_nthreads %s, py_nthreads = %s..." %  (
      test_suite.full_lenght, mpi_nprocs, omp_nthreads, py_nthreads)

  print(msg)

  #with open("readme.html", "w") as fh:
  #  fh.write(test_suite.make_readme(width=160, html=True))
  #sys.exit(1)

  results = test_suite.run_tests(build_env, workdir, runner, mpi_nprocs, py_nthreads,
                                 erase_files    = options.erase_files,
                                 make_html_diff = options.make_html_diff,
                                 sub_timeout    = options.sub_timeout
                                )
  do_patch = options.patch
  do_patch = do_patch and (results.nfailed !=0 or results.npassed != 0)
  if do_patch:
    results.patch_refs()

  #print results

  vrb_print("Execution completed.")
  return results.nfailed

###############################################################################

if __name__ == "__main__":
  sys.exit(main())
