#!/usr/bin/env python
from __future__ import division

import os
import sys
import shutil as shut

from os.path import join as pj, abspath as absp #, exists as pexists, basename

from subprocess import Popen, PIPE, call

try:
  from subprocess import CalledProcessError
except ImportError:
  class CalledProcessError(Exception):
    pass

__version__ = "0.1"
__author__  = "Matteo Giantomassi"

__all__ = [
"RestrictedShell",
"StringColorizer",
"Editor",
]

#############################################################################################################
# Helper functions

def pipe_commands(args1, args2):
  p1 = Popen(args1, stdout=PIPE)
  p2 = Popen(args2, stdin=p1.stdout, stdout=PIPE)
  return p2.communicate() # (stdout, stderr)

def unzip(gz_fname, dest=None):
  """decompress a gz file."""
  import gzip

  if not gz_fname.endswith(".gz"): raise ValueError("%s should end with .gz" % gz_fname)

  try:
    gz_fh = gzip.open(gz_fname, 'rb')
    file_content = gz_fh.read()
  finally:
    gz_fh.close() # Cannot use try, except, finally in python2-4

  try:
    if dest is None: dest = gz_fname[:-3]
    out_fh = open(dest, "wb")
    out_fh.write(file_content)
  finally:
    out_fh.close()

#############################################################################################################

def touch(fname):
  try:
    open(fname,"w").close()
    return 0
  except:
    raise IOError("trying to create file = %s" % fname) 

#############################################################################################################

def tail_file(fname, n, aslist=False):
  """Assumes a unix-like system."""

  args = ["tail", "-n " + str(n), fname]

  p = Popen(args, shell=False, stdout=PIPE, stderr=PIPE)
  ret_code = p.wait() 
  
  if ret_code != 0: 
    raise RuntimeError("return_code = %s, cmd = %s" %(ret_code, " ".join(args) ))

  if aslist:
    return p.stdout.readlines()
  else:
    return p.stdout.read()

#########################################################################################

def which(program):
  """
  python version of the unix tool which 
  locate a program file in the user's path
  Return None if program cannot be found.
  """
  def is_exe(fpath):
      return os.path.isfile(fpath) and os.access(fpath, os.X_OK)

  fpath, fname = os.path.split(program)
  if fpath:
      if is_exe(program):
          return program
  else:
      for path in os.environ["PATH"].split(os.pathsep):
          exe_file = os.path.join(path, program)
          if is_exe(exe_file):
              return exe_file

  return None

#########################################################################################

from string import maketrans, punctuation
_table = maketrans("","")

def strip_punct(s):
  "Remove punctuation characters from string s."
  return s.translate(_table, punctuation)

def tonumber(s):
  """Convert string to number, raise ValueError if s cannot be converted."""
  # Duck test. Much more readable than the ugly strfltrem routine in fldiff.pl
  try:
    stnum = s.upper().replace("D","E")  # D-01 is not recognized by python: Replace it with E.
    stnum = strip_punct(stnum)          # Remove punctuation chars.
    return float(stnum)                 # Try to convert.
  except ValueError:
    raise 
  except:
    raise RuntimeError("Don't know how to handle string: " + s)

def nums_and_text(line):
  "split line into (numbers, text)."
  tokens = line.split()
  text    = ""
  numbers = list()
  for tok in tokens:
     try:
       numbers.append( tonumber(tok) )
     except ValueError:
       text += " " + tok
  return numbers, text

#############################################################################################################
class RShellError(Exception): pass

class RestrictedShell(object):
  _key2command = {
  # key      (function,   nargs)
  "rm"    : (shut.rmtree, 2),
  "cp"    : (shut.copy,   2),
  "mv"    : (shut.move,   2),
  "touch":  (touch,       1),
  }

  def __init__(self, inp_dir, workdir, psps_dir):
    """Helper function executing simple commands passed via a string."""

    self.exceptions = []

    self.prefix2dir = { 
      "i" : absp(inp_dir),
      "w" : absp(workdir),
      "p" : absp(psps_dir),
    }

  def empty_exceptions(self):
    self.exceptions = []

  def execute(self, string):
    """
    Don't raise exceptions since python threads get stuck. 
    Exceptions are stored in self.exceptions
    """
    #print "executing %s" % string
    _key2command = RestrictedShell._key2command

    tokens = string.split()
    try:
      key, args  = tokens[0], tokens[1:]
      pres, key = key.split("_") # Assume command in the form pre_cmd
      cmd   = _key2command[key][0]
      expected_nargs = _key2command[key][1]
      #print pres, key, cmd, expected_nargs
      nargs = len(args)
      if nargs != expected_nargs:
        err_msg = " Too many arguments, cmd = %s, args = %s " % (cmd, args)
        self.exceptions.append(RShellError(err_msg))
        return 
    except:
      err_msg = "Not able to interpret the string: %s " % string
      self.exceptions.append(RShellError(err_msg))
      return

    new_args = []
    for (pref, arg) in zip(pres, args): 
      new_args.append( pj(self.prefix2dir[pref], arg)  )
    #print "new_args: ",new_args

    try:
      if nargs == 1:
        assert pres == "w"
        return cmd(new_args[0])
      elif nargs == 2:
        return cmd(new_args[0], new_args[1])
      else:
        raise NotImplementedError("nargs = %s is too large" % nargs)
    except:
      import sys
      err_msg = "Executing: " + string + "\n" + str(sys.exc_info()[1])
      self.exceptions.append(RShellError(err_msg))

#############################################################################################################

# Python cookbook, #475186
def stream_has_colours(stream):
  if not hasattr(stream, "isatty"): 
    return False

  if not stream.isatty():
    return False # auto color only on TTYs
  try:
    import curses
    curses.setupterm()
    return curses.tigetnum("colors") > 2
  except:
    return False # guess false in case of error

class StringColorizer(object):
  colours={"default":"",
           "blue":  "\x1b[01;34m",
           "cyan":  "\x1b[01;36m",
           "green": "\x1b[01;32m",
           "red"  : "\x1b[01;31m",
           # lighting colours.
           #"lred":    "\x1b[01;05;37;41m"
           }

  def __init__(self, stream):
     self.has_colours = stream_has_colours(stream)

  def __call__(self, string, colour):
    if self.has_colours: 
      code = self.colours.get(colour, "")
      if code:
        return code + string + "\x1b[00m"
      else:
        return string
    else:
      return string 

#############################################################################################################
def user_wants_to_exit():
  try:
    ans = raw_input("Do you want to continue [Y/n]")
  except EOFError:
    print "exc"
    return True

  if ans.lower().strip() in ["n", "no"]: return True
  return False

class Editor(object):
  def __init__(self, editor=None):
    if editor is None:
      self.editor = os.getenv("EDITOR", "vi")
    else: 
      self.editor = str(editor)

  def edit_file(self, fname):
    from subprocess import call 
    retcode = call([self.editor, fname])
    if retcode != 0:
      raise RuntimeError("trying to edit file: %s" % fname)

  def edit_files(self, fnames):
    for idx, fname in enumerate(fnames): 

      exit_status = self.edit_file(fname)
      if idx != len(fnames)-1 and user_wants_to_exit(): break

    return exit_status

def input_from_editor(message=None):
  if message: print message,
  from tempfile import mkstemp
  fd, fname = mkstemp(text=True)

  Editor().edit_file(fname)

  filobj = open(fname) 
  string = filobj.read() 
  filobj.close()
  return string

#############################################################################################################

class Patcher(object):
  interactive_patchers = [
    "vimdiff", 
    "kdiff3",
    "tkdiff",
  ]

  auto_patchers = [
    "patch",
  ]

  known_patchers = interactive_patchers + auto_patchers

  def __init__(self, patcher=None):
     if patcher is None:
       self.patcher = os.getenv("PATCHER", "kdiff3")
       if which(self.patcher) is None:
         print "using vimdiff"
         self.patcher = "vimdiff"

     else:
       if patcher not in Patcher.known_patchers: raise ValueError("%s is not supported" % patcher)
       self.patcher = str(patcher)

     if which(self.patcher) is None:
       err_msg = "Cannot find executable %s in user PATH" % self.patcher
       raise ValueError(err_msg)

  @property
  def is_interactive(self):
    return self.patcher in Patcher.interactive_patchers

  def patch(self, fromfile, tofile):

    if self.patcher == "patch":
      try: # diff -u to from | patch
        o, e = pipe_commands(["diff", "-u", tofile, fromfile], [self.patcher])
        return 0
      except:
        sys.stderr.write(e.read())
        raise CalledProcessError("%s: trying to patch  %s %s" % (self.patcher, fromfile, tofile))

    else:
      try:
        return call([self.patcher, fromfile, tofile])
      except CalledProcessError:
        raise CalledProcessError("%s: trying to patch  %s %s" % (self.patcher, fromfile, tofile))

  def patch_files(self, fromfiles, tofiles):
    nfiles = len(fromfiles)
    assert nfiles == len(tofiles)

    for idx, (fro, to) in enumerate(zip(fromfiles, tofiles)):  

      if self.is_interactive:
        print idx, fro, to
        exit_status = self.patch(fro, to)
        if idx != (nfiles-1) and user_wants_to_exit(): break

      else:
        try:
          exit_status = self.patch(fro, to)
        except CalledProcessError:
          raise

    return exit_status

#############################################################################################################
def pprint_table(table, out=sys.stdout, rstrip=False):
    """
    Prints out a table of data, padded for alignment
    Each row must have the same number of columns.

    out: Output stream (file-like object)
    table: The table to print. A list of lists.
    rstrip: if true, trailing withespaces are removed from the entries.
    """
    def max_width_col(table, col_idx):
        """Get the maximum width of the given column index"""
        return max([len(row[col_idx]) for row in table])

    if rstrip:
      for row_idx, row in enumerate(table):
        table[row_idx] = [c.rstrip() for c in row]

    col_paddings = []
    ncols = len(table[0])
    for i in range(ncols):
        col_paddings.append(max_width_col(table, i))

    for row in table:
        # left col
        out.write( row[0].ljust(col_paddings[0] + 1) )
        # rest of the cols
        for i in range(1, len(row)):
            col = row[i].rjust(col_paddings[i] + 2)
            out.write(col)
        out.write("\n")


if __name__ == "__main__":
  sys.exit(1)
  #editor = Editor()
  #editor.edit_files(["foo.txt", "bar.txt"])

  #string = input_from_editor()
  #print(string)

  #Patcher("patch").patch("hello.txt","world.txt")
  #o, e = pipe_commands(["ls"], ["wc", "-l"])
  #print(o)
