#!/usr/bin/env python

import sys
import os
import time
import datetime
import pprint
import shutil 
import textwrap
import platform
import tarfile
import cPickle as cpkl

from os.path import join as pj, abspath as absp, exists as pexists, basename, isfile
from StringIO import StringIO 
from socket import gethostname
from warnings import warn

from subprocess import Popen, PIPE

try: 
  from ConfigParser import SafeConfigParser, NoOptionError
except ImportError: # The ConfigParser module has been renamed to configparser in Python 3
  from configparser import SafeConfigParser, NoOptionError

try:
  from collections import namedtuple
except ImportError: # Added in python2.6.
  from tests.pymods.namedtuple import namedtuple

from jobrunner import JobRunner, JobRunnerError, TimeBomb
from tools import RestrictedShell, RShellError, StringColorizer, unzip, tail_file, pprint_table, Patcher
from xyaptu import xcopier

__version__ = "0.4"
__author__  = "Matteo Giantomassi"

__all__ = [
"BuildEnvironment",
"TestSuite",
]

#############################################################################################################
### Helper functions and tools

_MY_NAME = basename(__file__)[:-3] + "-" + __version__

__debug = False
#__debug = True
if __debug:
  def dbg_print(*args): print(args)
else:
  def dbg_print(*args): pass

def html_colorize_text(string, code): 
  return "<FONT COLOR='%s'>%s</FONT>" % (code, string)

_status2htmlcolor = {
  "succeeded" : lambda string : html_colorize_text(string, 'Green'),
  "passed"    : lambda string : html_colorize_text(string, 'DeepSkyBlue'),
  "failed"    : lambda string : html_colorize_text(string, 'Red'),
  "disabled"  : lambda string : html_colorize_text(string, 'Cyan'),
  "skipped"   : lambda string : html_colorize_text(string, 'Cyan'),
}

def status2html(status):
  return _status2htmlcolor[status](status)

def sec2str(seconds):
   return "%.2f" % seconds
   #if seconds < 60:
   #  return "%.2f" % seconds
   #else:
   #  return str(datetime.timedelta(seconds=seconds))

def str2html(string, end="<br>"): 
  lines = string.splitlines()
  return "<br>".join(lines) + end 

def args2htmltr(*args):
  string = ""
  for arg in args: string += "<td>" + str(arg) + "</td>"
  return string

def html_link(string, href=None):
  if href is not None:
    return "<a href='%s'>%s</a>" % (href, string)
  else:
    return "<a href='%s'>%s</a>" % (string, string)

def my_all(iterable):
  "Replacement for the builtin function all (added in python2.5)"
  for item in iterable:
    if not item: return False
  return True

def my_any(iterable):
  "Replacement for the builtin function any (added in python2.5)"
  for item in iterable:
    if item: return True
  return False

def has_exts(path, exts):
  "True if path ends with extensions exts"
  root, ext = os.path.splitext(path)
  if isinstance(exts, str):
    return ext == exts
  else:
    return ext in exts

def lazy__str__(func): 
  "Lazy decorator for __str__ methods"
  def oncall(*args, **kwargs):
    self = args[0]
    return "\n".join( [ str(k) + " : " + str(v) for (k,v) in self.__dict__.items()] )
  return oncall

# Helper functions for IO (with statement was added in py2.6) 
def lazy_read(fname):
  fh = open(fname, "r")
  line = fh.read()
  fh.close()
  return line

def lazy_readlines(fname):
  fh = open(fname, "r")
  lines = fh.readlines()
  fh.close()
  return lines

def lazy_write(fname, str):
  fh = open(fname, "w")
  fh.write(str)
  fh.close()

def lazy_writelines(fname, lines):
  fh = open(fname, "w")
  fh.writelines(lines)
  fh.close()

class Record(object):
  @lazy__str__
  def __str__(self): pass

#############################################################################################################

class FileToTest(object):
  "Info on the output file that is checked by fldiff"
  #            atr_name,   default, conversion function. None designes mandatory attributes.
  _attrbs = [ ("name",     None, str), 
              ("tolnlines",None, int),    # fldiff tolerances
              ("tolabs",   None, float),
              ("tolrel",   None, float),  
              ("fld_options","",str) ,     # options passed to fldiff.
              ("fldiff_fname","",str),
              ("hdiff_fname","",str),
            ] 

  def __init__(self, dic):

    for atr in FileToTest._attrbs: 
      atr_name = atr[0] 
      default  = atr[1]
      f        = atr[2]
      value = dic.get(atr_name, default)
      if value is None: raise ValueError("%s must be defined" % atr_name)

      value = f(value)
      if hasattr(value, "strip"): value = value.strip()
      self.__dict__[atr_name] = value

    # Postprocess fld_options
    self.fld_options = self.fld_options.split()
    for opt in self.fld_options:
      if not opt.startswith("-"): raise ValueError("Wrong fldiff option: %s" % opt)

  @lazy__str__
  def __str__(self): pass

  def compare(self, fldiff_path, ref_dir, workdir, timebomb=None, outf=sys.stdout):
    """
    Use fldiff_path to compare the reference file located in ref_dir with the output file located in workdir
    Analysis results are written to outf.
    """

    ref_fname = absp( pj(ref_dir, self.name) )
    # FIXME Hack due to the stdout-out ambiguity
    if not pexists(ref_fname) and ref_fname.endswith(".stdout"): ref_fname = ref_fname[:-7] + ".out"
    out_fname = absp( pj(workdir, self.name) )

    opts  = self.fld_options
    label = self.name
                                                                                                                  
    fld_result = wrap_fldiff(fldiff_path, ref_fname, out_fname, 
                              opts=opts, label=label, timebomb=timebomb, out_filobj=outf)

    (isok, status, msg) = fld_result.passed_within_tols(self.tolnlines, self.tolabs, self.tolrel)

    # Save comparison results.
    self.fld_isok   = isok
    self.fld_status = status
    self.fld_msg    = msg

    return (isok, status, msg)

#############################################################################################################
# Parsers used for the different TEST_INFO options  
class TestInfoParserError(Exception): 
  pass

# parsers used for the different options
def _str2filestotest(string):
  if not string: return []

  if ";" in string: 
    file_lines = string.split(";")
  else:
    file_lines = [string]
    
  files_to_test = []
  for line in file_lines:
    tokens = line.split(",")
    d = dict()
    d["name"] = tokens[0]
    for tok in tokens[1:]:
      k, v = [s.strip() for s in tok.split("=")]
      if k in d: 
        err_msg = "Found multiple occurences of keyword %s" % k
        raise TestInfoParserError(err_msg)
      d[k] = v
    files_to_test.append( FileToTest(d) )
  return tuple(files_to_test)

def _str2list(string):    return [s.strip() for s in string.split(",") if s]
def _str2intlist(string): return [int(item) for item in _str2list(string) ]
def _str2set(string):     return set([s.strip() for s in string.split(",") if s])
def _str2cmds(string):    return [s.strip() for s in string.split(";") if s]

# TEST_INFO specifications
TESTCNF_KEYWORDS = {
# keyword        : (parser, default, section, description)
# [setup]
"executable"     : (str       , None , "setup", "Name of the executable e.g. abinit"), 
"test_chain"     : (_str2list , ""   , "setup", "Used to defined a ChainOfTest namely that a list of tests that are connected together."),
"need_cpp_vars"  : (_str2set  , ""   , "setup", "CPP variables that should be defined in config.h in order to enable the test."),
"input_prefix"   : (str       , ""   , "setup", "Prefix for input files, mainly used for the ABINIT files file"),
"output_prefix"  : (str       , ""   , "setup", "Prefix for output files, mainly used for the ABINIT files file"),
"input_ddb"      : (str       , ""   , "setup", "The input DDB file read by anaddb"),
"input_gkk"      : (str       , ""   , "setup", "The input GKK file read by anaddb"),
# [files]
"files_to_test"  : (_str2filestotest, "", "files", "List with the output files that will be compared with the reference results.\n" + 
                                                   "format:\n file_name, tolnlines = int, tolabs = float, tolrel = float [,fld_options = -medium]\n" + 
                                                   "tolnlines: the tolerance on the number of differing lines\n" + 
                                                   "tolabs:the tolerance on the absolute error\n" + 
                                                   "tolrel: tolerance on the relative error\n fld_options: options passed to fldiff.pl (optional).\n" +
                                                   "Multiple files are separated by ; e.g.\n" +  
                                                   " foo.out, tolnlines = 2, tolabs = 0.1, tolrel = 1.0e-01;\n" +
                                                   "bar.out, tolnlines = 4, tolabs = 0.0, tolrel = 1.0e-01"
                                                       ), 
"psp_files"      : (_str2list,        "", "files", "List of pseudopotential files (located in the Psps_for_tests directory)."),
"extra_inputs"   : (_str2list,        "", "files", "List of extra input files."),
# [shell]
"pre_commands"   : (_str2cmds, "", "shell", "List of commands to execute before starting the test"),
"post_commands"  : (_str2cmds, "", "shell", "List of commands to execute after the test is completed"),
# [paral_info]
"max_nprocs"     : (int      ,  1 , "paral_info", "Maximum number of MPI processors (1 for sequential run)"),
"nprocs_to_test" : (_str2intlist, "","paral_info","List with the number of MPI processes that should be used for the test"),
"exclude_nprocs" : (_str2intlist, "","paral_info","List with the number of MPI processes that should not be used for the test"),
# [extra_info]
"author"         : (_str2set , "Unknown"                  , "extra_info", "Unknown","Authors of the test"), 
"keywords"       : (_str2set , ""                         , "extra_info", "List of keywords associated to the test"),
"description"    : (str      , "No description available",  "extra_info", "String containing extra information on the test"),
"references"     : (_str2list, "",  "extra_info", "List of references to papers or other articles"),
}

#TESTCNF_SECTIONS = set( [ TESTCNF_KEYWORDS[k][2] for k in TESTCNF_KEYWORDS ] )

# This extra list is hardcoded in order to have a fixed order of the sections in doc_testcfn_format.
# OrderedDict have been introduced in python2.7 sigh!
TESTCNF_SECTIONS = [
  "setup",
  "files",
  "shell",
  "paral_info",
  "extra_info",
]

# consistency check.
for key, tup in TESTCNF_KEYWORDS.items():
  if tup[2] not in TESTCNF_SECTIONS:
    raise ValueError("Please add the new section %s to TESTCNF_SECTIONS" % tup[2])

def line_starts_with_section_or_option(string):
  """True if string start with a TEST_INFO section or option."""
  from re import compile
  re_ncpu = compile("^NCPU_(\d+)$")
  s = string.strip()
  idx = s.find("=")
  if idx == -1: # might be a section.
    if s.startswith("[") and s.endswith("]"):
      if s[1:-1] in TESTCNF_SECTIONS: return 1 # [files]...
      if re_ncpu.search(s[1:-1]):  return 1    # [NCPU_1] ...
  else:
    if s[:idx].strip() in TESTCNF_KEYWORDS: return 2
  return 0

def doc_testcnf_format(fh=sys.stdout):
  """Automatic documentation of the configuration file that describes the test."""
  def writen(string): fh.write(string + "\n") 

  for section in TESTCNF_SECTIONS:
    writen("["+section+"]")
    for key in TESTCNF_KEYWORDS:
       tup = TESTCNF_KEYWORDS[key]
       if section == tup[2]:
         line_parser = tup[0]
         default = tup[1]
         if default is None: default = "Mandatory"
         #print section, key
         desc    = tup[3]
         if default:
           msg = "%s =  %s (%s)" % (key, desc, default)
         else:
           msg = "%s =  %s" % (key, desc)
         writen(msg)

#############################################################################################################

class TestInfo(object):
  """Container storing the options specified in the TEST_INFO section."""

  def __init__(self, dct):
    for k, v in dct.items(): self.__dict__[k] = v

    #if self.nprocs_to_test and self.test_chain:
    #  err_msg = "test_chain and nprocs_to_test are mutually exclusive"
    #  raise TestInfoParserError(err_msg)

    # Add the executable name to the list of keywords.
    self.add_keywords([self.executable])

  @lazy__str__
  def __str__(self): pass

  def add_cpp_vars(self, need_cpp_vars):
    self.need_cpp_vars = self.need_cpp_vars.union(need_cpp_vars)

  def add_keywords(self, keywords):
    self.keywords = self.keywords.union(keywords)

  def make_test_id(self):
    ## FIXME Assumes inp_fname is in the form name.in
    test_id = basename(self.inp_fname).split(".")[0]
    if self.ismulti_parallel: test_id = test_id + str(self.max_nprocs)
    return test_id

  @property
  def ismulti_parallel(self):
    return self._ismulti_paral

class TestInfoParser(object):
  """Parse the TEST_INFO section that describes the test."""

  def __init__(self, inp_fname, defaults=None):
    dbg_print("Parsing TEST_INFO section from input file : " + str(inp_fname))

    self.inp_fname = absp(inp_fname)
    self.inp_dir, x = os.path.split(self.inp_fname) 
    
    SENTINEL = "#%%"
    HEADER   = "<BEGIN TEST_INFO>\n"
    FOOTER   = "<END TEST_INFO>\n"
    #
    # Extract the lines that start with SENTINEL and parse the file.
    lines = lazy_readlines(inp_fname)
    lines = [ l.replace(SENTINEL,"",1).lstrip() for l in lines if l.startswith(SENTINEL) ]

    try: 
      start = lines.index(HEADER)
      stop  = lines.index(FOOTER)
    except ValueError:
      err_msg = "%s does not contain any valid testcnf section!" % inp_fname
      raise TestInfoParserError(err_msg)

    lines = lines[start+1:stop]
    if not lines: 
      err_msg = "%s does not contain any valid testcnf section!" % inp_fname
      raise TestInfoParserError(err_msg)

    # Hack to allow for options that occupy more than one line.
    string = ""
    for l in lines:
      if line_starts_with_section_or_option(l):
        string += l
      else:
        if l.startswith("#"): continue
        string = string.rstrip() + " " + l
    lines = [ l + "\n" for l in  string.split("\n") ]
    #print(lines)

    s = StringIO()
    s.writelines(lines)
    s.seek(0)
 
    class MySafeConfigParser(SafeConfigParser):
      "Wrap the get method of SafeConfigParser to disable the interpolation of raw_options."
      raw_options = ["description",]

      def get(self, section, option, raw=False, vars=None):
        if option in self.raw_options and section == TESTCNF_KEYWORDS[option][2]: 
          #print "Disabling interpolation for section = %s, option = %s" % (section, option)
          return SafeConfigParser.get(self, section, option, raw=True, vars=vars)
        else:
          return SafeConfigParser.get(self, section, option, raw, vars)

    self.parser = MySafeConfigParser(defaults) # Wrap the parser.
    self.parser.readfp(s)

    # Consistency check
    opt = "test_chain"
    section = TESTCNF_KEYWORDS[opt][2]
    pars    = TESTCNF_KEYWORDS[opt][0] 

    if self.parser.has_option(section, opt):
      string = self.parser.get(section, opt)
      chain = pars(string)
      ones = [chain.count(value) for value in chain]
      if sum(ones) != len(ones): 
        err_msg = "%s : test_chain contains repeated tests %s" % (inp_fname, string)
        raise TestInfoParserError(err_msg)

      #
      # Check whether (section, option) is correct.
      #_defs = [s.upper() for s in defaults] if defaults else []
      #err_msg = ""
      #for section in parser.sections():
      #  for opt in parser.options(section): 
      #    if opt.upper() in _defs: continue
      #    if opt not in TESTCNF_KEYWORDS:
      #      err_msg += "Unknown (section, option) = %s, %s\n" % (section, opt)
      #    elif section != TESTCNF_KEYWORDS[opt][2]:
      #      err_msg += "Wrong (section, option) = %s, %s\n" % (section, opt)
      #if err_msg: raise ValueError(err_msg)

  def generate_testinfo_nprocs(self, nprocs):
    """Return a record with the variables needed to handle the job with nprocs."""

    info = Record(); d = info.__dict__

    # First read and parse the global options.
    for key in TESTCNF_KEYWORDS:
      tup = TESTCNF_KEYWORDS[key]
      line_parser = tup[0]
      section = tup[2]
      if section in self.parser.sections():
        try:
          d[key] = self.parser.get(section, key)
        except NoOptionError:
          d[key] = tup[1] # Section exists but option is not specified. Use default value.
      else:
        d[key] = tup[1] # Section does not exist. Use default value.

      # Process the line
      try:
        d[key] = line_parser( d[key] )
      except:
        err_msg = "Wrong line:\n key = %s, d[key] = %s\n in file: %s" % (key, d[key], self.inp_fname)
        raise TestInfoParserError(err_msg)

    # At this point info contains the parsed global values. 
    # Now check if this is a parallel test and, in case, overwrite the values
    # using those reported in the [CPU_nprocs] sections.
    # Set also the value of info._ismulti_paral so that we know how to create the test id 
    if not info.nprocs_to_test:
      assert nprocs == 1
      info._ismulti_paral = False
    else:
      #print("multi parallel case")
      if nprocs not in info.nprocs_to_test:
        err_msg = "in file: %s. nprocs = %s > not in nprocs_to_test = %s" % (self.inp_fname, nprocs, info.nprocs_to_test)
        raise TestInfoParserError(err_msg)

      if nprocs > info.max_nprocs:
        err_msg = "in file: %s. nprocs = %s > max_nprocs = %s" % (self.inp_fname, nprocs, self.max_nprocs)
        raise TestInfoParserError(err_msg)

      # Redefine variables related to the number of CPUs.
      info._ismulti_paral = True
      info.nprocs_to_test = [nprocs]
      info.max_nprocs = nprocs
 
      info.exclude_nprocs = list(range(1, nprocs))
      #print self.inp_fname, nprocs, info.exclude_nprocs

      ncpu_section = "NCPU_" + str(nprocs)
      if not self.parser.has_section(ncpu_section): 
        err_msg = "Cannot find section %s in %s" % (ncpu_section, self.inp_fname)
        raise TestInfoParserError(err_msg)

      for key in self.parser.options(ncpu_section):
        if key in self.parser.defaults(): continue
        opt = self.parser.get(ncpu_section, key)
        tup = TESTCNF_KEYWORDS[key]
        line_parser = tup[0]
        #
        # Process the line and replace the global value.
        try:
          d[key] = line_parser(opt)
        except:
          err_msg = "In file: %s. Wrong line: key = %s, value = " % (self.inp_fname, key, d[key]) 
          raise TestInfoParserError(err_msg)

        #print self.inp_fname, d["max_nprocs"]

    # Add the name of the input file.
    info.inp_fname = self.inp_fname

    return TestInfo(d)

  @property
  def nprocs_to_test(self):
    key     = "nprocs_to_test"
    opt_parser = TESTCNF_KEYWORDS[key][0]
    default    = TESTCNF_KEYWORDS[key][1]
    section    = TESTCNF_KEYWORDS[key][2]

    try:
      opt = self.parser.get(section, key)
    except NoOptionError:
      opt = default 

    return opt_parser(opt)

  @property
  def is_testchain(self):
    opt = "test_chain"
    section = TESTCNF_KEYWORDS[opt][2]
    return self.parser.has_option(section, opt)

  def chain_inputs(self):
    assert self.is_testchain
    opt = "test_chain"
    section = TESTCNF_KEYWORDS[opt][2]
    parse   = TESTCNF_KEYWORDS[opt][0] 
                                             
    fnames = parse(self.parser.get(section, opt))
    return [pj(self.inp_dir, fname) for fname in fnames]

#############################################################################################################
def find_top_build_tree(start_path):
  """
  Return the absolute path of the ABINIT build tree. 
  Raise RuntimeError if build tree is not found.
  Assume that start_path is within the buildtree.
  """
  ntrials = 10
  abs_path = os.path.abspath(start_path)

  trial = 0
  while trial <= ntrials:
    config_h   = os.path.join(abs_path, "config.h")
    abinit_bin = os.path.join(abs_path, "src", "98_main", "abinit")
    # Check if we are in the top of the ABINIT source tree
    if  isfile(config_h) and isfile(abinit_bin): 
      return abs_path
    else:
      abs_path, tail = os.path.split(abs_path)
      trial += 1

  err_msg = "Cannot find the ABINIT build tree after %s trials" % ntrials
  raise RuntimeError(err_msg)

class BuildEnvironment(object):
  """Information on the build directory."""

  def __init__(self, build_dir, cygwin_instdir=None):
    """
    build_dir: Path to the top level directory of the buid.
    cygwin_instdir: Installation directory of cygwin. Defaults to '/cygwin'
    """
    # Try to figure out the top level directory of the build tree.
    try: 
      build_dir = find_top_build_tree(build_dir)
    except:
      raise

    self.uname    = platform.uname()
    self.hostname = gethostname()

    try:
      self.username = os.getlogin()
    except:
      self.username = "No_username"

    self.build_dir        = absp(build_dir)
    self.configh_path     = pj(self.build_dir, "config.h")
    self.binary_dir       = pj(self.build_dir, "src", "98_main")

    self._cygwin_instdir = "/cygwin"
    if cygwin_instdir is not None:
      self._cygwin_instdir = cygwin_instdir

    # Binaries that are not located in src/98_main
    self._external_bins = {
      "atompaw" : pj(self.build_dir, "fallbacks", "exports", "bin", "atompaw-abinit"),
      "timeout" : pj(self.build_dir, "tests", "Nightly", "timeout"),
    }

    # Check if it is a valid ABINIT build tree.
    if not ( isfile(self.configh_path) and isfile(self.path_of_bin("abinit")) ): 
      raise ValueError("%s is not a valid ABINIT build tree." % self.build_dir)

    # Get the list of CPP variables defined in the build.
    self.defined_cppvars = parse_configh_file(self.configh_path)

    if not self.has_bin("timeout"):
       warn("Cannot find timeout executable!")

  @lazy__str__
  def __str__(self): pass

  #def apath_of(self, *p):
  #  "The absolute path of p where p is one or more pathname components.
  #   Use path_of bin if p is a binary file "
  #  return pj(self.build_dir, *p)

  def issrctree(self):
    configac_path  = pj(self.build_dir, "configure.ac")
    abinitF90_path = pj(self.build_dir,"src","98_main","abinit.F90")
    return (isfile(configac_path) and isfile(abinitF90_path))

  def iscygwin(self):
    return "CYGWIN" in self.uname[0].upper()

  def _addext(self, string):
    "Append .exe extension, needed for cygwin"
    if self.iscygwin(): string += ".exe"
    return string

  def path_of_bin(self, bin_name):
    "Return the absolute path of bin_name."

    if bin_name in self._external_bins:
      bin_path = self._external_bins[bin_name]
    else: 
      bin_path = pj(self.binary_dir, bin_name) # It's in src/98_main

    bin_path = self._addext(bin_path)

    # Handle external bins that are installed system wide 
    # (such as atompaw on woopy)
    if bin_name in self._external_bins and not isfile(bin_path):
      # Search it in PATH.
      paths = os.getenv("PATH").split(os.pathsep)
      for p in paths:
        bin_path = pj(p, bin_name)
        if isfile(bin_path): break
      else:
        err_msg = "Cannot find path of bin_name %s, neither in the build directory nor in PATH %s" % (
                   bin_name, paths)
        raise RuntimeError(err_msg)

    return bin_path

  def cygwin_path_of_bin(self, bin_name):
    path = self.path_of_bin(bin_name)
    if self.iscygwin(): path = self._cygwin_instdir + path
    return path

  def has_bin(self, bin_name):
    "True if binary bin_name is present in the build."
    return isfile( self.path_of_bin(bin_name) )

  def cygwin_path(self, path):
    apath = absp(path)
    if self.iscygwin(): apath = self._cygwin_instdir + apath
    return apath

#############################################################################################################

def parse_configh_file(fname):
  """
  parse config.h, return list with the CCP variables that are #defined.
  Not very robust: does not handle instructions such as 

  #ifdef HAVE_FOO 
  #  define HAVE_BAR 1
  #endif

  Handling this case would require a real preprocessing with CPP and then the parsing.
  Not easy to implement in a portable way especially on IBM machines with XLF.
  """
  defined_cppvars = []
  fh = open(fname, "r")
  for l in fh:
    l = l.lstrip()
    if l.startswith("#define "):
      tokens = l.split()
      varname = tokens[1]
      if varname.startswith("HAVE_") and len(tokens) >= 3: 
        value = int(tokens[2])
        if value != 0: defined_cppvars.append(varname)
  fh.close()
  return defined_cppvars

#############################################################################################################

class FldiffResult(object): 
  """Container storing the results produced by fldiff.pl."""

  _attrbs = {
      "fname1"          : "first file provided to fldiff.",
      "fname2"          : "second file provided to fldiff.",
      "options"         : "options passed to fldiff.",
      "summary_line"    : "Summary given by fldiff.",
      "fatal_error"     : "True if file comparison cannot be done.",
      "ndiff_lines"     : "Number of different lines.",
      "abs_error"       : "Max absolute error.",
      "rel_error"       : "Max relative error.",
      "max_absdiff_ln"  : "Line number where the Max absolute error occurs.",
      "max_reldiff_ln"  : "Line number where the Max relative error occurs.",
  }

  def __init__(self, summary_line, err_msg, fname1, fname2, options):

    self.summary_line = summary_line.strip()
    self.err_msg      = err_msg.strip()
    self.fname1       = fname1 
    self.fname2       = fname2
    self.options      = options

    self.fatal_error  = False
    self.success      = False

    if "fatal" in summary_line:
      self.fatal_error = True
    elif "no significant difference" in summary_line:
      self.success     = True
      self.ndiff_lines = 0
      self.abs_error   = 0.0
      self.rel_error   = 0.0
    elif "different lines=" in summary_line:
      #Summary Case_84 : different lines= 5 , max abs_diff= 1.000e-03 (l.1003), max rel_diff= 3.704e-02 (l.1345)
      tokens = summary_line.split(",")
      for tok in tokens:
        if "different lines=" in tok:
          self.ndiff_lines = int(tok.split("=")[1])
        if "max abs_diff=" in tok:
          vals = tok.split("=")[1].split()
          self.abs_error = float(vals[0])
        if "max rel_diff=" in tok:
          vals = tok.split("=")[1].split()
          self.rel_error = float(vals[0])
    else:
      err_msg = "Wrong summary_line: " + str(summary_line)
      #raise ValueError(err_msg)
      warn(err_msg)
      self.fatal_error = True

  @lazy__str__
  def __str__(self): pass

  def passed_within_tols(self, tolnlines, tolabs, tolrel):

    status = "succeeded"; msg = ""
    if self.fatal_error:
      status  = "failed"
      msg     = "fldiff.pl fatal error:\n" + self.err_msg
    elif self.success:
      msg = "succeeded"
    else:
      abs_error   = self.abs_error
      rel_error   = self.rel_error
      ndiff_lines = self.ndiff_lines
      status = "failed"; fact = 1.5

      locs = locals()
      if abs_error > tolabs * fact: 
        msg = "failed: absolute error %(abs_error)s > %(tolabs)s" % locs
      elif rel_error > tolrel * fact: 
        msg = "failed: relative error %(rel_error)s > %(tolrel)s" % locs
      elif ndiff_lines > tolnlines:
        msg = "failed: erroneous lines %(ndiff_lines)s > %(tolnlines)s" % locs
      # FIXME passed or failed?
      elif abs_error > tolabs:
        msg = "within 1.5 of tolerance (absolute error %(abs_error)s, accepted %(tolabs)s )" % locs
      elif rel_error > tolrel:
        msg = "within 1.5 of tolerance (relative error %(rel_error)s, accepted %(tolrel)s )" % locs
      else:
        status = "passed"
        msg = "passed: absolute error %(abs_error)s < %(tolabs)s, relative error %(rel_error)s < %(tolrel)s" % locs

    isok = status in ["passed", "succeeded"]

    return (isok, status, msg) 

def wrap_fldiff(fldiff_path, fname1, fname2, opts=None, label=None, timebomb=None, out_filobj=sys.stdout):
  """Wraps fldiff.pl, return instance of FldiffResult."""
  # Usage: fldiff [-context] [ -ignore | -include ] [ -ignoreP | -includeP ] [ -easy | -medium | -ridiculous ] file1 file2 [label]
  #
  fld_options = "-ignore -ignoreP"	# Default options for fldiff script.
  if opts: fld_options = " ".join([fld_options] + [o for o in opts])
  fld_options = [s for s in fld_options.split()]

  if label is None: label = ""

  args = ["perl", fldiff_path] + fld_options + [fname1, fname2, label]
  cmd_str = " ".join(args)
  #print "about to execute ", cmd_str
  #start = time.time()

  if True or timebomb is None:
    #p = Popen(args, shell=True, stdout=PIPE, stderr=PIPE)
    p = Popen(cmd_str, shell=True, stdout=PIPE, stderr=PIPE)
    (stdout_data, stderr_data) = p.communicate()
    ret_code = p.returncode
    #ret_code = p.wait()
  else:
    #(p, ret_code) = timebomb.run(args, shell=False, stdout=PIPE, stderr=PIPE)
    (p, ret_code) = timebomb.run(cmd_str, shell=True, stdout=PIPE, stderr=PIPE)

  #print "fldiff done in %.2f s" % (time.time()-start)

  MAGIC_FLDEXIT = 4 # fldiff returns this value when some difference is found.    
                    # perl programmers have a different understanding of exit_status!
  err_msg = ""
  if ret_code not in [0, MAGIC_FLDEXIT]:
    #err_msg = p.stderr.read()
    err_msg = stderr_data

  lines = stdout_data.splitlines(True)
  #lines = p.stdout.readlines()
  if out_filobj and not hasattr(out_filobj, "writelines"): # Assume string
     lazy_writelines(out_filobj, lines)
  else:
    out_filobj.writelines(lines)

  # Parse the last line.
  try:
    summary_line = lines[-1] 
  except IndexError:
    summary_line = "fatal error: no summary line received from fldiff"

  return FldiffResult(summary_line, err_msg, fname1, fname2, fld_options)

#############################################################################################################

def test_from_input(inp_fname, abenv, keywords=None, need_cpp_vars=None, with_np=1):
  """
  Factory function for one test
  """
  np = with_np

  inp_fname = absp(inp_fname)
  #
  # Warning: Assumes inp_fname in the form name.in
  tin = os.path.basename(inp_fname)

  try: # Assumes some_path/Input/t30.in
    inpdir_path, x = os.path.split(inp_fname) 
  except:
    raise ValueError("%s is not a valid path" % inp_fname)

  parser = TestInfoParser(inp_fname)

  nprocs_to_test = parser.nprocs_to_test
  ntests = len(nprocs_to_test)  

  if ntests == 0:
    nprocs_to_test = [1,]
    ntests = 1

  test_info = parser.generate_testinfo_nprocs(with_np)

  # Add global cpp variables.
  test_info.add_cpp_vars(need_cpp_vars)

  # Add global keywords.
  test_info.add_keywords(keywords)

  # Single test with np processors.
  # Istanciate the appropriate subclass depending on the name of the executable. Default is BaseTest.
  cls = exec2class(test_info.executable)
  return cls(test_info, abenv)

def tests_from_inputs(input_fnames, abenv, keywords=None, need_cpp_vars=None):
  """
  Factory function. Return a list of tests generated from the TEST_INFO section reported
  in the input files inp_fnames.
  """

  if isinstance(input_fnames, str): input_fnames = [input_fnames,]

  inp_fnames = [absp(p) for p in input_fnames]

  out_tests = []

  while True:
    #
    try:
      inp_fname = inp_fnames.pop(0)
    except IndexError:
      break

    try: # Assumes some_path/Input/t30.in
      inpdir_path, x = os.path.split(inp_fname) 
    except:
      raise ValueError("%s is not a valid path" % inp_fname)

    parser = TestInfoParser(inp_fname)

    nprocs_to_test = parser.nprocs_to_test
    if len(nprocs_to_test) == 0: 
      nprocs_to_test = [1,]

    if not parser.is_testchain:

      for np in nprocs_to_test:
        # Generate single test with np MPI processors.
        test_info = parser.generate_testinfo_nprocs(np)

        test_info.add_cpp_vars(need_cpp_vars) # Add global cpp variables.
        test_info.add_keywords(keywords)      # Add global keywords.

        # Istanciate the appropriate subclass depending on the name of the executable. Default is BaseTest.
        cls = exec2class(test_info.executable)
        out_tests.append(cls(test_info, abenv))

    else:
      #print "got chain input ", inp_fname
      #print  parser.chain_inputs()
      # Build the chain with np nprocessors.
      for np in nprocs_to_test:
        tchain_list = []
        for cht_fname in parser.chain_inputs():
          t = test_from_input(cht_fname, abenv, keywords=keywords, need_cpp_vars=need_cpp_vars, with_np=np)
          tchain_list.append(t)

        if not tchain_list:
          err_msg = "tchain_list is empty, inp_fname",inp_fname
          raise RuntimeError(err_msg)

        out_tests.append(ChainOfTests(tchain_list))

      # Remove the input files of the chain
      for s in parser.chain_inputs()[1:]:
        try:
          idx = inp_fnames.index(s)
        except ValueError:
          err_msg = "%s not found in inp_fnames" % inp_fnames
          #print inp_fnames
          raise RuntimeError(err_msg)

        inp_fnames.pop(idx)

  return out_tests

#############################################################################################################
class TestError(Exception): 
  pass

class BaseTest(object):
  """
  Base class describing a single test. Tests associated to other executables should 
  sublcass BaseTest and redefine the method make_stdin.
  Then change exec2cls so that the appropriate instance is returned.
  """
  _possible_status = ["failed", "passed", "succeeded", "skipped", "disabled",]

  def __init__(self, test_info, abenv):
    dbg_print("Initializing BaseTest from inp_fname: ", test_info.inp_fname)

    self.inp_fname  = absp(test_info.inp_fname)
    self.abenv      = abenv
    self.id         = test_info.make_test_id() # The test identifier (takes into account the multi_parallel case)

    # FIXME Assumes inp_fname is in the form tests/suite_name/Input/name.in
    suite_name = os.path.dirname(self.inp_fname)
    suite_name = os.path.dirname(suite_name)

    self.suite_name = basename(suite_name)
    self.ref_dir    = abenv.apath_of("tests", suite_name, "Refs")
    self.inp_dir    = abenv.apath_of("tests", suite_name, "Input")

    self._executed = False
    self._status = None
    if basename(self.inp_fname).startswith("-"): self._status = "disabled"

    # Initial list of local files that should not be removed.
    self._files_to_keep = [] 

    # Default values.
    self.make_html_diff = 0   # 0 => Do not produce diff files in HTML format
                              # 1 => Produced HTML diff but only if test failed
                              # 2 => Produce HTML diff independently of the final status

    self.sub_timeout = 30     # Timeout for subprocesses (in seconds)

    self.erase_files = 2      # 0 => Keep all files.
                              # 1 => Remove files but only if the test passes or succeeds 
                              # 2 => Remove files even when the test fail.

    # Inglobate the attributes of test_info in self.
    err_msg = ""
    for k in test_info.__dict__: 
      if k in self.__dict__ and test_info.__dict__[k] != self.__dict__[k]: 
        err_msg += "Cannot overwrite key %s\n" % k
        #print(test_info.__dict__[k],  self.__dict__[k])
    if err_msg: raise TestError(err_msg)

    self.__dict__.update(test_info.__dict__)

    # Save authors' second names to speed up the search.
    # Well, let's hope that we don't have authors with the same second name!
    second_names = list()
    for string in self.author:
      idx = string.rfind(".") 
      f, s = ("", string)
      if idx != -1:
        try:
          f, s = string[:idx+1], string[idx+2:]
        except IndexError:
          raise ValueError("Wrong author name")

      if not f and s and s != "Unknown": 
        err_msg = "author first name is missing in file %s, string = %s " %(self.full_id, string)
        print err_msg
        #raise ValueError(err_msg)
      second_names.append(s)

    self._authors_snames = set(second_names)

  @lazy__str__
  def __str__(self): pass

  def stdin_readlines(self):  return lazy_readlines(self.stdin_fname)
  def stdin_read(self):       return lazy_read(self.stdin_fname)
  def stdout_readlines(self): return lazy_readlines(self.stdout_fname)
  def stdout_read(self):      return lazy_read(self.stdout_fname)
  def stderr_readlines(self): return lazy_readlines(self.stderr_fname)
  def stderr_read(self):      return lazy_read(self.stderr_fname)

  @property
  def full_id(self):
    return "["+self.suite_name+"]["+self.id+"]"

  @property
  def bin_path(self):
    return self.build_env.path_of_bin(self.executable)

  @property
  def cygwin_bin_path(self):
    return self.build_env.cygwin_path_of_bin(self.executable)

  def cygwin_path(self, path):
    return self.build_env.cygwin_path(path)

  def cpkl_dump(self, protocol=-1):
    """Save the instance in a cpickle file"""
    self.cpkl_fname = pj(self.workdir, self.id +".cpkl")
    fh = open(self.cpkl_fname,"wb")
    cpkl.dump(self, fh, protocol=protocol)
    self.keep_files(self.cpkl_fname)
    fh.close()

  def has_keywords(self, keywords):
    "True if test has keywords"
    return set(keywords).issubset(self.keywords)

  def has_authors(self, authors):
    return set(authors).issubset(self._authors_snames)

  def readme(self, width=100, html=True, abslink=True):
    string = self.description.lstrip()
    if self.references: string += "References:\n" + "\n".join(self.references)
    string = textwrap.dedent(string)
    string = textwrap.fill(string, width=width)
    if not html:
      return self.full_id + ":\n" + string
    else:  
      if abslink:
        link = html_link(self.full_id, self.inp_fname)
      else:
        # Use relative path so that we can upload the HTML file on 
        # the buildbot master and browse the pages.
        link = html_link(self.full_id, basename(self.inp_fname))
      string = link + "<br>" + string.replace("\n","<br>") +"\n"
    return string

  def make_stdin(self):
    t_stdin = StringIO()
    fh = open(self.inp_fname, "r") 
    t_stdin.writelines(fh)
    fh.close()
    t_stdin.flush()
    t_stdin.seek(0)
    return t_stdin

  def get_extra_inputs(self):
    "Copy extra inputs from inp_dir to workdir"
    #
    # First copy the main input file (useful for debugging the test)
    # Avoid raising exceptions as python threads do not handle them correctly.
    try:
       src  = self.inp_fname
       dest = pj(self.workdir, basename(self.inp_fname))
       shutil.copy(src, dest)
       self.keep_files(dest)  # Do not remove it after the test.
    except:
       self.exceptions.append( TestError("copying %s => %s" % (src,dest)) )

    for extra in self.extra_inputs:
      src  = pj(self.inp_dir, extra)
      dest = pj(self.workdir, extra)

      if not isfile(src): 
        self.exceptions.append( TestError("%s: no such file" % src) )
        continue

      shutil.copy(src, dest)
      if dest.endswith(".gz"):  # Decompress the file
        unzip(dest)
        dest = dest[:-3]
      #self.keep_files(dest)  # Do not remove dest after the test.

  @property
  def inputs_used(self):
    inputs = [self.inp_fname] + [pj(self.inp_dir, f) for f in self.extra_inputs]
    #
    # Add files appearing in the shell sections.
    for cmd_str in (self.pre_commands + self.post_commands):
      if cmd_str.startswith("iw_"): 
        tokens = cmd_str.split()
        inp = pj(self.inp_dir, tokens[1])
        inputs.append(inp)
    return inputs

  @property
  def status(self):
    if self._status in ["disabled", "skipped"]: return self._status
    all_fldstats = [f.fld_status for f in self.files_to_test]
    if "failed" in all_fldstats: return "failed"
    if "passed" in all_fldstats: return "passed"
    assert my_all( [ s == "succeeded" for s in all_fldstats])
    return "succeeded"

  @property
  def isok(self):
    return self.fld_isok and not self.exceptions

  @property
  def files_to_keep(self):
    return self._files_to_keep

  def keep_files(self, files):
    if isinstance(files, str):
      self._files_to_keep.append(files)
    else:
      self._files_to_keep.extend(files)

  def compute_nprocs(self, build_env, nprocs):
    """
    Compute the number of MPI processes that can be used for the test.
    Return (nprocs, string) 
    where nprocs = 0 if self cannot be executed, and string explains the 
    reason why the test is skipped.

    Note: A test cannot be executed if: 
      1) It requires some CPP variables that are not defined in the build.
      2) User asks for more MPI nodes than max_nprocs
      3) We have a multiparallel test (e.g. paral/tA.in) and nprocs is not in in nprocs_to_test
      4) nprocs is in exclude_nprocs
    """
    # !HAVE_FOO --> HAVE_FOO should not be present.
    err_msg = ""
    for var in self.need_cpp_vars: 
      if not var.startswith("!") and var not in build_env.defined_cppvars: 
          err_msg += "Build environment does not define the CPP variable %s\n" % var
      elif var[1:] in build_env.defined_cppvars: 
          err_msg += "Build environment defines the CPP variable %s\n" % var[1:]
                                                                                           
    if nprocs > self.max_nprocs:
      err_msg += "nprocs: %s > max_nprocs: %s\n" % (nprocs, self.max_nprocs)
                                                                                           
    if self.nprocs_to_test and nprocs != self.nprocs_to_test[0]:
      err_msg += "nprocs: %s != nprocs_to_test: %s\n" % (nprocs, self.nprocs_to_test[0])

    if nprocs in self.exclude_nprocs:
      err_msg += "nprocs: %s in exclude_nprocs: %s\n" % (nprocs, self.exclude_nprocs)
                                                                                           
    if err_msg: 
      real_nprocs = 0
      err_msg = self.full_id + ":\n" + err_msg
    else:
      real_nprocs = min(self.max_nprocs, nprocs)

    return (real_nprocs, err_msg)

  def run(self, build_env, runner, workdir, nprocs=1, **kwargs):
    """
    Run the test with nprocs MPI nodes in the build environment build_env using the JobRunner runner.
    Results are produced in directory workdir. kwargs is used to pass additional options
    This method must be thread-safe, DO NOT change build_env or runner.
    """

    import copy
    runner = copy.deepcopy(runner)
    start_time = time.time()

    workdir = absp(workdir)
    if not pexists(workdir): os.mkdir(workdir)
    self.workdir = workdir
                                               
    self.build_env = build_env

    self.exceptions = []
    self.fld_isok = True  # False if at least one file comparison fails.

    # Extract options from kwargs
    self.erase_files    = kwargs.get("erase_files", self.erase_files)
    self.make_html_diff = kwargs.get("make_html_diff", self.make_html_diff)
    self.sub_timeout    = kwargs.get("sub_timeout", self.sub_timeout)

    timeout = self.sub_timeout
    if self.build_env.has_bin("timeout") and timeout > 0.0:
      exec_path = self.build_env.path_of_bin("timeout")
      self.timebomb = TimeBomb(timeout, delay=0.05, exec_path = exec_path) 
    else:
      self.timebomb = TimeBomb(timeout, delay=0.05)

    str_colorizer = StringColorizer(sys.stdout)

    status2txtcolor = {
      "succeeded" : lambda string : str_colorizer(string,"green"),
      "passed"    : lambda string : str_colorizer(string,"blue"),
      "failed"    : lambda string : str_colorizer(string,"red"),
      "disabled"  : lambda string : str_colorizer(string,"cyan"),
      "skipped"   : lambda string : str_colorizer(string,"cyan"),
    }

    # Check whether the test can be executed.
    can_run = True
    if self._status == "disabled":
      msg = self.full_id + ": Disabled"
      can_run = False
      print status2txtcolor[self._status](msg)

    (self.nprocs, self.skip_msg) = self.compute_nprocs(self.build_env, nprocs)

    if self.skip_msg:
      self._status = "skipped"
      msg = self.full_id + ": Skipped"
      print status2txtcolor[self._status](msg)
      can_run = False

    self.run_etime = 0.0
    if can_run:
      # Execute pre_commands in workdir.
      rshell = RestrictedShell(self.inp_dir, self.workdir, self.abenv.psps_dir)

      for cmd_str in self.pre_commands: 
        rshell.execute(cmd_str) 
       
      if rshell.exceptions:
        self.exceptions.extend(rshell.exceptions)
        rshell.empty_exceptions()

      # Copy extra inputs in workdir (if any).
      self.get_extra_inputs()

      # Create stdin file in the workdir.
      self.stdin_fname  = pj(workdir, self.id + ".stdin")
      self.stdout_fname = pj(workdir, self.id + ".stdout")
      self.stderr_fname = pj(workdir, self.id + ".stderr")

      self.keep_files([self.stdin_fname, self.stdout_fname, self.stderr_fname])

      # Create input file.
      t_stdin = self.make_stdin()
      fh = open(self.stdin_fname, "w")
      fh.writelines(t_stdin)
      fh.close()

      # Run the code (run_etime is the wall time spent to execute the test)
      if runner.has_mpirun:
        bin_path = self.cygwin_bin_path
      else:
        bin_path = self.bin_path

      #try: 
      self.run_etime = runner.run(self.nprocs, bin_path, self.stdin_fname, self.stdout_fname, self.stderr_fname, cwd=workdir)
      #except (JobRunnerError,), job_err:
      #  #print job_err
      #  self.run_etime  = job_err.run_etime
      #  self.exceptions.append(job_err)
      if runner.exceptions:
        for exc in runner.exceptions: print exc
        self.exceptions.extend(runner.exceptions)

      # Execute post_commands in workdir.
      for cmd_str in self.post_commands: 
        rshell.execute(cmd_str) 

      if rshell.exceptions:
        self.exceptions.extend(rshell.exceptions)
        rshell.empty_exceptions()
      
      # Check final results: 
      # 1) use fldiff to compare ref and output files.
      # 2) fldiff stdout is redirected to fldiff_fname.
      for f in self.files_to_test:
        fldiff_fname = pj(self.workdir, f.name + ".fldiff")
        self.keep_files(fldiff_fname)
        fh = open(fldiff_fname,"w")

        f.fldiff_fname = fldiff_fname

        (isok, status, msg) = f.compare(self.abenv.fldiff_path, self.ref_dir, self.workdir, 
                                        timebomb = self.timebomb, 
                                        outf=fh)

        fh.close()

        self.keep_files( pj(self.workdir, f.name) )
        self.fld_isok = self.fld_isok and isok

        msg = " ".join([self.full_id, msg])
        print status2txtcolor[status](msg)

    self._executed = True
    self.tot_etime = time.time() - start_time

  def clean_workdir(self, other_test_files=None):
    "Remove the files produced in self.workdir."
    assert self._executed
    if not pexists(self.workdir) or self.erase_files == 0: return

    save_files = self._files_to_keep
    if other_test_files is not None: save_files += other_test_files

    if ( (self.erase_files == 1 and self.isok ) or self.erase_files == 2 ):
      entries = [ pj(self.workdir, e) for e in os.listdir(self.workdir)]
      for entry in entries:
        if entry in save_files: continue
        if isfile(entry): 
          os.remove(entry)
        else:
          raise NotImplementedError("Found directory: %s in workdir!!" % entry)

  def patch(self, patcher=None):
    assert self._executed
    for f in self.files_to_test:
      ref_fname = absp( pj(self.ref_dir, f.name) )
      out_fname = absp( pj(self.workdir,f.name) )
      raise NotImplementedError("patcher should be tested")
      from tests.pymods import Patcher
      Patcher(patcher).patch(out_fname, ref_fname)
                                                                                                                                  
  def make_html_diff_files(self):
    "Generate and write diff files in HTML format."

    assert self._executed
    if ( self.make_html_diff == 0 or 
         self._status in ["disabled", "skipped"] ): return 

    diffpy = self.abenv.apath_of("tests", "pymods", "diff.py")
                                                                                      
    for f in self.files_to_test:
      if f.fld_isok and self.make_html_diff == 1: 
        continue

      ref_fname = absp( pj(self.ref_dir, f.name) )

      if not isfile(ref_fname) and ref_fname.endswith(".stdout"): 
        ref_fname = ref_fname[:-7] + ".out"  # FIXME Hack due to the stdout-out ambiguity

      out_fname = absp( pj(self.workdir,f.name) )

      # Check whether output and ref file exist.
      out_exists = isfile(out_fname) 
      ref_exists = isfile(ref_fname) 

      hdiff_fname = absp( pj(self.workdir, f.name+".diff.html") )

      f.hdiff_fname = hdiff_fname
     
      x, ext = os.path.splitext(f.name)
      safe_hdiff = ext in [".out", ".stdout"] # Create HTML diff file only for these files

      if ref_exists and out_exists and safe_hdiff:
        out_opt = "-m"    # HTML side by side diff (can get stuck)
        #out_opt = "-t"   # For simple HTML table. (can get stuck)
        #args = ["python", diffpy, out_opt, "-f " + hdiff_fname, out_fname, ref_fname ]
        args = [diffpy, out_opt, "-f " + hdiff_fname, out_fname, ref_fname ]
        cmd = " ".join(args)
        #print "Diff", cmd

        (p, ret_code) = self.timebomb.run(cmd, shell=True, cwd=self.workdir)

        if ret_code != 0: 
          err_msg = "Timeout error (%s s) while executing %s, retcode = %s" % (
            self.timebomb.timeout, str(args), ret_code)
          self.exceptions.append(TestError(err_msg))
        else:
          self.keep_files(hdiff_fname)

  def write_pyreport(self, fh=None):
    """Write a python module that summarizes the results of the test."""
    assert self._executed

    close_fh = False
    if fh is None:
      fname = pj(self.workdir, "__test_report__.py")
      fh = open(fname, "w")
      close_fh = True
 
    self.keep_files(fh.name)

    # helper function.
    def writen(string, indent=4, nn=1): fh.write(indent*" " + string + nn*"\n") 

    writen("def report_" + self.id + "():", indent=0)
    writen("status = '" + str(self.status) + "'")
                                                                                       
    keys = ["cpkl_fname",] #"exceptions", "tot_etime", ] #"stdin_fname", "stdout_fname", "stderr_fname", 
    for k in keys:
      value = self.__dict__.get(k, None)
      if value is not None:
        writen( k + " = " + str(value) )
      else:
        writen( k + " = ''")
                                                                                       
    writen("return locals()")  # We'll use introspection to retrieve the data
    writen(100*"#", indent=0)  # without having to write a parser.

    if close_fh: fh.close()

  def write_html_report(self, fh=None, oc="oc"):
    """Write a HTML file summarizing the results of the test."""
    assert self._executed

    close_fh = False
    if fh is None:
      close_fh = True
      html_report = pj(self.workdir, "test_report.html")
      fh = open(html_report, "w")

    self.keep_files(fh.name)
    
    self.make_html_diff_files()

    # Try to read stdout and stderr.
    # Ignore errors (fock takes years to flush the stdout)
    stdout_text, stderr_text = 2*("",)
    nlast = 120
    if not self.fld_isok:
      try:
        stderr_text = str2html(self.stderr_read())
        stdout_text = str2html(tail_file(self.stdout_fname, nlast))
      except:
        pass

    ##################################################
    # Document Name Space that serves as the substitution 
    # namespace for instantiating a doc template).
    try:
      username = os.getlogin()
    except:
      username = "No_username"

    DNS = {
      "self"            : self,
      "page_title"      : "page_title",
      "user_name"       : username,
      "hostname"        : gethostname(),
      "Headings"        : ['File_to_test', 'Status', 'fld_output', 'fld_options', 'html_diff'] , 
      "nlast"           : nlast,
      "stderr_text"     : stderr_text,
      "stdout_text"     : stdout_text,
      # Functions and modules available in the template.
      "time"            : time,
      "pj"              : os.path.join,
      "basename"        : os.path.basename,
      "str2html"        : str2html,
      "sec2str"         : sec2str,
      "args2htmltr"     : args2htmltr,
      "html_link"       : html_link,
      "status2html"     : status2html
    }

    header = """
    <html>
     <head><title>$page_title</title></head>
     <body bgcolor="#FFFFFF" text="#000000">
    """

    if self.status in ["skipped", "disabled"]: 
      if self.status == "skipped": 
        template = str2html(self.skip_msg)
      else:
        template = "This test has been disabled!"
    else:
      template = """
        <hr>
        <h1>Results of test ${self.full_id}</h1>
           MPI nprocs =  ${self.nprocs}, 
           run_etime = ${sec2str(self.run_etime)} s, 
           tot_etime = ${sec2str(self.tot_etime)} s
         <br>
         ${html_link("stdin",  basename(self.stdin_fname))}, 
         ${html_link("stdout", basename(self.stdout_fname))},
         ${html_link("stderr", basename(self.stderr_fname))} 
        <p>
        <table width="100%" border="0" cellspacing="0" cellpadding="2">
          <tr valign="top" align="left">
          <py-open code = "for h in Headings:"> </py-open>
            <th>${h}</th>
          <py-close/>
          </tr>
          <py-open>for idx, f in enumerate(self.files_to_test):</py-open>
           <tr valign="top" align="left">
            <py-line code = "fld_link = html_link(basename(f.fldiff_fname))"/>
            <py-line code = "hdiff_link = html_link(basename(f.hdiff_fname))"/>
            <py-line code = "tab_row = args2htmltr(f.name, status2html(f.fld_status), fld_link, f.fld_options, hdiff_link)"/> 
            ${tab_row}
           </tr>
          <py-close/>
        </table>

        <py-open>for idx, f in enumerate(self.files_to_test):</py-open>
          <py-open code="if f.fld_status != 'succeeded':"/>
          <p> ${f.name} ${f.fld_msg} </p>
        <py-close/>

        <py-open code="if not self.fld_isok:"/>
          <py-open code="if self.exceptions:"/>
            <hr><p>
            <h1>Exceptions raised at run-time:</h1>
            <py-open code="for idx, e in enumerate(self.exceptions):"/>
              <p> $idx) ${str2html(str(e))}</p>
            <py-close/>
            <br>
          <py-close/>
          <hr><p>
          <h1>Standard Error of test ${self.id}:</h1>
            ${stderr_text}
          <hr><p>
          <h1>Standard output of test ${self.id} (last ${nlast} lines):</h1>
            ${stdout_text}
          <br>
        <py-close/>
        <p>
        <h3>Extra Information</h3>
        <py-line code = "authors = ', '.join([a for a in self.author])" />
        <p>Authors = ${authors}</p>
        <py-line code = "keys = ', '.join([k for k in self.keywords])" />
        <p>Keywords = ${keys}</p>
        <p>${self.readme(abslink=False)}</p>
      """

    footer = """
      <hr>
      Automatically generated by %s on %s. Logged on as %s@%s
      <hr>
      </body>
      </html> """ % (_MY_NAME, time.asctime(), username, gethostname())

    if "o" in oc: template = header + template
    if "c" in oc: template += footer

    # Set a filelike object to template 
    template_stream = StringIO(template)
    
    # Initialise an xyaptu xcopier, and call xcopy
    xcp = xcopier(DNS, ouf=fh)
    xcp.xcopy(template_stream)

    if close_fh: fh.close()

#############################################################################################################
#
# Subclasses needed to handle the different executables
#
#############################################################################################################

class AbinitTest(BaseTest):
  """Redefine the make_stdin method of BaseTest"""

  def make_stdin(self):
    t_stdin = StringIO()

    inp_fname = self.cygwin_path(self.inp_fname)
    t_stdin.write( inp_fname + "\n")

    out_fname = self.id + ".out"
    t_stdin.write( out_fname + "\n")

    # Prefix for input-output-temporary files
    if self.input_prefix:
      i_prefix = self.input_prefix
    else:
      i_prefix = self.id + "i"

    # Prefix for input-output-temporary files
    if self.output_prefix:
      o_prefix = self.output_prefix
    else:
      o_prefix = self.id + "o"

    t_prefix = self.id #+ "t"

    t_stdin.writelines( [l + "\n"  for l in [i_prefix, o_prefix, t_prefix]] )

    # Path to the pseudopotential files.
    # 1) pp files are searched in pspd_dir first then in workdir.
    psp_paths = [ pj(self.abenv.psps_dir, pname) for pname in self.psp_files ]

    for idx, psp in enumerate(psp_paths):
      if not isfile(psp): 
         pname = pj(self.workdir, basename(psp))
         if isfile(pname):
           psp_paths[idx] = pname # Use local pseudo.
         else:
           err_msg = "Cannot find pp file %s, neither in Psps_for_tests nor in self.workdir" % pname
           self.exceptions.append( TestError(err_msg) )

    psp_paths = [ self.cygwin_path(p) for p in psp_paths ] # Cygwin

    t_stdin.writelines( [p + "\n"  for p in psp_paths] )

    t_stdin.flush()
    t_stdin.seek(0)
    return t_stdin

#############################################################################################################

class AnaddbTest(BaseTest):

  def make_stdin(self):
    t_stdin = StringIO()

    inp_fname = self.cygwin_path(self.inp_fname)  # cygwin
    t_stdin.write( inp_fname + "\n")          # 1) formatted input file
    t_stdin.write( self.id + ".out" + "\n")   # 2) formatted output file e.g. t13.out

    iddb_fname = self.id + ".ddb.in" 
    if self.input_ddb:  
      iddb_fname = pj(self.workdir, self.input_ddb)  # Use output DDB of a previous run.

      if not isfile(iddb_fname): 
        self.exceptions.append( TestError("%s no such DDB file: " % iddb_fname) )

      iddb_fname = self.cygwin_path(iddb_fname)   # cygwin

    t_stdin.write( iddb_fname + "\n")         # 3) input derivative database e.g. t13.ddb.in

    t_stdin.write( self.id + ".md" + "\n")    # 4) output molecular dynamics e.g. t13.md

    input_gkk = self.id + ".gkk"
    if self.input_gkk: 
      input_gkk = pj(self.workdir, self.input_gkk) # Use output GKK of a previous run.
      if not isfile(input_gkk): 
        self.exceptions.append( TestError("%s no such GKK file: " % input_gkk) )

      input_gkk = self.cygwin_path(input_gkk)    # cygwin

    t_stdin.write( input_gkk + "\n")         # 5) input elphon matrix elements  (GKK file) :
    t_stdin.write( self.id + "\n")           # 6) base name for elphon output files e.g. t13

    input_ddk = self.id + ".ddk"
    if not isfile(input_ddk): # Try in input directory:
      input_ddk = pj(self.inp_dir, input_ddk)
                               # FIXME: Someone has to rewrite the treatment of the anaddb files file
      input_ddk = self.cygwin_path(input_ddk)

    t_stdin.write( input_ddk + "\n")   # 7) file containing ddk filenames for elphon/transport :

    t_stdin.flush()
    t_stdin.seek(0)
    return t_stdin

#############################################################################################################

class AimTest(BaseTest):

  def make_stdin(self):
    t_stdin = StringIO()

    inp_fname = self.cygwin_path(self.inp_fname)
    t_stdin.write( inp_fname + "\n")         # formatted input file e.g. .../Input/t57.in

    iden_fname = self.id + "i_DEN" 
    t_stdin.write( iden_fname + "\n")        # input density  e.g. t57i_DEN 
    t_stdin.write( self.id + "\n")           # t57

    # Path to the pseudopotential files.
    psp_paths = [ pj(self.abenv.psps_dir, pname) for pname in self.psp_files ]
    psp_paths = [ self.cygwin_path(p) for p in psp_paths ] # Cygwin

    t_stdin.writelines( [p + "\n"  for p in psp_paths] )
    #                                                                                                 
    t_stdin.flush()
    t_stdin.seek(0)
    return t_stdin

#############################################################################################################

class ConductiTest(BaseTest):
  def make_stdin(self):
    t_stdin = StringIO()

    inp_fname = self.cygwin_path(self.inp_fname)
    t_stdin.write( inp_fname + "\n")  # formatted input file e.g. .../Input/t57.in
    t_stdin.write( self.id + "\n")    # will be used as the prefix of the log file names e.g. t57
    #                                                                                                 
    t_stdin.flush()
    t_stdin.seek(0)
    return t_stdin

#############################################################################################################

class LwfTest(BaseTest):
  def make_stdin(self):
    t_stdin = StringIO()

    inp_fname = self.cygwin_path(self.inp_fname)
    t_stdin.write( inp_fname + "\n")     # formatted input file e.g. .../Input/t88.in
    inp_moldyn = pj(self.inp_dir, self.id + ".moldyn.in")
    inp_moldyn = self.cygwin_path(inp_moldyn)
    t_stdin.write( inp_moldyn + "\n")        # Input moldyn file.
    t_stdin.write( self.id + ".out\n")       # Output
    t_stdin.write( self.id + ".wandata\n")   # Output wandata 
    #                                                                                                 
    t_stdin.flush()
    t_stdin.seek(0)
    return t_stdin

#############################################################################################################

class OpticTest(BaseTest):
  def make_stdin(self):
    t_stdin = StringIO()

    inp_fname = self.cygwin_path(self.inp_fname)
    t_stdin.write( inp_fname + "\n")   # optic input file e.g. .../Input/t57.in
    t_stdin.write(self.id + ".out\n")  # Output. e.g t57.out
    t_stdin.write(self.id +"\n")       # Used as suffix to diff and prefix to log file names, and also for roots for temporaries
    #                                                                                                 
    t_stdin.flush()
    t_stdin.seek(0)
    return t_stdin

#############################################################################################################

class Band2epsTest(BaseTest):
  "How to waste lines of code just to test a F90 code that can be implemented with a few python commands!"
  def make_stdin(self):
    t_stdin = StringIO()
                                                                                                       
    inp_fname = self.cygwin_path(self.inp_fname)
    t_stdin.write( inp_fname + "\n")        # input file e.g. .../Input/t51.in
    t_stdin.write( self.id + ".out.eps\n")  # output file e.g. t51.out.eps

    inp_freq = pj(self.inp_dir, self.id + ".in_freq")
    inp_freq = self.cygwin_path(inp_freq)
    t_stdin.write(inp_freq + "\n")          # input freq file e.g Input/t51.in_freq

    inp_displ = pj(self.inp_dir, self.id + ".in_displ")
    inp_displ = self.cygwin_path(inp_displ)
    t_stdin.write(inp_displ + "\n")         # input displ file e.g Input/t51.in_displ
    #                                                                                                 
    t_stdin.flush()
    t_stdin.seek(0)
    return t_stdin

#############################################################################################################

class AtompawTest(BaseTest):

  def clean_workdir(self, other_test_files=None):
    "Keep all atompaw output files."

  @property
  def bin_path(self):
    return self.build_env.path_of_bin("atompaw")


# Subclasses associated to the executables.
def exec2class(exec_name):
  _exec2class = {  
    "abinit"   : AbinitTest, 
    "anaddb"   : AnaddbTest, 
    "aim"      : AimTest, 
    "conducti" : ConductiTest,
    "lwf"      : LwfTest,
    "atompaw"  : AtompawTest,
    "band2eps" : Band2epsTest,
    "optic"    : OpticTest,
  }
  try:
    return _exec2class[exec_name]
  except KeyError:
    return BaseTest

#############################################################################################################

class ChainOfTests(object):
   """
   A list of tests that should be executed at once. 
   Provides the same interface as the one provided by BaseTest 
   """
   def __init__(self, tests):
     self.tests = tuple([t for t in tests])

     self.inp_dir    = tests[0].inp_dir
     self.suite_name = tests[0].suite_name
     #
     # Consistency check.
     for t in tests:
       if (self.inp_dir    != t.inp_dir or  
           self.suite_name != t.suite_name): 
         raise TestError("All tests should be located in the same directory")

     all_keys = [t.keywords for t in self.tests]
     self.keywords = set()
     for ks in all_keys: self.keywords = self.keywords.union(ks)

     all_cpp_vars = [t.need_cpp_vars  for t in self.tests]
     self.need_cpp_vars = set() 
     for vs in all_cpp_vars: self.need_cpp_vars = self.need_cpp_vars.union(vs)

     self._files_to_keep = []

   def __len__(self): return len(self.tests)
   def __str__(self): return "\n".join([ str(t) for t in self ])
   def __iter__(self): 
     for t in self.tests: yield t

   def info_on_chain(self):
     attr_names = ["extra_inputs", "pre_commands", "post_commands"]
     string = "Info on chain: %s\n" % self.full_id

     nlinks = 0
     for test in self:
       string += test.full_id + "executable " + test.executable + ":\n"
       for (attr, value) in  test.__dict__.items():
         if ( value and (attr in attr_names or 
              attr.startswith("input_") or attr.startswith("output_")) ):
           string += "  %s = %s\n" % (attr, value)
           nlinks += 1

     return (string, nlinks)

   # A lot of boilerplate code!
   @property
   def id(self):
     return "-".join( [test.id for test in self] ) 

   @property
   def full_id(self):
     return "["+self.suite_name+"]["+self.id+"]"

   @property
   def max_nprocs(self):
     return max([test.max_nprocs for test in self])

   @property
   def _executed(self):
     return my_all([test._executed for test in self])

   @property
   def ref_dir(self):
     ref_dirs = [test.ref_dir for test in self]
     assert my_all( [dir == ref_dirs[0] for dir in ref_dirs] )
     return ref_dirs[0]

   def readme(self, width=100, html=True, abslink=True):
     string = ""
     if not html:
       string += "\n".join( [test.readme(width, html, abslink) for test in self] )
       string = self.full_id + ":\n" + string
     else:
       string += "<br>".join( [test.readme(width, html, abslink) for test in self] )
       string = "Test Chain " + self.full_id + ":<br>" + string
     return string

   @property
   def files_to_test(self):
     files = []
     for test in self: files.extend(test.files_to_test)
     return files

   @property
   def extra_inputs(self):
     extra_inputs = []
     for test in self: extra_inputs.extend(test.extra_inputs)
     return extra_inputs

   @property
   def inputs_used(self):
    inputs = []
    for test in self: inputs.extend(test.inputs_used)
    return inputs

   @property
   def run_etime(self):
     return sum([test.run_etime for test in self])
                                                    
   @property
   def tot_etime(self):
     return sum([test.tot_etime for test in self])

   @property
   def isok(self):
     return my_all([test.isok for test in self])

   @property
   def exceptions(self):
     excs = list()
     for test in self:
       excs.extend(test.exceptions)
     return excs

   @property
   def status(self):
     _stats = [test._status for test in self]
     if "disabled" in _stats or "skipped" in _stats: 
       assert my_all( [s==_stats[0] for s in _stats])
       return _stats[0]

     all_fldstats = [f.fld_status for f in self.files_to_test]
     if "failed" in all_fldstats: return "failed"
     if "passed" in all_fldstats: return "passed"
     assert my_all( [ s == "succeeded" for s in all_fldstats])
     return "succeeded"

   def keep_files(self, files):
     if isinstance(files, str):
       self._files_to_keep.append(files)
     else:
       self._files_to_keep.extend(files)

   @property
   def files_to_keep(self):
     # Files produced by the single tests.
     files_of_tests = []
     for test in self: 
       files_of_tests.extend(test.files_to_keep)

     # Add the files produced by self.
     self._files_to_keep += files_of_tests
     return self._files_to_keep

   def cpkl_dump(self, protocol=-1):
     self.cpkl_fname = pj(self.workdir, self.id +".cpkl")
     fh = open(self.cpkl_fname,"wb")
     cpkl.dump(self, fh, protocol=protocol)
     self.files_to_keep.append(self.cpkl_fname)
     fh.close()

   def has_keywords(self, keywords):
     return set(keywords).issubset(self.keywords)

   @property
   def _authors_snames(self):
     snames = set()
     for test in self: 
       snames = snames.union(test._authors_snames)
     return snames

   def has_authors(self, authors):
     return set(authors).issubset(self._authors_snames)

   def write_pyreport(self):
     fname = pj(self.workdir, "__test_report__.py")
     fh = open(fname, "w")
     #self.keep_files(fname)
     for test in self: test.write_pyreport(fh=fh)
     fh.close()

   def write_html_report(self):
     html_report = pj(self.workdir, "test_report.html")
     fh = open(html_report, "w")
     for idx, test in enumerate(self):
       oc = ""
       if idx == 0: oc += "o"
       if idx == (len(self)-1): oc += "c"
       test.write_html_report(fh=fh, oc=oc)
     fh.close()

   def run(self, build_env, runner, workdir, nprocs=1, **kwargs):

     workdir = absp(workdir)
     if not pexists(workdir): os.mkdir(workdir)
     self.workdir = workdir

     for test in self: 
       test.run(build_env, runner, workdir=self.workdir, nprocs=nprocs, **kwargs)

   def clean_workdir(self, other_test_files=None):
     for test in self: 
       test.clean_workdir(other_test_files=self.files_to_keep)

   def patch(self, patcher=None):
     for test in self:  
       test.patch(patcher)

#############################################################################################################

class TestSuite(object):
   """
   List of BaseTest instances. Provide methods to:

   1) select subset of tests according to keywords, authors, numbers
   2) run tests in parallel with python threads
   3) analyze the final results
   """
   def __init__(self, abenv, inp_files=None, test_list=None, keywords=None, need_cpp_vars=None):

     # Check arguments.
     args = [inp_files, test_list] 
     no_of_notnone = [ arg is not None for arg in args ].count(True)
     if no_of_notnone != 1: raise ValueError("Wrong args: " + str(args))

     self._executed = False
     self.abenv = abenv
     self.exceptions = list()

     if inp_files is not None:
       self.tests = tests_from_inputs(inp_files, abenv, keywords=keywords, need_cpp_vars=need_cpp_vars)

     elif test_list is not None:
       assert keywords is None
       assert need_cpp_vars is None
       self.tests = tuple(test_list)

     else:
       raise ValueError("Either inp_files or test_list must be specified!")

   def __str__(self): 
     return "\n".join( [str(t) for t in self.tests] )

   def __add__(self, other):
     test_list = [t for t in self] + [t for t in other]
     return TestSuite(self.abenv, test_list=test_list)

   def __len__(self): 
     return len(self.tests)

   def __iter__(self): 
     for t in self.tests: yield t

   def __getitem__(self, key):   # FIXME: this won't work for tutorial, paral and other test suites.
     "Called by self[key]"

     if isinstance(key, slice): 
       return self.__getslice(key)
     else:
       raise NotImplementedError("__getitem__ expects a slice instance")
       #for test in self:
       #  if test.id == key: return test
       #else:
       #  raise IndexError("%s not in TestSuite" % str(key))

   def __getslice(self, slice):
      start = slice.start
      if start is None: start = 1
      stop = slice.stop
      if stop is None: stop = 10000 # FIXME not very elegant, but cannot use len(self) since indices are not contigous
      assert slice.step is None     # Slices with steps (e.g. [1:4:2]) are not supported.

      test_list = list()
      for test in self:
        tokens = test.id.split("-")
        assert tokens[0][0] == "t"  # Assume fist character is "t"
        # extract the number from the name
        num = tokens[0][1:] 
        idx = num.rfind("_")
        if idx != -1: num = num[idx+1:]
        #print num
        num = int(num)
        if num in range(start, stop):  
          #print "got", num
          test_list.append(test)

      return TestSuite(self.abenv, test_list=test_list)

   @property
   def full_lenght(self):
     one = lambda : 1
     return sum( [getattr(test, "__len__", one)() for test in self] )
                                               
   @property
   def run_etime(self):
     assert self._executed
     return sum([test.run_etime for test in self])

   @property
   def keywords(self):
     keys = []
     for test in self: keys.extend(test.keywords)
     return set(keys)

   def has_keywords(self, keywords):
     return set(keywords).issubset(self.keywords)

   @property
   def need_cpp_vars(self):
     vars = list()
     for test in self: vars.extend(test.need_cpp_vars)
     return set(vars)

   def all_exceptions(self):
     "Return my exceptions + test exceptions" 
     all_excs = self.exceptions
     for test in self: 
       all_excs.extend(test.exceptions)
     return all_excs

   def cpkl_dump(self, protocol=-1):
     self.cpkl_fname = pj(self.workdir, "test_suite.cpkl")
     fh = open(self.cpkl_fname, "wb")
     cpkl.dump(self, fh, protocol=protocol)
     fh.close()

   def _tests_with_status(self, status):
      assert self._executed
      assert status in BaseTest._possible_status
      return [test for test in self if test.status == status]

   def succeeded_tests(self): return self._tests_with_status("succeeded")
   def passed_tests(self):    return self._tests_with_status("passed")
   def failed_tests(self):    return self._tests_with_status("failed")
   def skipped_tests(self):   return self._tests_with_status("skipped")
   def disabled_tests(self):  return self._tests_with_status("disabled")

   @property
   def targz_fname(self):
     """
     Location of the tarball file with the results in HTML format
     None if the tarball has not been created.
     """
     try:
       return self._targz_fname
     except:
       return None

   def create_targz_results(self):
     """
     Produce the tarball file results.tar.gz in the working directory.
     """
     assert self._executed
     exclude_exts = [".cpkl", ".py", "pyc", ]

     self._targz_fname = None
     ofname = pj(self.workdir,"results.tar.gz")

     try:
       targz = tarfile.open(ofname, "w:gz")
       for test in self:
         files = set(test.files_to_keep)
         save_files = [f for f in files if not has_exts(f, exclude_exts)]
         for p in save_files:
           # /foo/bar/suite_workdir/test_workdir/file --> test_workdir/t01/file
           rpath = os.path.relpath(p, start=self.workdir)
           #print "Adding to tarball:\n", p," with arcname ", rpath
           targz.add(p, arcname=rpath)
       targz.close()
       # 
       # Save the name of the tarball file.
       self._targz_fname = ofname 
     except:
       exc = sys.exc_info()[1]
       self.exceptions.append(exc)

   def sanity_check(self):
     all_full_ids = [test.full_id for test in self]
     if len(all_full_ids) != len(set(all_full_ids)): 
       raise ValueError("Cannot have more than two tests with the same full_id") 

   def run_tests(self, build_env, workdir, runner, nprocs=1, nthreads=1, **kwargs):
     """
     Main entry point for client code.
     """
     self.sanity_check()

     if len(self) == 0: raise ValueError("len(self) == 0")

     workdir = absp(workdir)
     if not pexists(workdir): os.mkdir(workdir)
     self.workdir = workdir

     self.nprocs = nprocs
     self.nthreads = nthreads

     def run_and_check_test(test):
       """Helper function that executes the test. Must be thread-safe."""

       testdir = absp(pj(self.workdir, test.suite_name + "_" + test.id))
       #print "%s Running in %s" % (test.full_id, testdir)

       # Run the test
       test.run(build_env, runner, testdir, nprocs=nprocs, **kwargs)

       # Write python module with summary.
       test.write_pyreport()

       # Write HTML summary
       test.write_html_report()
                                                          
       # Dump the object with cpickle.
       test.cpkl_dump()
                                                          
       # Remove useless files in workdir.
       test.clean_workdir()

     ##############################
     # And now let's run the tests
     ##############################
     start_time = time.time()

     if nthreads == 1:
       dbg_print("Sequential version")
       for test in self: run_and_check_test(test)

     elif nthreads > 1: 
       dbg_print("Threaded version with nthreads = %s" % nthreads)
       from threading import Thread
       # Internal replacement that provides task_done, join_with_timeout (py2.4 compliant)
       from pymods.myqueue import QueueWithTimeout 

       def worker():
         while True:
           test = q.get()
           run_and_check_test(test)
           q.task_done()
                                                    
       q = QueueWithTimeout()
       for i in range(nthreads):
         t = Thread(target=worker)
         t.setDaemon(True)
         t.start()
                                                    
       for test in self: q.put(test)
                                                    
       # Block until all tasks are done. Raise QueueTimeoutError after timeout seconds.
       timeout_1test = float(runner.timebomb.timeout)
       if timeout_1test <= 0.1: timeout_1test = 240.

       queue_timeout = 1.3 * timeout_1test * self.full_lenght / float(nthreads)
       q.join_with_timeout(queue_timeout)  

     # Run completed. 
     self._executed = True

     # Collect HTML files in a tarball
     self.create_targz_results()

     nsucc = len(self.succeeded_tests()) 
     npass = len(self.passed_tests())
     nfail = len(self.failed_tests())
     nskip = len(self.skipped_tests())
     ndisa = len(self.disabled_tests())
     ntot  = len(self)

     self.tot_etime = time.time() - start_time

     mean_etime = sum([test.run_etime for test in self]) / len(self)
     dev_etime = (sum([(test.run_etime - mean_etime)**2 for test in self]) / len(self) )**0.5

     print "Test suite completed in %.2f s (average time for test = %.2f s, stdev = %.2f s)" % ( 
       self.tot_etime, mean_etime, dev_etime)
     print "failed: %s, succeeded: %s, passed: %s, skipped: %s, disabled: %s" % (
       nfail, nsucc, npass, nskip, ndisa)
                                                                                              
     for test in self:
       if abs(test.run_etime - mean_etime) > 2*dev_etime:
         print "%s has run_etime %.2f s" % (test.full_id, test.run_etime)

     # Print summary table.
     stats_suite = dict()
     for test in self:
       if test.suite_name not in stats_suite:
         d = dict.fromkeys(BaseTest._possible_status, 0)
         d["run_etime"] = 0.0
         d["tot_etime"] = 0.0
         stats_suite[test.suite_name] = d

     for test in self:
       stats_suite[test.suite_name][test.status] += 1
       stats_suite[test.suite_name]["run_etime"] += test.run_etime
       stats_suite[test.suite_name]["tot_etime"] += test.tot_etime

     suite_names = stats_suite.keys()
     suite_names.sort()

     times = ["run_etime", "tot_etime"] 

     table = [ ["Suite"] + BaseTest._possible_status + times ]
     for suite_name in suite_names:
       stats = stats_suite[suite_name]
       row = [suite_name] + [str(stats[s]) for s in  BaseTest._possible_status] + ["%.2f" % stats[s] for s in times]
       table.append(row)

     pprint_table(table)

     fh = open(pj(self.workdir, "results.txt"), "w")
     pprint_table(table, out=fh)
     fh.close()

     try:
       username = os.getlogin()
     except:
       username = "No_username"

     # Create the HTML index.
     DNS = {
       "self"            : self,
       "runner"          : runner,
       "user_name"       : username,
       "hostname"        : gethostname(),
       "test_headings"   : ['ID', 'Status', 'run_etime (s)', 'tot_etime (s)'], 
       "suite_headings"  : ['failed', 'passed', 'succeeded', 'skipped', 'disabled'], 
       # Functions and modules available in the template.
       "time"        : time,
       "pj"          : os.path.join,
       "basename"    : basename,
       "str2html"    : str2html,
       "sec2str"     : sec2str,
       "args2htmltr" : args2htmltr,
       "html_link"   : html_link,
       "status2html" : status2html
     }

     fname = pj(self.workdir, "suite_report.html")
     fh = open(fname,"w")

     header = """
       <html>
       <head><title>Suite Summary</title></head>
       <body bgcolor="#FFFFFF" text="#000000">
        <hr>
        <h1>Suite Summary</h1>
         <table width="100%" border="0" cellspacing="0" cellpadding="2">
         <tr valign="top" align="left">
          <py-open code = "for h in suite_headings:"> </py-open>
          <th>${status2html(h)}</th>
         <py-close/>
         </tr>
         <tr valign="top" align="left">
         <py-open code = "for h in suite_headings:"> </py-open>
          <td> ${len(self._tests_with_status(h))} </td>
         <py-close/>
         </tr>
         </table>
         <p>
         tot_etime = ${sec2str(self.tot_etime)} <br>
         run_etime = ${sec2str(self.run_etime)} <br>
         no_pythreads = ${self.nthreads} <br>
         no_MPI = ${self.nprocs} <br>
         ${str2html(str(runner))}
        <hr>
     """

     table = """
        <p> 
        <h1>Test Results</h1>
        <table width="100%" border="0" cellspacing="0" cellpadding="2">
         <tr valign="top" align="left">
          <py-open code = "for h in test_headings:"> </py-open>
           <th>$h</th>
          <py-close/>
         </tr>
     """

     for status in BaseTest._possible_status:
       table += self._pyhtml_table_section(status)

     table += "</table>"

     footer = """
       <hr>
       <h1>Suite Info</h1>
         <py-line code = "keys = ', '.join(self.keywords)" />
         <p>Keywords = ${keys}</p>
         <py-line code = "cpp_vars = ', '.join(self.need_cpp_vars)"/>
         <p>Required CPP variables = ${cpp_vars}</p>
       <hr>
         Automatically generated by %s on %s. Logged on as %s@%s
       <hr>
       </body>
       </html> """ % (_MY_NAME, time.asctime(), username, gethostname() )

     template = header + table + footer

     template_stream = StringIO(template)
  
     # Initialise an xyaptu xcopier, and call xcopy
     xcp = xcopier(DNS, ouf=fh)
     xcp.xcopy(template_stream)
     fh.close()

     #self.cpkl_dump(protocol=0)

     return TestSuiteResults(self)

   def _pyhtml_table_section(self, status):
     # ['ID', 'Status', 'run_etime', 'tot_etime'], 
     string = """
        <py-open code="for test in self.%s_tests():"/>
         <py-line code = "report_link = pj(basename(test.workdir),'test_report.html') " />
         <tr valign="top" align="left">
          <td> ${html_link(test.full_id, report_link)}</td>
          <td> ${status2html(test.status)} </td>
          <td> ${sec2str(test.run_etime)} </td>
          <td> ${sec2str(test.tot_etime)} </td>
         </tr>
        <py-close/>
        """ % status
     return string

   def patch(self, patcher=None): 
     for test in self: test.patch(patcher)

   def select_tests(self, with_keys=None, exclude_keys=None, with_authors=None, exclude_authors=None):

     test_list =  [test for test in self]

     if with_keys:
       test_list = [test for test in test_list if test.has_keywords(with_keys)]

     if exclude_keys:
       test_list = [test for test in test_list if not test.has_keywords(exclude_keys)]

     if with_authors: 
       test_list = [test for test in test_list if test.has_authors(with_authors) ]

     if exclude_authors:
       test_list = [test for test in test_list if not test.has_authors(exclude_authors) ]

     return TestSuite(self.abenv, test_list=test_list)

   def make_readme(self, width=100, html=True):
     if not html:
       return "\n\n".join( [test.readme(width, html) for test in self] )
     else:
        header = """
         <html>
         <head><title>README FILE</title></head>
         <body bgcolor="#FFFFFF" text="#000000">
         <!-- Automatically generated by %s on %s. ****DO NOT EDIT**** -->""" % (_MY_NAME, time.asctime())

        body = "<hr>".join([test.readme(width, html) for test in self])

        footer = """
          <hr>
           Automatically generated by %s on %s. 
          <hr>
          </body>
          </html>""" % (_MY_NAME, time.asctime())

        return (header + body + footer)


class TestSuiteResults(object):

   def __init__(self, test_suite):
     assert test_suite._executed 
     self.failed_tests    = test_suite.failed_tests()
     self.passed_tests    = test_suite.passed_tests()
     self.succeeded_tests = test_suite.succeeded_tests() 
     self.skipped_tests   = test_suite.skipped_tests()
     self.disabled_tests  = test_suite.disabled_tests()

     self.targz_fname     = test_suite.targz_fname

   @lazy__str__
   def __str__(self): pass

   @property
   def nfailed(self):
     return len(self.failed_tests)

   @property
   def npassed(self):
     return len(self.passed_tests)

   def outref_files(self, status=None):
     assert status is None
     out_files, ref_files = [], []
     for test in (self.passed_tests + self.failed_tests):
       for f in test.files_to_test:
         out_files.append( pj(test.workdir, f.name) )
         ref_files.append( pj(test.ref_dir, f.name) )
     return out_files, ref_files

   def patch_refs(self):
     out_files, ref_files = self.outref_files()
     for r, o in zip(out_files, ref_files): print r, o
     patcher = Patcher()
     patcher.patch_files(out_files, ref_files)

   #def inspect_stdouts(self):
   #  out_files, ref_files = self.outref_files()
   #  patcher = Patcher()
   #  patcher.patch_files(out_files, ref_files)

   def cpkl_dump(self, cpkl_fname, protocol=-1):
     "Save the instance in a cpickle file"
     fh = open(cpkl_fname,"wb")
     cpkl.dump(self, fh, protocol=protocol)
     fh.close()

   #def inspect_diffs(self):
   #def write_pyreport(self):
   #def html_summary(self):

#############################################################################################################

def read_test_report(fname):
  # Import the module.
  from imp import load_source
  mod_path = absp(fname) 
  mod_name = mod_path 
  mod = load_source(mod_name, mod_path)
  #
  # Extract tests report.
  tag = "report_"
  for atr in mod.__dict__:
    if atr.startswith(tag): 
      # Get test id and retrieve the results.
      tid = atr.replace(tag,""); assert int(tid[1:]) 
      rep = Record()
      rep.__dict__ = mod.__dict__[atr]()
      return tid, rep
  else:
    raise ValueError("%s is not a valid __test_report__.py file" % fname)

#############################################################################################################

if __name__ == "__main__":
  doc_testcnf_format()
