# Script for running various tests of the parallel version of abinit
# Before running this script, read README in this directory.

# Copyright (C) 1998-2012 ABINIT group (XG,LSi)
# This file is distributed under the terms of the
# GNU General Public License, see ~abinit/COPYING
# or http://www.gnu.org/copyleft/gpl.txt .
# For the initials of contributors, see ~abinit/doc/developers/contributors.txt .
#
# Because of some execution problems, the script executes only one
# 'test' at a time (See the keyword 'HERE'), meaning that only one
# input file is considered, for different cases, from pure sequential
# to parallel with 32 processors.

# Also, only selected machine names are allowed, see the keyword 'MACH_LIST'
#
# Usage :
#  unix C-shell: Run machine_name sets [cases] >& log_file
#	 unix bash: Run machine_name sets [cases] > log_file 2>&1
#  DOS/Windows: perl Run.pl machine_name sets [cases] > log_file
#		sets or cases can be specified as a) a single set or case;
#			b) a range Start-Last or Start-end of sets or cases.
#
# For example :
# (Run ibm_pcpm A) >& log_file        run, for input file A,
#                                        all cases on the ibm_pcpm cluster
# (Run ibm_pcpm A 0) >& log_file      run, for input file A,
#                                        only case 0 on the ibm_pcpm cluster
# (Run t3e B 1-end) >& log_file     run, for input file B,
#                                        from case_1 to the end on a Cray-T3E
# (Run t3e C-D  1-2a) >& log_file     run, for input files C to D,
#                                        cases between 1 and 2a included,
#                                        on a Cray-T3E.
#*****************************************************************************

# Load ABINIT environment
$abinit_bindir = $ENV{'abinit_bindir'};
$abinit_rundir = $ENV{'abinit_rundir'};
$abinit_pspdir = $ENV{'abinit_pspdir'};
$abinit_inpdir = $ENV{'abinit_inpdir'};
$abinit_outdir = $ENV{'abinit_outdir'};
$nightly_bindir = $ENV{'nightly_bindir'};
$PERL = $ENV{'PERL'};
$PERL = 'perl' if ($PERL eq '');
$config_mpirun = $ENV{'MPI_RUNNER'};

# load reportdiff lib
# import lib
$top_testdir=$abinit_inpdir;
do "$top_testdir/Scripts/reportdiff.pl";

# Timeout support
#
# This is required for nightly builds, because e.g. binaries built
# with ifort 10.0 make the process hang forever when they crash.
$timeout = $ENV{'timeout'};
$timeout = '0';
$timeout = '0' if ($timeout eq '');

$, = ' ';               # set output field separator
$\ = "\n";              # set output record separator
#
@MachList=('t3e','sgi','tux','vpp','compaq_es40','magerit','sr8k','mpd','sun','test','spinoza','fock','max','lemaitre','sleepy(-compilo)','pwrg4','mpich2','generic','chum(-compilo)','cea','chpit-intel11','chpit-gcc44','testf-gcc43','testf-gcc44','bigmac','bigmac-g95','bigmac-gcc44','green','green-gcc42','green-g95','config','coba2-gcc44','coba2-intel','coba2-open64','buda_mpich2','buda-gcc43_openmpi','buda-gcc44_openmpi','buda-gcc45_openmpi', 'buda-intel11', 'inca', 'shiva', 'ibm6_mpich2', 'ibm6_openmpi','ibm6', 'win32_mpich2','littlebuda','green-intel11', 'ktulu','woopy', 'toum', 'hmem','max2','max2-openmpi','max2-open64','max2-ifort12','tikal-openmp','tikal','toum-ifort12','cronos','tikal-ifort12','vishnu','petrus','bigmac-gcc47','tikal-pgi','yquem');
@CompiloList = ('g95','pgi','gcc41','gcc42','gcc43','pathscale','intel','intel10','open64','sun');
@Cases = ('1','2','4','10','24','32');
@Sets = ('A','B','C','D','E','F','G','H','I','J','K','L','M','N','O','P','Q','R','S','T','U','V','W','X','Y','Z','AA','AB','AC','AD','AE','AF','AG','AH','AI','AJ','AK','AL','AM','AN','AO','AP','AQ','AR','AS','AT','AU','AV','AW','AX','AY','AZ','dfpt_01','dfpt_02','dfpt_03','dfpt_04');
$debug = 1;
$POE = 0;
# Check whether a machine name is provided
if ($#ARGV < 1) {
  print 'At least, two arguments must be provided giving machine name and set number';
  print "Choose among :@MachList" if ($#ARGV < 0);
  exit 16;
	}
print $ARGV[0];
print $ARGV[1];
print $ARGV[2];
print $ARGV[3];
#
$UNIXSLASH = '/';	# unix subdirectory delimitor in file paths (octal 057)
$DOSSLASH = '\\';	# subdirectory delimitor in DOS file paths (\ escaped)
# try unix 'uname' command in the Bourne shell manner
$rc = system('uname > uname.tmp 2>&1');
unlink ('uname.tmp');
$OStype = $ENV{'OSTYPE'};	# OSTYPE environment on unix systems (HP/UX excepted)
$OSname = $ENV{'OS'};
if ($rc == 0) {
	$SLASH = $UNIXSLASH;		# subdirectory delimitor in file paths
	$COPY_COMMAND = 'cp -p';	# unix copy file command
	$DIFF_COMMAND = 'diff -b';	# unix difference command
	$ERASE_COMMAND = 'rm -f';	# unix delete file command
	$XSUFX = '';		# no special suffix for executable module
	$UNAMERC = open(IP,"uname -a |");
	$_ = <IP>;
	chop $_;
	($LINonWIN,$wd2,$wd3,$wd4,$wd5,$wd6,$wd7,$wd8,$wd9) = split(' ',$_);
	close IP;
  	$LINonWIN = 'MINGW64_NT' if ( $LINonWIN eq 'CYGWIN_NT-6.1-WOW64');
  	$LINonWIN = 'MINGW64_NT' if ( $LINonWIN eq 'CYGWIN_NT-5.2-WOW64');
	$LINonWIN =~ m/(.*)(-)([0-9]+\.[0-9]+)/;
	$LINonWIN = $1 if ($& ne '');	# cleanup version number and keep plain uname
	if ($OSname eq 'Windows_NT' && $LINonWIN eq 'CYGWIN_NT') {
# cygwin shell works like a normal unix system
		$PERLSLASH = $UNIXSLASH;	# subdirectory delimitor for perl file paths
		$PLSUFX = '.pl';		# no special suffix for perl script
		}
	if ($OSname eq 'Windows_NT' && $LINonWIN eq 'MINGW64_NT') {
# cygwin shell works like a normal unix system
		$PERLSLASH = $UNIXSLASH;	# subdirectory delimitor for perl file paths
		$PLSUFX = '.pl';		# no special suffix for perl script
		}
	if ($OStype eq '' && $LINonWIN eq '') {
# since perl for Windows NT is not a PGI Workstation command (cygwin32) but a
# DOS module, calling the fldiff perl script by means of the "system" function
# requires DOS conventions
		$PERLSLASH = $DOSSLASH;   # subdirectory delimitor as for DOS file paths
		$PLSUFX = '.pl';	# DOS suffix for perl script
		}
	else {
# for perl under normal unix systems:
		$PERLSLASH = $UNIXSLASH;	# subdirectory delimitor for perl file paths
		$PLSUFX = '.pl';		# no special suffix for perl script
		}
	}
# When 'uname' command is unknown, the return code is non zero. Under Windows it may
# be 1 or 256 depending of the Perl version and the command interpretor; check OS
# environmental variable
elsif (($rc == 1 || $rc == 256) && $OSname eq 'Windows_NT') {			
	$OStype = $ENV{'OS'};
	$SLASH = $DOSSLASH;		# subdirectory delimitor in DOS file paths
	$COPY_COMMAND = 'copy';		# DOS copy file command
	$DIFF_COMMAND = 'fc /w /l /n';	# DOS difference command
# since unlink <file*> fails on some versions of perl for DOS, let's use del
	$ERASE_COMMAND = 'del /q';	# DOS delete file command
	$XSUFX = '.exe';		# DOS suffix for executable module
	$PERLSLASH = $DOSSLASH;   # subdirectory delimitor in DOS file path
	$PLSUFX = '.pl';		# DOS suffix for perl script
	$PERL = 'perl';			# perl.exe MUST be accessible through the DOS PATH
	}
else {
	print "unrecognized Operating System $OStype";
	exit 99;
	}

# by JMB : add support of mixed env cygwin + mingw compilation : MINGW64_NT
$CYGWIN = $LINonWIN eq 'MINGW64_NT' ? '/cygwin' : '';

if ($timeout eq '0') {
  $timeout_cmd = '';
  }
else {
  $timeout_cmd = &perlpath("$CYGWIN/$nightly_bindir/timeout$XSUFX")." $timeout ";
}
$CODE_SEQ = &perlpath("$CYGWIN/$abinit_bindir/abinit$XSUFX");	# codename for sequential version
$CODE_PAR = &perlpath("$CYGWIN/$abinit_bindir/abinit$XSUFX");	# codename for parallel version
$PSPS = &transpath("$CYGWIN$abinit_pspdir");		# pseudopotential directory  
# Set a default for the second and third pseudopotentials 
$PSP2 = &transpath("$PSPS/33as.SGS_mod");
$PSP3 = &transpath("$PSPS/33as.SGS_mod");
$FLDIFF = &perlpath("$PERL $abinit_rundir/fldiff.pl");	# fldiff script in utilities directory
#*****************************************************************************
 
# Check whether the machine name is allowed, and initialize its characteristics
# It is the place to add new machines and characteristics
# (but do not forget to mention the new name in the MACH_LIST variable above)

# The following characteristics are adapted for machines where neither lam
# nor mpich have to be used, and where there is always one thread per PE.
$MACH = $ARGV[0];
$TESTDIR = $ARGV[1];
$LAM = 0;
$MPICH = 0;
# List the set of cases for each machine (mask the second characteristics
# of tests, that may be 0, 1, 2, 4, 10, 24 or 32 ; 0, 1 and 2 are always allowed)
$CASE0 = 1;
$CASE1 = 1;
$CASE2 = 1;
$CASE4 = 1;
$CASE10 = 1;
$CASE24 = 1;
$CASE32 = 1;
# However, the list of cases depend on the test
#-----------------------------------
# For the t3e in berkeley and Garching
if ($MACH eq 't3e') {
  $MPIRUN_NP = 'mpprun -n';
	}
#-----------------------------------
# For a SGI R1000
elsif ($MACH eq 'sgi') {
  $MPIRUN_NP = 'mpprun -np';
	}
#-----------------------------------
# For the cluster of DEC/Linux "tux" machine at the UCL.
# Note that on this cluster, the local 
# node (tux0) is used for the first thread. 
# This explains why the number of lines
# of the cluster files is smaller than usual.
# 
elsif ($MACH eq 'tux') {
  $MPICH = 1;
  $MPICH_DIR = '/usr/local/mpi/bin';
  $MPIRUN_NP = "$MPICH_DIR/mpirun -np";
  $MACH0='tux0';
  $MACH1='tux2';
  $MACH2='tux3';
  $MACH3='tux5';
  $MACH4='tux6';
  $MACH5='tux7';
  $MACH6='tux8';
  $MACH7='tux9';
  $MACH8='tux10';
  $MACH9='tux11';
  unlink 'cluster1','cluster2','cluster4','cluster10';
	open (FILE,">cluster1") || die "Unable to open FILE cluster1";
	print FILE $MACH0;
	close (FILE);
	open (FILE,">cluster2") || die "Unable to open FILE cluster2";
	print FILE $MACH1;
	close (FILE);
	open (FILE,">cluster4") || die "Unable to open FILE cluster4";
        print FILE $MACH1;
        print FILE $MACH2;
        print FILE $MACH3;
	close (FILE);
	open (FILE,">cluster10") || die "Unable to open FILE cluster10";
        print FILE $MACH1;
        print FILE $MACH2;
        print FILE $MACH3;
        print FILE $MACH4;
        print FILE $MACH5;
        print FILE $MACH6;
        print FILE $MACH7;
        print FILE $MACH8;
        print FILE $MACH9;
	close (FILE);
	}
#--------------------------------------
#
elsif ($MACH eq 'magerit') {
  $MPICH = 1;
  $MPIRUN_NP = "mpirun -np";
  $LL_MACHINE_LIST = $ENV{'LL_MACHINE_LIST'};
  $CASE10 = 0;
  unlink 'cluster1','cluster2','cluster4';
	system("head -1 $LL_MACHINE_LIST > cluster1");
	system("head -2 $LL_MACHINE_LIST > cluster2");
	system("head -4 $LL_MACHINE_LIST > cluster4");
	}
#-----------------------------------
# For the iBM Power 5  machine at the PCPM lab, using MPICH
elsif ($MACH eq 'fock') {
  $MPICH = 2;
  $OPENMPI = 1;
  $MPICH_DIR = '/usr/local/openmpi/bin';
  $MPIRUN_NP = "$MPICH_DIR/mpirun -np";
  $MACH1 = 'fock';
  $CASE10 = 0;
  unlink 'cluster1','cluster2','cluster4';
        open (FILE,">cluster1") || die "Unable to open FILE cluster1";
        print FILE $MACH1;
        close (FILE);
        open (FILE,">cluster2") || die "Unable to open FILE cluster2";
        print FILE $MACH1;
        print FILE $MACH1;
        close (FILE);
        open (FILE,">cluster4") || die "Unable to open FILE cluster4";
                for ($i = 0; $i < 4; $i++) {
                print FILE $MACH1;
                }
        close (FILE);
        }
#-----------------------------------
# For the Fujitsu VPP700 at Riken
elsif ($MACH eq 'vpp') {
  $MPIRUN_NP = ' -np';
  $CASE10 = 0;
	}
#-----------------------------------
# For the Compaq/DEC 4 processors in LLN
elsif ($MACH eq 'compaq_es40') {
  $MPIRUN_NP = ' dmpirun -np';
  $CASE10 = 0;
	}
#-----------------------------------
# For the Sun V20 Opteron-based machine in LLN
elsif ($MACH eq 'lemaitre') {
  $MPIRUN_NP = '/usr/local/mpich-1.2.6-eth-intel9.1/bin/mpirun -np';
        }
#-----------------------------------
# For the Hitachi SR8000
elsif ($MACH eq 'sr8k' || $MACH eq 'mpd') {
  $MPIRUN_NP = 'mpiexec -n';
	}
# 
elsif ($MACH eq 'cea') {
  $MPIRUN_NP = 'cea_mprun -n ';
	}
#-----------------------------------
# For the Sun sunfire 2 proc
elsif ($MACH eq 'sun') {
  $MPIRUN_NP = '/opt/SUNWhpc/HPC5.0/bin/mprun -np';
  $CASE4 = 0;
  $CASE10 = 0;
	}
#-----------------------------------
# For a bi-processor SGI R14000
elsif ($MACH eq 'spinoza') {
  $MPIRUN_NP = 'mpirun -np';
  $CASE4 = 0;
  $CASE10 = 0;
        }
#-----------------------------------
# For the Intel PC XEON "sleepy" 2 procs (with hyperthreading)
elsif ($MACH =~ /^sleepy-?(.*)/) {
  $MPICH = 1;
  $MPICH_DIR = "/usr/local/mpi-$1/bin";
  if ($1 eq '') { $MPICH_DIR = "/usr/local/mpi-ifc9.1-64/bin"; }
# verify if $MPICH_DIR exists
  if (! -e $MPICH_DIR ) {
     print "The $MPICH_DIR folder is not present...";
     print "Choose among : @CompiloList";
     exit 15;
  }
# Note that one uses the mpirun from PGI, but the one from IFC is OK also ...
  $MPIRUN_NP = "$MPICH_DIR/mpirun -np";
  $MACH0='sleepy.pcpm.ucl.ac.be';
  $MACH1='sleepy.pcpm.ucl.ac.be:4';
  $MACH2='sleepy.pcpm.ucl.ac.be:2';
  $CASE10 = 0;
  unlink 'cluster1','cluster2','cluster4';
        open (FILE,">cluster1") || die "Unable to open FILE cluster1";
        print FILE $MACH1;
        close (FILE);
        open (FILE,">cluster2") || die "Unable to open FILE cluster2";
        print FILE $MACH2;
        close (FILE);
        open (FILE,">cluster4") || die "Unable to open FILE cluster4";
        print FILE $MACH1;
        close (FILE);
}
#-----------------------------------
# For the MacPro Mac OS X Leopard Server "bigmac" with biprocs Quad-Core Xeon 2.6Ghz (6Gb RAM)
elsif ($MACH eq 'bigmac') {
  $MPICH = 2;
  $OPENMPI = 1;
  $MPICH_DIR = '/opt/local/openmpi-gcc43_m64/bin';
  $MPIRUN_NP = "$MPICH_DIR/mpirun -np";
  $CASE10 = 0;
}
#-----------------------------------
# For the MacPro Mac OS X Leopard Server "bigmac" with biprocs Quad-Core Xeon 2.6Ghz (6Gb RAM)
elsif ($MACH eq 'bigmac-gcc44') {
  $MPICH = 2;
  $OPENMPI = 1;
  $MPICH_DIR = '/opt/openmpi-1.4.1_gcc-4.4.3_m64/bin';
  $MPIRUN_NP = "$MPICH_DIR/mpirun -np";
  $CASE10 = 0;
}
#-----------------------------------
# For the MacPro Mac OS X Leopard Server "bigmac" with biprocs Quad-Core Xeon 2.6Ghz (6Gb RAM)
elsif ($MACH eq 'bigmac-gcc47') {
  $MPICH = 2;
  $OPENMPI = 1;
  $MPICH_DIR = '/opt/openmpi16_gcc47/bin';
  $MPIRUN_NP = "$MPICH_DIR/mpirun -np";
  $CASE10 = 0;
}
#-----------------------------------
# For the MacPro Mac OS X Leopard Server "bigmac" with biprocs Quad-Core Xeon 2.6Ghz (6Gb RAM)
elsif ($MACH eq 'bigmac-g95') {
  $MPICH = 2;
  $OPENMPI = 1;
  $MPICH_DIR = '/usr/local/openmpi_g95/bin';
  $MPIRUN_NP = "$MPICH_DIR/mpirun -np";
  $CASE10 = 0;
}
#-----------------------------------
# For the  DELL cluster "green" with biprocs Quad-Core Xeon 2.6Ghz (8Gb RAM) per node
elsif ($MACH eq 'green') {
  $MPICH = 2;
  $OPENMPI = 1;
  $MPICH_DIR = '/cvos/shared/apps/openmpi/intel/64/current/bin';
  $MPIRUN_NP = "$MPICH_DIR/mpirun -np";
  $MACH0='green';
  $MACH1='green';
  $MACH2='green:2';
  $MACH3='green:4';
  $CASE10 = 0;
  unlink 'cluster1','cluster2','cluster4';
        open (FILE,">cluster1") || die "Unable to open FILE cluster1";
        print FILE $MACH1;
        close (FILE);
        open (FILE,">cluster2") || die "Unable to open FILE cluster2";
        print FILE $MACH2;
        close (FILE);
        open (FILE,">cluster4") || die "Unable to open FILE cluster4";
        print FILE $MACH3;
        close (FILE);
        }
#-----------------------------------
# For the  DELL cluster "green-gcc42" with biprocs Quad-Core Xeon 2.6Ghz (8Gb RAM) per node
elsif ($MACH eq 'green-gcc42') {
  $MPICH = 2;
  $OPENMPI = 1;
  $MPICH_DIR = '/cvos/shared/apps/openmpi/gcc/64/1.3.1/bin';
  $MPIRUN_NP = "$MPICH_DIR/mpirun -np";
  $MACH0='green';
  $MACH1='green';
  $MACH2='green:2';
  $MACH3='green:4';
  $CASE10 = 0;
  unlink 'cluster1','cluster2','cluster4';
        open (FILE,">cluster1") || die "Unable to open FILE cluster1";
        print FILE $MACH1;
        close (FILE);
        open (FILE,">cluster2") || die "Unable to open FILE cluster2";
        print FILE $MACH2;
        close (FILE);
        open (FILE,">cluster4") || die "Unable to open FILE cluster4";
        print FILE $MACH3;
        close (FILE);
        }
#-----------------------------------
# For the  DELL cluster "green-g95" with biprocs Quad-Core Xeon 2.6Ghz (8Gb RAM) per node
elsif ($MACH eq 'green-g95') {
  $MPICH = 2;
  $OPENMPI = 1;
  $MPICH_DIR = '/home/pcpm/buildbot/openmpi/bin';
  $MPIRUN_NP = "$MPICH_DIR/mpirun -np";
  $CASE10 = 0;
}
#-----------------------------------
# For the  DELL cluster "green-intel11" with biprocs Quad-Core Xeon 2.6Ghz (8Gb RAM) per node
elsif ($MACH eq 'green-intel11') {
  $MPICH = 2;
  $OPENMPI = 1;
  $MPICH_DIR = '/home/pcpm/buildbot/openmpi_intel11/bin';
  $MPIRUN_NP = "$MPICH_DIR/mpirun -np";
  $CASE10 = 0;
}
#-----------------------------------
# For the AMD SUN GALAXY X4200  "chum" 2 x AMD 2220 Dual Cores with OpenMPI or MPICH2
elsif ($MACH =~ /^chum-?(.*)/) {
  $MPICH = 2;
  $OPENMPI = 1;
#  $MPICH_DIR = "/usr/local/mpich2-$1/bin";
  $MPICH_DIR = "/usr/local/openmpi-$1/bin";
  if ($1 eq '') { $MPICH_DIR = "/usr/local/openmpi-intel/bin"; }
  if ($1 eq 'sun') { $MPICH_DIR = "/usr/local/mpich2-sun/bin"; $OPENMPI = 0;}
#  if ($1 eq 'open64') { $MPICH_DIR = "/usr/local/mpich2-open64/bin"; $OPENMPI = 0;}
# verify if $MPICH_DIR exists
  if (! -e $MPICH_DIR ) {
     print "The $MPICH_DIR folder is not present...";
     print "Choose among : @CompiloList";
     exit 15;
  }
print "The $MPICH_DIR will be used...($OPENMPI)";
#
  $MPIRUN_NP = "$MPICH_DIR/mpirun -np";
  $MACH0='chum.pcpm.ucl.ac.be';
  $MACH1='chum.pcpm.ucl.ac.be:4';
  $MACH2='chum.pcpm.ucl.ac.be:2';
  $CASE10 = 0;
  unlink 'cluster1','cluster2','cluster4';
        open (FILE,">cluster1") || die "Unable to open FILE cluster1";
        print FILE $MACH1;
        close (FILE);
        open (FILE,">cluster2") || die "Unable to open FILE cluster2";
        print FILE $MACH2;
        close (FILE);
        open (FILE,">cluster4") || die "Unable to open FILE cluster4";
        print FILE $MACH1;
        close (FILE);
}
#-----------------------------------
# For the  2 x Quad Xeon Nehalem 2.6GHz / 12GB  intel11 openmpi 1.3.x
elsif ($MACH eq 'buda-intel11') {
  $MPICH = 2;
  $OPENMPI = 1;
  $MPIRUN_NP = '/usr/local/openmpi_intel11/bin/mpirun --mca btl tcp,self -np';
  $CASE10 = 0;
        }
#-----------------------------------
# For the  2 x Quad Xeon Nehalem 2.6GHz / 12GB  gcc434 + mpich2
elsif ($MACH eq 'buda_mpich2') {
  $MPICH = 2;
  $OPENMPI = 0;
  $MPICH_DIR = "/usr/local/mpich2/bin";
  $MPIRUN_NP = '/usr/local/mpich2/bin/mpiexec -np';
  $CASE10 = 0;
        }
#-----------------------------------
# For the  2 x  Intel Core2 Quad CPU 3GHz / 8GB  gcc451 + mpich2-1.3.2
# Windows 7
elsif ($MACH eq 'woopy') {
  $MPICH = 2;
  $OPENMPI = 0;
  $MPICH_DIR = "/usr/local/mpich2/bin";
  $MPIRUN_NP = '/usr/local/mpich2/bin/mpiexec -localonly';
  $CASE10 = 0;
        }
#-----------------------------------
# For the  Intel Core i7 3.2GHz / 12Gb gcc462 + openmpi 1.4.3
elsif ($MACH eq 'petrus') {
  $MPICH = 2;
  $OPENMPI = 1;
  $MPICH_DIR = "/usr/lib64/mpi/gcc/openmpi/bin";
  $MPIRUN_NP = '/usr/lib64/mpi/gcc/openmpi/bin/mpirun -np';
  $CASE10 = 0;
        }
#-----------------------------------
# For the  Intel Core i7 3.4GHz / 8Gb gcc4.7.2+ openmpi 1.6.3
# CentOS 6.3 packages for openmpi and gcc + fttw3 3.3.3 
elsif ($MACH eq 'yquem') {
  $MPICH = 2;
  $OPENMPI = 1;
  $MPICH_DIR = "/usr/local/openmpi/bin";
  $MPIRUN_NP = '/usr/local/openmpi/bin/mpirun -np';
  $CASE10 = 0;
        }
#-----------------------------------
# For the  AMD FX-8120 3.1GHz / 8Gb gcc470 + openmpi 1.6.0
elsif ($MACH eq 'vishnu') {
  $MPICH = 2;
  $OPENMPI = 1;
  $MPICH_DIR = "/usr/local/openmpi/bin";
  $MPIRUN_NP = '/usr/local/openmpi/bin/mpirun -np';
  $CASE10 = 0;
        }
#-----------------------------------
# For the  2 x  Intel Xeon Quad CPU 2.93GHz / 12GB  ifort 12.1.4 + openMPI 1.5.4
elsif ($MACH eq 'tikal-ifort12') {
  $MPICH = 2;
  $OPENMPI = 1;
  $MPICH_DIR = "/usr/local/openmpi_ifort12/bin";
  $MPIRUN_NP = '/usr/local/openmpi_ifort12/bin/mpirun -np';
  $CASE10 = 0;
        }
#-----------------------------------
# For the  2 x  Intel Xeon Quad CPU 2.93GHz / 12GB  8GB  pgi 12.x + openMPI 1.6.2
elsif ($MACH eq 'tikal-pgi') {
  $MPICH = 2;
  $OPENMPI = 1;
  $MPICH_DIR = "/usr/local/openmpi_pgi/bin";
  $MPIRUN_NP = '/usr/local/openmpi_pgi/bin/mpirun -np';
  $CASE10 = 0;
        }
#-----------------------------------
# For the  2 x  Intel Xeon Quad CPU 2.93GHz / 12GB  8GB  psc + openMPI 1.4.3
elsif ($MACH eq 'tikal') {
  $MPICH = 2;
  $OPENMPI = 1;
  $MPICH_DIR = "/usr/local/openmpi/bin";
  $MPIRUN_NP = '/usr/local/openmpi/bin/mpirun -np';
  $CASE10 = 0;
        }
#-----------------------------------
# For the  2 x  Intel Core2 Quad CPU 3GHz / 8GB  psc + openMPI 1.4.3
elsif ($MACH eq 'tikal-openmp') {
  $MPICH = 2;
  $OPENMPI = 1;
  $MPICH_DIR = "/usr/local/openmpi_ifort12/bin";
  $MPIRUN_NP = '/usr/local/openmpi_ifort12/bin/mpirun -np';
  $CASE10 = 0;
        }
#-----------------------------------
# For the  2 x  AMD Opteron 2 x OctaCore CPU 2.6GHz / 16GB  open64 + openMPI 1.4.3
elsif ($MACH eq 'max2-open64') {
  $MPICH = 2;
  $OPENMPI = 1;
  $MPICH_DIR = "/usr/local/openmpi_open64/bin";
  $MPIRUN_NP = '/usr/local/openmpi_open64/bin/mpirun -np';
  $CASE10 = 1;
        }
#-----------------------------------
# For the  2 x  AMD Opteron 2 x OctaCore CPU 2.6GHz / 16GB  gcc46 + openMPI 1.4.5
elsif ($MACH eq 'max2-ifort12') {
  $MPICH = 2;
  $OPENMPI = 1;
  $MPICH_DIR = "/usr/local/openmpi_ifort12/bin";
  $MPIRUN_NP = '/usr/local/openmpi_ifort12/bin/mpirun -np';
  $CASE10 = 0;
        }
#-----------------------------------
# For the  2 x  AMD Opteron 2 x OctaCore CPU 2.6GHz / 16GB  gcc46 + openMPI 1.4.3
elsif ($MACH eq 'max2-openmpi') {
  $MPICH = 2;
  $OPENMPI = 1;
  $MPICH_DIR = "/usr/local/openmpi/bin";
  $MPIRUN_NP = '/usr/local/openmpi/bin/mpirun -np';
  $CASE10 = 1;
        }
#-----------------------------------
# For the  2 x  AMD Opteron 2 x OctaCore CPU 2.6GHz / 16GB  gcc46 + mpich2 1.3.1 with slurm support
elsif ($MACH eq 'max2') {
  $MPICH = 2;
  $OPENMPI = 0;
  $MPICH_DIR = "/usr/local/mpich2/bin";
  $MPIRUN_NP = '/usr/local/mpich2/bin/mpiexec -np';
  $CASE10 = 1;
        }
#-----------------------------------
# For the  2 x  AMD  16-Core CPU   gcc46 +  ???   (Jean-Michel, could you complete ?)
elsif ($MACH eq 'cronos') {
  $MPICH = 2;
  $OPENMPI = 1;
  $MPICH_DIR = "/usr/local/openmpi/bin";
  $MPIRUN_NP = '/usr/local/openmpi/bin/mpirun -np';
  $CASE10 = 0;
        }
#-----------------------------------
# For the  2 x  Intel Core2 Quad CPU 3GHz / 8GB  gcc46 + openMPI 1.5.4 rpm
elsif ($MACH eq 'toum') {
  $MPICH = 2;
  $OPENMPI = 1;
  $MPICH_DIR = "/usr/lib64/openmpi/bin";
  $MPIRUN_NP = '/usr/lib64/openmpi/bin/mpirun -np';
  $CASE10 = 0;
        }
#-----------------------------------
# For the  2 x  Intel Core2 Quad CPU 3GHz / 8GB  ifort12 + openMPI 1.4.4
elsif ($MACH eq 'toum-ifort12') {
  $MPICH = 2;
  $OPENMPI = 1;
  $MPICH_DIR = "/usr/local/openmpi_intel/bin";
  $MPIRUN_NP = '/usr/local/openmpi_intel/bin/mpirun -np';
  $CASE10 = 0;
        }
#-----------------------------------
# For the  1 x Intel Core2 Quad CPU 2.6GHz / 4GB  gcc451 + mpich2 + atlas
elsif ($MACH eq 'littlebuda') {
  $MPICH = 2;
  $OPENMPI = 0;
  $MPICH_DIR = "/usr/local/mpich2/bin";
  $MPIRUN_NP = '/usr/local/mpich2/bin/mpiexec -np';
  $CASE10 = 0;
        }
#-----------------------------------
# For the  1 x Intel Core2 Quad CPU 3GHz / 4GB  gcc442 + mpich2
elsif ($MACH eq 'inca') {
  $MPICH = 2;
  $OPENMPI = 0;
  $MPICH_DIR = "/usr/local/mpich2/bin";
  $MPIRUN_NP = '/usr/local/mpich2/bin/mpiexec -np';
  $CASE10 = 0;
        }
#-----------------------------------
# For the HP Z400 1 x Six Xeon Nehalem 3.3GHz / 12GB gcc451 + mpich2-1.3.x
elsif ($MACH eq 'shiva') {
  $MPICH = 2;
  $OPENMPI = 0;
  $MPICH_DIR = "/usr/local/mpich2/bin";
  $MPIRUN_NP = '/usr/local/mpich2/bin/mpiexec -np';
  $CASE10 = 0;
        }
#-----------------------------------
# For the IBM6  1 x Power6 Quad core 4.7GHz / 8GB  xlf 12 + POE
elsif ($MACH eq 'ibm6') {
  $MPICH = 2;
  $OPENMPI = 1;
  $POE = 1;
  $timeout_cmd = '';
  $MPIRUN_NP = '/usr/lpp/ppe.poe/bin/poe ';
#  $MPIRUN_NP = '/usr/bin/mpiexec -hfile /usr/local/etc/host.list -n';
  $CASE10 = 0;
        }
#-----------------------------------
# For the  1 x Power6 Quad core 4.7GHz / 8GB  xlf 12 + mpich2
elsif ($MACH eq 'ibm6_mpich2') {
  $MPICH = 2;
  $OPENMPI = 0;
  $MPIRUN_NP = '/usr/local/mpich2/bin/mpiexec -np';
  $CASE10 = 0;
        }
#-----------------------------------
# For the  1 x Power6 Quad core 4.7GHz / 8GB  xlf 12 + openmpi 1.4.x
elsif ($MACH eq 'ibm6_openmpi') {
  $MPICH = 2;
  $OPENMPI = 1;
  $MPIRUN_NP = '/usr/local/openmpi/bin/mpirun --mca btl sm,tcp,self -np';
  $CASE10 = 0;
        }
#-----------------------------------
# For the  2 x Quad Xeon Nehalem 2.6GHz / 12GB  gcc434 + openmpi 1.3.4
elsif ($MACH eq 'buda-gcc43_openmpi') {
  $MPICH = 2;
  $OPENMPI = 1;
  $MPIRUN_NP = '/usr/local/openmpi_gcc43/bin/mpirun --mca btl tcp,self -np';
  $CASE10 = 0;
        }
#-----------------------------------
# For the  2 x Quad Xeon Nehalem 2.6GHz / 12GB  gcc444 + openmpi 1.4.2
elsif ($MACH eq 'buda-gcc44_openmpi') {
  $MPICH = 2;
  $OPENMPI = 1;
  $MPIRUN_NP = '/usr/local/openmpi_gcc44/bin/mpirun --mca btl tcp,self -np';
  $CASE10 = 0;
        }
#
#-----------------------------------
# For the  2 x Quad Xeon Nehalem 2.6GHz / 12GB  gcc451 + openmpi 1.4.3
elsif ($MACH eq 'buda-gcc45_openmpi') {
  $MPICH = 2;
  $OPENMPI = 1;
  $MPIRUN_NP = '/usr/local/openmpi_gcc45/bin/mpirun --mca btl tcp,self -np';
  $CASE10 = 0;
        }
#-----------------------------------
# For the  2 x Quad Xeon Nehalem 2.6GHz / 12GB  gcc472 + openmpi 1.4.2
elsif ($MACH eq 'buda-gcc47_openmpi') {
 $MPICH = 2;
 $OPENMPI = 1;
 $MPIRUN_NP = '/usr/local/openmpi_gcc47/bin/mpirun --mca btl tcp,self -np';
 $CASE10 = 0;
       }
#-----------------------------------
# For the Bull 2 x Quad Xeon Nehalem 3GHz / 12GB  gcc432 + openmpi 1.2.9
elsif ($MACH eq 'testf-gcc43') {
  $MPICH = 2;
  $OPENMPI = 1;
  $MPIRUN_NP = '/usr/local/openmpi-gcc43/bin/mpirun -np';
  $CASE10 = 0;
        }
#-----------------------------------
# For TITANE
elsif ($MACH eq 'titane') {
  $MPICH = 2;
  $OPENMPI = 1;
  $MPIRUN_NP = 'mpirun -n';
  $CASE10 = 0;
        }
#-----------------------------------
# For the Bull 2 x Quad Xeon Nehalem 3GHz / 12GB gcc440 + openmpi 1.3.2
elsif ($MACH eq 'testf-gcc44') {
  $MPICH = 2;
  $OPENMPI = 1;
  $MPIRUN_NP = '/usr/local/openmpi-gcc44/bin/mpirun --mca btl tcp,self -np';
  $CASE10 = 0;
        }
#-----------------------------------
# For the Bull 2 x Quad Xeon Nehalem 3GHz / 12GB gcc471 + openmpi 1.6.1
elsif ($MACH eq 'testf-gcc47') {
  $MPICH = 2;
  $OPENMPI = 1;
  $MPIRUN_NP = '/usr/local/openmpi-gcc47/bin/mpirun --mca btl tcp,self -np';
  $CASE10 = 0;
        }
#-----------------------------------
# For the HP Z400 1 x Quad Xeon Nehalem 2.7GHz / 3GB open64 4.5.1 ( AMD version) + openmpi 1.4.4
elsif ($MACH eq 'coba2-open64') {
  $MPICH = 2;
  $OPENMPI = 1;
  $MPIRUN_NP = '/usr/local/openmpi_open64/bin/mpirun --mca btl tcp,self -np';
  $CASE10 = 0;
        }
#-----------------------------------
# For the HP Z400 1 x Quad Xeon Nehalem 2.7GHz / 3GB gcc44 + openmpi 1.3.3
elsif ($MACH eq 'coba2-gcc44') {
  $MPICH = 2;
  $OPENMPI = 1;
  $MPIRUN_NP = '/usr/local/openmpi_gcc44/bin/mpirun --mca btl tcp,self -np';
  $CASE10 = 0;
        }
#-----------------------------------
# For the HP Z400 1 x Quad Xeon Nehalem 2.7GHz / 3GB intel11 + openmpi 1.4.1 + mkl +fftw3
elsif ($MACH eq 'coba2-intel11') {
  $MPICH = 2;
  $OPENMPI = 1;
  $MPIRUN_NP = '/usr/local/openmpi_intel/bin/mpirun --mca btl tcp,self -np';
  $CASE10 = 0;
        }
#-----------------------------------
# For the HP Z400 1 x Quad Xeon Nehalem 2.7GHz / 3GB intel12.1.4 + openmpi 1.6.0 + mkl +fftw3
elsif ($MACH eq 'coba2-intel12') {
  $MPICH = 2;
  $OPENMPI = 1;
  $MPIRUN_NP = '/usr/local/openmpi_intel12/bin/mpirun --mca btl tcp,self -np';
  $CASE10 = 0;
        }
#-----------------------------------
# For the HMEM - CISM machine with nodes Quad 12Core Opteron@2.2GHz
elsif ($MACH eq 'hmem') {
  $MPICH = 2;
  $OPENMPI = 1;
  $MPIRUN_NP = '/usr/local/openmpi/1.4.3-intel/bin/mpirun --mca btl tcp,self -np';
        }
#-----------------------------------
# For the  1 x Intel Core2 Quad CPU 3GHz / 8GB  gcc44 + openmpi 1.4.1 + fftw3 ( all deb packages )
elsif ($MACH eq 'ktulu') {
  $MPICH = 2;
  $OPENMPI = 1;
  $MPIRUN_NP = '/usr/bin/mpirun --mca btl tcp,self -np';
  $CASE10 = 0;
        }
#-----------------------------------
# For the HP integrity Itanium 4 processor in LLN
elsif ($MACH eq 'chpit-gcc44') {
  $MPICH = 2;
  $OPENMPI = 1;
  $MPIRUN_NP = '/usr/local/openmpi-gcc44/bin/mpirun -np';
  $CASE10 = 0;
        }
#-----------------------------------
# For the HP integrity Itanium 4 processor in LLN
elsif ($MACH eq 'chpit-intel11') {
  $MPICH = 2;
  $OPENMPI = 1;
  $MPIRUN_NP = '/usr/local/openmpi-1.4.3-intel-11.1/bin/mpirun -np';
  $CASE10 = 0;
        }
#-----------------------------------
# For the MacBookPRo with Intel 2 cores "pwrg4"
elsif ($MACH eq 'pwrg4') {
  $MPICH = 1;
  $MPICH_DIR = "/usr/local/mpich-1.2.7p1-gcc-4.3.0/bin"; 
  $MPIRUN_NP = "$MPICH_DIR/mpirun -np";
  $MACH0='pwrg4.pcpm.ucl.ac.be';
  $MACH1='pwrg4.pcpm.ucl.ac.be:4';
  $MACH2='pwrg4.pcpm.ucl.ac.be:2';
  $CASE10 = 0;
  unlink 'cluster1','cluster2','cluster4';
        open (FILE,">cluster1") || die "Unable to open FILE cluster1";
        print FILE $MACH1;
        close (FILE);
        open (FILE,">cluster2") || die "Unable to open FILE cluster2";
        print FILE $MACH2;
        close (FILE);
        open (FILE,">cluster4") || die "Unable to open FILE cluster4";
        print FILE $MACH1;
        close (FILE);
        }
#-----------------------------------
# For the cluster of PowerPC Mac OS X "max" with biprocs with gm ( Myrinet )
elsif ($MACH eq 'max') {
  $MPICH = 2;
  $OPENMPI = 1;
  $MPICH_DIR = '/usr/local/openmpi/bin';
  $MPIRUN_NP = "$MPICH_DIR/mpirun --mca btl gm,self -np";
  $MACH0='maxi1 slots=1 max_slots=1';
  $MACH1='maxi2 slots=1 max_slots=1';
  $MACH2='maxi3 slots=1 max_slots=1';
  $MACH3='maxi4 slots=1 max_slots=1';
  $CASE10 = 1;
  unlink 'cluster1','cluster2','cluster4','cluster10';
        open (FILE,">cluster1") || die "Unable to open FILE cluster1";
        print FILE $MACH0;
        close (FILE);
        open (FILE,">cluster2") || die "Unable to open FILE cluster2";
        print FILE $MACH0;
        print FILE $MACH1;
        close (FILE);
        open (FILE,">cluster4") || die "Unable to open FILE cluster4";
        print FILE $MACH0;
        print FILE $MACH1;
        print FILE $MACH2;
        print FILE $MACH3;
        close (FILE);
        open (FILE,">cluster10") || die "Unable to open FILE cluster10";
        print FILE 'maxi1 slots=1 max_slots=1';
        print FILE 'maxi2 slots=1 max_slots=1';
        print FILE 'maxi3 slots=1 max_slots=1';
        print FILE 'maxi4 slots=1 max_slots=1';
        print FILE 'maxi5 slots=1 max_slots=1';
        print FILE 'maxi7 slots=1 max_slots=1';
        print FILE 'maxi8 slots=1 max_slots=1';
        print FILE 'maxi9 slots=1 max_slots=1';
        print FILE 'maxi10 slots=1 max_slots=1';
        print FILE 'maxi11 slots=2 max_slots=2';
        print FILE 'maxi12 slots=2 max_slots=2';
        print FILE 'maxi13 slots=2 max_slots=2';
        print FILE 'maxi14 slots=2 max_slots=2';
        print FILE 'maxi15 slots=2 max_slots=2';
        close (FILE);
   }
#-----------------------------------
# For the cluster of PowerPC Mac OS X "max" with biprocs
elsif ($MACH eq 'max-eth') {
  $MPICH = 2;
  $OPENMPI = 1;
  $MPICH_DIR = '/usr/local/openmpi-g95/bin';
  $MPIRUN_NP = "$MPICH_DIR/mpirun -np";
  $MACH0='max';
  $MACH1='max';
  $MACH2='max';
  $MACH3='max';
  $CASE10 = 0;
  unlink 'cluster1','cluster2','cluster4';
        open (FILE,">cluster1") || die "Unable to open FILE cluster1";
        print FILE $MACH0;
        close (FILE);
        open (FILE,">cluster2") || die "Unable to open FILE cluster2";
        print FILE $MACH0;
        print FILE $MACH1;
        close (FILE);
        open (FILE,">cluster4") || die "Unable to open FILE cluster4";
        print FILE $MACH0;
        print FILE $MACH1;
        print FILE $MACH2;
        print FILE $MACH3;
        close (FILE);
        }
#-----------------------------------
# For mpich2-1.0.5p4 (gmatteo@tux)
elsif ($MACH eq 'mpich2') {
# $MPIRUN_NP = '/opt/mpich2-1.0.5p4/bin/mpirun -np ';
 $MPIRUN_NP = '/opt/mpich2-1.2-ifc11/bin/mpirun -np ';
        }
#-----------------------------------
# generic
# use the mpirun that is in the PATH
elsif ($MACH eq 'generic') {
 $MPIRUN_NP = 'mpirun -np ';
        }
elsif ($MACH eq 'cea') {
 $MPIRUN_NP = 'cea_mprun -n  ';
        }
#-----------------------------------
# config
# parameters extracted from the build system
elsif ($MACH eq 'config') {
 $MPIRUN_NP = "$config_mpirun -np ";
 $CASE4     = 0;
 $CASE10    = 0;
 $CASE24    = 0;
 $CASE32    = 0;
        }
#-----------------------------------
# For win32_mpich2 i.e. Windows32 with cygwin and mpich2-1.3 (on cauac)
elsif ($MACH eq 'win32_mpich2') {
 $MPIRUN_NP = 'mpiexec -np ';
        }
#-----------------------------------
# This is a dummy machine name, for sequential tests only
# (One should check that this is only used in the sequential case)
elsif ($MACH eq 'test') {
 $MPIRUN_NP = ' ';
        }
#-----------------------------------
# For the hmem cluster, plain openmpi launch
elsif ($MACH eq 'hmem') {
  $MPICH = 2;
  $OPENMPI = 1;
  $MPIRUN_NP = 'mpirun -np';
  $CASE10 = 0;
        }
#-----------------------------------
else {
  print "The machine name $MACH is not allowed";
  print "Choose among : @MachList";
  exit 12;
	}

print "Testing the Abinit code on the $MACH $OStype platform";
# Here stop the examination of machine names
# and the definition of their characteristics
#*******************************************************************************
$_ = $ARGV[2];
$hitreport = m/report/;     # check for the keyword report
if ($hitreport) {
        print "Will make report";
        }
else {
        $hitnum = m/[A-Z]{1,2}/;	# check for upper case
        $hitrng = m/([A-Z]{1,2})-(.*)/;	# check for range
        $hit_tutoparal_dfpt = m/dfpt_0[1-4]/;	# check for number between 01 and 04
        if ($hitrng) {
	        $FirstSet = $1;
	        $LastSet = $2;
	        $LastSet = '' if ($LastSet eq 'end');
	        }
        elsif ($hitnum) {	# if last set is omitted, run only first one
	        $FirstSet = $ARGV[2];
	        $LastSet = $ARGV[2];
	        }
        elsif ($hit_tutoparal_dfpt) {       # if last set is omitted, run only first one
                $FirstSet = $ARGV[2];
                $LastSet = $ARGV[2];
                }
        else {
	        print "Invalid sets: $ARGV[2]";
	        exit 24;
	        }
# set label for default first and last test number
        $FirstCase = '1';
        $LastCase = '32';
        $_ = $ARGV[3];
        $hit = m/(.*)-(.*)/;	# check for range
	        if ($hit) {
		        $FirstCase = $1;
		        $LastCase = $2;
		        $LastCase = '32' if ($LastCase eq 'end');
		        }
	        elsif ($ARGV[3] ne '') {	# if last test is omitted, run only first one
		        $FirstCase = $ARGV[3];
		        $LastCase = $ARGV[3];
		        }

        print "Following tests will be run: sets $FirstSet to $LastSet, cases $FirstCase to $LastCase";
        }

#################################################
# set a date flag to make a new directory today #
#################################################
($sec,$min,$hour,$mday,$ymon,$yyear,$wday,$yday,$isdst)=localtime(time);
$ymon++;	# ymon was 0-11
$yyear +=1900;	# yyear was relative to 1900
$YYYYMMDD = sprintf("%4.4d",$yyear).sprintf("%2.2d",$ymon).sprintf("%2.2d",$mday);

$WORK_DIR = 'tmp-'.$MACH.'_'.$YYYYMMDD;
if (! -e $WORK_DIR || ! -d $WORK_DIR) {
	mkdir ($WORK_DIR,0755);		# Mode 0755 ignored under DOS-Windows
	}
else {
	print "Can not create directory, $WORK_DIR already exists";
	}

print "cd $WORK_DIR";
chdir ("$WORK_DIR");

if ($hitreport) {
        print "Make report";
        &doreport();
        exit 0;
        }

# **************************************** 
# Loop on all the sets
#
if ( $TESTDIR eq 'paral') {
        $CASE24=0;
        $CASE32=0;
}
if ( $TESTDIR eq 'mpiio') {
        $CASE10=0;
        $CASE24=0;
        $CASE32=0;
}
if ( $TESTDIR eq 'tutoparal') {
        $CASE0=0 ;
        $CASE1=0 ;
        $CASE2=0 ;
        $CASE4=0 ;
        $CASE10=0;
}
#
foreach $set (@Sets) {
# Jump to first test to be run
        next if ($T1 eq '' && $set ne $FirstSet);
        last if ($T1 eq $LastSet);
        $T1 = $set ;
        print "Make test set $T1" ;

#for ($T1 = $FirstSet; $T1 le $LastSet; $T1 ++) {
#        last if ($T1 eq 'ZZ');
# Make preliminary operations that might differ for different tests

# HERE, the set of possible tests and corresponding input file
	if ($T1 eq 'A' || $T1 eq 'B') {
	  $PSP1 = "$PSPS/14si.psp";
	  }
#
	elsif ($T1 eq 'C') {
	  $PSP1 = "$PSPS/13al.pspgth";
	  $PSP2 = "$PSPS/33as.SGS_mod";
	  }
#
	elsif ($T1 eq 'D') {
	  $PSP1 = "$PSPS/42mo.pspnc";
          $CASE10=0;
	  }
#
	elsif ($T1 eq 'E' || $T1 eq 'H' || $T1 eq 'J') {
	  $PSP1 = "$PSPS/31ga.SGS_mod";
          $PSP2 = "$PSPS/33as.SGS_mod";
	  $OPTFLDIFF = '-medium';
	  }
#
	elsif ($T1 eq 'F') {
	  $PSP1 = "$PSPS/13al.pspgth";
          $PSP2 = "$PSPS/33as.SGS_mod";
          unlink 'tFi_DS1_DEN','tFi_DS2_DEN';
	  system("$COPY_COMMAND tC1o_DEN tFi_DS1_DEN"); 
	  system("$COPY_COMMAND tC1o_DEN tFi_DS2_DEN"); 
	  }
#
	elsif ($T1 eq 'G') {
	  $PSP1 = "$PSPS/13al.pspgth";
          $PSP2 = "$PSPS/33as.SGS_mod";
	  $OPTFLDIFF = '-medium';
	  unlink 'tGi_WFK','tGi_WFQ';
	  system("$COPY_COMMAND tC1o_WFK tGi_WFK");
	  system("$COPY_COMMAND tC1o_WFK tGi_WFQ");
	  }
#
	elsif ($T1 eq 'I') {
	  $PSP1 = "$PSPS/26fe.pspnc";
	  $OPTFLDIFF = '-easy';
          $CASE10=0;
	  }
#
        elsif ($T1 eq 'K') {
          $PSP1 = "$PSPS/7n.psphgh";
          }
#
        elsif ($T1 eq 'L') {
          $PSP1 = "$PSPS/41nb_001023.pspfhi";
          $PSP2 = "$PSPS/03li.pspfhi";
          $PSP3 = "$PSPS/08o_001023.pspfhi";
          $OPTFLDIFF = '-easy';
          }
#
        elsif ($T1 eq 'M') {
          $PSP1 = "$PSPS/14si.pspnc";
          $CASE10=0;
          }
#
        elsif ($T1 eq 'N') {
          $PSP1 = "$PSPS/14si.pspnc";
          $CASE10=0;
          }
#
        elsif ($T1 eq 'O') {
          $PSP1 = "$PSPS/14si.pspnc";
          $CASE10=0;
          }
#This is to test MPI and paral_kgb, only with 4 procs in parallel
        elsif ($T1 eq 'P') {
          $PSP1 = "$PSPS/14si.phoney_mod";
          $OPTFLDIFF = '-easy';
          $CASE1= 0;
          $CASE2= 0;
          $CASE10=0;
          }
#This is to test MPI_IO norm conserving (with the MPI flag), only with 4 procs in parallel
        elsif ($T1 eq 'Q') {
          $PSP1 = "$PSPS/14si.phoney_mod";
	  $CASE0= 0;
	  $CASE1= 0;
	  $CASE2= 0;
	  $CASE10=0;
          }
#This is to test MPI and paral_kgb, only with 4 procs in parallel, paw case
        elsif ($T1 eq 'R') {
          $PSP1 = "$PSPS/6c_lda.paw";
          $OPTFLDIFF = '-ridiculous';
          $CASE1= 0;
          $CASE2= 0;
          $CASE10=0;
          }
#This is to test MPI_IO PAW (with the MPI flag), only with 4 procs in parallel
        elsif ($T1 eq 'S') {
          $PSP1 = "$PSPS/6c_lda.paw";
	  $CASE0= 0;
	  $CASE1= 0;
	  $CASE2= 0;
	  $CASE10=0;
          }
#This is to test the recursion code, in sequential and with 4 procs in parallel
        elsif ($T1 eq 'T') {
          $PSP1 = "$PSPS/2he.psphgh";
	  $CASE1= 0;
	  $CASE2= 0;
	  $CASE10=0;
          }
#This is to test the KSS generation parallelized over k-points
        elsif ($T1 eq 'U') {
          $PSP1 = "$PSPS/14si.pspnc";
          $CASE10=0;
          }
#This is to test the SC-GW with advanced memory management in parallel
        elsif ($T1 eq 'V') {
          $PSP1 = "$PSPS/11na.pspnc";
          $CASE10=0;
          }
	$FLDreport = "fldiff.set$T1.report";
	unlink $FLDreport;
	$RUNfile = 'abfiles.run';
	if ($LAM == 1) {
		$cmd = "$LAMDIR/wipe ../cluster1";
		print $cmd if ($debug > 1);
		system($cmd);
		$cmd = "$LAMDIR/wipe ../cluster2";
		print $cmd if ($debug > 1);
		system($cmd);
		}
#This is to test MPI and paral_kgb, only with 4 procs in parallel, paw case
        elsif ($T1 eq 'W') {
          $PSP1 = "$PSPS/6c_lda.paw";
          $OPTFLDIFF = '-easy';
          $CASE1= 0;
          $CASE2= 0;
          $CASE10=0;
          }
#This is to test the triple BandFFT-kpt parallelization in sequential and on 4 procs in parallel
        elsif ($T1 eq 'X') {
          $PSP1 = "$PSPS/6c_lda.paw";
	  $CASE1= 0;
	  $CASE2= 0;
	  $CASE10=0;
          }
#Same as X with also test on spin component parallelization, bandpp, istwfk=2 and wfoptalg=14.  
        elsif ($T1 eq 'Y') {
          $PSP1 = "$PSPS/6c_lda.paw";
	  $CASE1= 0;
	  $CASE2= 0;
	  $CASE10=0;
          }
#Tddft test
        elsif ($T1 eq 'Z') {
          $PSP1 = "$PSPS/31ga.pspnc";
          $PSP2 = "$PSPS/33as.pspnc";
          $OPTFLDIFF = '-medium';
          $PSP1 = "$PSPS/7n.psphgh";
          }
#PAW Berrys Phase calculation of Born effective charge in AlAs
        elsif ($T1 eq 'AA') {
          $PSP1 = "$PSPS/al_ps.abinit.paw";
          $PSP2 = "$PSPS/as_ps.paw";
	  $CASE10=0;
          $OPTFLDIFF = '-easy';
          }
#Electron-phonon renormalization of the electronic eigenenergies
        elsif ($T1 eq 'AB') {
          $PSP1 = "$PSPS/6c.pspnc";
          }
#String method
        elsif ($T1 eq 'AC') {
          $PSP1 = "$PSPS/1h.pspnc";
          $PSP2 = "$PSPS/1h.pspnc";
          }
#Parallelization over spinorial components
        elsif ($T1 eq 'AD') {
          $PSP1 = "$PSPS/83bi.paw";
          $CASE1= 0;
          $CASE2= 0;
          $CASE10=0;
          }
#Automatic Parallelization
        elsif ($T1 eq 'AE') {
          $PSP1 = "$PSPS/6c_lda.paw";
          $OPTFLDIFF = '-easy';
          $CASE1= 0;
          $CASE2= 0;
          $CASE10=0;
          }    
#Parallelization of SC DMFT over k-points
        elsif ($T1 eq 'AF') {
          $PSP1 = "$PSPS/28ni.paw";
          $PSP2 = "$PSPS/8o.2.paw";
	  $OPTFLDIFF = '-medium';
          $CASE2= 0;
#          $CASE10=0;
          }
#Parallelization of NSC DMFT over bands and k-point
        elsif ($T1 eq 'AG') {
          $PSP1 = "$PSPS/28ni.paw";
          $PSP2 = "$PSPS/8o.2.paw";
	  $OPTFLDIFF = '-medium';
          $CASE1= 0;
          $CASE2= 0;
          $CASE10=0;
          }
#Automatic Parallelization with unoccupied processor
        elsif ($T1 eq 'AH') {
          $PSP1 = "$PSPS/6c_lda.paw";
          $OPTFLDIFF = '-easy';
          $CASE1= 0;
          $CASE2= 0;
          $CASE10=0;
          }
#Parallelization on perturbations
        elsif ($T1 eq 'AI') {
          $PSP1 = "$PSPS/31ga.SGS_mod";
          $PSP2 = "$PSPS/33as.SGS_mod";
          $OPTFLDIFF = '-easy';
          $CASE1= 0;
           }
#Tutorial Parallelism : automatic testing dfpt 03, BaTiO3 ground state calculation
        elsif ($T1 eq 'dfpt_03') {
          $PSP1 = "$PSPS/56ba.psp_mod";
          $PSP2 = "$PSPS/22ti.psp_mod";
          $PSP3 = "$PSPS/8o.psp_mod";
          $CASE32= 0;
          }
#Tutorial Parallelism : automatic testing dfpt 04, BaTiO3 linear response calculation
        elsif ($T1 eq 'dfpt_04') {
          $PSP1 = "$PSPS/56ba.psp_mod";
          $PSP2 = "$PSPS/22ti.psp_mod";
          $PSP3 = "$PSPS/8o.psp_mod";
          $OPTFLDIFF = '-medium';
          $CASE24= 0;
          unlink 'tdfpt_04i_WFK','tdfpt_04i_WFQ';
          system("$COPY_COMMAND tdfpt_0324o_WFK tdfpt_04i_WFK");
          system("$COPY_COMMAND tdfpt_0324o_WFK tdfpt_04i_WFQ");
          }

# ****************************************
	foreach $case (@Cases) {
# Jump to first test to be run
		next if ($T2 eq '' && $case ne $FirstCase);
#
#Here, set characteristics of the test : the name of the test_case is $T1$T2 
		$T2 = $case;
		if ( ($T2 eq '0' && $CASE0 == 0) || ($T2 eq '1' && $CASE1 == 0) || 
                     ($T2 eq '2' && $CASE2 == 0) || ($T2 eq '4' && $CASE4 == 0) || 
		     ($T2 eq '10' && $CASE10 == 0) || ($T2 eq '24' && $CASE24 == 0) ||
                     ($T2 eq '32' && $CASE32 == 0)  ) {
			last if ($T2 eq $LastCase); 
			next;
			}
		print "\nSet $T1 / Case $T2\n";
#Set some variables according to the value of $T2
		$MODE = $T2 eq '0' ? 'SEQ' : 'PAR';
		$CLUST = "../cluster$T2";
		$NPAR = $T2;
		$APPENDI = "i";
		$APPENDO = "o";
		unlink $RUNfile;
		system ("$ERASE_COMMAND t$T1$T2*");	# unlink with metachar
#Initialize the files file
		open (FILE,">$RUNfile") || die "Unable to open FILE $RUNfile";
		print FILE "$CYGWIN$abinit_inpdir/$TESTDIR/Input/t$T1.in";
		print FILE "t$T1$T2.out";
		print FILE "t$T1$APPENDI";
		print FILE "t$T1$T2$APPENDO";
		print FILE "t$T1$T2";
		print FILE $PSP1;
		print FILE $PSP2;
		print FILE $PSP3;
		close (FILE);
#Run the job
		if ($MACH eq 'max') {
			$MachFile = "--hostfile $CLUST";
   			# problem with DS5 of test E ( see below for reporting )
			if ($T2 eq '10' && $T1 eq 'E') { $NPAR = 16; };
			}
		elsif ( $MACH =~ /^max2/ ) { 
   			# problem with DS5 of test E ( see below for reporting )
			if ($T2 eq '10' && $T1 eq 'E') { $NPAR = 16; };
			}
		elsif ($MPICH == 1 && $MODE eq 'PAR') { 
			$MachFile = "-machinefile $CLUST";
			}
		elsif ($POE == 1 ) { 
			$MachFile = "-hostfile /usr/local/etc/host.list";
			}
		else {
			$MachFile = '';
			}
# if not OPENMPI, start mpd (MPICH2)
		if ($OPENMPI == 0) {
			$MPDSTAT=`$MPICH_DIR/mpdtrace`;
			if ( $MACH =~ /^chum/ ) {
			   if ( $MPDSTAT =~ !/^chum/ ) {
        			system("$MPICH_DIR/mpd &");
				sleep 3;
			   }
			}
			if ( $MACH =~ /^buda/ ) {
			   if ( $MPDSTAT =~ !/^buda/ ) {
        			system("$MPICH_DIR/mpd &");
				sleep 3;
			   }
			}
# REM : inca use now pm=gforker then no more needs mpd
			if ( $MACH =~ /^i_nca/ ) {
			   if ( $MPDSTAT =~ !/^i_nca/ ) {
        			system("$MPICH_DIR/mpd &");
				sleep 3;
			   }
			}
		}

		if ($LAM == 1 && $MODE eq 'PAR') { 
			$cmd = "$LAMDIR/lamboot $CLUST ";
			print $cmd if ($debug > 1);
			system($cmd);
		}
#
		if ($MODE eq 'SEQ') {
			$CALL_CODE = ($T2 eq '0' && $MACH eq 'sr8k') ? "prun $CODE_SEQ" : $CODE_SEQ;
			$CALL_CODE = $timeout_cmd.$CALL_CODE;
			}
		else {
			$CALL_CODE = $MACH eq 'vpp' ? "$CODE_PAR $MPIRUN_NP $NPAR" : "$MPIRUN_NP $NPAR $MachFile $CODE_PAR";
			$CALL_CODE = $MACH eq 'ibm6' ? "$MPIRUN_NP $CODE_PAR $MachFile -procs $NPAR" : "$MPIRUN_NP $NPAR $MachFile $CODE_PAR";
			$CALL_CODE = $timeout_cmd.$CALL_CODE;
			}
   		# reset the $NPAR to 10 for reporting -> tE10.out
                if ($MACH eq 'max') {
                        if ($NPAR eq '16') { $NPAR = 10; };
                }       
                if ($MACH =~ /^max2/) {
                        if ($NPAR eq '16') { $NPAR = 10; };
                }       
		#
		$cmd = "$CALL_CODE < $RUNfile > t$T1$T2.log";
		print $cmd if ($debug > 1);
		system($cmd);

		# make sure there exists an output file
		system("touch t$T1$T2.out");
#
		if ($LAM == 1 && $MODE eq 'PAR') {
		$cmd = "$LAMDIR/wipe $CLUST";
		print $cmd if ($debug > 1);
		system($cmd);
		}
#Analyze the output
		$cmd = "$DIFF_COMMAND t$T1$T2.out $abinit_inpdir/$TESTDIR/Refs/t$T1$NPAR.out > diff.t$T1$T2";
		print $cmd if ($debug > 1);
		system($cmd);
# append label with test number to report file
		open (FLDREP,">>$FLDreport") || die "Unable to open FLDREP for test $T1$T2";
		print FLDREP "\n","Case_$T1$T2";	# Case_NN
		close (FLDREP);
		$cmd = "$FLDIFF -ignore $OPTFLDIFF t$T1$T2.out $abinit_inpdir/$TESTDIR/Refs/t$T1$NPAR.out Case_$T1$T2 >> $FLDreport";
		print $cmd if ($debug > 1);
		system($cmd);
#Eventually exit
		last if ($T2 eq $LastCase);
		}
# End of loop on $T1
	}
# ***************************************************************************
sub transpath {
	local($path) = @_;
#
# purpose: translate unix-like path of file to DOS-like according to $SLASH
# argument:
#	$path = path to be translated
# output: this subroutine acts as a function and returns the path
# according to host conventions
	
	$path =~ tr/\057/\\/ if ($SLASH eq '\\');
	return $path;
	}

# ****************************************
sub perlpath {
	local($path) = @_;
#
# purpose: translate unix-like path of file to DOS-like according to $PERLSLASH.
#   This is necessary when calling a DOS command like perl under PGI Workstation.
# argument:
#	$path = path to be translated
# output: this subroutine acts as a function and returns the path
# according to host conventions
	
	$path =~ tr/\057/\\/ if ($PERLSLASH eq '\\');
	return $path;
	}
# ****************************************
sub doreport {
        local($TID) = @_;
#
# purpose: exercise lwf
# arguments:
#               $TID = test identification (1 to 3 characters string) that will be used as suffix to diff
# and prefix to log file names
#               $p1,..$p5 = 1 to 5 parameters [re]defining some option or the file names (fn1 to fn4) that
# will be put into the "files" file named "lwf.run";
# the parameter format for a file name is: fnI=path
# the parameter format for an option is: opt=option ; see setfnopt
#       $fn1 = main input to lwf
# $fn2 = moldyn input to lwf
# $fn3 = output file name that will be compared with reference.
# $fn4 = auxiliary output file name
# set default files names derived from $TID:
# Prepare filenames
        $fn1 = "$abinit_inpdir/$TESTDIR/Input/report.in";
        $reportref = &transpath($fn1);
        $fn2 = "fldiff.set?.report" ;
        $fldiffreport = &transpath($fn2);
        $fn2b = "fldiff.set??.report" ;
        $fldiffreportb = &transpath($fn2b);
        $fn2c = "fldiff.set???*.report" ;
        $fldiffreportc = &transpath($fn2c);
        $fn3 = "summaryfldiff" ;
        $summaryfldiff = &transpath($fn3);
        $fn4 = "report" ;
        $shortreport = &transpath($fn4);
# Create summary of tests
        unlink($summaryfldiff);
        system ("grep --no-filename 'Summary' $fldiffreport > $summaryfldiff");
        system ("grep --no-filename 'Summary' $fldiffreportb >> $summaryfldiff");
        system ("grep --no-filename 'Summary' $fldiffreportc >> $summaryfldiff");
# Compare with reference files
        print "Prepare the short report";
        unlink($shortreport);
        &reportdiff("$summaryfldiff","$reportref","$shortreport");  # compare floating point numbers
        return;                 # go read next line from configuration file
        }

# ****************************************
