/* abi_common.h */

/*
 * Copyright (C) 2008-2012 ABINIT Group (MG)
 *
 * This file is part of the ABINIT software package. For license information,
 * please see the COPYING file in the top-level directory of the ABINIT source
 * distribution.
 *
 */

#ifndef _ABINIT_COMMON_H
#define _ABINIT_COMMON_H

/*
 * Language standards requires the existance of pre-defined macros
 * Microsoft Visual C++ does not define __STDC__,
 * Sun Workshop 4.2 supports C94 without setting __STDC_VERSION__ to the proper value
 */

#if defined (__STDC__)
# define PREDEF_STANDARD_C_1989    /** ANSI X3.159-1989 **/
# if defined (__STDC_VERSION__)
#  define PREDEF_STANDARD_C_1990   /** ISO/IEC 9899:1990 **/
#  if (__STDC_VERSION__ >= 199409L)
#   define PREDEF_STANDARD_C_1994  /** ISO/IEC 9899-1:1994 **/
#  endif
#  if (__STDC_VERSION__ >= 199901L)
#   define PREDEF_STANDARD_C_1999  /** ISO/IEC 9899:1999 **/
#  endif
# endif
#endif

/** #define DEBUG_MODE **/

/** define WHEREARG __FILE__,__LINE__**/
#ifdef HAVE_FC_LONG_LINES
# define NEWLINE ;
#else
# define NEWLINE \newline
#endif
/** define WHEREARG NEWLINE __FILE__, NEWLINE __LINE__ **/

/** this does not work with gfort, pgi, **/

#if defined (FC_GNU) || defined(FC_G95) || defined (FC_PGI)
#define QUOTEME(x)     'x'
#else
#define QUOTEME(x)     #x
#endif

#define BYTE_SIZE(array)  PRODUCT(SHAPE(array)) * DBLE(KIND(array))

/** debugging macros so we can pin down message provenance at a glance
#define WHERESTR "[file %s, line %d] "
**/

/*
 * ABI_  basic abinit macros.
 * DBG_  macros for debugging. Defined only if abinit is compiled in DEBUG_MODE.
 * MSG_  macros for logging.
 * */

#ifdef HAVE_FC_LONG_LINES
#  define ABI_CHECK(expr,str) if (.not.(expr)) call assert(.FALSE.,str,__FILE__,__LINE__)
/** #  define ABI_CHECK(expr,str) call assert((expr), str,__FILE__,__LINE__) **/
#else
#  define ABI_CHECK(expr,str) call assert((expr), str)
#endif

#if defined HAVE_FC_LONG_LINES
#  define ABI_CHECK_MPI(ierr,msg)      call check_mpi_ierr(ierr,msg,"PERS",__FILE__,__LINE__)
#  define ABI_CHECK_MPI_PERS(ierr,msg) call check_mpi_ierr(ierr,msg,"PERS",__FILE__,__LINE__)
#else
#  define ABI_CHECK_MPI(ierr,msg) call check_mpi_ierr(ierr,msg,"PERS")
#  define ABI_CHECK_MPI_PERS(ierr,msg) call check_mpi_ierr(ierr,msg,"PERS")
#endif


/* Macros for memory checking and profiling *
 * Remember: subtract number of newlines in lineno otherwise the value might be misleading
#define HAVE_MEM_PROFILE 0
*/

#ifdef HAVE_MEM_PROFILING 
# ifdef HAVE_FC_LONG_LINES
#  define ABI_ALLOCATE(ARR,SIZE) \
   allocate(ARR SIZE,stat=ABI_ALLOC_STAT) NEWLINE \
   call memocc(ABI_ALLOC_STAT,product(shape(ARR))*kind(ARR),QUOTEME(ARR),ABI_FUNC)
#  define ABI_DEALLOCATE(ARR) \
   ABI_ALLOC_SIZE=-product(shape(ARR))*kind(ARR) NEWLINE \
   deallocate(ARR,stat=ABI_ALLOC_STAT) NEWLINE \
   call memocc(ABI_ALLOC_STAT,ABI_ALLOC_SIZE,QUOTEME(ARR),ABI_FUNC)
# else
#  define ABI_ALLOCATE(ARR,SIZE) \
   allocate(ARR SIZE,stat=ABI_ALLOC_STAT) NEWLINE \
   call memocc(ABI_ALLOC_STAT,product(shape(ARR))*kind(ARR),QUOTEME(ARR),ABI_FUNC)
#  define ABI_DEALLOCATE(ARR) \
   ABI_ALLOC_SIZE=-product(shape(ARR))*kind(ARR) NEWLINE \
   deallocate(ARR,stat=ABI_ALLOC_STAT) NEWLINE \
   call memocc(ABI_ALLOC_STAT,ABI_ALLOC_SIZE,QUOTEME(ARR),ABI_FUNC)
# endif
#else
# define ABI_ALLOCATE(ARR,SIZE) \
  allocate(ARR SIZE,stat=ABI_ALLOC_STAT)
# define ABI_DEALLOCATE(ARR) \
  deallocate(ARR,stat=ABI_ALLOC_STAT)
#endif

#define ABI_DATATYPE_ALLOCATE(ARR,SIZE) \
   allocate(ARR SIZE,stat=ABI_ALLOC_STAT)
#define ABI_DATATYPE_DEALLOCATE(ARR) \
   deallocate(ARR,stat=ABI_ALLOC_STAT)

#define TMP_ABI_12MPI_ALLOCATE(ARR,SIZE) \
   allocate(ARR SIZE)
#define TMP_ABI_12MPI_DEALLOCATE(ARR) \
   deallocate(ARR)

/* Shorthand versions for lazy programmers */
#define ABI_MALLOC(ARR,SIZE) ABI_ALLOCATE(ARR,SIZE)
#define ABI_FREE(ARR) ABI_DEALLOCATE(ARR) 
/*
#define ABI_SFREE(ARR) if allocated(ARR) ABI_DEALLOCATE(ARR) 
#define ABI_SFREE_PTR(PT) if associated(PT) ABI_DEALLOCATE(PT) 
*/

#ifdef DEBUG_MODE

#  ifdef HAVE_FC_LONG_LINES
#    define DBG_CHECK(expr,str) if (.not.expr) call assert((expr), str,__FILE__,__LINE__)
#    define DBG_CHKPT(value) write(std_out,*)__FILE__,":",__LINE__,":",value
#    define DBG_ENTER(mode) call sentinel(1,mode,__FILE__,__LINE__)
#    define DBG_EXIT(mode)  call sentinel(2,mode,__FILE__,__LINE__)
#  else
#    define DBG_CHECK(expr,str) if (.not.expr) call assert((expr),str)
#    define DBG_CHKPT(value) write(std_out,*)value
#    define DBG_ENTER(mode) call sentinel(1,mode)
#    define DBG_EXIT(mode)  call sentinel(2,mode)
#  endif

#else
#  define DBG_CHECK(expr,str)
#  define DBG_CHKPT(value)
#  define DBG_ENTER(mode)
#  define DBG_EXIT(mode)
#endif

/* Macro for basic messages */
#ifdef HAVE_FC_LONG_LINES

#  define MSG_COMMENT(msg)      call msg_hndl(msg,"COMMENT","COLL",__FILE__,__LINE__)
#  define MSG_WARNING(msg)      call msg_hndl(msg,"WARNING","COLL",__FILE__,__LINE__)
#  define MSG_ERROR(msg)        call msg_hndl(msg,"ERROR"  ,"COLL",__FILE__,__LINE__)
#  define MSG_BUG(msg)          call msg_hndl(msg,"BUG"    ,"COLL",__FILE__,__LINE__)

#  define MSG_ERROR_NOSTOP(msg,ierr) \
   ierr=ierr+1;call msg_hndl(msg,"ERROR","COLL",__FILE__,__LINE__,NOSTOP=.TRUE.)

#  define ETSF_CHECK_ERROR(lstat,Error_data)   call abietsf_msg_hndl(lstat,Error_data,"COLL",__FILE__,__LINE__)
#  define ETSF_CHECK_MYERROR(lstat,Error_data) call abietsf_msg_hndl(lstat,Error_data,"PERS",__FILE__,__LINE__)
#  define ETSF_WARN(lstat,Error_data) call abietsf_warn(lstat,Error_data,"COLL",__FILE__,__LINE__)

#else
/*
 * Safe macros for emergency cases!
 * Useful if __FILE__ expands to the full path name exceeding
 * the max number of Fortran columns. ISO doesn't define any standard!
 */
#  define MSG_COMMENT(msg)      call msg_hndl(msg,"COMMENT","COLL")
#  define MSG_WARNING(msg)      call msg_hndl(msg,"WARNING","COLL")
#  define MSG_ERROR(msg)        call msg_hndl(msg,"ERROR"  ,"COLL")
#  define MSG_BUG(msg)          call msg_hndl(msg,"BUG"    ,"COLL")

#  define MSG_ERROR_NOSTOP(msg,ierr) \
   ierr=ierr+1;call msg_hndl(msg,"ERROR","COLL",NOSTOP=.TRUE.)

#  define ETSF_CHECK_ERROR(lstat,Error_data)   call abietsf_msg_hndl(lstat,Error_data,"COLL")
#  define ETSF_CHECK_MYERROR(lstat,Error_data) call abietsf_msg_hndl(lstat,Error_data,"PERS")
#  define ETSF_WARN(lstat,Error_data) call abietsf_warn(lstat,Error_data,"COLL")

#endif

/* Macro for checking whether allocation was successful */
#define ABI_CHECK_ALLOC(msg) if (ABI_ALLOC_STAT/=0) MSG_ERROR(msg)

/* Does the compiler support => in declarations? */
#ifdef HAVE_FC_NULL
#  define SET2NULL => null()
#else
#  define SET2NULL
#endif


/* Does the compiler support allocatable arrays in datatypes? */
#ifdef HAVE_FC_ALLOCATABLE_DTARRAYS
#  define DTA_ALLOCATABLE_TG  allocatable,target
#  define DTA_ALLOCATABLE     allocatable
#  define DTA_SFREE(arr)      if (allocated(arr)) deallocate(arr)
#  define DTA_NULLIFY(arr)

#else
/* Have to use pointers in datatypes instead of allocatable arrays */
#  define DTA_ALLOCATABLE_TG  pointer
#  define DTA_ALLOCATABLE     pointer
#  define DTA_SFREE(arr)      if (associated(arr)) deallocate(arr)
#  define DTA_NULLIFY(arr)    nullify(arr)
#endif

/* Dummy use of unused arguments to silence compiler warnings */
#define ABI_UNUSED(var) if (.FALSE.) call unused_var(var)

#if defined HAVE_DEV_TIMER
#  define ABI_TIMER_START(key) call abi_timer(1,ABI_FUNC//":"//key)
#  define ABI_TIMER_STOP(key)  call abi_timer(2,ABI_FUNC//":"//key)
#  define DEV_TIMER_START(key) call abi_timer(1,ABI_FUNC//":"//key)
#  define DEV_TIMER_STOP(key)  call abi_timer(2,ABI_FUNC//":"//key)
#else
#  define ABI_TIMER_START(key) 
#  define ABI_TIMER_STOP(key) 
#  define DEV_TIMER_START(key) 
#  define DEV_TIMER_STOP(key) 
#endif

#ifdef HAVE_TIMER_PAPI
#ifdef HAVE_FC_LONG_LINES
#  define XPAPI_CHECK(check,msg) if (check/=PAPI_OK) call xpapi_handle_error(check,msg,__FILE__,__LINE__) 
#else
#  define XPAPI_CHECK(check,msg) if (check/=PAPI_OK) call xpapi_handle_error(check,msg)
#endif
#else
#  define XPAPI_CHECK(check,msg) 
#endif

/* Error handler for NetCDF calls */
#ifdef HAVE_FC_LONG_LINES
#define NETCDF_CHECK(ncerr,msg) call netcdf_ioerr(ncerr,msg,__FILE__,__LINE__) 
#else
#define NETCDF_CHECK(ncerr,msg) call netcdf_ioerr(ncerr,msg)
#endif

/* Portable support for /dev/null */
#if defined HAVE_OS_WINDOWS
#define NULL_FILE "NUL"
#else
#define NULL_FILE "/dev/null"
#endif

/* OpenMP support */
#ifndef HAVE_OMP_COLLAPSE
#define COLLAPSE(x) 
#endif

/* DFTI macros (should be declared in m_dfti but build-sys tests complain */

#define HAVE_FFT_MKL
#undef HAVE_FFT_MKL

#ifdef HAVE_FC_LONG_LINES
#  define DFTI_CHECK(status) if (status/=0) call check_status(status,__FILE__,__LINE__)
#else
#  define DFTI_CHECK(status) if (status/=0) call check_status(status)
#endif

#endif 
/* _ABINIT_COMMON_H */
