!{\src2tex{textfont=tt}}
!!****m* ABINIT/m_matlu
!! NAME
!!  m_matlu
!!
!! FUNCTION
!!
!! COPYRIGHT
!! Copyright (C) 2006-2012 ABINIT group (BAmadon)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~abinit/COPYING
!! or http://www.gnu.org/copyleft/gpl.txt .
!!
!! INPUTS
!!
!! OUTPUT
!!
!! PARENTS
!!
!! CHILDREN
!!
!! NOTES
!!  subroutines in this module must never call
!!   a subroutine of m_oper, m_green, m_self
!!   in order to avoid circular dependancies
!!
!! SOURCE

#if defined HAVE_CONFIG_H
#include "config.h"
#endif

#include "abi_common.h"

MODULE m_matlu

 use defs_basis
 use defs_datatypes
 use m_errors
 use m_profiling

 implicit none

 private 

 public :: init_matlu
 public :: inverse_matlu
 public :: destroy_matlu
 public :: diff_matlu
 public :: add_matlu
 public :: nullify_matlu
 public :: print_matlu
 public :: sym_matlu
 public :: copy_matlu
 public :: gather_matlu
 public :: zero_matlu
 public :: trace_matlu
 public :: diag_matlu
 public :: rotate_matlu
 public :: shift_matlu
 public :: checkdiag_matlu
 public :: prod_matlu
 public :: conjg_matlu
 public :: ln_matlu
!!***
 

!!****t* m_matlu/matlu_type
!! NAME
!!  matlu_type
!!
!! FUNCTION
!!  This structured datatype contains an matrix for the correlated subspace
!!
!! SOURCE

 type, public :: matlu_type ! for each atom

  integer :: lpawu         
  ! value of the angular momentum for correlated electrons

!  integer :: natom       
   ! number of atoms (given for each atom, not useful..could be changed)
!
!  integer :: mband
!  ! Number of bands
!      
!  integer :: mbandc
!  ! Total number of bands in the Kohn-Sham Basis for PAW+DMFT
!
!  integer :: natpawu         ! Number of correlated atoms 
!
!  integer :: nkpt
!  ! Number of k-point in the IBZ.
  character(len=12) :: whichmatlu
  ! describe the type of local matrix computed (greenLDA, etc..)
!
  integer :: nspinor
  ! Number of spinorial component
!
  integer :: nsppol
  ! Number of polarization 

  complex(dpc), pointer :: mat(:,:,:,:,:)
  ! local quantity
      
 end type matlu_type

!----------------------------------------------------------------------


CONTAINS  !========================================================================================
!!***

!!****f* m_matlu/init_matlu
!! NAME
!! init_matlu
!!
!! FUNCTION
!!  Allocate variables used in type matlu_type.
!!
!! INPUTS
!!  natom = number of atoms
!!  nspinor = number of spinorial components
!!  nsppol = number of polarisation components
!!  lpawu_natom(natom) = value of lpawu for all atoms
!!  maltu <type(matlu_type)>= density matrix in the local orbital basis and related variables
!!
!! OUTPUTS
!!  maltu <type(matlu_type)>= density matrix in the local orbital basis and related variables
!!
!! PARENTS
!!      datafordmft,hubbard_one,m_green,m_matlu,m_oper
!!
!! CHILDREN
!!      checkdiag_matlu
!!
!! SOURCE

subroutine init_matlu(natom,nspinor,nsppol,lpawu_natom,matlu)

 use defs_basis
 use defs_datatypes
 use m_crystal, only : crystal_structure

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
#undef ABI_FUNC
#define ABI_FUNC 'init_matlu'
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer :: natom,nspinor,nsppol
!type
 integer, intent(in) :: lpawu_natom(natom)
 type(matlu_type), intent(inout) :: matlu(natom)
!Local variables ------------------------------------
 integer :: iatom,lpawu

!************************************************************************

! matlu%mband       = mband
! matlu%dmftbandf   = dmftbandf
! matlu%dmftbandi   = dmftbandi
! matlu%nkpt        = nkpt
! matlu%mbandc  = 0
 matlu%nsppol      = nsppol
 matlu%nspinor     = nspinor
 call nullify_matlu(matlu,natom)
 do iatom=1,natom
  lpawu=lpawu_natom(iatom)
  matlu(iatom)%lpawu=lpawu
  if(lpawu.ne.-1) then
   ABI_ALLOCATE(matlu(iatom)%mat,(2*lpawu+1,2*lpawu+1,nsppol,nspinor,nspinor))
   matlu(iatom)%mat=czero
  else
   ABI_ALLOCATE(matlu(iatom)%mat,(0,0,nsppol,nspinor,nspinor))
  endif
 enddo

end subroutine init_matlu
!!***

!!****f* m_matlu/nullify_matlu
!! NAME
!! nullify_matlu
!!
!! FUNCTION
!!  nullify matlu
!!
!! INPUTS
!!  maltu <type(matlu_type)>= density matrix in the local orbital basis and related variables
!!  natom = number of atoms
!!
!! OUTPUT
!!  maltu <type(matlu_type)>= density matrix in the local orbital basis and related variables
!!
!! PARENTS
!!      hubbard_one,m_green,m_matlu,m_oper
!!
!! CHILDREN
!!      checkdiag_matlu
!!
!! SOURCE

subroutine nullify_matlu(matlu,natom)

 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
#undef ABI_FUNC
#define ABI_FUNC 'nullify_matlu'
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer, intent(in) :: natom
 type(matlu_type),intent(inout) :: matlu(natom)
!Local variables-------------------------------
 integer :: iatom

!*********************************************************************

 do iatom=1,natom
  nullify(matlu(iatom)%mat)
 enddo


end subroutine nullify_matlu
!!***

!!****f* m_matlu/zero_matlu
!! NAME
!! zero_matlu
!!
!! FUNCTION
!!  zero_matlu 
!!
!! INPUTS
!!  maltu <type(matlu_type)>= density matrix in the local orbital basis and related variables
!!  natom = number of atoms
!!
!! OUTPUT
!!  maltu <type(matlu_type)>= density matrix in the local orbital basis and related variables
!!
!! PARENTS
!!      m_green,m_matlu
!!
!! CHILDREN
!!      checkdiag_matlu
!!
!! SOURCE

subroutine zero_matlu(matlu,natom)

 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
#undef ABI_FUNC
#define ABI_FUNC 'zero_matlu'
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer, intent(in) :: natom
 type(matlu_type),intent(inout) :: matlu(natom)
!Local variables-------------------------------
 integer :: iatom

!*********************************************************************

 do iatom=1,natom
  matlu(iatom)%mat=czero
 enddo


end subroutine zero_matlu
!!***

!!****f* m_matlu/destroy_matlu
!! NAME
!! destroy_matlu
!!
!! FUNCTION
!!  deallocate matlu
!!
!! INPUTS
!!  maltu <type(matlu_type)>= density matrix in the local orbital basis and related variables
!!  natom = number of atoms
!!
!! OUTPUT
!!
!! PARENTS
!!      datafordmft,hubbard_one,m_green,m_matlu,m_oper
!!
!! CHILDREN
!!      checkdiag_matlu
!!
!! SOURCE

subroutine destroy_matlu(matlu,natom)

 use defs_basis
 use m_crystal, only : crystal_structure

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
#undef ABI_FUNC
#define ABI_FUNC 'destroy_matlu'
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer, intent(in) :: natom
 type(matlu_type),intent(inout) :: matlu(natom)

!Local variables-------------------------------
 integer :: iatom

! *********************************************************************

 do iatom=1,natom
  if ( associated(matlu(iatom)%mat) )   then
    ABI_DEALLOCATE(matlu(iatom)%mat)
  end if
 enddo

end subroutine destroy_matlu
!!***

!!****f* m_matlu/copy_matlu
!! NAME
!! copy_matlu
!!
!! FUNCTION
!!  Copy matlu1 into matlu2 
!!
!! INPUTS
!!  maltu1 <type(matlu_type)>= density matrix matlu1 in the local orbital basis and related variables
!!  natom = number of atoms
!!
!! OUTPUT
!!  maltu2 <type(matlu_type)>= density matrix matlu2 in the local orbital basis and related variables
!!
!! PARENTS
!!      datafordmft,hubbard_one,m_matlu,m_oper,m_self,spectral_function
!!
!! CHILDREN
!!      checkdiag_matlu
!!
!! SOURCE

subroutine copy_matlu(nmat1,nmat2,natom)

 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
#undef ABI_FUNC
#define ABI_FUNC 'copy_matlu'
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!type
 integer, intent(in) :: natom
 type(matlu_type),intent(in) :: nmat1(natom)
 type(matlu_type),intent(out) :: nmat2(natom)

!Local variables-------------------------------
 integer :: iatom, lpawu
! *********************************************************************

 do iatom=1,natom
  lpawu=nmat1(iatom)%lpawu
  if(lpawu.ne.-1) then
   nmat2(iatom)%mat=nmat1(iatom)%mat
  endif
 enddo


end subroutine copy_matlu
!!***

!!****f* m_matlu/print_matlu
!! NAME
!! print_matlu
!!
!! FUNCTION
!!
!! INPUTS
!!  maltu <type(matlu_type)>= density matrix in the local orbital basis and related variables
!!  natom= number of atoms
!!  prtopt= option for printing
!!  opt_diag=   0   print non diagonal matrix (real or complex according to nspinor)
!!             -1   print non diagonal complex matrix
!!            >=1   print diagonal matrix (real or complex according to nspinor)
!!  opt_ab_out=  0  print matrix on std_out
!!             /=0  print matrix on ab_out
!!
!! OUTPUT
!!
!! PARENTS
!!      datafordmft,hubbard_one,impurity_solve,m_green,m_matlu,m_oper,m_self
!!      psichi_renormalization
!!
!! CHILDREN
!!      checkdiag_matlu
!!
!! SOURCE

subroutine print_matlu(matlu,natom,prtopt,opt_diag,opt_ab_out)

 use defs_basis
 use m_crystal, only : crystal_structure

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
#undef ABI_FUNC
#define ABI_FUNC 'print_matlu'
 use interfaces_14_hidewrite
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!type
 integer, intent(in):: natom
 type(matlu_type),intent(in) :: matlu(natom)
 integer, intent(in) :: prtopt
 integer, optional, intent(in) :: opt_diag,opt_ab_out
!Local variables-------------------------------
 integer :: iatom,ispinor,ispinor1,isppol,m1,m,lpawu,nspinor,nsppol,optdiag,optab_out,arg_out
 character(len=500) :: message
 type(matlu_type), allocatable :: matlu_tmp(:)
 character(len=9),parameter :: dspinm(2,2)= RESHAPE((/"n        ","mx       ","my       ","mz       "/),(/2,2/))
! *********************************************************************
 if(present(opt_diag)) then
   optdiag=opt_diag
 else
   optdiag=0
 endif
 if(present(opt_ab_out)) then
   optab_out=opt_ab_out
 else
   optab_out=0
 endif
 if(optab_out==0) then
   arg_out=std_out
 else
   arg_out=ab_out
 endif
 nspinor=matlu(1)%nspinor
 nsppol=matlu(1)%nsppol

 do iatom = 1 , natom
   lpawu=matlu(iatom)%lpawu
   if(lpawu/=-1) then
     write(message,'(2a,i4)')  ch10,'   -------> For Correlated Atom', iatom
     call wrtout(arg_out,  message,'COLL')
     do isppol = 1 , nsppol
       if(nspinor == 1) then
         write(message,'(a,10x,a,i3,i3)')  ch10,'-- polarization spin component',isppol
         call wrtout(arg_out,  message,'COLL')
       endif
       do ispinor = 1 , nspinor
         do ispinor1 = 1, nspinor
           if(nspinor == 2) then
             write(message,'(a,10x,a,i3,i3)')  ch10,'-- spin components',ispinor,ispinor1
             call wrtout(arg_out,  message,'COLL')
           endif
           if(optdiag<=0) then
             do m1=1, 2*lpawu+1
               if(optdiag==0) then
                 if(nspinor==1.and.abs(prtopt)>0) &
&                  write(message,'(5x,20f10.5)') (REAL(matlu(iatom)%mat(m1,m,isppol,ispinor,ispinor1)),m=1,2*lpawu+1)
!&                  write(message,'(5x,14(2f15.11,2x))')((matlu(iatom)%mat(m1,m,isppol,ispinor,ispinor1)),m=1,2*lpawu+1)
                 if(nspinor==2.and.abs(prtopt)>0) &
&                  write(message,'(5x,14(2f9.5,2x))')((matlu(iatom)%mat(m1,m,isppol,ispinor,ispinor1)),m=1,2*lpawu+1)
!&                  write(message,'(5x,14(2f15.11,2x))')((matlu(iatom)%mat(m1,m,isppol,ispinor,ispinor1)),m=1,2*lpawu+1)
               else if(optdiag==-1) then
                 write(message,'(5x,14(2f10.5,2x))')((matlu(iatom)%mat(m1,m,isppol,ispinor,ispinor1)),m=1,2*lpawu+1)
               endif
               call wrtout(arg_out,  message,'COLL')
             enddo
           elseif (optdiag>=1) then
             if(nspinor==1.and.abs(prtopt)>0) &
&             write(message,'(5x,20f10.5)') (REAL(matlu(iatom)%mat(m,m,isppol,ispinor,ispinor1)),m=1,2*lpawu+1)
             if(nspinor==2.and.abs(prtopt)>0) &
&             write(message,'(5x,14(2f9.5,2x))')((matlu(iatom)%mat(m,m,isppol,ispinor,ispinor1)),m=1,2*lpawu+1)
!            write(std_out,'(5x,14(2f9.5,2x))')((matlu(iatom)%mat(m1,m,isppol,ispinor,ispinor1)),m=1,2*lpawu+1)
             call wrtout(arg_out,  message,'COLL')
           endif
         end do ! ispinor1
       end do ! ispinor
       if(nspinor==2.and.prtopt>=5) then
         ABI_DATATYPE_ALLOCATE(matlu_tmp,(0:natom))
         call init_matlu(natom,nspinor,nsppol,matlu(1:natom)%lpawu,matlu_tmp)
         matlu_tmp(iatom)%mat(m1,m,isppol,1,1)= matlu(iatom)%mat(m1,m,isppol,1,1)+matlu(iatom)%mat(m1,m,isppol,2,2)
         matlu_tmp(iatom)%mat(m1,m,isppol,2,2)= matlu(iatom)%mat(m1,m,isppol,1,1)+matlu(iatom)%mat(m1,m,isppol,2,2)
         matlu_tmp(iatom)%mat(m1,m,isppol,1,2)= matlu(iatom)%mat(m1,m,isppol,1,2)+matlu(iatom)%mat(m1,m,isppol,2,1)
         matlu_tmp(iatom)%mat(m1,m,isppol,2,1)= &
&           (matlu(iatom)%mat(m1,m,isppol,1,2)+matlu(iatom)%mat(m1,m,isppol,2,1))*cmplx(0_dp,1_dp)
         do ispinor = 1 , nspinor
           do ispinor1 = 1, nspinor
             write(message,'(a,10x,a,a)')  ch10,'-- spin components',dspinm(ispinor,ispinor1)
             call wrtout(arg_out,  message,'COLL')
             write(message,'(5x,14(2f9.5,2x))')((matlu_tmp(iatom)%mat(m1,m,isppol,ispinor,ispinor1)),m=1,2*lpawu+1)
             call wrtout(arg_out,  message,'COLL')
           end do ! ispinor1
         end do ! ispinor
         call destroy_matlu(matlu_tmp,natom)
         call nullify_matlu(matlu_tmp,natom)
         ABI_DATATYPE_DEALLOCATE(matlu_tmp)
       endif
     enddo ! isppol
!     if(nsppol==1.and.nspinor==1) then
!       write(message,'(a,10x,a,i3,a)')  ch10,'-- polarization spin component',isppol+1,' is identical'
!       call wrtout(arg_out,  message,'COLL')
!     endif
   endif ! lpawu/=1
 enddo ! natom


end subroutine print_matlu
!!***

!{\src2tex{textfont=tt}}
!!****f* m_matlu/sym_matlu
!! NAME
!! sym_matlu
!!
!! FUNCTION
!! Symetrise local quantity.
!!
!! COPYRIGHT
!! Copyright (C) 2005-2012 ABINIT group (BAmadon)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~abinit/COPYING
!! or http://www.gnu.org/copyleft/gpl.txt .
!!
!! INPUTS
!!  cryst_struc <type(crystal_structure)>=crystal structure data
!!  gloc(natom) <type(matlu_type)>= density matrix in the local orbital basis and related variables
!!  pawang <type(pawang)>=paw angular mesh and related data
!!
!! OUTPUT
!!  gloc(natom) <type(matlu_type)>= density matrix symetrized in the local orbital basis and related variables
!!
!! SIDE EFFECTS
!!
!! NOTES
!!
!! PARENTS
!!      datafordmft,hubbard_one,m_green,psichi_renormalization
!!
!! CHILDREN
!!      checkdiag_matlu
!!
!! SOURCE
 subroutine sym_matlu(cryst_struc,gloc,pawang)
 use defs_basis
 use defs_datatypes
 use defs_wvltypes
 use m_crystal, only : crystal_structure

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
#undef ABI_FUNC
#define ABI_FUNC 'sym_matlu'
 use interfaces_41_geometry
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 type(crystal_structure),intent(in) :: cryst_struc
 type(pawang_type),intent(in) :: pawang
!arrays
 type(matlu_type),intent(inout) :: gloc(cryst_struc%natom)
!Local variables-------------------------------
!scalars
 integer :: at_indx,iatom,irot,ispinor,ispinor1,isppol,lpawu,m1,m2,m3,m4,mu
 integer :: natom,ndim,nsppol,nspinor,nu
 complex(dpc) :: sumrho,summag(3),rotmag(3),ci
 real(dp) :: zarot2
!arrays
! complex(dpc),allocatable :: glocnm(:,:,:,:,:)
 type(matlu_type),allocatable :: glocnm(:)
! complex(dpc),allocatable :: glocnms(:,:,:,:,:)
 type(matlu_type),allocatable :: glocnms(:)
 type(matlu_type),allocatable :: glocsym(:)
 real(dp),allocatable :: symrec_cart(:,:,:)

! DBG_ENTER("COLL")

 ci=cone
 nspinor=gloc(1)%nspinor
 nsppol=gloc(1)%nsppol
 natom=cryst_struc%natom

 ABI_DATATYPE_ALLOCATE(glocnm,(natom))
 ABI_DATATYPE_ALLOCATE(glocnms,(natom))
 ABI_DATATYPE_ALLOCATE(glocsym,(natom))
 call init_matlu(natom,nspinor,nsppol,gloc(1:natom)%lpawu,glocnm)
 call init_matlu(natom,nspinor,nsppol,gloc(1:natom)%lpawu,glocnms)
 call init_matlu(natom,nspinor,nsppol,gloc(1:natom)%lpawu,glocsym)


!=========  Case nspinor ==1 ========================

 if (nspinor==1) then
  ispinor=1
  ispinor1=1
  do iatom=1,cryst_struc%natom
   do isppol=1,nsppol
    if(gloc(iatom)%lpawu/=-1) then
     lpawu=gloc(iatom)%lpawu
     do m1=1, 2*lpawu+1
      do m2=1, 2*lpawu+1
       do irot=1,cryst_struc%nsym
        at_indx=cryst_struc%indsym(4,irot,iatom)
        do m3=1, 2*lpawu+1
         do m4=1, 2*lpawu+1
          zarot2=pawang%zarot(m3,m1,lpawu+1,irot)*pawang%zarot(m4,m2,lpawu+1,irot) 
          glocsym(iatom)%mat(m1,m2,isppol,ispinor,ispinor1)=&
&          glocsym(iatom)%mat(m1,m2,isppol,ispinor,ispinor1)&
&          +gloc(at_indx)%mat(m3,m4,isppol,ispinor,ispinor1)*zarot2
         end do  ! m3
        end do  ! m4 
       end do  ! irot
       glocsym(iatom)%mat(m1,m2,isppol,ispinor,ispinor1)=&
&       glocsym(iatom)%mat(m1,m2,isppol,ispinor,ispinor1)/cryst_struc%nsym
      end do ! m2
     end do ! m1
    endif ! lpawu/=-1
   end do ! isppol
  end do ! iatom
!==  Put glocsym into gloc
  do iatom=1,cryst_struc%natom
    if(gloc(iatom)%lpawu/=-1) then
      gloc(iatom)%mat=glocsym(iatom)%mat
!      gloc(iatom)%mat(:,:,1,:,:)=(glocsym(iatom)%mat(:,:,1,:,:) &
!&      + glocsym(iatom)%mat(:,:,2,:,:))/two
!      gloc(iatom)%mat(:,:,2,:,:)= gloc(iatom)%mat(:,:,1,:,:)
!      write(std_out,*) "WARNING: SYM non mag"
!      write(ab_out,*) "WARNING: SYM non mag"
    endif
  end do ! iatom

!=========  Case nspinor ==2 ========================

 else if (nspinor==2) then

!== Allocate temporary arrays
  do iatom=1,cryst_struc%natom
   if(gloc(iatom)%lpawu/=-1) then
    ndim=2*gloc(iatom)%lpawu+1
    ABI_DEALLOCATE(glocnm(iatom)%mat)
    ABI_DEALLOCATE(glocnms(iatom)%mat)
    ABI_DEALLOCATE(glocsym(iatom)%mat)
    ABI_ALLOCATE(glocnm(iatom)%mat,(ndim,ndim,nsppol,4,1))
    ABI_ALLOCATE(glocnms(iatom)%mat,(ndim,ndim,nsppol,4,1))
    ABI_ALLOCATE(glocsym(iatom)%mat,(ndim,ndim,nsppol,2,2))
   endif
  enddo
  ABI_ALLOCATE(symrec_cart,(3,3,cryst_struc%nsym))

!==  Compute symrec_cart
  do irot=1,cryst_struc%nsym
   call symredcart(cryst_struc%gprimd,cryst_struc%rprimd,symrec_cart(:,:,irot),cryst_struc%symrec(:,:,irot))
  end do

!==  Compute density matrix in density and magnetisation representation
  call chg_repr_matlu(gloc,glocnm,cryst_struc%natom,option=1,prtopt=1)

!==  Do the sum over symetrized density matrix (in n,m repr)
  isppol=1
  do iatom=1,cryst_struc%natom
   if(gloc(iatom)%lpawu/=-1) then
    lpawu=gloc(iatom)%lpawu
    ndim=2*gloc(iatom)%lpawu+1
    do m1=1, 2*lpawu+1
     do m2=1, 2*lpawu+1
      sumrho=czero
      rotmag=czero
      do irot=1,cryst_struc%nsym
       summag=czero
       at_indx=cryst_struc%indsym(4,irot,iatom)
       do m3=1, 2*lpawu+1
        do m4=1, 2*lpawu+1
         zarot2=pawang%zarot(m3,m2,lpawu+1,irot)*pawang%zarot(m4,m1,lpawu+1,irot) 
         sumrho=sumrho +  glocnm(at_indx)%mat(m4,m3,isppol,1,1)  * zarot2 
         do mu=1,3
          summag(mu)=summag(mu) + glocnm(at_indx)%mat(m4,m3,isppol,mu+1,1) * zarot2 
         enddo
        end do ! m3
       end do !m4

!       ==  special case of magnetization 
       do nu=1,3
        do mu=1,3
         rotmag(mu)=rotmag(mu)+symrec_cart(mu,nu,irot)*summag(nu) 
        end do
       end do
!      write(std_out,'(a,3i4,2x,3(2f10.5,2x))') "rotmag",irot,m1,m2,(rotmag(mu),mu=1,3)
      end do ! irot

!       ==  Normalizes sum
      sumrho=sumrho/cryst_struc%nsym 
!        sumrho=glocnm(isppol,1,iatom,m1,m2) ! test without sym
      glocnms(iatom)%mat(m1,m2,isppol,1,1)=sumrho
      do mu=1,3
       rotmag(mu)=rotmag(mu)/cryst_struc%nsym 
!          rotmag(mu)=glocnm(isppol,mu+1,iatom,m1,m2) ! test without sym
       glocnms(iatom)%mat(m1,m2,isppol,mu+1,1)=rotmag(mu)
      enddo
     end do  ! m2
    end do ! m1
   endif ! lpawu/=-1
  end do ! iatom

!==  Compute back density matrix in upup dndn updn dnup representation
  call chg_repr_matlu(glocsym,glocnms,cryst_struc%natom,option=-1,prtopt=1)

!==  Put glocsym into gloc
  do iatom=1,cryst_struc%natom
    if(gloc(iatom)%lpawu/=-1) then
      gloc(iatom)%mat=glocsym(iatom)%mat
!      gloc(iatom)%mat(:,:,1,:,:)=(glocsym(iatom)%mat(:,:,1,:,:) &
!&      + glocsym(iatom)%mat(:,:,2,:,:))/two
!      gloc(iatom)%mat(:,:,2,:,:)= gloc(iatom)%mat(:,:,1,:,:)
!      write(std_out,*) "WARNING: SYM non mag"
!      write(ab_out,*) "WARNING: SYM non mag"
    endif
  end do ! iatom

  ABI_DEALLOCATE(symrec_cart)
 endif

 call destroy_matlu(glocnm,cryst_struc%natom)
 call destroy_matlu(glocnms,cryst_struc%natom)
 call destroy_matlu(glocsym,cryst_struc%natom)
 ABI_DATATYPE_DEALLOCATE(glocnm)
 ABI_DATATYPE_DEALLOCATE(glocnms)
 ABI_DATATYPE_DEALLOCATE(glocsym)
!==============end of nspinor==2 case ===========


! DBG_EXIT("COLL")

 end subroutine sym_matlu
!!***

!!****f* m_matlu/inverse_matlu
!! NAME
!! inverse_matlu
!!
!! FUNCTION
!! Inverse local quantity.
!!
!! COPYRIGHT
!! Copyright (C) 2005-2012 ABINIT group (BAmadon)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~abinit/COPYING
!! or http://www.gnu.org/copyleft/gpl.txt .
!!
!! INPUTS
!!  matlu(natom)%(nsppol,nspinor,nspinor,ndim,ndim) :: input quantity to inverse
!!  natom=number of atoms in cell.
!!  prtopt= option to define level of printing
!!
!! OUTPUT
!!  matlu(natom)%(nsppol,nspinor,nspinor,ndim,ndim) :: inverse of input matrix
!!
!! SIDE EFFECTS
!!
!! NOTES
!!
!! PARENTS
!!      m_oper
!!
!! CHILDREN
!!      checkdiag_matlu
!!
!! SOURCE
 subroutine inverse_matlu(matlu,natom,prtopt)
 use defs_basis
 use defs_datatypes
 use defs_wvltypes
 use m_crystal, only : crystal_structure

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
#undef ABI_FUNC
#define ABI_FUNC 'inverse_matlu'
 use interfaces_32_util
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer, intent(in) :: natom
 integer, intent(in) :: prtopt
!arrays
 type(matlu_type),intent(inout) :: matlu(natom)
!Local variables-------------------------------
 integer :: iatom,tndim
 integer :: nsppol,nspinor
!scalars
 type(coeff2c_type),allocatable :: gathermatlu(:)
 !************************************************************************


 nspinor=matlu(1)%nspinor
 nsppol=matlu(1)%nsppol
 if(prtopt>0) then
 endif
 ABI_DATATYPE_ALLOCATE(gathermatlu,(natom))
 do iatom=1,natom
   if(matlu(iatom)%lpawu.ne.-1) then
     tndim=nsppol*nspinor*(2*matlu(iatom)%lpawu+1)
     ABI_ALLOCATE(gathermatlu(iatom)%value,(tndim,tndim))
     gathermatlu(iatom)%value=czero
   endif
 enddo

 call gather_matlu(matlu,gathermatlu,natom,option=1,prtopt=1)
 do iatom=1,natom
   if(matlu(iatom)%lpawu.ne.-1) then
     tndim=nsppol*nspinor*(2*matlu(iatom)%lpawu+1)
     call matcginv_dpc(gathermatlu(iatom)%value,tndim,tndim)
   endif
 enddo
 call gather_matlu(matlu,gathermatlu,natom,option=-1,prtopt=1)

 do iatom=1,natom
   if(matlu(iatom)%lpawu.ne.-1) then
     ABI_DEALLOCATE(gathermatlu(iatom)%value)
   endif
 enddo
 ABI_DATATYPE_DEALLOCATE(gathermatlu)
 end subroutine inverse_matlu
!!***

!!****f* m_matlu/diff_matlu
!! NAME
!! diff_matlu
!!
!! FUNCTION
!!
!! INPUTS
!!  char1 = character describing matlu1
!!  char2 = character describing matlu2
!!  matlu1(natom) <type(matlu_type)>= density matrix 1 in the local orbital basis and related variables
!!  matlu2(natom) <type(matlu_type)>= density matrix 2 in the local orbital basis and related variables
!!  natom = number of atoms
!!  option =1      if diff> toldiff , stop 
!!          0      print diff and toldiff
!!          else   do not test and do not print
!!  toldiff = maximum value for the difference between matlu1 and matlu2
!!
!! OUTPUT
!!
!! PARENTS
!!      datafordmft,m_green,m_oper
!!
!! CHILDREN
!!      checkdiag_matlu
!!
!! SOURCE
subroutine diff_matlu(char1,char2,matlu1,matlu2,natom,option,toldiff,ierr)

 use defs_basis
 use m_paw_dmft, only : paw_dmft_type
 use m_crystal, only : crystal_structure

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
#undef ABI_FUNC
#define ABI_FUNC 'diff_matlu'
 use interfaces_14_hidewrite
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!type
 integer,intent(in) :: natom,option
 type(matlu_type), intent(in) :: matlu1(natom),matlu2(natom)
 character(len=*), intent(in) :: char1,char2
 real(dp),intent(in) :: toldiff
 integer,intent(out), optional :: ierr

!local variables-------------------------------
 integer :: iatom,idiff,ispinor,ispinor1,isppol,m1,m,lpawu,nspinor,nsppol
 real(dp) :: matludiff
 character(len=500) :: message
! *********************************************************************
 nsppol=matlu1(1)%nsppol
 nspinor=matlu1(1)%nspinor

 matludiff=zero
 idiff=0
 do iatom = 1 , natom
  lpawu=matlu1(iatom)%lpawu
  if(lpawu/=-1) then
   do isppol = 1 , nsppol
    do ispinor = 1 , nspinor
     do ispinor1 = 1, nspinor
      do m1 = 1 , 2*lpawu+1
       do m = 1 ,  2*lpawu+1
        idiff=idiff+1
        matludiff=matludiff+ &
&        sqrt( real(matlu1(iatom)%mat(m1,m,isppol,ispinor,ispinor1)      &
&            -      matlu2(iatom)%mat(m1,m,isppol,ispinor,ispinor1))**2  &
&            + imag(matlu1(iatom)%mat(m1,m,isppol,ispinor,ispinor1)      &
&            -      matlu2(iatom)%mat(m1,m,isppol,ispinor,ispinor1))**2  )
!       write(std_out,*) m,m1,matlu1(iatom)%mat(m1,m,isppol,ispinor,ispinor1),matlu2(iatom)%mat(m1,m,isppol,ispinor,ispinor1),matludiff
       enddo
      enddo
     end do ! ispinor1
    end do ! ispinor
   enddo ! isppol
  endif ! lpawu/=1
 enddo ! natom
 matludiff=matludiff/float(idiff)
 if(option==1.or.option==0) then
  if( matludiff < toldiff ) then
   write(message,'(5a,6x,3a,4x,e12.4,a,e12.4)') ch10,&
&   '   ** Differences between ',trim(char1),' and ',ch10,trim(char2),' are small enough:',&
&   ch10,matludiff,' is lower than',toldiff
   call wrtout(std_out,message,'COLL')
   if(present(ierr)) ierr=0
  else
   write(message,'(5a,3x,3a,3x,e12.4,a,e12.4)') ch10,&
&   '   Warning : Differences between ',trim(char1),' and ',ch10,trim(char2),' is too large:',&
&   ch10,matludiff,' is larger than',toldiff
   call wrtout(std_out,message,'COLL')
!   write(message,'(8a,4x,e12.4,a,e12.4)') ch10,"  Matrix for ",trim(char1)
   write(message,'(a,3x,a)') ch10,trim(char1)
   call wrtout(std_out,message,'COLL')
   call print_matlu(matlu1,natom,prtopt=1,opt_diag=-1)
   write(message,'(a,3x,a)') ch10,trim(char2)
   call wrtout(std_out,message,'COLL')
   call print_matlu(matlu2,natom,prtopt=1,opt_diag=-1)
   if(option==1) then 
     MSG_ERROR("Cannot continue")
   end if
   if(present(ierr)) ierr=-1
  endif
 endif
 
end subroutine diff_matlu
!!***

!!****f* m_matlu/add_matlu
!! NAME
!! add_matlu
!!
!! FUNCTION
!!
!! INPUTS
!!  maltu1 <type(matlu_type)>= density matrix matlu1 in the local orbital basis and related variables
!!  maltu2 <type(matlu_type)>= density matrix matlu2 in the local orbital basis and related variables
!!  natom = number of atoms
!!  sign_matlu2= 1 add matlu1 and matlu2
!!              -1 substract matlu2 to matlu1
!!
!! OUTPUT
!!  maltu3 <type(matlu_type)>= density matrix matlu3, sum/substract matlu1 and matlu2
!!
!! PARENTS
!!      dyson,m_green
!!
!! CHILDREN
!!      checkdiag_matlu
!!
!! SOURCE
subroutine add_matlu(matlu1,matlu2,matlu3,natom,sign_matlu2)

 use defs_basis
 use m_paw_dmft, only : paw_dmft_type
 use m_crystal, only : crystal_structure

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
#undef ABI_FUNC
#define ABI_FUNC 'add_matlu'
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!type
 integer,intent(in) :: natom,sign_matlu2
 type(matlu_type), intent(in) :: matlu1(natom),matlu2(natom)
 type(matlu_type), intent(out) :: matlu3(natom)

!local variables-------------------------------
 integer :: iatom
! *********************************************************************

 do iatom = 1 , natom
   matlu3(iatom)%mat=matlu1(iatom)%mat+float(sign_matlu2)*matlu2(iatom)%mat
 enddo ! natom
 
end subroutine add_matlu
!!***

!!****f* m_matlu/chg_repr_matlu
!! NAME
!! chg_repr_matlu
!!
!! FUNCTION
!! Change representation of density matrix (useful for nspinor=2)
!!
!! COPYRIGHT
!! Copyright (C) 2005-2012 ABINIT group (BAmadon)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~abinit/COPYING
!! or http://www.gnu.org/copyleft/gpl.txt .
!!
!! INPUTS
!!  glocspsp(natom)%(nsppol,nspinor,nspinor,ndim,ndim) :: density matrix in the spin spin representation
!!  glocnm(natom)%(nsppol,nspinor,nspinor,ndim,ndim) :: density matrix in the magnetization representation
!!  natom=number of atoms in cell.
!!  option= 1 glocspsp is input, glocnm is computed
!!  option= -1 glocspsp is computed, glocnm is input
!!  prtopt= option to define level of printing
!!
!! OUTPUT
!!  glocspsp(natom)%(nsppol,nspinor,nspinor,ndim,ndim) :: density matrix in the spin spin representation
!!  glocnm(natom)%(nsppol,nspinor,nspinor,ndim,ndim) :: density matrix in the magnetization representation
!!
!! SIDE EFFECTS
!!
!! NOTES
!!
!! PARENTS
!!      m_matlu
!!
!! CHILDREN
!!      checkdiag_matlu
!!
!! SOURCE
 subroutine chg_repr_matlu(glocspsp,glocnm,natom,option,prtopt)
 use defs_basis
 use defs_datatypes
 use defs_wvltypes
 use m_crystal, only : crystal_structure

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
#undef ABI_FUNC
#define ABI_FUNC 'chg_repr_matlu'
 use interfaces_14_hidewrite
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: natom,option,prtopt
!arrays
 type(matlu_type),intent(inout) :: glocspsp(natom)
 type(matlu_type),intent(inout) :: glocnm(natom)
!Local variables-------------------------------
!scalars
 integer :: iatom,isppol,lpawu,m1,m2,ndim,nsppol,mu
 complex(dpc) :: ci
 character(len=500) :: message

! DBG_ENTER("COLL")

 ci=j_dpc

!==  Compute density matrix in density magnetisation representation
 if (option==1) then
  nsppol=glocspsp(1)%nsppol
  do isppol=1,nsppol
   do iatom=1,natom
    if(glocspsp(iatom)%lpawu/=-1) then
     ndim=2*glocspsp(iatom)%lpawu+1
     do m1=1,ndim
      do m2=1,ndim
       glocnm(iatom)%mat(m1,m2,isppol,1,1)=glocspsp(iatom)%mat(m1,m2,isppol,1,1) &
&                                         +glocspsp(iatom)%mat(m1,m2,isppol,2,2)
       glocnm(iatom)%mat(m1,m2,isppol,4,1)=glocspsp(iatom)%mat(m1,m2,isppol,1,1) &
&                                         -glocspsp(iatom)%mat(m1,m2,isppol,2,2)
       glocnm(iatom)%mat(m1,m2,isppol,2,1)=glocspsp(iatom)%mat(m1,m2,isppol,1,2) &
&                                         +glocspsp(iatom)%mat(m1,m2,isppol,2,1)
       glocnm(iatom)%mat(m1,m2,isppol,3,1)= &
&                               cmplx((imag(glocspsp(iatom)%mat(m1,m2,isppol,2,1))   &
&                                     -imag(glocspsp(iatom)%mat(m1,m2,isppol,1,2))), &
&                                    (-real(glocspsp(iatom)%mat(m1,m2,isppol,2,1))+  &
&                                      real(glocspsp(iatom)%mat(m1,m2,isppol,1,2))),kind=dp)
      enddo  ! m2
     enddo ! m1
     if(abs(prtopt)>=3) then
      write(message,'(a)') "        -- in n, m repr "
      call wrtout(std_out,  message,'COLL')
      do mu=1,4
       do m1=1,ndim
        write(message,'(8x,(14(2f9.5,2x)))')(glocnm(iatom)%mat(m1,m2,isppol,mu,1),m2=1,ndim)
        call wrtout(std_out,  message,'COLL')
       enddo ! m1
       write(message,'(a)') ch10
       call wrtout(std_out,  message,'COLL')
      enddo ! mu
     endif ! prtopt >3
    endif ! lpawu/=-1
   enddo
  enddo

!==  Compute back density matrix in upup dndn updn dnup representation
 else  if (option==-1) then
  isppol=1
  do iatom=1,natom
   if(glocnm(iatom)%lpawu/=-1) then
    lpawu=glocnm(iatom)%lpawu
    ndim=2*glocnm(iatom)%lpawu+1
    do m1=1, 2*lpawu+1
     do m2=1, 2*lpawu+1
      glocspsp(iatom)%mat(m1,m2,isppol,1,1)=half*(glocnm(iatom)%mat(m1,m2,isppol,1,1)+glocnm(iatom)%mat(m1,m2,isppol,4,1))
      glocspsp(iatom)%mat(m1,m2,isppol,2,2)=half*(glocnm(iatom)%mat(m1,m2,isppol,1,1)-glocnm(iatom)%mat(m1,m2,isppol,4,1))
      glocspsp(iatom)%mat(m1,m2,isppol,1,2)=half*(glocnm(iatom)%mat(m1,m2,isppol,2,1)-ci*glocnm(iatom)%mat(m1,m2,isppol,3,1))
      glocspsp(iatom)%mat(m1,m2,isppol,2,1)=half*(glocnm(iatom)%mat(m1,m2,isppol,2,1)+ci*glocnm(iatom)%mat(m1,m2,isppol,3,1))
     end do  ! m2
    end do ! m1
    if(abs(prtopt)>6) then
     write(message,'(a)') "        -- in spin spin repr "
     call wrtout(std_out,  message,'COLL')
     do mu=1,4
      do m1=1,ndim
       write(message,'(8x,14(2f9.5,2x))')(glocspsp(iatom)%mat(m1,m2,isppol,mu,1),m2=1,ndim)
       call wrtout(std_out,  message,'COLL')
      enddo
      write(message,'(a)') ch10
      call wrtout(std_out,  message,'COLL')
     enddo
    endif !prtopt>3
   endif ! lpawu/=-1
  end do ! iatom
 else  
  message = "stop in chg_repr_matlu"
  MSG_ERROR(message)
 endif


! DBG_EXIT("COLL")

 end subroutine chg_repr_matlu
!!***

!!****f* m_matlu/trace_matlu
!! NAME
!! trace_matlu
!!
!! FUNCTION
!!  Compute the trace of the matlu matrix
!!
!! INPUTS
!!  maltu(natom) <type(matlu_type)>= density matrix in the 
!!               local orbital basis and related variables
!!  natom = number of atoms
!!
!! OUTPUT
!!  trace_loc(natom,nsppol+1)= trace for each atoms and each polarization,
!!                             trace_loc(iatom,nsppol+1) is
!!                             the full trace over polarization also.
!!
!! PARENTS
!!      m_green,m_oper
!!
!! CHILDREN
!!      checkdiag_matlu
!!
!! SOURCE
 subroutine trace_matlu(matlu,natom,trace_loc,itau)

 use defs_basis
 use defs_datatypes

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
#undef ABI_FUNC
#define ABI_FUNC 'trace_matlu'
 use interfaces_14_hidewrite
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!type
 integer, intent(in) :: natom
 type(matlu_type), intent(in) :: matlu(natom)
 real(dp),intent(inout),target,optional :: trace_loc(natom,matlu(1)%nsppol+1)
 integer, intent(in),optional :: itau

!local variables-------------------------------
 integer :: iatom,isppol,ispinor,m,lpawu
 integer :: nsppol,nspinor
 real(dp), pointer :: traceloc(:,:)
 character(len=500) :: message
! *********************************************************************
 nsppol=matlu(1)%nsppol
 nspinor=matlu(1)%nspinor
 if(present(trace_loc)) then
   traceloc=>trace_loc
 else
   ABI_ALLOCATE(traceloc,(natom,matlu(1)%nsppol+1))
 endif

 traceloc=zero
 do iatom = 1 , natom
   lpawu=matlu(iatom)%lpawu
   if(lpawu/=-1) then 
      write(message,'(2a,i4)')  ch10,'   -------> For Correlated Atom', iatom
     if(.not.present(itau)) call wrtout(std_out,  message,'COLL')
     if(present(itau)) then
       if (itau>0) call wrtout(std_out,  message,'COLL')
     endif
     do isppol = 1 , nsppol
       do ispinor = 1 , nspinor
         do m = 1 ,  2*lpawu+1
           traceloc(iatom,isppol)=traceloc(iatom,isppol)+&
&           matlu(iatom)%mat(m,m,isppol,ispinor,ispinor)
         enddo 
       enddo 
      traceloc(iatom,nsppol+1)=traceloc(iatom,nsppol+1)+traceloc(iatom,isppol)
     enddo 
     if(nsppol==1.and.nspinor==1)  traceloc(iatom,nsppol+1)=traceloc(iatom,nsppol+1)*two
     if(.not.present(itau)) then 
       write(message,'(8x,a,f12.6)')   'Nb of Corr. elec. from G(w) is:'&
&       ,traceloc(iatom,nsppol+1)
       call wrtout(std_out,  message,'COLL')
     endif
     if(present(itau)) then
       if(itau==1) then
         write(message,'(8x,a,f12.6)')   'Nb of Corr. elec. from G(tau) is:'&
&         ,traceloc(iatom,nsppol+1)
         call wrtout(std_out,  message,'COLL')
       else if(itau==-1) then
         write(message,'(8x,a,f12.6)')   'Nb: Sum of the values of G0(tau=0-) is:'&
&       ,traceloc(iatom,nsppol+1)
         call wrtout(std_out,  message,'COLL')
       else if(itau==4) then
         write(message,'(8x,a,f12.6)')   'Trace of matlu matrix is:'&
&         ,traceloc(iatom,nsppol+1)
         call wrtout(std_out,  message,'COLL')
       endif
     endif
   endif
 enddo
 if(.not.present(trace_loc)) then
  ABI_DEALLOCATE(traceloc)
 endif

 end subroutine trace_matlu
!!***

!!****f* m_matlu/gather_matlu
!! NAME
!! gather_matlu
!!
!! FUNCTION
!! Create new array from matlu
!!
!! COPYRIGHT
!! Copyright (C) 2005-2012 ABINIT group (BAmadon)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~abinit/COPYING
!! or http://www.gnu.org/copyleft/gpl.txt .
!!
!! INPUTS
!!  gloc(natom) <type(matlu_type)>        = density matrix in the spin spin representation
!!  gatherloc(natom) <type(coeff2c_type)> = density matrix where spin and angular momentum are gathered in the same index
!!  natom=number of atoms in cell.
!!  option= 1 go from gloc to gathergloc
!!  option= -1 go from gathergloc to gloc
!!  prtopt= option to define level of printing
!!
!! OUTPUT
!!  gloc(natom) <type(matlu_type)>        = density matrix in the spin spin representation
!!  gatherloc(natom) <type(coeff2c_type)> = density matrix where spin and angular momentum are gathered in the same index
!!
!! SIDE EFFECTS
!!
!! PARENTS
!!      m_matlu,psichi_renormalization
!!
!! CHILDREN
!!      checkdiag_matlu
!!
!! SOURCE
 subroutine gather_matlu(gloc,gathergloc,natom,option,prtopt)
 use defs_basis
 use defs_datatypes
 use defs_wvltypes
 use m_crystal, only : crystal_structure

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
#undef ABI_FUNC
#define ABI_FUNC 'gather_matlu'
 use interfaces_14_hidewrite
!End of the abilint section

 implicit none

! type  matlus_type 
!  SEQUENCE
!  complex(dpc), pointer :: mat(:,:)
! end type matlus_type

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: natom,option,prtopt
 type(coeff2c_type), intent(inout) :: gathergloc(natom)
 type(matlu_type),intent(inout) :: gloc(natom)
!Local variables-------------------------------
!scalars
 integer :: iatom,im1,im2,ispinor,ispinor1,isppol,isppol1
 integer :: jc1,jc2,ml1,ml2,ndim,nspinor,nsppol,tndim
 complex(dpc) :: ci
 character(len=500) :: message

! DBG_ENTER("COLL")
 nsppol=gloc(1)%nsppol
 nspinor=gloc(1)%nspinor

 ci=j_dpc


 do iatom=1,natom
   if(gloc(iatom)%lpawu.ne.-1) then
!==-------------------------------------

     ndim=2*gloc(iatom)%lpawu+1
     tndim=nsppol*nspinor*ndim

!== Put norm into array "gathergloc"
     jc1=0
     do isppol=1,nsppol
       do ispinor=1,nspinor
         do ml1=1,ndim
           jc1=jc1+1
           jc2=0
           do isppol1=1,nsppol
             do ispinor1=1,nspinor
               do ml2=1,ndim
                 jc2=jc2+1
                 if(option==1) then
                   if(isppol==isppol1) then
                     gathergloc(iatom)%value(jc1,jc2)=gloc(iatom)%mat(ml1,ml2,isppol,ispinor,ispinor1)
                   endif
                 else if(option==-1) then
                   if(isppol==isppol1) then
                     gloc(iatom)%mat(ml1,ml2,isppol,ispinor,ispinor1)=gathergloc(iatom)%value(jc1,jc2)
                   endif
                 endif
               enddo
             enddo ! ispinor1
           enddo ! isppol1
         enddo 
       enddo !ispinor
     enddo ! isppol
   endif
 enddo ! iatom
 if(option==1.and.prtopt==3) then
   do iatom=1,natom
     if(gloc(iatom)%lpawu.ne.-1) then
       tndim=nsppol*nspinor*(2*gloc(iatom)%lpawu+1)
       write(message,'(2a,i5)') ch10,' (gathermatlu:) For atom', iatom
       call wrtout(std_out,message,'COLL')
       do im1=1,tndim
         write(message,'(12(1x,18(1x,"(",f9.3,",",f9.3,")")))')&
&         (gathergloc(iatom)%value(im1,im2),im2=1,tndim)
         call wrtout(std_out,message,'COLL')
       end do
     endif
   enddo ! iatom
 else if(option==-1.and.prtopt==3) then
   call print_matlu(gloc,natom,prtopt)
 endif



! DBG_EXIT("COLL")

 end subroutine gather_matlu
!!***

!!****f* m_matlu/diag_matlu
!! NAME
!! diag_matlu
!!
!! FUNCTION
!! Diagonalize matlu matrix
!!
!! COPYRIGHT
!! Copyright (C) 2005-2012 ABINIT group (BAmadon)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~abinit/COPYING
!! or http://www.gnu.org/copyleft/gpl.txt .
!!
!! INPUTS
!!  matlu(natom)%(nsppol,nspinor,nspinor,ndim,ndim) :: input quantity to diagonalize
!!  natom=number of atoms 
!!  prtopt= option to define level of printing
!!  nsppol_imp= if 1, one can diagonalize with the same matrix the Up
!!   and Dn matlu matrix. It is convenient.
!!
!! OUTPUT
!!  matlu_diag(natom)%(nsppol,nspinor,nspinor,ndim,ndim) :: diagonalized density matrix
!!  eigvectmatlu(natom) <type(coeff2c_type)> = Eigenvectors corresponding to the diagonalization
!!
!! SIDE EFFECTS
!!
!! NOTES
!!
!! PARENTS
!!      hubbard_one
!!
!! CHILDREN
!!      checkdiag_matlu
!!
!! SOURCE
 subroutine diag_matlu(matlu,matlu_diag,natom,prtopt,eigvectmatlu,nsppol_imp)
 use defs_basis
 use defs_datatypes
 use defs_wvltypes
 use m_crystal, only : crystal_structure

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
#undef ABI_FUNC
#define ABI_FUNC 'diag_matlu'
 use interfaces_14_hidewrite
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer, intent(in) :: natom
 integer, intent(in) :: prtopt
!arrays
 type(matlu_type),intent(inout) :: matlu(natom)
 type(matlu_type),intent(out) :: matlu_diag(natom)
 type(coeff2c_type),optional,intent(out) :: eigvectmatlu(natom,matlu(1)%nsppol)
 integer,optional,intent(in) :: nsppol_imp
!Local variables-------------------------------
!scalars
 integer :: iatom,im,im1,im2,im3,imc,imc1,info,ispinor,ispinor1,isppol,lwork,tndim
 integer :: nsppol,nsppolimp,nspinor
 character(len=500) :: message
!arrays
 type(coeff2c_type),allocatable :: gathermatlu(:)
 real(dp),allocatable :: eig(:),rwork(:)
 complex(dpc),allocatable :: zwork(:)
 logical :: donotdiag,print_temp_mat2
 complex(dpc),allocatable :: temp_mat(:,:)
 complex(dpc),allocatable :: temp_mat2(:,:)
!************************************************************************

 call zero_matlu(matlu_diag,natom)
 nsppol=matlu(1)%nsppol
 if(present(nsppol_imp)) then
   nsppolimp=nsppol_imp
 else 
   nsppolimp=nsppol
 endif
 nspinor=matlu(1)%nspinor

 donotdiag=.true.
! ===========================
! Check is diagonalization is necessary and how
! ===========================
 do isppol=1,matlu(1)%nsppol
   do iatom=1,natom
      if(matlu(iatom)%lpawu.ne.-1) then
       tndim=(2*matlu(iatom)%lpawu+1)
        do im1=1,tndim
          do im2=1,tndim
            do ispinor=1,nspinor
              do ispinor1=1,nspinor
                if(abs(matlu(iatom)%mat(im1,im2,isppol,ispinor,ispinor1))>tol8.and.&
&                 (im1/=im2.or.ispinor/=ispinor1)) then
!                 if matrix is diagonal: do not diagonalize
                  donotdiag=.false.
                  exit
                endif
              enddo
            enddo
          enddo
        enddo
      endif
   enddo
 enddo

 if(donotdiag) then
   do isppol=1,matlu(1)%nsppol
     do iatom=1,natom
        if(matlu(iatom)%lpawu.ne.-1) then
          tndim=(2*matlu(iatom)%lpawu+1)
          eigvectmatlu(iatom,isppol)%value(:,:)=czero
          do im1=1,tndim
            eigvectmatlu(iatom,isppol)%value(im1,im1)=cone
          enddo
        endif
     enddo
   enddo
   call copy_matlu(matlu,matlu_diag,natom)
   write(message,'(a)')  "   Diagonalisation of matlu will not be performed"
   call wrtout(std_out,message,'COLL')
   return
 endif


! Below, gathermatlu is used to store eigenvectors (after diagonalization) 
! For nsppol=2, and if nsppolimp=1, these eigenvectors computed for isppol=1, and applied to
! rotate matrix for isppol=2. It is the reason why the sum below is only
! from 1 to nsppolimp !



 do isppol=1,nsppolimp ! 
! ===========================
! Define gathermatlu
! ===========================
   ABI_DATATYPE_ALLOCATE(gathermatlu,(natom))
   do iatom=1,natom
     if(matlu(iatom)%lpawu.ne.-1) then
       tndim=nspinor*(2*matlu(iatom)%lpawu+1)
       ABI_ALLOCATE(gathermatlu(iatom)%value,(tndim,tndim))
!debug       temp_mat2=czero
       gathermatlu(iatom)%value=czero
     endif
   enddo
   if(nsppol==1.and.nspinor==2) then
     call gather_matlu(matlu,gathermatlu,natom,option=1,prtopt=1)
   else if((nsppol==2.or.nsppol==1).and.nspinor==1) then
     do iatom=1,natom
       if(matlu(iatom)%lpawu.ne.-1) then
         do im1=1,tndim
           do im2=1,tndim
             gathermatlu(iatom)%value(im1,im2)=matlu(iatom)%mat(im1,im2,isppol,1,1)
           enddo
         enddo
       endif
     enddo
   endif
! ===========================
! Diagonalize
! ===========================
   do iatom=1,natom
     if(matlu(iatom)%lpawu.ne.-1) then
       if(isppol<=nsppolimp) then
         tndim=nspinor*(2*matlu(iatom)%lpawu+1)
         lwork=2*tndim-1
         ABI_ALLOCATE(rwork,(3*tndim-2))
         ABI_ALLOCATE(zwork,(lwork))
         ABI_ALLOCATE(eig,(tndim))
         if(prtopt>=4) then
           write(message,'(a,i4,a,i4)')  "       BEFORE DIAGONALIZATION for atom",iatom,"  and isppol",isppol
           call wrtout(std_out,message,'COLL')
           do im1=1,tndim
             write(message,'(12(1x,18(1x,"(",f9.3,",",f9.3,")")))')&
&             (gathermatlu(iatom)%value(im1,im2),im2=1,tndim)
             call wrtout(std_out,message,'COLL')
           end do
         endif
!           write(std_out,*)"diag"
         call zheev('v','u',tndim,gathermatlu(iatom)%value,tndim,eig,zwork,lwork,rwork,info)
         if(prtopt>=3) then
           write(message,'(a)') ch10
           call wrtout(std_out,message,'COLL')
           write(message,'(a,i4,a,i4)')  "       EIGENVECTORS for atom",iatom,"  and isppol",isppol
           call wrtout(std_out,message,'COLL')
           do im1=1,tndim
             write(message,'(12(1x,18(1x,"(",f9.3,",",f9.3,")")))')&
&             (gathermatlu(iatom)%value(im1,im2),im2=1,tndim)
             call wrtout(std_out,message,'COLL')
           end do
         endif
!       write(std_out,*) "eig",eig
! ===========================
! Put eigenvalue in matlu_diag
! ===========================
         imc=0
         do ispinor=1,nspinor
           do im1=1,2*matlu(iatom)%lpawu+1
             imc=imc+1
             matlu_diag(iatom)%mat(im1,im1,isppol,ispinor,ispinor)=eig(imc)
!debug             temp_mat2(imc,imc)=eig(imc)
           enddo
         enddo
         if(prtopt>=2) then
!            write(message,'(a,12(1x,18(1x,"(",f9.3,",",f9.3,")")))')&
!&            ch10,(eig(im1),im1=1,tndim)
!             call wrtout(std_out,message,'COLL')
            write(std_out,*) "EIG", eig
         endif
         ABI_DEALLOCATE(zwork)
         ABI_DEALLOCATE(rwork)
         ABI_DEALLOCATE(eig)     
!     endif
!   enddo
! ===========================
! Keep eigenvectors gathermatlu
! ===========================
         if (present(eigvectmatlu)) then
           tndim=nspinor*(2*matlu(iatom)%lpawu+1)
           eigvectmatlu(iatom,isppol)%value(:,:)=gathermatlu(iatom)%value(:,:) 
!           write(std_out,*) "eigvect in diag_matlu"
!           do im1=1,tndim
!             write(message,'(12(1x,18(1x,"(",f9.3,",",f9.3,")")))')&
!&             (gathermatlu(iatom)%value(im1,im2),im2=1,tndim)
!             call wrtout(std_out,message,'COLL')
!           end do
         endif


! ==================================================================
       endif
       if(nsppolimp==1.and.matlu(1)%nsppol==2) then
! If necessary rotate levels for this other spin, assuming the same
! rotation matrix: it have to be checked afterwards that the matrix is
! diagonal
! ===================================================================
         ABI_ALLOCATE(temp_mat,(tndim,tndim))
         ABI_ALLOCATE(temp_mat2,(tndim,tndim))
         temp_mat(:,:)=czero
!        input matrix: gathermatlu
!        rotation matrix: eigvectmatlu
!        intermediate matrix: temp_mat
!        result matrix: temp_mat2
         do im1=1,tndim
           do im2=1,tndim
             gathermatlu(iatom)%value(im1,im2)=matlu(iatom)%mat(im1,im2,2,1,1)
           enddo
         enddo
         if(prtopt>=3) then
           write(message,'(a,i4,a,i4)')  "       GATHERMATLU for atom",iatom," inside if nsppolimp==1"
           call wrtout(std_out,message,'COLL')
           do im1=1,tndim
             write(message,'(12(1x,18(1x,"(",f9.3,",",f9.3,")")))')&
&             (gathermatlu(iatom)%value(im1,im2),im2=1,tndim)
             call wrtout(std_out,message,'COLL')
           end do
         endif
         call zgemm('n','n',tndim,tndim,tndim,cone,gathermatlu(iatom)%value,tndim,&
&          eigvectmatlu(iatom,1)%value,tndim,czero,temp_mat                ,tndim)
         call zgemm('c','n',tndim,tndim,tndim,cone,eigvectmatlu(iatom,1)%value,tndim,&
&          temp_mat                   ,tndim,czero,temp_mat2,tndim)
         eigvectmatlu(iatom,2)%value=eigvectmatlu(iatom,1)%value
         imc=0
         print_temp_mat2=.true.
         do ispinor=1,nspinor
           do im1=1,2*matlu(iatom)%lpawu+1
             imc=imc+1
             imc1=0
             do ispinor1=1,nspinor
               do im2=1,2*matlu(iatom)%lpawu+1
                 imc1=imc1+1
                 matlu_diag(iatom)%mat(im1,im2,2,ispinor,ispinor1)=temp_mat2(imc,imc1)
                 if (imc/=imc1.and.(abs(temp_mat2(imc,imc1))>tol5)) then
                   write(message,'(3a,i4)') ch10,'diag_matlu= Matrix for spin number 2 obtained with', &
&                   ' eigenvectors from diagonalization for spin nb 1 is non diagonal for atom:',iatom
                   call wrtout(std_out,message,'COLL')
                   print_temp_mat2=.true.
                 endif
!                 write(std_out,*) temp_mat2(imc,imc1)
!debug             temp_mat2(imc,imc)=eig(imc)
               enddo
             enddo
           enddo
         enddo
         if(print_temp_mat2.and.prtopt>=3) then
           write(message,'(a)')  "       temp_mat2"
           call wrtout(std_out,message,'COLL')
           do im1=1,tndim
             write(message,'(12(1x,18(1x,"(",f9.3,",",f9.3,")")))')&
&             (temp_mat2(im1,im3),im3=1,tndim)
             call wrtout(std_out,message,'COLL')
           end do
           if(iatom==2) call leave_new('COLL')
         endif
         ABI_DEALLOCATE(temp_mat)
         ABI_DEALLOCATE(temp_mat2)
       endif


     endif ! end lpawu/=-1

!debug!!  for check only
!debug     if(matlu(iatom)%lpawu.ne.-1) then
!debug       allocate(temp_mat(tndim,tndim))
!debug       temp_mat(:,:)=czero
!debug!      input matrix: temp_mat2
!debug!      rotation matrix: gathermatlu
!debug!      intermediate matrix: temp_mat
!debug!      result matrix: temp_mat2
!debug       call zgemm('n','c',tndim,tndim,tndim,cone,temp_mat2   ,tndim,&
!debug&        gathermatlu(iatom)%value,tndim,czero,temp_mat                ,tndim)
!debug       call zgemm('n','n',tndim,tndim,tndim,cone,gathermatlu(iatom)%value,tndim,&
!debug&        temp_mat                   ,tndim,czero,temp_mat2,tndim)
!debug         write(std_out,*) "result"
!debug         do im1=1,tndim
!debug           write(message,'(12(1x,18(1x,"(",f9.3,",",f9.3,")")))')&
!debug&           (temp_mat2(im1,im2),im2=1,tndim)
!debug           call wrtout(std_out,message,'COLL')
!debug         end do
!debug       deallocate(temp_mat)
!debug     endif ! lpawu

   enddo  ! iatom
! End loop over atoms
! ===========================
   do iatom=1,natom
     if(matlu(iatom)%lpawu.ne.-1) then
       ABI_DEALLOCATE(gathermatlu(iatom)%value)
     endif
   enddo
   ABI_DATATYPE_DEALLOCATE(gathermatlu)
 enddo ! isppol

 end subroutine diag_matlu
!!***

!!****f* m_matlu/rotate_matlu
!! NAME
!! rotate_matlu
!!
!! FUNCTION
!! Rotate matlu matrix
!!
!! COPYRIGHT
!! Copyright (C) 2005-2012 ABINIT group (BAmadon)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~abinit/COPYING
!! or http://www.gnu.org/copyleft/gpl.txt .
!!
!! INPUTS
!!  matlu(natom)%(nsppol,nspinor,nspinor,ndim,ndim) :: input quantity to rotate
!!  rot_mat(natom) <type(coeff2c_type)> = Rotation matrix (usually from diag_matlu)
!!  natom=number of atoms in cell.
!!  prtopt= option to define level of printing
!!
!! OUTPUT
!!  matlu(natom)%(nsppol,nspinor,nspinor,ndim,ndim) :: Rotated matrix
!!
!! SIDE EFFECTS
!!
!! NOTES
!!
!! PARENTS
!!      hubbard_one
!!
!! CHILDREN
!!      checkdiag_matlu
!!
!! SOURCE
 subroutine rotate_matlu(matlu,rot_mat,natom,prtopt,inverse)
 use defs_basis
 use defs_datatypes
 use defs_wvltypes
 use m_crystal, only : crystal_structure

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
#undef ABI_FUNC
#define ABI_FUNC 'rotate_matlu'
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer, intent(in) :: natom
 integer, intent(in) :: prtopt
 integer, intent(in) :: inverse
!arrays
 type(matlu_type),intent(inout) :: matlu(natom)
 type(coeff2c_type),optional,intent(inout) :: rot_mat(natom,matlu(1)%nsppol)
!Local variables-------------------------------
!scalars
 integer :: iatom,im1,im2,isppol
 integer :: nsppol,nspinor,tndim
!arrays
 type(coeff2c_type),allocatable :: gathermatlu(:)
! type(coeff2c_type),allocatable :: rot_mat_orig(:,:)
 type(coeff2c_type),allocatable :: rot_mat_orig(:)
 complex(dpc),allocatable :: temp_mat(:,:)
!************************************************************************
 if(prtopt==1) then
 endif
 nsppol=matlu(1)%nsppol
 nspinor=matlu(1)%nspinor
! ABI_DATATYPE_ALLOCATE(rot_mat_orig,(natom,matlu(1)%nsppol))

 do isppol=1,nsppol

! ===========================
! Define gathermatlu and rot_mat_orig and allocate
! ===========================
   ABI_DATATYPE_ALLOCATE(rot_mat_orig,(natom))
   ABI_DATATYPE_ALLOCATE(gathermatlu,(natom))
   do iatom=1,natom
     if(matlu(iatom)%lpawu.ne.-1) then
       tndim=nspinor*(2*matlu(iatom)%lpawu+1)
       ABI_ALLOCATE(gathermatlu(iatom)%value,(tndim,tndim))
       gathermatlu(iatom)%value=czero
!       ABI_ALLOCATE(rot_mat_orig(iatom,isppol)%value,(tndim,tndim))
!       rot_mat_orig(iatom,isppol)%value(:,:)=rot_mat(iatom,isppol)%value(:,:)
       ABI_ALLOCATE(rot_mat_orig(iatom)%value,(tndim,tndim))
       rot_mat_orig(iatom)%value(:,:)=rot_mat(iatom,isppol)%value(:,:)
     endif
   enddo
   if(nsppol==1.and.nspinor==2) then
     call gather_matlu(matlu,gathermatlu,natom,option=1,prtopt=1)
   else if((nsppol==2.or.nsppol==1).and.nspinor==1) then
     do iatom=1,natom
       if(matlu(iatom)%lpawu.ne.-1) then
         do im1=1,tndim
           do im2=1,tndim
             gathermatlu(iatom)%value(im1,im2)=matlu(iatom)%mat(im1,im2,isppol,1,1)
           enddo
         enddo
       endif
     enddo
   endif
!debug         write(std_out,*) "gathermatlu in rotate matlu"
!debug         do im1=1,tndim
!debug           write(message,'(12(1x,18(1x,"(",f9.3,",",f9.3,")")))')&
!debug&           (gathermatlu(1)%value(im1,im2),im2=1,tndim)
!debug           call wrtout(std_out,message,'COLL')
!debug         end do

! ===========================
! If necessary, invert rot_mat
! ===========================
   if(inverse==1) then
     do iatom=1,natom
       if(matlu(iatom)%lpawu.ne.-1) then
         tndim=nspinor*(2*matlu(iatom)%lpawu+1)
           do im1=1,tndim
             do im2=1,tndim
!               rot_mat(iatom,isppol)%value(im1,im2)=conjg(rot_mat_orig(iatom,isppol)%value(im2,im1))
               rot_mat(iatom,isppol)%value(im1,im2)=conjg(rot_mat_orig(iatom)%value(im2,im1))
             enddo
           enddo
       endif ! lpawu
     enddo ! iatom
   endif

! ===========================
! Rotate
! ===========================
   do iatom=1,natom
     if(matlu(iatom)%lpawu.ne.-1) then
       tndim=nspinor*(2*matlu(iatom)%lpawu+1)
       ABI_ALLOCATE(temp_mat,(tndim,tndim))
       temp_mat(:,:)=czero
!      input matrix: gathermatlu
!      rotation matrix: rot_mat
!      intermediate matrix: temp_mat
!      result matrix: gathermatlu
       call zgemm('n','c',tndim,tndim,tndim,cone,gathermatlu(iatom)%value   ,tndim,&
&        rot_mat(iatom,isppol)%value,tndim,czero,temp_mat                ,tndim)
       call zgemm('n','n',tndim,tndim,tndim,cone,rot_mat(iatom,isppol)%value,tndim,&
&        temp_mat                   ,tndim,czero,gathermatlu(iatom)%value,tndim)
       ABI_DEALLOCATE(temp_mat)
     endif ! lpawu
   enddo ! iatom
!debug     do iatom=1,natom
!debug       if(matlu(iatom)%lpawu.ne.-1) then
!debug         write(std_out,*) "eigvect in rotate_matlu 2"
!debug         do im1=1,tndim
!debug           write(message,'(12(1x,18(1x,"(",f9.3,",",f9.3,")")))')&
!debug&           (gathermatlu(iatom)%value(im1,im2),im2=1,tndim)
!debug           call wrtout(std_out,message,'COLL')
!debug         end do
!debug       endif
!debug     enddo
!debug                 call leave_new('COLL')

! ===========================
! Choose inverse rotation: reconstruct correct rot_mat from rot_mat_orig
! ===========================
   if(inverse==1) then
     do iatom=1,natom
       if(matlu(iatom)%lpawu.ne.-1) then
         tndim=nspinor*(2*matlu(iatom)%lpawu+1)
           do im1=1,tndim
             do im2=1,tndim
!               rot_mat(iatom,isppol)%value(im1,im2)=rot_mat_orig(iatom,isppol)%value(im1,im2)
               rot_mat(iatom,isppol)%value(im1,im2)=rot_mat_orig(iatom)%value(im1,im2)
             enddo
           enddo
       endif ! lpawu
     enddo ! iatom
   endif

! ===========================
! Put data into matlu(iatom)
! ===========================
   if(nsppol==1.and.nspinor==2) then
     call gather_matlu(matlu,gathermatlu,natom,option=-1,prtopt=1)
   else if((nsppol==2.or.nsppol==1).and.nspinor==1) then
     do iatom=1,natom
       if(matlu(iatom)%lpawu.ne.-1) then
         do im1=1,tndim
           do im2=1,tndim
             matlu(iatom)%mat(im1,im2,isppol,1,1)= gathermatlu(iatom)%value(im1,im2)
           enddo
         enddo
       endif
     enddo
   endif ! test nsppol/nspinor
! ===========================
! Deallocations
! ===========================
   do iatom=1,natom
     if(matlu(iatom)%lpawu.ne.-1) then
       ABI_DEALLOCATE(gathermatlu(iatom)%value)
!       ABI_DEALLOCATE(rot_mat_orig(iatom,isppol)%value)
       ABI_DEALLOCATE(rot_mat_orig(iatom)%value)
     endif
   enddo
   ABI_DATATYPE_DEALLOCATE(gathermatlu)
   ABI_DATATYPE_DEALLOCATE(rot_mat_orig)
 enddo ! isppol

 end subroutine rotate_matlu
!!***

!!****f* m_matlu/shift_matlu
!! NAME
!! shift_matlu
!!
!! FUNCTION
!! shift matlu matrix
!!
!! COPYRIGHT
!! Copyright (C) 2005-2011 ABINIT group (BAmadon)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~abinit/COPYING
!! or http://www.gnu.org/copyleft/gpl.txt .
!!
!! INPUTS
!!  matlu(natom)%(nsppol,nspinor,nspinor,ndim,ndim) :: input quantity to rotate
!!  natom=number of atoms in cell.
!!  shift= shift of the diagonal part.
!!
!! OUTPUT
!!  matlu(natom)%(nsppol,nspinor,nspinor,ndim,ndim) :: shifted matrix
!!
!! SIDE EFFECTS
!!
!! NOTES
!!
!! PARENTS
!!      m_green,m_self
!!
!! CHILDREN
!!      checkdiag_matlu
!!
!! SOURCE
 subroutine shift_matlu(matlu,natom,shift,signe)
 use defs_basis
 use defs_datatypes
 use defs_wvltypes
 use m_crystal, only : crystal_structure

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
#undef ABI_FUNC
#define ABI_FUNC 'shift_matlu'
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer, intent(in) :: natom
!arrays
 type(matlu_type),intent(inout) :: matlu(natom)
 real(dp),intent(in) :: shift(natom)
 integer, optional,intent(in) :: signe
!Local variables-------------------------------
!scalars
 integer :: iatom,im,ispinor,isppol
 integer :: lpawu,signe_used
! character(len=500) :: message
!arrays
!************************************************************************
 signe_used=1
 if(present(signe)) then
   if(signe==-1) signe_used=-1
 endif
 do iatom=1,natom
   lpawu=matlu(iatom)%lpawu
   if(lpawu.ne.-1) then
     do im=1,2*lpawu+1
       do isppol=1,matlu(1)%nsppol
         do ispinor=1,matlu(1)%nspinor
           matlu(iatom)%mat(im,im,isppol,ispinor,ispinor)=&
&           matlu(iatom)%mat(im,im,isppol,ispinor,ispinor)-signe_used*shift(iatom)
         enddo
       enddo
     enddo
   endif
 enddo

 end subroutine shift_matlu
!!***

!!****f* m_matlu/checkdiag_matlu
!! NAME
!! checkdiag_matlu
!!
!! FUNCTION
!! Check that matlu is diagonal in the orbital index with given precision
!!
!! COPYRIGHT
!! Copyright (C) 2005-2011 ABINIT group (BAmadon)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~abinit/COPYING
!! or http://www.gnu.org/copyleft/gpl.txt .
!!
!! INPUTS
!!  matlu(natom)%(nsppol,nspinor,nspinor,ndim,ndim) :: input quantity to rotate
!!  natom=number of atoms in cell.
!!  tol : precision
!!
!! SIDE EFFECTS
!!
!! NOTES
!!
!! PARENTS
!!      m_matlu
!!
!! CHILDREN
!!      checkdiag_matlu
!!
!! SOURCE
 subroutine checkdiag_matlu(matlu,natom,tol)
 use defs_basis
 use defs_datatypes
 use defs_wvltypes
 use m_crystal, only : crystal_structure

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
#undef ABI_FUNC
#define ABI_FUNC 'checkdiag_matlu'
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 real(dp),intent(in) :: tol
 integer, intent(in) :: natom
!arrays
 type(matlu_type),intent(inout) :: matlu(natom)
!Local variables-------------------------------
!scalars
 integer :: iatom,im,im1,ispinor,ispinor1,isppol
 integer :: lpawu
 character(len=500) :: message
!arrays
!************************************************************************
 do iatom=1,natom
   lpawu=matlu(iatom)%lpawu
   if(lpawu.ne.-1) then
     do im=1,2*lpawu+1
       do im1=1,2*lpawu+1
         do isppol=1,matlu(1)%nsppol
           do ispinor=1,matlu(1)%nspinor
!              if(im/=im1) write(std_out,*) "im,im1",im,im1,matlu(iatom)%mat(im,im1,isppol,ispinor,ispinor)
             if(im/=im1.and.(abs(matlu(iatom)%mat(im,im1,isppol,ispinor,ispinor))>tol))  then
               write(message,'(a,2e16.5)')" checkdiag_matlu: BUG ",matlu(iatom)%mat(im,im1,isppol,ispinor,ispinor),tol
               MSG_BUG(message)
             endif
             do ispinor1=1,matlu(1)%nspinor
               if(ispinor/=ispinor1.and.(abs(matlu(iatom)%mat(im,im1,isppol,ispinor,ispinor1))>tol))  then
                 write(message,'(a,2e16.5)')" checkdiag_matlu : BUG ",matlu(iatom)%mat(im,im1,isppol,ispinor,ispinor1),tol
                 MSG_BUG(message)
               endif
             enddo
           enddo ! ispinor
         enddo ! isppol
       enddo ! im1
     enddo ! im
   endif ! lpawu
 enddo ! iatom

 end subroutine checkdiag_matlu
!!***

!!****f* m_matlu/prod_matlu
!! NAME
!! prod_matlu
!!
!! FUNCTION
!! Do the matrix product of two matlus
!!
!! COPYRIGHT
!! Copyright (C) 2005-2011 ABINIT group (BAmadon)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~abinit/COPYING
!! or http://www.gnu.org/copyleft/gpl.txt .
!!
!! INPUTS
!!  matlu1(natom)%(nsppol,nspinor,nspinor,ndim,ndim) :: input quantity 
!!  matlu2(natom)%(nsppol,nspinor,nspinor,ndim,ndim) :: input quantity
!!
!! OUTPUT
!!  matlu3(natom)%(nsppol,nspinor,nspinor,ndim,ndim) :: output quantity
!!
!! SIDE EFFECTS
!!
!! NOTES
!!
!! PARENTS
!!
!! CHILDREN
!!      checkdiag_matlu
!!
!! SOURCE
 subroutine prod_matlu(matlu1,matlu2,matlu3,natom)
 use defs_basis
 use defs_datatypes
 use defs_wvltypes
 use m_crystal, only : crystal_structure

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
#undef ABI_FUNC
#define ABI_FUNC 'prod_matlu'
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer, intent(in) :: natom
!arrays
 type(matlu_type), intent(in) :: matlu1(natom),matlu2(natom)
 type(matlu_type), intent(out) :: matlu3(natom)
!Local variables-------------------------------
!scalars
 integer :: iatom,im1,im2,im3,ispinor1,ispinor2,ispinor3,isppol
 integer :: lpawu
!arrays
!************************************************************************
 call zero_matlu(matlu3,natom)
 do iatom=1,natom
   lpawu=matlu1(iatom)%lpawu
   if(lpawu.ne.-1) then
     do isppol=1,matlu1(1)%nsppol
       do ispinor1=1,matlu1(1)%nspinor
         do ispinor2=1,matlu1(1)%nspinor
           do ispinor3=1,matlu1(1)%nspinor
             do im1=1,2*lpawu+1
               do im2=1,2*lpawu+1
                 do im3=1,2*lpawu+1
                   matlu3(iatom)%mat(im1,im2,isppol,ispinor1,ispinor2)= &
&                    matlu3(iatom)%mat(im1,im2,isppol,ispinor1,ispinor2)+ &
&                    matlu1(iatom)%mat(im1,im3,isppol,ispinor1,ispinor3)*&
&                    matlu2(iatom)%mat(im3,im2,isppol,ispinor3,ispinor2)
                 enddo ! im3
               enddo ! im2
             enddo ! im1
           enddo ! ispinor3
         enddo ! ispinor2
       enddo ! ispinor1
     enddo ! isppol
   endif ! lpawu
 enddo ! iatom

 end subroutine prod_matlu
!!***

!!****f* m_matlu/conjg_matlu
!! NAME
!! conjg_matlu
!!
!! FUNCTION
!! conjugate of input matlu
!!
!! COPYRIGHT
!! Copyright (C) 2005-2011 ABINIT group (BAmadon)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~abinit/COPYING
!! or http://www.gnu.org/copyleft/gpl.txt .
!!
!! INPUTS
!!  matlu1(natom)%(nsppol,nspinor,nspinor,ndim,ndim) :: input quantity 
!!
!! SIDE EFFECTS
!!
!! NOTES
!!
!! PARENTS
!!
!! CHILDREN
!!      checkdiag_matlu
!!
!! SOURCE
 subroutine conjg_matlu(matlu1,natom)
 use defs_basis
 use defs_datatypes
 use defs_wvltypes

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
#undef ABI_FUNC
#define ABI_FUNC 'conjg_matlu'
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer, intent(in) :: natom
!arrays
 type(matlu_type), intent(inout) :: matlu1(natom)
!Local variables-------------------------------
!scalars
 integer :: iatom,im1,im2,ispinor2,ispinor1,isppol
 integer :: lpawu
!arrays
!************************************************************************
 do iatom=1,natom
   lpawu=matlu1(iatom)%lpawu
   if(lpawu.ne.-1) then
     do isppol=1,matlu1(1)%nsppol
       do ispinor1=1,matlu1(1)%nspinor
         do ispinor2=1,matlu1(1)%nspinor
           do im1=1,2*lpawu+1
             do im2=1,2*lpawu+1
               matlu1(iatom)%mat(im1,im2,isppol,ispinor1,ispinor2)= &
&               conjg(matlu1(iatom)%mat(im1,im2,isppol,ispinor1,ispinor2))
             enddo ! im2
           enddo ! im1
         enddo ! ispinor2
       enddo ! ispinor1
     enddo ! isppol
   endif ! lpawu
 enddo ! iatom

 end subroutine conjg_matlu
!!***

!!****f* m_matlu/ln_matlu
!! NAME
!! ln_matlu
!!
!! FUNCTION
!! Compute the logarithm of matlu (only if diagonal for the moment)
!!
!! COPYRIGHT
!! Copyright (C) 2005-2011 ABINIT group (BAmadon)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~abinit/COPYING
!! or http://www.gnu.org/copyleft/gpl.txt .
!!
!! INPUTS
!!  matlu1(natom)%(nsppol,nspinor,nspinor,ndim,ndim) :: input quantity 

!! SIDE EFFECTS
!!
!! NOTES
!!
!! PARENTS
!!
!! CHILDREN
!!      checkdiag_matlu
!!
!! SOURCE
 subroutine ln_matlu(matlu1,natom)
 use defs_basis
 use defs_datatypes
 use defs_wvltypes

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
#undef ABI_FUNC
#define ABI_FUNC 'ln_matlu'
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer, intent(in) :: natom
!arrays
 type(matlu_type), intent(inout) :: matlu1(natom)
!Local variables-------------------------------
!scalars
 integer :: iatom,im,ispinor,isppol
 integer :: lpawu
 character(len=500) :: message
!arrays
!************************************************************************
 call checkdiag_matlu(matlu1,natom,tol8)
 do iatom=1,natom
   lpawu=matlu1(iatom)%lpawu
   if(lpawu.ne.-1) then
     do isppol=1,matlu1(1)%nsppol
       do ispinor=1,matlu1(1)%nspinor
         do im=1,2*lpawu+1
           if( real(matlu1(iatom)%mat(im,im,isppol,ispinor,ispinor))<zero) then
             write(message,'(2a,2es13.5,a)') ch10," ln_matlu: PROBLEM " &
&             , matlu1(iatom)%mat(im,im,isppol,ispinor,ispinor)
             MSG_ERROR(message)
           endif
           matlu1(iatom)%mat(im,im,isppol,ispinor,ispinor)= &
&           log(matlu1(iatom)%mat(im,im,isppol,ispinor,ispinor))
         enddo ! im
       enddo ! ispinor
     enddo ! isppol
   endif ! lpawu
 enddo ! iatom

 end subroutine ln_matlu

END MODULE m_matlu
!!***
