!{\src2tex{textfont=tt}}
!!****m* ABINIT/m_wfk
!! NAME
!!  m_wfk
!!
!! FUNCTION
!!
!! COPYRIGHT
!! Copyright (C) 2009-2012 ABINIT group (MG)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~abinit/COPYING
!! or http://www.gnu.org/copyleft/gpl.txt .
!! For the initials of contributors, see ~abinit/doc/developers/contributors.txt.
!!
!! PARENTS
!!
!! CHILDREN
!!
!! SOURCE

#if defined HAVE_CONFIG_H
#include "config.h"
#endif

#include "abi_common.h"

MODULE m_wfk

 use defs_basis
 use defs_datatypes
 use m_profiling
 use m_errors
#if defined HAVE_MPI2 && defined HAVE_MPI_IO
 use mpi
#endif
 use m_xmpi
 use m_mpiotk
#ifdef HAVE_TRIO_ETSF_IO
 use etsf_io
#endif

 use defs_abitypes,  only : hdr_type
 use m_io_tools,     only : get_unit, mvrecord
 use m_header,       only : hdr_skip, hdr_io, hdr_comm, hdr_clean, hdr_mpio_skip

 implicit none

#if defined HAVE_MPI1 && defined HAVE_MPI_IO
 include 'mpif.h'
#endif

 private

!public procedures.
 public :: wfk_open_read
 !public :: wfk_open_write
 public :: wfk_close
 public :: wfk_read_band_block
 public :: wfk_read_bmask
 !public :: wfk_write_band_block
 !public :: wfk_write_bmask

 ! Profiling tools
 public :: wfk_prof
!!***

 integer,private,parameter :: WFK_NOMODE    = 0
 integer,private,parameter :: WFK_READMODE  = 1
 integer,private,parameter :: WFK_WRITEMODE = 2

 integer,private,parameter :: REC_HDR=0
 integer,private,parameter :: REC_NPW=1
 integer,private,parameter :: REC_KG =2
 integer,private,parameter :: REC_EIG=3
 integer,private,parameter :: REC_CG =4
 integer,private,parameter :: REC_NUM=REC_CG

 integer(XMPI_OFFSET_KIND),private,parameter :: WFK_CG_MAXBSIZE = 2000 * (1024.0_dp**2)
   ! Maximum size (in bytes) of the block of wavefunctions that are (read|written) 
   ! in a single MPI-IO call. (Some MPI-IO implementation crashes if we try to 
   ! (read|write) a big chunk of data with a single call.

!----------------------------------------------------------------------

!!****t* m_wfk/wfk_t
!! NAME
!!
!! FUNCTION
!!
!! SOURCE

 type,public :: wfk_t

  integer :: fh
   ! file handler:
   !  unit number if IO_MODE_FORTRAN
   !  MPI file handler if IO_MODE_MPI
   !  Netcdf file handler if IO_MODE_NETCDF 

  integer :: accesswff
   ! Method to access the wavefunction file:
   !   IO_MODE_FORTRAN for usual Fortran IO routines
   !   IO_MODE_MPI if MPI/IO routines.
   !   IO_MODE_ETSF, NetCDF format read via etsf-io.

  integer :: mband
  ! Max number of bands stored on file (MAX(Hdr%nband))

  integer :: formeig=0 
   ! formeig=format of the eigenvalues
   !    0 => vector of eigenvalues
   !    1 => hermitian matrix of eigenvalues
   ! TODO: this should be reported somewhere in the WFK file, at present is passed to wfk_open

  integer :: fform
   ! File type format of the header

  integer :: rw_mode = WFK_NOMODE
   ! (Read|Write) mode 

  character(len=fnlen) :: fname = ABI_NOFILE
   ! File name 

  integer :: master
   ! master node of the IO procedure 

  integer :: my_rank
   ! index of my processor in the MPI communicator comm

  integer :: nproc
   ! number of processors in comm

  integer :: comm
   ! MPI communicator 

  type(hdr_type) :: Hdr
   ! Abinit header.

  integer :: f90_fptr(3) = (/0,0,0/)
  ! The position of the file pointer used for sequential access with Fortran-IO.
  !  f90_fprt(1) = Index of the k-point associated to the block.
  !  f90_fprt(2) = the spin associated to the block.
  !  f90_fprt(3) = Record Type (see REC_* variables).
  !  (/0,0,0/) corresponds to the beginning of the file.

  integer,pointer :: recn_ks(:,:,:)  => null()
   ! recn_ks(k,s,1) : record number of  (npw, nspinor, nband_disk)
   ! recn_ks(k,s,2) : record number of the (k+G) vectors.
   ! recn_ks(k,s,3) : record number of the eigenvalues.
   ! recn_ks(k,s,4) : record number of the first wavefunction in  the wf coefficients block.

  integer(XMPI_OFFSET_KIND),pointer :: offset_ks(:,:,:)  => null()
   ! offset_ks(k,s,1) : offset of the record: npw, nspinor, nband_disk.
   ! offset_ks(k,s,2) : offset of the Second record: (k+G) vectors.
   ! offset_ks(k,s,3) : offset of the third record eigenvalues.
   ! offset_ks(k,s,4) : offset of the fourth record (wavefunction coefficients).

  !TODO
  !this should be the output of a hdr method!
  integer(XMPI_OFFSET_KIND) :: hdr_offset
   ! Header offset

  !integer(XMPI_OFFSET_KIND) :: offset
   ! offset for MPI-IO access

  !integer(XMPI_OFFSET_KIND) :: offset_rec
   ! offset position of start record

  !integer(XMPI_OFFSET_KIND)  :: offwff, off_recs
   ! offwff   offset position of unformatted wavefunction disk file
   ! off_recs offset position of start record
   ! (used in parallel MPI-IO)

   integer(XMPI_OFFSET_KIND) :: cg_maxbsize

 end type wfk_t
!!***

!----------------------------------------------------------------------

!!****t* m_wfk/wfkview_t
!! NAME
!!
!! FUNCTION
!!
!! SOURCE

 !type,public :: wfkview_t
 !  logical,pointer :: b_mask(:)        => null()
 !  integer,pointer :: myg_to_gfile(:)  => null()
 !end type wfkview_t

 !public :: vfkview_init
 !public :: vfkview_free

CONTAINS
!!***

!----------------------------------------------------------------------

!!****f* m_wfk/wfk_open_read
!! NAME
!!  wfk_open_read
!!
!! FUNCTION
!!   Open the WFK file for reading.
!!
!! INPUTS
!!  accesswff = File format option.
!!  comm = MPI communicator (used for MPI-IO)
!!  formeig = 0 for GS wavefunctions, 1 for RF wavefunctions.
!!  fh = Fortran unit numer for  Only used if accesswff == IO_MODE_FORTRAN
!!  fname = Name of the file
!!
!! OUTPUT
!!  Wfk<type(wfk_t)> = WFK handler initialized and set in read mode
!!
!! PARENTS
!!      m_wfk,m_wfs
!!
!! CHILDREN
!!      cwtime,hdr_clean,hdr_comm,hdr_io,wfk_close,wfk_open_read
!!      wfk_read_band_block,wfk_read_bmask,wrtout
!!
!! SOURCE

subroutine wfk_open_read(Wfk,fname,formeig,accesswff,fh,comm)


!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
#undef ABI_FUNC
#define ABI_FUNC 'wfk_open_read'
 use interfaces_14_hidewrite
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: accesswff,comm,formeig,fh
 character(len=fnlen),intent(in) :: fname
 type(wfk_t),intent(out) :: Wfk

!Local variables-------------------------------
!scalars
 integer,parameter :: rdwr1=1
 integer :: ierr,mpi_err,funt
 character(len=500) :: msg
#ifdef HAVE_MPI_IO
 integer :: fform
#endif
#ifdef HAVE_TRIO_ETSF_IO
 logical :: lstat
 type(etsf_io_low_error) :: Error
#endif
!arrays

!************************************************************************

!Initialize the mandatory data of the Wfk datastructure
 Wfk%rw_mode     = WFK_READMODE
 Wfk%cg_maxbsize = WFK_CG_MAXBSIZE

 Wfk%fname     = fname
 Wfk%formeig   = formeig
 Wfk%accesswff = accesswff
 Wfk%fh        = fh
 Wfk%comm      = comm

 Wfk%master    = 0
 Wfk%my_rank   = xcomm_rank(comm)
 Wfk%nproc     = xcomm_size(comm)
 !
 ! Master node reads fform and the Header with Fortran IO
 if (Wfk%my_rank == Wfk%master) then

   funt = get_unit()
   open(unit=funt, file=fname, form="unformatted", status="old", iostat=ierr)
   ABI_CHECK(ierr==0,'Opening '//TRIM(fname)//' as old-unformatted')

   call hdr_io(Wfk%fform,Wfk%Hdr,rdwr1,funt)
   close(funt)
   !call hdr_io(Wfk%fform,Wfk%Hdr,4,std_out)
 end if
 !
 ! Broadcast fform and the header.
 if (Wfk%nproc > 1) then
   call xcast_mpi(Wfk%fform,Wfk%master,Wfk%comm,ierr)
   call hdr_comm(Wfk%Hdr,Wfk%master,Wfk%my_rank,Wfk%comm)
 end if
 !
 ! Useful dimensions
 Wfk%mband = MAXVAL(Wfk%Hdr%nband)

 ! Precompute offsets for MPI-IO access and number of records for Fortran IO.
 !call wfk_compute_offsets(Wfk)

 ierr=0
 SELECT CASE (accesswff)

 CASE (IO_MODE_FORTRAN) 
   ! All processors see a local Fortran binary file.
   ! Each node opens the file, skip the header and set f90_fptr.
   open(unit=Wfk%fh,file=Wfk%fname,form='unformatted',status="old",iostat=ierr)
   ABI_CHECK(ierr==0,'Opening '//TRIM(Wfk%fname)//' as old-unformatted')

   call wfk_compute_offsets(Wfk)

   call hdr_skip(Wfk%fh,ierr)
   Wfk%f90_fptr = (/1,1,REC_NPW/)

 CASE (IO_MODE_MPI) 
   ! In the parallel case, only the master open the file
#ifdef HAVE_MPI_IO
   call MPI_FILE_OPEN(Wfk%comm, Wfk%fname, MPI_MODE_RDONLY, MPI_INFO_NULL, Wfk%fh, mpi_err)
   ABI_CHECK_MPI(mpi_err,"MPI_FILE_OPEN")

   call hdr_mpio_skip(Wfk%fh,fform,Wfk%hdr_offset)
   !write(std_out,*)"after skip hdr_offset",Wfk%hdr_offset

   if (Wfk%hdr_offset > 0) then
     call wfk_compute_offsets(Wfk)
   else
     MSG_ERROR("hdr_offset <=0")
   end if
#else
   MSG_ERROR("MPI-IO not enabled")
#endif

 CASE (IO_MODE_ETSF)
   write(msg,'(3a,i0)' )' Opening file:',TRIM(Wfk%fname)," on unit ",Wfk%fh
   call wrtout(std_out, msg, 'COLL')

#ifdef HAVE_TRIO_ETSF_IO
   call etsf_io_low_open_modify(Wfk%fh, Wfk%fname, lstat, Error_data = Error)
   ETSF_CHECK_ERROR(lstat, Error)
#else
   MSG_ERROR("ETSF-IO not enabled")
#endif

 CASE DEFAULT
   write(msg, '(7a,i0,3a)' )&
&   '  For the time being the input variable accesswff is restricted ',ch10,&
&   '  to 0 (all cases), 1 (in case MPI is enabled),',ch10,&
&   '  or 3 (only sequential, and if the NetCDF and ETSF_IO libraries have been enabled).',ch10,&
&   '  Its value is accesswff=',accesswff,'.',ch10,&
&   '  Action : change accesswff or use ABINIT in parallel or enable NetCDF and/or ETSF_IO.'
   MSG_ERROR(msg)
 END SELECT

end subroutine wfk_open_read
!!***

!----------------------------------------------------------------------

!!****f* m_wfk/wfk_close
!! NAME
!!  wfk_close
!!
!! FUNCTION
!!  Close the wavefunction file handler and release the memory allocated
!!
!! PARENTS
!!      m_wfk,m_wfs
!!
!! CHILDREN
!!      cwtime,hdr_clean,hdr_comm,hdr_io,wfk_close,wfk_open_read
!!      wfk_read_band_block,wfk_read_bmask,wrtout
!!
!! SOURCE

subroutine wfk_close(Wfk)


!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
#undef ABI_FUNC
#define ABI_FUNC 'wfk_close'
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 type(wfk_t),intent(inout) :: Wfk
!arrays
!Local variables-------------------------------
!scalars
 integer :: ierr
 character(len=500) :: msg
#ifdef HAVE_TRIO_ETSF_IO
 logical :: lstat
 type(etsf_io_low_error) :: Error
#endif

! *************************************************************************

 Wfk%fname   = ABI_NOFILE
 Wfk%rw_mode = WFK_NOMODE

 SELECT CASE (Wfk%accesswff)

 CASE (IO_MODE_FORTRAN) 
   close(unit=Wfk%fh)

 CASE (IO_MODE_MPI)
#ifdef HAVE_MPI_IO
   call MPI_FILE_CLOSE(Wfk%fh,ierr)
   ABI_CHECK_MPI(ierr,"FILE_CLOSE!")
#endif

 CASE (IO_MODE_ETSF)
#ifdef HAVE_TRIO_ETSF_IO
   call etsf_io_low_close(Wfk%fh, lstat, Error_data = Error)
   ETSF_CHECK_ERROR(lstat, Error)
#endif

 CASE DEFAULT
   write(msg, '(7a,i0,3a)' )&
&   '  For the time being the input variable accesswff is restricted ',ch10,&
&   '  to 0 (all cases), 1 (in case MPI is enabled),',ch10,&
&   '  or 3 (only sequential, and if the NetCDF and ETSF_IO libraries have been enabled).',ch10,&
&   '  Its value is accesswff=',Wfk%accesswff,'.',ch10,&
&   '  Action : change accesswff or use ABINIT in parallel or enable NetCDF and/or ETSF_IO.'
   MSG_ERROR(msg)
 END SELECT
 !
 ! Free memory.
 call hdr_clean(Wfk%Hdr)

 if (associated(Wfk%recn_ks)) then
   ABI_FREE(Wfk%recn_ks)
 end if

 if (associated(Wfk%offset_ks)) then
   ABI_FREE(Wfk%offset_ks)
 end if

end subroutine wfk_close
!!***

!----------------------------------------------------------------------

!!****f* m_wfk/wfk_read_band_block
!! NAME
!!  wfk_read_band_block
!!
!! FUNCTION
!!  Read a block of contigous bands.
!!
!! INPUTS
!!  Wfk<type(wfk_t)>=
!!  band_block(2)=Initial and final band index.
!!  ik_ibz=Index of the k-point in the IBZ.
!!  spin=Spin index
!!
!! OUTPUTS
!!  [kg_k=(:,:)] = G-vectors
!!  [eig_k(:,:)] = Eigenvectors
!!  [cg_k(:,:)]  = Fourier coefficients
!!
!! PARENTS
!!      m_wfk,m_wfs
!!
!! CHILDREN
!!      cwtime,hdr_clean,hdr_comm,hdr_io,wfk_close,wfk_open_read
!!      wfk_read_band_block,wfk_read_bmask,wrtout
!!
!! SOURCE

subroutine wfk_read_band_block(Wfk,band_block,ik_ibz,spin,kg_k,eig_k,cg_k) 


!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
#undef ABI_FUNC
#define ABI_FUNC 'wfk_read_band_block'
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: ik_ibz,spin !,mband,rdcg,rdeig,npw_k,nband_k
 type(wfk_t),intent(inout) :: Wfk
!arrays
 integer,intent(in) :: band_block(2)
 integer,intent(out),optional,target :: kg_k(:,:)  !(3,rdcg*npw_k)
 real(dp),intent(out),optional,target :: cg_k(:,:) ! cg_k(2,rdcg*cgsize2) !(2,npw_k*nspinor*nband)
 real(dp),intent(out),optional,target :: eig_k(:)  !(rdeig*(2*mband)**formeig*mband) !,occ_k(mband)

!Local variables-------------------------------
!scalars
 integer :: ierr,npw_disk,nspinor_disk,nband_disk,band 
 integer :: ipw,my_bcount,npwso,npw_tot,nb_block
 integer :: npw_read,nspinor_read,nband_read
 character(len=500) :: msg
#ifdef HAVE_MPI_IO
 integer :: mpi_err,bufsz
 integer(XMPI_OFFSET_KIND) :: my_offset !,my_offpad !,fmarker
 integer :: sizes(2),subsizes(2),starts(2),statux(MPI_STATUS_SIZE)
#endif
#ifdef HAVE_TRIO_ETSF_IO
 logical :: lstat
 !type(etsf_main) :: main_folder
 type(etsf_basisdata) :: wave_folder
 type(etsf_electrons) :: electrons_folder
 type(etsf_io_low_error) :: Error
#endif

!************************************************************************

 ABI_CHECK(Wfk%rw_mode==WFK_READMODE, "Wfk must be in READMODE")

 if (Wfk%Hdr%headform<40) then 
   write(msg,'(a,i0)')"Too old headform : ",Wfk%Hdr%headform
   MSG_ERROR(msg)
 end if
 !
 ! Look before you leap 
 npw_disk     = Wfk%Hdr%npwarr(ik_ibz)
 nspinor_disk = Wfk%Hdr%nspinor 
 nband_disk   = Wfk%Hdr%nband(ik_ibz + (spin-1)*Wfk%Hdr%nkpt)
 nb_block     = (band_block(2) - band_block(1) + 1)
 npw_tot      = npw_disk * nspinor_disk * nb_block

 if (PRESENT(kg_k)) then
   ABI_CHECK(SIZE(kg_k,DIM=2) >= npw_disk,"kg_k too small")
 end if

 if (PRESENT(eig_k)) then
   ABI_CHECK(SIZE(eig_k) >= nband_disk, "eig_k too small")
 end if

 if (PRESENT(cg_k)) then
   ABI_CHECK(SIZE(cg_k, DIM=2) >= npw_tot, "Too small cg_k")
 end if

 SELECT CASE (Wfk%accesswff)

 CASE (IO_MODE_FORTRAN) 

   ! Rewind the file to have the correct (k,s) block (if needed)
   call wfk_seek(Wfk,ik_ibz,spin)
   !
   ! Read the first record: npw, nspinor, nband_disk
   !read(Wfk%fh, iostat=ierr) npw_disk, nspinor_disk, nband_disk
   read(Wfk%fh, iostat=ierr) npw_read, nspinor_read, nband_read

   if (ierr/=0)then
     write(msg,'(4a)') &
      " Error while reading the (npw,nspinor,nband) record of the wfk file", TRIM(Wfk%fname),ch10,&
      ' Your file is likely not correct.'
     MSG_ERROR(msg)
   end if

   if ( ANY( (/npw_read, nspinor_read, nband_read/) /= (/npw_disk, nspinor_disk, nband_disk/) )) then
     MSG_ERROR("Mismatch between WFK and HDR")
   end if
   !
   ! The second record: (k+G) vectors
   if (PRESENT(kg_k)) then
     read(Wfk%fh, iostat=ierr) kg_k(1:3,1:npw_disk)
   else
     read(Wfk%fh, iostat=ierr) ! kg_k(1:3,1:npw_disk)
   end if

   if (ierr/=0)then
     write(msg,'(4a)')&
&     " Error while reading the k+g record of the WFK file",TRIM(Wfk%fname),ch10,&
&     " Your file is likely not correct."
     MSG_ERROR(msg)
   end if
   !
   ! The third record: eigenvalues
   if (Wfk%formeig==0) then

     if (PRESENT(eig_k)) then
       read(Wfk%fh, iostat=ierr) eig_k(1:nband_disk)
     else
       read(Wfk%fh, iostat=ierr) ! eig_k(1:nband_disk)
     end if
    
     if (ierr/=0)then
       write(msg,'(3a,i4,2a,i4,5a)') &
&       " Error while reading the eigenvalue record of the WFK file",TRIM(Wfk%fname),ch10,&
&       " Your file is likely not correct."
       MSG_ERROR(msg)
     end if

   else 
     MSG_ERROR("formeig != 0 not yet coded")
   end if
   !
   ! The wave-functions.
   if (PRESENT(cg_k)) then
     !
     npwso = npw_disk*nspinor_disk
     my_bcount = 0
     do band=1,nband_disk
       if (band >= band_block(1) .and. band <= band_block(2)) then
         ipw = my_bcount * npwso
         my_bcount = my_bcount + 1
         read(Wfk%fh, iostat=ierr) cg_k(1:2,ipw+1:ipw+npwso)
       else
         read(Wfk%fh, iostat=ierr) ! cg_k(1:2,ipw+1:ipw+npwso)
       end if
       if (ierr/=0) EXIT
     end do
     !
   else
     !
     do band=1,nband_disk
       read(Wfk%fh, iostat=ierr) ! cg_k(1:2,ipw+1:ipw+npwso)
       if (ierr/=0) EXIT
     end do
     !
   end if

   if (ierr/=0)then
     write(msg,'(3a,i4,2a,i4,5a)') &
      " Error while reading the cg record of the WFK file",TRIM(Wfk%fname),ch10,&
      " Your file is likely not correct."
     MSG_ERROR(msg)
   end if
   ! 
   ! Reached the end of the (k,s) block. Update f90_fptr
   if (ik_ibz < Wfk%Hdr%nkpt) then
     Wfk%f90_fptr = (/ik_ibz+1,spin,REC_NPW/)
   else
     Wfk%f90_fptr = (/1,spin+1,REC_NPW/)
   end if

 CASE (IO_MODE_MPI) 

#ifdef HAVE_MPI_IO
   if (PRESENT(kg_k)) then
     my_offset = Wfk%offset_ks(ik_ibz,spin,REC_KG) + xmpio_bsize_frm

     call MPI_FILE_READ_AT_ALL(Wfk%fh,my_offset,kg_k,3*npw_disk,MPI_INTEGER,statux,mpi_err)
     ABI_CHECK_MPI(mpi_err,"reading REC_KG")
   end if

   if (PRESENT(eig_k)) then
     my_offset = Wfk%offset_ks(ik_ibz,spin,REC_EIG) + xmpio_bsize_frm

     call MPI_FILE_READ_AT_ALL(Wfk%fh,my_offset,eig_k,nband_disk,MPI_DOUBLE_PRECISION,statux,mpi_err)
     ABI_CHECK_MPI(mpi_err,"reading REC_EIG")
   end if

   if (PRESENT(cg_k)) then
     my_offset = Wfk%offset_ks(ik_ibz,spin,REC_CG)
     sizes    = (/npw_disk*nspinor_disk, nband_disk/)
     subsizes = (/npw_disk*nspinor_disk, band_block(2)-band_block(1)+1/)
     bufsz = 2 * SIZE(cg_k, DIM=2)
     starts = (/1, band_block(1)/)

     call mpiotk_read_fsuba_all_dp2D(Wfk%fh,my_offset,sizes,subsizes,starts,&
&      bufsz,cg_k,Wfk%cg_maxbsize,Wfk%comm,ierr) 
     ABI_CHECK(ierr==0,"Fortran record too big")
   end if
#endif

 CASE (IO_MODE_ETSF)

   MSG_ERROR("Not Coded")
#ifdef HAVE_TRIO_ETSF_IO
   ABI_CHECK(Wfk%formeig==0,"formeig !=0 not compatible with ETSF-IO")

   ! Read the first record: npw, nspinor, nband_disk
   call etsf_io_low_read_dim(Wfk%fh, "number_of_spinor_components", nspinor_disk, lstat, error_data=error)
   ETSF_CHECK_MYERROR(lstat,error)
                                                                                                                               
   call etsf_io_low_read_var(Wfk%fh, "number_of_coefficients", npw_disk, lstat, start=(/ik_ibz/), error_data=error)
   ETSF_CHECK_MYERROR(lstat,error)
                                                                                                                               
   call etsf_io_low_read_var(Wfk%fh, "number_of_states", nband_disk, lstat, start=(/ik_ibz,spin/), error_data=error)
   ETSF_CHECK_MYERROR(lstat,error)

   ! Read reduced_coordinates_of_plane_waves for this k point.
   wave_folder%reduced_coordinates_of_plane_waves%data2D => kg_k
   wave_folder%red_coord_pw__kpoint_access = ik_ibz
   call etsf_io_basisdata_get(Wfk%fh, wave_folder, lstat, error)
   ETSF_CHECK_MYERROR(lstat,error)

   ! Read eigenvalues.
   electrons_folder%eigenvalues%data1D => eig_k
   electrons_folder%eigenvalues__kpoint_access = ik_ibz
   electrons_folder%eigenvalues__spin_access = spin
   !electrons_folder%occupations%data1D => occ
   !electrons_folder%occupations__kpoint_access = ik_ibz
   !electrons_folder%occupations__spin_access = spin
   call etsf_io_electrons_get(Wfk%fh, electrons_folder, lstat, error)
   ETSF_CHECK_MYERROR(lstat,error)

   MSG_ERROR("Cg reading with ETSF not coded yet")
#if 0
   ! get the coefficients_of_wavefunctions
   ! main_folder%coefficients_of_wavefunctions%data2D => cg(:, icg + 1:icg + npw * nspinor * nband)
   ! With g95, the association done above sometime leads to segfaults.
   ! So we allocate a temporary array to store the wfs of our kpt.
   ABI_ALLOCATE(main_folder%coefficients_of_wavefunctions%data2D,(2,npw*nspinor*nband))
   main_folder%wfs_coeff__kpoint_access = ik_ibz
   main_folder%wfs_coeff__spin_access = spin
   main_folder%wfs_coeff__number_of_states = nband
   main_folder%wfs_coeff__number_of_coefficients = npw
   call etsf_io_main_get(Wfk%fh, main_folder, lstat, error)
   ETSF_CHECK_MYERROR(lstat,error)
   ! Now we copy our values and deallocate the temporary array.
   ! cg(:,icg+1:icg+npw*nspinor*nband)=main_folder%coefficients_of_wavefunctions%data2D
   ! this is better than the previous instruction to optimize virtual memory.
   do iband=1,nband
     ipw=(iband-1)*npwso
     cg(:,icg+ipw+1:icg+ipw+npwso)=main_folder%coefficients_of_wavefunctions%data2D(:,ipw+1:ipw+npwso)
   end do
   ABI_DEALLOCATE(main_folder%coefficients_of_wavefunctions%data2D)
#endif
#endif

 CASE DEFAULT
   write(msg, '(7a,i0,3a)' )&
&   '  For the time being the input variable accesswff is restricted ',ch10,&
&   '  to 0 (all cases), 1 (in case MPI is enabled),',ch10,&
&   '  or 3 (only sequential, and if the NetCDF and ETSF_IO libraries have been enabled).',ch10,&
&   '  Its value is accesswff=',Wfk%accesswff,'.',ch10,&
&   '  Action : change accesswff or use ABINIT in parallel or enable NetCDF and/or ETSF_IO.'
   MSG_ERROR(msg)
 END SELECT

end subroutine wfk_read_band_block
!!***

!----------------------------------------------------------------------

!!****f* m_wfk/wfk_read_bmask
!! NAME
!!  wfk_read_bmask
!!
!! FUNCTION
!!  Read a set of bands specified by a mask
!!
!! INPUTS
!!  Wfk<type(wfk_t)>=
!!  ik_ibz=Index of the k-point in the IBZ.
!!  spin=Spin index
!!
!! OUTPUTS
!!  [kg_k=(:,:)] = G-vectors
!!  [eig_k(:,:)] = Eigenvectors
!!  [cg_k(:,:)]  = Fourier coefficients
!!
!! PARENTS
!!      m_wfk,m_wfs
!!
!! CHILDREN
!!      cwtime,hdr_clean,hdr_comm,hdr_io,wfk_close,wfk_open_read
!!      wfk_read_band_block,wfk_read_bmask,wrtout
!!
!! SOURCE

subroutine wfk_read_bmask(Wfk,wfk_bmask,ik_ibz,spin,kg_k,eig_k,cg_k) 


!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
#undef ABI_FUNC
#define ABI_FUNC 'wfk_read_bmask'
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: ik_ibz,spin
 type(wfk_t),intent(inout) :: Wfk
!arrays
 logical,intent(in) :: wfk_bmask(Wfk%mband)
 integer,intent(out),optional,target :: kg_k(:,:)  !(3,npw_k)
 real(dp),intent(out),optional,target :: cg_k(:,:) !(2,npw_k*nspinor*nband)
 real(dp),intent(out),optional,target :: eig_k(:)  !((2*mband)**formeig*mband) !,occ_k(mband)

!Local variables-------------------------------
!scalars
 integer :: ierr,npw_disk,nspinor_disk,nband_disk,band,ipw,my_bcount,cnt,npwso,npw_tot,pt1,pt2
 integer :: npw_read,nspinor_read,nband_read,nb_tot,ncount,my_bcnt,my_maxb
 character(len=500) :: msg
#ifdef HAVE_MPI_IO
 integer :: mpi_err,cgscatter_type,cg_type,method,block,nblocks,nbxblock
 integer :: bstart,bstop,bufsz,ugsz,brest,max_nband
 integer(XMPI_OFFSET_KIND) :: my_offset,base_ofs !,fmarker !my_offpad,
 integer :: band_block(2),sizes(2),subsizes(2),starts(2),statux(MPI_STATUS_SIZE)
 integer,allocatable :: block_length(:),block_type(:)
 integer(XMPI_ADDRESS_KIND),allocatable :: block_displ(:)
 real(dp),allocatable :: buffer(:,:)
#endif

!************************************************************************

 ABI_CHECK(Wfk%rw_mode==WFK_READMODE, "Wfk must be in READMODE")

 if (Wfk%Hdr%headform<40) then 
   write(msg,'(a,i0)')"Too old headform : ",Wfk%Hdr%headform
   MSG_ERROR(msg)
 end if
 !
 ! Look before you leap 
 npw_disk = Wfk%Hdr%npwarr(ik_ibz)
 nspinor_disk = Wfk%Hdr%nspinor
 nband_disk = Wfk%Hdr%nband(ik_ibz + (spin-1)*Wfk%Hdr%nkpt)
 nb_tot = COUNT(wfk_bmask)
 npw_tot = nb_tot* nspinor_disk * npw_disk

 if (PRESENT(kg_k)) then
   ABI_CHECK((SIZE(kg_k,DIM=2) >= npw_disk),"kg_k too small")
 end if
 !                                                                             
 if (PRESENT(eig_k)) then
   ABI_CHECK(SIZE(eig_k) >= nband_disk, "eig_k too small")
 end if
 !                                                                             
 if (PRESENT(cg_k)) then
  ABI_CHECK(SIZE(cg_k, DIM=2) >= npw_tot, "Too small cg_k")
 end if

 ! Rewind the file to have the correct (k,s) block (if needed)
 call wfk_seek(Wfk,ik_ibz,spin)

 SELECT CASE (Wfk%accesswff)

 CASE (IO_MODE_FORTRAN) 
   !
   ! Read the first record: npw, nspinor, nband_disk
   read(Wfk%fh, iostat=ierr) npw_read, nspinor_read, nband_read
   !read(Wfk%fh, iostat=ierr) npw_disk, nspinor_disk, nband_disk

   if (ierr/=0)then
     write(msg,'(4a)') &
      " Error while reading the (npw,nspinor,nband) record of the wfk file", TRIM(Wfk%fname),ch10,&
      ' Your file is likely not correct.'
     MSG_ERROR(msg)
   end if

   if ( ANY( (/npw_read, nspinor_read, nband_read/) /= (/npw_disk, nspinor_disk, nband_disk/) )) then
     MSG_ERROR("Mismatch between WFK and HDR")
   end if
   !
   ! The second record: (k+G) vectors
   if (PRESENT(kg_k)) then
     read(Wfk%fh, iostat=ierr) kg_k(1:3,1:npw_disk)
   else
     read(Wfk%fh, iostat=ierr) ! kg_k(1:3,1:npw_disk)
   end if

   if (ierr/=0)then
     write(msg,'(4a)')&
&     " Error while reading the k+g record of the WFK file",TRIM(Wfk%fname),ch10,&
&     " Your file is likely not correct."
     MSG_ERROR(msg)
   end if
   !
   ! The third record: eigenvalues
   if (Wfk%formeig==0) then

     if (PRESENT(eig_k)) then
       read(Wfk%fh, iostat=ierr) eig_k(1:nband_disk)
     else
       read(Wfk%fh, iostat=ierr) ! eig_k(1:nband_disk)
     end if
    
     if (ierr/=0)then
       write(msg,'(3a,i4,2a,i4,5a)') &
&       " Error while reading the eigenvalue record of the WFK file",TRIM(Wfk%fname),ch10,&
&       " Your file is likely not correct."
       MSG_ERROR(msg)
     end if

   else 
     MSG_ERROR("formeig != 0 not yet coded")
   end if
   !
   ! The wave-functions.
   if (PRESENT(cg_k)) then
     npwso = npw_disk*nspinor_disk
     my_bcount = 0
     do band=1,nband_disk
       if (wfk_bmask(band)) then
         ipw = my_bcount * npwso
         my_bcount = my_bcount + 1
         read(Wfk%fh, iostat=ierr) cg_k(1:2,ipw+1:ipw+npwso)
       else
         read(Wfk%fh, iostat=ierr) ! cg_k(1:2,ipw+1:ipw+npwso)
       end if
       if (ierr/=0) EXIT
     end do
     !
   else
     !
     do band=1,nband_disk
       read(Wfk%fh, iostat=ierr) ! cg_k(1:2,ipw+1:ipw+npwso)
       if (ierr/=0) EXIT
     end do
     !
   end if

   if (ierr/=0)then
     write(msg,'(3a,i4,2a,i4,5a)') &
      " Error while reading the cg record of the WFK file",TRIM(Wfk%fname),ch10,&
      " Your file is likely not correct."
     MSG_ERROR(msg)
   end if
   ! 
   ! Reached the end of the (k,s) block. Update f90_fptr
   if (ik_ibz < Wfk%Hdr%nkpt) then
     Wfk%f90_fptr = (/ik_ibz+1,spin,REC_NPW/)
   else
     Wfk%f90_fptr = (/1,spin+1,REC_NPW/)
   end if

 CASE (IO_MODE_MPI) 
#ifdef HAVE_MPI_IO
   if (PRESENT(kg_k)) then
     my_offset = Wfk%offset_ks(ik_ibz,spin,REC_KG) + xmpio_bsize_frm

     call MPI_FILE_READ_AT_ALL(Wfk%fh,my_offset,kg_k,3*npw_disk,MPI_INTEGER,statux,mpi_err)
     ABI_CHECK_MPI(mpi_err,"reading REC_KG")
     !write(std_out,*)"kg_k",kg_k
   end if

   if (PRESENT(eig_k)) then
     my_offset = Wfk%offset_ks(ik_ibz,spin,REC_EIG) + xmpio_bsize_frm

     call MPI_FILE_READ_AT_ALL(Wfk%fh,my_offset,eig_k,nband_disk,MPI_DOUBLE_PRECISION,statux,mpi_err)
     ABI_CHECK_MPI(mpi_err,"reading REC_EIG")
     !write(std_out,*)"eig_k",eig_k
   end if

   if (PRESENT(cg_k)) then
     method = 0

     SELECT CASE (method)
     CASE (0)
       ! DATA SIEVING: 
       !   read max_nband states in chuncks of nbxblock, then extract my states according to bmask.
       !
       ! MAX number of bands read by the procs in the communicator 
       my_maxb = nband_disk
       do band=nband_disk,1,-1
         if (wfk_bmask(band)) then
           my_maxb = band
           EXIT
         end if
       end do
       call xmax_mpi(my_maxb,max_nband,Wfk%comm,mpi_err)
       !max_nband = nband_disk
       !
       ! MPI-IO crashes if we try to read a large number of bands in a single call.
       nbxblock = max_nband
       if ((2*npw_disk*nspinor_disk*nbxblock*xmpi_bsize_dp) > Wfk%cg_maxbsize) then
         nbxblock = Wfk%cg_maxbsize / (2*npw_disk*nspinor_disk*xmpi_bsize_dp)
         if (nbxblock == 0) nbxblock = 50
       end if
       !nbxblock = 2

       nblocks = max_nband / nbxblock
       brest   = MOD(max_nband, nbxblock)
       if (brest /= 0) nblocks = nblocks + 1

       write(std_out,*)"in buffered bmask"
       write(std_out,*)"nb_tot",nb_tot,"nblocks",nblocks,"nbxblock",nbxblock

       base_ofs = Wfk%offset_ks(ik_ibz,spin,REC_CG)
       sizes = (/npw_disk*nspinor_disk, nband_disk/)

       my_bcnt = 0  ! index of my band in cg_k
       do block=1,nblocks
         bstart = 1 + (block-1) * nbxblock
         bstop  = bstart + nbxblock - 1 
         if (bstop > max_nband) bstop = max_nband
         !
         ! Allocate and read the buffer
         band_block = (/bstart, bstop/)
         ugsz = npw_disk*nspinor_disk
         bufsz = 2*ugsz*(bstop-bstart+1)
         ABI_MALLOC(buffer, (2,bufsz))

         !write(std_out,*)"  bstart,bstop, ",band_block
         subsizes = (/npw_disk*nspinor_disk, band_block(2)-band_block(1)+1/)
         starts = (/1, bstart/)

         call mpiotk_read_fsuba_all_dp2D(Wfk%fh,base_ofs,sizes,subsizes,starts,&
&          bufsz,buffer,Wfk%cg_maxbsize,Wfk%comm,ierr) 
         ABI_CHECK(ierr==0,"Fortran record too big")

         ! Extract my bands from buffer.
         do band=bstart,bstop
           if (wfk_bmask(band)) then
             my_bcnt = my_bcnt + 1
             pt1 = 1 + (my_bcnt - 1) * ugsz
             pt2 = 1 + (band - bstart) * ugsz
             cg_k(:,pt1:pt1+ugsz-1) = buffer(:,pt2:pt2+ugsz-1)
           end if
         end do

         ABI_FREE(buffer)
       end do

     CASE (1,2)
       call MPI_Type_contiguous(npw_disk*nspinor_disk,MPI_DOUBLE_COMPLEX,cg_type,mpi_err)
       ABI_CHECK_MPI(mpi_err,"type_contigous")

       if (method==1) then
         ncount = nb_tot
         ABI_MALLOC(block_length,(ncount+2))
         ABI_MALLOC(block_type, (ncount+2))
         ABI_MALLOC(block_displ,(ncount+2))

         block_length(1)=1
         block_displ (1)=0
         block_type  (1)=MPI_LB

         my_bcount = 1
         do band=1,Wfk%mband
           if (wfk_bmask(band)) then
             my_bcount = my_bcount + 1
             block_length(my_bcount) = 1
             block_type(my_bcount) = cg_type
             block_displ(my_bcount) = xmpio_bsize_frm + &
&              (band-1) * (2*npw_disk*nspinor_disk*xmpi_bsize_dp + 2*xmpio_bsize_frm)
           end if
         end do

         block_length(ncount+2) = 1
         block_displ (ncount+2) = block_displ(my_bcount)
         block_type  (ncount+2) = MPI_UB

       else if (method==2) then
         ! this file view is not efficient but it's similar to the 
         ! one used in wff_readwrite. Let's see if MPI-IO likes it!
         ncount = nb_tot* nspinor_disk * npw_disk

         ABI_MALLOC(block_length,(ncount+2))
         ABI_MALLOC(block_type, (ncount+2))
         ABI_MALLOC(block_displ,(ncount+2))
                                             
         block_length(1)=1
         block_displ (1)=0
         block_type  (1)=MPI_LB
         !
         ! The view starts at REC_CG
         cnt = 1
         do band=1,Wfk%mband
           if (wfk_bmask(band)) then
             base_ofs =  xmpio_bsize_frm + &
&              (band-1) * (2*npw_disk*nspinor_disk*xmpi_bsize_dp + 2*xmpio_bsize_frm)
             do ipw=1,npw_disk*nspinor_disk
               cnt = cnt + 1
               block_length(cnt) = 1
               block_type(cnt)   = MPI_DOUBLE_COMPLEX
               block_displ(cnt)  = base_ofs + 2*(ipw-1)*xmpi_bsize_dp 
             end do
           end if
         end do

         block_length(ncount+2) = 1
         block_displ (ncount+2) = block_displ(cnt)
         block_type  (ncount+2) = MPI_UB
       end if

       call xmpio_type_struct(ncount+2,block_length,block_displ,block_type,cgscatter_type,mpi_err)
       ABI_CHECK_MPI(mpi_err,"type_struct")

       ABI_FREE(block_length)
       ABI_FREE(block_type)
       ABI_FREE(block_displ)

       call MPI_TYPE_FREE(cg_type, mpi_err)
       ABI_CHECK_MPI(mpi_err,"MPI_TYPE_FREE")

       my_offset = Wfk%offset_ks(ik_ibz,spin,REC_CG)

       call MPI_FILE_SET_VIEW(Wfk%fh, my_offset, MPI_BYTE, cgscatter_type, 'native', MPI_INFO_NULL, mpi_err)
       ABI_CHECK_MPI(mpi_err,"SET_VIEW")
                                                                                                                 
       call MPI_TYPE_FREE(cgscatter_type, mpi_err)
       ABI_CHECK_MPI(mpi_err,"MPI_TYPE_FREE")

       call MPI_FILE_READ_ALL(Wfk%fh, cg_k, npw_tot, MPI_DOUBLE_COMPLEX, MPI_STATUS_IGNORE, mpi_err)
       ABI_CHECK_MPI(mpi_err,"FILE_READ_ALL")

     CASE DEFAULT
       MSG_ERROR("Wrong method")
     END SELECT
   end if
#endif

 CASE (IO_MODE_ETSF)
   MSG_ERROR("Not Coded")

 CASE DEFAULT
   write(msg, '(7a,i0,3a)' )&
&   '  For the time being the input variable accesswff is restricted ',ch10,&
&   '  to 0 (all cases), 1 (in case MPI is enabled),',ch10,&
&   '  or 3 (only sequential, and if the NetCDF and ETSF_IO libraries have been enabled).',ch10,&
&   '  Its value is accesswff=',Wfk%accesswff,'.',ch10,&
&   '  Action : change accesswff or use ABINIT in parallel or enable NetCDF and/or ETSF_IO.'
   MSG_ERROR(msg)
 END SELECT

end subroutine wfk_read_bmask
!!***

!----------------------------------------------------------------------

!!****f* m_wfk/wfk_seek
!! NAME
!!  wfk_seek
!!
!! FUNCTION
!!   Move the internal file pointer so that it points to the
!!   block (ik_ibz, spin). Needed only if accesswff==IO_MODE_FORTRAN 
!!
!! INPUTS
!!   ik_ibz,spin = (k-point,spin) indices
!! 
!! SIDE EFFECTS
!!   Wfk<type(Wfk_t)> : modifies Wfk%f90_fptr and the internal F90 file pointer.
!! PARENTS
!!      m_wfk
!!
!! CHILDREN
!!      cwtime,hdr_clean,hdr_comm,hdr_io,wfk_close,wfk_open_read
!!      wfk_read_band_block,wfk_read_bmask,wrtout
!!
!! SOURCE

subroutine wfk_seek(Wfk,ik_ibz,spin)


!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
#undef ABI_FUNC
#define ABI_FUNC 'wfk_seek'
!End of the abilint section

 implicit none

!Arguments ------------------------------------
 integer,intent(in)  :: ik_ibz,spin
 type(Wfk_t),intent(inout) :: Wfk

!Local variables-------------------------------
 integer :: ierr,ik_fpt,spin_fpt,recn_wanted,recn_fpt,rec_type

! *************************************************************************

 SELECT CASE (Wfk%accesswff)
 CASE (IO_MODE_FORTRAN)
   !
   ! Actual position inside the file
   ik_fpt   = Wfk%f90_fptr(1)
   spin_fpt = Wfk%f90_fptr(2)
   rec_type = Wfk%f90_fptr(3)
   recn_fpt = Wfk%recn_ks(ik_fpt,spin_fpt, rec_type)

   recn_wanted = Wfk%recn_ks(ik_ibz,spin, REC_NPW)

   !write(std_out,*)(recn_wanted - recn_fpt) 
   call mvrecord(Wfk%fh, (recn_wanted - recn_fpt) ,ierr)
   ABI_CHECK(ierr==0,"error in mvrecord")
   Wfk%f90_fptr = (/ik_ibz, spin, REC_NPW/)
 END SELECT 

end subroutine wfk_seek
!!***

!----------------------------------------------------------------------

!!****f* m_wfkfile/wfk_compute_offsets
!! NAME
!!  wfk_compute_offsets
!!
!! FUNCTION
!!
!! INPUTS
!!
!! OUTPUT
!!
!! PARENTS
!!      m_wfk
!!
!! CHILDREN
!!      cwtime,hdr_clean,hdr_comm,hdr_io,wfk_close,wfk_open_read
!!      wfk_read_band_block,wfk_read_bmask,wrtout
!!
!! SOURCE

subroutine wfk_compute_offsets(Wfk)

 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
#undef ABI_FUNC
#define ABI_FUNC 'wfk_compute_offsets'
!End of the abilint section

 implicit none

!Arguments ------------------------------------
 type(wfk_t),intent(inout) :: Wfk

!Local variables-------------------------------
!scalars
 integer :: spin,ik_ibz,band,npw_k,nband_k,bsize_frm,mpi_type_frm,base
 integer(XMPI_OFFSET_KIND) :: offset

! *************************************************************************

 ABI_CHECK(Wfk%formeig==0,"formeig != 0 not coded")

 SELECT CASE (Wfk%accesswff)

 CASE (IO_MODE_FORTRAN) 
   !
   ! Compute record numbers for Fortran IO
   ABI_MALLOC(Wfk%recn_ks,(Wfk%Hdr%nkpt,Wfk%Hdr%nsppol,4))

   ! TODO this should point to the end of the Header!
   ! if we want to have meaningfull absolute recn.
   base = 0 
   do spin=1,Wfk%Hdr%nsppol
     do ik_ibz=1,Wfk%Hdr%nkpt
       Wfk%recn_ks(ik_ibz,spin,1) = base + 1
       Wfk%recn_ks(ik_ibz,spin,2) = base + 2
       if (Wfk%formeig==0) then
         Wfk%recn_ks(ik_ibz,spin,3) = base + 3
       else 
         MSG_ERROR("formeig != 0 not coded")
       end if
       base = Wfk%recn_ks(ik_ibz,spin,3)
       do band=1,Wfk%Hdr%nband(ik_ibz + (spin-1)*Wfk%Hdr%nkpt)
         base = base + 1
         Wfk%recn_ks(ik_ibz,spin,4) = base
       end do
     end do
   end do

 CASE (IO_MODE_MPI) 
   !
   ! Compute offsets for MPI-IO.
   ABI_MALLOC(Wfk%offset_ks,(Wfk%Hdr%nkpt,Wfk%Hdr%nsppol,4))

   bsize_frm    = xmpio_bsize_frm    ! Byte length of the Fortran record marker.
   mpi_type_frm = xmpio_mpi_type_frm ! MPI type of the record marker.

   ! fform must be 0 ??
   ! The offset of the Header. TODO
   offset = Wfk%hdr_offset ! hdr_offset(Hdr)

   !write(std_out,*)ABI_FUNC//"hdr_offset",offset

   do spin=1,Wfk%Hdr%nsppol
     do ik_ibz=1,Wfk%Hdr%nkpt
       npw_k   = Wfk%Hdr%npwarr(ik_ibz)
       nband_k = Wfk%Hdr%nband(ik_ibz + (spin-1)*Wfk%Hdr%nkpt)
       !
       !---------------------------------------------------------------------------
       ! First record: npw, nspinor, nband_disk
       !---------------------------------------------------------------------------
       Wfk%offset_ks(ik_ibz,spin,REC_NPW) = offset

       if (Wfk%Hdr%headform>=40) then 
         ! npw, nspinor, nband_disk
         offset = offset +  3*xmpi_bsize_int + 2*bsize_frm
       else 
         MSG_ERROR("Old headforms < 40 are not supported")
       end if
       Wfk%offset_ks(ik_ibz,spin,REC_KG) = offset
       !
       !---------------------------------------------------------------------------
       ! Second record: (k+G) vectors
       ! kg_k(1:3,1:npw_k)
       !---------------------------------------------------------------------------
       offset = offset + 3*npw_k*xmpi_bsize_int + 2*bsize_frm
       Wfk%offset_ks(ik_ibz,spin,REC_EIG) = offset
       !
       !---------------------------------------------------------------------------
       ! Third record: eigenvalues
       !---------------------------------------------------------------------------
       if (Wfk%formeig==0) then
         ! eigen(1:nband_k), occ(1:nband_k)
         offset = offset + 2*nband_k*xmpi_bsize_dp + 2*bsize_frm 
       else if (Wfk%formeig==1) then
         MSG_ERROR("formeig==1 not coded") ! TODO 
         !do iband=1,nband1
         !  read (unitwf,iostat=ios) eigen(1+indxx:2*nband1+indxx)
         !end do
       else 
         MSG_ERROR("Wrong formeig")
       end if
       Wfk%offset_ks(ik_ibz,spin,REC_CG) = offset
       !
       !---------------------------------------------------------------------------
       ! Wavefunction coefficients 
       !---------------------------------------------------------------------------
       !do band=1,nband_k
       !  write(unitwf) cg_k(1:2,npw_k*nspinor)
       !end do
       offset = offset + nband_k * (2*npw_k*Wfk%Hdr%nspinor*xmpi_bsize_dp + 2*bsize_frm)
     end do ! ik_ibz
   end do ! spin
   !
   !do spin=1,Wfk%Hdr%nsppol
   !  do ik_ibz=1,Wfk%Hdr%nkpt
   !    write(std_out,*)"offset_ks",ik_ibz,spin,Wfk%offset_ks(ik_ibz,spin,:)
   !  end do
   !end do

   ! Check for possible wraparound errors.
   if (ANY(Wfk%offset_ks <= 0)) then
     MSG_ERROR("Found negative offset. File too large for MPI-IO!!!")
   end if

 END SELECT

end subroutine wfk_compute_offsets
!!***

!----------------------------------------------------------------------

!!****f* m_wfk/wfk_prof
!! NAME
!!  wfk_prof
!!
!! FUNCTION
!!
!! INPUTS
!! 
!! PARENTS
!!      kss2wfk
!!
!! CHILDREN
!!      cwtime,hdr_clean,hdr_comm,hdr_io,wfk_close,wfk_open_read
!!      wfk_read_band_block,wfk_read_bmask,wrtout
!!
!! SOURCE

subroutine wfk_prof(wfk_fname,nband,comm)


!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
#undef ABI_FUNC
#define ABI_FUNC 'wfk_prof'
 use interfaces_14_hidewrite
 use interfaces_18_timing
!End of the abilint section

 implicit none

!Arguments ------------------------------------
 integer,intent(in) :: nband,comm
 character(len=*),intent(in) :: wfk_fname

!Local variables-------------------------------
!scalars
 integer,parameter :: formeig0=0,rdwr1=1
 integer :: accesswff,wfk_unt,band,ik_ibz,spin,ierr,ii,option
 integer :: npw_disk,nband_disk,mcg,fform,nband_read
 integer :: master,my_rank,nproc
 real(dp) :: cpu,wall
 character(len=500) :: msg
 type(hdr_type) :: Hdr
 type(Wfk_t) :: Wfk
!arrays
 integer,parameter :: io_modes(2) = (/IO_MODE_FORTRAN, IO_MODE_MPI/)
 logical,allocatable :: my_bmask(:)
 integer,allocatable :: kg_k(:,:)
 real(dp),allocatable :: eig_k(:),cg_k(:,:)

! *************************************************************************

 call wrtout(std_out,ABI_FUNC//": about to read "//TRIM(wfk_fname),"COLL")

 my_rank = xcomm_rank(comm)
 nproc   = xcomm_size(comm)
 master  = 0

 wfk_unt = get_unit()
 if (my_rank == master) then
   open(unit=wfk_unt, file=wfk_fname, form="unformatted", status="old", iostat=ierr)
   ABI_CHECK(ierr==0,'Opening '//TRIM(wfk_fname)//' as old-unformatted')
   call hdr_io(fform,Hdr,rdwr1,wfk_unt)
   close(wfk_unt)
 end if
 if (nproc > 1) then
   call hdr_comm(Hdr,master,my_rank,comm)
 end if

 do ii=1,SIZE(io_modes)
   accesswff = io_modes(ii)
   do option=1,2
     call cwtime(cpu,wall,"start")

     if (option==1) then
       !
       do spin=1,Hdr%nsppol
         do ik_ibz=1,Hdr%nkpt
           npw_disk   = Hdr%npwarr(ik_ibz)
           nband_disk = Hdr%nband(ik_ibz+(spin-1)*Hdr%nkpt)

           nband_read = nband
           if (nband_read <=0) nband_read = nband_disk
           if (nband_read > nband_disk) then
             write(msg,'(a,2(i0,1x))')&
&             " nband_read cannot be greater than nband_disk while: ",nband_read,nband_disk
             MSG_ERROR(msg)
           end if

           mcg = npw_disk*Hdr%nspinor*nband_read

           ABI_MALLOC(eig_k,((2*nband_disk)**formeig0*nband_disk))
           ABI_MALLOC(kg_k,(3,npw_disk))
           ABI_MALLOC(cg_k,(2,mcg))
           ABI_CHECK_ALLOC("out of memory in cg_k") 

           call wfk_open_read(Wfk,wfk_fname,formeig0,accesswff,wfk_unt,comm)

           call wfk_read_band_block(Wfk,(/1,nband_read/),ik_ibz,spin,kg_k=kg_k,eig_k=eig_k,cg_k=cg_k)

           call wfk_close(Wfk)
           !
           ! Read the block of bands for this (k,s).
           !mband = nband_wfd
           !ABI_MALLOC(occ_k,(mband))
           !     call rwwf(cg_k,eig_k,formeig0,headform0,icg0,ik_ibz,spin,kg_k,mband,mcg,Wfd%MPI_enreg,nband_wfd,&
           !&       nband_disk,npw_disk,Wfd%nspinor,occ_k,option1,optkg1,tim_rwwf0,Wff)
           !ABI_FREE(occ_k)

           ABI_FREE(eig_k)
           ABI_FREE(kg_k)
           ABI_FREE(cg_k)
         end do !ik_ibz
       end do !spin

     else

       do spin=1,Hdr%nsppol
         do ik_ibz=1,Hdr%nkpt
           npw_disk   = Hdr%npwarr(ik_ibz)
           nband_disk = Hdr%nband(ik_ibz+(spin-1)*Hdr%nkpt)

           nband_read = nband
           if (nband_read <=0) nband_read = nband_disk
           if (nband_read > nband_disk) then
             write(msg,'(a,2(i0,1x))')&
&             " nband_read cannot be greater than nband_disk while: ",nband_read,nband_disk
             MSG_ERROR(msg)
           end if

           ABI_MALLOC(my_bmask,(MAXVAL(Hdr%nband)))
           my_bmask=.FALSE.
           do band=1,nband_read
             my_bmask(band) = .TRUE.
           end do

           ABI_MALLOC(eig_k,((2*nband_disk)**formeig0*nband_disk))
           ABI_MALLOC(kg_k,(3,npw_disk))

           mcg = npw_disk*Hdr%nspinor*COUNT(my_bmask)
           ABI_MALLOC(cg_k,(2,mcg))
           ABI_CHECK_ALLOC("out of memory in cg_k") 

           call wfk_open_read(Wfk,wfk_fname,formeig0,accesswff,wfk_unt,comm)

           call wfk_read_bmask(Wfk,my_bmask,ik_ibz,spin,kg_k=kg_k,eig_k=eig_k,cg_k=cg_k) 

           call wfk_close(Wfk)

           ABI_FREE(my_bmask)
           ABI_FREE(eig_k)
           ABI_FREE(kg_k)
           ABI_FREE(cg_k)
         end do !ik_ibz
       end do !spin
       !
     end if

     call cwtime(cpu,wall,"stop")
     write(msg,'(3(a,i2),2(a,f8.2))')&
&      " accesswff: ",accesswff,", nproc: ",nproc,", option: ",option,", cpu: ",cpu,", wall:",wall
     call wrtout(std_out,msg,"COLL")
   end do
 end do

 call hdr_clean(Hdr)

end subroutine wfk_prof
!!***

END MODULE m_wfk
!!***
