!{\src2tex{textfont=tt}}
!!****m* ABINIT/m_header
!! NAME
!! m_header
!!
!! FUNCTION
!! This module contains the definition of the abinit header (TODO) and methods acting on the data type.
!!
!! COPYRIGHT
!! Copyright (C) 2008-2012 ABINIT group (XG, MB, MT, DC, MG)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~abinit/COPYING
!! or http://www.gnu.org/copyleft/gpl.txt .
!! For the initials of contributors, see ~abinit/doc/developers/contributors.txt.
!!
!! PARENTS
!!
!! CHILDREN
!!
!! SOURCE

#if defined HAVE_CONFIG_H
#include "config.h"
#endif

#include "abi_common.h"

MODULE m_header

 use defs_basis
 use m_xmpi
 use m_profiling
 use m_errors
 use m_wffile
#if defined HAVE_MPI2
 use mpi
#endif
#if defined HAVE_TRIO_ETSF_IO
 use etsf_io_low_level
 use etsf_io
#endif


 use m_copy,          only : deep_copy
 use m_io_tools,      only : flush_unit
 use defs_wvltypes,   only : wvl_internal_type
 use defs_datatypes,  only : bandstructure_type, pseudopotential_type, pawtab_type
 use defs_abitypes,   only : hdr_type, dataset_type
 use m_pawrhoij,      only : pawrhoij_type, rhoij_alloc, rhoij_copy, rhoij_free, rhoij_io

 implicit none

 private
!!**

#if defined HAVE_MPI1
 include 'mpif.h'
#endif

 public :: m_header_init           ! Initialize the global variable HDR_fforms stored in this module.
 public :: hdr_fform2ftype         ! Return the filetype flag, a string with the name of the file and the patch level from fform.
 public :: hdr_ftype2fform         ! Returns the last fform associated to the filetype ftype.
 public :: hdr_init                ! Initialize the header structured datatype and most of its content from dtset and psps.
 public :: hdr_init_lowlvl         ! Low level initialization method for Hdr (no dtset).
 public :: hdr_nullify             ! Set all pointers to null.
 public :: hdr_clean               ! Deallocates the components of the header.
 public :: hdr_copy                ! Deep copy of the Header.
 public :: hdr_get_nelect_byocc    ! Returns the number of electrons calculated from Hdr%occ
 public :: isknown_headform        ! Returns .TRUE. is headform is in HDR_KNOWN_HEADFORMS.
 public :: hdr_mpio_skip           ! Skip the abinit header using MPI-IO routines and returns the offset of the
                                   ! first Fortran record after the header.
 public :: hdr_comm                ! Transmits the header datatype with MPI.
 public :: hdr_io_etsf             ! I/O of the hdr_type with ETSF-IO.
 public :: hdr_update              ! Update the header structured datatype.

 public :: hdr_skip
 public :: hdr_io

 interface hdr_skip  ! Generic interface of the routines hdr_skip
   module procedure hdr_skip_int
   module procedure hdr_skip_wfftype
 end interface hdr_skip

 interface hdr_io  !Generic interface of the routines hdr_io
   module procedure hdr_io_wfftype
   module procedure hdr_io_int
 end interface hdr_io

 integer,public,parameter :: HDR_NHEADFORMS=9
 ! Number of abinit header formats used so far.

 integer,public,parameter :: HDR_KNOWN_HEADFORMS( HDR_NHEADFORMS ) = (/23,34,40,41,42,44,53,56,57/)
 ! The list of headforms used so far.

 integer,public,parameter :: HDR_LATEST_HEADFORM = HDR_KNOWN_HEADFORMS( HDR_NHEADFORMS )
 ! The latest headform to be used for writing.

 ! Each filetype is denoted by the filetype flag HDR_* and a patch level.
 ! To add a new filetype do the following:
 !   1) add a new integer filetype flag
 !   2) increase HDR_NFTYPES
 !   3) Modify m_header_init to add a new entry in the array HDR_fforms.

 integer,public,parameter ::   &
&  HDR_WF_PW      = 1,         &
&  HDR_DENSITY    = 2,         &
&  HDR_POTENTIAL  = 3,         &
&  HDR_WF_WVL     = 4,         &
&  HDR_SCREENING  = 5,         &
&  HDR_BS_HAM     = 6,         &
&  HDR_BS_EIG     = 7,         &
&  HDR_BS_HAYDOCK = 8,         &
&  HDR_NFTYPES    = HDR_BS_HAYDOCK

!----------------------------------------------------------------------

 type,private :: fform_t
   !integer :: ftype
   character(len=500) :: fname
   integer,pointer :: fforms(:)
 end type fform_t

 type(fform_t),private,save,allocatable :: HDR_fforms(:)
 ! HDR_fforms(HDR_NFTYPES)
 ! The internal databases with the correspondence ftype ==> (fform, patch_level)

CONTAINS  !===========================================================
!!***

!----------------------------------------------------------------------

!!****f* m_header/m_header_init
!! NAME
!! m_header_init
!!
!! FUNCTION
!!  Initialize the global variable HDR_fforms stored in this module.
!!
!! OUTPUT
!!  ierr=A nonzero values signals an inconsistency in the internal tables.
!!
!! SIDE EFFECTS
!!   HDR_fforms(HDR_NFTYPES): Database completely initialized.
!!
!! PARENTS
!!      abi_init_globals
!!
!! CHILDREN
!!      mpi_barrier,mpi_bcast,rhoij_alloc
!!
!! SOURCE

subroutine m_header_init(ierr)

 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
#undef ABI_FUNC
#define ABI_FUNC 'm_header_init'
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(out) :: ierr
!Local variables-------------------------------
!scalars
 integer :: ft,nft

! *************************************************************************

 !@fform_t
 nft = HDR_NFTYPES
 ABI_DATATYPE_ALLOCATE(HDR_fforms,(nft))

 do ft=1,nft
   select case(ft)

   case (HDR_WF_PW)
     HDR_fforms(ft)%fname = "wf_planewave"
     ABI_MALLOC(HDR_fforms(ft)%fforms,(2))
     HDR_fforms(ft)%fforms = (/1,2/)

   case (HDR_DENSITY)
     HDR_fforms(ft)%fname = "density"
     ABI_MALLOC(HDR_fforms(ft)%fforms,(2))
     HDR_fforms(ft)%fforms = (/51,52/)

   case (HDR_POTENTIAL)
     HDR_fforms(ft)%fname = "potential"
     ABI_MALLOC(HDR_fforms(ft)%fforms,(2))
     HDR_fforms(ft)%fforms = (/101,102/)

   case (HDR_WF_WVL)
     HDR_fforms(ft)%fname = "wf_wavelet"
     ABI_MALLOC(HDR_fforms(ft)%fforms,(1))
     HDR_fforms(ft)%fforms = (/202/)

   ! FIXME Screening part
   case (HDR_SCREENING)
     HDR_fforms(ft)%fname = "screening"
     ABI_MALLOC(HDR_fforms(ft)%fforms,(2))
     HDR_fforms(ft)%fforms = (/1002,1102/)

   case (HDR_BS_HAM)
     HDR_fforms(ft)%fname = "bs_hamiltonian"
     ABI_MALLOC(HDR_fforms(ft)%fforms,(1))
     HDR_fforms(ft)%fforms = (/301/)

   case (HDR_BS_EIG)
     HDR_fforms(ft)%fname = "bs_eigenstates"
     ABI_MALLOC(HDR_fforms(ft)%fforms,(1))
     HDR_fforms(ft)%fforms = (/401/)

   case (HDR_BS_HAYDOCK)
     HDR_fforms(ft)%fname = "bs_haydock"
     ABI_MALLOC(HDR_fforms(ft)%fforms,(1))
     HDR_fforms(ft)%fforms = (/501/)

   case default
     ierr=ierr+1
   end select

 end do

end subroutine m_header_init
!!***

!----------------------------------------------------------------------

!!****f* m_header/hdr_fform2ftype
!! NAME
!! hdr_fform2ftype
!!
!! FUNCTION
!!  Return the filetype flag, a string with the name of the file and the patch level from the
!!  input fform.
!!
!! INPUTS
!!  fform=The value of fform read from the header.
!!
!! OUTPUT
!!  fname=The name of the file.
!!  ftype=The integer flag giving the filetype.
!!  patch_level=The
!!  ierr=A nonzero value signals that fform is not allowed
!!
!! PARENTS
!!
!! CHILDREN
!!      mpi_barrier,mpi_bcast,rhoij_alloc
!!
!! SOURCE

subroutine hdr_fform2ftype(fform,fname,ftype,patch_level,ierr)

 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
#undef ABI_FUNC
#define ABI_FUNC 'hdr_fform2ftype'
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: fform
 integer,intent(out) :: ftype,patch_level,ierr
 character(len=500),intent(out):: fname

!Local variables-------------------------------
!scalars
 integer :: ii,patch

! *************************************************************************

 !@fform_t
 ierr=0
 do ii=1,SIZE(HDR_fforms)
   if ( ANY( fform == HDR_fforms(ii)%fforms) ) then
     ierr = ierr+1
     ftype = ii
     fname = HDR_fforms(ii)%fname
     do patch=1,SIZE(HDR_fforms(ii)%fforms)
       if (fform == HDR_fforms(ii)%fforms(patch)) then
         patch_level=patch-1
         EXIT
       end if
     end do
   end if
 end do

 if (ierr==0) then ! Invalid fform, raise the error.
   ierr=1
 else if (ierr>1) then
   MSG_ERROR("Internal database is inconsistent")
 else
  ierr=0
 end if

end subroutine hdr_fform2ftype
!!***

!----------------------------------------------------------------------

!!****f* m_header/hdr_ftype2fform
!! NAME
!! hdr_ftype2fform
!!
!! FUNCTION
!!  Returns the last fform associated to the filetype ftype.
!!
!! INPUTS
!!  ftype
!!
!! OUTPUT
!!  fform
!!  ierr
!!
!! PARENTS
!!
!! CHILDREN
!!      mpi_barrier,mpi_bcast,rhoij_alloc
!!
!! SOURCE

subroutine hdr_ftype2fform(ftype,fform,ierr)

 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
#undef ABI_FUNC
#define ABI_FUNC 'hdr_ftype2fform'
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: ftype
 integer,intent(out) :: fform,ierr

!Local variables-------------------------------
!scalars
 integer :: last

! *************************************************************************

 !@fform_t
 ierr=0
 if ( ftype<1 .or. ftype>SIZE(HDR_fforms) ) then ! Wrong ftype.
  ierr=1; RETURN
 end if
 !
 ! Return the last fform associated to this ftype.
 last = SIZE(HDR_fforms(ftype)%fforms)
 fform = HDR_fforms(ftype)%fforms(last)

end subroutine hdr_ftype2fform
!!***

!----------------------------------------------------------------------

!!****f* m_header/hdr_init
!! NAME
!! hdr_init
!!
!! FUNCTION
!! This subroutine initializes the header structured datatype
!! and most of its content from dtset and psps, and put default values for
!! evolving variables.
!!
!! INPUTS
!! bstruct <type(bandstructure_type)>=band structure information
!!  including Brillouin zone description
!! codvsn=code version
!! dtset <type(dataset_type)>=all input variables for this dataset
!! mpi_atmtab(:)=--optional-- indexes of the atoms treated by current proc
!! mpi_comm_atom=--optional-- MPI communicator over atoms
!! pawtab(ntypat*usepaw) <type(pawtab_type)>=paw tabulated starting data
!! pertcase=index of the perturbation, or 0 if GS calculation
!! psps <type(pseudopotential_type)>=all the information about psps
!! my_atomtab(:)=Index of the atoms (in global numbering ) treated by current proc (Optional)
!!
!! OUTPUT
!! hdr <type(hdr_type)>=the header, initialized, and for most part of
!!   it, contain its definite values, except for evolving variables
!!
!! PARENTS
!!      gstate,loper3,nonlinear,respfn,setup_bse,setup_screening,setup_sigma
!!
!! CHILDREN
!!      mpi_barrier,mpi_bcast,rhoij_alloc
!!
!! SOURCE

subroutine hdr_init(bstruct,codvsn,dtset,hdr,pawtab,pertcase,psps,wvl, &
&                   mpi_atmtab,mpi_comm_atom) ! optional arguments (parallelism)


 use defs_basis
 use defs_wvltypes

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
#undef ABI_FUNC
#define ABI_FUNC 'hdr_init'
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: pertcase
 integer,intent(in),optional :: mpi_comm_atom
 character(len=6),intent(in) :: codvsn
 type(bandstructure_type),intent(in) :: bstruct
 type(dataset_type),intent(in) :: dtset
 type(hdr_type),intent(out) :: hdr
 type(pseudopotential_type),intent(in) :: psps
 type(wvl_internal_type),intent(in) :: wvl
!arrays
 integer,optional,target,intent(in) :: mpi_atmtab(:)
 type(pawtab_type),intent(in) :: pawtab(dtset%ntypat*psps%usepaw)

!Local variables-------------------------------
!scalars
 character(len=500) :: msg

! *************************************************************************

 !@hdr_type

! More checking would be needed ...
 if (dtset%ntypat/=psps%ntypat) then
   write(msg,'(a,2i0)')' dtset%ntypat and psps%ntypat differs. They are :',dtset%ntypat,psps%ntypat
   MSG_ERROR(msg)
 end if

 if (dtset%npsp/=psps%npsp) then
   write(msg,'(a,2i0)')' dtset%npsp and psps%npsp differs. They are :',dtset%npsp,psps%npsp
   MSG_ERROR(msg)
 end if

 if (present(mpi_comm_atom)) then
   if (present(mpi_atmtab)) then
     call hdr_init_lowlvl(hdr,bstruct,psps,pawtab,wvl,codvsn,&
&     pertcase,dtset%natom,dtset%nsym,dtset%nspden,dtset%ecut,dtset%pawecutdg,dtset%ecutsm,dtset%dilatmx,&
&     dtset%intxc,dtset%ixc,dtset%stmbias,dtset%usewvl,dtset%pawcpxocc,dtset%ngfft,dtset%ngfftdg,dtset%so_psp,&
&     dtset%qptn, dtset%rprimd_orig(:,:,1),dtset%xred_orig(:,:,1),dtset%symrel,dtset%tnons,dtset%symafm,dtset%typat,&
&     mpi_comm_atom=mpi_comm_atom,mpi_atmtab=mpi_atmtab)
   else
     call hdr_init_lowlvl(hdr,bstruct,psps,pawtab,wvl,codvsn,&
&     pertcase,dtset%natom,dtset%nsym,dtset%nspden,dtset%ecut,dtset%pawecutdg,dtset%ecutsm,dtset%dilatmx,&
&     dtset%intxc,dtset%ixc,dtset%stmbias,dtset%usewvl,dtset%pawcpxocc,dtset%ngfft,dtset%ngfftdg,dtset%so_psp,&
&     dtset%qptn, dtset%rprimd_orig(:,:,1),dtset%xred_orig(:,:,1),dtset%symrel,dtset%tnons,dtset%symafm,dtset%typat,&
&     mpi_comm_atom=mpi_comm_atom)
   end if
 else
   call hdr_init_lowlvl(hdr,bstruct,psps,pawtab,wvl,codvsn,&
&   pertcase,dtset%natom,dtset%nsym,dtset%nspden,dtset%ecut,dtset%pawecutdg,dtset%ecutsm,dtset%dilatmx,&
&   dtset%intxc,dtset%ixc,dtset%stmbias,dtset%usewvl,dtset%pawcpxocc,dtset%ngfft,dtset%ngfftdg,dtset%so_psp,&
&   dtset%qptn, dtset%rprimd_orig(:,:,1),dtset%xred_orig(:,:,1),dtset%symrel,dtset%tnons,dtset%symafm,dtset%typat)
 end if

end subroutine hdr_init
!!***

!----------------------------------------------------------------------

!!****f* m_header/hdr_nullify
!! NAME
!! hdr_nullify
!!
!! FUNCTION
!! This subroutine set all the pointers to null()
!!
!! INPUTS
!! hdr <type(hdr_type)>=the header
!!
!! PARENTS
!!      elphon,m_gamma,m_io_gkk,m_io_screening,read_el_veloc,read_gkk
!!
!! CHILDREN
!!      mpi_barrier,mpi_bcast,rhoij_alloc
!!
!! SOURCE

subroutine hdr_nullify(hdr)

 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
#undef ABI_FUNC
#define ABI_FUNC 'hdr_nullify'
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 type(hdr_type),intent(inout) :: hdr

! *************************************************************************
!@hdr_type

! integer
 nullify(hdr%istwfk)
 nullify(hdr%lmn_size)
 nullify(hdr%nband)
 nullify(hdr%npwarr)
 nullify(hdr%pspcod)
 nullify(hdr%pspdat)
 nullify(hdr%pspso)
 nullify(hdr%pspxc)
 nullify(hdr%so_psp)
 nullify(hdr%symafm)
 nullify(hdr%symrel)
 nullify(hdr%typat)

! real
 nullify(hdr%kptns)
 nullify(hdr%occ)
 nullify(hdr%tnons)
 nullify(hdr%wtk)
 nullify(hdr%xred)
 nullify(hdr%zionpsp)
 nullify(hdr%znuclpsp)
 nullify(hdr%znucltypat)

! string arrays
 nullify(hdr%title)

! types
 nullify(hdr%pawrhoij)

end subroutine hdr_nullify
!!***

!----------------------------------------------------------------------

!!****f* m_header/hdr_clean
!! NAME
!! hdr_clean
!!
!! FUNCTION
!! This subroutine deallocates the components of the header structured datatype
!!
!! INPUTS
!! hdr <type(hdr_type)>=the header
!!
!! OUTPUT
!!  (only deallocate)
!!
!! PARENTS
!!      bethe_salpeter,compare_interpol,conducti_nc,conducti_paw
!!      conducti_paw_core,cut3d,elphon,emispec_paw,gstate,gw_tools,initaim
!!      inpgkk,inwffil,ioarr,kss2wfk,linear_optics_paw,loper3,m_bse_io,m_ebands
!!      m_gamma,m_gwannier,m_io_gkk,m_io_kss,m_io_screening,m_wannier2abinit
!!      m_wfk,m_wfs,macroave,mrggkk,nonlinear,optic,read_el_veloc,read_gkk
!!      respfn,screening,setup_screening,setup_sigma,sigma,suscep
!!
!! CHILDREN
!!      mpi_barrier,mpi_bcast,rhoij_alloc
!!
!! SOURCE

subroutine hdr_clean(hdr)

 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
#undef ABI_FUNC
#define ABI_FUNC 'hdr_clean'
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 type(hdr_type),intent(inout) :: hdr

!Local variables-------------------------------

! *************************************************************************

 DBG_ENTER("COLL")

 !@hdr_type

 !integer
 if (associated(hdr%istwfk))   then
   ABI_FREE(hdr%istwfk)
 end if
 if (associated(hdr%lmn_size))   then
   ABI_FREE(hdr%lmn_size)
 end if
 if (associated(hdr%nband))   then
   ABI_FREE(hdr%nband)
 end if
 if (associated(hdr%npwarr))   then
   ABI_FREE(hdr%npwarr)
 end if

 if (associated(hdr%pspcod))   then
   ABI_FREE(hdr%pspcod)
 end if
 if (associated(hdr%pspdat))   then
   ABI_FREE(hdr%pspdat)
 end if
 if (associated(hdr%pspso))   then
   ABI_FREE(hdr%pspso)
 end if
 if (associated(hdr%pspxc))   then
   ABI_FREE(hdr%pspxc)
 end if
 if (associated(hdr%so_psp))   then
   ABI_FREE(hdr%so_psp)
 end if
 if (associated(hdr%symafm))   then
   ABI_FREE(hdr%symafm)
 end if
 if (associated(hdr%symrel))   then
   ABI_FREE(hdr%symrel)
 end if
 if (associated(hdr%typat))   then
   ABI_FREE(hdr%typat)
 end if

 !real
 if (associated(hdr%kptns))   then
   ABI_FREE(hdr%kptns)
 end if
 if (associated(hdr%occ))   then
   ABI_FREE(hdr%occ)
 end if
 if (associated(hdr%tnons))   then
   ABI_FREE(hdr%tnons)
 end if
 if (associated(hdr%wtk))   then
   ABI_FREE(hdr%wtk)
 end if
 if (associated(hdr%xred))   then
   ABI_FREE(hdr%xred)
 end if
 if (associated(hdr%zionpsp))   then
   ABI_FREE(hdr%zionpsp)
 end if
 if (associated(hdr%znuclpsp))   then
   ABI_FREE(hdr%znuclpsp)
 end if
 if (associated(hdr%znucltypat))   then
   ABI_FREE(hdr%znucltypat)
 end if

 !string arrays
 if(associated(hdr%title))   then
   ABI_FREE(hdr%title)
 end if

 if (hdr%usepaw==1 .and. associated(hdr%pawrhoij) ) then
  call rhoij_free(hdr%pawrhoij)
  ABI_DATATYPE_DEALLOCATE(hdr%pawrhoij)
 end if

 DBG_EXIT("COLL")

end subroutine hdr_clean
!!***

!----------------------------------------------------------------------

!!****f* m_header/hdr_copy
!! NAME
!! hdr_copy
!!
!! FUNCTION
!! Make a deep copy of the abinit header.
!!
!! INPUTS
!!  Hdr_in=The header to be copied.
!!
!! OUTPUT
!!  Hdr_cp=The deep copy of Hdr_in.
!!
!! NOTES
!!  The present version deals with versions of the header up to 56.
!!
!! PARENTS
!!      m_io_kss,m_io_screening,m_wannier2abinit,m_wfs
!!
!! CHILDREN
!!      mpi_barrier,mpi_bcast,rhoij_alloc
!!
!! SOURCE

subroutine hdr_copy(Hdr_in,Hdr_cp)

 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
#undef ABI_FUNC
#define ABI_FUNC 'hdr_copy'
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 type(hdr_type),intent(in) :: Hdr_in
 type(hdr_type),intent(inout) :: Hdr_cp

!Local variables-------------------------------
!scalars
 integer :: cplex
 character(len=500) :: msg
!arrays
 integer,pointer :: nlmn(:)
! *************************************************************************

!TODO add method to nullify header, deep_copy might crash if
!a pointer in the Hdr is not initialized to null(), tipically
!this happens for PAW quantities but we always check Hdr%usepaw

 !@hdr_type
 if (Hdr_in%headform>57) then
  write(msg,'(3a,i3,2a)')&
&  ' hdr_copy deals with versions of the header only up to 57. ',ch10,&
&  ' However headform = ',Hdr_in%headform,ch10,&
&  ' Change the source to add the changes done in the new version. '
  MSG_ERROR(msg)
 end if

!=== Integer values ===
 Hdr_cp%bantot   = Hdr_in%bantot
 Hdr_cp%date     = Hdr_in%date
 Hdr_cp%headform = Hdr_in%headform
 Hdr_cp%intxc    = Hdr_in%intxc
 Hdr_cp%ixc      = Hdr_in%ixc
 Hdr_cp%natom    = Hdr_in%natom
 Hdr_cp%nkpt     = Hdr_in%nkpt
 Hdr_cp%npsp     = Hdr_in%npsp
 Hdr_cp%nspden   = Hdr_in%nspden
 Hdr_cp%nspinor  = Hdr_in%nspinor
 Hdr_cp%nsppol   = Hdr_in%nsppol
 Hdr_cp%nsym     = Hdr_in%nsym
 Hdr_cp%ntypat   = Hdr_in%ntypat
 Hdr_cp%occopt   = Hdr_in%occopt
 Hdr_cp%pertcase = Hdr_in%pertcase
 Hdr_cp%usepaw   = Hdr_in%usepaw
 Hdr_cp%usewvl   = Hdr_in%usewvl

 ! === Integer arrays ===
 Hdr_cp%ngfft(:)   = Hdr_in%ngfft(:)
 Hdr_cp%nwvlarr(:) = Hdr_in%nwvlarr(:)

!=== Integer pointers ====
 call deep_copy( Hdr_in%istwfk,  Hdr_cp%istwfk   )
 call deep_copy( Hdr_in%lmn_size,Hdr_cp%lmn_size )
 call deep_copy( Hdr_in%nband,   Hdr_cp%nband    )
 call deep_copy( Hdr_in%npwarr,  Hdr_cp%npwarr   )
 call deep_copy( Hdr_in%pspcod,  Hdr_cp%pspcod )
 call deep_copy( Hdr_in%pspdat,  Hdr_cp%pspdat )
 call deep_copy( Hdr_in%pspso ,  Hdr_cp%pspso  )
 call deep_copy( Hdr_in%pspxc ,  Hdr_cp%pspxc  )
 call deep_copy( Hdr_in%so_psp,  Hdr_cp%so_psp )
 call deep_copy( Hdr_in%symafm,  Hdr_cp%symafm )
 call deep_copy( Hdr_in%symrel,  Hdr_cp%symrel )
 call deep_copy( Hdr_in%typat ,  Hdr_cp%typat  )

!=== Real variables ====
 Hdr_cp%ecut        = Hdr_in%ecut
 Hdr_cp%ecutdg      = Hdr_in%ecutdg
 Hdr_cp%ecutsm      = Hdr_in%ecutsm
 Hdr_cp%ecut_eff    = Hdr_in%ecut_eff
 Hdr_cp%etot        = Hdr_in%etot
 Hdr_cp%fermie      = Hdr_in%fermie
 Hdr_cp%residm      = Hdr_in%residm
 Hdr_cp%stmbias     = Hdr_in%stmbias
 Hdr_cp%tphysel     = Hdr_in%tphysel
 Hdr_cp%tsmear      = Hdr_in%tsmear

 Hdr_cp%qptn(:)     = Hdr_in%qptn(:)
 Hdr_cp%rprimd(:,:) = Hdr_in%rprimd(:,:)

!=== Real pointers ===
 call deep_copy( Hdr_in%kptns     ,Hdr_cp%kptns     )
 call deep_copy( Hdr_in%occ       ,Hdr_cp%occ       )
 !write(std_out,*)"DEBUG: ",Hdr_in%occ
 !write(std_out,*)"DEBUG: ",Hdr_in%bantot
 !if (associated(Hdr_cp%occ)) write(std_out,*)"DEBUG: ",Hdr_cp%occ
 call deep_copy( Hdr_in%tnons     ,Hdr_cp%tnons     )
 call deep_copy( Hdr_in%wtk       ,Hdr_cp%wtk       )
 call deep_copy( Hdr_in%xred      ,Hdr_cp%xred      )
 call deep_copy( Hdr_in%zionpsp   ,Hdr_cp%zionpsp   )
 call deep_copy( Hdr_in%znuclpsp  ,Hdr_cp%znuclpsp  )
 call deep_copy( Hdr_in%znucltypat,Hdr_cp%znucltypat)

!=== Character pointers ===
 Hdr_cp%codvsn = Hdr_in%codvsn
! THIS DOES NOT WORK ON XLF: Hdr_cp%title string length becomes huge and segfaults
! call deep_copy( Hdr_in%title,Hdr_cp%title )
 ABI_MALLOC(Hdr_cp%title,(Hdr_cp%npsp))
 Hdr_cp%title = Hdr_in%title

!=== For PAW have to copy Pawrhoij ====
!* TODO alchemy requires a different treatment but for the moment it is not available within PAW.
 if (Hdr_in%usepaw==1) then
  cplex = Hdr_in%Pawrhoij(1)%cplex
  nlmn => Hdr_in%lmn_size(1:Hdr_in%ntypat)
  ABI_DATATYPE_ALLOCATE(Hdr_cp%Pawrhoij,(Hdr_in%natom))
  call rhoij_alloc(cplex,Hdr_in%nspden,Hdr_in%nspinor,Hdr_in%nsppol,Hdr_cp%Pawrhoij,Hdr_in%typat,lmnsize=nlmn)
  call rhoij_copy(Hdr_in%Pawrhoij,Hdr_cp%Pawrhoij)
 end if

end subroutine hdr_copy
!!***

!----------------------------------------------------------------------

!!****f* m_header/hdr_get_nelect_byocc
!! NAME
!! hdr_get_nelect_byocc
!!
!! FUNCTION
!!  Return the number of electrons from the occupation numbers
!!  thus taking into account a possible additional charge or alchemy.
!!
!! INPUTS
!!  Hdr<hdr_type>
!!
!! OUTPUT
!!  nelect=Number of electrons in the unit cell.
!!
!! PARENTS
!!
!! CHILDREN
!!
!! SOURCE

function hdr_get_nelect_byocc(Hdr) result(nelect)

 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
#undef ABI_FUNC
#define ABI_FUNC 'hdr_get_nelect_byocc'
!End of the abilint section

 implicit none

!Arguments ---------------------------------------------
!scalars
 type(hdr_type),intent(in) :: Hdr
 real(dp) :: nelect

!Local variables ---------------------------------------
!scalars
 integer :: idx,isppol,ikibz,nband_k

! *************************************************************************

!* Cannot use znucl because we might have additional charge or alchemy.
 nelect=zero ; idx=0
 do isppol=1,Hdr%nsppol
   do ikibz=1,Hdr%nkpt
     nband_k=Hdr%nband(ikibz+(isppol-1)*Hdr%nkpt)
     nelect = nelect + Hdr%wtk(ikibz)*SUM(Hdr%occ(idx+1:idx+nband_k))
     idx=idx+nband_k
   end do
 end do

!Might also check also Hdr%znuclpsp(:) to avoid round off errors

end function hdr_get_nelect_byocc
!!***

!----------------------------------------------------------------------

!!****f* m_header/isknown_headform
!! NAME
!!  isknown_headform
!!
!! FUNCTION
!!  Returns .TRUE. if headform is one of the allowed values.
!!
!! INPUTS
!!  headform
!!
!! SOURCE

function isknown_headform(headform) result(ans)

 use defs_basis

!Arguments ------------------------------------

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
#undef ABI_FUNC
#define ABI_FUNC 'isknown_headform'
!End of the abilint section

 integer,intent(in) :: headform
 logical :: ans

! *************************************************************************

 ans = ANY(headform == HDR_KNOWN_HEADFORMS)

end function isknown_headform
!!***

!----------------------------------------------------------------------

!!****f* m_header/hdr_init_lowlvl
!! NAME
!! hdr_init_lowlvl
!!
!! FUNCTION
!! This subroutine initializes the header structured datatype
!! and most of its content from psps and other input variables that
!! are passed explicitly. It also use default values for evolving variables.
!! Note that Dtset is not required thus rendering the initialization of the header
!! much easier.
!!
!! INPUTS
!! bstruct <type(bandstructure_type)>=band structure information
!!  including Brillouin zone description
!! codvsn=code version
!! mpi_atmtab(:)=--optional-- indexes of the atoms treated by current proc
!! mpi_comm_atom=--optional-- MPI communicator over atoms
!! pawtab(ntypat*usepaw) <type(pawtab_type)>=paw tabulated starting data
!! pertcase=index of the perturbation, or 0 if GS calculation
!! psps <type(pseudopotential_type)>=all the information about psps
!! For the meaning of the other varialble see the definition of dataset_type.
!!
!! OUTPUT
!! hdr <type(hdr_type)>=the header, initialized, and for most part of
!!   it, contain its definite values, except for evolving variables
!!
!! PARENTS
!!      m_header,m_wfs
!!
!! CHILDREN
!!      mpi_barrier,mpi_bcast,rhoij_alloc
!!
!! SOURCE

subroutine hdr_init_lowlvl(hdr,bstruct,psps,pawtab,wvl,&
&  codvsn,pertcase,natom,nsym,nspden,ecut,pawecutdg,ecutsm,dilatmx,&
&  intxc,ixc,stmbias,usewvl,pawcpxocc,ngfft,ngfftdg,so_psp,qptn,&
&  rprimd,xred,symrel,tnons,symafm,typat, &
&  mpi_atmtab,mpi_comm_atom) ! optional arguments (parallelism)


 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
#undef ABI_FUNC
#define ABI_FUNC 'hdr_init_lowlvl'
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: natom,nsym,nspden,intxc,ixc,usewvl,pawcpxocc,pertcase
 integer, intent(in),optional :: mpi_comm_atom
 real(dp),intent(in) :: ecut,ecutsm,dilatmx,stmbias,pawecutdg
 character(len=6),intent(in) :: codvsn
 type(bandstructure_type),intent(in) :: bstruct
 type(pseudopotential_type),intent(in) :: psps
 type(wvl_internal_type),intent(in) :: wvl
 type(hdr_type),intent(out) :: hdr
!arrays
 integer,intent(in) :: typat(natom)
 integer,intent(in) ::  so_psp(psps%npsp)
 integer,intent(in) :: symrel(3,3,nsym),symafm(nsym)
 integer,intent(in) :: ngfft(18),ngfftdg(18)
 integer,optional,target,intent(in) :: mpi_atmtab(:)
 real(dp),intent(in) :: tnons(3,nsym)
 real(dp),intent(in) :: qptn(3) ! the wavevector, in case of a perturbation
 real(dp),intent(in) :: rprimd(3,3),xred(3,natom)
 type(pawtab_type),intent(in) :: pawtab(psps%ntypat*psps%usepaw)

!Local variables-------------------------------
!scalars
 integer :: bantot,date,nkpt,npsp,ntypat,nsppol,nspinor
 integer :: idx,isppol,ikpt,iband,ipsp
#ifndef HAVE_DFT_BIGDFT
 character(len=500) :: msg
#endif
 character(len=8) :: date_time
!arrays

! *************************************************************************

 !@hdr_type
 call date_and_time(date_time)
 read(date_time,'(i8)')date

 npsp   = psps%npsp
 ntypat = psps%ntypat
 nkpt   = bstruct%nkpt
 nsppol = bstruct%nsppol
 nspinor= bstruct%nspinor
 bantot = bstruct%bantot

!Transfer dimensions and other scalars to hdr.
 hdr%intxc    =intxc
 hdr%ixc      =ixc
 hdr%natom    =natom
 hdr%npsp     =npsp
 hdr%nspden   =nspden
 hdr%nspinor  =nspinor
 hdr%nsym     =nsym
 hdr%ntypat   =ntypat
 hdr%bantot   =bantot
 hdr%nkpt     =nkpt
 hdr%nsppol   =nsppol
 hdr%usepaw   =psps%usepaw
 hdr%usewvl   =usewvl !hdr%nwvlarr will be set later since the number !of wavelets have not yet been computed.
 hdr%occopt   =bstruct%occopt
 hdr%codvsn   =codvsn
 hdr%date     =date
 hdr%headform =HDR_LATEST_HEADFORM ! Initialize with the latest headform
 hdr%pertcase =pertcase
 hdr%ecut     =ecut
 hdr%ecutsm   =ecutsm
 hdr%ecut_eff =ecut * (dilatmx)**2
 hdr%stmbias  =stmbias
 hdr%tphysel  =bstruct%tphysel
 hdr%tsmear   =bstruct%tsmear
 hdr%qptn     =qptn
 hdr%rprimd   =rprimd      ! Evolving data

!Default for other data  (all evolving data)
 hdr%etot     =1.0d20
 hdr%fermie   =1.0d20
 hdr%residm   =1.0d20

!Allocate all components of hdr

!Transfer data from bstruct
 ABI_MALLOC(hdr%istwfk,(nkpt))
 hdr%istwfk(1:nkpt)      =bstruct%istwfk(1:nkpt)
 ABI_MALLOC(hdr%kptns,(3,nkpt))
 hdr%kptns(:,:)          =bstruct%kptns(:,:)
 ABI_MALLOC(hdr%nband,(nkpt*nsppol))
 hdr%nband(1:nkpt*nsppol)=bstruct%nband(1:nkpt*nsppol)
 ABI_MALLOC(hdr%npwarr,(nkpt))
 hdr%npwarr(:)           =bstruct%npwarr(:)
 ABI_MALLOC(hdr%wtk,(nkpt))
 hdr%wtk(:)=bstruct%wtk(:)

!Transfer data from psps
 ABI_MALLOC(hdr%pspcod,(npsp))
 hdr%pspcod    =psps%pspcod
 ABI_MALLOC(hdr%pspdat,(npsp))
 hdr%pspdat    =psps%pspdat
 ABI_MALLOC(hdr%pspso,(npsp))
 hdr%pspso     =psps%pspso
 ABI_MALLOC(hdr%pspxc,(npsp))
 hdr%pspxc     =psps%pspxc
 ABI_MALLOC(hdr%znuclpsp,(npsp))
 hdr%znuclpsp  =psps%znuclpsp
 ABI_MALLOC(hdr%znucltypat,(ntypat))
 hdr%znucltypat=psps%znucltypat
 ABI_MALLOC(hdr%zionpsp,(npsp))
 hdr%zionpsp   =psps%zionpsp
 ABI_MALLOC(hdr%title,(npsp))
 do ipsp=1,psps%npsp
   write(hdr%title(ipsp), "(A)") psps%title(ipsp)(1:132)
 end do

 ABI_MALLOC(hdr%so_psp,(npsp))
 hdr%so_psp=so_psp

 ABI_MALLOC(hdr%symafm,(nsym))
 hdr%symafm(1:min(size(symafm),size(hdr%symafm)))=symafm(1:min(size(symafm),size(hdr%symafm)))

 ABI_MALLOC(hdr%symrel,(3,3,nsym))
 hdr%symrel(:,:,1:min(size(symrel,3),size(hdr%symrel,3))) =symrel(:,:,1:min(size(symrel,3),size(hdr%symrel,3)))

 ABI_MALLOC(hdr%tnons,(3,nsym))
 hdr%tnons(:,1:min(size(tnons,2),size(hdr%tnons,2)))=tnons(:,1:min(size(tnons,2),size(hdr%tnons,2)))

 ABI_MALLOC(hdr%typat,(natom))
 hdr%typat(1:natom) =typat(1:natom)  ! PMA : in tests/v2/t11 size(dtset%typat) is bigger dtset%natom
 ABI_MALLOC(hdr%xred,(3,natom))
 hdr%xred(:,1:natom)=xred(:,1:natom) ! Evolving data

 if (psps%usepaw==1)then
   ABI_DATATYPE_ALLOCATE(hdr%pawrhoij,(natom))
   !Values of nspden/nspinor/nsppol are dummy ones; they are overwritten later (by hdr_update)
   if (present(mpi_comm_atom)) then
     if (present(mpi_atmtab)) then
       call rhoij_alloc(pawcpxocc,nspden,nspinor,nsppol,hdr%pawrhoij,typat, &
&                       pawtab=pawtab,mpi_comm_atom=mpi_comm_atom,mpi_atmtab=mpi_atmtab)
     else
       call rhoij_alloc(pawcpxocc,nspden,nspinor,nsppol,hdr%pawrhoij,typat, &
&                       pawtab=pawtab,mpi_comm_atom=mpi_comm_atom)
     end if
   else
     call rhoij_alloc(pawcpxocc,nspden,nspinor,nsppol,hdr%pawrhoij,typat,pawtab=pawtab)
   end if
 end if

 if (psps%usepaw==1) then
   hdr%ngfft(:) =ngfftdg(1:3)
 else if (usewvl==1) then
#if defined HAVE_DFT_BIGDFT
   hdr%ngfft(:) = (/ wvl%Glr%d%n1i, wvl%Glr%d%n2i, wvl%Glr%d%n3i /)
#else
 write(msg, '(a,a,a,a)' ) ch10,&
& ' hdr_init_lowlvl : BigDFT library is not compiled.', ch10, &
& '   Action, used the flag --enable-bigdft when configuring.'
 MSG_ERROR(msg)
#endif
 else
   hdr%ngfft(:) =ngfft(1:3)
 end if

!Transfer paw data
 ABI_MALLOC(hdr%lmn_size,(npsp))
 if(psps%usepaw==1) then
   hdr%ecutdg   =pawecutdg
   hdr%lmn_size(1:npsp)=pawtab(1:npsp)%lmn_size
 else
   hdr%ecutdg=hdr%ecut
   hdr%lmn_size(:)=psps%lmnmax
 end if

 !call get_eneocc_vect(bstruct,'occ',hdr%occ)  ! Evolving data
 ABI_MALLOC(hdr%occ,(bantot))
 hdr%occ(:)=zero; idx=0
 do isppol=1,nsppol
   do ikpt=1,nkpt
     do iband=1,hdr%nband(ikpt+(isppol-1)*nkpt)
       idx=idx+1
       hdr%occ(idx)=bstruct%occ(iband,ikpt,isppol)
     end do
   end do
 end do

end subroutine hdr_init_lowlvl
!!***

!----------------------------------------------------------------------

!!****f* m_header/hdr_mpio_skip
!! NAME
!!  hdr_mio_skip
!!
!! FUNCTION
!!   Skip the abinit header in MPI-IO mode. This routine uses local MPI-IO calls hence
!!   it can be safely called by master node only. Note however that in this case the
!!   offset has to be communicated to the other nodes.
!!
!! INPUTS
!!  mpio_fh=MPI-IO file handler
!!
!! OUTPUT
!!  fform=kind of the array in the file
!!  offset=The offset of the Fortran record located immediately below the Abinit header.
!!
!! SOURCE

subroutine hdr_mpio_skip(mpio_fh,fform,offset)

 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
#undef ABI_FUNC
#define ABI_FUNC 'hdr_mpio_skip'
!End of the abilint section

 implicit none

!Arguments ------------------------------------
 integer,intent(in) :: mpio_fh
 integer,intent(out) :: fform
 integer(kind=XMPI_OFFSET_KIND),intent(out) :: offset

!Local variables-------------------------------
!scalars
 integer :: bsize_frm,mpi_type_frm
#ifdef HAVE_MPI_IO
 integer :: headform,ierr,mu,usepaw,npsp
!arrays
 integer(kind=MPI_OFFSET_KIND) :: fmarker,positloc
 integer :: statux(MPI_STATUS_SIZE)
#endif

! *************************************************************************

 offset = 0
 fform  = 0

 bsize_frm    = xmpio_bsize_frm    ! bsize_frm= Byte length of the Fortran record marker.
 mpi_type_frm = xmpio_mpi_type_frm ! MPI type of the record marker.

#ifdef HAVE_MPI_IO
!Reading the first record of the file -------------------------------------
!read (unitfi)   codvsn,headform,..............
 positloc = bsize_frm + 6*xmpi_bsize_ch
 call MPI_FILE_READ_AT(mpio_fh,positloc,fform,1,MPI_INTEGER,statux,ierr)
 !write(std_out,*)" MPI_FILE_READ_AT test fform: ",fform

 if (ANY(fform == (/1,2,51,52,101,102/) )) then
   headform=22  ! This is the old format !read (unitfi) codvsn,fform
 else
   !read (unitfi)codvsn,headform,fform
   call MPI_FILE_READ_AT(mpio_fh,positloc,headform,1,MPI_INTEGER,statux,ierr)
   positloc = positloc + xmpi_bsize_int
   call MPI_FILE_READ_AT(mpio_fh,positloc,fform,1,MPI_INTEGER,statux,ierr)
   !write(std_out,*)" MPI_FILE_READ_AT: headform, fform: ",headform,fform
 end if

 ! Skip first record.
 call xmpio_read_frm(mpio_fh,offset,xmpio_at,fmarker,ierr)

!Reading the second record of the file: read(unitfi) bantot, hdr%date, hdr%intxc.................
!Read npsp and usepaw.
 positloc  = offset + bsize_frm + xmpi_bsize_int*13
 call MPI_FILE_READ_AT(mpio_fh,positloc,npsp,1,MPI_INTEGER,statux,ierr)

 usepaw=0
 if (headform >= 44) then
   positloc = positloc +  xmpi_bsize_int*4
   call MPI_FILE_READ_AT(mpio_fh,positloc,usepaw,1,MPI_INTEGER,statux,ierr)
 end if

 ! Skip second record.
 call xmpio_read_frm(mpio_fh,offset,xmpio_at,fmarker,ierr)

 ! Skip the rest of the file ---------------------------------------------
 do mu=1,2+npsp
   call xmpio_read_frm(mpio_fh,offset,xmpio_at,fmarker,ierr)
 end do

 if (headform>=44.and.usepaw==1) then ! skip rhoij records.
   call xmpio_read_frm(mpio_fh,offset,xmpio_at,fmarker,ierr)
   call xmpio_read_frm(mpio_fh,offset,xmpio_at,fmarker,ierr)
 end if

#else
 MSG_ERROR("hdr_mpio_skip cannot be used when MPI-IO is not enabled")
 ABI_UNUSED(mpio_fh)
#endif

end subroutine hdr_mpio_skip
!!***

!----------------------------------------------------------------------

!!****f* m_header/hdr_io_wfftype
!! NAME
!! hdr_io_wfftype
!!
!! FUNCTION
!! This subroutine deals with the I/O of the hdr_type
!! structured variables (read/write/echo).
!! According to the value of rdwr, it reads the header
!! of a file, writes it, or echo the value of the structured
!! variable to a file.
!! Note that, when reading, different records of hdr
!! are allocated here, according to the values of the
!! read variables. Records of hdr should be deallocated
!! correctly by a call to hdr_clean when hdr is not used anymore.
!! Two instances of the hdr_io routines are defined :
!!  hdr_io_int to which only the unit number is given
!!  hdr_io_wfftype to which a wffil datatype is given
!!
!! INPUTS
!!  rdwr= if 1, read the hdr structured variable from the header of the file,
!!        if 2, write the header to unformatted file
!!        if 3, echo part of the header to formatted file (records 1 and 2)
!!        if 4, echo the header to formatted file
!!        if 5, read the hdr without rewinding (unformatted)
!!        if 6, write the hdr without rewinding (unformatted)
!!  unitfi=unit number of the file (unformatted if rdwr=1, 2, 5 or 6 formatted if rdwr=3,4)
!!
!! OUTPUT
!!  (see side effects)
!!
!! SIDE EFFECTS
!!  The following variables are both input or output :
!!  fform=kind of the array in the file
!!   if rdwr=1,5 : will be output ; if the reading fail, return fform=0
!!   if rdwr=2,3,4,6 : should be input, will be written or echo to file
!!  hdr <type(hdr_type)>=the header structured variable
!!   if rdwr=1,5 : will be output
!!   if rdwr=2,3,4,6 : should be input, will be written or echo to file
!!
!! NOTES
!! In all cases, the file is supposed to be open already
!! When reading (rdwr=1) or writing (rdwr=2), rewind the file
!! When echoing (rdwr=3) does not rewind the file.
!! When reading (rdwr=5) or writing (rdwr=6), DOES NOT rewind the file
!!
!! PARENTS
!!
!! CHILDREN
!!      mpi_barrier,mpi_bcast,rhoij_alloc
!!
!! SOURCE

subroutine hdr_io_wfftype(fform,hdr,rdwr,wff)

 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
#undef ABI_FUNC
#define ABI_FUNC 'hdr_io_wfftype'
!End of the abilint section

 implicit none

!Arguments ------------------------------------
 integer,intent(inout) :: fform
 integer,intent(in) :: rdwr
 type(hdr_type),intent(inout) :: hdr
 type(wffile_type),intent(inout) :: wff

!Local variables-------------------------------
#if defined HAVE_MPI
 integer :: ierr
#endif

! *************************************************************************

 DBG_ENTER("COLL")

 if ( wff%accesswff==IO_MODE_FORTRAN .or. &
& (wff%accesswff==IO_MODE_FORTRAN_MASTER .and.wff%master==wff%me).or. &
& (wff%accesswff==IO_MODE_MPI  .and.wff%master==wff%me)    ) then
   call hdr_io_int(fform,hdr,rdwr,wff%unwff)
 end if

#if defined HAVE_MPI
!In the parallel case, if the files were not local, need to bcast the data
 if(rdwr==1)then
   if (wff%accesswff==IO_MODE_FORTRAN_MASTER .or. wff%accesswff==IO_MODE_MPI) then
     if (wff%spaceComm/=MPI_COMM_SELF) then
       call MPI_BCAST(fform,1,MPI_INTEGER,wff%master,wff%spaceComm,ierr)
       call hdr_comm(hdr,wff%master,wff%me,wff%spaceComm)
     end if
     wff%headform=hdr%headform
     if(wff%accesswff==IO_MODE_MPI)then
       call hdr_skip_wfftype(wff,ierr)
     end if
   end if
 end if
#if defined HAVE_MPI_IO
 if (rdwr == 2 .and. wff%accesswff==IO_MODE_MPI) then
   if (wff%spaceComm/=MPI_COMM_SELF) then
     call MPI_BARRIER(wff%spaceComm,ierr)
   end if
   wff%headform=hdr%headform
   call hdr_skip_wfftype(wff,ierr)
 end if
#endif
 if (rdwr==5) wff%headform=hdr%headform
#else
 if (rdwr==1.or.rdwr==5) wff%headform=hdr%headform
#endif

 DBG_EXIT("COLL")

end subroutine hdr_io_wfftype
!!***

!----------------------------------------------------------------------

!!****f* hdr_io/hdr_io_int
!! NAME
!! hdr_io_int
!!
!! FUNCTION
!! This subroutine deals with the I/O of the hdr_type structured variables (read/write/echo).
!! According to the value of rdwr, it reads the header of a file, writes it, or echo the value of the structured
!! variable to a file. Note that, when reading, different records of hdr are allocated here, according to the values of the
!! read variables. Records of hdr should be deallocated correctly by a call to hdr_clean when hdr is not used anymore.
!! Two instances of the hdr_io routines are defined :
!!   hdr_io_int to which only the unit number is given
!!   hdr_io_wfftype to which a wffil datatype is given
!!
!! INPUTS
!!  rdwr= if 1, read the hdr structured variable from the header of the file,
!!        if 2, write the header to unformatted file
!!        if 3, echo part of the header to formatted file (records 1 and 2)
!!        if 4, echo the header to formatted file
!!        if 5, read the hdr without rewinding (unformatted)
!!        if 6, write the hdr without rewinding (unformatted)
!!  unitfi=unit number of the file (unformatted if rdwr=1, 2, 5 or 6 formatted if rdwr=3,4)
!!
!! OUTPUT
!!  (see side effects)
!!
!! SIDE EFFECTS
!!  The following variables are both input or output :
!!  fform=kind of the array in the file
!!   if rdwr=1,5 : will be output ; if the reading fail, return fform=0
!!   if rdwr=2,3,4,6 : should be input, will be written or echo to file
!!  hdr <type(hdr_type)>=the header structured variable
!!   if rdwr=1,5 : will be output
!!   if rdwr=2,3,4,6 : should be input, will be written or echo to file
!!
!! NOTES
!! In all cases, the file is supposed to be open already
!! When reading (rdwr=1) or writing (rdwr=2), rewind the file
!! When echoing (rdwr=3) does not rewind the file.
!! When reading (rdwr=5) or writing (rdwr=6), DOES NOT rewind the file
!!
!! PARENTS
!!      m_header
!!
!! CHILDREN
!!      mpi_barrier,mpi_bcast,rhoij_alloc
!!
!! SOURCE

subroutine hdr_io_int(fform,hdr,rdwr,unitfi)

 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
#undef ABI_FUNC
#define ABI_FUNC 'hdr_io_int'
!End of the abilint section

 implicit none

!Arguments ------------------------------------
 integer,intent(inout) :: fform
 integer,intent(in) :: rdwr,unitfi
 type(hdr_type),intent(inout) :: hdr

!Local variables-------------------------------
 integer :: bantot,headform,iatom,ierr,ii,ikpt,ipsp,isym
 integer :: lloc,lmax,mmax,natom,nkpt,npsp,nsppol
 integer :: nsym,ntypat
 character(len=500) :: message
 character(len=6) :: codvsn
 real(dp) :: acell(3)

!*************************************************************************

 DBG_ENTER("COLL")

!-------------------------------------------------------------------------
!Reading the header of an unformatted file
!-------------------------------------------------------------------------

 if(rdwr==1 .or. rdwr==5)then

   if (rdwr==1) rewind(unitfi)

!  Reading the first record of the file ------------------------------------

   read(unitfi,iostat=ierr)codvsn,fform
   if (ierr /=0) then
     fform=0
     return   ! This is to allow treatment of old epsm1 format
   end if

   if(fform==1   .or. &
&   fform==2   .or. &
&   fform==51  .or. &
&   fform==52  .or. &
&   fform==101 .or. &
&   fform==102       )then
!    This is the old format
     headform=22

   else

!    Format beyond 22 have a different first line, so need reading again the first line

     backspace (unitfi)
     read (unitfi) codvsn,headform,fform

     if(headform/=23 .and. &
&     headform/=34 .and. &
&     headform/=40 .and. &
&     headform/=41 .and. &
&     headform/=42 .and. &
&     headform/=44 .and. &
&     headform/=53 .and. &
&     headform/=56 .and. &
&     headform/=57         )then
       write(message,'(a,i0,3a,i0,3a)')&
&       '  The first line of the (WF, DEN or POT) file read in unit ',unitfi,' is erroneous.',ch10,&
&       '  headform is ',headform,', while it should be 23, 34, 40, 41, 42, 44, 53 or 56 or 57.',ch10,&
&       '  Action : check the correctness of your file.'
       MSG_ERROR(message)
     end if

   end if

   hdr%codvsn=codvsn
   hdr%headform=headform
!  fform is not a record of hdr_type

!  Reading the second record of the file ------------------------------------

!  Initialize the values that are not present for all versions (exception : npsp)
   hdr%nspden=1
   hdr%nspinor=1
   hdr%occopt=1
   hdr%pertcase=1
   hdr%usepaw=0
   hdr%usewvl=0
   hdr%ecut=zero
   hdr%ecutdg=zero
   hdr%ecutsm=zero
   hdr%qptn(1:3)=zero
   hdr%stmbias=zero
   hdr%tphysel=zero
   hdr%tsmear=zero

   if(headform==22)then

     read(unitfi) bantot, hdr%date, hdr%intxc, hdr%ixc, natom, hdr%ngfft(1:3),&
&     nkpt, nsppol, nsym, ntypat,&
&     acell, hdr%ecut_eff, hdr%rprimd
     npsp=ntypat

   else if(headform==23)then

!    Compared to v2.2, add nspden, nspinor, occopt
     read(unitfi) bantot, hdr%date, hdr%intxc, hdr%ixc, natom, hdr%ngfft(1:3),&
&     nkpt, hdr%nspden, hdr%nspinor, nsppol, nsym, ntypat, hdr%occopt,&
&     acell, hdr%ecut_eff, hdr%rprimd
     npsp=ntypat

   else if(headform==34)then

!    Compared to v2.3, subtract acell, and add npsp
     read(unitfi) bantot, hdr%date, hdr%intxc, hdr%ixc, natom, hdr%ngfft(1:3),&
&     nkpt, hdr%nspden, hdr%nspinor, nsppol, nsym, npsp, ntypat, hdr%occopt,&
&     hdr%ecut_eff, hdr%rprimd

   else if(headform==40)then

!    Compared to v3.4, add ecut, ecutsm, tphysel, tsmear
     read(unitfi) bantot, hdr%date, hdr%intxc, hdr%ixc, natom, hdr%ngfft(1:3),&
&     nkpt, hdr%nspden, hdr%nspinor, nsppol, nsym, npsp, ntypat, hdr%occopt,&
&     hdr%ecut, hdr%ecutsm, hdr%ecut_eff, hdr%rprimd, hdr%tphysel, hdr%tsmear

   else if(headform==41)then

!    Compared to v4.0, add pertcase and qptn(3)
     read(unitfi) bantot, hdr%date, hdr%intxc, hdr%ixc, natom, hdr%ngfft(1:3),&
&     nkpt, hdr%nspden, hdr%nspinor, nsppol, nsym, npsp, ntypat, hdr%occopt, hdr%pertcase,&
&     hdr%ecut, hdr%ecutsm, hdr%ecut_eff, hdr%qptn(1:3), hdr%rprimd, hdr%tphysel, hdr%tsmear

   else if(headform==42)then

!    Compared to v4.1, add stmbias
     read(unitfi) bantot, hdr%date, hdr%intxc, hdr%ixc, natom, hdr%ngfft(1:3),&
&     nkpt, hdr%nspden, hdr%nspinor, nsppol, nsym, npsp, ntypat, hdr%occopt, hdr%pertcase,&
&     hdr%ecut, hdr%ecutsm, hdr%ecut_eff, hdr%qptn(1:3), hdr%rprimd,&
&     hdr%stmbias, hdr%tphysel, hdr%tsmear

   else if(headform>=44 .and. headform<57)then

!    Compared to v4.2, add usepaw and ecutdg
     read(unitfi) bantot, hdr%date, hdr%intxc, hdr%ixc, natom, hdr%ngfft(1:3),&
&     nkpt, hdr%nspden, hdr%nspinor, nsppol, nsym, npsp, ntypat, hdr%occopt, hdr%pertcase,&
&     hdr%usepaw, hdr%ecut, hdr%ecutdg, hdr%ecutsm, hdr%ecut_eff, hdr%qptn(1:3), hdr%rprimd,&
&     hdr%stmbias, hdr%tphysel, hdr%tsmear

   else if(headform>=57)then

!    Compared to v4.4, add usewvl
     read(unitfi) bantot, hdr%date, hdr%intxc, hdr%ixc, natom, hdr%ngfft(1:3),&
&     nkpt, hdr%nspden, hdr%nspinor, nsppol, nsym, npsp, ntypat, hdr%occopt, hdr%pertcase,&
&     hdr%usepaw, hdr%ecut, hdr%ecutdg, hdr%ecutsm, hdr%ecut_eff, hdr%qptn(1:3), hdr%rprimd,&
&     hdr%stmbias, hdr%tphysel, hdr%tsmear, hdr%usewvl

   end if

   hdr%bantot=bantot
   hdr%natom =natom
   hdr%nkpt  =nkpt
   hdr%npsp  =npsp
   hdr%nsppol=nsppol
   hdr%nsym  =nsym
   hdr%ntypat =ntypat

   if(hdr%ecutsm>tol6 .and. headform<44 .and. .not.(fform==51.or.fform==52.or.fform==101.or.fform==102))then
     write(message,'(a,es16.6,9a)')&
&     '  The value of ecutsm is',hdr%ecutsm,', while the file has been produced prior to v4.4 .',ch10,&
&     '  The definition of the smearing function has changed, so that you are not allowed',ch10,&
&     '  to restart from a old wavefunction file. By contrast, you can restart from an old',ch10,&
&     '  potential or density file, and perform a self-consistent cycle with a new ABINIT version.',ch10,&
&     '  Action : produce a density or potential file using the old version of ABINIT, and restart from it.'
     MSG_ERROR(message)
   end if

!  Allocate all parts of hdr that need to be --------------------------------
   ABI_ALLOCATE(hdr%istwfk,(nkpt))
   ABI_ALLOCATE(hdr%kptns,(3,nkpt))
   ABI_ALLOCATE(hdr%lmn_size,(npsp))
   ABI_ALLOCATE(hdr%nband,(nkpt*nsppol))
   ABI_ALLOCATE(hdr%npwarr,(nkpt))
   ABI_ALLOCATE(hdr%occ,(bantot))
   ABI_ALLOCATE(hdr%pspcod,(npsp))
   ABI_ALLOCATE(hdr%pspdat,(npsp))
   ABI_ALLOCATE(hdr%pspso,(npsp))
   ABI_ALLOCATE(hdr%pspxc,(npsp))
   ABI_ALLOCATE(hdr%so_psp,(npsp))
   ABI_ALLOCATE(hdr%symafm,(nsym))
   ABI_ALLOCATE(hdr%symrel,(3,3,nsym))
   ABI_ALLOCATE(hdr%title,(npsp))
   ABI_ALLOCATE(hdr%tnons,(3,nsym))
   ABI_ALLOCATE(hdr%typat,(natom))
   ABI_ALLOCATE(hdr%wtk,(nkpt))
   ABI_ALLOCATE(hdr%xred,(3,natom))
   ABI_ALLOCATE(hdr%zionpsp,(npsp))
   ABI_ALLOCATE(hdr%znuclpsp,(npsp))
   ABI_ALLOCATE(hdr%znucltypat,(ntypat))
   if(hdr%usepaw==1)  then
     ABI_DATATYPE_ALLOCATE(hdr%pawrhoij,(natom))
   end if

!  Reading the third record of the file ------------------------------------

!  Initialize the values that are not present for all versions
   hdr%istwfk(:)=1
   hdr%so_psp(:)=1
   hdr%symafm(:)=1

   if(headform==22 .and. (fform==1 .or. fform==51 .or. fform==101))then

!    This is very old (pre-2.0) format !
     read(unitfi) hdr%nband(:), hdr%npwarr(:), hdr%symrel(:,:,:), &
&     hdr%typat(:), hdr%kptns(:,:), hdr%occ(:), &
&     hdr%tnons(:,:), hdr%znucltypat(:)

   else if(headform==22 .or. headform==23 .or. headform==34)then

!    Compared to pre v2.0, add istwfk
     read(unitfi) hdr%nband(:), hdr%npwarr(:), hdr%symrel(:,:,:), &
&     hdr%typat(:), hdr%istwfk(:), hdr%kptns(:,:), hdr%occ(:), &
&     hdr%tnons(:,:), hdr%znucltypat(:)

   else if(headform>=40 .and. headform < 50)then

!    Compared to pre v4.0, add so_psp and symafm, and switch istwfk

     read(unitfi)  hdr%istwfk(:), hdr%nband(:), hdr%npwarr(:), &
&     hdr%so_psp(:), hdr%symafm(:), hdr%symrel(:,:,:), &
&     hdr%typat(:), hdr%kptns(:,:), hdr%occ(:), &
&     hdr%tnons(:,:), hdr%znucltypat(:)

   else if(headform>=50)then

!    Compared to pre v5.0, add wtk
     read(unitfi)  hdr%istwfk(:), hdr%nband(:), hdr%npwarr(:), &
&     hdr%so_psp(:), hdr%symafm(:), hdr%symrel(:,:,:), &
&     hdr%typat(:), hdr%kptns(:,:), hdr%occ(:), &
&     hdr%tnons(:,:), hdr%znucltypat(:), hdr%wtk(:)

   end if

!  Reading the records with psp information ---------------------------------

!  Initialize the values that are not present for all versions
   hdr%pspso(:)=1
   hdr%lmn_size(:)=0

   if(headform==22)then

     do ipsp=1,npsp
       read(unitfi) hdr%title(ipsp), hdr%znuclpsp(ipsp), &
&       hdr%zionpsp(ipsp), hdr%pspdat(ipsp), hdr%pspcod(ipsp), &
&       hdr%pspxc(ipsp), lmax, lloc, mmax
     end do

   else if(headform==23)then

!    Compared to 2.2, add pspso
     do ipsp=1,npsp
       read(unitfi) hdr%title(ipsp), hdr%znuclpsp(ipsp), &
&       hdr%zionpsp(ipsp), hdr%pspso(ipsp), hdr%pspdat(ipsp), &
&       hdr%pspcod(ipsp), hdr%pspxc(ipsp), lmax, lloc, mmax
     end do

   else if(headform==34 .or. headform==40 .or. headform==41 &
&     .or. headform==42)then

!    Compared to 2.3, suppress lmax, lloc, mmax
     do ipsp=1,npsp
       read(unitfi) hdr%title(ipsp), hdr%znuclpsp(ipsp), &
&       hdr%zionpsp(ipsp), hdr%pspso(ipsp), hdr%pspdat(ipsp), &
&       hdr%pspcod(ipsp), hdr%pspxc(ipsp)
     end do

   else if(headform>=44)then

!    Compared to 4.2, add lmn_size
     do ipsp=1,npsp
       read(unitfi) hdr%title(ipsp), hdr%znuclpsp(ipsp), &
&       hdr%zionpsp(ipsp), hdr%pspso(ipsp), hdr%pspdat(ipsp), &
&       hdr%pspcod(ipsp), hdr%pspxc(ipsp), hdr%lmn_size(ipsp)
     end do

   end if

!  Reading the final record of the header  ---------------------------------

!  Initialize the values that are not present for all versions
   hdr%fermie=zero

   if(headform==22)then
     read(unitfi) hdr%residm, hdr%xred(:,:), hdr%etot
   else if(headform==23 .or. headform==34 .or. headform>=40)then
     read(unitfi) hdr%residm, hdr%xred(:,:), hdr%etot, hdr%fermie
   end if

   if (hdr%usepaw==1) then ! Reading the Rhoij tab if the PAW method was used.
     call rhoij_io(hdr%pawrhoij,unitfi,hdr%nsppol,hdr%nspinor,hdr%nspden,hdr%lmn_size,hdr%typat,headform,"Read")
   end if

!  -------------------------------------------------------------------------
!  Writing the header of an unformatted file
!  -------------------------------------------------------------------------

 else if(rdwr==2 .or. rdwr==6)then

!  natom,nkpt,npsp,ntypat... are not defined in this section :
!  always address them from hdr

   if(rdwr==2) rewind(unitfi)

!  Writing always use last format version
   headform=57
!$headform= HDR_LATEST_HEADFORM TODO  rationalize hdr methods and deps. so that we can use the vars. in m_header.
   write(unitfi) hdr%codvsn, headform, fform

   write(unitfi) hdr%bantot, hdr%date, hdr%intxc, hdr%ixc, &
&   hdr%natom, hdr%ngfft(1:3), hdr%nkpt, &
&   hdr%nspden, hdr%nspinor, &
&   hdr%nsppol, hdr%nsym, hdr%npsp, hdr%ntypat, hdr%occopt, hdr%pertcase,&
&   hdr%usepaw, hdr%ecut, hdr%ecutdg, hdr%ecutsm, hdr%ecut_eff, &
&   hdr%qptn, hdr%rprimd, hdr%stmbias, hdr%tphysel, hdr%tsmear, &
&   hdr%usewvl

   write(unitfi) hdr%istwfk(:), hdr%nband(:), hdr%npwarr(:),&
&   hdr%so_psp(:), hdr%symafm(:), hdr%symrel(:,:,:), &
&   hdr%typat(:), hdr%kptns(:,:), hdr%occ(:), &
&   hdr%tnons(:,:), hdr%znucltypat(:), hdr%wtk(:)

   do ipsp=1,hdr%npsp
     write(unitfi) hdr%title(ipsp), hdr%znuclpsp(ipsp), &
&     hdr%zionpsp(ipsp), hdr%pspso(ipsp), hdr%pspdat(ipsp), &
&     hdr%pspcod(ipsp), hdr%pspxc(ipsp), hdr%lmn_size(ipsp)
   end do

   write(unitfi) hdr%residm, hdr%xred(:,:), hdr%etot, hdr%fermie

   if (hdr%usepaw==1) then
     call rhoij_io(hdr%pawrhoij,unitfi,hdr%nsppol,hdr%nspinor,hdr%nspden,hdr%lmn_size,hdr%typat,headform,"Write")
   end if

!  -------------------------------------------------------------------------
!  Writing the header of a formatted file
!  -------------------------------------------------------------------------

 else if(rdwr==3 .or. rdwr==4)then

   write(unitfi, '(a)' )&
&   ' ==============================================================================='
   if(rdwr==3)write(unitfi, '(a)' ) ' ECHO of part of the ABINIT file header '
   if(rdwr==4)write(unitfi, '(a)' ) ' ECHO of the ABINIT file header '
   write(unitfi, '(a)' ) ' '
   write(unitfi, '(a)' ) ' First record :'
   write(unitfi, '(a,a6,2i5)' )  '.codvsn,headform,fform = ',hdr%codvsn, hdr%headform, fform   ! Do not worry about 22 format

   write(unitfi, '(a)' ) ' '
   write(unitfi, '(a)' ) ' Second record :'
   write(unitfi, '(a,4i6)') ' bantot,intxc,ixc,natom  =',hdr%bantot, hdr%intxc, hdr%ixc, hdr%natom
   write(unitfi, '(a,4i6)') ' ngfft(1:3),nkpt         =',hdr%ngfft(1:3), hdr%nkpt

   if(hdr%headform>=23)then
     write(unitfi, '(a,2i6)') ' nspden,nspinor          =',hdr%nspden, hdr%nspinor
   end if

   if(hdr%headform<=23)then
     write(unitfi, '(a,4i6)' ) ' nsppol,nsym,ntypat,occopt=',hdr%nsppol,hdr%nsym,hdr%ntypat,hdr%occopt
   else if(hdr%headform<=40)then
     write(unitfi, '(a,5i6)' ) ' nsppol,nsym,npsp,ntypat,occopt=',hdr%nsppol,hdr%nsym,hdr%npsp,hdr%ntypat,hdr%occopt

   else if(hdr%headform==41 .or. hdr%headform==42)then
     write(unitfi, '(a,6i6)' )&
&     ' nsppol,nsym,npsp,ntypat,occopt,pertcase=',hdr%nsppol,hdr%nsym,hdr%npsp,hdr%ntypat,hdr%occopt,hdr%pertcase
   else if(hdr%headform>=44)then
     write(unitfi, '(a,4i6)' ) ' nsppol,nsym,npsp,ntypat =',hdr%nsppol,hdr%nsym,hdr%npsp,hdr%ntypat
     write(unitfi, '(a,3i6)' ) ' occopt,pertcase,usepaw  =',hdr%occopt,hdr%pertcase,hdr%usepaw
   end if

   if(hdr%headform==40 .or. hdr%headform==41 .or. hdr%headform==42)then
     write(unitfi, '(a,2es18.10)') ' ecut,ecutsm             =',hdr%ecut, hdr%ecutsm
   else if(hdr%headform>=44)then
     write(unitfi, '(a,3es18.10)') ' ecut,ecutdg,ecutsm      =',hdr%ecut, hdr%ecutdg, hdr%ecutsm
   end if

   write(unitfi, '(a, es18.10)' ) ' ecut_eff                =',hdr%ecut_eff

   if(hdr%headform>=41)then
     write(unitfi, '(a,3es18.10)') ' qptn(1:3)               =',hdr%qptn(1:3)
   end if

   write(unitfi, '(a,3es18.10)' ) ' rprimd(1:3,1)           =',hdr%rprimd(1:3,1)
   write(unitfi, '(a,3es18.10)' ) ' rprimd(1:3,2)           =',hdr%rprimd(1:3,2)
   write(unitfi, '(a,3es18.10)' ) ' rprimd(1:3,3)           =',hdr%rprimd(1:3,3)

   if(hdr%headform==40.or.hdr%headform==41)then
     write(unitfi, '(a,2es18.10)') ' tphysel,tsmear          =',hdr%tphysel, hdr%tsmear
   else if(hdr%headform>=42)then
     write(unitfi, '(a,3es18.10)') ' stmbias,tphysel,tsmear  =',hdr%stmbias,hdr%tphysel, hdr%tsmear
   end if

   write(unitfi, '(a)' )
   if(rdwr==3)then
     write(unitfi, '(a,i3,a)' ) ' The header contain ',hdr%npsp+2,' additional records.'
   else

     write(unitfi, '(a)' ) ' Third record :'
     write(unitfi, '(a,(12i4,8x))') ' istwfk=',hdr%istwfk(:)
     write(unitfi, '(a,(12i4,8x))') ' nband =',hdr%nband(:)
     write(unitfi, '(a,(10i5,8x))') ' npwarr=',hdr%npwarr(:)

     if(hdr%headform>=40)then
       write(unitfi, '(a,(12i4,8x))') ' so_psp=',hdr%so_psp(:)
     end if

     if(hdr%headform>=40)then
       write(unitfi, '(a)') ' symafm='
       write(unitfi, '(8x,24i3,8x)') hdr%symafm(:)
     end if

     write(unitfi, '(a)' ) ' symrel='
     do isym=1,hdr%nsym/2
       write(unitfi, '(a,9i4,a,9i4)' ) '        ',hdr%symrel(:,:,2*isym-1),'  ',hdr%symrel(:,:,2*isym)
     end do
     if(2*(hdr%nsym/2)/=hdr%nsym)write(unitfi, '(a,9i4)' ) '        ',hdr%symrel(:,:,hdr%nsym)

     write(unitfi, '(a,(12i4,8x))') ' type  =',hdr%typat(:)
     write(unitfi, '(a)' ) ' kptns =                 (max 50 k-points will be written)'
     do ikpt=1,min(hdr%nkpt,50)
       write(unitfi, '(a,3es16.6)' ) '        ',hdr%kptns(:,ikpt)
     end do
     write(unitfi, '(a)' ) ' wtk ='
     do ikpt=1,hdr%nkpt,10
       write(unitfi, '(a,10f6.2)' ) '        ',hdr%wtk(ikpt:min(hdr%nkpt,ikpt + 10 - 1))
     end do
     write(unitfi, '(a)' ) '   occ ='
     do ii=1,hdr%bantot,10
       write(unitfi, '(a,10f6.2)') '        ',hdr%occ(ii:min(hdr%bantot,ii+10-1))
     end do
     write(unitfi, '(a)' ) ' tnons ='
     do isym=1,hdr%nsym/2
       write(unitfi, '(a,3f10.6,a,3f10.6)' ) '        ',hdr%tnons(:,2*isym-1),'  ',hdr%tnons(:,2*isym)
     end do
     if(2*(hdr%nsym/2)/=hdr%nsym)write(unitfi, '(a,3f10.6)' ) '        ',hdr%tnons(:,hdr%nsym)
     write(unitfi, '(a,(10f6.2,8x))') '  znucl=',hdr%znucltypat(:)
     write(unitfi,'(a)')

     write(unitfi, '(a)' ) ' Pseudopotential info :'
     do ipsp=1,hdr%npsp
       write(unitfi,'(a,a)' ) ' title=',trim(hdr%title(ipsp))
       write(unitfi,'(a,f6.2,a,f6.2,a,i3,a,i6,a,i3,a,i3)' ) &
&       '  znuclpsp=',hdr%znuclpsp(ipsp),    ', zionpsp=',  hdr%zionpsp(ipsp),&
&       ', pspso=' , hdr%pspso(ipsp),  ', pspdat=',hdr%pspdat(ipsp),          &
&       ', pspcod=', hdr%pspcod(ipsp), ', pspxc=', hdr%pspxc(ipsp)
       if(hdr%headform>=44)then
         if(hdr%usepaw==1)then
           write(unitfi,'(a,i3)' ) '  lmn_size=', hdr%lmn_size(ipsp)
         else
           write(unitfi,'(a,i3)' ) '  lmnmax  =', hdr%lmn_size(ipsp)
         end if
       end if

     end do

     write(unitfi, '(a)' ) ' '
     write(unitfi, '(a)' ) ' Last record :'
     write(unitfi, '(a,es16.6,es22.12,es16.6)' )' residm,etot,fermie=',hdr%residm, hdr%etot, hdr%fermie
     write(unitfi, '(a)' ) ' xred ='
     do iatom=1,hdr%natom
       write(unitfi, '(a,3es16.6)' ) '        ',hdr%xred(:,iatom)
     end do

     if(hdr%usepaw==1)then
       call rhoij_io(hdr%pawrhoij,unitfi,hdr%nsppol,hdr%nspinor,hdr%nspden,hdr%lmn_size,hdr%typat,headform,"Echo")
     end if

     if(rdwr==3)write(unitfi, '(a)' ) ' End the ECHO of part of the ABINIT file header '
     if(rdwr==4)write(unitfi, '(a)' ) ' End the ECHO of the ABINIT file header '
     write(unitfi,'(a)')' ==============================================================================='

   end if ! rdwr is 3 or 4
   call flush_unit(unitfi)
 end if ! choice read/write/echo

 DBG_EXIT("COLL")

end subroutine hdr_io_int
!!***

!----------------------------------------------------------------------

!!****f* m_header/hdr_io_etsf
!! NAME
!! hdr_io_etsf
!!
!! FUNCTION
!! This subroutine deals with the I/O of the hdr_type
!! structured variables (read/write/echo).
!! It handles variables according to the ETSF format, whenever
!! possible and uses new variables when not available in the ETSF
!! format.
!! According to the value of rdwr, it reads the header
!! of a file, writes it, or echo the value of the structured
!! variable to a file.
!! Note that, when reading, different records of hdr
!! are allocated here, according to the values of the
!! read variables. Records of hdr should be deallocated
!! correctly by a call to hdr_clean when hdr is not used anymore.
!!
!! INPUTS
!!  rdwr= if 1, read the hdr structured variable from the header of the netCDF file,
!!        if 2, write the header to unformatted netCDF file
!!        if 3, echo part of the header to formatted file (records 1 and 2)
!!        if 4, echo the header to formatted file
!!        if 5, read the hdr without rewinding (unformatted), identical to 1 for netCDF
!!        if 6, read the hdr without rewinding (unformatted), identical to 2 for netCDF
!!  unitwff=the unit of the open NetCDF file.
!!
!! OUTPUT
!!  (see side effects)
!!
!! SIDE EFFECTS
!!  The following variables are both input or output :
!!  fform=kind of the array in the file
!!   if rdwr=1,5 : will be output ; if the reading fail, return fform=0
!!   if rdwr=2,3,4,6 : should be input, will be written or echo to file
!!  hdr <type(hdr_type)>=the header structured variable
!!   if rdwr=1,5 : will be output
!!   if rdwr=2,3,4,6 : should be input, will be written or echo to file
!!
!! NOTES
!!
!! PARENTS
!!      cut3d,inwffil,ioarr,kss2wfk,m_ebands,m_io_kss,m_io_screening
!!      m_sigma_results,m_wfs,outwf,pawmkaewf,wfk_read_ene
!!
!! CHILDREN
!!      mpi_barrier,mpi_bcast,rhoij_alloc
!!
!! SOURCE

subroutine hdr_io_etsf(fform,hdr,rdwr,unitwff)

 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
#undef ABI_FUNC
#define ABI_FUNC 'hdr_io_etsf'
 use interfaces_14_hidewrite
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: rdwr,unitwff
 integer,intent(inout) :: fform
 type(hdr_type),intent(inout) :: hdr

!Local variables-------------------------------
#if defined HAVE_TRIO_ETSF_IO
 type(etsf_dims) :: dims
 type(etsf_kpoints) :: kpoints
 type(etsf_basisdata) :: basisdata
 type(etsf_geometry) :: geometry
 type(etsf_electrons) :: electrons
 type(etsf_io_low_error) :: error_data
 logical :: lstat
 character(len = 500)  :: message
 integer :: rhoijdim1, rhoijdim2, nresolution
 integer :: headform, iatom, itypat
 character(len=etsf_charlen), target :: basis_set
 real(dp), target :: ecut, fermie
 real(dp), target :: rprimd(3, 3)
!temp variables
! integer :: cplex, ilmn, irhoij, ispden, lmn2_size, nselect
! real(dp), allocatable :: rhoij(:,:,:)
#endif

! *************************************************************************

#if defined HAVE_TRIO_ETSF_IO
 write(message, '(A,I0)' ) ' hdr_io_etsf: accessing ABINIT specific data from unit ', unitwff
 call wrtout(std_out, message, 'COLL')

 if(rdwr==1 .or. rdwr==5)then
!  We switch off from define mode.
   call etsf_io_low_set_write_mode(unitwff, lstat, error_data = error_data)
   ETSF_CHECK_ERROR(lstat,error_data)

!  In case the file is just ETSF valid, we ignore the missing variables
!  and we use defualt values.
   hdr%codvsn   = "ETSF  "
   fform        = 1
   headform     = 57

!  First, we read the declaration of code, fform ...
!  We ignore errors, assuming that the file is at least ETSF valid.
   call etsf_io_low_read_var(unitwff, "codvsn", hdr%codvsn, 6, lstat, error_data = error_data)
   if (lstat) then ! We pad the returned string with " " instead of "\0"
     call strip(hdr%codvsn)
   end if

   call etsf_io_low_read_var(unitwff, "fform", fform, lstat, error_data = error_data)
   ETSF_CHECK_ERROR(lstat,error_data)

   call etsf_io_low_read_var(unitwff, "headform", hdr%headform, lstat, error_data = error_data)
   ETSF_CHECK_ERROR(lstat,error_data)

   if (headform <= 42) then
     write(message,'(a,i0,3a)')&
&     '  headform is ',headform,', while it should be > 42.',ch10,&
&     '  Action : check the correctness of your file.'
     MSG_ERROR(message)
   end if

!  Then, read dimensions handled by ETSF
   call etsf_io_dims_get(unitwff, dims, lstat, error_data)
   ETSF_CHECK_ERROR(lstat,error_data)

!  Copy dimensions to hdr structure
!  FIXME: don't handle k_dependent = 1
   hdr%bantot   = dims%max_number_of_states * dims%number_of_kpoints * dims%number_of_spins
   hdr%natom    = dims%number_of_atoms
   hdr%nkpt     = dims%number_of_kpoints
   hdr%nspden   = dims%number_of_components
   hdr%nspinor  = dims%number_of_spinor_components
   hdr%nsppol   = dims%number_of_spins
   hdr%nsym     = dims%number_of_symmetry_operations
   hdr%ntypat   = dims%number_of_atom_species
   hdr%ngfft(1) = dims%number_of_grid_points_vector1
   hdr%ngfft(2) = dims%number_of_grid_points_vector2
   hdr%ngfft(3) = dims%number_of_grid_points_vector3

!  We read other dimensions, not handled by ETSF format.
!  In case the file is just ETSF valid, we ignore the missing dimensions and we use default values.

   hdr%npsp    = hdr%ntypat
   call etsf_io_low_read_dim(unitwff, "npsp", hdr%npsp, lstat, error_data = error_data)
   ETSF_WARN(lstat,error_data)

   rhoijdim1   = 1
   call etsf_io_low_read_dim(unitwff, "rhoijdim1", rhoijdim1, lstat, error_data = error_data)
   ETSF_WARN(lstat,error_data)

   rhoijdim2   = hdr%nspden
   call etsf_io_low_read_dim(unitwff, "rhoijdim2", rhoijdim2, lstat, error_data = error_data)
   ETSF_WARN(lstat,error_data)

   hdr%usepaw  = 0
   call etsf_io_low_read_var(unitwff, "usepaw", hdr%usepaw, lstat, error_data = error_data)
   ETSF_WARN(lstat,error_data)

   hdr%usewvl  = 0
   call etsf_io_low_read_var(unitwff, "usewvl", hdr%usewvl, lstat, error_data = error_data)
   ETSF_WARN(lstat,error_data)

   nresolution=0
   if (hdr%usewvl == 1) then
!    This value must be 2...
     call etsf_io_low_read_dim(unitwff, "number_of_wavelet_resolutions",nresolution, lstat, error_data = error_data)
     ETSF_WARN(lstat,error_data)

!    We set the right ngfft, adding the padding space for wavelets.
     hdr%ngfft = hdr%ngfft + 31
   end if

!  Allocate all parts of hdr that need to be
   ABI_ALLOCATE(hdr%istwfk,(hdr%nkpt))
   ABI_ALLOCATE(hdr%lmn_size,(hdr%npsp))
   ABI_ALLOCATE(hdr%nband,(hdr%nkpt*hdr%nsppol))
   ABI_ALLOCATE(hdr%npwarr,(hdr%nkpt))
   ABI_ALLOCATE(hdr%pspcod,(hdr%npsp))
   ABI_ALLOCATE(hdr%pspdat,(hdr%npsp))
   ABI_ALLOCATE(hdr%pspso,(hdr%npsp))
   ABI_ALLOCATE(hdr%pspxc,(hdr%npsp))
   ABI_ALLOCATE(hdr%so_psp,(hdr%npsp))
   ABI_ALLOCATE(hdr%symafm,(hdr%nsym))
   ABI_ALLOCATE(hdr%symrel,(3,3,hdr%nsym))
   ABI_ALLOCATE(hdr%typat,(hdr%natom))
   ABI_ALLOCATE(hdr%kptns,(3,hdr%nkpt))
   ABI_ALLOCATE(hdr%occ,(hdr%bantot))
   ABI_ALLOCATE(hdr%tnons,(3,hdr%nsym))
   ABI_ALLOCATE(hdr%wtk,(hdr%nkpt))
   ABI_ALLOCATE(hdr%xred,(3,hdr%natom))
   ABI_ALLOCATE(hdr%znuclpsp,(hdr%npsp))
   ABI_ALLOCATE(hdr%znucltypat,(hdr%ntypat))
   ABI_ALLOCATE(hdr%zionpsp,(hdr%npsp))
   ABI_ALLOCATE(hdr%title,(hdr%npsp))
   if(hdr%usepaw==1)  then
     ABI_DATATYPE_ALLOCATE(hdr%pawrhoij,(hdr%natom))
   end if

!  We get then all variables included in ETSF
   if (hdr%usewvl == 0) then
     basisdata%kinetic_energy_cutoff => ecut
     basisdata%number_of_coefficients => hdr%npwarr
     call etsf_io_basisdata_get(unitwff, basisdata, lstat, error_data)
     ETSF_CHECK_ERROR(lstat,error_data)
   else
     call etsf_io_low_read_var(unitwff, "number_of_wavelets", hdr%nwvlarr, lstat, error_data = error_data)
     ETSF_CHECK_ERROR(lstat,error_data)
   end if

   electrons%fermi_energy => fermie
   electrons%number_of_states%data1D => hdr%nband
   electrons%occupations%data1D => hdr%occ

   call etsf_io_electrons_get(unitwff, electrons, lstat, error_data)
   ETSF_CHECK_ERROR(lstat,error_data)

   geometry%primitive_vectors => rprimd
   geometry%reduced_symmetry_matrices => hdr%symrel
   geometry%atom_species => hdr%typat
   geometry%reduced_symmetry_translations => hdr%tnons
   geometry%reduced_atom_positions => hdr%xred
   geometry%atomic_numbers => hdr%znucltypat

   if (hdr%npsp == hdr%ntypat) then
     geometry%valence_charges => hdr%zionpsp
   end if

   call etsf_io_geometry_get(unitwff, geometry, lstat, error_data)
   ETSF_CHECK_ERROR(lstat,error_data)

   kpoints%reduced_coordinates_of_kpoints => hdr%kptns
   kpoints%kpoint_weights => hdr%wtk

   call etsf_io_kpoints_get(unitwff, kpoints, lstat, error_data)
   ETSF_CHECK_ERROR(lstat,error_data)

   hdr%fermie = fermie
   hdr%ecut   = ecut
   hdr%rprimd = rprimd
   hdr%znuclpsp(1:hdr%npsp) = hdr%znucltypat(1:hdr%npsp)

!  We get all other variables
!  In case the file is just ETSF valid, we ignore the missing variables and we use default values.

   hdr%date = 0
   call etsf_io_low_read_var(unitwff, "date", hdr%date, lstat, error_data = error_data)
   ETSF_WARN(lstat,error_data)

   hdr%ecut_eff = hdr%ecut
   call etsf_io_low_read_var(unitwff, "ecut_eff", hdr%ecut_eff, lstat, error_data = error_data)
   ETSF_WARN(lstat,error_data)

   hdr%ecutsm = zero
   call etsf_io_low_read_var(unitwff, "ecutsm", hdr%ecutsm, lstat, error_data = error_data)
   ETSF_WARN(lstat,error_data)

   hdr%etot = zero
   call etsf_io_low_read_var(unitwff, "etot", hdr%etot, lstat, error_data = error_data)
   ETSF_WARN(lstat,error_data)

   hdr%intxc = 0
   call etsf_io_low_read_var(unitwff, "intxc", hdr%intxc, lstat, error_data = error_data)
   ETSF_WARN(lstat,error_data)

   hdr%ixc = 1
   call etsf_io_low_read_var(unitwff, "ixc", hdr%ixc, lstat, error_data = error_data)
   ETSF_WARN(lstat,error_data)

   hdr%occopt = 1
   call etsf_io_low_read_var(unitwff, "occopt", hdr%occopt, lstat, error_data = error_data)
   ETSF_WARN(lstat,error_data)

   hdr%pertcase = 0
   call etsf_io_low_read_var(unitwff, "pertcase", hdr%pertcase, lstat, error_data = error_data)
   ETSF_WARN(lstat,error_data)

   hdr%qptn(:) = 0
   call etsf_io_low_read_var(unitwff, "qptn", hdr%qptn, lstat, error_data = error_data)
   ETSF_WARN(lstat,error_data)

   hdr%residm = zero
   call etsf_io_low_read_var(unitwff, "residm", hdr%residm, lstat, error_data = error_data)
   ETSF_WARN(lstat,error_data)

   hdr%stmbias = zero
   call etsf_io_low_read_var(unitwff, "stmbias", hdr%stmbias, lstat, error_data = error_data)
   ETSF_WARN(lstat,error_data)

   hdr%tphysel  = zero
   call etsf_io_low_read_var(unitwff, "tphysel", hdr%tphysel, lstat, error_data = error_data)
   ETSF_WARN(lstat,error_data)

   hdr%tsmear = zero
   call etsf_io_low_read_var(unitwff, "tsmear", hdr%tsmear, lstat, error_data = error_data)
   ETSF_WARN(lstat,error_data)

   hdr%ecutdg = hdr%ecut
   call etsf_io_low_read_var(unitwff, "ecutdg", hdr%ecutdg, lstat, error_data = error_data)
   ETSF_WARN(lstat,error_data)

!  test for old wavefunction style
   if(hdr%ecutsm>tol6 .and. headform<44 .and. &
&   .not.(fform==51.or.fform==52.or.fform==101.or.fform==102)) then
     write(message,'(a,es16.6,13a)' )&
&     '  The value of ecutsm is',hdr%ecutsm, &
&     ', while the file has been produced prior to v4.4 .',ch10,&
&     '  The definition of the smearing function has changed,', &
&     ' so that you are not allowed',ch10,&
&     '  to restart from a old wavefunction file. By contrast,', &
&     ' you can restart from an old',ch10,&
&     '  potential or density file, and perform a self-consistent', &
&     ' cycle with a new ABINIT version.',ch10,&
&     '  Action : produce a density or potential file using the old', &
&     ' version of ABINIT, and restart from it.'
     MSG_ERROR(message)
   end if

!  Multidimensional variables.
!  The case of istwfk is always 1, since ETSF don't use the time reversal symetry.
   hdr%istwfk(:)   = 1

   hdr%pspcod(:) = 0
   call etsf_io_low_read_var(unitwff, "pspcod", hdr%pspcod, lstat, error_data = error_data)
   ETSF_WARN(lstat,error_data)

   hdr%pspdat(:) = 0
   call etsf_io_low_read_var(unitwff, "pspdat", hdr%pspdat, lstat, error_data = error_data)
   ETSF_WARN(lstat,error_data)

   hdr%pspso(:) = 0
   call etsf_io_low_read_var(unitwff, "pspso", hdr%pspso, lstat, error_data = error_data)
   ETSF_WARN(lstat,error_data)

   hdr%pspxc(:) = 0
   call etsf_io_low_read_var(unitwff, "pspxc", hdr%pspxc, lstat, error_data = error_data)
   ETSF_WARN(lstat,error_data)

   hdr%so_psp(:) = 1
   call etsf_io_low_read_var(unitwff, "so_psp", hdr%so_psp, lstat, error_data = error_data)
   ETSF_WARN(lstat,error_data)

   hdr%symafm(:) = 1
   call etsf_io_low_read_var(unitwff, "symafm", hdr%symafm, lstat, error_data = error_data)
   ETSF_WARN(lstat,error_data)

   hdr%title(:) = ""
   call etsf_io_low_read_var(unitwff, "title", hdr%title, 132, lstat, error_data = error_data)
   if (lstat) then ! Pad the returned string with " " instead of "\0"
     do itypat = 1, size(hdr%title), 1
       call strip(hdr%title(itypat))
     end do
   end if

   call etsf_io_low_read_var(unitwff, "zionpsp", hdr%zionpsp, lstat, error_data = error_data)
   ETSF_WARN(lstat,error_data)

   call etsf_io_low_read_var(unitwff, "znuclpsp", hdr%znuclpsp, lstat, error_data = error_data)
   ETSF_WARN(lstat,error_data)

   hdr%lmn_size = 1
   if (headform>=44) then ! Compared to 4.2, add lmn_size and
     call etsf_io_low_read_var(unitwff, "lmn_size", hdr%lmn_size, lstat, error_data = error_data)
!    DC: remove the PAW specific variables, they are not in accordance
!    with 59_io_mpi/hdr_io.F90. This part should be rewritten totally.
     if (hdr%usepaw==1) then
       write(message, '(9a)' )&
&       '  The support for the internal variables of PAW are not yet',ch10,&
&       '  available with ETSF output. Restarting calculation from this',ch10,&
&       '  will not be possible.',ch10,&
&       '  Action : produce a density or potential file using the old',ch10,&
&       '  binary format of ABINIT, and restart from it.'
       MSG_ERROR(message)
       do iatom=1,hdr%natom
         nullify(hdr%pawrhoij(iatom)%rhoijselect)
         nullify(hdr%pawrhoij(iatom)%rhoijp)
         hdr%pawrhoij(iatom)%ngrhoij = 0
         hdr%pawrhoij(iatom)%lmnmix_sz = 0
         hdr%pawrhoij(iatom)%use_rhoij_ = 0
         hdr%pawrhoij(iatom)%use_rhoijres = 0
       end do
!      !!    call etsf_io_low_read_var(unitwff, "rhoijdim1", rhoijdim1, lstat, error_data = error_data)
!      !!    allocate (rhoij(rhoijdim1,hdr%nspden,hdr%natom))
!      !!    call etsf_io_low_read_var(unitwff, "rhoij", rhoij, lstat, error_data = error_data)
!      !!    if (.not.lstat) goto 1000
!      !!
!      !!    cplex=1;if (rhoijdim1/=hdr%natom) cplex=2
!      !!    call rhoij_alloc(cplex,hdr%nspden,hdr%nspinor,hdr%nsppol,hdr%pawrhoij,hdr%typat,lmnsize=hdr%lmn_size)
!      !!    do iatom=1,hdr%natom
!      !!     itypat=hdr%typat(iatom)
!      !!     lmn2_size=hdr%lmn_size(itypat)*(hdr%lmn_size(itypat)+1)/2
!      !!     nselect=0
!      !!     if (cplex==1) then
!      !!      do ilmn=1,lmn2_size
!      !!       if (any(abs(rhoij(ilmn,:,iatom))>tol10)) then
!      !!        nselect=nselect+1
!      !!        hdr%pawrhoij(iatom)%rhoijselect(nselect)=ilmn
!      !!        do ispden=1,hdr%nspden
!      !!         hdr%pawrhoij(iatom)%rhoijp(nselect,ispden)=rhoij(ilmn,ispden,iatom)
!      !!        end do
!      !!       end if
!      !!      end do
!      !!     else
!      !!      do ilmn=1,lmn2_size
!      !!       if (any(abs(rhoij(2*ilmn-1:2*ilmn,:,iatom))>tol10)) then
!      !!        nselect=nselect+1
!      !!        hdr%pawrhoij(iatom)%rhoijselect(nselect)=ilmn
!      !!        hdr%pawrhoij(iatom)%rhoijp(2*nselect-1,ispden)=rhoij(2*ilmn-1,ispden,iatom)
!      !!        hdr%pawrhoij(iatom)%rhoijp(2*nselect  ,ispden)=rhoij(2*ilmn  ,ispden,iatom)
!      !!       end if
!      !!      end do
!      !!     end if
!      !!     if (nselect<lmn2_size) then
!      !!      hdr%pawrhoij(iatom)%rhoijselect(nselect+1:lmn2_size)=0
!      !!      do ispden=1,hdr%nspden
!      !!       hdr%pawrhoij(iatom)%rhoijp(cplex*nselect+1:cplex*lmn2_size,ispden)=zero
!      !!      end do
!      !!     end if
!      !!     hdr%pawrhoij(iatom)%nrhoijsel=nselect
!      !!    end do
!      !!    deallocate(rhoij)
     end if
   end if

!  BigDFT private variables.
!  First implementation, we assume that the number of wavelet resolutions
!  is 2. Latter, we may add this value to hdr.

   lstat = .true.

!  -------------------------------------------------------------------------
!  Writing the header of an unformatted file
!  -------------------------------------------------------------------------
 else if(rdwr==2 .or. rdwr==6)then
!  We switch to write mode.
   call etsf_io_low_set_write_mode(unitwff, lstat, error_data = error_data)
   ETSF_CHECK_ERROR(lstat,error_data)

!  Associate and write values to ETSF groups.
   if (hdr%usewvl == 0) then  ! Plane wave case.
     ecut = hdr%ecut
     write(basis_set, "(A)") "plane_waves"
     basisdata%basis_set => basis_set
     basisdata%kinetic_energy_cutoff => ecut
     basisdata%number_of_coefficients => hdr%npwarr
   else  ! Wavelet case.
     write(basis_set, "(A)") "daubechies_wavelets"
     basisdata%basis_set => basis_set
!    Required variable than should enter the standard.
     call etsf_io_low_write_var(unitwff, "number_of_wavelets", hdr%nwvlarr, lstat, error_data = error_data)
     ETSF_CHECK_ERROR(lstat,error_data)
   end if

   call etsf_io_basisdata_put(unitwff, basisdata, lstat, error_data)
   ETSF_CHECK_ERROR(lstat,error_data)

   fermie = hdr%fermie
   electrons%fermi_energy => fermie
   electrons%number_of_states%data1D => hdr%nband
   electrons%occupations%data1D => hdr%occ

   call etsf_io_electrons_put(unitwff, electrons, lstat, error_data)
   ETSF_CHECK_ERROR(lstat,error_data)

   rprimd = hdr%rprimd
   geometry%primitive_vectors => rprimd
   geometry%reduced_symmetry_matrices => hdr%symrel
   geometry%atom_species => hdr%typat
   geometry%reduced_symmetry_translations => hdr%tnons
   geometry%reduced_atom_positions => hdr%xred
   geometry%atomic_numbers => hdr%znucltypat
   if (hdr%npsp == hdr%ntypat) then
     geometry%valence_charges => hdr%zionpsp
   end if

   call etsf_io_geometry_put(unitwff, geometry, lstat, error_data)
   ETSF_CHECK_ERROR(lstat,error_data)

   kpoints%reduced_coordinates_of_kpoints => hdr%kptns
   kpoints%kpoint_weights => hdr%wtk

   call etsf_io_kpoints_put(unitwff, kpoints, lstat, error_data)
   ETSF_CHECK_ERROR(lstat,error_data)

!  Write non-ETSF variables.
   call etsf_io_low_write_var(unitwff, "date", hdr%date, lstat, error_data = error_data)
   ETSF_CHECK_ERROR(lstat,error_data)

   call etsf_io_low_write_var(unitwff, "codvsn", hdr%codvsn, 6, lstat, error_data = error_data)
   ETSF_CHECK_ERROR(lstat,error_data)

   call etsf_io_low_write_var(unitwff, "ecut_eff", hdr%ecut_eff, lstat, error_data = error_data)
   ETSF_CHECK_ERROR(lstat,error_data)

   call etsf_io_low_write_var(unitwff, "ecutsm", hdr%ecutsm, lstat, error_data = error_data)
   ETSF_CHECK_ERROR(lstat,error_data)

   call etsf_io_low_write_var(unitwff, "etot", hdr%etot, lstat, error_data = error_data)
   ETSF_CHECK_ERROR(lstat,error_data)

   call etsf_io_low_write_var(unitwff, "headform", 44, lstat, error_data = error_data)
   ETSF_CHECK_ERROR(lstat,error_data)

   call etsf_io_low_write_var(unitwff, "fform", fform, lstat, error_data = error_data)
   ETSF_CHECK_ERROR(lstat,error_data)

   call etsf_io_low_write_var(unitwff, "intxc", hdr%intxc, lstat, error_data = error_data)
   ETSF_CHECK_ERROR(lstat,error_data)

   call etsf_io_low_write_var(unitwff, "ixc", hdr%ixc, lstat, error_data = error_data)
   ETSF_CHECK_ERROR(lstat,error_data)

   call etsf_io_low_write_var(unitwff, "occopt", hdr%occopt, lstat, error_data = error_data)
   ETSF_CHECK_ERROR(lstat,error_data)

   call etsf_io_low_write_var(unitwff, "pertcase", hdr%pertcase, lstat, error_data = error_data)
   ETSF_CHECK_ERROR(lstat,error_data)

   call etsf_io_low_write_var(unitwff, "residm", hdr%residm, lstat, error_data = error_data)
   ETSF_CHECK_ERROR(lstat,error_data)

   call etsf_io_low_write_var(unitwff, "stmbias", hdr%stmbias, lstat, error_data = error_data)
   ETSF_CHECK_ERROR(lstat,error_data)

   call etsf_io_low_write_var(unitwff, "tphysel", hdr%tphysel, lstat, error_data = error_data)
   ETSF_CHECK_ERROR(lstat,error_data)

   call etsf_io_low_write_var(unitwff, "tsmear", hdr%tsmear, lstat, error_data = error_data)
   ETSF_CHECK_ERROR(lstat,error_data)

!  Version 44 add usepaw ecutdg
   call etsf_io_low_write_var(unitwff, "ecutdg", hdr%ecutdg, lstat, error_data = error_data)
   ETSF_CHECK_ERROR(lstat,error_data)

   call etsf_io_low_write_var(unitwff, "usepaw", hdr%usepaw, lstat, error_data = error_data)
   ETSF_CHECK_ERROR(lstat,error_data)

!  Array variables.
   call etsf_io_low_write_var(unitwff, "pspcod", hdr%pspcod, lstat, error_data = error_data)
   ETSF_CHECK_ERROR(lstat,error_data)

   call etsf_io_low_write_var(unitwff, "pspdat", hdr%pspdat, lstat, error_data = error_data)
   ETSF_CHECK_ERROR(lstat,error_data)

   call etsf_io_low_write_var(unitwff, "pspso", hdr%pspso, lstat, error_data = error_data)
   ETSF_CHECK_ERROR(lstat,error_data)

   call etsf_io_low_write_var(unitwff, "pspxc", hdr%pspxc, lstat, error_data = error_data)
   ETSF_CHECK_ERROR(lstat,error_data)

   call etsf_io_low_write_var(unitwff, "qptn", hdr%qptn, lstat, error_data = error_data)
   ETSF_CHECK_ERROR(lstat,error_data)

   call etsf_io_low_write_var(unitwff, "so_psp", hdr%so_psp, lstat, error_data = error_data)
   ETSF_CHECK_ERROR(lstat,error_data)

   call etsf_io_low_write_var(unitwff, "symafm", hdr%symafm, lstat, error_data = error_data)
   ETSF_CHECK_ERROR(lstat,error_data)

   call etsf_io_low_write_var(unitwff, "title", hdr%title, 132, lstat, error_data = error_data)
   ETSF_CHECK_ERROR(lstat,error_data)

   call etsf_io_low_write_var(unitwff, "znuclpsp", hdr%znuclpsp, lstat, error_data = error_data)
   ETSF_CHECK_ERROR(lstat,error_data)

   if (hdr%npsp /= hdr%ntypat) then
     call etsf_io_low_write_var(unitwff, "zionpsp", hdr%zionpsp, lstat, error_data = error_data)
     ETSF_CHECK_ERROR(lstat,error_data)
   end if

!  Version 44 add lmn_size and rhoij
   call etsf_io_low_write_var(unitwff, "lmn_size", hdr%lmn_size, lstat, error_data = error_data)
   ETSF_CHECK_ERROR(lstat,error_data)

!  DC: remove the PAW specific variables, they are not in accordance
!  with 59_io_mpi/hdr_io.F90. This part should be rewritten totally.
   if (hdr%usepaw == 1) then
     write(message, '(12a)' ) ch10,&
&     ' hdr_io_etsf : WARNING -',ch10,&
&     '  The support for the internal variables of PAW are not yet',ch10,&
&     '  available with ETSF output. Restarting calculation from this',ch10,&
&     '  will not be possible.',ch10,&
&     '  Action : produce a density or potential file using the old',ch10,&
&     '  binary format of ABINIT, and restart from it.'
     call wrtout(std_out, message, 'COLL')
!    !!   rhoijdim1 = maxval(hdr%lmn_size)
!    !!   rhoijdim1 = hdr%pawrhoij(1)%cplex*rhoijdim1*(rhoijdim1+1)/2
!    !!   call etsf_io_low_write_var(unitwff, "rhoijdim1", rhoijdim1, lstat, error_data = error_data)
!    !!   allocate (rhoij(rhoijdim1,hdr%nspden,hdr%natom))
!    !!   do iatom=1,hdr%natom
!    !!    itypat=hdr%typat(iatom)
!    !!    lmn2_size = hdr%lmn_size(itypat)*(hdr%lmn_size(itypat)+1)/2
!    !!    cplex=hdr%pawrhoij(iatom)%cplex
!    !!    do ispden=1,hdr%nspden
!    !!     rhoij(1:cplex*lmn2_size,ispden,iatom)=zero
!    !!     if (cplex==1) then
!    !!      do irhoij=1,hdr%pawrhoij(iatom)%nrhoijsel
!    !!       ilmn=hdr%pawrhoij(iatom)%rhoijselect(irhoij)
!    !!       rhoij(ilmn,ispden,iatom)=hdr%pawrhoij(iatom)%rhoijp(irhoij,ispden)
!    !!      end do
!    !!     else
!    !!      do irhoij=1,hdr%pawrhoij(iatom)%nrhoijsel
!    !!       ilmn=hdr%pawrhoij(iatom)%rhoijselect(irhoij)
!    !!       rhoij(2*ilmn-1,ispden,iatom)=hdr%pawrhoij(iatom)%rhoijp(2*irhoij-1,ispden)
!    !!       rhoij(2*ilmn  ,ispden,iatom)=hdr%pawrhoij(iatom)%rhoijp(2*irhoij  ,ispden)
!    !!      end do
!    !!     end if
!    !!    end do
!    !!   end do
!    !!   call etsf_io_low_write_var(unitwff, "rhoij", rhoij, lstat, error_data = error_data)
!    !!   if (.not.lstat) goto 1000
!    !!   deallocate (rhoij)
   end if
!  BigDFT variables.
   call etsf_io_low_write_var(unitwff, "usewvl", hdr%usewvl, lstat, error_data = error_data)
   ETSF_CHECK_ERROR(lstat,error_data)

 else if(rdwr==3 .or. rdwr==4)then

   call hdr_io_int(fform, hdr, rdwr, unitwff)
   lstat = .true.

 end if ! choice read/write/echo
#endif

 return
 fform=0 ; return   ! This is to allow treatment of old epsm1 format

end subroutine hdr_io_etsf
!!***

!----------------------------------------------------------------------

!!****f* m_header/hdr_skip_int
!! NAME
!! hdr_skip_int
!!
!! FUNCTION
!! Skip wavefunction or density file header, after having rewound the file.
!! Two instances of the hdr_skip routines are defined :
!!  hdr_skip_int to which only the unit number is given
!!  hdr_skip_wfftype to which a wffil datatype is given
!!
!! INPUTS
!!  unit = number of unit to be read
!!
!! OUTPUT
!!  ierr = error code returned by the MPI calls
!!
!! SIDE EFFECTS
!!
!! NOTES
!! No checking performed, since hdr_skip is assumed to be used only
!! on temporary wavefunction files.
!! This initialize further reading and checking by rwwf
!!
!! PARENTS
!!
!! CHILDREN
!!      mpi_barrier,mpi_bcast,rhoij_alloc
!!
!! SOURCE

subroutine hdr_skip_int(unitfi,ierr)

 use defs_basis
 use m_profiling
 use m_wffile

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
#undef ABI_FUNC
#define ABI_FUNC 'hdr_skip_int'
!End of the abilint section

 implicit none

!Arguments ------------------------------------
 integer,intent(in) :: unitfi
 integer,intent(out) :: ierr

!Local variables-------------------------------
 type(wffile_type) :: wff

! *************************************************************************

!Use default values for wff
 wff%unwff=unitfi
 wff%accesswff=IO_MODE_FORTRAN
 wff%me=0
 wff%master=0
!Then, transmit to hdr_skip_wfftype
 call hdr_skip_wfftype(wff,ierr)

end subroutine hdr_skip_int
!!***

!----------------------------------------------------------------------

!!****f* m_header/hdr_skip_wfftype
!! NAME
!! hdr_skip_wfftype
!!
!! FUNCTION
!! Skip wavefunction or density file header, after having rewound the file.
!! Two instances of the hdr_skip routines are defined :
!!  hdr_skip_int to which only the unit number is given
!!  hdr_skip_wfftype to which a wffil datatype is given
!!
!! INPUTS
!!  unit = number of unit to be read
!!
!! OUTPUT
!!  ierr = error code returned by the MPI calls
!!
!! NOTES
!! No checking performed, since hdr_skip is assumed to be used only
!! on temporary wavefunction files.
!! This initialize further reading and checking by rwwf
!!
!! PARENTS
!!      m_header
!!
!! CHILDREN
!!      mpi_barrier,mpi_bcast,rhoij_alloc
!!
!! SOURCE

subroutine hdr_skip_wfftype(wff,ierr)

 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
#undef ABI_FUNC
#define ABI_FUNC 'hdr_skip_wfftype'
!End of the abilint section

 implicit none

!Arguments ------------------------------------
 type(wffile_type),intent(inout) :: wff
 integer, intent(out) :: ierr

!Local variables-------------------------------
 integer :: headform,mu,npsp,unit,usepaw
 integer :: integers(17)
 character(len=6) :: codvsn
#if defined HAVE_MPI_IO
 integer(kind=MPI_OFFSET_KIND) :: delim_record,posit,positloc
 integer :: statux(MPI_STATUS_SIZE)
#endif

!*************************************************************************

 unit=wff%unwff
 ierr=0

 if( wff%accesswff==IO_MODE_FORTRAN .or. (wff%accesswff==IO_MODE_FORTRAN_MASTER.and.wff%master==wff%me) ) then

   rewind (unit)

!  Pick off headform from WF file
   read(unit) codvsn,headform                  ! XG040806 This does not work, but I do not understand why ?!
!  read(unit) integers(1),headform             ! This works ...
!  ! MT012408 Because codvsn is char*6 and not an integer !
   if(headform==1   .or. headform==2   .or. &
&   headform==51  .or. headform==52  .or.   &
&   headform==101 .or. headform==102       ) headform=22

   if (headform<44) then
     read (unit) integers(1:13),npsp
   else
     read (unit) integers(1:13),npsp,integers(15:17),usepaw
   end if

!  Skip rest of header records
   do mu=1,2+npsp
     read (unit)
   end do
   if ((headform>=44).and.(usepaw==1)) then
     read (unit)
     read (unit)
   end if

#if defined HAVE_MPI_IO
 else if(wff%accesswff==IO_MODE_MPI)then

   headform=wff%headform
   if(headform==1   .or. headform==2   .or. &
&   headform==51  .or. headform==52  .or. &
&   headform==101 .or. headform==102       ) headform=22

!  Causes all previous writes to be transferred to the storage device
   call flush_unit(wff%unwff)
   call MPI_FILE_SYNC(wff%fhwff,ierr)

!  Check FORTRAN record marker length (only at first call)
   if (wff%nbOct_recMarker<=0) then
     call getRecordMarkerLength_wffile(wff)
   end if

   if (wff%master==wff%me) then

!    Reading the first record of the file -------------------------------------
!    read (unitfi)   codvsn,headform,..............
     posit = 0
     call rwRecordMarker(1,posit,delim_record,wff,ierr)

!    Reading the second record of the file ------------------------------------
!    read(unitfi) bantot, hdr%date, hdr%intxc.................
!    Pick off npsp and usepaw from WF file
     positloc  = posit + wff%nbOct_recMarker + wff%nbOct_int*13
     call MPI_FILE_READ_AT(wff%fhwff,positloc,npsp,1,MPI_INTEGER,statux,ierr)
!    call MPI_FILE_READ_AT_ALL(wff%fhwff,positloc,npsp,1,MPI_INTEGER,statux,ierr)
     if (headform >= 44) then
       positloc = positloc +  wff%nbOct_int*4
       call MPI_FILE_READ_AT(wff%fhwff,positloc,usepaw,1,MPI_INTEGER,statux,ierr)
!      call MPI_FILE_READ_AT_ALL(wff%fhwff,positloc,usepaw,1,MPI_INTEGER,statux,ierr)
     end if
     call rwRecordMarker(1,posit,delim_record,wff,ierr)

!    Reading the rest of the file ---------------------------------------------
     do mu=1,2+npsp
       call rwRecordMarker(1,posit,delim_record,wff,ierr)
     end do
     if ((headform>=44).and.(usepaw==1)) then
       call rwRecordMarker(1,posit,delim_record,wff,ierr)
       call rwRecordMarker(1,posit,delim_record,wff,ierr)
     end if

     wff%offwff=posit
   end if

   if (wff%spaceComm/=MPI_COMM_SELF) then
     call MPI_BCAST(wff%offwff,1,wff%offset_mpi_type,wff%master,wff%spaceComm,ierr)
   end if
#endif
 end if

end subroutine hdr_skip_wfftype
!!***

!----------------------------------------------------------------------

!!****f* m_header/hdr_update
!! NAME
!! hdr_update
!!
!! FUNCTION
!! This subroutine update the header structured datatype.
!! Most of its records had been initialized correctly, but some corresponds
!! to evolving variables, or change with the context (like fform),
!! This routine is to be called before writing the header
!! to a file, in order to have up-to-date information.
!!
!! INPUTS
!! bantot=total number of bands
!! etot=total energy (Hartree)
!! fermie=Fermi energy (Hartree)
!! mpi_atmtab(:)=--optional-- indexes of the atoms treated by current proc
!! mpi_comm_atom=--optional-- MPI communicator over atoms
!! natom=number of atoms
!! residm=maximal residual
!! rprimd(3,3)=dimensional primitive translations for real space (bohr)
!! occ(bantot)=occupancies for each band and k point
!! pawrhoij(natom*usepaw) <type(pawrhoij_type)>= -PAW only- atomic occupancies
!! usepaw= 0 for non paw calculation; =1 for paw calculation
!! xred(3,natom)= relative coords of atoms in unit cell (dimensionless)
!!
!! OUTPUT
!! hdr <type(hdr_type)>=the header, initialized, and for most part of
!!   it, contain its definite values, except for evolving variables
!!
!! PARENTS
!!      afterscfloop,gstate,loper3,m_wfs,nonlinear,respfn,scfcv,setup_bse
!!      setup_screening,setup_sigma,vtorho
!!
!! CHILDREN
!!      mpi_barrier,mpi_bcast,rhoij_alloc
!!
!! SOURCE

subroutine hdr_update(bantot,etot,fermie,hdr,natom,residm,rprimd,occ,pawrhoij,usepaw,xred, &
&                     mpi_comm_atom,mpi_atmtab) ! optional arguments (parallelism)

 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
#undef ABI_FUNC
#define ABI_FUNC 'hdr_update'
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: bantot,natom,usepaw
 integer,optional,intent(in) :: mpi_comm_atom
 real(dp),intent(in) :: etot,fermie,residm
 type(hdr_type),intent(out) :: hdr
!arrays
 integer,optional,target,intent(in) :: mpi_atmtab(:)
 real(dp),intent(in) :: occ(bantot),rprimd(3,3),xred(3,natom)
 type(pawrhoij_type),intent(inout) :: pawrhoij(:)

!Local variables-------------------------------
!scalars
!arrays

! *************************************************************************

!Update of the "evolving" data
 hdr%etot     =etot
 hdr%fermie   =fermie
 hdr%residm   =residm
 hdr%rprimd(:,:)=rprimd(:,:)
 hdr%occ(:)   =occ(:)
 hdr%xred(:,:)=xred(:,:)

 if (usepaw==1) then
   if (present(mpi_comm_atom)) then
     if (present(mpi_atmtab)) then
       call rhoij_copy(pawrhoij,hdr%pawrhoij,mpi_comm_atom=mpi_comm_atom,mpi_atmtab=mpi_atmtab)
     else
       call rhoij_copy(pawrhoij,hdr%pawrhoij,mpi_comm_atom=mpi_comm_atom)
     end if
   else
     call rhoij_copy(pawrhoij,hdr%pawrhoij)
   end if
 end if

end subroutine hdr_update
!!***

!----------------------------------------------------------------------

!!****f* m_header/hdr_comm
!! NAME
!! hdr_comm
!!
!! FUNCTION
!! This subroutine transmits the header structured datatype
!! initialized on one processor (or a group of processor),
!! to the other processors. It also allocate the needed
!! part of the header.
!!
!! INPUTS
!!  master = id of the master process
!!  me = id of the current process
!!  comm = id of the space communicator handler
!!
!! OUTPUT
!!  (no output)
!!
!! SIDE EFFECTS
!!  hdr <type(hdr_type)>=the header. For the master, it is already
!!   initialized entirely, while for the other procs, everything has
!!   to be transmitted.
!!
!! NOTES
!! This routine is called only in the case of MPI version of the code.
!!
!! PARENTS
!!      elphon,m_header,m_io_kss,m_io_screening,m_wfk,read_gkk
!!
!! CHILDREN
!!      mpi_barrier,mpi_bcast,rhoij_alloc
!!
!! SOURCE

subroutine hdr_comm(hdr,master,me,comm)

 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
#undef ABI_FUNC
#define ABI_FUNC 'hdr_comm'
!End of the abilint section

 implicit none
#if defined HAVE_MPI1
 include 'mpif.h'
#endif

!Arguments ------------------------------------
 integer, intent(in) :: master,me,comm
 type(hdr_type),intent(inout) :: hdr

!Local variables-------------------------------
#ifdef HAVE_MPI
 integer :: bantot,cplex,iatom,ierr,index,index2,ipsp,ispden,list_size,list_size2,natom,nkpt
 integer :: npsp,nsel,nspden,nsppol,nsym,nrhoij,ntypat
 integer,allocatable :: list_int(:)
 real(dp),allocatable :: list_dpr(:)
 character(len=fnlen) :: list_tmp
 character(len=fnlen),allocatable :: list_char(:)
#endif

! *************************************************************************

 DBG_ENTER("COLL")

#if defined HAVE_MPI
!Transmit the integer scalars
 list_size=20
 ABI_ALLOCATE(list_int,(list_size))
 if (master==me)then
   list_int(1)=hdr%bantot
   list_int(2)=hdr%date
   list_int(3)=hdr%headform
   list_int(4)=hdr%intxc
   list_int(5)=hdr%ixc
   list_int(6)=hdr%natom
   list_int(7)=hdr%nkpt
   list_int(8)=hdr%npsp
   list_int(9)=hdr%nspden
   list_int(10)=hdr%nspinor
   list_int(11)=hdr%nsppol
   list_int(12)=hdr%nsym
   list_int(13)=hdr%ntypat
   list_int(14)=hdr%occopt
   list_int(15)=hdr%pertcase
   list_int(16)=hdr%usepaw
   list_int(17:19)=hdr%ngfft(1:3)
   list_int(20)=hdr%usewvl
 end if
 call MPI_BARRIER(comm,ierr)
 call MPI_BCAST(list_int,list_size,MPI_INTEGER,0,comm,ierr)
 if(master/=me)then
   hdr%bantot  =list_int(1)
   hdr%date    =list_int(2)
   hdr%headform=list_int(3)
   hdr%intxc   =list_int(4)
   hdr%ixc     =list_int(5)
   hdr%natom   =list_int(6)
   hdr%nkpt    =list_int(7)
   hdr%npsp    =list_int(8)
   hdr%nspden  =list_int(9)
   hdr%nspinor =list_int(10)
   hdr%nsppol  =list_int(11)
   hdr%nsym    =list_int(12)
   hdr%ntypat  =list_int(13)
   hdr%occopt  =list_int(14)
   hdr%pertcase=list_int(15)
   hdr%usepaw  =list_int(16)
   hdr%ngfft(1:3)=list_int(17:19)
   hdr%usewvl  =list_int(20)
 end if
 ABI_DEALLOCATE(list_int)

 bantot=hdr%bantot
 natom =hdr%natom
 nkpt  =hdr%nkpt
 npsp  =hdr%npsp
 nspden=hdr%nspden
 nsppol=hdr%nsppol
 nsym  =hdr%nsym
 ntypat=hdr%ntypat

 if(master/=me)then
!  Allocate all components of hdr
   ABI_ALLOCATE(hdr%istwfk,(nkpt))
   ABI_ALLOCATE(hdr%nband,(nkpt*nsppol))
   ABI_ALLOCATE(hdr%npwarr,(nkpt))
   ABI_ALLOCATE(hdr%pspcod,(npsp))
   ABI_ALLOCATE(hdr%pspdat,(npsp))
   ABI_ALLOCATE(hdr%pspso,(npsp))
   ABI_ALLOCATE(hdr%pspxc,(npsp))
   ABI_ALLOCATE(hdr%lmn_size,(npsp))
   ABI_ALLOCATE(hdr%so_psp,(npsp))
   ABI_ALLOCATE(hdr%symafm,(nsym))
   ABI_ALLOCATE(hdr%symrel,(3,3,nsym))
   ABI_ALLOCATE(hdr%typat,(natom))
   ABI_ALLOCATE(hdr%kptns,(3,nkpt))
   ABI_ALLOCATE(hdr%occ,(bantot))
   ABI_ALLOCATE(hdr%tnons,(3,nsym))
   ABI_ALLOCATE(hdr%wtk,(nkpt))
   ABI_ALLOCATE(hdr%xred,(3,natom))
   ABI_ALLOCATE(hdr%zionpsp,(npsp))
   ABI_ALLOCATE(hdr%znuclpsp,(npsp))
   ABI_ALLOCATE(hdr%znucltypat,(ntypat))
   ABI_ALLOCATE(hdr%title,(npsp))
 end if

!Transmit the integer arrays
 list_size=nkpt*(2+nsppol)+6*npsp+10*nsym+natom
 ABI_ALLOCATE(list_int,(list_size))
 if (master==me)then
   list_int(1      :nkpt             )=hdr%istwfk ; index=nkpt
   list_int(1+index:nkpt*nsppol+index)=hdr%nband  ; index=index+nkpt*nsppol
   list_int(1+index:nkpt       +index)=hdr%npwarr ; index=index+nkpt
   list_int(1+index:npsp       +index)=hdr%pspcod ; index=index+npsp
   list_int(1+index:npsp       +index)=hdr%pspdat ; index=index+npsp
   list_int(1+index:npsp       +index)=hdr%pspso  ; index=index+npsp
   list_int(1+index:npsp       +index)=hdr%pspxc  ; index=index+npsp
   list_int(1+index:npsp       +index)=hdr%lmn_size ; index=index+npsp
   list_int(1+index:npsp       +index)=hdr%so_psp ; index=index+npsp
   list_int(1+index:nsym       +index)=hdr%symafm ; index=index+nsym
   list_int(1+index:nsym*3*3   +index)=reshape(hdr%symrel,(/3*3*nsym/))
   index=index+nsym*3*3
   list_int(1+index:natom      +index)=hdr%typat   ; index=index+natom
 end if
 call MPI_BARRIER(comm,ierr)
 call MPI_BCAST(list_int,list_size,MPI_INTEGER,0,comm,ierr)
 if(master/=me)then
   hdr%istwfk=list_int(1      :nkpt             ) ; index=nkpt
   hdr%nband =list_int(1+index:nkpt*nsppol+index) ; index=index+nkpt*nsppol
   hdr%npwarr=list_int(1+index:nkpt       +index) ; index=index+nkpt
   hdr%pspcod=list_int(1+index:npsp       +index) ; index=index+npsp
   hdr%pspdat=list_int(1+index:npsp       +index) ; index=index+npsp
   hdr%pspso =list_int(1+index:npsp       +index) ; index=index+npsp
   hdr%pspxc =list_int(1+index:npsp       +index) ; index=index+npsp
   hdr%lmn_size=list_int(1+index:npsp     +index) ; index=index+npsp
   hdr%so_psp =list_int(1+index:npsp   +index) ; index=index+npsp
   hdr%symafm=list_int(1+index:nsym       +index) ; index=index+nsym
   hdr%symrel=reshape(list_int(1+index:nsym*3*3   +index),(/3,3,nsym/))
   index=index+nsym*3*3
   hdr%typat  =list_int(1+index:natom      +index) ; index=index+natom
 end if
 ABI_DEALLOCATE(list_int)

!Transmit the double precision scalars and arrays
 list_size=21+3*nkpt+nkpt+bantot+3*nsym+3*natom+2*npsp+ntypat
 ABI_ALLOCATE(list_dpr,(list_size))
 if (master==me)then
   list_dpr(1)=hdr%ecut_eff
   list_dpr(2)=hdr%etot
   list_dpr(3)=hdr%fermie
   list_dpr(4)=hdr%residm
   list_dpr(5:13)=reshape(hdr%rprimd(1:3,1:3),(/9/))
   list_dpr(14)=hdr%ecut
   list_dpr(15)=hdr%ecutdg
   list_dpr(16)=hdr%ecutsm
   list_dpr(17)=hdr%tphysel
   list_dpr(18)=hdr%tsmear
   list_dpr(19:21)=hdr%qptn(1:3)                                 ; index=21
   list_dpr(1+index:3*nkpt +index)=reshape(hdr%kptns,(/3*nkpt/)) ; index=index+3*nkpt
   list_dpr(1+index:nkpt   +index)=hdr%wtk                       ; index=index+nkpt
   list_dpr(1+index:bantot +index)=hdr%occ                       ; index=index+bantot
   list_dpr(1+index:3*nsym +index)=reshape(hdr%tnons,(/3*nsym/)) ; index=index+3*nsym
   list_dpr(1+index:3*natom+index)=reshape(hdr%xred,(/3*natom/)) ; index=index+3*natom
   list_dpr(1+index:npsp   +index)=hdr%zionpsp                   ; index=index+npsp
   list_dpr(1+index:npsp   +index)=hdr%znuclpsp                  ; index=index+npsp
   list_dpr(1+index:ntypat  +index)=hdr%znucltypat               ; index=index+ntypat
 end if
 call MPI_BARRIER(comm,ierr)
 call MPI_BCAST(list_dpr,list_size,MPI_DOUBLE_PRECISION,0,comm,ierr)
 if(master/=me)then
   hdr%ecut_eff=list_dpr(1)
   hdr%etot    =list_dpr(2)
   hdr%fermie  =list_dpr(3)
   hdr%residm  =list_dpr(4)
   hdr%rprimd  =reshape(list_dpr(5:13),(/3,3/))
   hdr%ecut    =list_dpr(14)
   hdr%ecutdg  =list_dpr(15)
   hdr%ecutsm  =list_dpr(16)
   hdr%tphysel =list_dpr(17)
   hdr%tsmear  =list_dpr(18)
   hdr%qptn(1:3)=list_dpr(19:21)                                    ; index=21
   hdr%kptns   =reshape(list_dpr(1+index:3*nkpt +index),(/3,nkpt/)) ; index=index+3*nkpt
   hdr%wtk     =list_dpr(1+index:nkpt   +index)                     ; index=index+nkpt
   hdr%occ     =list_dpr(1+index:bantot +index)                     ; index=index+bantot
   hdr%tnons   =reshape(list_dpr(1+index:3*nsym +index),(/3,nsym/)) ; index=index+3*nsym
   hdr%xred    =reshape(list_dpr(1+index:3*natom+index),(/3,natom/)); index=index+3*natom
   hdr%zionpsp =list_dpr(1+index:npsp   +index)                     ; index=index+npsp
   hdr%znuclpsp=list_dpr(1+index:npsp   +index)                     ; index=index+npsp
   hdr%znucltypat=list_dpr(1+index:ntypat  +index)                  ; index=index+ntypat
 end if
 ABI_DEALLOCATE(list_dpr)

!Transmit the characters
 list_size=npsp+1
 ABI_ALLOCATE(list_char,(list_size))
 if (master==me)then
   list_char(1)       =hdr%codvsn  ! Only 6 characters are stored in list_char(1)
   list_char(2:npsp+1)=hdr%title
 end if
 call MPI_BARRIER(comm,ierr)
 call MPI_BCAST(list_char,list_size*fnlen,MPI_CHARACTER,0,comm,ierr)
 if(master/=me)then
   list_tmp=list_char(1)
   hdr%codvsn=list_tmp(1:6)
   do ipsp=2,npsp+1
     list_tmp =list_char(ipsp)
     hdr%title =list_tmp(1:fnlen)
   end do
 end if
 ABI_DEALLOCATE(list_char)

!Transmit the structured variables in case of PAW
 if (hdr%usepaw==1) then

   nrhoij=0
   if (master==me)then
     cplex=hdr%pawrhoij(1)%cplex
     nspden=hdr%pawrhoij(1)%nspden
     do iatom=1,natom
       nrhoij=nrhoij+hdr%pawrhoij(iatom)%nrhoijsel
     end do
   end if
   call MPI_BARRIER(comm,ierr)
   call MPI_BCAST(nrhoij,1,MPI_INTEGER,0,comm,ierr)
   call MPI_BCAST(cplex ,1,MPI_INTEGER,0,comm,ierr)
   call MPI_BCAST(nspden,1,MPI_INTEGER,0,comm,ierr)

   list_size=natom+nrhoij;list_size2=nspden*nrhoij*cplex
   ABI_ALLOCATE(list_int,(list_size))
   ABI_ALLOCATE(list_dpr,(list_size2))
   if (master==me)then
     index=0;index2=0
     do iatom=1,natom
       nsel=hdr%pawrhoij(iatom)%nrhoijsel
       list_int(1+index)=nsel
       list_int(2+index:1+nsel+index)=hdr%pawrhoij(iatom)%rhoijselect(1:nsel)
       index=index+1+nsel
       do ispden=1,nspden
         list_dpr(1+index2:nsel*cplex+index2)=hdr%pawrhoij(iatom)%rhoijp(1:nsel*cplex,ispden)
         index2=index2+nsel*cplex
       end do
     end do
   end if
   call MPI_BARRIER(comm,ierr)
   call MPI_BCAST(list_int,list_size,MPI_INTEGER,0,comm,ierr)
   call MPI_BCAST(list_dpr,list_size2,MPI_DOUBLE_PRECISION,0,comm,ierr)
   if(master/=me)then
     index=0;index2=0
     ABI_DATATYPE_ALLOCATE(hdr%pawrhoij,(natom))
     call rhoij_alloc(cplex,nspden,hdr%nspinor,hdr%nsppol,hdr%pawrhoij,hdr%typat,lmnsize=hdr%lmn_size)
     do iatom=1,natom
       nsel=list_int(1+index)
       hdr%pawrhoij(iatom)%nrhoijsel=nsel
       hdr%pawrhoij(iatom)%rhoijselect(1:nsel)=list_int(2+index:1+nsel+index)
       index=index+1+nsel
       do ispden=1,nspden
         hdr%pawrhoij(iatom)%rhoijp(1:nsel*cplex,ispden)=list_dpr(1+index2:nsel*cplex+index2)
         index2=index2+nsel*cplex
       end do
     end do
   end if
   ABI_DEALLOCATE(list_int)
   ABI_DEALLOCATE(list_dpr)

 end if

#else
!Dummy arguments
 ABI_UNUSED((/me,master,comm,hdr%bantot/))
#endif

 DBG_EXIT("COLL")

end subroutine hdr_comm
!!***

END MODULE m_header
!!***
