!{\src2tex{textfont=tt}}
!!****f* ABINIT/pawxc3_gga
!! NAME
!! pawxc3_gga
!!
!! FUNCTION
!! PAW only
!! Compute first-order change of XC potential and contribution to
!! 2nd-order change of XC energy inside a PAW sphere.
!! LDA+GGA - USE THE DENSITY OVER A WHOLE SPHERICAL GRID (r,theta,phi)
!!
!! COPYRIGHT
!! Copyright (C) 2009-2012 ABINIT group (MT)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~abinit/COPYING
!! or http://www.gnu.org/copyleft/gpl.txt .
!! For the initials of contributors, see ~abinit/doc/developers/contributors.txt.
!! This routine has been written from rhohxc
!!
!! INPUTS
!!  corexc1(cplex_den*pawrad%mesh_size)=first-order change of core density on radial grid
!!  cplex_den= if 1, 1st-order densities are REAL, if 2, COMPLEX
!!  cplex_vxc= if 1, 1st-order XC potential is complex, if 2, COMPLEX
!!  ixc= choice of exchange-correlation scheme
!!  kxc(pawrad%mesh_size,pawang%angl_size,nkxc)=GS xc kernel
!!  lm_size=size of density array rhor (see below)
!!  lmselect(lm_size)=select the non-zero LM-moments of input density rhor1
!!  nhat1(cplex_den*pawrad%mesh_size,lm_size,nspden)=first-order change of compensation density
!!                                        (total in 1st half and spin-up in 2nd half if nspden=2)
!!  nkxc=second dimension of the kxc array
!!  nspden=number of spin-density components
!!  option=0  compute both 2nd-order XC energy and 1st-order potential
!!         1  compute only 1st-order XC potential
!!         2  compute only 2nd-order XC energy, XC potential is temporary computed here
!!         3  compute only 2nd-order XC energy, XC potential is input in vxc1(:)
!!  pawang <type(pawang_type)>=paw angular mesh and related data
!!  pawrad <type(pawrad_type)>=paw radial mesh and related data
!!  rhor1(cplex_den*pawrad%mesh_size,lm_size,nspden)=first-order change of density
!!  usecore= 1 if core density has to be used in Exc/Vxc ; 0 otherwise
!!  usexcnhat= 0 if compensation density does not have to be used
!!             1 if compensation density has to be used in d2Exc only
!!             2 if compensation density (nhat) has to be used in d2Exc and Vxc1
!!  xclevel= XC functional level
!!
!! OUTPUT
!!  == if option=0 or 2 or 3 ==
!!    d2enxc   =returned exchange-cor. contribution to 2nd-order XC energy
!!    d2enxc_im=returned IMAGINARY PART of exchange-cor. contribution to 2nd-order XC energy
!!              (optional argument)
!!
!! SIDE EFFECTS
!!    vxc1(cplex_vxc*pawrad%mesh_size,pawang%angl_size,nspden)=1st-order XC potential
!!      Output if option==0 or 1
!!      Unused if option==2
!!      Input  if option==3
!!
!! PARENTS
!!      pawxc3
!!
!! CHILDREN
!!      deducer0,nderiv_gen,simp_gen,timab
!!
!! SOURCE

#if defined HAVE_CONFIG_H
#include "config.h"
#endif

#include "abi_common.h"

subroutine pawxc3_gga(corexc1,cplex_den,cplex_vxc,d2enxc,ixc,kxc,lm_size,lmselect,nhat1,nkxc,nspden,&
&                 option,pawang,pawrad,rhor1,usecore,usexcnhat,vxc1,xclevel,&
&                 d2enxc_im) ! optional

 use m_profiling

 use defs_basis
 use defs_datatypes
 use m_errors
 use m_radmesh, only : simp_gen, nderiv_gen, deducer0

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
#undef ABI_FUNC
#define ABI_FUNC 'pawxc3_gga'
 use interfaces_18_timing
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: cplex_den,cplex_vxc,ixc,lm_size,nkxc,nspden,option,usecore,usexcnhat,xclevel
 real(dp),intent(out) :: d2enxc
 real(dp),intent(out),optional :: d2enxc_im
 type(pawang_type),intent(in) :: pawang
 type(pawrad_type),intent(in) :: pawrad
!arrays
 logical,intent(in) :: lmselect(lm_size)
 real(dp),intent(in) :: corexc1(cplex_den*pawrad%mesh_size)
 real(dp),intent(in) :: kxc(pawrad%mesh_size,pawang%angl_size,nkxc)
 real(dp),intent(in) :: nhat1(cplex_den*pawrad%mesh_size,lm_size,nspden*((usexcnhat+1)/2))
 real(dp),intent(in),target :: rhor1(cplex_den*pawrad%mesh_size,lm_size,nspden)
 real(dp),intent(inout),target :: vxc1(cplex_vxc*pawrad%mesh_size,pawang%angl_size,nspden)

!Local variables-------------------------------
!scalars
 integer :: ii,ilm,ipts,ir,ispden,jr,lm_size_eff,npts,nrad,nspden2
 logical :: need_impart
 real(dp),parameter :: tol24=tol12*tol12
 real(dp) :: coeff_grho_corr,coeff_grho_dn,coeff_grho_up
 real(dp) :: coeff_grhoim_corr,coeff_grhoim_dn,coeff_grhoim_up
 real(dp) :: factor,factor_gxc
 real(dp) :: grho_grho,grho_grho1,grho_grho1_up,grho_grho1_dn
 real(dp) :: grho_grho1im,grho_grho1im_up,grho_grho1im_dn
 real(dp) :: rho_dn,rho_up,rhoim_dn,rhoim_up
 real(dp) :: ro11i,ro11r,ro12i,ro12r,ro21i,ro21r,ro22i,ro22r
 real(dp) :: v11i,v11r,v12i,v12r,v21i,v21r,v22i,v22r,vxcrho
 character(len=500) :: msg
!arrays
 real(dp) :: g0(3),g0_dn(3),g0_up(3),g1(3),g1_dn(3),g1_up(3)
 real(dp) :: g1im(3),g1im_dn(3),g1im_up(3)
 real(dp) :: gxc1i(3,2),gxc1r(3,2)
 real(dp) :: tsec(2)
 real(dp),allocatable :: dgxc1(:),drho1(:,:),drho1core(:,:),dylmdr(:,:,:),ff(:),gg(:)
 real(dp),allocatable :: grho_updn(:,:,:),grho1_updn(:,:,:),gxc1(:,:,:,:)
 real(dp),allocatable :: rho_updn(:,:),rho1_updn(:,:),rho1arr(:,:,:)
 real(dp),allocatable,target :: rhohat1(:,:,:)
 real(dp),pointer :: rho1_(:,:,:),vxc1_(:,:,:)

! *************************************************************************

 DBG_ENTER("COLL")

 call timab(81,1,tsec)

!----------------------------------------------------------------------
!----- Check options
!----------------------------------------------------------------------

 if(option<0.or.option>3) then
   msg='  Wrong option !'
   MSG_BUG(msg)
 end if
 if(option/=3) then
   if (xclevel==1.and.nkxc/=2*min(nspden,2)-1) then
     msg='  nkxc must be 1 or 3 !'
     MSG_BUG(msg)
   end if
   if(xclevel==2.and.nkxc/=23) then
     msg='  nkxc should be 23 for GGA !'
     MSG_BUG(msg)
   end if
 end if
 if(nspden==4.and.option/=3) then
   msg='  nspden=4 not implemented (for vxc) !'
   MSG_ERROR(msg)
 end if
 if(pawang%angl_size==0) then
   msg='  pawang%angl_size=0 !'
   MSG_BUG(msg)
 end if
 if(.not.associated(pawang%ylmr)) then
   msg='  pawang%ylmr must be allocated !'
   MSG_BUG(msg)
 end if
 if(xclevel==2.and.(.not.associated(pawang%ylmrgr))) then
   msg='  pawang%ylmrgr must be allocated !'
   MSG_BUG(msg)
 end if

!----------------------------------------------------------------------
!----- Initializations / allocations
!----------------------------------------------------------------------

 nrad=pawrad%mesh_size
 npts=pawang%angl_size
 lm_size_eff=min(lm_size,pawang%ylm_size)

 need_impart=present(d2enxc_im)
 if (option/=1) then
   d2enxc=zero
   if (need_impart) d2enxc_im=zero
 end if
 if (option<=1) vxc1(:,:,:)=zero

!Special case: no XC applied
 if (ixc==0.or.(nkxc==0.and.option/=3)) then
   msg='  Note that no xc is applied (ixc=0).'
   MSG_WARNING(msg)
   return
 end if

 ABI_ALLOCATE(rho1arr,(cplex_den*nrad,nspden,npts))
 if (usexcnhat>0) then
   ABI_ALLOCATE(rhohat1,(cplex_den*nrad,lm_size,nspden))
   rhohat1(:,:,:)=rhor1(:,:,:)+nhat1(:,:,:)
 end if

 if (option==2) then
   ABI_ALLOCATE(vxc1_,(cplex_vxc*nrad,npts,nspden))
 else
   vxc1_ => vxc1
 end if

!Need gradients and additional allocations in case of GGA
 if (xclevel==2.and.option/=3) then
   nspden2=2 ! Force spin-polarized
   ABI_ALLOCATE(rho_updn,(nrad,nspden2))
   ABI_ALLOCATE(rho1_updn,(cplex_den*nrad,nspden2))
   ABI_ALLOCATE(grho_updn,(cplex_den*nrad,nspden2,3))
   ABI_ALLOCATE(grho1_updn,(cplex_den*nrad,nspden2,3))
   ABI_ALLOCATE(gxc1,(cplex_vxc*nrad,3,pawang%ylm_size,nspden2))
   gxc1=zero
   if (usecore==1) then
     ABI_ALLOCATE(drho1core,(nrad,cplex_den))
     if (cplex_den==1)  then
       call nderiv_gen(drho1core,corexc1,1,pawrad)
     else
       ABI_ALLOCATE(ff,(nrad))
       ABI_ALLOCATE(gg,(nrad))
       do ir=1,nrad
         ff(ir)=corexc1(2*ir-1)
         gg(ir)=corexc1(2*ir  )
       end do
       call nderiv_gen(drho1core(:,1),ff,1,pawrad)
       call nderiv_gen(drho1core(:,2),gg,1,pawrad)
       ABI_DEALLOCATE(ff)
       ABI_DEALLOCATE(gg)
     end if
   end if
!  Convert Ylm derivatives from normalized to standard cartesian coordinates
!  dYlm/dr_i = { dYlm/dr_i^hat - Sum_j[ dYlm/dr_j^hat (r_j/r)] } * (1/r)
   ABI_ALLOCATE(dylmdr,(3,npts,pawang%ylm_size))
   do ilm=1,pawang%ylm_size
     do ipts=1,npts
       factor=sum(pawang%ylmrgr(1:3,ilm,ipts)*pawang%anginit(1:3,ipts))
       dylmdr(1:3,ipts,ilm)=pawang%ylmrgr(1:3,ilm,ipts)-factor*pawang%anginit(1:3,ipts)
     end do
   end do
 end if

!----------------------------------------------------------------------
!----- Accumulate and store 1st-order change of XC potential
!----------------------------------------------------------------------

!Do loop on the angular part (theta,phi)
 do ipts=1,npts

!  Copy the input 1st-order density for this (theta,phi)
   rho1arr(:,:,ipts)=zero
   if (usexcnhat< 2) rho1_=>rhor1
   if (usexcnhat==2) rho1_=>rhohat1
   do ispden=1,nspden
     do ilm=1,lm_size_eff
       if (lmselect(ilm)) rho1arr(:,ispden,ipts)=rho1arr(:,ispden,ipts) &
&       +rho1_(:,ilm,ispden)*pawang%ylmr(ilm,ipts)
     end do
   end do
   if (usecore==1) then
     rho1arr(:,1,ipts)=rho1arr(:,1,ipts)+corexc1(:)
     if (nspden==2) rho1arr(:,2,ipts)=rho1arr(:,2,ipts)+half*corexc1(:)
   end if

   if (option/=3) then

!    =======================================================================
!    ======================= LDA ===========================================
!    =======================================================================
     if (xclevel==1.or.ixc==13) then

!      Non-spin-polarized
       if(nspden==1)then
         if (cplex_vxc==1) then
           if (cplex_den==1) then  ! cplex_vxc==1 and cplex_den==1
             vxc1_(1:nrad,ipts,1)=kxc(1:nrad,ipts,1)*rho1arr(1:nrad,1,ipts)
           else                    ! cplex_vxc==1 and cplex_den==2
             do ir=1,nrad
               vxc1_(ir,ipts,1)=kxc(ir,ipts,1)*rho1arr(2*ir-1,1,ipts)
             end do
           end if
         else
           if (cplex_den==1) then  ! cplex_vxc==2 and cplex_den==1
             do ir=1,nrad
               vxc1_(2*ir-1,ipts,1)=kxc(ir,ipts,1)*rho1arr(ir,1,ipts)
               vxc1_(2*ir  ,ipts,1)=zero
             end do
           else                    ! cplex_vxc==2 and cplex_den==2
             do ir=1,nrad
               vxc1_(2*ir-1,ipts,1)=kxc(ir,ipts,1)*rho1arr(2*ir-1,1,ipts)
               vxc1_(2*ir  ,ipts,1)=kxc(ir,ipts,1)*rho1arr(2*ir  ,1,ipts)
             end do
           end if
         end if

!        Spin-polarized
       else
         if (cplex_vxc==1) then
           if (cplex_den==1) then  ! cplex_vxc==1 and cplex_den==1
             do ir=1,nrad
               rho_up=rho1arr(ir,2,ipts);rho_dn=rho1arr(ir,1,ipts)-rho_up
               vxc1_(ir,ipts,1)=kxc(ir,ipts,1)*rho_up+kxc(ir,ipts,2)*rho_dn
               vxc1_(ir,ipts,2)=kxc(ir,ipts,2)*rho_up+kxc(ir,ipts,3)*rho_dn
             end do
           else                    ! cplex_vxc==1 and cplex_den==2
             do ir=1,nrad
               jr=2*ir-1
               rho_up=rho1arr(jr,2,ipts);rho_dn=rho1arr(jr,1,ipts)-rho_up
               vxc1_(ir,ipts,1)=kxc(ir,ipts,1)*rho_up+kxc(ir,ipts,2)*rho_dn
               vxc1_(ir,ipts,2)=kxc(ir,ipts,2)*rho_up+kxc(ir,ipts,3)*rho_dn
             end do
           end if
         else
           if (cplex_den==1) then  ! cplex_vxc==2 and cplex_den==1
             do ir=1,nrad
               jr=2*ir-1
               rho_up=rho1arr(ir,2,ipts);rho_dn=rho1arr(ir,1,ipts)-rho_up
               vxc1_(jr,ipts,1)=kxc(ir,ipts,1)*rho_up+kxc(ir,ipts,2)*rho_dn
               vxc1_(jr,ipts,2)=kxc(ir,ipts,2)*rho_up+kxc(ir,ipts,3)*rho_dn
             end do
           else                    ! cplex_vxc==2 and cplex_den==2
             do ir=1,nrad
               jr=2*ir
               rho_up  =rho1arr(jr-1,2,ipts);rho_dn  =rho1arr(jr-1,1,ipts)-rho_up
               rhoim_up=rho1arr(jr  ,2,ipts);rhoim_dn=rho1arr(jr  ,1,ipts)-rhoim_up
               vxc1_(jr-1,ipts,1)=kxc(ir,ipts,1)*rho_up  +kxc(ir,ipts,2)*rho_dn
               vxc1_(jr  ,ipts,1)=kxc(ir,ipts,1)*rhoim_up+kxc(ir,ipts,2)*rhoim_dn
               vxc1_(jr-1,ipts,2)=kxc(ir,ipts,2)*rho_up  +kxc(ir,ipts,3)*rho_dn
               vxc1_(jr  ,ipts,2)=kxc(ir,ipts,2)*rhoim_up+kxc(ir,ipts,3)*rhoim_dn
             end do
           end if
         end if
       end if

     else
!      =======================================================================
!      ======================= GGA ===========================================
!      =======================================================================

!      Like in mkvxcgga3, everything is treated as spin-polarized (nspden2=2)
!      FOR NSPDEN=1, should eliminate computation of gxc1(...,2), vxc1(...,2)

!      Transfer the ground-state density and its gradient to (up,dn) storage
       if (nspden2==1) then
         rho_updn(:,1)=half*kxc(:,ipts,16)
       else if (nspden==1) then
         rho_updn(:,1)=half*kxc(:,ipts,16)
         rho_updn(:,2)=rho_updn(:,1)
       else if (nspden==2) then
         rho_updn(:,1)=kxc(:,ipts,17)
         rho_updn(:,2)=kxc(:,ipts,16)-rho_updn(:,1)
       end if
       do ii=1,3
         if (nspden2==1) then
           grho_updn(:,1,ii)=half*kxc(:,ipts,16+2*ii)
         else if (nspden==1) then
           grho_updn(:,1,ii)=half*kxc(:,ipts,16+2*ii)
           grho_updn(:,2,ii)=grho_updn(:,1,ii)
         else if (nspden==2) then
           grho_updn(:,1,ii)=kxc(:,ipts,17+2*ii)
           grho_updn(:,2,ii)=kxc(:,ipts,16+2*ii)-grho_updn(:,1,ii)
         end if
       end do

!      First store the 1st-order density and its gradient in (up+dn,up) format
       rho1_updn(:,:)=rho1arr(:,:,ipts);grho1_updn(:,:,1:3)=zero
       ABI_ALLOCATE(drho1,(nrad,cplex_den))
       if (cplex_den==1) then
         ABI_ALLOCATE(ff,(nrad))
         do ispden=1,min(nspden,nspden2)
           do ilm=1,lm_size_eff
             if (lmselect(ilm)) then
               ff(1:nrad)=rho1_(1:nrad,ilm,ispden)
               call nderiv_gen(drho1,ff,1,pawrad)
               ff(2:nrad)=ff(2:nrad)/pawrad%rad(2:nrad)
               call deducer0(ff,nrad,pawrad)
               do ii=1,3
                 grho1_updn(1:nrad,ispden,ii)=grho1_updn(1:nrad,ispden,ii) &
&                 +drho1(1:nrad,1)*pawang%ylmr(ilm,ipts)*pawang%anginit(ii,ipts) &
&                 +ff(1:nrad)*dylmdr(ii,ipts,ilm)
               end do
             end if
           end do
         end do
         ABI_DEALLOCATE(ff)
       else
         ABI_ALLOCATE(ff,(nrad))
         ABI_ALLOCATE(gg,(nrad))
         do ispden=1,min(nspden,nspden2)
           do ilm=1,lm_size_eff
             if (lmselect(ilm)) then
               do ir=1,nrad
                 ff(ir)=rho1_(2*ir-1,ilm,ispden)
                 gg(ir)=rho1_(2*ir  ,ilm,ispden)
               end do
               call nderiv_gen(drho1(:,1),ff,1,pawrad)
               call nderiv_gen(drho1(:,2),gg,1,pawrad)
               ff(2:nrad)=ff(2:nrad)/pawrad%rad(2:nrad)
               gg(2:nrad)=gg(2:nrad)/pawrad%rad(2:nrad)
               call deducer0(ff,nrad,pawrad)
               call deducer0(gg,nrad,pawrad)
               do ii=1,3
                 do ir=2,nrad
                   jr=2*ir
                   grho1_updn(jr-1,ispden,ii)=grho1_updn(jr-1,ispden,ii) &
&                   +drho1(ir,1)*pawang%ylmr(ilm,ipts)*pawang%anginit(ii,ipts) &
&                   +ff(ir)*dylmdr(ii,ipts,ilm)
                   grho1_updn(jr  ,ispden,ii)=grho1_updn(jr  ,ispden,ii) &
&                   +drho1(ir,2)*pawang%ylmr(ilm,ipts)*pawang%anginit(ii,ipts) &
&                   +gg(ir)*dylmdr(ii,ipts,ilm)
                 end do
               end do
             end if
           end do
         end do
         ABI_DEALLOCATE(ff)
         ABI_DEALLOCATE(gg)
       end if
       if (usecore==1) then
         if (cplex_den==1) then
           do ispden=1,min(nspden,nspden2)
             factor=one;if (ispden==2) factor=half
             do ii=1,3
               grho1_updn(1:nrad,ispden,ii)=grho1_updn(1:nrad,ispden,ii) &
&               +factor*drho1core(1:nrad,1)*pawang%anginit(ii,ipts)
             end do
           end do
         else
           do ispden=1,min(nspden,nspden2)
             factor=one;if (ispden==2) factor=half
             do ii=1,3
               do ir=1,nrad
                 jr=2*ir
                 grho1_updn(jr-1,ispden,ii)=grho1_updn(jr-1,ispden,ii) &
&                 +factor*drho1core(ir,1)*pawang%anginit(ii,ipts)
                 grho1_updn(jr  ,ispden,ii)=grho1_updn(jr  ,ispden,ii) &
&                 +factor*drho1core(ir,2)*pawang%anginit(ii,ipts)
               end do
             end do
           end do
         end if
       end if
       ABI_DEALLOCATE(drho1)

!      Translate the 1st-order density and its gradient  in (up,dn) format
       if (nspden2==1) then
         rho1_updn(:,1)=half*rho1_updn(:,1)
         grho1_updn(:,1,1:3)=half*grho1_updn(:,1,1:3)
       else if (nspden==1) then
         rho1_updn(:,1)=half*rho1_updn(:,1)
         rho1_updn(:,2)=rho1_updn(:,1)
         grho1_updn(:,1,1:3)=half*grho1_updn(:,1,1:3)
         grho1_updn(:,2,1:3)=grho1_updn(:,1,1:3)
       else if (nspden==2) then
         rho1_updn(:,2)=rho1_updn(:,1)-rho1_updn(:,2)
         rho1_updn(:,1)=rho1_updn(:,1)-rho1_updn(:,2)
         grho1_updn(:,2,1:3)=grho1_updn(:,1,1:3)-grho1_updn(:,2,1:3)
         grho1_updn(:,1,1:3)=grho1_updn(:,1,1:3)-grho1_updn(:,2,1:3)
       end if

!      Apply XC kernel
       factor_gxc=four_pi;if (nspden2==1) factor_gxc=two_pi
       do ir=1,nrad
         if (cplex_vxc==1) then  ! cplex_vxc==1 and (cplex_den==1 or cplex_den=2)
           jr=cplex_den*(ir-1)+1
           g0_up(:)=grho_updn(ir,1,:)    ! grad of spin-up GS rho
           g0_dn(:)=grho_updn(ir,2,:)    ! grad of spin-down GS rho
           g0(:)=g0_up(:)+g0_dn(:)       ! grad of GS rho
           g1_up(:)=grho1_updn(jr,1,:)   ! grad of spin-up rho1
           g1_dn(:)=grho1_updn(jr,2,:)   ! grad of spin-down rho1
           g1(:)=g1_up(:)+g1_dn(:)       ! grad of GS rho1
           grho_grho1_up=g1_up(1)*g0_up(1)+g1_up(2)*g0_up(2)+g1_up(3)*g0_up(3)
           grho_grho1_dn=g1_dn(1)*g0_dn(1)+g1_dn(2)*g0_dn(2)+g1_dn(3)*g0_dn(3)
           grho_grho1   =g1   (1)*g0   (1)+g1   (2)*g0   (2)+g1   (3)*g0   (3)
           grho_grho    =g0   (1)*g0   (1)+g0   (2)*g0   (2)+g0   (3)*g0   (3)
           vxc1_(ir,ipts,1)=(kxc(ir,ipts, 1)+kxc(ir,ipts, 9))*rho1_updn(jr,1) &
&           +kxc(ir,ipts,10)                 *rho1_updn(jr,2) &
&           +kxc(ir,ipts, 5)*grho_grho1_up &
&           +kxc(ir,ipts,13)*grho_grho1
           if (nspden==2) then
             vxc1_(ir,ipts,2)=(kxc(ir,ipts, 2)+kxc(ir,ipts,11))*rho1_updn(jr,2) &
&             +kxc(ir,ipts,10)                 *rho1_updn(jr,1) &
&             +kxc(ir,ipts, 6)*grho_grho1_dn &
&             +kxc(ir,ipts,14)*grho_grho1
           end if
           coeff_grho_corr=kxc(ir,ipts,13)*rho1_updn(jr,1)+kxc(ir,ipts,14)*rho1_updn(jr,2) &
&           +kxc(ir,ipts,15)*grho_grho1
           coeff_grho_up  =kxc(ir,ipts, 5)*rho1_updn(jr,1)+kxc(ir,ipts, 7)*grho_grho1_up
           coeff_grho_dn  =kxc(ir,ipts, 6)*rho1_updn(jr,2)+kxc(ir,ipts, 8)*grho_grho1_dn
           gxc1r(1:3,1)=g1_up(1:3)*(kxc(ir,ipts, 3)+kxc(ir,ipts,12))+g1_dn(1:3)*kxc(ir,ipts,12) &
&           +g0_up(1:3)*coeff_grho_up+g0(1:3)*coeff_grho_corr
           gxc1r(1:3,2)=g1_dn(1:3)*(kxc(ir,ipts, 4)+kxc(ir,ipts,12))+g1_up(1:3)*kxc(ir,ipts,12) &
&           +g0_dn(1:3)*coeff_grho_dn+g0(1:3)*coeff_grho_corr
           if (grho_grho<tol24) gxc1r(:,:)=zero ! ???
           do ispden=1,nspden2
             do ilm=1,pawang%ylm_size
               do ii=1,3
                 gxc1(ir,ii,ilm,ispden)=gxc1(ir,ii,ilm,ispden) &
                 +gxc1r(ii,ispden)*pawang%ylmr(ilm,ipts)*pawang%angwgth(ipts)*factor_gxc
               end do
             end do
           end do
!          abirules
           if (.false.) write(std_out,*) coeff_grhoim_corr,coeff_grhoim_dn,coeff_grhoim_up
           if (.false.) write(std_out,*) grho_grho1im,grho_grho1im_up,grho_grho1im_dn
           if (.false.) write(std_out,*) gxc1i,g1im,g1im_dn,g1im_up
!          else
!          if (cplex_den==1) then  ! cplex_vxc==2 and cplex_den==1
!          jr=2*ir-1
!          g0_up(:)=grho_updn(ir,1,:)     ! grad of spin-up GS rho
!          g0_dn(:)=grho_updn(ir,2,:)     ! grad of spin-down GS rho
!          g0(:)=g0_up(:)+g0_dn(:)        ! grad of GS rho
!          g1_up(:)=grho1_updn(ir,1,:)    ! grad of spin-up rho1
!          g1_dn(:)=grho1_updn(ir,2,:)    ! grad of spin-down rho1
!          g1(:)=g1_up(:)+g1_dn(:)        ! grad of GS rho1
!          grho_grho1_up=g1_up(1)*g0_up(1)+g1_up(2)*g0_up(2)+g1_up(3)*g0_up(3)
!          grho_grho1_dn=g1_dn(1)*g0_dn(1)+g1_dn(2)*g0_dn(2)+g1_dn(3)*g0_dn(3)
!          grho_grho1   =g1(1)*g0(1)+g1(2)*g0(2)+g1(3)*g0(3)
!          vxc1_(jr  ,ipts,1)=(kxc(ir,ipts,1)+kxc(ir,ipts,9))*rho1_updn(ir,1) &
!          &             +kxc(ir,ipts,10)*rho1_updn(ir,2) &
!          &             +kxc(ir,ipts,5)*grho_grho1_up &
!          &             +kxc(ir,ipts,13)*grho_grho1
!          vxc1_(jr+1,ipts,1)=zero
!          vxc1_(jr  ,ipts,2)=(kxc(ir,ipts,2)+kxc(ir,ipts,11))*rho1_updn(ir,2) &
!          &             +kxc(ir,ipts,10)*rho1_updn(ir,1) &
!          &             +kxc(ir,ipts,6)*grho_grho1_dn &
!          &             +kxc(ir,ipts,14)*grho_grho1
!          vxc1_(jr+1,ipts,2)=zero
!          coeff_grho_corr=kxc(ir,ipts,13)*rho1_updn(ir,1)+kxc(ir,ipts,14)*rho1_updn(ir,2) &
!          &             +kxc(ir,ipts,15)*grho_grho1
!          coeff_grho_up  =kxc(ir,ipts,5)*rho1_updn(ir,1)+kxc(ir,ipts,7)*grho_grho1_up
!          coeff_grho_dn  =kxc(ir,ipts,6)*rho1_updn(ir,2)+kxc(ir,ipts,8)*grho_grho1_dn
!          grho1_updn(jr  ,1,:)=g1_up(:)*(kxc(ir,ipts,3)+kxc(ir,ipts,12))+g1_dn(:)*kxc(ir,ipts,12) &
!          &             +g0_up(:)*coeff_grho_up+g0(:)*coeff_grho_corr
!          grho1_updn(jr+1,1,:)=zero
!          grho1_updn(jr,2,:)=g1_dn(:)*(kxc(ir,ipts,4)+kxc(ir,ipts,12))+g1_up(:)*kxc(ir,ipts,12) &
!          &             +g0_dn(:)*coeff_grho_dn+g0(:)*coeff_grho_corr
!          grho1_updn(jr+1,2,:)=zero
!          else                    ! cplex_vxc==2 and cplex_den==2
!          jr=2*ir-1
!          g0_up(:)=grho_updn(ir,1,:)       ! grad of spin-up GS rho
!          g0_dn(:)=grho_updn(ir,2,:)       ! grad of spin-down GS rho
!          g0(:)=g0_up(:)+g0_dn(:)          ! grad of GS rho
!          g1_up(:)  =grho1_updn(jr  ,1,:)  ! grad of spin-up rho1
!          g1im_up(:)=grho1_updn(jr+1,1,:)  ! grad of spin-up rho1, im part
!          g1_dn(:)  =grho1_updn(jr  ,2,:)  ! grad of spin-down rho1
!          g1im_dn(:)=grho1_updn(jr+1,2,:)  ! grad of spin-down rho1, im part
!          g1(:)=g1_up(:)+g1_dn(:)          ! grad of GS rho1
!          g1im(:)=g1im_up(:)+g1im_dn(:)    ! grad of GS rho1, im part
!          grho_grho1_up=g1_up(1)*g0_up(1)+g1_up(2)*g0_up(2)+g1_up(3)*g0_up(3)
!          grho_grho1_dn=g1_dn(1)*g0_dn(1)+g1_dn(2)*g0_dn(2)+g1_dn(3)*g0_dn(3)
!          grho_grho1   =g1(1)*g0(1)+g1(2)*g0(2)+g1(3)*g0(3)
!          grho_grho1im_up=g1im_up(1)*g0_up(1)+g1im_up(2)*g0_up(2)+g1im_up(3)*g0_up(3)
!          grho_grho1im_dn=g1im_dn(1)*g0_dn(1)+g1im_dn(2)*g0_dn(2)+g1im_dn(3)*g0_dn(3)
!          grho_grho1im   =g1im(1)*g0(1)+g1im(2)*g0(2)+g1im(3)*g0(3)
!          vxc1_(jr  ,ipts,1)=(kxc(ir,ipts,1)+kxc(ir,ipts,9))*rho1_updn(jr,1) &
!          &             +kxc(ir,ipts,10)*rho1_updn(jr,2) &
!          &             +kxc(ir,ipts,5)*grho_grho1_up &
!          &             +kxc(ir,ipts,13)*grho_grho1
!          vxc1_(jr+1,ipts,1)=(kxc(ir,ipts,1)+kxc(ir,ipts,9))*rho1_updn(jr+1,1) &
!          &             +kxc(ir,ipts,10)*rho1_updn(jr+1,2) &
!          &             +kxc(ir,ipts,5)*grho_grho1im_up &
!          &             +kxc(ir,ipts,13)*grho_grho1im
!          vxc1_(jr  ,ipts,2)=(kxc(ir,ipts,2)+kxc(ir,ipts,11))*rho1_updn(jr,2) &
!          &             +kxc(ir,ipts,10)*rho1_updn(jr,1) &
!          &             +kxc(ir,ipts,6)*grho_grho1_dn &
!          &             +kxc(ir,ipts,14)*grho_grho1
!          vxc1_(jr+1,ipts,2)=(kxc(ir,ipts,2)+kxc(ir,ipts,11))*rho1_updn(jr+1,2) &
!          &             +kxc(ir,ipts,10)*rho1_updn(jr+1,1) &
!          &             +kxc(ir,ipts,6)*grho_grho1im_dn &
!          &             +kxc(ir,ipts,14)*grho_grho1im
!          coeff_grho_corr  =kxc(ir,ipts,13)*rho1_updn(jr,1)+kxc(ir,ipts,14)*rho1_updn(jr,2) &
!          &             +kxc(ir,ipts,15)*grho_grho1
!          coeff_grhoim_corr=kxc(ir,ipts,13)*rho1_updn(jr+1,1)+kxc(ir,ipts,14)*rho1_updn(jr+1,2) &
!          &             +kxc(ir,ipts,15)*grho_grho1im
!          coeff_grho_up    =kxc(ir,ipts,5)*rho1_updn(jr,1)+kxc(ir,ipts,7)*grho_grho1_up
!          coeff_grhoim_up  =kxc(ir,ipts,5)*rho1_updn(jr+1,1)+kxc(ir,ipts,7)*grho_grho1im_up
!          coeff_grho_dn    =kxc(ir,ipts,6)*rho1_updn(jr,2)+kxc(ir,ipts,8)*grho_grho1_dn
!          coeff_grhoim_dn  =kxc(ir,ipts,6)*rho1_updn(jr+1,2)+kxc(ir,ipts,8)*grho_grho1im_dn
!          grho1_updn(jr  ,1,:)=g1_up(:)*(kxc(ir,ipts,3)+kxc(ir,ipts,12))+g1_dn(:)*kxc(ir,ipts,12) &
!          &             +g0_up(:)*coeff_grho_up+g0(:)*coeff_grho_corr
!          grho1_updn(jr+1,1,:)=g1im_up(:)*(kxc(ir,ipts,3)+kxc(ir,ipts,12))+g1im_dn(:)*kxc(ir,ipts,12) &
!          &             +g0_up(:)*coeff_grhoim_up+g0(:)*coeff_grhoim_corr
!          grho1_updn(jr  ,2,:)=g1_dn(:)*(kxc(ir,ipts,4)+kxc(ir,ipts,12))+g1_up(:)*kxc(ir,ipts,12) &
!          &             +g0_dn(:)*coeff_grho_dn+g0(:)*coeff_grho_corr
!          grho1_updn(jr+1,2,:)=g1im_dn(:)*(kxc(ir,ipts,4)+kxc(ir,ipts,12))+g1im_up(:)*kxc(ir,ipts,12) &
!          &             +g0_dn(:)*coeff_grhoim_dn+g0(:)*coeff_grhoim_corr
!          end if ! cplex_den
!          do ispden=1,nspden2
!          do ilm=1,pawang%ylm_size
!          do ii=1,3
!          gxc1(jr  ,ii,ilm,ispden)=gxc1(jr  ,ii,ilm,ispden) &
!          +gxc1r(ii,ispden)*pawang%ylmr(ilm,ipts)*pawang%angwgth(ipts)*factor_gxc
!          gxc1(jr+1,ii,ilm,ispden)=gxc1(jr+1,ii,ilm,ispden) &
!          +gxc1i(ii,ispden)*pawang%ylmr(ilm,ipts)*pawang%angwgth(ipts)*factor_gxc
!          end do
!          end do
!          end do
         end if   ! cplex_vxc

       end do ! ir

     end if ! LDA or GGA

   end if ! option/=3

!  ----- End of the loop on npts (angular part)
 end do

!Deallocate memory
 nullify(rho1_)
 if (usexcnhat>0)  then
   ABI_DEALLOCATE(rhohat1)
 end if
 if (xclevel==2.and.option/=3) then
   ABI_DEALLOCATE(rho_updn)
   ABI_DEALLOCATE(rho1_updn)
   ABI_DEALLOCATE(grho_updn)
   ABI_DEALLOCATE(grho1_updn)
   if (usecore==1)  then
     ABI_DEALLOCATE(drho1core)
   end if
 end if

!----------------------------------------------------------------------
!----- If GGA, modify potential with term from density gradient
!----------------------------------------------------------------------
 if (xclevel==2.and.ixc/=13.and.option/=3) then
!  Compute divergence of gxc1 and substract it from Vxc
   ABI_ALLOCATE(dgxc1,(cplex_vxc*nrad))
!  Need to multiply gxc by 2 in the non-polarised case
   factor=one;if (nspden2==1) factor=two
   if (cplex_vxc==1) then
     ABI_ALLOCATE(ff,(nrad))
     do ispden=1,nspden
!      do ispden=1,nspden2
       do ilm=1,pawang%ylm_size
         do ii=1,3
           ff(1:nrad)=gxc1(1:nrad,ii,ilm,ispden)
           call nderiv_gen(dgxc1,ff,1,pawrad)
           ff(2:nrad)=ff(2:nrad)/pawrad%rad(2:nrad)
           call deducer0(ff,nrad,pawrad)
           do ipts=1,npts
             vxc1_(1:nrad,ipts,ispden)=vxc1_(1:nrad,ipts,ispden) &
&             -factor*(dgxc1(1:nrad)*pawang%anginit(ii,ipts)*pawang%ylmr(ilm,ipts) &
&             +ff(1:nrad)*dylmdr(ii,ipts,ilm))
           end do
         end do
       end do
     end do
     ABI_DEALLOCATE(ff)
!    else
!    ABI_ALLOCATE(ff,(nrad))
!    ABI_ALLOCATE(gg,(nrad))
!    do ispden=1,nspden
!    !      do ispden=1,nspden2
!    do ilm=1,pawang%ylm_size
!    do ii=1,3
!    do ir=1,nrad
!    jr=2*ir
!    ff(ir)=gxc1(jr-1,ii,ilm,ispden)
!    gg(ir)=gxc1(jr  ,ii,ilm,ispden)
!    end do
!    call nderiv_gen(dgxc1(1:nrad)       ,ff,1,pawrad)
!    call nderiv_gen(dgxc1(nrad+1:2*nrad),gg,1,pawrad)
!    ff(2:nrad)=ff(2:nrad)/pawrad%rad(2:nrad)
!    gg(2:nrad)=gg(2:nrad)/pawrad%rad(2:nrad)
!    call deducer0(ff,nrad,pawrad)
!    call deducer0(gg,nrad,pawrad)
!    do ipts=1,npts
!    do ir=1,nrad
!    jr=2*ir
!    vxc1_(jr-1,ipts,ispden)=vxc1_(jr-1,ipts,ispden) &
!    &               -factor*(dgxc1(ir)*pawang%anginit(ii,ipts)*pawang%ylmr(ilm,ipts) &
!    &                       +ff(ir)*dylmdr(ii,ipts,ilm))
!    vxc1_(jr  ,ipts,ispden)=vxc1_(jr  ,ipts,ispden) &
!    &               -factor*(dgxc1(ir)*pawang%anginit(ii,ipts)*pawang%ylmr(ilm,ipts) &
!    &                       +gg(ir)*dylmdr(ii,ipts,ilm))
!    end do
!    end do
!    end do
!    end do
!    end do
!    ABI_DEALLOCATE(ff)
!    ABI_DEALLOCATE(gg)
!    end if
     ABI_DEALLOCATE(dgxc1)
   end if

 end if ! GGA

!----------------------------------------------------------------------
!----- Accumulate and store 2nd-order change of XC energy
!----------------------------------------------------------------------
 if (option/=1) then

!  Do loop on the angular part (theta,phi)
   do ipts=1,npts

!    For usexnhat=1 particular case, add now compensation density
     if (usexcnhat==1) then
       do ispden=1,nspden
         do ilm=1,lm_size_eff
           if (lmselect(ilm)) rho1arr(:,ispden,ipts)=rho1arr(:,ispden,ipts)+nhat1(:,ilm,ispden)*pawang%ylmr(ilm,ipts)
         end do
       end do
     end if

!    ----- Calculate d2Exc=Int[Vxc^(1)^*(r).n^(1)(r).dr]
     ABI_ALLOCATE(ff,(nrad))
     if (need_impart) then
       ABI_ALLOCATE(gg,(nrad))
     end if

!    COLLINEAR MAGNETISM
     if (nspden/=4) then
       if (cplex_vxc==1.and.cplex_den==1) then       ! cplex_vxc==1 and cplex_den==1
         ff(:)=vxc1_(:,ipts,1)*rho1arr(:,nspden,ipts)
         if (nspden==2) ff(:)=ff(:)+vxc1_(:,ipts,2)*(rho1arr(:,1,ipts)-rho1arr(:,2,ipts))
         if (need_impart) gg(:)=zero
       else if (cplex_vxc==2.and.cplex_den==2) then  ! cplex_vxc==2 and cplex_den==2
         if (.not.need_impart) then      ! Real part only
           do ir=1,nrad
             jr=2*ir;v11r=vxc1_(jr-1,ipts,1);v11i=vxc1_(jr,ipts,1)
             ro11r=rho1arr(jr-1,nspden,ipts);ro11i=rho1arr(jr,nspden,ipts)
             ff(ir)=v11r*ro11r+v11i*ro11i
           end do
           if (nspden==2) then
             do ir=1,nrad
               jr=2*ir;v22r=vxc1_(jr-1,ipts,2);v22i=vxc1_(jr,ipts,2)
               ro22r=rho1arr(jr-1,1,ipts)-rho1arr(jr-1,2,ipts)
               ro22i=rho1arr(jr  ,1,ipts)-rho1arr(jr  ,2,ipts)
               ff(ir)=ff(ir)+v22r*ro22r+v22i*ro22i
             end do
           end if
         else
           do ir=1,nrad                  ! Real and imaginary parts
             jr=2*ir;v11r=vxc1_(jr-1,ipts,1);v11i=vxc1_(jr,ipts,1)
             ro11r=rho1arr(jr-1,nspden,ipts);ro11i=rho1arr(jr,nspden,ipts)
             ff(ir)=v11r*ro11r+v11i*ro11i
             gg(ir)=v11r*ro11i-v11i*ro11r
           end do
           if (nspden==2) then
             do ir=1,nrad
               jr=2*ir;v22r=vxc1_(jr-1,ipts,2);v22i=vxc1_(jr,ipts,2)
               ro22r=rho1arr(jr-1,1,ipts)-rho1arr(jr-1,2,ipts)
               ro22i=rho1arr(jr  ,1,ipts)-rho1arr(jr  ,2,ipts)
               ff(ir)=ff(ir)+v22r*ro22r+v22i*ro22i
               gg(ir)=gg(ir)+v22r*ro22i-v22i*ro22r
             end do
           end if
         end if
       else                                          ! other cases for cplex_vxc and cplex_den
         v11i=zero;ro11i=zero
         do ir=1,nrad
           jr=cplex_vxc*(ir-1)+1
           v11r=vxc1_(jr,ipts,1);if (cplex_vxc==2) v11i=vxc1_(jr+1,ipts,1)
           jr=cplex_den*(ir-1)+1
           ro11r=rho1arr(jr,nspden,ipts);if (cplex_den==2) ro11i=rho1arr(jr+1,nspden,ipts)
           ff(ir)=v11r*ro11r+v11i*ro11i
           if (need_impart) gg(ir)=v11r*ro11i-v11i*ro11r
         end do
         if (nspden==2) then
           v22i=zero;ro22i=zero
           do ir=1,nrad
             jr=cplex_vxc*(ir-1)+1
             v22r=vxc1_(jr,ipts,2);if (cplex_vxc==2) v22i=vxc1_(jr+1,ipts,2)
             jr=cplex_den*(ir-1)+1
             ro22r=rho1arr(jr,1,ipts)-rho1arr(jr,2,ipts)
             if (cplex_den==2) ro22i=rho1arr(jr+1,1,ipts)-rho1arr(jr+1,2,ipts)
             ff(ir)=ff(ir)+v22r*ro22r+v22i*ro22i
             gg(ir)=gg(ir)+v22r*ro22i-v22i*ro22r
           end do
         end if
       end if ! cplex_vxc and cplex_den

!      NON-COLLINEAR MAGNETISM
     else
       if (cplex_vxc==1.and.cplex_den==1) then   ! cplex_vxc==1 and cplex_den==1
         ff(:)=half*(vxc1_(:,ipts,1)*(rho1arr(:,1,ipts)+rho1arr(:,4,ipts)) &
&         +vxc1_(:,ipts,2)*(rho1arr(:,1,ipts)-rho1arr(:,4,ipts))) &
&         +vxc1_(:,ipts,3)*rho1arr(:,2,ipts) &
&         -vxc1_(:,ipts,4)*rho1arr(:,3,ipts)
         if (need_impart) gg(:)=zero
       else                                      ! other cases for cplex_vxc and cplex_den

!        V is stored as : v^11, v^22, V^12, i.V^21 (each are complex)
!        N is stored as : n, m_x, m_y, mZ          (each are complex)
         do ir=1,nrad
           jr=cplex_vxc*(ir-1)+1
           v11r= vxc1_(jr,ipts,1);v22r= vxc1_(jr,ipts,2)
           v12r= vxc1_(jr,ipts,3);v21i=-vxc1_(jr,ipts,1)
           if (cplex_vxc==2) then
             v11i= vxc1_(jr+1,ipts,1);v22i= vxc1_(jr+1,ipts,2)
             v12i= vxc1_(jr+1,ipts,3);v21r= vxc1_(jr+1,ipts,1)
           else
             v11i=zero;v22i=zero
             v12i=zero;v21i=zero
           end if
           jr=cplex_den*(ir-1)+1
           ro11r= rho1arr(jr,1,ipts)+rho1arr(jr,4,ipts)
           ro22r= rho1arr(jr,1,ipts)-rho1arr(jr,4,ipts)
           ro12r= rho1arr(jr,2,ipts);ro12i=-rho1arr(jr,3,ipts)
           ro21r= rho1arr(jr,2,ipts);ro21i= rho1arr(jr,3,ipts)
           if (cplex_den==2) then
             ro11i=rho1arr(jr+1,1,ipts)+rho1arr(jr+1,4,ipts)
             ro22i=rho1arr(jr+1,1,ipts)-rho1arr(jr+1,4,ipts)
             ro12r=ro12r+rho1arr(jr+1,3,ipts);ro12i=ro12i+rho1arr(jr+1,2,ipts)
             ro21r=ro21r-rho1arr(jr+1,3,ipts);ro21i=ro21i+rho1arr(jr+1,2,ipts)
           else
             ro11i=zero;ro22i=zero
           end if
!          Real part
           ff(ir)=half*(v11r*ro11r+v11i*ro11i+v22r*ro22r+v22i*ro22i &
&           +v12r*ro12r+v12i*ro12i+v21r*ro21r+v21i*ro21i)
!          Imaginary part
           if (need_impart) gg(ir)=half*(v11r*ro11i-v11i*ro11r+v22r*ro22i-v22i*ro22r &
&           +v12r*ro12i-v12i*ro12r+v21r*ro21i-v21i*ro21r)
         end do
       end if ! cplex_vxc and cplex_den
     end if ! nspden

     ff(:)=ff(:)*pawrad%rad(:)**2
     call simp_gen(vxcrho,ff,pawrad)
     d2enxc=d2enxc+vxcrho*pawang%angwgth(ipts)
     ABI_DEALLOCATE(ff)

     if (need_impart) then
       gg(:)=gg(:)*pawrad%rad(:)**2
       call simp_gen(vxcrho,gg,pawrad)
       d2enxc_im=d2enxc_im+vxcrho*pawang%angwgth(ipts)
       ABI_DEALLOCATE(gg)
     end if

!    ----- End of the loop on npts (angular part)
   end do

 end if  ! option/=1

!Add the four*pi factor of the angular integration
 if (option/=1) then
   d2enxc=d2enxc*four_pi
   if (need_impart) d2enxc_im=d2enxc_im*four_pi
 end if

!Free memory
 ABI_DEALLOCATE(rho1arr)
 if (option==2) then
   ABI_DEALLOCATE(vxc1_)
 end if
 if (xclevel==2.and.option/=3) then
   ABI_DEALLOCATE(gxc1)
   ABI_DEALLOCATE(dylmdr)
 end if

 call timab(81,2,tsec)

 DBG_EXIT("COLL")

 end subroutine pawxc3_gga
!!***
