!!****f* ABINIT/fftw3_fourdp
!! NAME
!!  fftw3_fourdp
!!
!! FUNCTION
!! Driver routine for 3D FFT of lengths nx, ny, nz. Mainly used for densities or potentials.
!! FFT Transform is out-of-place
!!
!! COPYRIGHT
!! Copyright (C) 1998-2012 ABINIT group (MG)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~abinit/COPYING
!! or http://www.gnu.org/copyleft/gpl.txt .
!! For the initials of contributors, see ~abinit/doc/developers/contributors.txt .
!!
!! INPUTS
!! cplex=1 if fofr is real, 2 if fofr is complex
!! nx,ny,nz=Number of point along the three directions.
!! ldx,ldy,ldz=Leading dimension of the array.
!! ndat = Number of FFTS
!! isign= +1 : fofg(G) => fofr(R); 
!!        -1 : fofr(R) => fofg(G)
!! fofg(2,ldx*ldy*ldz*ndat)=The array to be transformed.
!! [fftw_flags]=Flags used to create the plan. They can be combined with the "+" operator. Defaults to FFTW_ESTIMATE.
!!
!! OUTPUT 
!! fofr(cplex,ldx*ldy*ldz*ndat)=The FFT of fofg
!!
!! PARENTS
!!      fourdp
!!
!! CHILDREN
!!      dfti_c2c_op,fftw3_c2c_op,sg_fft,xscal
!!
!! SOURCE

#if defined HAVE_CONFIG_H
#include "config.h"
#endif

#include "abi_common.h"

subroutine fftw3_fourdp(cplex,nx,ny,nz,ldx,ldy,ldz,ndat,isign,fofg,fofr,fftw_flags)

 use defs_basis
 use m_profiling
 use m_fftw3
 use m_errors

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
#undef ABI_FUNC
#define ABI_FUNC 'fftw3_fourdp'
!End of the abilint section

 implicit none 

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: cplex,nx,ny,nz,ldx,ldy,ldz,ndat,isign
 integer,optional,intent(in) :: fftw_flags
!arrays
 real(dp),intent(inout) :: fofg(2*ldx*ldy*ldz*ndat)
 real(dp),intent(inout) :: fofr(cplex*ldx*ldy*ldz*ndat)

!Local variables-------------------------------
!scalars
 integer :: my_flags

! *************************************************************************

 my_flags=FFTW_ESTIMATE; if (PRESENT(fftw_flags)) my_flags= fftw_flags

 select case (cplex)
   case (2) ! Complex to Complex.

     select case (isign)
       case (FFTW_BACKWARD)  ! +1
         call fftw3_many_dft_op(nx,ny,nz,ldx,ldy,ldz,ndat,isign,fofg,fofr,fftw_flags=my_flags)

       case (FFTW_FORWARD) ! -1
         call fftw3_many_dft_op(nx,ny,nz,ldx,ldy,ldz,ndat,isign,fofr,fofg,fftw_flags=my_flags)

       case default
         MSG_BUG("Wrong isign")
     end select 

   case (1) ! Real case.

     select case (isign)
       case (FFTW_FORWARD) ! -1; R --> G 
         call fftw3_r2c_op(nx,ny,nz,ldx,ldy,ldz,ndat,fofr,fofg,fftw_flags=my_flags)

       case (FFTW_BACKWARD) ! +1; G --> R
         call fftw3_c2r_op(nx,ny,nz,ldx,ldy,ldz,ndat,fofg,fofr,fftw_flags=my_flags)

       case default
         MSG_BUG("Wrong isign")
     end select

   case default 
     MSG_BUG(" Wrong value for cplex")
 end select 

end subroutine fftw3_fourdp
!!***

!----------------------------------------------------------------------

!!****f* ABINIT/fourdp_c2c_ip
!! NAME
!!  fourdp_c2c_ip
!!
!! FUNCTION
!!  In-place FFT transform of a complex array in double precision.
!!  It calls FFTW3 if available, fallback to fourdp if FFTW3 support is missing. 
!!
!! INPUTS
!! isign=sign of Fourier transform exponent: current convention uses
!!   +1 for transforming from G to r, 
!!   -1 for transforming from r to G.
!! nfft=number of FFT grid points.
!! ngfft(18)=contain all needed information about 3D FFT, see ~abinit/doc/input_variables/vargs.htm#ngfft
!!
!! SIDE EFFECTS
!!  ff(nfft*ndat)= Changed in output, filled with the FFT results.
!!
!! PARENTS
!!      calc_sig_ppm_eet,debug_tools,m_fft_prof,m_oscillators,m_paw_pwij
!!      m_shirley
!!
!! CHILDREN
!!      dfti_c2c_op,fftw3_c2c_op,sg_fft,xscal
!!
!! SOURCE

subroutine fourdp_c2c_ip(ngfft,nfft,ndat,isign,ff)

 use defs_basis
 use m_errors
 use m_profiling

 use m_blas,         only : xscal
 use m_fftw3,        only : fftw3_c2c_ip
 use m_dfti,         only : dfti_c2c_ip

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
#undef ABI_FUNC
#define ABI_FUNC 'fourdp_c2c_ip'
 use interfaces_53_ffts, except_this_one => fourdp_c2c_ip
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: isign,nfft,ndat
!arrays
 integer,intent(in) :: ngfft(18)
 complex(dpc),intent(inout) :: ff(nfft*ndat)

!Local variables ------------------------------
!scalars
 integer,parameter :: cplex2=2
 integer :: ifft,ix,iy,iz,nx,ny,nz,ldx,ldy,ldz,dat,fftalga
!arrays
 real(dp),allocatable :: arr(:,:,:,:),ftarr(:,:,:,:) 

! *************************************************************************

 nx =ngfft(1); ny=ngfft(2); nz=ngfft(3)
 ldx=nx;      ldy=ny;      ldz=nz ! No augmentation here.

 fftalga = ngfft(7)/100

 SELECT CASE (fftalga)

 CASE (FFT_FFTW3) ! Call the FFTW wrapper

 call fftw3_c2c_ip(nx,ny,nz,ldx,ldy,ldz,ndat,isign,ff)

 CASE (FFT_DFTI) 

 call dfti_c2c_ip(nx,ny,nz,ldx,ldy,ldz,ndat,isign,ff)

 CASE DEFAULT
!Fallback to sg_fft 
!1) we have to change shape and type complex -> double because we have an explicit interface. 
!2) transform is out-of-place here

 ABI_MALLOC(arr,  (2,ldx,ldy,ldz))
 ABI_MALLOC(ftarr,(2,ldx,ldy,ldz))
 
 do dat=1,ndat
!  
!  Copy input data in arr
   ifft = (dat-1) * nfft
   do iz=1,nz
     do iy=1,ny
       do ix=1,nx
         ifft = ifft + 1
         arr(1,ix,iy,iz) = DBLE (ff(ifft))
         arr(2,ix,iy,iz) = AIMAG(ff(ifft)) 
       end do 
     end do
   end do
   
   call sg_fft(ngfft(8),ldx,ldy,ldz,nx,ny,nz,arr,ftarr,DBLE(isign))
!  
!  Copy results stored in ftarr
   ifft = (dat-1) * nfft
   do iz=1,nz
     do iy=1,ny
       do ix=1,nx
         ifft = ifft + 1
         ff(ifft) = DCMPLX(ftarr(1,ix,iy,iz), ftarr(2,ix,iy,iz))
       end do 
     end do
   end do
   
 end do
 
 ABI_FREE(arr)
 ABI_FREE(ftarr)
 
 if (isign==-1) then  ! Scale the transform
   call xscal(nfft*ndat,one/nfft,ff,1)
 end if

 END SELECT

end subroutine fourdp_c2c_ip
!!***

!----------------------------------------------------------------------

!!****f* ABINIT/fourdp_c2c_op
!! NAME
!!  fourdp_c2c_op
!!
!! FUNCTION
!!  Out-of-place FFT transform of a complex array in double precision.
!!  It calls FFTW3 if available, fallback to fourdp if FFTW3 support is missing. 
!!
!! INPUTS
!! isign=sign of Fourier transform exponent: current convention uses
!!   +1 for transforming from G to r, 
!!   -1 for transforming from r to G.
!! nfft= number of FFT grid points.
!! ndat=number of FFTs
!! ngfft(18)=contain all needed information about 3D FFT, see ~abinit/doc/input_variables/vargs.htm#ngfft
!! ff(nfft*ndat)=The input array to be transformed.
!!
!! SIDE EFFECTS
!! OUTPUT 
!!  gg(nfft*ndat)= The FFT results.
!!
!! PARENTS
!!      calc_sig_ppm_eet,m_fft_prof,m_oscillators
!!
!! CHILDREN
!!      dfti_c2c_op,fftw3_c2c_op,sg_fft,xscal
!!
!! SOURCE

subroutine fourdp_c2c_op(ngfft,nfft,ndat,isign,ff,gg)

 use defs_basis
 use m_errors
 use m_profiling

 use m_blas,         only : xscal
 use m_fftw3,        only : fftw3_c2c_op
 use m_dfti,         only : dfti_c2c_op

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
#undef ABI_FUNC
#define ABI_FUNC 'fourdp_c2c_op'
 use interfaces_53_ffts, except_this_one => fourdp_c2c_op
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: isign,nfft,ndat
!arrays
 integer,intent(in) :: ngfft(18)
 complex(dpc),intent(in) :: ff(nfft*ndat)
 complex(dpc),intent(out) :: gg(nfft*ndat)

!Local variables ------------------------------
!scalars
 integer,parameter :: cplex2=2
 integer :: ifft,ix,iy,iz,nx,ny,nz,ldx,ldy,ldz,dat,fftalga
!arrays
 real(dp),allocatable :: arr(:,:,:,:),ftarr(:,:,:,:) 

! *************************************************************************

 fftalga = ngfft(7)/100

 nx=ngfft(1); ny=ngfft(2); nz=ngfft(3)
 ldx=nx;      ldy=ny;      ldz=nz ! no augmentation

 SELECT CASE (fftalga)

 CASE (FFT_FFTW3) ! call FFTW3 wrapper
 call fftw3_c2c_op(nx,ny,nz,ldx,ldy,ldz,ndat,isign,ff,gg)

 CASE (FFT_DFTI) 
 call dfti_c2c_op(nx,ny,nz,ldx,ldy,ldz,ndat,isign,ff,gg)

 CASE DEFAULT
!Fallback to sg_fft 
!(we have to change shape and type complex -> double because we have an explicit interface. 

 ABI_MALLOC(arr,   (2,ldx,ldy,ldz))
 ABI_MALLOC(ftarr, (2,ldx,ldy,ldz))

 do dat=1,ndat
!  
!  Copy input data in arr
   ifft = (dat-1) * nfft
   do iz=1,nz
     do iy=1,ny
       do ix=1,nx
         ifft = ifft + 1
         arr(1,ix,iy,iz) = DBLE (ff(ifft))
         arr(2,ix,iy,iz) = AIMAG(ff(ifft)) 
       end do 
     end do
   end do

   call sg_fft(ngfft(8),ldx,ldy,ldz,nx,ny,nz,arr,ftarr,DBLE(isign))
!  
!  Copy results stored in ftarr
   ifft = (dat-1) * nfft
   do iz=1,nz
     do iy=1,ny
       do ix=1,nx
         ifft = ifft + 1
         gg(ifft) = DCMPLX(ftarr(1,ix,iy,iz), ftarr(2,ix,iy,iz))
       end do 
     end do
   end do

 end do

 ABI_FREE(arr)
 ABI_FREE(ftarr)

 if (isign==-1) then  ! Scale the transform
   call xscal(nfft*ndat,one/nfft,gg,1)
 end if

 END SELECT

end subroutine fourdp_c2c_op
!!***

!----------------------------------------------------------------------
