!{\src2tex{textfont=tt}}
!!****m* ABINIT/m_abihist
!! NAME
!! m_abihist
!!
!! FUNCTION
!! This module contains definition the type abihist
!! and its related routines
!!
!! Datatypes :
!!
!! * abihist     : Historical record of atomic positions
!!                 forces and cell parameters
!!
!!
!! Subroutines :
!!
!! * abihist_ini
!! * abihist_nullify
!! * abihist_fin
!! * abihist_bcast
!! * abihist_compare
!! * hist2var
!! * var2hist
!! * vel2hist
!!
!!
!! COPYRIGHT
!! Copyright (C) 2001-2012 ABINIT group (XG)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~abinit/COPYING
!! or http://www.gnu.org/copyleft/gpl.txt .
!!
!! SOURCE

#if defined HAVE_CONFIG_H
#include "config.h"
#endif

#include "abi_common.h"

module m_abihist

 use m_profiling
 use defs_basis
 use m_xmpi

 implicit none

 private

 public ::             &
      abihist,         &
      abihist_ini,     &
      abihist_nullify, &
      abihist_fin,     &
      abihist_bcast,   &
      abihist_compare, &
      hist2var,        &
      var2hist,        &
      vel2hist


!----------------------------------------------------------------------
!!***

!!****t* m_abihist/abihist
!! NAME
!! abihist
!!
!! FUNCTION
!! This type has several vectors, and index scalars to store
!! a proper history of previous evaluations of forces and
!! stresses,velocities,positions and energies
!!
!! It contains:
!! * mxhist                  : Maximum size of history
!! * ihist                   : index of history
!! * histA(3,mxhist)         : Acell
!! * histE(mxhist)           : Energy
!! * histEk(mxhist)          : Ionic Kinetic Energy
!! * histEnt(mxhist)         : Entropy
!! * histT(mxhist)           : Time (Or iteration number for GO)
!! * histR(3,3,mxhist)       : Rprimd
!! * histS(6,mxhist)         : Strten
!! * histV(3,natom,mxhist)   : Velocity
!! * histXF(3,natom,4,mxhist): Xcart,Xred,Fcart,Fred
!!
!! NOTES
!! The vectors are not allocated because in some cases
!! not all the vectors are needed, in particular a history
!! of stresses is only needed if optcell/=0, and a history
!! of velocities is needed for ionmov==1
!!
!! Store acell, rprimd and strten even with optcell/=0
!! represent a waste of 12x (dp)[Usually 8 Bytes] per
!! iteration, the reason to store all the records is
!! because some routines (eg bfgs.F90) uses the metric (gmet)
!! for initialize the hessian and we need rprimd for that.
!!
!! SOURCE

 type abihist

! scalars
! Index of the last element on all records
    integer :: ihist
! Maximun size of the historical records
    integer :: mxhist
! Booleans to know if some arrays are changing
    logical :: isVused  ! If velocities are changing
    logical :: isARused ! If Acell and Rprimd are changing

! arrays
! Vector of (x,y,z)X(mxhist)
    real(dp), pointer :: histA(:,:)
! Vector of (mxhist) values of energy
    real(dp), pointer :: histE(:)
! Vector of (mxhist) values of ionic kinetic energy
    real(dp), pointer :: histEk(:)
! Vector of (mxhist) values of Entropy
    real(dp), pointer :: histEnt(:)
! Vector of (mxhist) values of time (relevant for MD calculations)
    real(dp), pointer :: histT(:)
! Vector of (x,y,z)X(x,y,z)X(mxhist)
    real(dp), pointer :: histR(:,:,:)
! Vector of (stress [6])X(mxhist)
    real(dp), pointer :: histS(:,:)
! Vector of (x,y,z)X(natom)X(mxhist) values of velocity
    real(dp), pointer :: histV(:,:,:)
! Vector of (x,y,z)X(natom)X(xcart,xred,fcart,fred)X(mxhist)
    real(dp), pointer :: histXF(:,:,:,:)

 end type abihist
!!***

!----------------------------------------------------------------------

contains  !=============================================================

!----------------------------------------------------------------------
!!***

!!****f* m_abihist/abihist_ini
!! NAME
!! abihist_ini
!!
!! FUNCTION
!! Initialize the hist structure for a given number
!! configurations and number of atoms
!!
!! INPUTS
!!  natom = Number of atoms per unitary cell
!!  mxhist = Maximal number of records store  
!!
!! OUTPUT
!!  hist <type(abihist)> = The hist to initialize
!!
!! SIDE EFFECTS
!!
!! PARENTS
!!      m_abiimages,mover
!!
!! CHILDREN
!!
!! NOTES
!!
!! SOURCE

subroutine abihist_ini(hist,natom,mxhist)


!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
#undef ABI_FUNC
#define ABI_FUNC 'abihist_ini'
!End of the abilint section

 implicit none

!Arguments ------------------------------------

 type(abihist) ,intent(out) :: hist
 integer       ,intent(in)  :: natom
 integer       ,intent(in)  :: mxhist

! ***************************************************************

!Initialize indexes
 hist%ihist=1
 hist%mxhist=mxhist

!Allocate all the histories
 ABI_ALLOCATE(hist%histA,(3,mxhist))
 ABI_ALLOCATE(hist%histE,(mxhist))
 ABI_ALLOCATE(hist%histEk,(mxhist))
 ABI_ALLOCATE(hist%histEnt,(mxhist))
 ABI_ALLOCATE(hist%histT,(mxhist))
 ABI_ALLOCATE(hist%histR,(3,3,mxhist))
 ABI_ALLOCATE(hist%histS,(6,mxhist))
 ABI_ALLOCATE(hist%histV,(3,natom,mxhist))
 ABI_ALLOCATE(hist%histXF,(3,natom,4,mxhist))

end subroutine abihist_ini
!!***

!----------------------------------------------------------------------

!!****f* m_abihist/abihist_nullify
!! NAME
!! abihist_nullify
!!
!! FUNCTION
!! Nullify all the pointers in a hist
!!
!! INPUTS
!!
!! OUTPUT
!!
!! SIDE EFFECTS
!!  hist <type(abihist)> = The hist to nullify
!!
!! PARENTS
!!      m_abiimages,mover
!!
!! CHILDREN
!!
!! NOTES
!!  At present 8 variables are present in abihist
!!  if a new variable is added in abihist it should
!!  be added also for nullify here
!!
!! SOURCE

subroutine abihist_nullify(hist)


!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
#undef ABI_FUNC
#define ABI_FUNC 'abihist_nullify'
!End of the abilint section

 implicit none

!Arguments ------------------------------------

 type(abihist),intent(inout) :: hist

! ***************************************************************

! Vector of (x,y,z)X(mxhist)
 nullify(hist%histA)
! Vector of (mxhist) values of energy
 nullify(hist%histE)
! Vector of (mxhist) values of ionic kinetic energy
 nullify(hist%histEk)
! Vector of (mxhist) values of Entropy
 nullify(hist%histEnt)
! Vector of (mxhist) values of time (relevant for MD calculations)
 nullify(hist%histT)
! Vector of (x,y,z)X(x,y,z)X(mxhist)
 nullify(hist%histR)
! Vector of (stress [6])X(mxhist)
 nullify(hist%histS)
! Vector of (x,y,z)X(natom)X(mxhist) values of velocity
 nullify(hist%histV)
! Vector of (x,y,z)X(natom)X(xcart,xred,fcart,fred)X(mxhist)
 nullify(hist%histXF)

end subroutine abihist_nullify
!!***

!----------------------------------------------------------------------

!!****f* m_abihist/abihist_fin
!! NAME
!! abihist_fin
!!
!! FUNCTION
!! Deallocate all the pointers in a hist
!!
!! INPUTS
!!
!! OUTPUT
!!
!! SIDE EFFECTS
!!  hist <type(abihist)> = The hist to deallocate
!!
!! PARENTS
!!      m_abihist,m_abiimages,mover
!!
!! CHILDREN
!!
!! NOTES
!!  At present 8 variables are present in abihist
!!  if a new variable is added in abihist it should
!!  be added also for deallocate here
!!
!! SOURCE

subroutine abihist_fin(hist)


!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
#undef ABI_FUNC
#define ABI_FUNC 'abihist_fin'
!End of the abilint section

 implicit none

!Arguments ------------------------------------

 type(abihist),intent(inout) :: hist

! ***************************************************************

! Vector of (x,y,z)X(mxhist)
 if (associated(hist%histA))  then
   ABI_DEALLOCATE(hist%histA)
 end if
! Vector of (mxhist) values of energy
 if (associated(hist%histE))  then
   ABI_DEALLOCATE(hist%histE)
 end if
! Vector of (mxhist) values of ionic kinetic energy
 if (associated(hist%histEk))  then
   ABI_DEALLOCATE(hist%histEk)
 end if
! Vector of (mxhist) values of Entropy
 if (associated(hist%histEnt))  then
   ABI_DEALLOCATE(hist%histEnt)
 end if
! Vector of (mxhist) values of time (relevant for MD calculations)
 if (associated(hist%histT))  then
   ABI_DEALLOCATE(hist%histT)
 end if
! Vector of (x,y,z)X(x,y,z)X(mxhist)
 if (associated(hist%histR))  then
   ABI_DEALLOCATE(hist%histR)
 end if
! Vector of (stress [6])X(mxhist)
 if (associated(hist%histS))  then
   ABI_DEALLOCATE(hist%histS)
 end if
! Vector of (x,y,z)X(natom)X(mxhist) values of velocity
 if (associated(hist%histV))  then
   ABI_DEALLOCATE(hist%histV)
 end if
! Vector of (x,y,z)X(natom)X(xcart,xred,fcart,fred)X(mxhist)
 if (associated(hist%histXF))  then
   ABI_DEALLOCATE(hist%histXF)
 end if

end subroutine abihist_fin
!!***

!----------------------------------------------------------------------

!!****f* m_abihist/abihist_bcast
!! NAME
!! abihist_bcast
!!
!! FUNCTION
!! Broadcast a hist datastructure (from a root process to all others)
!!
!! INPUTS
!!  master=ID of the sending node in spaceComm
!!  spaceComm=MPI Communicator
!!
!! OUTPUT
!!
!! SIDE EFFECTS
!!  hist <type(abihist)> = The hist to broadcast
!!
!! PARENTS
!!      mover
!!
!! CHILDREN
!!
!! NOTES
!!  At present 8 variables are present in abihist
!!  if a new variable is added in abihist it should
!!  be added also for broadcast here
!!
!! SOURCE

subroutine abihist_bcast(hist,master,spaceComm)


!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
#undef ABI_FUNC
#define ABI_FUNC 'abihist_bcast'
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: master,spaceComm
 type(abihist),intent(inout) :: hist

!Local variables-------------------------------
!scalars
 integer :: bufsize,ierr,indx,nproc,rank
 integer :: sizeA,sizeA1,sizeA2
 integer :: sizeE,sizeEk,sizeEnt,sizeT
 integer :: sizeR,sizeR1,sizeR2,sizeR3
 integer :: sizeS,sizeS1,sizeS2
 integer :: sizeV,sizeV1,sizeV2,sizeV3
 integer :: sizeXF,sizeXF1,sizeXF2,sizeXF3,sizeXF4
!arrays
 integer,allocatable :: buffer_i(:)
 real(dp),allocatable :: buffer_r(:)

! ***************************************************************

 ierr=0
 nproc=xcomm_size(spaceComm)
 if (nproc<=1) return

 rank=xcomm_rank(spaceComm)

!=== Broadcast integers and logicals
 ABI_ALLOCATE(buffer_i,(4))
 if (rank==master) then
   buffer_i(1)=hist%ihist
   buffer_i(2)=hist%mxhist
   buffer_i(3)=0;if (hist%isVused)  buffer_i(3)=1
   buffer_i(4)=0;if (hist%isARused) buffer_i(4)=1
 end if
 call xcast_mpi(buffer_i,master,spaceComm,ierr)
 if (rank/=master) then
   hist%ihist=buffer_i(1)
   hist%mxhist=buffer_i(2)
   hist%isVused =(buffer_i(3)==1)
   hist%isARused=(buffer_i(4)==1)
 end if
 ABI_DEALLOCATE(buffer_i)

!If history is empty, return
 if (hist%mxhist==0.or.hist%ihist==0) return

!=== Broadcast sizes of arrays
ABI_ALLOCATE(buffer_i,(18))
 if (rank==master) then
   sizeA1=size(hist%histA,1);sizeA2=size(hist%histA,2)
   sizeE=size(hist%histE,1);sizeEk=size(hist%histEk,1);
   sizeEnt=size(hist%histEnt,1);sizeT=size(hist%histT,1)
   sizeR1=size(hist%histR,1);sizeR2=size(hist%histR,2);sizeR3=size(hist%histR,3)
   sizeS1=size(hist%histS,1);sizeS2=size(hist%histS,2)
   sizeV1=size(hist%histV,1);sizeV2=size(hist%histV,2);sizeV3=size(hist%histV,3)
   sizeXF1=size(hist%histXF,1);sizeXF2=size(hist%histXF,2);sizeXF3=size(hist%histXF,3)
   sizeXF4=size(hist%histXF,4)
   buffer_i(1)=sizeA1  ;buffer_i(2)=sizeA2
   buffer_i(3)=sizeE   ;buffer_i(4)=sizeEk
   buffer_i(5)=sizeEnt ;buffer_i(6)=sizeT 
   buffer_i(7)=sizeR1  ;buffer_i(8)=sizeR2  
   buffer_i(9)=sizeR3  ;buffer_i(10)=sizeS1 
   buffer_i(11)=sizeS2 ;buffer_i(12)=sizeV1 
   buffer_i(13)=sizeV2 ;buffer_i(14)=sizeV3 
   buffer_i(15)=sizeXF1;buffer_i(16)=sizeXF2
   buffer_i(17)=sizeXF3;buffer_i(18)=sizeXF4
 end if
 call xcast_mpi(buffer_i,master,spaceComm,ierr)
 if (rank/=master) then
   sizeA1 =buffer_i(1) ;sizeA2 =buffer_i(2)
   sizeE  =buffer_i(3) ;sizeEk =buffer_i(4)
   sizeEnt =buffer_i(5);sizeT  =buffer_i(6) 
   sizeR1 =buffer_i(7) ;sizeR2 =buffer_i(8) 
   sizeR3 =buffer_i(9) ;sizeS1 =buffer_i(10)
   sizeS2 =buffer_i(11);sizeV1 =buffer_i(12)
   sizeV2 =buffer_i(13);sizeV3 =buffer_i(14)
   sizeXF1=buffer_i(15);sizeXF2=buffer_i(16)
   sizeXF3=buffer_i(17);sizeXF4=buffer_i(18)   
   
 end if
 ABI_DEALLOCATE(buffer_i)

!=== Broadcast reals
 sizeA=sizeA1*sizeA2;sizeR=sizeR1*sizeR2*sizeR3;sizeS=sizeS1*sizeS2
 sizeV=sizeV1*sizeV2*sizeV3;sizeXF=sizeXF1*sizeXF2*sizeXF3*sizeXF4
 bufsize=sizeA+sizeE+sizeEk+sizeEnt+sizeT+sizeR+sizeS+sizeV+sizeXF
 ABI_ALLOCATE(buffer_r,(bufsize))
 if (rank==master) then
   indx=0
   buffer_r(indx+1:indx+sizeA)=reshape(hist%histA(1:sizeA1,1:sizeA2),(/sizeA/))
   indx=indx+sizeA
   buffer_r(indx+1:indx+sizeE)=hist%histE(1:sizeE)
   indx=indx+sizeE
   buffer_r(indx+1:indx+sizeEk)=hist%histEk(1:sizeEk)
   indx=indx+sizeEk
   buffer_r(indx+1:indx+sizeEnt)=hist%histEnt(1:sizeEnt)
   indx=indx+sizeEnt
   buffer_r(indx+1:indx+sizeT)=hist%histT(1:sizeT)
   indx=indx+sizeT
   buffer_r(indx+1:indx+sizeR)=reshape(hist%histR(1:sizeR1,1:sizeR2,1:sizeR3),(/sizeR/))
   indx=indx+sizeR
   buffer_r(indx+1:indx+sizeS)=reshape(hist%histS(1:sizeS1,1:sizeS2),(/sizeS/))
   indx=indx+sizeS
   buffer_r(indx+1:indx+sizeV)=reshape(hist%histV(1:sizeV1,1:sizeV2,1:sizeV3),(/sizeV/))
   indx=indx+sizeV
   buffer_r(indx+1:indx+sizeXF)=reshape(hist%histXF(1:sizeXF1,1:sizeXF2,1:sizeXF3,1:sizeXF4),(/sizeXF/))
 else
   call abihist_fin(hist)
   ABI_ALLOCATE(hist%histA,(sizeA1,sizeA2))
   ABI_ALLOCATE(hist%histE,(sizeE))
   ABI_ALLOCATE(hist%histEk,(sizeEk))
   ABI_ALLOCATE(hist%histEnt,(sizeEnt))
   ABI_ALLOCATE(hist%histT,(sizeT))
   ABI_ALLOCATE(hist%histR,(sizeR1,sizeR2,sizeR3))
   ABI_ALLOCATE(hist%histS,(sizeS1,sizeS2))
   ABI_ALLOCATE(hist%histV,(sizeV1,sizeV2,sizeV3))
   ABI_ALLOCATE(hist%histXF,(sizeXF1,sizeXF2,sizeXF3,sizeXF4))
 end if
 call xcast_mpi(buffer_r,master,spaceComm,ierr)
 if (rank/=master) then
   indx=0
   hist%histA(1:sizeA1,1:sizeA2)=reshape(buffer_r(indx+1:indx+sizeA),(/sizeA1,sizeA2/))
   indx=indx+sizeA
   hist%histE(1:sizeE)=buffer_r(indx+1:indx+sizeE)
   indx=indx+sizeE
   hist%histEk(1:sizeEk)=buffer_r(indx+1:indx+sizeEk)
   indx=indx+sizeEk
   hist%histEnt(1:sizeEnt)=buffer_r(indx+1:indx+sizeEnt)
   indx=indx+sizeEnt
   hist%histT(1:sizeT)=buffer_r(indx+1:indx+sizeT)
   indx=indx+sizeT
   hist%histR(1:sizeR1,1:sizeR2,1:sizeR3)=reshape(buffer_r(indx+1:indx+sizeR),(/sizeR1,sizeR2,sizeR3/))
   indx=indx+sizeR
   hist%histS(1:sizeS1,1:sizeS2)=reshape(buffer_r(indx+1:indx+sizeS),(/sizeS1,sizeS2/))
   indx=indx+sizeS
   hist%histV(1:sizeV1,1:sizeV2,1:sizeV3)=reshape(buffer_r(indx+1:indx+sizeV),(/sizeV1,sizeV2,sizeV3/))
   indx=indx+sizeV
   hist%histXF(1:sizeXF1,1:sizeXF2,1:sizeXF3,1:sizeXF4)=reshape(buffer_r(indx+1:indx+sizeXF), &
&                                                       (/sizeXF1,sizeXF2,sizeXF3,sizeXF4/))
 end if
 ABI_DEALLOCATE(buffer_r)

end subroutine abihist_bcast
!!***

!----------------------------------------------------------------------

!{\src2tex{textfont=tt}}
!!****f* m_abihist/var2hist
!!
!! NAME
!! var2hist
!!
!! FUNCTION
!! Set the values of the history "hist"
!! with the values of xcart, xred, acell and rprimd
!!
!! COPYRIGHT
!! Copyright (C) 1998-2012 ABINIT group (DCA, XG, GMR)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~abinit/COPYING
!! or http://www.gnu.org/copyleft/gpl.txt .
!! For the initials of contributors,
!! see ~abinit/doc/developers/contributors.txt .
!!
!! INPUTS
!! natom = number of atoms
!! xred(3,natom) = reduced dimensionless atomic
!!                           coordinates
!! xcart(3,natom)= cartesian dimensional atomic
!!                           coordinates (bohr)
!! acell(3)    = length scales of primitive translations (bohr)
!! rprimd(3,3) = dimensionlal real space primitive translations
!!               (bohr)
!!
!! OUTPUT
!!
!! SIDE EFFECTS
!! hist<type abihist>=Historical record of positions, forces
!!      |                    acell, stresses, and energies,
!!      |                    contains:
!!      | mxhist:  Maximun number of records
!!      | ihist:   Index of present record
!!      | histA:   Historical record of acell(A) and rprimd(R)
!!      | histE:   Historical record of energy(E)
!!      | histEk:  Historical record of kinetic energy(Ek)
!!      | histEnt: Historical record of Entropy
!!      | histT:   Historical record of time(T)
!!      | histR:   Historical record of rprimd(R)
!!      | histS:   Historical record of strten(S)
!!      | histV:   Historical record of velocity(V)
!!      | histXF:  Historical record of positions(X) and forces(F)
!!
!! PARENTS
!!      mover,pred_bfgs,pred_delocint,pred_diisrelax,pred_isokinetic
!!      pred_isothermal,pred_langevin,pred_moldyn,pred_nose,pred_srkna14
!!      pred_steepdesc,pred_verlet
!!
!! CHILDREN
!!
!! SOURCE

subroutine var2hist(acell,hist,natom,rprim,rprimd,xcart,xred,zDEBUG)

!Arguments ------------------------------------
!scalars

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
#undef ABI_FUNC
#define ABI_FUNC 'var2hist'
 use interfaces_41_geometry
!End of the abilint section

integer,intent(in) :: natom
type(abihist),intent(inout) :: hist
logical,intent(in) :: zDEBUG
!arrays
real(dp),intent(in) :: acell(3)
real(dp),intent(in) :: rprimd(3,3)
real(dp),intent(in) :: rprim(3,3)
real(dp),intent(in) :: xcart(3,natom)
real(dp),intent(in) :: xred(3,natom)

!Local variables-------------------------------
!scalars
integer :: kk

! *************************************************************

 hist%histXF(:,:,1,hist%ihist)=xcart(:,:)
 hist%histXF(:,:,2,hist%ihist)=xred(:,:)
 hist%histA(:,hist%ihist)     =acell(:)
 hist%histR(:,:,hist%ihist)   =rprimd(:,:)

 if(zDEBUG)then
   write (std_out,*) 'Atom positions and cell parameters '
   write (std_out,*) 'ihist: ',hist%ihist
   write (std_out,*) 'xcart:'
   do kk=1,natom
     write (std_out,*) xcart(:,kk)
   end do
   write (std_out,*) 'xred:'
   do kk=1,natom
     write (std_out,*) xred(:,kk)
   end do
   write(std_out,*) 'rprim:'
   do kk=1,3
     write(std_out,*) rprim(:,kk)
   end do
   write(std_out,*) 'rprimd:'
   do kk=1,3
     write(std_out,*) rprimd(:,kk)
   end do
   write(std_out,*) 'acell:'
   write(std_out,*) acell(:)
   call chkrprimd(acell,rprim,rprimd,std_out)
 end if

end subroutine var2hist
!!***

!----------------------------------------------------------------------

!{\src2tex{textfont=tt}}
!!****f* m_abihist/hist2var
!!
!! NAME
!! hist2var
!!
!! FUNCTION
!! Set the values of xcart, xred, acell and rprimd
!! with the values that comes from the history "hist"
!!
!! COPYRIGHT
!! Copyright (C) 1998-2012 ABINIT group (DCA, XG, GMR)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~abinit/COPYING
!! or http://www.gnu.org/copyleft/gpl.txt .
!! For the initials of contributors,
!! see ~abinit/doc/developers/contributors.txt .
!!
!! INPUTS
!! natom = Number of atoms
!! hist<type abihist>=Historical record of positions, forces
!!                           acell, stresses, and energies,
!! zDebug = If true some output will be printed
!!
!! OUTPUT
!!  xred(3,natom) = reduced dimensionless atomic
!!                           coordinates
!!  xcart(3,natom)= cartesian dimensional atomic
!!                           coordinates (bohr)
!!  acell(3)    = length scales of primitive translations (bohr)
!!  rprim(3,3) = dimensionless real space primitive translations
!!  rprimd(3,3) = dimensional real space primitive translations
!!                (bohr)
!!
!! SIDE EFFECTS
!!
!! PARENTS
!!      mover,pred_bfgs,pred_delocint,pred_diisrelax,pred_isokinetic
!!      pred_isothermal,pred_langevin,pred_moldyn,pred_nose,pred_srkna14
!!      pred_steepdesc,pred_verlet
!!
!! CHILDREN
!!
!! SOURCE

subroutine hist2var(acell,hist,natom,rprim,rprimd,xcart,xred,zDEBUG)

!Arguments ------------------------------------
!scalars

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
#undef ABI_FUNC
#define ABI_FUNC 'hist2var'
 use interfaces_41_geometry
!End of the abilint section

integer,intent(in) :: natom
type(abihist),intent(in) :: hist
logical,intent(in) :: zDEBUG
!arrays
real(dp),intent(out) :: acell(3)
real(dp),intent(out) :: rprimd(3,3)
real(dp),intent(out) :: rprim(3,3)
real(dp),intent(out) :: xcart(3,natom)
real(dp),intent(out) :: xred(3,natom)

!Local variables-------------------------------
!scalars
integer :: jj,kk

! *************************************************************

 xcart(:,:) =hist%histXF(:,:,1,hist%ihist)
 xred(:,:)  =hist%histXF(:,:,2,hist%ihist)
 acell(:)   =hist%histA(:,hist%ihist)
 rprimd(:,:)=hist%histR(:,:,hist%ihist)

!Compute rprim from rprimd and acell
 do kk=1,3
   do jj=1,3
     rprim(jj,kk)=rprimd(jj,kk)/acell(kk)
   end do
 end do

 if(zDEBUG)then
   write (std_out,*) 'Atom positions and cell parameters '
   write (std_out,*) 'ihist: ',hist%ihist
   write (std_out,*) 'xcart:'
   do kk=1,natom
     write (std_out,*) xcart(:,kk)
   end do
   write (std_out,*) 'xred:'
   do kk=1,natom
     write (std_out,*) xred(:,kk)
   end do
   write(std_out,*) 'rprim:'
   do kk=1,3
     write(std_out,*) rprim(:,kk)
   end do
   write(std_out,*) 'rprimd:'
   do kk=1,3
     write(std_out,*) rprimd(:,kk)
   end do
   write(std_out,*) 'acell:'
   write(std_out,*) acell(:)
   call chkrprimd(acell,rprim,rprimd,std_out)
 end if

end subroutine hist2var
!!***

!----------------------------------------------------------------------

!{\src2tex{textfont=tt}}
!!****f* m_abihist/vel2hist
!!
!! NAME
!! vel2hist
!!
!! FUNCTION
!! Set the values of the history "hist"
!! related with velocities, ie
!! The array of velocities and Kinetic Energy
!!
!! COPYRIGHT
!! Copyright (C) 1998-2012 ABINIT group (DCA, XG, GMR)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~abinit/COPYING
!! or http://www.gnu.org/copyleft/gpl.txt .
!! For the initials of contributors,
!! see ~abinit/doc/developers/contributors.txt .
!!
!! INPUTS
!! amass = mass of the atoms
!! natom = number of atoms
!! vel(3,natom)= Velocities of the atoms
!!
!! OUTPUT
!!
!! SIDE EFFECTS
!! hist<type abihist>=Historical record of positions, forces
!!                           acell, stresses, and energies,
!!
!! PARENTS
!!      mover
!!
!! CHILDREN
!!
!! SOURCE

subroutine vel2hist(amass,hist,natom,vel)

!Arguments ------------------------------------
!scalars

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
#undef ABI_FUNC
#define ABI_FUNC 'vel2hist'
!End of the abilint section

integer,intent(in) :: natom
type(abihist),intent(inout) :: hist
!arrays
real(dp),intent(in) :: vel(3,natom)
real(dp),intent(in) :: amass(natom)

!Local variables-------------------------------
!scalars
integer :: ii,jj
real(dp) :: ekin

! *************************************************************

!Store the velocities
 hist%histV(:,:,hist%ihist)=vel(:,:)

!Compute the Ionic Kinetic energy
 ekin=0.0
 do ii=1,natom
   do jj=1,3
     ekin=ekin+0.5_dp*amass(ii)*vel(jj,ii)**2
   end do
 end do

!Store the Ionic Kinetic Energy
 hist%histEk(hist%ihist)=ekin

end subroutine vel2hist
!!***

!----------------------------------------------------------------------

!{\src2tex{textfont=tt}}
!!****f* ABINIT/abihist_compare
!! NAME
!! abihist_compare
!!
!! FUNCTION
!! Compare 2 HIST records
!!
!! COPYRIGHT
!! Copyright (C) 1998-2012 ABINIT group (DCA, XG, GMR, SE)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~abinit/COPYING
!! or http://www.gnu.org/copyleft/gpl.txt .
!!
!! INPUTS
!!  hist_in <type(abihist)>
!!  tolerance
!!
!! OUTPUT
!!  similar= 1 the records are consitent
!!           0 the records are not consitent
!!
!! SIDE EFFECTS
!!  hist_out <type(abihist)>
!!
!! PARENTS
!!      mover
!!
!! CHILDREN
!!
!! SOURCE


subroutine abihist_compare(hist_in,hist_out,natom,similar,tolerance)

!Arguments ------------------------------------
!scalars

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
#undef ABI_FUNC
#define ABI_FUNC 'abihist_compare'
!End of the abilint section

integer,intent(in) :: natom
integer,intent(out) :: similar
real(dp),intent(in) :: tolerance
type(abihist),intent(in) :: hist_in
type(abihist),intent(inout) :: hist_out

!arrays

!Local variables-------------------------------
!scalars
integer :: kk,jj
real(dp) :: maxdiff,diff
real(dp) :: x,y

! ***************************************************************

 similar=1

 write(std_out,*) 'Using values from history, iteration:',hist_in%ihist
 write(std_out,*) 'Differences between present history and values stored'
 write(std_out,*) 'on the previous history.(Relative difference)'

 x=hist_out%histXF(1,1,1,hist_out%ihist)
 y=hist_in%histXF(1,1,1,hist_in%ihist)
 maxdiff=2*abs(x-y)/(abs(x)+abs(y))
 do kk=1,natom
   do jj=1,3
     x=hist_out%histXF(jj,kk,1,hist_out%ihist)
     y=hist_in%histXF(jj,kk,1,hist_in%ihist)
     diff=2*abs(x-y)/(abs(x)+abs(y))
     if (diff>maxdiff) maxdiff=diff
   end do
 end do
 write(std_out,'(a,e12.5)') 'xcart:    ',maxdiff
 if (maxdiff>tolerance) similar=0


 x=hist_out%histXF(1,1,2,hist_out%ihist)
 y=hist_in%histXF(1,1,2,hist_in%ihist)
 maxdiff=2*abs(x-y)/(abs(x)+abs(y))
 do kk=1,natom
   do jj=1,3
     x=hist_out%histXF(jj,kk,2,hist_out%ihist)
     y=hist_in%histXF(jj,kk,2,hist_in%ihist)
     diff=2*abs(x-y)/(abs(x)+abs(y))
     if (diff>maxdiff) maxdiff=diff
   end do
 end do
 write(std_out,'(a,e12.5)') 'xred:     ',maxdiff
 if (maxdiff>tolerance) similar=0


 x=hist_out%histR(1,1,hist_out%ihist)
 y=hist_in%histR(1,1,hist_in%ihist)
 maxdiff=2*abs(x-y)/(abs(x)+abs(y))
 do kk=1,3
   do jj=1,3
     x=hist_out%histR(jj,kk,hist_out%ihist)
     y=hist_in%histR(jj,kk,hist_in%ihist)
     diff=2*abs(x-y)/(abs(x)+abs(y))
     if (diff>maxdiff) maxdiff=diff
   end do
 end do
 write(std_out,'(a,e12.5)') 'rprimd:   ',maxdiff
 if (maxdiff>tolerance) similar=0


 x=hist_out%histA(1,hist_out%ihist)
 y=hist_in%histA(1,hist_in%ihist)
 maxdiff=2*abs(x-y)/(abs(x)+abs(y))
 do kk=1,3
   x=hist_out%histA(kk,hist_out%ihist)
   y=hist_in%histA(kk,hist_in%ihist)
   diff=2*abs(x-y)/(abs(x)+abs(y))
   if (diff>maxdiff) maxdiff=diff
 end do
 write(std_out,'(a,e12.5)') 'acell:    ',maxdiff
 if (maxdiff>tolerance) similar=0

 if (similar==1) then

   hist_out%histV(:,:,hist_out%ihist)=   hist_in%histV(:,:,hist_in%ihist)
   hist_out%histE(hist_out%ihist)=       hist_in%histE(hist_in%ihist)
   hist_out%histEk(hist_out%ihist)=      hist_in%histEk(hist_in%ihist)
   hist_out%histEnt(hist_out%ihist)=     hist_in%histEnt(hist_in%ihist)
   hist_out%histT(hist_out%ihist)=       hist_in%histT(hist_in%ihist)
   hist_out%histXF(:,:,3,hist_out%ihist)=hist_in%histXF(:,:,3,hist_in%ihist)
   hist_out%histXF(:,:,4,hist_out%ihist)=hist_in%histXF(:,:,4,hist_in%ihist)
   hist_out%histS(:,hist_out%ihist)=     hist_in%histS(:,hist_in%ihist)

 end if

end subroutine abihist_compare
!!***

end module m_abihist
!!***
