!{\src2tex{textfont=tt}}
!!****m* ABINIT/defs_abitypes
!! NAME
!! defs_abitypes
!!
!! FUNCTION
!! This module contains definitions of high-level structured datatypes for the
!! ABINIT package.
!!
!! If you are sure a new high-level structured datatype is needed,
!! write it here, and DOCUMENT it properly (not all datastructure here are
!! well documented, it is a shame ...).
!! Do not forget : you will likely be the major winner if you document
!! properly.
!! Proper documentation of a structured datatype means :
!!  (1) Mention it in the list just below
!!  (2) Describe it in the NOTES section
!!  (3) Put it in alphabetical order in the the main section of this module
!!  (4) Document each of its records, except if they are described elsewhere
!!      (this exception is typically the case of the dataset associated with
!!      input variables, for which there is a help file)
!!  (5) Declare variables on separated lines in order to reduce the occurence of bzr conflicts.
!!
!! List of datatypes :
!! * aim_dataset_type : the "dataset" for aim
!! * anaddb_dataset_type : the "dataset" for anaddb
!! * bandfft_kpt_type : the "dataset" for triple band-fft-kpt parallelization
!! * datafiles_type: gather all the variables related to files
!! * dataset_type : the "dataset" for the main abinit code
!! * MPI_type : the data related to MPI parallelization
!! * hdr_type : the header of wf, den and pot files
!!
!! COPYRIGHT
!! Copyright (C) 2001-2012 ABINIT group (XG)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~abinit/COPYING
!! or http://www.gnu.org/copyleft/gpl.txt .
!!
!! SOURCE

#if defined HAVE_CONFIG_H
#include "config.h"
#endif

#include "abi_common.h"


module defs_abitypes

 use defs_basis
 use m_pawrhoij, only : pawrhoij_type
 use m_profiling

 implicit none

!Structures
!!***

!!****t* defs_abitypes/aim_dataset_type
!! NAME
!! aim_dataset_type
!!
!! FUNCTION
!! The aim_dataset_type structured datatype
!! gathers all the input variables for the aim code
!!
!! SOURCE

 type aim_dataset_type

! WARNING : if you modify this datatype, please check whether there might be creation/destruction/copy routines,
! declared in another part of ABINIT, that might need to take into account your modification.

! Variables should be declared on separated lines in order to reduce the occurence of bzr conflicts.

! Since all these input variables are described in the aim_help.html
! file, they are not described in length here ...

! Integer
  integer :: crit
  integer :: denout
  integer :: dltyp
  integer :: gpsurf
  integer :: irho
  integer :: ivol
  integer :: lapout
  integer :: nsa
  integer :: nsb
  integer :: nsc

  integer :: batom  !! Warning : corresponds to the input variable atom
  integer :: foll   !! Warning : corresponds to the input variable follow
  integer :: isurf  !! Warning : corresponds to the input variable surf
  integer :: irsur  !! Warning : corresponds to the input variable rsurf
  integer :: nph    !! Warning : corresponds to the input variable nphi
  integer :: npt    !! Warning : corresponds to the input variable inpt
  integer :: nth    !! Warning : corresponds to the input variable ntheta
  integer :: plden  !! Warning : not documented in help file ?!

  integer :: ngrid(3)

! Real
  real(dp) :: atrad
  real(dp) :: coff1
  real(dp) :: coff2
  real(dp) :: dpclim
  real(dp) :: folstp
  real(dp) :: lgrad
  real(dp) :: lgrad2
  real(dp) :: lstep
  real(dp) :: lstep2
  real(dp) :: maxatd
  real(dp) :: maxcpd
  real(dp) :: phimax
  real(dp) :: phimin

  real(dp) :: dr0    !! Warning : correspond to the input variable radstp
  real(dp) :: phi0   !! Warning : correspond to the input variable rsurdir(2)
  real(dp) :: rmin   !! Warning : correspond to the input variable ratmin
  real(dp) :: th0    !! Warning : correspond to the input variable rsurdir(1)
  real(dp) :: themax !! Warning : correspond to the input variable thetamax
  real(dp) :: themin !! Warning : correspond to the input variable thetamin

  real(dp) :: foldep(3)
  real(dp) :: scal(3)
  real(dp) :: vpts(3,4)

 end type aim_dataset_type
!!***

!----------------------------------------------------------------------

!!****t* defs_abitypes/anaddb_dataset_type
!! NAME
!! anaddb_dataset_type
!!
!! FUNCTION
!! The anaddb_dataset_type structured datatype
!! gather all the input variables for the anaddb code.
!!
!! SOURCE

 type anaddb_dataset_type

! WARNING : if you modify this datatype, please check whether there might be creation/destruction/copy routines,
! declared in another part of ABINIT, that might need to take into account your modification.

! Variables should be declared on separated lines in order to reduce the occurence of bzr conflicts.


! Since all these input variables are described in the anaddb_help.html
! file, they are not described in length here ...
! Integer
  integer :: alphon
  integer :: asr
  integer :: brav
  integer :: chneut
  integer :: dieflag
  integer :: dipdip
  integer :: dossum
  integer :: ep_scalprod
  integer :: eivec
  integer :: elaflag
  integer :: elphflag
  integer :: enunit
  integer :: gkk2write
  integer :: gkk_rptwrite
  integer :: gkqwrite
  integer :: iavfrq
  integer :: ifcana
  integer :: ifcflag
  integer :: ifcout
  integer :: ifltransport
  integer :: instrflag
  integer :: natfix
  integer :: natifc
  integer :: natom
  integer :: natprj_bs
  integer :: nchan
  integer :: nfreq
  integer :: ngrids
  integer :: nlflag
  integer :: nph1l
  integer :: nph2l
  integer :: nqpath
  integer :: nqshft
  integer :: nsphere
  integer :: nstrfix
  integer :: ntemper
  integer :: nwchan
  integer :: outscphon
  integer :: piezoflag
  integer :: polflag
  integer :: prtdos
  integer :: prtmbm
  integer :: prtfsurf
  integer :: prtnest
  integer :: prtsrlr  ! print the short-range/long-range decomposition of phonon freq.
  integer :: qrefine
  integer :: ramansr
  integer :: relaxat
  integer :: relaxstr
  integer :: rfmeth
  integer :: selectz
  integer :: symdynmat
  integer :: telphint
  integer :: thmflag
  integer :: qgrid_type
  integer :: ep_b_min
  integer :: ep_b_max
  integer :: ep_int_gkk
  integer :: ep_keepbands
  integer :: ep_nqpt
  integer :: ep_prt_yambo
  integer :: symgkq
  integer :: use_k_fine

  integer :: ngqpt(9)             ! ngqpt(9) instead of ngqpt(3) is needed in wght9.f
  integer :: istrfix(6)
  integer :: ng2qpt(3)
  integer :: kptrlatt(3,3)
  integer :: kptrlatt_fine(3,3)

! Real(dp)
  real(dp) :: a2fsmear
  real(dp) :: dosdeltae
  real(dp) :: dossmear
  real(dp) :: dostol
  real(dp) :: elphsmear
  real(dp) :: elph_fermie
  real(dp) :: ep_extrael
  real(dp) :: freeze_displ
  real(dp) :: frmax
  real(dp) :: frmin
  real(dp) :: temperinc
  real(dp) :: tempermin
  real(dp) :: thmtol
  real(dp) :: mustar
  real(dp) :: rifcsph

  real(dp) :: q1shft(3,4)
  real(dp) :: q2shft(3)
  real(dp) :: targetpol(3)

! Integer pointers
  integer, pointer :: atifc(:)
   ! atifc(natom) WARNING : there is a transformation of this input variable, in chkin9
   ! This should be changed ...

  integer, pointer :: iatfix(:)
  ! iatfix(natom)

  integer, pointer :: iatprj_bs(:)

! Real pointers
  real(dp), pointer :: qnrml1(:)
  ! qnrml1(nph1l)

  real(dp), pointer :: qnrml2(:)
  ! qnrml2(nph2l)

  real(dp), pointer :: qpath(:,:)
  ! qpath(3,nqpath)

  real(dp), pointer :: qph1l(:,:)
  ! qph1l(3,nph1l)

  real(dp), pointer :: qph2l(:,:)
  ! qph2l(3,nph2l)

  real(dp), pointer :: ep_qptlist(:,:)
  ! qph2l(3,ep_nqpt)

 end type anaddb_dataset_type
!!***

!----------------------------------------------------------------------

!!****t* defs_abitypes/dataset_type
!! NAME
!! dataset_type
!!
!! FUNCTION
!! The dataset_type structured datatype gather all the input variables,
!! except those that are labelled NOT INTERNAL.
!! For one dataset, it is initialized in driver.f, and will not change
!! at all during the treatment of the dataset.
!! The "evolving" input variables are also stored, with their
!! name appended with _orig, to make clear that this is the original
!! value, decided by the user, and not a possibly modified, intermediate value.
!! The following input variables are NOT INTERNAL, that is, they
!! are input variables used to determine other input variables,
!! after suitable processing, and do not appear anymore afterwards
!! (so, they do not appear as components of a dataset_type variable) :
!! cpuh,cpum(but cpus is present),fband,kptbounds,ndivk,ndism,nobj,
!! objaat,objbat,objaax,objbax,objan,objbn,objarf,objbrf,objaro,objbro
!! objatr,objbtr,vaclst,vacuum
!!
!! SOURCE

type dataset_type

! WARNING : if you modify this datatype, please check whether there might be
! creation/destruction/copy routines, declared in another part of ABINIT,
! that might need to take into account your modification.

! Variables should be declared on separated lines in order to reduce the occurence
! of bzr conflicts.

! Since all these input variables are described in the abinit_help.html and
! associated html files they are not described in length here ...

! Integer
 integer :: accesswff
 integer :: awtr
 integer :: bandpp
 integer :: bdeigrf
 integer :: berryopt
 integer :: berrysav
 integer :: berrystep
 integer :: brvltt
 integer :: bs_nstates
 integer :: bs_hayd_term
 integer :: builtintest
 integer :: cd_full_grid
 integer :: cd_frqim_method
 integer :: cd_customnimfrqs
 integer :: chkexit
 integer :: chkgwcomp
 integer :: chkprim
 integer :: chksymbreak
 integer :: cineb_start
 integer :: delayperm
 integer :: dmatpuopt
 integer :: dmatudiag
 integer :: dmft_dc
 integer :: dmft_iter
 integer :: dmft_nwli
 integer :: dmft_nwlo
 integer :: dmft_rslf
 integer :: dmft_read_occnd
 integer :: dmft_solv
 integer :: dmftbandi
 integer :: dmftbandf
 integer :: dmftcheck
 integer :: enunit
 integer :: exchn2n3d
 integer :: fftgw
 integer :: fft_opt_lob
 integer :: frzfermi
 integer :: ga_algor
 integer :: ga_fitness
 integer :: ga_n_rules
 integer :: getcell
 integer :: getddk
 integer :: getden
 integer :: getgam_eig2nkq
 integer :: getkss
 integer :: getocc
 integer :: getpawden
 integer :: getqps
 integer :: getscr
 integer :: getsuscep
 integer :: getvel
 integer :: getwfk
 integer :: getwfq
 integer :: getxcart
 integer :: getxred
 integer :: get1den
 integer :: get1wf
 integer :: getbseig
 integer :: getbsreso
 integer :: getbscoup
 integer :: gethaydock
 integer :: goprecon
 integer :: gw_customnfreqsp
 integer :: gw_eet
 integer :: gw_eet_nband
 integer :: gw_eet_inclvkb
 integer :: gwcalctyp
 integer :: gwcomp
 integer :: gwgamma
 integer :: gwrpacorr
 integer :: gw_npoles
 integer :: gw_nqlwl
 integer :: gw_nstep
 integer :: gw_sigxcore
 integer :: gw_reconst_scr
 integer :: gw_frqim_inzgrid
 integer :: gw_frqre_inzgrid
 integer :: gw_frqre_tangrid
 integer :: gw_use_pole_scr
 integer :: gwmem
 integer :: gwpara
 integer :: gw_sctype
 integer :: iboxcut
 integer :: icoulomb
 integer :: icutcoul
 integer :: idyson
 integer :: ieig2rf
 integer :: ikhxc
 integer :: imgmov
 integer :: inclvkb
 integer :: intexact
 integer :: intxc
 integer :: ionmov
 integer :: iprcch
 integer :: iprcel
 integer :: iprctfvw
 integer :: iprcfc
 integer :: irandom
 integer :: irdddk
 integer :: irdden
 integer :: irdhaydock
 integer :: irdkss
 integer :: irdpawden
 integer :: irdqps
 integer :: irdscr
 integer :: irdsuscep
 integer :: irdwfk
 integer :: irdwfq
 integer :: ird1den
 integer :: ird1wf
 integer :: irdbseig
 integer :: irdbsreso
 integer :: irdbscoup
 integer :: iscf
 integer :: isecur
 integer :: istatimg
 integer :: istatr
 integer :: istatshft
 integer :: ixc
 integer :: ixcpositron
 integer :: jdtset !  jdtset contains the actual number of the dataset
 integer :: jellslab
 integer :: kptopt
 integer :: kssform
 integer :: ldgapp
 integer :: localrdwf
 integer :: maxnsym
 integer :: mband
 integer :: mep_solver
 integer :: mffmem
 integer :: mgfft
 integer :: mgfftdg
 integer :: mkmem
 integer :: mkqmem
 integer :: mk1mem
 integer :: nnos
 integer :: mpw
 integer :: mqgrid
 integer :: mqgriddg
 integer :: natom
 integer :: natpawu
 integer :: natrd
 integer :: natsph
 integer :: natvshift
 integer :: nbandkss
 integer :: nbandsus
 integer :: nbdblock
 integer :: nbdbuf
 integer :: nberry
 integer :: nconeq
 integer :: nctime
 integer :: ndtset
 integer :: ndynimage
 integer :: ndyson
 integer :: neb_algo
 integer :: nfft
 integer :: nfftdg
 integer :: nfreqim
 integer :: nfreqre
 integer :: nfreqsp
 integer :: nfreqsus
 integer :: diismemory
 integer :: nimage
 integer :: nkptgw
 integer :: nkpt
 integer :: nline
 integer :: nnsclo
 integer :: nomegasf
 integer :: nomegasi
 integer :: nomegasrd
 integer :: npband
 integer :: npfft
 integer :: npimage
 integer :: npkpt
 integer :: nppert
 integer :: npspinor
 integer :: npsp
 integer :: npspalch
 integer :: npulayit
 integer :: npweps
 integer :: npwkss
 integer :: npwsigx
 integer :: npwwfn
 integer :: np_slk
 integer :: nqpt
 integer :: nqptdm
 integer :: nscforder
 integer :: nsheps
 integer :: nshiftk
 integer :: nshsigx
 integer :: nshwfn
 integer :: nspden
 integer :: nspinor
 integer :: nsppol
 integer :: nstep
 integer :: nsym
 integer :: ntime
 integer :: ntimimage
 integer :: ntypalch
 integer :: ntypat
 integer :: ntyppure
 integer :: nwfshist
 integer :: occopt
 integer :: optcell
 integer :: optdriver
 integer :: optforces
 integer :: optfreqsus
 integer :: optnlxccc
 integer :: optstress
 integer :: ortalg
 integer :: paral_atom
 integer :: paral_kgb
 integer :: paral_rf
 integer :: pawcpxocc
 integer :: pawcross
 integer :: pawfatbnd
 integer :: pawlcutd
 integer :: pawlmix
 integer :: pawmixdg
 integer :: pawnhatxc
 integer :: pawnphi
 integer :: pawntheta
 integer :: pawnzlm
 integer :: pawoptmix
! integer :: pawprtden deprecated variable => "merged" with prtden
 integer :: pawprtdos
 integer :: pawprtvol
 integer :: pawprtwf
 integer :: pawprt_k
 integer :: pawprt_b
 integer :: pawspnorb
 integer :: pawstgylm
 integer :: pawsushat
 integer :: pawusecp
 integer :: macro_uj
 integer :: pawujat
 integer :: pawxcdev
 integer :: pitransform
 integer :: positron
 integer :: posnstep
 integer :: ppmodel
 integer :: prepanl
 integer :: prepgkk
 integer :: prepscphon
 integer :: prtbbb
 integer :: prtbltztrp
 integer :: prtcif
 integer :: prtcml
 integer :: prtcs
 integer :: prtden
 integer :: prtdensph
 integer :: prtdipole
 integer :: prtdos
 integer :: prtdosm
 integer :: prtefg
 integer :: prteig
 integer :: prtelf
 integer :: prtfc
 integer :: prtfsurf
 integer :: prtgden
 integer :: prtgeo
 integer :: prtgkk
 integer :: prtkden
 integer :: prtkpt
 integer :: prtlden
 integer :: prtnabla
 integer :: prtnest
 integer :: prtpmp
 integer :: prtposcar
 integer :: prtpot
 integer :: prtspcur
 integer :: prtstm
 integer :: prtsuscep
 integer :: prtvha
 integer :: prtvhxc
 integer :: prtvol
 integer :: prtvolimg
 integer :: prtvxc
 integer :: prtwant
 integer :: prtwf
 integer :: prtxml
 integer :: prt1dm
 integer :: ptgroupma
 integer :: random_atpos
 integer :: recgratio
 integer :: recnpath
 integer :: recnrec
 integer :: recptrott
 integer :: rectesteg
 integer :: restartxf
 integer :: rfasr
 integer :: rfddk
 integer :: rfelfd
 integer :: rfmeth
 integer :: rfphon
 integer :: rfstrs
 integer :: rfuser
 integer :: rf1elfd
 integer :: rf1phon
 integer :: rf2elfd
 integer :: rf2phon
 integer :: rf3elfd
 integer :: rf3phon
 integer :: signperm
 integer :: smdelta
 integer :: spgaxor
 integer :: spgorig
 integer :: spgroup
 integer :: spmeth
 integer :: string_algo
 integer :: suskxcrs
 integer :: symmorphi
 integer :: symchi
 integer :: symsigma
 integer :: td_mexcit
 integer :: tfkinfunc
 integer :: timopt
 integer :: tl_nprccg
 integer :: ucrpa
 integer :: use_gpu_cuda
 integer :: usedmatpu
 integer :: usedmft
 integer :: useexexch
 integer :: usekden
 integer :: usepaw
 integer :: usepawu
 integer :: userec
 integer :: useria
 integer :: userib
 integer :: useric
 integer :: userid
 integer :: userie
 integer :: usewvl
 integer :: usexcnhat
 integer :: useylm
 integer :: use_slk
 integer :: vacnum
 integer :: vdw_nfrag
 integer :: vdw_xc
 integer :: wfoptalg
 integer :: wvl_nprccg
 integer :: w90iniprj
 integer :: w90prtunk
 integer :: xclevel

!Integer arrays
 integer :: bdberry(4)
 integer :: bravais(11)
 integer :: cd_subset_freq(2)
 integer :: jfielddir(3)
 integer :: kptrlatt(3,3)
 integer :: ga_rules(30)
 integer :: ngfft(18)
 integer :: ngfftdg(18)
 integer :: nloalg(5)
 integer :: qprtrb(3)
 integer :: rfatpol(2)
 integer :: rfdir(3)
 integer :: rf1atpol(2)
 integer :: rf1dir(3)
 integer :: rf2atpol(2)
 integer :: rf2dir(3)
 integer :: rf3atpol(2)
 integer :: rf3dir(3)
 integer :: scphon_supercell(3)
 integer :: supercell(3)
 integer :: ucrpa_bands(2)
 integer :: vdw_supercell(3)
 integer :: vdw_typfrag(100)
!Integer pointers
 integer, pointer ::  algalch(:)    ! SET2NULL ! algalch(ntypalch)
 integer, pointer ::  bdgw(:,:,:)   ! SET2NULL ! bdgw(2,nkptgw,nsppol)
 integer, pointer ::  dynimage(:)   ! SET2NULL ! dynimage(nimage or mxnimage)
 integer, pointer ::  iatfix(:,:)   ! SET2NULL ! iatfix(3,natom)
 integer, pointer ::  iatsph(:)     ! SET2NULL ! iatsph(natsph)
 integer, pointer ::  istwfk(:)     ! SET2NULL ! istwfk(nkpt)
 integer, pointer ::  kberry(:,:)   ! SET2NULL ! kberry(3,nberry)
 integer, pointer ::  lexexch(:)    ! SET2NULL ! lexexch(ntypat)
 integer, pointer ::  lpawu(:)      ! SET2NULL ! lpawu(ntypat)
 integer, pointer ::  nband(:)      ! SET2NULL ! nband(nkpt*nsppol)
 integer, pointer ::  prtatlist(:)  ! SET2NULL ! prtatlist(natom)
 integer, pointer ::  so_psp(:)     ! SET2NULL ! so_psp(npsp)
 integer, pointer ::  symafm(:)     ! SET2NULL ! symafm(nsym)
 integer, pointer ::  symrel(:,:,:) ! SET2NULL ! symrel(3,3,nsym)
 integer, pointer ::  typat(:)      ! SET2NULL ! typat(natom)

!Real
 real(dp) :: bmass
 real(dp) :: boxcutmin
 real(dp) :: bxctmindg
 real(dp) :: cd_halfway_freq
 real(dp) :: cd_max_freq
 real(dp) :: charge
 real(dp) :: cpus
 real(dp) :: ddamp
 real(dp) :: diecut
 real(dp) :: diegap
 real(dp) :: dielam
 real(dp) :: dielng
 real(dp) :: diemac
 real(dp) :: diemix
 real(dp) :: diemixmag
 real(dp) :: dilatmx
 real(dp) :: dmft_mxsf
 real(dp) :: dmft_tollc
 real(dp) :: dosdeltae
 real(dp) :: dtion
 real(dp) :: ecut
 real(dp) :: ecuteps
 real(dp) :: ecutsigx
 real(dp) :: ecutsm
 real(dp) :: ecutwfn
 real(dp) :: effmass
 real(dp) :: elph2_imagden
 real(dp) :: eshift
 real(dp) :: esmear
 real(dp) :: exchmix
 real(dp) :: fband
 real(dp) :: fermie_nest
 real(dp) :: spinmagntarget
 real(dp) :: freqim_alpha
 real(dp) :: freqremin
 real(dp) :: freqremax
 real(dp) :: freqspmin
 real(dp) :: freqspmax
 real(dp) :: freqsusin
 real(dp) :: freqsuslo
 real(dp) :: friction
 real(dp) :: fxcartfactor
 real(dp) :: ga_opt_percent
 real(dp) :: gwencomp
 real(dp) :: gw_eet_scale
 real(dp) :: gw_toldfeig
 real(dp) :: kptnrm
 real(dp) :: kptrlen
 real(dp) :: maxestep
 real(dp) :: mdf_epsinf
 real(dp) :: mdwall
 real(dp) :: mep_mxstep
 real(dp) :: nelect
 real(dp) :: noseinert
 real(dp) :: omegasimax
 real(dp) :: omegasrdmax
 real(dp) :: pawecutdg
 real(dp) :: pawovlp
 real(dp) :: pawujrad
 real(dp) :: pawujv
 real(dp) :: posocc
 real(dp) :: postoldfe
 real(dp) :: postoldff
 real(dp) :: ppmfrq
 real(dp) :: recrcut
 real(dp) :: recefermi
 real(dp) :: rectolden
 real(dp) :: rhoqpmix
 real(dp) :: rcut
 real(dp) :: sciss
 real(dp) :: scphon_temp
 real(dp) :: slabwsrad
 real(dp) :: slabzbeg
 real(dp) :: slabzend
 real(dp) :: soenergy
 real(dp) :: spbroad
 real(dp) :: spnorbscl
 real(dp) :: stmbias
 real(dp) :: strfact
 real(dp) :: strprecon
 real(dp) :: td_maxene
 real(dp) :: tl_radius
 real(dp) :: toldfe
 real(dp) :: toldff
 real(dp) :: tolimg
 real(dp) :: tolmxf
 real(dp) :: tolrde
 real(dp) :: tolrff
 real(dp) :: tolsym
 real(dp) :: tolvrs
 real(dp) :: tolwfr
 real(dp) :: tphysel
 real(dp) :: tsmear
 real(dp) :: userra
 real(dp) :: userrb
 real(dp) :: userrc
 real(dp) :: userrd
 real(dp) :: userre
 real(dp) :: vacwidth
 real(dp) :: vdw_tol
 real(dp) :: vis
 real(dp) :: wvl_hgrid
 real(dp) :: wvl_crmult
 real(dp) :: wvl_frmult
 real(dp) :: xc_denpos
 real(dp) :: xc_tb09_c
 real(dp) :: zcut

!Real arrays
 real(dp) :: boxcenter(3)
 real(dp) :: bfield(3)
 real(dp) :: dfield(3)
 real(dp) :: efield(3)
 real(dp) :: genafm(3)
 real(dp) :: goprecprm(3)
 real(dp) :: neb_spring(2)
 real(dp) :: pol(3)
 real(dp) :: polcen(3)
 real(dp) :: qptn(3)
 real(dp) :: red_efield(3)
 real(dp) :: red_dfield(3)
 real(dp) :: red_efieldbar(3)
 real(dp) :: strtarget(6)
 real(dp) :: vcutgeo(3)
 real(dp) :: vprtrb(2)
 real(dp) :: zeemanfield(3)
 real(dp) :: mdtemp(2)

!Real pointers
 real(dp), pointer :: acell_orig(:,:)   !SET2NULL  ! acell_orig(3,nimage)
 real(dp), pointer :: amu_orig(:,:)     !SET2NULL  ! amu(ntypat,nimage)
 real(dp), pointer :: atvshift(:,:,:)   !SET2NULL  ! atvshift(16,nsppol,natom)
 real(dp), pointer :: cd_imfrqs(:)      !SET2NULL  ! cd_imfrqs(cd_customnimfrqs)
 real(dp), pointer :: corecs(:)         !SET2NULL  ! corecs(ntypat)
 real(dp), pointer :: densty(:,:)       !SET2NULL  ! densty(ntypat,4)
 real(dp), pointer :: dmatpawu(:,:,:,:) !SET2NULL  ! dmatpawu(2*lpawu+1,2*lpawu+1,nsppol*nspinor,natpu)
                                                   !   where natpu=number of atoms with lpawu/=1
 real(dp), pointer :: f4of2_sla(:)      !SET2NULL  ! f4of2_sla(ntypat)
 real(dp), pointer :: f6of2_sla(:)      !SET2NULL  ! f6of2_sla(ntypat)
 real(dp), pointer :: gw_qlwl(:,:)      !SET2NULL  ! gw_qlwl(3,gw_nqlwl)
 real(dp), pointer :: gw_freqsp(:)      !SET2NULL  ! gw_freqsp(gw_customnfreqsp)
 real(dp), pointer :: jpawu(:)          !SET2NULL  ! jpawu(ntypat)
 real(dp), pointer :: kpt(:,:)          !SET2NULL  ! kpt(3,nkpt)
 real(dp), pointer :: kptgw(:,:)        !SET2NULL  ! kptgw(3,nkptgw)
 real(dp), pointer :: kptns(:,:)        !SET2NULL  ! kptns(3,nkpt) k-points renormalized and shifted.
                                                  ! The ones that should be used inside the code.

 real(dp), pointer :: mixalch_orig(:,:,:) !SET2NULL ! mixalch_orig(npspalch,ntypalch,nimage)
 real(dp), pointer :: occ_orig(:)         !SET2NULL ! occ_orig(mband*nkpt*nsppol)
 real(dp), pointer :: pimass(:)           !SET2NULL ! pimass(ntypat)
 real(dp), pointer :: ptcharge(:)         !SET2NULL ! ptcharge(ntypat)
 real(dp), pointer :: qmass(:)            !SET2NULL ! qmass(nnos)
 real(dp), pointer :: qptdm(:,:)          !SET2NULL ! qptdm(3,nqptdm)
 real(dp), pointer :: quadmom(:)          !SET2NULL ! quadmom(ntypat)
 real(dp), pointer :: ratsph(:)           !SET2NULL ! ratsph(ntypat)
 real(dp), pointer :: rprim_orig(:,:,:)   !SET2NULL ! rprim_orig(3,3,nimage)
 real(dp), pointer :: rprimd_orig(:,:,:)  !SET2NULL ! rprimd_orig(3,3,nimage)
 real(dp), pointer :: shiftk(:,:)         !SET2NULL ! shiftk(3,nshiftk)
 real(dp), pointer :: spinat(:,:)         !SET2NULL ! spinat(3,natom)
 real(dp), pointer :: tnons(:,:)          !SET2NULL ! tnons(3,nsym)
 real(dp), pointer :: upawu(:)            !SET2NULL ! upawu(ntypat)
 real(dp), pointer :: vel_cell_orig(:,:,:)!SET2NULL ! vel_cell_orig(3,3,nimage)
 real(dp), pointer :: vel_orig(:,:,:)     !SET2NULL ! vel_orig(3,natom,nimage)
 real(dp), pointer :: wtatcon(:,:,:)      !SET2NULL ! wtatcon(3,natom,nconeq)
 real(dp), pointer :: wtk(:)              !SET2NULL ! wtk(nkpt)
 real(dp), pointer :: xred_orig(:,:,:)    !SET2NULL ! xred_orig(3,natom,nimage)
 real(dp), pointer :: ziontypat(:)        !SET2NULL ! ziontypat(ntypat)
 real(dp), pointer :: znucl(:)            !SET2NULL ! znucl(npsp)


!BEGIN VARIABLES FOR @Bethe-Salpeter
 integer :: bs_algorithm
 integer :: bs_haydock_niter
 integer :: bs_exchange_term
 integer :: bs_coulomb_term
 integer :: bs_calctype
 integer :: bs_coupling

 real(dp) :: bs_haydock_tol(2)

 integer :: bs_loband

 real(dp) :: bs_eh_cutoff(3)
 real(dp) :: bs_freq_mesh(3)
!END VARIABLES FOR @Bethe-Salpeter.

 integer :: gpu_linalg_limit ! MT sept2012: had to add this keyword at the end
                             ! to get BigDFT automatic tests work on shiva and littlebuda (why????)

 end type dataset_type
!!***


!----------------------------------------------------------------------

!!****t* defs_abitypes/MPI_type
!! NAME
!! MPI_type
!!
!! FUNCTION
!! The MPI_type structured datatype gather different information
!! about the MPI parallelisation : number of processors,
!! the index of my processor, the different groups of processors, etc ...
!!
!! SOURCE

 type MPI_type

! WARNING : if you modify this datatype, please check whether there might be creation/destruction/copy routines,
! declared in another part of ABINIT, that might need to take into account your modification.
! Variables should be declared on separated lines in order to reduce the occurence of bzr conflicts.

! *****************************************************************************************
! Please make sure that initmpi_seq is changed so that any variable or any flag in MPI_type
! is initialized with the value used for sequential executions.
! In particular any MPI communicator should be set to MPI_COMM_SELF
! ************************************************************************************

  ! Set of variables for parallelism, that do NOT depend on input variables.
  ! These are defined for each dataset

! ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
! Main variables for parallelisation
  integer :: comm_world       ! number of the world communicator MPI COMM WORLD
  integer :: me               ! number of my processor in the group of all processors
  integer :: nproc            ! number of processors

  integer :: me_g0         ! if set to 1, means that the current processor
                           ! is taking care of the G(0 0 0) planewave.
! ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
! This is for the parallelisation over atoms (PAW)
   integer :: comm_atom                   ! Communicator over atoms
   integer :: nproc_atom                  ! Size of the communicator over atoms
   integer :: my_natom                    ! Number of atoms treated by current proc
   integer,pointer :: my_atmtab(:)        ! Indexes of the atoms treated by current processor
! ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
! This is for the parallelisation over perturbations
   integer :: paral_pert ! to activate parallelisation over perturbations for linear response
   integer :: comm_pert         ! communicator for calculating perturbations
   integer :: comm_cell_pert    ! general communicator over all processors treating the same cell
   integer :: me_pert           ! number of my processor in my group of perturbations
   integer :: nproc_pert        ! number of processors in my group of perturbations
   integer, pointer :: distrb_pert(:)  ! distrb_pert(1:npert)
      ! index of processor treating each perturbation
! ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
! This is for the parallelisation over images
   integer :: paral_img     ! Flag activated if parallelization over image is on
   integer :: my_nimage        ! Number of images of the cell treated by current proc (i.e. local nimage)
   integer :: comm_img      ! Communicator over all images
   integer :: me_img        ! Index of my processor in the comm. over all images
   integer :: nproc_img     ! Size of the communicator over all images
   integer,pointer :: distrb_img(:)
      ! distrb_img(1:dtset%nimage)
      ! index of processor treating each image (in comm_img communicator)
   integer,pointer :: my_imgtab(:)
      ! index_img(1:my_nimage) indexes of images treated by current proc
! ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
! This is for the parallelisation over the cell
   integer :: comm_cell  ! local Communicator over all processors treating the same cell
   integer :: me_cell    ! Index of my processor in the comm. over one cell
   integer :: nproc_cell ! Size of the communicator over one cell
! ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
! This is for the parallelisation over fft
   integer :: comm_fft               ! Communicator over fft
   integer :: me_fft         ! Rank of my processor in my group of FFT
   integer :: nproc_fft      ! number of processors in my group of FFT
! ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
! This is for the parallelisation over bands
   integer :: paralbd
    ! paralbd=0 : (no //ization on bands)
    ! paralbd=1 : (//ization on bands)
   integer :: comm_band              ! Communicator over bands
   integer :: me_band ! Rank of my proc in my group of bands
   integer :: nproc_band             ! Number of procs on which we distribute bands
! ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
! This is for the spinor parallelisation
   integer :: paral_spinor             ! Flag: activation of parallelization over spinors
   integer :: comm_spinor              ! Communicator over spinors
   integer :: me_spinor                ! Rank of my proc in the communicator over spinors
                                     ! Note: me_spinor is related to the index treated by current proc
                                     !       (nspinor_index= mpi_enreg%me_spinor + 1)
   integer :: nproc_spinor             ! Number of procs on which we distribute spinors
! ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
! This is for the kpt/nsppol parallelisation
   integer :: comm_kpt               ! Communicator over kpt
   integer :: me_kpt                 ! Rank of my proc in the communicator over kpt
   integer :: nproc_kpt              ! Number of procs on which we distribute kpt
   integer, pointer :: proc_distrb(:,:,:)
    ! proc_distrb(nkpt,mband,nsppol)
    ! number of the processor that will treat
    ! each band in each k point.
! ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
! This is for the band-FFT-kpt-spinor parallelisation
   integer :: paral_kgb              ! Flag: activation of parallelization over kpt/band/fft
   integer :: bandpp                 ! # of Bands Per Processor
   integer :: comm_bandspinorfft            ! Cartesian communicator over band-fft-spinor
   integer :: comm_bandfft           ! Cartesian communicator over the band-fft
   integer :: comm_kptband           ! Communicator over kpt-band subspace
   integer :: comm_spinorfft           ! Communicator over fft-spinors subspace
   integer :: comm_bandspinor          ! Communicator over band-spinors subspace
   integer, pointer :: my_kgtab(:,:)
    ! Indexes of kg treated by current proc
   integer, pointer :: my_kpttab(:)
    ! Indicates the correspondence between the ikpt and ikpt_this_proc
! ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
! This is for the parallelisation over kpt/nsppol in the Berry Phase case
   integer, pointer :: kptdstrb(:,:,:)
    ! kptdstrb(me,ineigh,ikptloc)
    ! tab of processors required for mv_3dte.f and berryphase_new.f
   integer, pointer :: kpt_loc2fbz_sp(:,:,:)
    ! kpt_loc2fbz_sp(nproc, dtefield%fmkmem_max, 2)
    ! K-PoinT LOCal TO Full Brilloin Zone and Spin Polarization
    ! given a processor and the local number of the k-point for this proc,
    ! give the number of the k-point in the FBZ and the isppol;
    ! necessary for synchronisation in berryphase_new
    ! kpt_loc2fbz(iproc, ikpt_loc,1) = ikpt
    ! kpt_loc2fbz(iproc, ikpt_loc,2) = isppol
   integer, pointer :: kpt_loc2ibz_sp(:,:,:)
   integer, pointer :: mkmem(:)
! ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
! This is for the wavelet parallelisation
   logical :: usewvl
   ! Array to store the description of the scaterring in real space of
   ! the potentials and density. It is allocated to (0:nproc-1,4).
   ! The four values are:
   ! - the density size in z direction ( = ngfft(3)) ;
   ! - the potential size in z direction ( <= ngfft(3)) ;
   ! - the position of the first value in the complete array ;
   ! - the shift for the potential in the array.
   ! This array is a pointer on a BigDFT handled one.
   integer, pointer :: nscatterarr(:,:)
   ! Array to store the total size (of this proc) of the potentails arrays when
   ! the memory is distributed following nscatterarr.
   ! This array is a pointer on a BigDFT handled one.
   integer, pointer :: ngatherarr(:,:)
   ! Store the ionic potential size in z direction.
   integer :: ngfft3_ionic
   ! End wavelet additions

 end type MPI_type
!!***

!----------------------------------------------------------------------

!!****t* defs_datatypes/datafiles_type
!! NAME
!! datafiles_type
!!
!! FUNCTION
!! The datafiles_type structures datatype gather all the variables
!! related to files, such as filename, and file units.
!! For one dataset, it is initialized in 95_drive/dtfil_init1.F90,
!! and will not change at all during the treatment of the dataset.
!!
!! SOURCE

 type datafiles_type

! WARNING : if you modify this datatype, please check whether there might be creation/destruction/copy routines,
! declared in another part of ABINIT, that might need to take into account your modification.


! These keywords are only used in algorithms using images of the cell
  integer :: getwfk_from_image
   ! index of image from which read WFK file (0 if standard WFK)
   !    -1: the same image as current one
   !     0: no image
   !    >0: index of an image
  integer :: getden_from_image
   ! index of image from which read DEN file (0 if standard DEN)
   !    -1: the same image as current one
   !     0: no image
   !    >0: index of an image
  integer :: getpawden_from_image
   ! index of image from which read PAWDEN file (0 if standard PAWDEN)
   !    -1: the same image as current one
   !     0: no image
   !    >0: index of an image

  integer :: ireadden
   ! ireadden non-zero  if the den file must be read

  integer :: ireadkden
   ! ireadkden non-zero  if the kden file must be read

  integer :: ireadwf
   ! if(optdriver/=1), that is, no response-function computation,
   !   ireadwf non-zero  if the wffk file must be read
   !   (if irdwfk non-zero or getwfk non-zero)
   ! if(optdriver==1), that is, response-function computation,
   !   ireadwf non-zero  if the wff1 file must be read
   !   (if ird1wf non-zero or get1wf non-zero)

  integer :: unchi0  ! unit number for chi0 files
  integer :: unddb   ! unit number for Derivative DataBase
  integer :: unddk   ! unit number for ddk 1WF file
  integer :: unem1ggp! unit number for Epsilon minus one (G,Gp) file
  integer :: unkg    ! unit number for k+G data
  integer :: unkgq   ! unit number for k+G+q data
  integer :: unkg1   ! unit number for first-order k+G+q data
  integer :: unkss   ! unit number for KSS file
  integer :: unqps   ! unit number for QPS file
  integer :: unscr   ! unit number for SCR file
  integer :: unwff1  ! unit number for wavefunctions, number one
  integer :: unwff2  ! unit number for wavefunctions, number two
  integer :: unwffgs ! unit number for ground-state wavefunctions
  integer :: unwffkq ! unit number for k+q ground-state wavefunctions
  integer :: unwft1  ! unit number for wavefunctions, temporary one
  integer :: unwft2  ! unit number for wavefunctions, temporary two
  integer :: unwftgs ! unit number for ground-state wavefunctions, temporary
  integer :: unwftkq ! unit number for k+q ground-state wavefunctions, temporary
  integer :: unylm   ! unit number for Ylm(k) data
  integer :: unylm1  ! unit number for first-order Ylm(k+q) data
  integer :: unpaw   ! unit number for temporary PAW data (for ex. rhoij_nk) (Paw only)
  integer :: unpaw1  ! unit number for temporary PAW first-order cprj1=<c1_k,q|p>(1) data
  integer :: unpawq  ! unit number for temporary PAW cprjq=<c+_k+q|p> at k+qdata
  integer :: unpos   ! unit number for restart molecular dynamics
  integer :: undelI  ! unit number for incompleteness matrix

  character(len=fnlen) :: filnam_ds(5)
   ! if no dataset mode, the five names from the standard input :
   !   ab_in, ab_out, abi, abo, tmp
   ! if dataset mode, the same 5 filenames, appended with //'_DS'//trim(jdtset)

  character(len=fnlen) :: fildensin
   ! if no dataset mode             : abi//'DEN'
   ! if dataset mode, and getden==0 : abi//'_DS'//trim(jdtset)//'DEN'
   ! if dataset mode, and getden/=0 : abo//'_DS'//trim(jgetden)//'DEN'

  character(len=fnlen) :: filkdensin
   ! if no dataset mode             : abi//'KDEN'
   ! if dataset mode, and getden==0 : abi//'_DS'//trim(jdtset)//'KDEN'
   ! if dataset mode, and getden/=0 : abo//'_DS'//trim(jgetden)//'KDEN'

  character(len=fnlen) :: filpawdensin
   ! if no dataset mode             : abi//'PAWDEN'
   ! if dataset mode, and getden==0 : abi//'_DS'//trim(jdtset)//'PAWDEN'
   ! if dataset mode, and getden/=0 : abo//'_DS'//trim(jgetden)//'PAWDEN'

! character(len=fnlen) :: filpsp(ntypat)
   ! the filenames of the pseudopotential files, from the standard input.

  character(len=fnlen) :: filstat
   ! tmp//'_STATUS'

  character(len=fnlen) :: fnamewffk
   ! the name of the ground-state wavefunction file to be read (see driver.F90)

  character(len=fnlen) :: fnamewffq
   ! the name of the k+q ground-state wavefunction file to be read (see driver.F90)
   ! only useful in the response-function case

  character(len=fnlen) :: fnamewffddk
   ! the generic name of the ddk response wavefunction file(s) to be read (see driver.F90)
   ! (the final name is formed by appending the number of the perturbation)
   ! only useful in the response-function case

  character(len=fnlen) :: fnamewff1
   ! the generic name of the first-order wavefunction file(s) to be read (see driver.F90)
   ! (the final name is formed by appending the number of the perturbation)
   ! only useful in the response-function case

  character(len=fnlen) :: fildens1in   ! to be described by MVeithen

  character(len=fnlen) :: fname_tdwf

  character(len=fnlen) :: fname_w90

  character(len=fnlen) :: fnametmp_wf1
  character(len=fnlen) :: fnametmp_wf2
  character(len=fnlen) :: fnametmp_1wf1
  character(len=fnlen) :: fnametmp_1wf2
  character(len=fnlen) :: fnametmp_wfgs
  character(len=fnlen) :: fnametmp_wfkq
   ! Set of filemanes formed from trim(dtfil%filnam_ds(5))//APPEN where APPEN is _WF1, _WF2 ...
   ! See dtfil_init

  character(len=fnlen) :: fnametmp_kg
  character(len=fnlen) :: fnametmp_kgq
  character(len=fnlen) :: fnametmp_kg1
  character(len=fnlen) :: fnametmp_dum
  character(len=fnlen) :: fnametmp_ylm
  character(len=fnlen) :: fnametmp_ylm1
  character(len=fnlen) :: fnametmp_paw
  character(len=fnlen) :: fnametmp_paw1
  character(len=fnlen) :: fnametmp_pawq
   ! Set of filemanes formed from trim(dtfil%filnam_ds(5))//APPEN where APPEN is _KG, _DUM, followed
   ! by the index of the processor.
   ! See dtfil_init

  character(len=fnlen) :: fnametmp_cg
  character(len=fnlen) :: fnametmp_cprj
  character(len=fnlen) :: fnametmp_eig
  character(len=fnlen) :: fnametmp_1wf1_eig
  character(len=fnlen) :: fnametmp_fft
  character(len=fnlen) :: fnametmp_kgs
  character(len=fnlen) :: fnametmp_sustr
  character(len=fnlen) :: fnametmp_tdexcit
  character(len=fnlen) :: fnametmp_tdwf

!@Bethe-Salpeter
! New files introduced for the Bethe-Salpeter part.

   character(len=fnlen) :: fnameabi_bsham_reso
!   ! if no dataset mode             : abi//'BSR'
!   ! if dataset mode, and getbsreso==0 : abi//'_DS'//trim(jdtset)//'BSR'
!   ! if dataset mode, and getbsreso/=0 : abo//'_DS'//trim(jget_reso_bsham)//'BSR'

   character(len=fnlen) :: fnameabi_bsham_coup
!   ! if no dataset mode             : abi//'BSC'
!   ! if dataset mode, and getbscoup==0 : abi//'_DS'//trim(jdtset)//'BSC'
!   ! if dataset mode, and getbscoup/=0 : abo//'_DS'//trim(jget_coup_bsham)//'BSC'
!
  character(len=fnlen) :: fnameabi_bseig
   ! The name of the file containing the eigenstates and eigenvalues of the Bethe-Salpeter Hamiltonian
   ! if no dataset mode             : abi//'BS_EIG'
   ! if dataset mode, and getbseig==0 : abi//'_DS'//trim(jdtset)//'BS_EIG'
   ! if dataset mode, and getbseig/=0 : abo//'_DS'//trim(jget_bseig)//'BS_EIG'

   character(len=fnlen) :: fnameabi_haydock
   ! The prefix used to construct the names of the files containing the coefficients of the
   ! continued fractions produced by the Haydock iterative algorithm.
   ! if no dataset mode             : abi//'HAYDOCK'
   ! if dataset mode, and gethaydock==0 : abi//'_DS'//trim(jdtset)//'HAYDOCK'
   ! if dataset mode, and gethaydock/=0 : abo//'_DS'//trim(jget_bseig)//'HAYDOCK'

!END @BEthe-Salpeter

!The following filenames do not depend on itimimage, iimage and itime loops.
!Note the following convention:
!  fnameabo_* are filenames used for ouput results (writing)
!  fnameabi_* are filenames used for data the should be read by the code.
!  fnametmp_* are filenames used for temporary files that should be erased at the end of each dataset.
!
!If a file does not have the corresponding "abi" or the corresponding "abo" name, that means that
!that particular file is only used for writing or for reading results, respectively.

  character(len=fnlen) :: fnameabi_hes
  character(len=fnlen) :: fnameabi_kss            ! KSS file
  character(len=fnlen) :: fnameabi_phfrq
  character(len=fnlen) :: fnameabi_phvec
  character(len=fnlen) :: fnameabi_qps
  character(len=fnlen) :: fnameabi_scr            ! SCReening file (symmetrized inverse dielectric matrix)
  character(len=fnlen) :: fnameabi_sus            ! KS independent-particle polarizability file

  character(len=fnlen) :: fnameabo_ae_wfk
  character(len=fnlen) :: fnameabo_ddb
  character(len=fnlen) :: fnameabo_den
  character(len=fnlen) :: fnameabo_dos
  character(len=fnlen) :: fnameabo_eelf
  character(len=fnlen) :: fnameabo_eig
  character(len=fnlen) :: fnameabo_eigi2d
  character(len=fnlen) :: fnameabo_eigr2d
  character(len=fnlen) :: fnameabo_em1
  character(len=fnlen) :: fnameabo_em1_lf
  character(len=fnlen) :: fnameabo_em1_nlf
  character(len=fnlen) :: fnameabo_gkk
  character(len=fnlen) :: fnameabo_gw
  character(len=fnlen) :: fnameabo_gw_nlf_mdf
  character(len=fnlen) :: fnameabo_kss
  character(len=fnlen) :: fnameabo_moldyn
  character(len=fnlen) :: fnameabo_pot
  character(len=fnlen) :: fnameabo_qps            ! Quasi-Particle band structure file.
  character(len=fnlen) :: fnameabo_qp_den
  character(len=fnlen) :: fnameabo_qp_pawden      ! Full QP density
  character(len=fnlen) :: fnameabo_qp_dos
  character(len=fnlen) :: fnameabo_qp_eig
  character(len=fnlen) :: fnameabo_rpa
  character(len=fnlen) :: fnameabo_rpa_nlf_mdf
  character(len=fnlen) :: fnameabo_scr
  character(len=fnlen) :: fnameabo_sgm
  character(len=fnlen) :: fnameabo_sgr
  character(len=fnlen) :: fnameabo_sig
  character(len=fnlen) :: fnameabo_spcur
  character(len=fnlen) :: fnameabo_sus
  character(len=fnlen) :: fnameabo_vso
  character(len=fnlen) :: fnameabo_wan
  character(len=fnlen) :: fnameabo_wfk
  character(len=fnlen) :: fnameabo_wfq
  character(len=fnlen) :: fnameabo_w90
  character(len=fnlen) :: fnameabo_1wf
  character(len=fnlen) :: fnameabo_delI

!The following filenames are initialized only iniside itimimage, iimage and itime loops,
!and are appended with the adequate specifier 'app'.

  character(len=fnlen) :: fnameabo_app
  character(len=fnlen) :: fnameabo_app_atmden_core
  character(len=fnlen) :: fnameabo_app_atmden_full
  character(len=fnlen) :: fnameabo_app_atmden_val
  character(len=fnlen) :: fnameabo_app_n_tilde
  character(len=fnlen) :: fnameabo_app_n_one
  character(len=fnlen) :: fnameabo_app_nt_one
  character(len=fnlen) :: fnameabo_app_bxsf
  character(len=fnlen) :: fnameabo_app_cif
  character(len=fnlen) :: fnameabo_app_cml_xml
  character(len=fnlen) :: fnameabo_app_den
  character(len=fnlen) :: fnameabo_app_dos
  character(len=fnlen) :: fnameabo_app_elf
  character(len=fnlen) :: fnameabo_app_elf_down
  character(len=fnlen) :: fnameabo_app_elf_up
  character(len=fnlen) :: fnameabo_app_eig
  character(len=fnlen) :: fnameabo_app_fatbands
  character(len=fnlen) :: fnameabo_app_gden1
  character(len=fnlen) :: fnameabo_app_gden2
  character(len=fnlen) :: fnameabo_app_gden3
  character(len=fnlen) :: fnameabo_app_geo
  character(len=fnlen) :: fnameabo_app_kden
  character(len=fnlen) :: fnameabo_app_lden
  character(len=fnlen) :: fnameabo_app_nesting
  character(len=fnlen) :: fnameabo_app_pawden
  character(len=fnlen) :: fnameabo_app_pot
  character(len=fnlen) :: fnameabo_app_opt
  character(len=fnlen) :: fnameabo_app_opt2
  character(len=fnlen) :: fnameabo_app_stm
  character(len=fnlen) :: fnameabo_app_vha
  character(len=fnlen) :: fnameabo_app_vhxc
  character(len=fnlen) :: fnameabo_app_vxc
  character(len=fnlen) :: fnameabo_app_wfk
  character(len=fnlen) :: fnameabo_app_1dm

  character(len=fnlen) :: fnametmp_app_den
  character(len=fnlen) :: fnametmp_app_kden

 end type datafiles_type
!!***

!----------------------------------------------------------------------

!!****t* defs_abitypes/hdr_type
!! NAME
!! hdr_type
!!
!! FUNCTION
!! It contains all the information needed to write a header for a
!! wf, den or pot file.
!! The structure of the header is explained in the abinit_help.html and other associated html files.
!! The datatype is considered as an object, to which are attached a whole
!! set of "methods", actually, different subroutines.
!! A few of these subroutines are : hdr_init, hdr_update, hdr_clean,
!! hdr_check, hdr_io, hdr_skip.
!!
!! SOURCE

 type hdr_type

! WARNING : if you modify this datatype, please check whether there might be creation/destruction/copy routines,
! declared in another part of ABINIT, that might need to take into account your modification.

  integer :: bantot        ! total number of bands (sum of nband on all kpts and spins)
  integer :: date          ! starting date
  integer :: headform      ! format of the header
  integer :: intxc         ! input variable
  integer :: ixc           ! input variable
  integer :: natom         ! input variable
  integer :: nkpt          ! input variable
  integer :: npsp          ! input variable
  integer :: nspden        ! input variable
  integer :: nspinor       ! input variable
  integer :: nsppol        ! input variable
  integer :: nsym          ! input variable
  integer :: ntypat        ! input variable
  integer :: occopt        ! input variable
  integer :: pertcase      ! the index of the perturbation, 0 if GS calculation
  integer :: usepaw        ! input variable (0=norm-conserving psps, 1=paw)
  integer :: usewvl        ! input variable (0=plane-waves, 1=wavelets)

  real(dp) :: ecut                  ! input variable
  real(dp) :: ecutdg                ! input variable (ecut for NC psps, pawecutdg for paw)
  real(dp) :: ecutsm                ! input variable
  real(dp) :: ecut_eff              ! ecut*dilatmx**2 (dilatmx is an input variable)
  real(dp) :: etot                  ! EVOLVING variable
  real(dp) :: fermie                ! EVOLVING variable
  real(dp) :: residm                ! EVOLVING variable
  real(dp) :: stmbias               ! input variable
  real(dp) :: tphysel               ! input variable
  real(dp) :: tsmear                ! input variable

! This record is not a part of the hdr_type, although it is present in the
! header of the files. This is because it depends on the kind of file
! that is written, while all other information does not depend on it.
! It was preferred to let it be initialized or defined outside of hdr_type.
! integer :: fform         ! file format

  integer :: ngfft(3)      ! input variable
  integer :: nwvlarr(2)    ! nwvlarr(2) array holding the number of wavelets for each resolution.

  ! MG: nullifying the pointers renders the object undefined, maybe intent(out) somewhere?
  integer, pointer :: istwfk(:)     !SET2NULL
  ! input variable istwfk(nkpt)

  integer, pointer :: lmn_size(:)   !SET2NULL
  ! lmn_size(npsp) from psps

  integer, pointer :: nband(:)      !SET2NULL
  ! input variable nband(nkpt*nsppol)

  integer, pointer :: npwarr(:)     !SET2NULL
  ! npwarr(nkpt) array holding npw for each k point

  integer, pointer :: pspcod(:)    !SET2NULL
  ! pscod(npsp) from psps

  integer, pointer :: pspdat(:)   !SET2NULL
  ! psdat(npsp) from psps

  integer, pointer :: pspso(:)    !SET2NULL
  ! pspso(npsp) from psps

  integer, pointer :: pspxc(:)    !SET2NULL
  ! pspxc(npsp) from psps

  integer, pointer :: so_psp(:)   !SET2NULL
  ! input variable so_psp(npsp)

  integer, pointer :: symafm(:)   !SET2NULL
  ! input variable symafm(nsym)

  integer, pointer :: symrel(:,:,:)    !SET2NULL
  ! input variable symrel(3,3,nsym)

  integer, pointer :: typat(:)    !SET2NULL
  ! input variable typat(natom)

  real(dp) :: qptn(3)               ! the wavevector, in case of a perturbation
  real(dp) :: rprimd(3,3)           ! EVOLVING variables

  real(dp), pointer :: kptns(:,:)   !SET2NULL
  ! input variable kptns(3,nkpt)

  real(dp), pointer :: occ(:)       !SET2NULL
  ! EVOLVING variable occ(bantot)

  real(dp), pointer :: tnons(:,:)   !SET2NULL
  ! input variable tnons(3,nsym)

  real(dp), pointer :: wtk(:)       !SET2NULL
  ! weight of kpoints wtk(nkpt)

  real(dp), pointer :: xred(:,:)    !SET2NULL
  ! EVOLVING variable xred(3,natom)

  real(dp), pointer :: zionpsp(:)   ! SET2NULL
  ! zionpsp(npsp) from psps

  real(dp), pointer :: znuclpsp(:)  ! SET2NULL
  ! znuclpsp(npsp) from psps
  ! Note the difference between znucl and znuclpsp !

  real(dp), pointer :: znucltypat(:)  !SET2NULL
  ! znucltypat(ntypat) from alchemy

  character(len=6) :: codvsn
  ! version of the code

  character(len=132), pointer :: title(:)  !SET2NULL
  ! title(npsp) from psps

  type(pawrhoij_type), pointer :: pawrhoij(:)  !SET2NULL
  ! EVOLVING variable, only for paw

!Should make a list of supplementary infos
! MG: TODO For postprocessing purposes, it is quite useful to
!  have kptrlatt as well as nshiftk and shiftk. also kptopt is useful
!  to know if time reversal can be employed
  !real(dp) :: nelect
  !integer :: kptrlatt(3,3)
  !integer :: nshiftk
  !integer :: kptopt
  !integer,pointer :: shiftk(:,:) SET2NULL

 end type hdr_type
!!***

!----------------------------------------------------------------------

!!****t* defs_abitypes/ab_dimensions
!! NAME
!! ab_dimensions
!!
!! FUNCTION
!! One record for each dimension of arrays used in ABINIT.
!! Will be used to e.g.:
!! - contain the maximum size attained over all datasets (mxvals)
!! - indicate whether this dimension is the same for all datasets or not (multivals).
!! Used for example inside outvars
!!
!! SOURCE

 type ab_dimensions

! WARNING : if you modify this datatype, please check whether there might be creation/destruction/copy routines,
! declared in another part of ABINIT, that might need to take into account your modification.

    integer :: ga_n_rules
    integer :: gw_nqlwl
    integer :: lpawu
    integer :: mband
    integer :: mband_upper ! Maybe this one could be removed
    integer :: natom
    integer :: natpawu
    integer :: natsph
    integer :: natvshift
    integer :: nberry
    integer :: nconeq
    integer :: nfreqsp
    integer :: nimage
    integer :: nimfrqs
    integer :: nkpt
    integer :: nkptgw
    integer :: nnos
    integer :: nqptdm
    integer :: nshiftk
    integer :: nsp
    integer :: nspinor
    integer :: nsppol
    integer :: nsym
    integer :: ntypalch
    integer :: ntypat

 end type ab_dimensions

end module defs_abitypes
!!***
