!{\src2tex{textfont=tt}}
!!****m* ABINIT/defs_datatypes
!! NAME
!! defs_datatypes
!!
!! FUNCTION
!! This module contains definitions of all structured datatypes for the
!! ABINIT package.
!! If you want to add one new datatype, please, examine first whether
!! another datatype might meet your need (e.g. adding some records to it).
!! Then, if you are sure your new structured datatype is needed,
!! write it here, and DOCUMENT it properly (not all datastructure here are
!! well documented, it is a shame ...).
!! Do not forget : you will likely be the major winner if you document
!! properly.
!! Proper documentation of a structured datatype means :
!!  (1) Mention it in the list just below
!!  (2) Describe it in the NOTES section
!!  (3) Put it in alphabetical order in the the main section of this module
!!  (4) Document each of its records, except if they are described elsewhere
!!      (this exception is typically the case of the dataset associated with
!!      input variables, for which there is a help file)
!!
!! List of datatypes :
!! * bandstructure_type : different information about the band structure
!! * bcp_type : a "bonding critical point" for aim
!! * datafil_type : the data (units,filenames) related to files
!! * epsilonm1_results : for GW part of ABINIT, results of screening
!! * pawang_type : for PAW, ANGular mesh discretization and related data
!! * pawfgr_type : for PAW, Fine rectangular GRid parameters and related data
!! * pawfgrtab_type : for PAW, various arrays giving data related to fine grid for a given atom
!! * pawrad_type : for PAW, RADial mesh discretization and related data
!! * pawtab_type : for PAW, TABulated data initialized at start
!! * paw_an_type : for PAW, various arrays given on ANgular mesh or ANgular moments
!! * paw_ij_type : for PAW, various arrays given on (i,j) (partial waves) channels
!! * pseudopotential_type : for norm-conserving pseudopotential, all the
!!   information
!! * pspheader_paw_type : for PAW, the header of the atomic file
!! * pspheader_type : for norm-conserving pseudopotentials, the header of the file
!!
!! COPYRIGHT
!! Copyright (C) 2001-2012 ABINIT group (XG)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~abinit/COPYING
!! or http://www.gnu.org/copyleft/gpl.txt .
!!
!! NOTES
!! PAW structured datatypes to be described ...
!! * pawang_type : ANGular mesh discretization and related data
!! * pawfgr_type : Fine rectangular GRid parameters and related data
!! * pawfgrtab_type : various arrays giving data related to fine grid for a given atom
!! * pawrad_type :  RADial mesh discretization and related data
!! * pawtab_type : TABulated data initialized at start
!! * paw_an_type : various arrays given on ANgular mesh or
!! * paw_ij_type : various arrays given on (i,j) (partial waves) channels
!! * pspheader_paw_type: the header of the atomic file
!!
!! SOURCE

#if defined HAVE_CONFIG_H
#include "config.h"
#endif

#include "abi_common.h"

module defs_datatypes

 use defs_basis
 use defs_parameters

 implicit none

!Structures

!!***

!!****t* defs_datatypes/bandstructure_type
!! NAME
!! bandstructure_type
!!
!! FUNCTION
!! It contains different information about the band structure: eigenenergies, residuals, derivative of occupation
!! numbers vs energy in case of metallic occupations and Brillouin zone according to the context: k points,
!! occupation numbers, storage mode of wavefunctions, weights ...
!! For example, the initial Brillouin zone, set up in the dataset, will be treated in the response function part of
!! the code, to give a reduced Brillouin zone different from the original one, due to the breaking of the symmetries
!! related to the existence of a wavevector, or the lack of time-reversal invariance.
!!
!! SOURCE

 type bandstructure_type

! WARNING : if you modify this datatype, please check whether there might be creation/destruction/copy routines,
! declared in another part of ABINIT, that might need to take into account your modification.

  integer :: bantot                ! total number of bands (sum(nband(:))
  integer :: mband                 ! Max number of bands i.e MAXVAL(nband) (to dimension arrays)
  integer :: nkpt                  ! number of k points
  integer :: nspinor               ! 1 for collinear, 2 for noncollinear.
  integer :: nsppol                ! number of spin-polarizations
  integer :: occopt                ! Occupation option, see input variable.

  !integer :: kptopt
  !real(dp) :: tolwfr
  !real(dp),pointer :: resid(mband*nkpt*nsppol)
  !resid(mband*nkpt*nsppol)=residuals (hartree**2)

  real(dp) :: entropy              ! Entropy associated with the smearing (adimensional)
  real(dp) :: fermie               ! Fermi energy
  real(dp) :: nelect               ! Number of electrons.
  real(dp) :: tphysel              ! Physical temperature of electrons.
  real(dp) :: tsmear               ! Temperature of smearing.

  integer,pointer :: istwfk(:)   SET2NULL
  ! istwfk(nkpt)
  ! Storage mode at each k point.

  integer,pointer :: nband(:)  SET2NULL
  ! nband(nkpt*nsppol)
  ! Number of bands at each k point and spin-polarisation.

  integer,pointer :: npwarr(:)   SET2NULL
  ! npwarr(nkpt)
  ! Number of plane waves at each k point.

  real(dp),pointer :: kptns(:,:)  SET2NULL
  ! kptns(3,nkpt)
  ! k-point vectors.

  real(dp),pointer :: eig(:,:,:)   SET2NULL
  ! eig(mband,nkpt,nsppol)
  ! Eigenvalues of each band.

  !real(dp),pointer :: im_eig(:,:,:)   SET2NULL
  ! imeig(mband,nkpt,nsppol)
  ! Imaginary part of the eigenvalues (used in the BSE code).

  real(dp),pointer :: occ(:,:,:)   SET2NULL
  ! occ(mband,nkpt,nsppol)
  ! occupation of each band.

  real(dp),pointer :: doccde(:,:,:)   SET2NULL
  ! doccde(mband,nkpt,nsppol)
  ! derivative of the occupation of each band wrt energy (needed for RF).

  real(dp),pointer :: wtk(:)  SET2NULL
  ! wtk(nkpt)
  ! weight of each k point, normalized to one.

 end type bandstructure_type
!!***

!----------------------------------------------------------------------

!!****t* defs_datatypes/bcp_type
!! NAME
!! bcp_type
!!
!! FUNCTION
!! a "bonding critical point" for aim
!!
!! SOURCE

 type bcp_type

! WARNING : if you modify this datatype, please check whether there might be creation/destruction/copy routines,
! declared in another part of ABINIT, that might need to take into account your modification.


! Integer
  integer :: iat     !! number of the bonding atom inside a primitive cell
  integer :: ipos    !! number of the primitive cell of the bonding atom

! Real
  real(dp) :: chg     !! charge at the critical point
  real(dp) :: diff(3) !! three distances : AT-CP,BAT-CP,AT-BAT
  real(dp) :: ev(3)   !! eigenvalues of the Hessian
  real(dp) :: pom(3)  !! position of the bonding atom
  real(dp) :: rr(3)   !! position of the bcp
  real(dp) :: vec(3,3)!! eigenvectors of the Hessian
  real(dp) :: vv(3)   !! position of the bcp relative to the central atom

 end type bcp_type
!!***

!----------------------------------------------------------------------

!!****t* defs_datatypes/macro_uj_type
!! NAME
!! dtmacro_uj
!!
!! FUNCTION
!! This data type contains the potential shifts and the occupations
!! for the determination of U and J for the DFT+U calculations.
!! iuj=1,2: non-selfconsistent calculations. iuj=3,4 selfconsistent calculations.
!! iuj=2,4  => pawujsh<0 ; iuj=1,3 => pawujsh >0,
!!
!! SOURCE

 type macro_uj_type

! Integer
  integer :: iuj        !! dataset treated
  integer :: nat        !! number of atoms U (J) is determined on
  integer :: ndtset     !! total number of datasets
  integer :: nspden     !! number of densities treated
  integer :: macro_uj   !! which mode the determination runs in
  integer :: pawujat    !! which atom U (J) is determined on
  integer :: pawprtvol  !! controlling amount of output
  integer :: option     !! controls the determination of U (1 with compensating charge bath, 2 without)
  integer :: dmatpuopt  !! controls the renormalisation of the PAW projectors

! Real
  real(dp) :: diemix    !! mixing parameter
  real(dp) :: mdist     !! maximal distance of ions
  real(dp) :: pawujga   !! gamma for inversion of singular matrices
  real(dp) :: ph0phiint !! integral of phi(:,1)*phi(:,1)
  real(dp) :: pawujrad  !! radius to which U should be extrapolated.
  real(dp) :: pawrad    !! radius of the paw atomic data

! Integer arrays
  integer , pointer  :: scdim(:)   !! size of supercell

! Real arrays
  real(dp) , pointer :: occ(:,:)    !! occupancies after a potential shift: occ(ispden,nat)
  real(dp) , pointer :: rprimd(:,:) !! unit cell for symmetrization
  real(dp) , pointer :: vsh(:,:)    !! potential shifts on atoms, dimensions: nspden,nat
  real(dp) , pointer :: xred(:,:)   !! atomic position for symmetrization
  real(dp) , pointer :: wfchr(:)    !! wfchr(1:3): zion, n and l of atom on which projection is done
                                    !! wfchr(4:6): coefficients ai of a0+a1*r+a2*r^2, fit to the wfc for
                                    !! r< r_paw
  real(dp), pointer :: zioneff(:)   !! zioneff(ij_proj), "Effective charge"*n "seen" at r_paw, deduced from Phi at r_paw, n:
                                    !! pricipal quantum number; good approximation to model wave function outside PAW-sphere through

 end type macro_uj_type
!!***

!----------------------------------------------------------------------

!!****t* defs_datatypes/pawang_type
!! NAME
!! pawang_type
!!
!! FUNCTION
!! For PAW, ANGular mesh discretization and related data
!!
!! SOURCE

 type pawang_type

! WARNING : if you modify this datatype, please check whether there might be creation/destruction/copy routines,
! declared in another part of ABINIT, that might need to take into account your modification.

!Integer scalars

  integer :: angl_size
   ! Dimension of paw angular mesh
   ! angl_size=ntheta*nphi

  integer :: l_max
   ! Maximum value of angular momentum l+1

  integer :: l_size_max
   ! Maximum value of angular momentum +1
   ! leading to non-zero Gaunt coefficients
   ! l_size_max = 2*l_max-1

  integer :: ngnt
   ! Number of non zero Gaunt coefficients

  integer :: ntheta, nphi
   ! Dimensions of paw angular mesh

  integer :: nsym
   ! Number of symmetry elements in space group

  integer :: gnt_option
   ! Option for Gaunt coefficients:
   !  gnt_option==0, Gaunt coeffs are not computed (and not allocated)
   !  gnt_option==1, Gaunt coeffs are computed up to 2*l_size_max-1
   !  gnt_option==2, Gaunt coeffs are computed up to l_size_max

  integer :: use_ls_ylm
   ! Flag: use_ls_ylm=1 if pawang%ls_ylm pointer is allocated

  integer :: ylm_size
   ! Size of ylmr/ylmrgr arrays

!Integer arrays

  integer, pointer :: gntselect(:,:)
   ! gntselect(l_size_max**2,l_max**2*(l_max**2+1)/2)
   ! Selection rules for Gaunt coefficients stored as (LM,ij) where ij is in packed form.
   ! (if gntselect>0, Gaunt coeff. is non-zero)

!Real (real(dp)) arrays

  real(dp), pointer :: anginit(:,:)
   ! anginit(3,angl_size)
   ! For each point of the angular mesh, gives the coordinates
   ! of the corresponding point on an unitary sphere
   ! Not used in present version (5.3)

  real(dp), pointer :: angwgth(:)
   ! angwgth(angl_size)
   ! For each point of the angular mesh, gives the weight
   ! of the corresponding point on an unitary sphere

  real(dp), pointer :: ls_ylm(:,:,:)
   ! ls_ylm(2,l_max**2*(l_max**2+1)/2,2)
   ! LS operator in the real spherical harmonics basis
   ! ls_ylm(ilm1m2,ispin)= <sigma, y_lm1| LS |y_lm2, sigma_prime>

  real(dp), pointer :: realgnt(:)
   ! realgnt(ngnt)
   ! Non zero real Gaunt coefficients

  real(dp), pointer :: ylmr(:,:)
   ! ylmr(ylm_size,angl_size)
   ! Real Ylm calculated in real space

  real(dp), pointer :: ylmrgr(:,:,:)
   ! ylmrgr(1:3,ylm_size,angl_size)
   ! First gradients of real Ylm calculated in real space (cart. coordinates)

  real(dp), pointer :: zarot(:,:,:,:)
   !  zarot(l_size_max,l_size_max,l_max,nsym)
   !  Coeffs of the transformation of real spherical
   !  harmonics under the symmetry operations symrec.

 end type pawang_type
!!***

!----------------------------------------------------------------------

!!****t* defs_datatypes/pawfgr_type
!! NAME
!! pawfgr_type
!!
!! FUNCTION
!! For PAW, Fine rectangular GRid parameters and related data
!!
!! SOURCE

 type pawfgr_type

! WARNING : if you modify this datatype, please check whether there might be creation/destruction/copy routines,
! declared in another part of ABINIT, that might need to take into account your modification.

!Integer scalars

  integer :: mgfft, nfft
   ! Values of mffft and nfft for the fine rectangular grid:
   !   mgfft= max(ngfft(i)) [max. size of 1D FFT grid]
   !   nfft=ngfft1*ngfft2*ngfft3 [number of pts in the FFT box]

  integer :: mgfftc, nfftc
   ! Values of mffft and nfft for the COARSE rectangular grid:
   !   mgfftc= max(ngfftc(i)) [max. size of 1D FFT grid]
   !   nfftc=ngfftc1*ngfftc2*ngfftc3 [number of pts in the FFT box]

  integer :: usefinegrid
   ! Flag: =1 if a double-grid is used to convert spherical data
   !       to Fourier grid. =0 otherwise

!Integer arrays

  integer, pointer :: coatofin(:)
   ! coatofin(nfftc)
   ! Index of the points of the coarse grid on the fine grid

  integer, pointer :: fintocoa(:)
   ! fintocoa(nfft)
   ! Index of the points of the fine grid on the coarse grid
   !  (=0 if the point of the fine grid does not belong to the coarse grid)

  integer :: ngfft(18)
   ! ngfft(1:18)=integer array with FFT box dimensions and other
   ! information on FFTs, for the fine rectangular grid

  integer :: ngfftc(18)
   ! ngfft(1:18)=integer array with FFT box dimensions and other
   ! information on FFTs, for the COARSE rectangular grid

 end type pawfgr_type
!!***

!----------------------------------------------------------------------

!!****t* defs_datatypes/pawfgrtab_type
!! NAME
!! pawfgrtab_type
!!
!! FUNCTION
!! For PAW, various arrays giving data related to fine grid for a given atom
!!
!! SOURCE

 type pawfgrtab_type

! WARNING : if you modify this datatype, please check whether there might be creation/destruction/copy routines,
! declared in another part of ABINIT, that might need to take into account your modification.

!Integer scalars

  integer :: cplex
   ! cplex=1 if potentials/densities are real, 2 if they are complex

  integer :: expiqr_allocated
   ! 1 if expiqr() is allocated (and computed)

  integer :: itypat
   ! itypat=type of the atom

  integer :: l_size
   ! 1+maximum value of l leading to non zero Gaunt coeffs
   ! for the considered atom type

  integer :: gylm_allocated
   ! 1 if gylm() is allocated (and computed)

  integer :: gylmgr_allocated
   ! 1 if gylmgr() is allocated (and computed)

  integer :: gylmgr2_allocated
   ! 1 if gylmgr2() is allocated (and computed)

  integer :: nfgd
   ! Number of Fine rectangular GriD points
   ! in the paw sphere around considered atom

  integer :: nhatfr_allocated
   ! 1 if nhatfr() is allocated (and computed)

  integer :: nhatfrgr_allocated
   ! 1 if nhatfrgr() is allocated (and computed)

  integer :: nspden
   ! Number of spin-density components

  integer :: rfgd_allocated
   ! 1 if rfgd() is allocated (and computed)

!Integer arrays
  !integer :: ngfftf(18)
  ! Info on the dense FFT mesh.

  integer, pointer :: ifftsph(:)
   ! ifftsph(nfgd)
   ! Array giving the FFT index (fine grid) of a point in the paw
   ! sphere around considered atom (ifftsph=ix+n1*(iy-1+n2*(iz-1))

!Real (real(dp)) arrays

  real(dp), pointer :: expiqr(:,:)
   ! expiqr(2,nfgd)
   ! Gives exp(-i.q.r) on the fine rectangular grid
   ! where q is the phonons wavevector

  real(dp), pointer :: gylm(:,:)
   ! gylm(nfgd,l_size*l_size)
   ! Gives g_l(r)*Y_lm(r) on the fine rectangular grid
   ! around considered atom

  real(dp), pointer :: gylmgr(:,:,:)
   ! gylmgr(3,nfgd,l_size*l_size)
   ! Gives the gradient of g_l(r)*Y_lm(r) wrt cart. coordinates
   ! on the fine rectangular grid around considered atom

  real(dp), pointer :: gylmgr2(:,:,:)
   ! gylmgr(6,nfgd,l_size*l_size)
   ! Gives the second gradient of g_l(r)*Y_lm(r) wrt cart. coordinates
   ! on the fine rectangular grid around considered atom

  real(dp), pointer :: nhatfr(:,:)
   ! nhatfr(nfgd,nspden)
   ! Gives the "frozen part" of 1st-order compensation charge density
   ! on the fine rectangular grid around considered atom
   ! Only used in response function calculations

  real(dp), pointer :: nhatfrgr(:,:,:)
   ! nhatfrgr(3,nfgd,nspden)
   ! Gives the gradients (wrt cart. coordinates)
   ! of "frozen part" of 1st-order compensation charge density
   ! on the fine rectangular grid around considered atom
   ! Only used in response function calculations

  real(dp), pointer :: rfgd(:,:)
   ! r(3,nfgd)
   ! Gives all R vectors (r-r_atom) on the Fine rectangular GriD
   ! around considered atom in Cartesian coordinates.

 end type pawfgrtab_type
!!***

!-------------------------------------------------------------------------

!!****t* defs_datatypes/pawrad_type
!! NAME
!! pawrad_type
!!
!! FUNCTION
!! For PAW, RADial mesh discretization and related data
!!
!! SOURCE

 type pawrad_type

! WARNING : if you modify this datatype, please check whether there might be creation/destruction/copy routines,
! declared in another part of ABINIT, that might need to take into account your modification.

!Integer scalars

  integer :: int_meshsz
   ! Mesh size used in integrals computation
   ! Integrals will be computed up to r(int_meshsz)

  integer :: mesh_size
   ! Dimension of radial mesh

  integer :: mesh_type
   ! Type of mesh
   !     1=regular grid: r(i)=(i-1)*AA
   !     2=logarithmic grid: r(i)=AA*(exp[BB*(i-1)]-1)
   !     3=logarithmic grid: r(i>1)=AA*exp[BB*(i-1)] and r(1)=0
   !     4=logarithmic grid: r(i)=-AA*ln[1-BB*(i-1)] with BB=1/n

!Real (real(dp)) scalars

  real(dp) :: lstep
   ! Exponential step of the mesh (BB parameter above)
   ! Defined only if mesh type is logarithmic

  real(dp) :: rmax
   ! Max. value of r = rad(mesh_size)

  real(dp) :: rstep
   ! Radial step of the mesh (AA parameter above)

  real(dp) :: stepint
   ! Radial step used to convert any function from the
   ! present grid onto a regular grid in order to
   ! integrate it using trapeze method

!Real (real(dp)) arrays

  real(dp), pointer :: rad(:)
   ! rad(mesh_size)
   ! Coordinates of all the points of the mesh

  real(dp), pointer :: radfact(:)
   ! radfact(mesh_size)
   ! Factor used to compute radial integrals
   ! Before being integrated on the present mesh,
   ! any function is multiplied by this factor

  real(dp), pointer :: simfact(:)
   ! simfact(mesh_size)
   ! Factor used to compute radial integrals by the a Simpson scheme
   ! Integral[f] = Sum_i [simfact(i)*f(i)]

 end type pawrad_type
!!***

!!****t* defs_datatypes/pawtab_type
!! NAME
!! pawtab_type
!!
!! FUNCTION
!! For PAW, TABulated data initialized at start
!!
!! SOURCE

 type pawtab_type

! WARNING : if you modify this datatype, please check whether there might be creation/destruction/copy routines,
! declared in another part of ABINIT, that might need to take into account your modification.

!Integer scalars

  integer :: basis_size
   ! Number of elements for the paw nl basis on the considered atom type

  integer :: has_kij
   ! if 1, onsite matrix elements of the kinetic operator are allocated
   ! if 2, onsite matrix elements of the kinetic operator are computed and stored

  integer :: has_nabla
   ! if 1, onsite matrix elements of the nabla operator are allocated.
   ! if 2, onsite matrix elements of the nabla operator are computed and stored.

  integer :: has_vhntzc
   ! if 1, space for vhntzc is allocated
   ! if 2, vhntzc has been read from PAW file and stored

  integer :: has_vhnzc
   ! if 1, space for vhnzc is allocated
   ! if 2, vhnzc has been computed and stored

  integer :: ij_proj
   ! Number of (i,j) elements for the orbitals on which U acts (PAW+U only)
   ! on the considered atom type (ij_proj=1 (1 projector), 3 (2 projectors)...)
   ! Also used for local exact-exchange

  integer :: ij_size
   ! Number of (i,j) elements for the symetric paw basis
   ! on the considered atom type (ij_size=basis_size*(basis_size+1)/2)

  integer :: lcut_size
   ! Maximum value of l+1 leading to non zero Gaunt coeffs
   ! modified by dtset%pawlcutd
   ! lcut_size=min(2*l_max,dtset%pawlcutd)+1

  integer :: l_size
   ! Maximum value of l+1 leading to non zero Gaunt coeffs
   ! l_size=2*l_max-1

  integer :: lexexch
   ! lexexch gives l on which local exact-exchange is applied for a given type of atom.

  integer :: lmn_size
   ! Number of (l,m,n) elements for the paw basis

  integer :: lmn2_size
   ! lmn2_size=lmn_size*(lmn_size+1)/2
   ! where lmn_size is the number of (l,m,n) elements for the paw basis

  integer :: lmnmix_sz
   ! lmnmix_sz=number of klmn=(lmn,lmn_prime) verifying l<=lmix and l_prime<=lmix

  integer :: lpawu
   ! lpawu gives l on which U is applied for a given type of atom.

  integer :: nproju
   ! nproju is the number of projectors for orbitals on which paw+u acts.
   ! Also used for local exact-exchange

  integer :: mesh_size
   ! Dimension of radial mesh

  integer :: mqgrid
   ! Number of points in the reciprocal space grid on which
   ! the radial functions (tcorespl, tvalespl...) are specified
   ! Same as psps%mqgrid_vl

  integer :: shape_lambda
   ! Lambda parameter in gaussian shapefunction (shape_type=2)

  integer :: shape_type
   ! Radial shape function type
   ! shape_type=-1 ; g(r)=numeric (read from psp file)
   ! shape_type= 1 ; g(r)=[sin(pi*r/rshp)/(pi*r/rshp)]**2 if r<=rshp, zero if r>rshp
   ! shape_type= 2 ; g(r)=exp[-(r/sigma)**lambda]
   ! shape_type= 3 ; gl(r)=Alpha(1,l)*jl(q(1,l)*r)+Alpha(2,l)*jl(q(2,l)*r) for each l

  integer :: useexexch
   ! useexexch=0 ; do not use local exact-exchange
   ! useexexch=1 ; use local exact-exchange

  integer :: usepawu
   ! usepawu=0 ; do not use PAW+U formalism
   ! usepawu=1 ; use PAW+U formalism (Full localized limit)
   ! usepawu=2 ; use PAW+U formalism (Around Mean Field)

  integer :: usetcore
   ! Flag controling use of pseudized core density (0 if tncore=zero)

  integer :: usetvale
   ! Flag controling use of pseudized valence density (0 if tnval is unknown)

  integer :: usexcnhat
   ! 0 if compensation charge density is not included in XC terms
   ! 1 if compensation charge density is included in XC terms

!Real (real(dp)) scalars

  real(dp) :: dncdq0
   ! Gives 1/q d(tNcore(q))/dq for q=0
   ! (tNcore(q) = FT of pseudo core density)

  real(dp) :: dnvdq0
   ! Gives 1/q d(tNvale(q))/dq for q=0
   ! (tNvale(q) = FT of pseudo valence density)

  real(dp) :: exccore
   ! Exchange-correlation energy for the core density

  real(dp) :: exchmix
   ! mixing of exact exchange; default is 0.25 (PBE0)

  real(dp) :: f4of2_sla
   ! Ratio of Slater Integrals F4 and F2

  real(dp) :: f6of2_sla
   ! Ratio of Slater Integrals F6 and F4

  real(dp) :: jpawu
   ! jpawu
   ! Value of J parameter for paw+u for a given type.

  integer :: mqgrid_shp
   ! Number of points in the reciprocal space grid on which
   ! the radial shape functions (shapefncg) are given

  real(dp) :: rpaw
   ! Radius of PAW sphere

  real(dp) :: rshp
   ! Compensation charge radius (if r>rshp, g(r)=zero)

  real(dp) :: shape_sigma
   ! Sigma parameter in gaussian shapefunction (shape_type=2)

  real(dp) :: upawu
   ! upawu
   ! Value of U parameter for paw+u for a given type.


!Integer arrays

  integer, pointer :: indklmn(:,:)
   ! indklmn(6,lmn2_size)
   ! Array giving klm, kln, abs(il-jl), (il+jl), ilm and jlm for each klmn=(ilmn,jlmn)
   ! Note: ilmn=(il,im,in) and ilmn<=jlmn

  integer, pointer :: klmntomn(:,:)
   ! klmntomn(4,lmn2_size)
   ! Array giving im, jm ,in, and jn for each klmn=(ilmn,jlmn)
   ! Note: ilmn=(il,im,in) and ilmn<=jlmn
   ! NB: klmntomn is an application and not a bijection

  integer, pointer :: kmix(:)
   ! kmix(lmnmix_sz)
   ! Indirect array selecting the klmn=(lmn,lmn_prime) verifying l<=lmix and l_prime<=lmix

  integer, pointer :: lnproju(:)
   ! lnproju(nproju) gives ln (index for phi) for each projectors on which U acts (PAW+U only)
   ! nproju is 1 or 2 and  is the number of projectors for correlated orbitals
   ! Also used for local exact-exchange

!Real (real(dp)) arrays

  real(dp), pointer :: coredens(:)
   ! coredens(mesh_size)
   ! Gives the core density of the atom

  real(dp), pointer :: dij0(:)
   ! dij0(lmn2_size)
   ! Part of the Dij term (non-local operator) completely
   ! calculated in the atomic data part

  real(dp), pointer :: dltij(:)
   ! dltij(lmn2_size)
   ! Factor used to compute sums over klmn=(ilmn,jlmn)
   ! ((ilmn,ilmn) term has to be added once)
   ! dltij(klmn)=1 if ilmn=jlmn, else dltij(klmn)=2

  real(dp), pointer :: dshpfunc(:,:,:)
   ! shapefunc(mesh_size,l_size,4)
   ! Gives the 4 first derivatives of  radial shape function
   ! for each l component; used only if shape_type=-1

  real(dp), pointer :: eijkl(:,:)
   ! eijkl(lmn2_size,lmn2_size)
   ! Part of the Dij term (non-local operator) that depends only from
   ! the projected occupation coeffs in the self-consistent loop

  real(dp), pointer :: fk(:,:)
   ! fk(6,4)
   ! Slater integrals used for local exact exchange

  real(dp), pointer :: gnorm(:)
   ! gnorm(l_size)
   ! Give the the normalization factor of each radial shape function

  real(dp),pointer :: kij(:)
   ! Onsite matrix elements <phi|\kinetic|phj>-<tphi|\kinetic|tphj>

  real(dp),pointer :: nabla_ij(:,:,:)
   ! nabla_ij(3,lmn_size,lmn_size))
   ! Onsite matrix elements <phi|\nabla|phj>-<tphi|\nabla|tphj>

  real(dp), pointer :: phi(:,:)
   ! phi(mesh_size, basis_size)
   ! Gives the paw electron wavefunctions on the radial grid

  real(dp), pointer :: phiphj(:,:)
   ! phiphj(mesh_size,ij_size)
   ! Useful product Phi(:,i)*Phi(:,j)

  real(dp), pointer :: phiphjint(:)
   ! phiphjint(ij_proj)
   ! Integration of Phi(:,i)*Phi(:,j) for LDA+U/local exact-exchange occupation matrix

  real(dp), pointer :: ph0phiint(:)
   ! ph0phjint(ij_proj)
   ! Integration of Phi(:,1)*Phi(:,j) for LDA+DMFT projections

  real(dp), pointer :: qgrid_shp(:)
   ! qgrid_shp(mqgrid_shp)
   ! Grid of points in reciprocal space on which the shape functions are given

  real(dp), pointer :: qijl(:,:)
   ! qijl(l_size**2,lmn2_size)
   ! The qijl are the moments of the charge density difference between
   ! the AE and PS partial wave for each channel (i,j). They take part
   ! to the building of the compensation charge

  real(dp), pointer :: rad_for_spline(:)
   ! rad_for_spline(mesh_size)
   ! Radial mesh used to spline quantities on radial mesh;
   ! Allocated and used only when
   !     shape_type=-1 (numerical shape function)
   !  or usedvloc=1 (use of vloc derivative)

  real(dp), pointer :: rhoij0(:)
   ! rhoij0(lmn2_size)
   ! Initial guess for rhoij

  real(dp), pointer :: shape_alpha(:,:)
   ! shape_alpha(2,l_size)
   ! Alpha_i parameters in Bessel shapefunctions (shape_type=3)

  real(dp), pointer :: shape_q(:,:)
   ! shape_q(2,l_size)
   ! Q_i parameters in Bessel shapefunctions (shape_type=3)

  real(dp), pointer :: shapefunc(:,:)
   ! shapefunc(mesh_size,l_size)
   ! Gives the normalized radial shape function for each l component

  real(dp), pointer :: shapefncg(:,:,:)
   ! shapefncg(mqgrid_shp,2,l_size)
   ! Gives the spherical Fourier transform of the radial shape function
   ! for each l component (for each qgrid_shp(i)) + second derivative

  real(dp), pointer :: sij(:)
   ! sij(lmn2_size)
   ! Nonlocal part of the overlap operator

  real(dp), pointer :: tcoredens(:)
   ! tcoredens(mesh_size)
   ! Gives the pseudo core density of the atom

  real(dp), pointer :: tcorespl(:,:)
   ! tcorespl(mqgrid,2)
   ! Gives the pseudo core density in reciprocal space on a regular grid

  real(dp), pointer :: tphi(:,:)
   ! tphi(mesh_size,basis_size)
   ! Gives, on the radial grid, the paw atomic pseudowavefunctions

  real(dp), pointer :: tphitphj(:,:)
   ! tphitphj(mesh_size,ij_size)
   ! Useful product tPhi(:,i)*tPhi(:,j)

  real(dp), pointer :: tvalespl(:,:)
   ! tvalespl(mqgrid,2)
   ! Gives the pseudo valence density in reciprocal space on a regular grid

  real(dp), pointer :: Vee(:,:,:,:)
   ! PAW+U:
   ! Screened interaction matrix deduced from U and J parameters
   ! computed on the basis of orbitals on which U acts.

  real(dp), pointer :: Vex(:,:,:,:,:)
   ! Local exact-exchange:
   ! Screened interaction matrix deduced from calculation of Slater integrals
   ! computed on the basis of orbitals on which local exact exchange acts.

  real(dp), pointer :: VHntZC(:)
   ! VHntZC(mesh_size)
   ! Hartree potential for pseudized Zc density, v_H[\tilde{n}_{Zc}]
   ! read in from PAW file

  real(dp), pointer :: VHnZC(:)
   ! VHnZC(mesh_size)
   ! Hartree potential for Zc density, v_H[n_{Zc}]
   ! constructed from core density in PAW file (see psp7in.F90)

  real(dp), pointer :: zioneff(:)
   ! zioneff(ij_proj)
   ! "Effective charge"*n "seen" at r_paw, deduced from Phi at r_paw, n:
   ! pricipal quantum number
   ! good approximation to model wave function outside PAW-sphere through

 end type pawtab_type
!!***

!-------------------------------------------------------------------------

!!****t* defs_datatypes/paw_an_type
!! NAME
!! paw_an_type
!!
!! FUNCTION
!! For PAW, various arrays given on ANgular mesh or ANgular moments
!!
!! SOURCE

 type paw_an_type

! WARNING : if you modify this datatype, please check whether there might be creation/destruction/copy routines,
! declared in another part of ABINIT, that might need to take into account your modification.

!Integer scalars

  integer :: angl_size
   ! Dimension of paw angular mesh (angl_size=ntheta*nphi)

  integer :: cplex
   ! cplex=1 if potentials/densities are real, 2 if they are complex

  integer :: has_kxc
   ! set to 1 if xc kernels kxc1 and kxct1 are allocated and used
   !        2 if they are already computed

  integer :: has_vhartree
   ! set to 1 if vh1 and vht1 are allocated and used
   !        2 if they are already computed

  integer :: has_vxc
   ! set to 1 if vxc1 and vxct1 are allocated and used
   !        2 if they are already computed

  integer :: has_vxcval
   ! set to 1 if vxc1_val and vxct1_val are allocated and used
   !        2 if they are already computed

  integer :: itypat
   ! itypat=type of the atom

  integer :: lm_size
   ! lm_size=(l_size)**2
   ! l is Maximum value of l+1 leading to non zero Gaunt coeffs (l_size=2*l_max+1)

  integer :: mesh_size
   ! Dimension of radial mesh

  integer :: nkxc1
   ! number of independent components of Kxc1 and Kxct1
   ! (usually 3 for LDA, 23 for GGA)

  integer :: nspden
   ! Number of spin-density components

!Logical arrays

  logical, pointer :: lmselect(:)
   ! lmselect(lm_size)
   ! lmselect(ilm)=select the non-zero LM-moments of "one-center" densities/potentials

!Real (real(dp)) arrays

  real(dp), pointer :: kxc1 (:,:,:)
   ! kxc1(cplex*mesh_size,lm_size or angl_size,nkxc1)
   ! Gives xc kernel inside the sphere
   !   (theta,phi) values of kernel if pawxcdev=0
   !   LM-moments of kernel if pawxcdev/=0

  real(dp), pointer :: kxct1 (:,:,:)
   ! kxct1(cplex*mesh_size,lm_size or angl_size,nkxc1)
   ! Gives xc pseudo kernel inside the sphere
   !   (theta,phi) values of kernel if pawxcdev=0
   !   LM-moments of kernel if pawxcdev/=0

  real(dp), pointer :: vh1 (:,:,:)
   ! vh1(cplex*mesh_size,lm_size,nspden)
   ! Gives Hartree potential LM-moments inside the sphere

  real(dp), pointer :: vht1 (:,:,:)
   ! vht1(cplex*mesh_size,lm_size,nspden)
   ! Gives Hartree tilde potential LM-moments inside the sphere

  real(dp), pointer :: vxc1 (:,:,:)
   ! vxc1(cplex*mesh_size,lm_size or angl_size,nspden)
   ! Gives xc potential inside the sphere
   !   (theta,phi) values of potential if pawxcdev=0
   !   LM-moments of potential if pawxcdev/=0

  real(dp), pointer :: vxc1_val (:,:,:)
   ! vxc1_val(cplex*mesh_size,lm_size or angl_size,nspden) (Usually real, Mainly used for GW)
   ! Gives xc potential inside the sphere arising from valence only electrons
   !   (theta,phi) values of potential if pawxcdev=0
   !   LM-moments of potential if pawxcdev/=0

  real(dp), pointer :: vxct1 (:,:,:)
   ! vxct1(cplex*mesh_size,angl_size,nspden)
   ! Gives xc pseudo potential inside the sphere
   !   (theta,phi) values of potential if pawxcdev=0
   !   LM-moments of potential if pawxcdev/=0

  real(dp), pointer :: vxct1_val (:,:,:)
   ! vxct1_val(cplex*mesh_size,angl_size,nspden) (Usually real, Mainly used for GW)
   ! Gives xc pseudo potential inside the sphere
   !   (theta,phi) values of potential if pawxcdev=0
   !   LM-moments of potential if pawxcdev/=0

  real(dp), pointer :: vxc_ex (:,:,:)
   ! vxc_ex(cplex*mesh_size,angl_size,nspden)
   ! Gives xc  potential for local exact exchange inside the sphere
   !   (theta,phi) values of potential if pawxcdev=0
   !   LM-moments of potential if pawxcdev/=0

 end type paw_an_type
!!***

!-------------------------------------------------------------------------

!!****t* defs_datatypes/Paw_an_flags_type
!! NAME
!! Paw_an_flags_type
!!
!! FUNCTION
!! For PAW, the various "has flags" defined in the paw_an_type
!!
!! SOURCE

 type Paw_an_flags_type

  integer :: has_kxc
  integer :: has_vhartree
  integer :: has_vxc
  integer :: has_vxcval

 end type Paw_an_flags_type
!!***

!----------------------------------------------------------------------

!!****t* defs_datatypes/paw_ij_type
!! NAME
!! paw_ij_type
!!
!! FUNCTION
!! For PAW, various arrays given on (i,j) (partial waves) channels
!!
!! SOURCE

 type paw_ij_type

! WARNING : if you modify this datatype, please check whether there might be creation/destruction/copy routines,
! declared in another part of ABINIT, that might need to take into account your modification.

!Integer scalars

  integer :: cplex
   ! cplex=1 if all on-site PAW quantities are real, 2 if they are complex
   ! cplex=2 is useful for RF calculations

  integer :: cplex_dij
   ! cplex=1 if dij are real, 2 if they are complex

  integer :: has_dij
   ! 1 if dij is allocated
   ! 2 if dij is already computed

  integer :: has_dij0
   ! 1 if dij0 is allocated
   ! 2 if dij0 is already computed

  integer :: has_dijexxc
   ! 1 if dijexxc is associated and used, 0 otherwise
   ! 2 if dijexxc is already computed

  integer :: has_dijfr
   ! 1 if dijfr is allocated
   ! 2 if dijfr is already computed

  integer :: has_dijhartree
   ! 1 if dijhartree is allocated
   ! 2 if dijhartree is already computed

  integer :: has_dijhat
   ! 1 if dijhat is allocated
   ! 2 if dijhat is already computed

  integer :: has_dijso
   ! 1 if dijso is associated and used, 0 otherwise
   ! 2 if dijso is already computed

  integer :: has_dijU
   ! 1 if dijU is associated and used, 0 otherwise
   ! 2 if dijU is already computed

  integer :: has_dijxc
   ! 1 if dijxc is associated and used, 0 otherwise
   ! 2 if dijxc is already computed

  integer :: has_dijxc_hat
   ! 1 if dijxc_hat is associated and used, 0 otherwise
   ! 2 if dijxc_hat is already computed

  integer :: has_dijxc_val
   ! 1 if dijxc_val is associated and used, 0 otherwise
   ! 2 if dijxc_val is already computed

  integer :: has_exexch_pot
   ! 1 if PAW+(local exact exchange) potential is allocated

  integer :: has_pawu_occ
   ! 1 if PAW+U occupations are allocated

  integer :: itypat
   ! itypat=type of the atom

  integer :: lmn_size
   ! Number of (l,m,n) elements for the paw basis

  integer :: lmn2_size
   ! lmn2_size=lmn_size*(lmn_size+1)/2
   ! where lmn_size is the number of (l,m,n) elements for the paw basis

  integer :: ndij
   ! Number of components of dij
   ! Usually ndij=nspden, except for nspinor==2 (where ndij=nspinor**2)

  integer :: nspden
   ! Number of spin-density components (may be different from dtset%nspden if spin-orbit)

  integer :: nsppol
   ! Number of independant spin-components

!Real (real(dp)) arrays

  real(dp), pointer :: dij(:,:)
   ! dij(cplex_dij*lmn2_size,ndij)
   ! Dij term (non-local operator)
   ! May be complex if cplex_dij=2
   !  dij(:,:,1) contains Dij^up-up
   !  dij(:,:,2) contains Dij^dn-dn
   !  dij(:,:,3) contains Dij^up-dn (only if nspinor=2)
   !  dij(:,:,4) contains Dij^dn-up (only if nspinor=2)

  real(dp), pointer :: dij0(:)
   ! dij0(lmn2_size)
   ! Atomic part of Dij (read from PAW dataset)

  real(dp), pointer :: dijexxc(:,:)
   ! dijexxc(cplex_dij*lmn2_size,ndij)
   ! Onsite matrix elements of the Fock operator

  real(dp), pointer :: dijfr(:,:)
   ! dijhat(cplex_dij*lmn2_size,ndij)
   ! For response function calculation only
   ! RF Frozen part of Dij (depends on q vector but not on 1st-order wave function)
   ! Same storage as Dij (see above)

  real(dp), pointer :: dijhartree(:)
   ! dijhartree(cplex*lmn2_size)
   ! Dij_hartree term
   ! Contains all contributions to Dij from hartree
   ! Warning: Dimensioned by cplex, not cplex_dij
   ! Same storage as Dij (see above)

  real(dp), pointer :: dijhat(:,:)
   ! dijhat(cplex_dij*lmn2_size,ndij)
   ! Dij_hat term (non-local operator) i.e \sum_LM \int_FFT Q_{ij}^{LM} vtrial
   ! Same storage as Dij (see above)

  real(dp), pointer :: dijU(:,:)
   ! dijU(cplex_dij*lmn2_size,ndij)
   ! Onsite matrix elements of the U part of the PAW Hamiltonian.
   ! Same storage as Dij (see above)

  real(dp), pointer :: dijso(:,:)
   ! dijso(cplex_dij*lmn2_size,ndij)
   ! Onsite matrix elements of L.S i.e <phi_i|L.S|phi_j>
   ! Same storage as Dij (see above)

  real(dp), pointer :: dijxc(:,:)
   ! dijxc(cplex_dij*lmn2_size,ndij)
   ! Onsite matrix elements of vxc i.e
   ! <phi_i|vxc[n1+nc]|phi_j> - <tphi_i|vxc(tn1+nhat+tnc]|tphi_j>
   ! Same storage as Dij (see above)

  real(dp), pointer :: dijxc_hat(:,:)
   ! dijxc_hat(cplex_dij*lmn2_size,ndij)
   ! Dij_hat term i.e \sum_LM \int_FFT Q_{ij}^{LM} Vxc
   ! Same storage as Dij (see above)

  real(dp), pointer :: dijxc_val(:,:)
   ! dijxc_val(cplex_dij*lmn2_size,ndij)
   ! Onsite matrix elements of valence-only vxc i.e
   ! <phi_i|vxc[n1]|phi_j> - <tphi_i|vxc(tn1+nhat]|tphi_j>
   ! Same storage as Dij (see above)

  real(dp), pointer :: noccmmp(:,:,:,:)
   ! noccmmp(cplex_dij,2*lpawu+1,2*lpawu+1,nocc_nspden)
   ! cplex_dij=1 if collinear
   ! cplex_dij=2 if spin orbit is used
   ! cplex_dij=2 is used if non-collinear (for coherence, it is not necessary in this case, however)
   ! gives occupation matrix for lda+u (computed in setnoccmmp)
   ! Stored as: noccmmp(:,:,1)=   n^{up,up}_{m,mp}
   !            noccmmp(:,:,2)=   n^{dn,dn}_{m,mp}
   !            noccmmp(:,:,3)=   n^{up,dn}_{m,mp}
   !            noccmmp(:,:,4)=   n^{dn,up}_{m,mp}
   ! noccmmp(m,mp,:) is computed from rhoij(klmn) with  m=klmntomn(2)>mp=klmntomn(1)

  real(dp), pointer :: nocctot(:)
   ! nocctot(nspden)
   ! gives trace of occupation matrix for lda+u (computed in pawdenpot)
   ! for each value of ispden (1 or 2)

  real(dp), pointer :: vpawx(:,:,:)
   ! vpawx(2*lexexch+1,2*lexexch+1,nspden)
   ! exact exchange potential

 end type paw_ij_type
!!***

!----------------------------------------------------------------------

!!****t* defs_datatypes/paw_ij_flags_type
!! NAME
!! paw_ij_flags_type
!!
!! FUNCTION
!! For PAW, the various "has flags" defined in the paw_ij_type
!!
!! SOURCE

 type paw_ij_flags_type

  integer :: has_dij
  integer :: has_dij0
  integer :: has_dijexxc
  integer :: has_dijfr
  integer :: has_dijhartree
  integer :: has_dijhat
  integer :: has_dijso
  integer :: has_dijU
  integer :: has_dijxc
  integer :: has_dijxc_hat
  integer :: has_dijxc_val
  integer :: has_exexch_pot
  integer :: has_pawu_occ

 end type paw_ij_flags_type
!!***

!----------------------------------------------------------------------

!!****t* defs_datatypes/pseudopotential_gth_type
!! NAME
!! pseudopotential_gth_type
!!
!! FUNCTION
!! This structure is a sub-structure of pseudopotential_type used to
!! store parameters from the GTH pseudo-potentials. All arrays have
!! indices running on 1:npsp for each read pseudo-file. The 'set' array
!! is a check array, since several different pseudo can be used in a simulation
!! it set a flag for each npsp if params have been set or not. This is
!! redundant with psps%pspcod in the way that when psps%pspcod(i) is 2,
!! then gth_params%set(i) is .true.. GTH pseudo previous to wavelets introduction
!! doesn't have geometric informations. These have been added on the last line.
!! It is three radius informations, the %hasGeometry flag is there to know
!! which kind of pseudo has been read.
!!
!! SOURCE

 type pseudopotential_gth_type

! WARNING : if you modify this datatype, please check whether there might be creation/destruction/copy routines,
! declared in another part of ABINIT, that might need to take into account your modification.

  real(dp), pointer :: psppar(:, :, :)
   ! These are {rloc, C(1...4)} coefficients for psppar(0, :, :) indices,
   ! Followed by the h coefficients for psppar(1:2, :, :) indices.
   !  size (0:2, 0:4, npsp)

  real(dp), pointer :: radii_cov(:)
   ! The covalence radii for each pseudo (?) size (npsp)

  real(dp), pointer :: radii_cf(:, :)
   ! Cut-off radii for core part and long-range part.
   ! radii_cf(:, 1) is for the long-range cut-off and
   ! radii_cf(:, 2) is for the core cut-off. size (npsp, 2)

  integer, pointer :: semicore(:)
   ! The semicore code, indicated as an integer.
   ! The integer is the n_s + 4*n_p + 16* n_d + 64* n_f
   ! where n_l are the number of semicore orbitals for a given angular momentum
   ! starting from the lower level of course

  real(dp), pointer :: psp_k_par(:, :, :)
   ! Spin orbit coefficients in HGH/GTH formats: k11p etc... see psp3ini.F90
   !   dimension = num l channels, 3 coeffs, num psp = (1:lmax+1,1:3,npsp)

  logical, pointer :: hasGeometry(:)
   ! Flag for geometric informations in the pseudo. size (npsp)

  logical, pointer :: set(:)
   ! Consistency array, used for checking size (npsp)

 end type pseudopotential_gth_type
!!***

!----------------------------------------------------------------------

!!****t* defs_datatypes/pseudopotential_type
!! NAME
!! pseudopotential_type
!!
!! FUNCTION
!! This structured datatype contains all the information about one
!! norm-conserving pseudopotential, including the description of the local
!! and non-local parts, the different projectors, the non-linear core
!! correction ...
!!
!! SOURCE

 type pseudopotential_type

! WARNING : if you modify this datatype, please check whether there might be creation/destruction/copy routines,
! declared in another part of ABINIT, that might need to take into account your modification.


! Integer scalars

  integer :: dimekb
   ! Dimension of Ekb
   ! ->Norm conserving : Max. number of Kleinman-Bylander energies
   !                     for each atom type
   !                     dimekb=lnmax (lnmax: see this file)
   ! ->PAW : Max. number of Dij coefficients connecting projectors
   !                     for each atom type
   !                     dimekb=lmnmax*(lmnmax+1)/2 (lmnmax: see this file)

  integer :: lmnmax
   !  If useylm=0, max number of (l,n)   comp. over all type of psps (lnproj)
   !  If useylm=1, max number of (l,m,n) comp. over all type of psps (lmnproj)
   !  If mpspso is 2, lmnmax takes into account the spin-orbit projectors,
   !  so, it is equal to the max of lmnprojso or lnprojso, see pspheader_type

  integer :: lnmax
   !  Max. number of (l,n) components over all type of psps
   !  If mpspso is 2, lmnmax takes into account the spin-orbit projectors,
   !  so, it is equal to the max of lnprojso, see pspheader_type

  integer :: mproj    ! TO BE SUPPRESSED
   ! Maximum number of non-local projectors over all angular momenta
   !  and type of psps
   ! 0 only if all psps are local

  integer :: mpsang
   ! Highest angular momentum of non-local projectors over all type of psps.
   ! shifted by 1 : for all local psps, mpsang=0; for largest s, mpsang=1,
   ! for largest p, mpsang=2; for largest d, mpsang=3; for largest f, mpsang=4
   ! This gives also the number of non-local "channels"

  integer :: mpspso
   ! mpspso is set to 1 if none of the psps is used with a spin-orbit part (that
   !  is, if the user input variable so_psp is not equal
   !  to 1 in at least one case
   ! otherwise, it is set to 2

  integer :: mpssoang
   ! Maximum number of channels, including those for treating the spin-orbit coupling
   ! when mpspso=1, mpssoang=mpsang
   ! when mpspso=2, mpssoang=2*mpsang-1

  integer :: mqgrid_ff
   ! Number of points in the reciprocal space grid on which
   ! the radial functions ffspl are specified

  integer :: mqgrid_vl
   ! Number of points in the reciprocal space grid on which
   ! the radial functions vlspl are specified

  integer :: mtypalch
   ! Maximum number of alchemical pseudo atoms. If non-zero,
   ! the mechanism to generate mixing of pseudopotentials is activated

  integer :: npsp
   ! Number of types of pseudopotentials

  integer :: npspalch
   ! Number of types of pseudopotentials use for alchemical purposes

  integer :: ntypat
   ! Number of types of atoms (might be alchemy wrt pseudopotentials)

  integer :: ntypalch
   ! Number of types of alchemical pseudoatoms

  integer :: ntyppure
   ! Number of types of pure pseudoatoms

  integer :: n1xccc
   ! Number of radial points for the description of the pseudo-core charge
   ! (in the framework of the non-linear XC core correction)

  integer :: optnlxccc
   ! Option for the choice of non-linear XC core correction treatment (see the input variable)

  integer :: positron
   ! Option for the choice of type of GS calculation (electron or positron)

  integer :: usepaw
   ! if usepaw=0 , use norm-conserving psps part of the code
   ! is usepaw=1 , use paw part of the code

  integer :: useylm
   ! governs the way the nonlocal operator is to be applied:
   !   1=using Ylm, 0=using Legendre polynomials

! Logical scalars

  logical :: vlspl_recipSpace
   ! governs if vlspl is compute in reciprocal space or in real
   ! space (when available).

! Integer arrays

  integer, pointer :: algalch(:)   SET2NULL
   ! algalch(ntypalch)
   ! For each type of pseudo atom, the algorithm to mix the pseudopotentials

  integer, pointer :: indlmn(:,:,:)  SET2NULL
   ! indlmn(6,lmnmax,ntypat)
   ! For each type of psp,
   ! array giving l,m,n,lm,ln,spin for i=ln  (if useylm=0)
   !                                or i=lmn (if useylm=1)

  integer, pointer :: pspdat(:)  SET2NULL
   ! pspdat(ntypat)
   ! For each type of psp, the date of psp generation, as given by the psp file

  integer, pointer :: pspcod(:)  SET2NULL
   ! pspcod(npsp)
   ! For each type of psp, the format -or code- of psp generation,
   !  as given by the psp file

  integer, pointer :: pspso(:)  SET2NULL
   ! pspso(ntypat)
   ! For each type of psp, 1 if no spin-orbit component is taken
   ! into account, 2 if a spin-orbit component is used

  integer, pointer :: pspxc(:)   SET2NULL
   ! pspxc(ntypat)
   ! For each type of psp, the XC functional that was used to generate it,
   ! as given by the psp file

! Real (real(dp)) arrays

  real(dp), pointer :: ekb(:,:)  SET2NULL
   ! ekb(dimekb,ntypat*(1-usepaw))
   !  ->NORM-CONSERVING PSPS ONLY:
   !    (Real) Kleinman-Bylander energies (hartree)
   !           for number of basis functions (l,n) (lnmax)
   !           and number of atom types (ntypat)
   ! NOTE (MT) : ekb (norm-conserving) is now diagonal (one dimension
   !             lnmax); it would be easy to give it a second
   !             (symmetric) dimension by putting
   !             dimekb=lnmax*(lnmax+1)/2
   !             in the place of dimekb=lmnmax.

  real(dp), pointer :: ffspl(:,:,:,:)  SET2NULL
   ! ffspl(mqgrid_ff,2,lnmax,ntypat)
   ! Gives, on the radial grid, the different non-local projectors,
   ! in both the norm-conserving case, and the PAW case

  real(dp), pointer :: mixalch(:,:)  SET2NULL
   ! mixalch(npspalch,ntypalch)
   ! Mixing coefficients to generate alchemical pseudo atoms

  real(dp), pointer :: qgrid_ff(:)  SET2NULL
   ! qgrid_ff(mqgrid_ff)
   ! The coordinates of all the points of the radial grid for the nl form factors

  real(dp), pointer :: qgrid_vl(:)  SET2NULL
   ! qgrid_vl(mqgrid_vl)
   ! The coordinates of all the points of the radial grid for the local part of psp

  real(dp), pointer :: vlspl(:,:,:)  SET2NULL
   ! vlspl(mqgrid_vl,2,ntypat)
   ! Gives, on the radial grid, the local part of each type of psp.

  real(dp), pointer :: dvlspl(:,:,:)  SET2NULL
   ! dvlspl(mqgrid_vl,2,ntypat)
   ! Gives, on the radial grid, the first derivative of the local
   ! part of each type of psp (computed when the flag 'vlspl_recipSpace' is true).

  real(dp), pointer :: xcccrc(:)  SET2NULL
   ! xcccrc(ntypat)
   ! Gives the maximum radius of the pseudo-core charge, for each type of psp.

  real(dp), pointer :: xccc1d(:,:,:)  SET2NULL
   ! xccc1d(n1xccc*(1-usepaw),6,ntypat)
   ! Norm-conserving psps only
   ! The component xccc1d(n1xccc,1,ntypat) is the pseudo-core charge
   ! for each type of atom, on the radial grid. The components
   ! xccc1d(n1xccc,ideriv,ntypat) give the ideriv-th derivative of the
   ! pseudo-core charge with respect to the radial distance.

  real(dp), pointer :: zionpsp(:)  SET2NULL
   ! zionpsp(npsp)
   ! For each pseudopotential, the ionic pseudo-charge
   ! (giving raise to a long-range coulomb potential)

  real(dp), pointer :: ziontypat(:)  SET2NULL
   ! ziontypat(ntypat)
   !  For each type of atom (might be alchemy wrt psps), the ionic pseudo-charge
   ! (giving raise to a long-range coulomb potential)

  real(dp), pointer :: znuclpsp(:)  SET2NULL
   ! znuclpsp(npsp)
   ! The atomic number of each pseudopotential

  real(dp), pointer :: znucltypat(:)  SET2NULL
   ! znucltypat(ntypat)
   ! The atomic number of each type of atom (might be alchemy wrt psps)

! Character arrays

  character(len=fnlen), pointer :: filpsp(:)  SET2NULL
   ! filpsp(ntypat)
   ! The filename of the pseudopotential

  character(len=fnlen), pointer :: title(:)   SET2NULL
   ! title(ntypat)
   ! The content of first line read from the psp file

  type(pseudopotential_gth_type) :: gth_params
   ! Types for pseudo-potentials that are based on parameters. Currently, only
   ! GTH are supported (see pseudopotential_gth_type). To add one, one should
   ! create an initialisation method and a destruction method in 02psp (see
   ! psp2params.F90). These methods are called in driver().

 end type pseudopotential_type
!!***

!----------------------------------------------------------------------

!!****t* defs_datatypes/pspheader_paw_type
!! NAME
!! pspheader_paw_type
!!
!! FUNCTION
!! The pspheader_paw_type structured datatype gather additional information
!! about a PAW pseudopotential file, from its header.
!!
!! SOURCE

 type pspheader_paw_type

! WARNING : if you modify this datatype, please check whether there might be creation/destruction/copy routines,
! declared in another part of ABINIT, that might need to take into account your modification.

  integer :: basis_size    ! Number of elements of the wf basis ((l,n) quantum numbers)
  integer :: l_size        ! Maximum value of l+1 leading to a non zero Gaunt coefficient
  integer :: lmn_size      ! Number of elements of the paw basis
  integer :: mesh_size     ! Dimension of (main) radial mesh
  integer :: pawver        ! Version number of paw psp format
  integer :: shape_type    ! Type of shape function
  real(dp) :: rpaw         ! Radius for paw spheres
  real(dp) :: rshp         ! Cut-off radius of shape function

 end type pspheader_paw_type
!!***

!----------------------------------------------------------------------

!!****t* defs_datatypes/pspheader_type
!! NAME
!! pspheader_type
!!
!! FUNCTION
!! The pspheader_type structured datatype gather different information
!! about a pseudopotential file, from its header.
!!
!! SOURCE

 type pspheader_type

! WARNING : if you modify this datatype, please check whether there might be creation/destruction/copy routines,
! declared in another part of ABINIT, that might need to take into account your modification.

  integer :: nproj(0:3) ! number of scalar projectors for each angular momentum
  integer :: nprojso(3) ! number of spin-orbit projectors for each angular momentum
  integer :: lmax       ! maximum l quantum number (-1 if only local)
                        ! Example : s only       -> lmax=0
                        !           s and p      -> lmax=1
                        !           d only       -> lmax=2
  integer :: pspcod     ! code number of the pseudopotential
  integer :: pspdat     ! date of generation of the pseudopotential
  integer :: pspxc      ! exchange-correlation functional
  integer :: pspso      ! spin-orbit characteristics
  integer :: xccc       ! =0 if no XC core correction, non-zero if XC core correction

  real(dp) :: zionpsp       ! charge of the ion made of core electrons only
  real(dp) :: znuclpsp      ! atomic number of the nuclei

  real(dp) :: GTHradii(0:4) ! Radii values for GTH (and HGH) family potentials

  character(len=fnlen) :: filpsp   ! name of the psp file
  character(len=fnlen) :: title    ! content of first line read from the psp file

  type(pspheader_paw_type) :: pawheader ! only for PAW psps. See above

 end type pspheader_type
!!***

end module defs_datatypes
!!***
