!{\src2tex{textfont=tt}}
!!****f* ABINIT/make_efg_paw
!! NAME
!! make_efg_paw
!!
!! FUNCTION
!! Compute the electric field gradient due to the PAW cores
!!
!! COPYRIGHT
!! Copyright (C) 2005-2009 ABINIT group (JJ,MT)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~ABINIT/Infos/copyright
!! or http://www.gnu.org/copyleft/gpl.txt .
!!
!! INPUTS
!!  natom=number of atoms in cell.
!!  ntypat=number of atom types
!!  pawang <type(pawang_type)>=paw angular mesh and related data
!!  pawrad(ntypat) <type(pawrad_type)>=paw radial mesh and related data
!!  pawrhoij(natom) <type(pawrhoij_type)>= paw rhoij occupancies and related data
!!  pawtab(ntypat) <type(pawtab_type)>=paw tabulated starting data
!!  typat(natom)=type (integer) for each atom
!!
!! OUTPUT
!!  efg(3,3,natom), the 3x3 efg tensor at each site due to PAW

!! NOTES
!! This routine computes the electric field gradient, specifically the components
!! $\partial^2 V/\partial x_\alpha \partial x_\beta$ of the potential generated by the valence
!! electrons, at each atomic site in the unit cell. Key references: Kresse and Joubert, ``From
!! ultrasoft pseudopotentials to the projector augmented wave method'', Phys. Rev. B. 59, 1758--1775 (1999),
!! and Profeta, Mauri, and Pickard, ``Accurate first principles prediction of $^{17}$O NMR parameters in
!! SiO$_2$: Assignment of the zeolite ferrierite spectrum'', J. Am. Chem. Soc. 125, 541--548 (2003). See in particular
!! Eq. 11 and 12 of Profeta et al., but note that their sum over occupied states times 2 for occupation number is
!! replaced in the Kresse and Joubert formulation by the sum over $\rho_{ij}$ occupations for each basis element pair.
!!
!! PARENTS
!!      calc_efg
!!
!! CHILDREN
!!      deducer0,simp_gen
!!
!! SOURCE

#if defined HAVE_CONFIG_H
#include "config.h"
#endif

subroutine make_efg_paw(efg,natom,ntypat,pawang,pawrhoij,pawrad,pawtab,typat)

 use defs_basis
 use defs_datatypes

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_11util
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: natom,ntypat
 type(pawang_type),intent(in) :: pawang
!arrays
 integer,intent(in) :: typat(natom)
 real(dp),intent(out) :: efg(3,3,natom)
 type(pawrad_type),intent(in) :: pawrad(ntypat)
 type(pawrhoij_type),intent(in) :: pawrhoij(natom)
 type(pawtab_type),intent(in) :: pawtab(ntypat)

!Local variables-------------------------------
!scalars
 integer :: iatom,ils,ilslm,irhoij,isel,ispden,itypat,klm,klmn,kln,lmax,lmin,lnspden
 integer :: mesh_size,mm
 real(dp) :: c1,c2,c3,intg,ro
 character(len=500) :: message
!arrays
 real(dp) :: rg2(-2:2)
 real(dp),allocatable :: ff(:)

! ************************************************************************

!DEBUG
!write(*,*)' make_efg_paw : enter'
!ENDDEBUG

 efg(:,:,:) = zero

!the following factors arise in expanding the angular dependence of the electric field gradient tensor in
!terms of real spherical harmonics. The real spherical harmonics are as in the routine initylmr.F90; see
!in particular also http://www1.elsevier.com/homepage/saa/eccc3/paper48/eccc3.html
 c1 = sqrt(16.0*pi/5.0)
 c2 = sqrt(4.0*pi/5.0)
 c3 = sqrt(12.0*pi/5.0)

!loop over atoms in cell
 do iatom = 1, natom
  itypat = typat(iatom)
  mesh_size=pawrad(itypat)%mesh_size
  allocate(ff(mesh_size))

! loop over spin components
! nspden = 1: just a single component
! nspden = 2: two components, loop over them
! nspden = 4: total is in component 1, only one of interest
  if ( pawrhoij(iatom)%nspden == 2 ) then 
   lnspden = 2
  else
   lnspden = 1
  end if
  do ispden=1,lnspden

!  Loop over non-zero elements of rhoij
   do irhoij=1,pawrhoij(iatom)%nrhoijsel
    klmn=pawrhoij(iatom)%rhoijselect(irhoij)
    klm =pawtab(itypat)%indklmn(1,klmn)
    kln =pawtab(itypat)%indklmn(2,klmn)
    lmin=pawtab(itypat)%indklmn(3,klmn)
    lmax=pawtab(itypat)%indklmn(4,klmn)

!   Select only l=2, because the EFG tensor is pure rank 2
    if (lmin<=2.and.lmax>=2) then

!    Computation of <phi_i|1/r^3|phi_j>- <tphi_i|1/r^3|tphi_j>
!    the EFG tensor has radial dependence 1/r^3
     ff(2:mesh_size)=(pawtab(itypat)%phiphj(2:mesh_size,kln)&
&     -pawtab(itypat)%tphitphj(2:mesh_size,kln))&
&     /pawrad(itypat)%rad(2:mesh_size)**3
     call deducer0(ff,mesh_size,pawrad(itypat))
     call simp_gen(intg,ff,pawrad(itypat))

!    Real gaunt coefficients selection
     ils=2 !l=2 only
     rg2(:)=zero
     do mm=-ils,ils
      ilslm=ils*ils+ils+mm+1
      isel=pawang%gntselect(ilslm,klm)
      if (isel>0) rg2(mm)=pawang%realgnt(isel) ! these are the non-zero <Ylm|Y2mm|Yl'm'> matrix elements
     end do

!    Accumulation of (PAW) EFG
!    in the following line, the summand over rhoij, we multiply by -1 so that the electron density rhoij is converted
!    to an electron charge density
     if (pawrhoij(iatom)%cplex == 1) then
      ro= -pawtab(itypat)%dltij(klmn)*pawrhoij(iatom)%rhoijp(irhoij,ispden)*intg
     else
      ro= -pawtab(itypat)%dltij(klmn)*pawrhoij(iatom)%rhoijp(2*irhoij-1,ispden)*intg
     end if
     efg(1,1,iatom) = efg(1,1,iatom)+(-c2*rg2(0)+c3*rg2(2))*ro
     efg(2,2,iatom) = efg(2,2,iatom)+(-c2*rg2(0)-c3*rg2(2))*ro
     efg(3,3,iatom) = efg(3,3,iatom)+c1*rg2(0)*ro
     efg(1,2,iatom) = efg(1,2,iatom)+c3*rg2(-2)*ro
     efg(1,3,iatom) = efg(1,3,iatom)+c3*rg2(1) *ro
     efg(2,3,iatom) = efg(2,3,iatom)+c3*rg2(-1)*ro

    end if  ! Select only l=2
   end do   ! Loop on non-zero rhoij

  end do    ! Loop on spin components
  deallocate(ff)

! Symmetrization of EFG
  efg(2,1,iatom) = efg(1,2,iatom)
  efg(3,1,iatom) = efg(1,3,iatom)
  efg(3,2,iatom) = efg(2,3,iatom)

 end do     ! Loop on atoms

!DEBUG
!write(6,*)' make_efg_paw : exit '
!stop
!ENDDEBUG

 end subroutine make_efg_paw
!!***
