!{\src2tex{textfont=tt}}
!!****f* ABINIT/setup_Qmesh
!! NAME
!! setup_Qmesh
!!
!! FUNCTION
!! Initialize and construct a bz_mesh_type datatype 
!! gathering information on the q-mesh used for GW calculations
!!
!! COPYRIGHT
!!  Copyright (C) 2008-2009 ABINIT group (MG)
!!  This file is distributed under the terms of the
!!  GNU General Public License, see ~abinit/COPYING
!!  or http://www.gnu.org/copyleft/gpl.txt .
!!
!! INPUTS
!! nqibz=number of irreducible q-points
!! nsym=number of symmetry operations
!! prtvol=verbosity level
!! timrev=1 if time-reversal cannot be used, 2 otherwise
!! qibz(3,nqibz)=irreducible q-points
!! symrec(3,3,nsym)=symmetry operations in reciprocal space
!!
!! OUTPUT
!! Qmesh<bz_mesh_type>=datatype gathering information on the q point sampling. see defs_datatypes.F90
!!
!! SIDE EFFECTS
!!
!! NOTES
!!
!! PARENTS
!!      setup_qmesh,setup_sigma
!!
!! CHILDREN
!!      findnq,findq,metric,setup_qmesh
!!
!! SOURCE

#if defined HAVE_CONFIG_H
#include "config.h"
#endif

subroutine setup_Qmesh(nqibz,Cryst,prtvol,qibz,Qmesh)

 use defs_basis
 use defs_datatypes
 use m_io_tools, only : flush_unit
 use m_errors,   only : assert
 use m_bz_mesh

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_01manage_mpi
 use interfaces_12geometry
 use interfaces_15gw, except_this_one => setup_Qmesh
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: nqibz,prtvol
 type(BZ_mesh_type),intent(inout) :: Qmesh
 type(Crystal_structure),intent(in) :: Cryst
!arrays
 real(dp),intent(in) :: qibz(3,nqibz)

!Local variables-------------------------------
!scalars
 integer :: iq_bz,iq_ibz,isym,itim,nqbz,nqbzX,nsym,timrev
 real(dp) :: ucvol
 logical :: ltest
 character(len=500) :: msg
!arrays
 integer,allocatable :: qtab(:),qtabi(:),qtabo(:)
 integer,pointer :: symrec(:,:,:)
 real(dp) :: Sq(3),gmet(3,3),gprimd(3,3),rmet(3,3)
 real(dp),allocatable :: qbz(:,:),wtq(:)

! *************************************************************************

#if defined DEBUG_MODE
 write(msg,'(a)')' setup_qmesh : enter'
 call wrtout(std_out,msg,'COLL') 
 call flush_unit(std_out)
#endif

 call NullifyBzMesh(Qmesh)

 ltest=(Cryst%timrev==1.or.Cryst%timrev==2)
 call assert(ltest,"timrev should be 1 or 2",__FILE__,__LINE__)

 nsym   =  Cryst%nsym
 timrev =  Cryst%timrev
 symrec => Cryst%symrec

 call metric(gmet,gprimd,-1,rmet,Cryst%rprimd,ucvol)
 Qmesh%gmet   = gmet
 Qmesh%gprimd = gprimd

 nqbzX=nqibz*nsym*timrev
 allocate(qbz(3,nqbzX),qtab(nqbzX),qtabo(nqbzX),qtabi(nqbzX),wtq(nqibz))
 qbz(:,:)=0 ; qtab(:)=0 ; qtabo(:)=0 ; qtabi(:)=0

 call identq(qibz,nqibz,nqbzX,REAL(symrec,dp),nsym,timrev,wtq,qbz,qtab,qtabi,qtabo,nqbz,verbose=1)

 do iq_bz=1,nqbz
  isym=qtabo(iq_bz) ; iq_ibz=qtab(iq_bz) ; itim=(3-qtabi(iq_bz))/2
  call dosym(REAL(symrec(:,:,isym),dp),itim,qibz(:,iq_ibz),Sq(:))
  if (ANY(ABS(qbz(:,iq_bz)-Sq(:) )>1.0d-4)) then
   write(msg,'(4a,3f6.3,a,3f6.3,2a,9i3,2a)')ch10,&
&   ' setup_qmesh : ERROR - ',ch10,&
&   ' qpoint ',qbz(:,iq_bz),' is the symmetric of ',qibz(:,iq_ibz),ch10,&
&   ' through operation ',symrec(:,:,isym),ch10,&
&   ' however a non zero umklapp G_o vector is required and this is not yet allowed'
   call wrtout(std_out,msg,'COLL') ; write(*,*) Sq,qbz(:,iq_bz) 
   call leave_new('COLL')
  end if
 end do 
 !
 ! ==== Create data structure to store information on q-points ====
 ! * Dimensions
 Qmesh%nbz    = nqbz
 Qmesh%nibz   = nqibz      
 Qmesh%timrev = timrev
 Qmesh%nsym   = nsym

 Qmesh%ntetra_irr=0   ! no tetrahedrons for the moment
 !
 ! * Arrays
 allocate(Qmesh%ibz(3,nqibz)) ; Qmesh%ibz(:,:) = qibz(:,1:nqibz) 
 allocate(Qmesh%wt(nqibz))    ; Qmesh%wt(:)    = wtq(1:nqibz)

 allocate(Qmesh%bz(3,nqbz))   ; Qmesh%bz(:,:)  = qbz(:,1:nqbz)
 allocate(Qmesh%tab(nqbz))    ; Qmesh%tab(:)   = qtab (1:nqbz)
 allocate(Qmesh%tabi(nqbz))   ; Qmesh%tabi(:)  = qtabi(1:nqbz)
 allocate(Qmesh%tabo(nqbz))   ; Qmesh%tabo(:)  = qtabo(1:nqbz)

 !
 ! TODO For the time being these arrays are not used however they should be defined just
 ! to be consistent
 nullify(Qmesh%tabp)       
 nullify(Qmesh%rottb)
 nullify(Qmesh%rottbm1)

 !call print_bz_mesh(Qmesh,prtvol=prtvol)
 deallocate(qbz,qtab,qtabi,qtabo,wtq)

#if defined DEBUG_MODE
 write(msg,'(a)')' setup_qmesh : exit'
 call wrtout(std_out,msg,'COLL') 
 call flush_unit(std_out)
#endif

end subroutine setup_Qmesh
!!***


!!****f* ABINIT/find_Qmesh
!! NAME
!! find_Qmesh
!!
!! FUNCTION
!!  Find the q-mesh defined as all the possible differences between k-points
!!  Find the irreducible q-points using a special treatment for the Gamma point.
!!  Then call setup_kmesh to initialize the Qmesh datatype
!!
!! COPYRIGHT
!!  Copyright (C) 2008-2009 ABINIT group (MG)
!!  This file is distributed under the terms of the
!!  GNU General Public License, see ~abinit/COPYING
!!  or http://www.gnu.org/copyleft/gpl.txt .
!!
!! INPUTS
!!  Cryst<crystal_structure>=datatype gathering info on the unit cell and symmetries
!!    %nsym=number of symmetry operations
!!    %symrec(3,3,nsym)=symmetry operations in reciprocal space
!!  prtvol=verbosity level
!!  timrev=1 if time-reversal cannot be used, 2 otherwise
!!  Kmesh<bz_mesh_type>=datatype gathering information on the k-mesh
!!
!! OUTPUT
!!  Qmesh<bz_mesh_type>=datatype gathering information on the q point sampling. see defs_datatypes.F90
!!
!! SIDE EFFECTS
!!
!! NOTES
!!
!! PARENTS
!!      joint_dos,mrgscr,setup_screening
!!
!! CHILDREN
!!      findnq,findq,metric,setup_qmesh
!!
!! SOURCE

#if defined HAVE_CONFIG_H
#include "config.h"
#endif

subroutine find_Qmesh(Cryst,Kmesh,Qmesh,prtvol)

 use defs_basis
 use defs_datatypes

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_12geometry
 use interfaces_15gw, except_this_one => find_Qmesh
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: prtvol
 type(BZ_mesh_type),intent(in) :: Kmesh
 type(BZ_mesh_type),intent(out) :: Qmesh
 type(Crystal_structure),intent(in) :: Cryst

!Local variables-------------------------------
!scalars
 integer :: nqibz,nsym,timrev
 real(dp) :: ucvol
 logical :: avoid_zero
 character(len=500) :: msg
!arrays
 integer,pointer :: symrec(:,:,:)
 real(dp) :: gmet(3,3),gprimd(3,3),rmet(3,3)
 real(dp),allocatable :: qibz(:,:)

! *************************************************************************

 timrev = Cryst%timrev
 nsym   = Cryst%nsym
 symrec => Cryst%symrec

 call metric(gmet,gprimd,-1,rmet,Cryst%rprimd,ucvol)
 !
 ! === Find number of q-points q=k-kp ===  
 call findnq(Kmesh%nbz,Kmesh%bz,nsym,symrec,nqibz,timrev) 
 !
 ! === Find q-points === 
 allocate(qibz(3,nqibz)) 
 avoid_zero=.TRUE.

 call findq(Kmesh%nbz,Kmesh%bz,nsym,symrec,gprimd,nqibz,qibz,timrev,avoid_zero)
 !
 ! === Create Qmesh datatype starting from the IBZ ===
 ! Here I should call setup_Kmesh, just to keep it simple but I have to 
 ! solve some problems with the small q
 call setup_Qmesh(nqibz,Cryst,prtvol,qibz,Qmesh)
 deallocate(qibz)

end subroutine find_Qmesh
!!***

!!****f* ABINIT/findnq
!! NAME
!! findnq
!!
!! FUNCTION
!! Identify the number of q-points in the IBZ by which the k-points in BZ differ
!! (count the q points in the k-point difference set)
!!
!! COPYRIGHT
!! Copyright (C) 1999-2009 ABINIT group (GMR, VO, LR, RWG, MT, MG)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~abinit/COPYING
!! or http://www.gnu.org/copyleft/gpl.txt .
!! For the initials of contributors, see ~abinit/doc/developers/contributors.txt .
!!
!! INPUTS
!!  kbz(3,nkbz)=coordinates of k points in BZ
!!  timrev=2 if time-reversal symmetry is used, 1 otherwise
!!  nkbz=number of k points in Brillouin zone
!!  nsym=number of symmetry operations
!!  symrec(3,3,nsym)=symmetry operations in reciprocal space
!!
!! OUTPUT
!!  nqibz=number of q points
!!
!! PARENTS
!!      rdm,setup_qmesh
!!
!! CHILDREN
!!      dosym,memerr,wrtout
!!
!! SOURCE

#if defined HAVE_CONFIG_H
#include "config.h"
#endif

subroutine findnq(nkbz,kbz,nsym,symrec,nqibz,timrev)

 use defs_basis
 use m_bz_mesh

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_01manage_mpi
 use interfaces_15gw, except_this_one => findnq
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: timrev,nkbz,nsym
 integer,intent(out) :: nqibz
!arrays
 integer,intent(in) :: symrec(3,3,nsym)
 real(dp),intent(in) :: kbz(3,nkbz)

!Local variables ------------------------------
!scalars
 integer :: ifound,ik,isym,iq,istat,memory_exhausted,nqall,nqallm,itim
 character(len=500) :: msg
!arrays
 integer :: gtemp(3),g0(3)
 real(dp) :: qposs(3),qrot(3)
 real(dp),allocatable :: qall(:,:)
!************************************************************************

 ! === Infinite do-loop to be able to allocate sufficient memory ===
 nqallm=1000 ; memory_exhausted=0
 do 
  allocate(qall(3,nqallm),stat=istat)
  if (istat/=0) call memerr('findnq','qall',3*nqallm,'dp')
  nqall=0
  ! === Loop over all k-points in BZ, forming k-k1 ===
  do ik=1,nkbz
   qposs(:)=kbz(:,ik)-kbz(:,1)
   ! === Check whether this q (or its equivalent) has already been found ===
   ! * Use spatial inversion instead of time reversal whenever possible.
   ifound=0
   do iq=1,nqall
    do itim=1,timrev
     do isym=1,nsym
      !FIXME this is for g95
      call dosym(REAL(symrec(:,:,isym),dp),itim,qall(:,iq),qrot)
      if (isamek(qrot,qposs,g0)) ifound=ifound+1
     end do
    end do
   end do

   if (ifound==0) then
    nqall=nqall+1
    !
    ! === If not yet found, check that the allocation is big enough ===
    if (nqall>nqallm) then
     memory_exhausted=1 ; deallocate(qall)
     nqallm=nqallm*2    ; EXIT ! Exit the do ik=1 loop
    end if
    ! === Add to the list ===
    qall(:,nqall)=qposs(:)
   end if
  end do

  if (memory_exhausted==0) EXIT
 end do !infinite loop

 deallocate(qall)
 nqibz=nqall

 write(msg,'(a,i8)')' number of q-points found ',nqibz
 call wrtout(std_out,msg,'COLL') ; call wrtout(ab_out,msg,'COLL')

end subroutine findnq
!!***

!!****f* ABINIT/findq
!! NAME
!! findq
!!
!! FUNCTION
!! Identify the q-points by which the k-points in BZ differ
!!
!! COPYRIGHT
!! Copyright (C) 1999-2009 ABINIT group (GMR, VO, LR, RWG, MT, MG)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~abinit/COPYING
!! or http://www.gnu.org/copyleft/gpl.txt .
!! For the initials of contributors, see ~abinit/doc/developers/contributors.txt .
!!
!! INPUTS
!!  gprimd(3,3)=dimensional reciprocal space primitive translations
!!  kbz(3,nkbz)=coordinates of k points in BZ
!!  timrev=2 if time-reversal symmetry is used, 1 otherwise
!!  nkbz=number of k points in Brillouin zone
!!  nsym=number of symmetry operations
!!  nqibz=number of q points in the IBZ by which k points differ (computed in findnq)
!!  symrec(3,3,nsym)=symmetry operations in reciprocal space
!!
!! OUTPUT
!!  qibz(3,nqibz)=coordinates of q points by which k points differ
!!
!! PARENTS
!!      rdm,setup_qmesh
!!
!! CHILDREN
!!      bz1,dosym,wrtout
!!
!! SOURCE

#if defined HAVE_CONFIG_H
#include "config.h"
#endif

subroutine findq(nkbz,kbz,nsym,symrec,gprimd,nqibz,qibz,timrev,avoid_zero)

 use defs_basis
 use m_numeric_tools, only : is_zero
 use m_bz_mesh,       only : isamek

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_01manage_mpi
 use interfaces_15gw, except_this_one => findq
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: nkbz,nqibz,nsym,timrev
 logical,intent(in) :: avoid_zero
!arrays
 integer,intent(in) :: symrec(3,3,nsym)
 real(dp),intent(in) :: gprimd(3,3),kbz(3,nkbz)
 real(dp),intent(out) :: qibz(3,nqibz)

!Local variables ------------------------------
!scalars
 integer :: ii,ik,iq,iqp,isym,itim,jj,nq0
 real(dp) :: shift,tolq0
 logical :: found
 character(len=500) :: msg
!arrays
 integer :: g0(3),gtemp(3)
 real(dp) :: gmet(3,3),qposs(3),qrot(3)

!************************************************************************

 write(msg,'(a)')' find q-points q = k - k1 and translate in first BZ'
 call wrtout(std_out,msg,'COLL') ; call wrtout(ab_out,msg,'COLL')

 ! Compute reciprocal space metrics
 do ii=1,3
  gmet(ii,:)=gprimd(1,ii)*gprimd(1,:)+&
&            gprimd(2,ii)*gprimd(2,:)+&
&            gprimd(3,ii)*gprimd(3,:)
 end do

 tolq0=0.001_dp !old behaviour
 !
 ! === Loop over all k-points in BZ, forming k-k1 ===
 ! iq is the no. of q-points found, zero at the beginning
 iq=0
 do ik=1,nkbz
  qposs(:)=kbz(:,ik)-kbz(:,1)
  ! === Check whether this q (or its equivalent) has already been found ===
  found=.FALSE.
  do iqp=1,iq
   do itim=1,timrev
    do isym=1,nsym
     !FIXME this is for g95
     call dosym(REAL(symrec(:,:,isym),dp),itim,qibz(:,iqp),qrot)
     if (isamek(qrot,qposs,g0)) found=.TRUE.
    end do
   end do
  end do
  if (.not.found) then
   iq=iq+1
   if (iq>nqibz) then 
    write(msg,'(2a)')ch10,&
&    ' findq : BUG in findnq: iq>nqibz '
    call wrtout(std_out,msg,'COLL') ; call leave_new('COLL')
   end if 
   qibz(:,iq)=qposs(:)
  end if
 end do

 if (iq/=nqibz) then 
  write(msg,'(2a)')ch10,&
&  ' findq : BUG in findnq: iq/=nqibz '
  call wrtout(std_out,msg,'COLL') ; call leave_new('COLL')
 end if 
 !
 ! Translate q-points (calculated for info) to 1st BZ
 !
!MG it seems that bz1 sometimes does not work, 
!in SiO2 for example I got 0.333333   -0.666667    0.000000
!it is better to use canon9 also because if an irred qpoint 
!lies outside the 1BZ then most probably we have to consider
!an umklapp G0 vector to reconstruct the full BZ. The 
!correct treatment of this case is not yet implemented yet, see csigme.F90
!Anyway I should check this new method because likely mrscr will complain
!The best idea consists in  writing a new subroutine canon10 
!wich reduce the q point in the interval [-1/2,1/2[ which is
!supposed to be the scope of bz1. Just to obtain the same results as 
!the automatic tests
!FIXME for the moment use old version, easy for debugging

 do iq=1,nqibz
  call bz1(qibz(:,iq),gtemp,gmet)
 end do

!DEBUG
!do iq=1,nqibz
!do ii=1,3
!call canon9(qibz(ii,iq),qibz(ii,iq),shift)
!end do
!end do 
!ENDDEBUG
 write(msg,'(a)')' q-points [reduced coordinates]'
 call wrtout(std_out,msg,'COLL') ; call wrtout(ab_out,msg,'COLL')

 nq0=0
 do jj=1,nqibz
  if (is_zero(qibz(:,jj),tolq0).and.avoid_zero) then
   qibz(1,jj)=0.000010
   qibz(2,jj)=0.000020
   qibz(3,jj)=0.000030
   nq0=nq0+1
  end if
  write(msg,'(3f12.6)') (qibz(ii,jj),ii=1,3)
  call wrtout(std_out,msg,'COLL') ; call wrtout(ab_out,msg,'COLL')
 end do
 
 !write(msg,'(a)')ch10
 !call wrtout(std_out,msg,'COLL') ; call wrtout(ab_out,msg,'COLL')
 write(std_out,*) ; write(ab_out,*)

 if (nq0/=1) then 
  write(msg,'(4a,i2,5a)')ch10,&
&  ' findq : ERROR ',ch10,&
&  ' Found ',nq0,' "small" qpoints ',ch10,&
&  ' Check the q-mesh and, if it is correct, decrease the tolerance value ',ch10,&
&  ' below which two points are considered equivalent. '
  call wrtout(std_out,msg,'COLL') ; call leave_new('COLL')
 end if

end subroutine findq
!!***

!!****f* ABINIT/identq
!! NAME
!! identq
!!
!! FUNCTION
!! Identify q-points in whole BZ
!!
!! COPYRIGHT
!! Copyright (C) 1999-2008 ABINIT group (GMR, VO, LR, RWG, MG)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~abinit/COPYING
!! or http://www.gnu.org/copyleft/gpl.txt .
!! For the initials of contributors, see ~abinit/doc/developers/contributors.txt .
!!
!! INPUTS
!!  qibz(3,nqibz)=coordinates of the q-points in the IBZ
!!  timrev=if 2, time-reversal symmetry is used, 1 otherwise
!!  nqibz= number of q points in IBZ
!!  nqbzX= maximum number of q points in BZ
!!  nsym= number of symmetry operations
!!  symrec(3,3,nsym)= symmetry operations in reciprocal space
!!
!! OUTPUT
!!  qbz(3,nqbzX)= q-points in whole BZ
!!  qtab(nqbzX)= table giving for each q-point in the BZ (qBZ), the corresponding
!!   irreducible point (qIBZ), where qBZ= (IS) qIBZ and I is the inversion or the identity
!!  qtabi(nqbzX)= for each q-point in the BZ defines whether inversion has to be 
!!   considered in the relation qBZ=(IS) qIBZ (1 => only S; -1 => -S)  
!!  qtabo(nqbzX)= the symmetry operation S in the array op that takes qIBZ to each qBZ
!!  nqbz= no. of q-points in whole BZ
!!  wtq(nqibz)=weight of each irred q-point (normalized to one)
!!
!! NOTES
!!  For q close to zero only one q-point is counted (i.e. not the
!!  rotations and reflections of the little q, nor other little
!!  q-points in the qIBZ set)
!!
!! TODO 
!! Merge this routine with identk
!!
!! PARENTS
!!      rdm,setup_qmesh
!!
!! CHILDREN
!!      dosym,flush_unit,wrtout
!!
!! SOURCE

#if defined HAVE_CONFIG_H
#include "config.h"
#endif

subroutine identq(qibz,nqibz,nqbzX,symrec,nsym,timrev,wtq,qbz,qtab,qtabi,qtabo,nqbz,verbose)

 use defs_basis
 use m_bz_mesh
 use m_numeric_tools, only : is_zero
 use m_io_tools,      only : flush_unit

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_01manage_mpi
 use interfaces_15gw, except_this_one => identq
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: nqbzX,nqibz,nsym,timrev,verbose
 integer,intent(out) :: nqbz
!arrays
 integer,intent(out) :: qtab(nqbzX),qtabi(nqbzX),qtabo(nqbzX)
 real(dp),intent(in) :: qibz(3,nqibz),symrec(3,3,nsym)
 real(dp),intent(out) :: qbz(3,nqbzX),wtq(nqibz)

!Local variables ------------------------------
!scalars
 integer :: div4,id,ii,iold,iq_bz,iq_ibz,isym,itim,jj,jqp,nqeq0,res
 real(dp) :: tolq0
 character(len=100) :: frmt
 character(len=500) :: msg
!arrays
 integer :: G0(3)
 real(dp) :: qnew(3)

!************************************************************************

#if defined DEBUG_MODE
 write(msg,'(a)')' identq: identifying q-points'
 call wrtout(std_out,msg,'COLL') 
 call flush_unit(std_out)
#endif

 tolq0=0.001_dp !old behaviour

 ! Zero number of q-points found and zero small q vectors found
 nqbz=0 ; nqeq0=0
 !
 ! === Loop over q-points in IBZ ===
 do iq_ibz=1,nqibz
  wtq(iq_ibz)=zero
  ! 
  ! * If q is close to zero, treat it differently as described above
  if (is_zero(qibz(:,iq_ibz),tolq0)) then

   if (nqeq0==0) then
    nqeq0=1
    nqbz=nqbz+1
    if (nqbz>nqbzX) then
     write(msg,'(5a)')ch10,&
&     ' identq : BUG ',ch10,&
&     ' nqbzX too small ',ch10
     call wrtout(std_out,msg,'COLL') ; call leave_new('COLL')
    end if
    wtq(iq_ibz)=wtq(iq_ibz)+one
    qbz(:,nqbz)=qibz(:,iq_ibz)
    qtab(nqbz)=iq_ibz
    qtabo(nqbz)=1
    qtabi(nqbz)=1
   else 
    write(msg,'(6a)')ch10,&
&    ' identq : ERROR -,',ch10,&
&    ' it seems that there are at least two "small" q-points in the grid ',ch10,&
&    ' check q-grid and coding in identq.F90 '
    call wrtout(std_out,msg,'COLL') ; call leave_new('COLL')
   end if

  else
   ! === Loop over symmetry operations S and inversion/identity I ===
   ! * Use spatial inversion instead of time reversal whenever possible.
   do itim=1,timrev
    do isym=1,nsym
     ! * Form SI q
     call dosym(symrec(:,:,isym),itim,qibz(:,iq_ibz),qnew)
      !    
      ! Check whether it has already been found (to within a RL vector)
      ! Here there is a problem since if an umklapp G_o vector is required (Sq1 = q2 +¨G_) 
      ! then during the reconstruction of \tilde\espilon^{-1} we have to calculate G-G_o, see csigme.F90
     iold=0
     do iq_bz=1,nqbz
      if (isamek(qnew,qbz(:,iq_bz),G0)) iold=iold+1
     end do

     if (iold==0) then ! we have a new q-point
      nqbz=nqbz+1
      if (nqbz>nqbzx) then
       write(msg,'(4a)')ch10,&
&       ' identq : BUG -',ch10,&
&       ' nqbzX too small '
       call wrtout(std_out,msg,'COLL') ; call leave_new('COLL')
      end if
      wtq(iq_ibz)=wtq(iq_ibz)+one
      qbz(:,nqbz)=qnew(:)
      qtab(nqbz)=iq_ibz
      qtabo(nqbz)=isym
      qtabi(nqbz)=3-2*itim
     end if
    end do
   end do
  end if

 end do

 ! === Normalize weights to 1 ===
 wtq(:) = wtq(:)/SUM(wtq)
 !
 ! === Print out results ===
 if (verbose>0) then 
  write(msg,'(2a,i4,3a)')ch10,&
&  ' q-points in irreducible wedge (IBZ) ',nqibz,ch10,&
&  ' q-points [reciprocal lattice units]:',ch10
  call wrtout(std_out,msg,'COLL')
  do jj=1,nqibz
   write(msg,'(i5,3f12.6)')jj,(qibz(ii,jj),ii=1,3)
   call wrtout(std_out,msg,'COLL')
  end do

  write(msg,'(3a,i2,3a,i4,2a)')ch10,ch10,' together with the ',nsym,&
&  ' symmetry operations and inversion',ch10,&
&  ' have yielded',nqbz,' q-points in Brillouin Zone (BZ):',ch10 
  call wrtout(std_out,msg,'COLL')

  write(frmt,*)'(i5,2x,4(3f7.3,2x))'
  div4=nqbz/4 ; res=mod(nqbz,4)

  do id=0,div4-1 
   jj=4*id+1 
   write(msg,frmt)jj,((qbz(ii,jqp),ii=1,3),jqp=jj,jj+3)
   call wrtout(std_out,msg,'COLL')
  end do
  if (res/=0) then
   write(frmt,*)'(i5,2x,',res,'(3f7.3,2x),a)'
   write(msg,frmt)4*div4+1,((qbz(ii,jqp),ii=1,3),jqp=4*div4+1,nqbz),ch10
   call wrtout(std_out,msg,'COLL')
  end if
 end if !verbose

#if defined DEBUG_MODE
 write(msg,'(a)')' identq : ended'
 call wrtout(std_out,msg,'COLL') 
 call flush_unit(std_out)
#endif

end subroutine identq
!!***
