!{\src2tex{textfont=tt}}
!!****m* ABINIT/m_crystal
!! NAME
!! m_crystal
!!
!! FUNCTION
!! Module containing the definition of the Crystal_structure data type and methods used to handle it. 
!!
!! COPYRIGHT
!!  Copyright (C) 2008-2009 ABINIT group (MG)
!!  This file is distributed under the terms of the
!!  GNU General Public License, see ~abinit/COPYING
!!  or http://www.gnu.org/copyleft/gpl.txt .
!!
!! INPUTS
!!
!! OUTPUT
!!
!! NOTES
!!
!! PARENTS
!!
!! CHILDREN
!!  
!!
!! SOURCE

#if defined HAVE_CONFIG_H
#include "config.h"
#endif

#include "abi_common.h"

MODULE m_crystal

 use defs_basis
 use m_errors 

 private 

 public ::             &
  InitCrystal,         &  ! Main Creation method.
  InitCrystalFromHdr,  &  ! Initialize the object using the info reported in the header.
  DestroyCrystal,      &  ! Free the structure.
  PrintCrystal,        &  ! Print dimensions and basic info stored in the object
  print_symmetries,    &  ! Helper function to print symmetries in a nice format.
  idx_spatial_inversion   ! Return the index of the spatial inversion, 0 if not present

  !integer,private,parameter :: inversion=RESHAPE(/-1,0,0,0,-1,0,0,0,-1/),(/3.3/)

CONTAINS  !===========================================================
!!***

!!****f* m_crystal/InitCrystal
!! NAME
!!  InitCrystal 
!!
!! FUNCTION
!!  Initialize a Crystal_structure data type.
!!  Ideally the routine should work in two different modes:
!!  Either the symmetries are directly supplied or the space group
!!  is determined starting from the definition of the unit cell.
!!  Only the first method is implemented, the second one should be
!!  a wrapper for the symmetry finder library. To implement the 
!!  second case I have to add additional entries in the object
!!  and I have also to pass an object describing the (optional) geometry builder.
!!
!! INPUTS
!!  natom=number of atom
!!  ntypat=number of type of atoms
!!  nsym=number of symmetry operations
!!  rprimd(3,3)=dimensional lattive vector (real space)
!!  typat(natom)=type of each atom
!!  xred(3,natom)=reduced coordinates of each atom
!!  symrel(3,3) [optional]=symmetry operations in real space
!!  space_group=Space group (0 if not available)
!!  tnons(3,nsym) [optional]=fractional Translations
!!  symafm(nsym) [optional]=  ferromagnetic symmetries
!!  remove_inv [optional]= if .TRUE. the inversion is removed from the set of symmetries
!!  timrev ==2 => take advantage of time-reversal symmetry
!!         ==1 ==> do not use time-reversal symmetry 
!!
!! OUTPUT
!!  Cryst<Crystal_structure>= the object completely initialized.
!!
!! TODO
!!  Add additional entries in the class:
!!  1) Info on space and point group (generators?).
!!  2) alchemy
!!  3) masses and nuclear (pseudo&AE) charge
!!  4) forces stresses, velocities.
!!  5) constraints for the relaxation
!!  6) Likely I will need also info on the electric field and berryopt
!!
!! PARENTS
!!
!! CHILDREN
!!      wrtout
!!
!! SOURCE

subroutine InitCrystal(Cryst,space_group,natom,npsp,ntypat,nsym,rprimd,typat,xred,&
& ziontypat,znucl,timrev,use_antiferro,remove_inv,title,&
& symrel,tnons,symafm) ! Optional
    
 use defs_basis
 use defs_datatypes
 use m_numeric_tools, only : set2unit

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_01manage_mpi
 use interfaces_11util
 use interfaces_12geometry
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: natom,ntypat,nsym,timrev,space_group,npsp
 type(Crystal_structure),intent(inout) :: Cryst
 logical,intent(in) :: remove_inv,use_antiferro
!arrays
 integer,intent(in) :: typat(natom)
 integer,optional,intent(in) :: symrel(3,3,nsym),symafm(nsym)
 real(dp),intent(in) :: xred(3,natom),rprimd(3,3),ziontypat(ntypat),znucl(npsp)
 real(dp),optional,intent(in) :: tnons(3,nsym)
 character(len=*),intent(in) :: title(ntypat) 

!Local variables-------------------------------
!scalars
 integer :: iat,indx,itypat,pinv,isym,nsym_noI
 real(dp) :: ucvol
 character(len=500) :: msg      
!arrays
 integer :: symrec(3,3),inversion(3,3)
 real(dp) :: gprimd(3,3),gmet(3,3),rmet(3,3)
 real(dp) :: spinrot(4)
 integer,pointer :: symrel_noI(:,:,:)
 integer,allocatable :: indsym(:,:,:)
 real(dp),pointer :: tnons_noI(:,:)
! *************************************************************************

  call NullifyCrystal_(Cryst)

  Cryst%natom  = natom 
  Cryst%ntypat = ntypat
  Cryst%npsp   = npsp

  Cryst%space_group = space_group

  call metric(gmet,gprimd,-1,rmet,rprimd,ucvol)
  
  Cryst%ucvol  = ucvol
  Cryst%rprimd = rprimd 
  Cryst%rmet   = rmet
  Cryst%gmet   = gmet
  Cryst%gprimd = gprimd

  Cryst%angdeg(1)=ACOS(Cryst%rmet(2,3)/SQRT(Cryst%rmet(2,2)*Cryst%rmet(3,3)))/two_pi*360.0d0
  Cryst%angdeg(2)=ACOS(Cryst%rmet(1,3)/SQRT(Cryst%rmet(1,1)*Cryst%rmet(3,3)))/two_pi*360.0d0
  Cryst%angdeg(3)=ACOS(Cryst%rmet(1,2)/SQRT(Cryst%rmet(1,1)*Cryst%rmet(2,2)))/two_pi*360.0d0

  allocate(Cryst%typat(natom),Cryst%xred(3,natom),Cryst%xcart(3,natom)) 
  allocate(Cryst%ziontypat(ntypat))
  allocate(Cryst%znucl(npsp))

  Cryst%typat     = typat 
  Cryst%xred      = xred 
  Cryst%ziontypat = ziontypat
  Cryst%znucl     = znucl

  call xredxcart(natom,1,rprimd,Cryst%xcart,Cryst%xred)

  allocate(Cryst%title(ntypat))
  Cryst%title=title
  !
  ! === Generate index table of atoms, in order for them to be used type after type ===
  allocate(Cryst%atindx(natom),Cryst%atindx1(natom))
  allocate(Cryst%nattyp(ntypat))

  indx=1
  do itypat=1,ntypat
   Cryst%nattyp(itypat)=0
   do iat=1,natom
    if (Cryst%typat(iat)==itypat) then
     Cryst%atindx (iat )=indx 
     Cryst%atindx1(indx)=iat
     indx=indx+1
     Cryst%nattyp(itypat)=Cryst%nattyp(itypat)+1
    end if
   end do
  end do

  Cryst%timrev = timrev
  call set2unit(inversion) ; inversion=-inversion

  if (PRESENT(symrel).and.PRESENT(tnons).and.PRESENT(symafm)) then 
   if (.not.remove_inv) then
    ! * Just a copy 
    Cryst%nsym= nsym
    allocate(Cryst%symrel(3,3,nsym),Cryst%symrec(3,3,nsym))
    allocate(Cryst%tnons(3,nsym),Cryst%symafm(nsym))
    Cryst%symrel=symrel 
    Cryst%tnons=tnons
    Cryst%symafm=symafm
    Cryst%use_antiferro = use_antiferro
    Cryst%has_inversion=.FALSE.
    do isym=1,nsym
     call mati3inv(symrel(:,:,isym),symrec)
     Cryst%symrec(:,:,isym)=symrec
     if (ALL(symrel(:,:,isym)==inversion)) Cryst%has_inversion=.TRUE. 
    end do
   else 
    ! * Remove inversion, just to be compatible with old GW implementation
    ! TODO should be removed!
    call remove_inversion(nsym,symrel,tnons,nsym_noI,symrel_noI,tnons_noI,pinv)
    Cryst%nsym=nsym_noI
    allocate(Cryst%symrel(3,3,nsym_noI),Cryst%symrec(3,3,nsym_noI))
    allocate(Cryst%tnons(3,nsym_noI),Cryst%symafm(nsym_noI))
    Cryst%symrel=symrel_noI
    Cryst%tnons=tnons_noI
    Cryst%has_inversion=.FALSE.
    if (ANY(symafm==-1)) then 
     write(msg,'(a)')' First of all solve the problem with inversion before adding ferromagnetic symmetries '
     call wrtout(std_out,msg,'COLL') ; call leave_new('COLL')
    end if
    Cryst%symafm=1
    Cryst%use_antiferro=use_antiferro 
    do isym=1,nsym_noI
     call mati3inv(symrel_noI(:,:,isym),symrec)
     Cryst%symrec(:,:,isym)=symrec
    end do
    deallocate(symrel_noI,tnons_noI)
   end if

  else
   ! * Find symmetries symrec,symrel,tnons,symafm
   ! TODO This should be a wrapper around the abinit library whose usage is not so straightforward
   MSG_BUG('not yet implemented')
  end if

  Cryst%isymmorphic = ALL(ABS(Cryst%tnons)<tol6)

  ! === Obtain a list of rotated atoms ===
  ! $ R^{-1} (xred(:,iat)-\tau) = xred(:,iat_sym) + R_0 $ 
  ! * indsym(4,  isym,iat) gives iat_sym in the original unit cell.
  ! * indsym(1:3,isym,iat) gives the lattice vector $R_0$.
  ! 
  allocate(indsym(4,Cryst%nsym,natom))
  call symatm(indsym,natom,Cryst%nsym,Cryst%symrec,Cryst%tnons,Cryst%typat,Cryst%xred)

  allocate(Cryst%indsym(4,Cryst%nsym,natom))
  Cryst%indsym=indsym  
  deallocate(indsym)

  ! === Rotation in spinor space ===
  allocate(Cryst%spinrot(4,Cryst%nsym))
  do isym=1,Cryst%nsym
   call getspinrot(Cryst%rprimd,spinrot,Cryst%symrel(:,:,isym))
   Cryst%spinrot(:,isym)=spinrot(:)
  end do

end subroutine InitCrystal
!!***

!!****f* m_crystal/InitCrystalFromHdr 
!! NAME
!!  InitCrystalFromHdr
!!
!! FUNCTION
!!  initialize a Crystal_structure data type starting from the abinit header.
!!
!! INPUTS
!!  Hdr<Hdr_type>=the abinit header
!!  timrev ==2 => take advantage of time-reversal symmetry
!!         ==1 ==> do not use time-reversal symmetry 
!!  remove_inv [optional]= if .TRUE. the inversion symmetry is removed from the set of operations
!!  even though it is present in the header
!!
!! OUTPUT
!!  Cryst <Crystal_structure>= the data type filled with data reported in the abinit header 
!!
!! TODO
!!  Add information on the use of time-reversal in the Abinit header.
!!
!! PARENTS
!!      gw_etsf_io,mlwfovlp_qp,mrgscr,rdm,setup_screening,setup_sigma
!!
!! CHILDREN
!!      wrtout
!!
!! SOURCE

subroutine InitCrystalFromHdr(Hdr,Cryst,timrev,remove_inv)

 use defs_basis 
 use defs_datatypes

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
!End of the abilint section

 implicit none

!Arguments ------------------------------------
 type(Hdr_type),intent(in) :: Hdr
 type(Crystal_structure),intent(out) :: Cryst 
 integer,intent(in) :: timrev
 logical,optional,intent(in) :: remove_inv

!Local variables-------------------------------
 integer :: space_group
 logical :: rinv,ltest,use_antiferro
! *********************************************************************

 rinv=.FALSE. ; if (PRESENT(remove_inv)) rinv=remove_inv
 use_antiferro=(Hdr%nspden==2.and.Hdr%nsppol==1)

 ! === consistency check ===
 ltest = (timrev==1.or.timrev==2)
 call assert(ltest,"Wrong value for timrev (1|2)",__FILE__,__LINE__)
 if (use_antiferro) then
  ltest = (ANY(Hdr%symafm==-1))
  call assert(ltest,"Wrong combination of nspden, nsppol, symafm.",__FILE__,__LINE__)
 end if

 space_group=0 !FIXME not known

 call InitCrystal(Cryst,space_group,Hdr%natom,Hdr%npsp,Hdr%ntypat,Hdr%nsym,Hdr%rprimd,Hdr%typat,Hdr%xred,&
& Hdr%zionpsp,Hdr%znuclpsp,timrev,use_antiferro,rinv,Hdr%title,&
& Hdr%symrel,Hdr%tnons,Hdr%symafm) ! Optional

end subroutine InitCrystalFromHdr
!!***

!!****f* m_crystal/NullifyCrystal_
!! NAME
!! NullifyCrystal_
!!
!! FUNCTION
!!  Nullify the pointers in the Crystal_structure data type. [PRIVATE]
!!
!! INPUTS
!!
!! OUTPUT
!!
!! PARENTS
!!
!! CHILDREN
!!      wrtout
!!
!! SOURCE

subroutine NullifyCrystal_(Cryst)

 use defs_basis
 use defs_datatypes

 implicit none

!Arguments ------------------------------------
 type(Crystal_structure),intent(inout) :: Cryst
! *********************************************************************

! integer
 nullify(Cryst%indsym )
 nullify(Cryst%symafm )
 nullify(Cryst%symrec )
 nullify(Cryst%symrel )
 nullify(Cryst%atindx )   
 nullify(Cryst%atindx1)   
 nullify(Cryst%typat  )   
 nullify(Cryst%nattyp )   

! real
 nullify(Cryst%tnons    )
 nullify(Cryst%xcart    )
 nullify(Cryst%xred     )
 nullify(Cryst%ziontypat)
 nullify(Cryst%znucl    )
 nullify(Cryst%title    )
 nullify(Cryst%spinrot  )

end subroutine NullifyCrystal_
!!***

!!****f* m_crystal/DestroyCrystal
!! NAME
!!  DestroyCrystal 
!!
!! FUNCTION
!!  Destroy the dinamic arrays in a Crystal_structure data type
!!
!! INPUTS
!!
!! OUTPUT
!!
!! PARENTS
!!      gw_tools,mlwfovlp_qp,mrgscr,rdm,screening,sigma
!!
!! CHILDREN
!!      wrtout
!!
!! SOURCE

subroutine DestroyCrystal(Cryst)

 use defs_basis
 use defs_datatypes
 use m_io_tools, only : flush_unit

 implicit none

!Arguments ------------------------------------
 type(Crystal_structure),intent(inout) :: Cryst

!Local variables-------------------------------
 character(len=500) :: msg
! *********************************************************************

 DBG_ENTER("COLL")

!integer
 if (associated(Cryst%indsym )) deallocate(Cryst%indsym )   
 if (associated(Cryst%symafm )) deallocate(Cryst%symafm )   
 if (associated(Cryst%symrec )) deallocate(Cryst%symrec )   
 if (associated(Cryst%symrel )) deallocate(Cryst%symrel )   
 if (associated(Cryst%atindx )) deallocate(Cryst%atindx )   
 if (associated(Cryst%atindx1)) deallocate(Cryst%atindx1)   
 if (associated(Cryst%typat  )) deallocate(Cryst%typat  )   
 if (associated(Cryst%nattyp )) deallocate(Cryst%nattyp )   

!real
 if (associated(Cryst%tnons    )) deallocate(Cryst%tnons    )   
 if (associated(Cryst%xcart    )) deallocate(Cryst%xcart    )   
 if (associated(Cryst%xred     )) deallocate(Cryst%xred     )   
 if (associated(Cryst%ziontypat)) deallocate(Cryst%ziontypat)
 if (associated(Cryst%znucl    )) deallocate(Cryst%znucl    )
 if (associated(Cryst%title    )) deallocate(Cryst%title    )
 if (associated(Cryst%spinrot  )) deallocate(Cryst%spinrot  )   

 DBG_EXIT("COLL")

end subroutine DestroyCrystal
!!***

!!****f* m_crystal/PrintCrystal
!! NAME
!!  PrintCrystal 
!!
!! FUNCTION
!!  Print the content of Crystal_structure data type
!!
!! INPUTS
!!
!! OUTPUT
!!
!! PARENTS
!!      rdm,setup_screening,setup_sigma
!!
!! CHILDREN
!!      wrtout
!!
!! SOURCE

subroutine PrintCrystal(Cryst,unit,mode_paral,prtvol) 

 use defs_basis
 use defs_datatypes

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_01manage_mpi
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 type(Crystal_structure),intent(in) :: Cryst
 integer,optional,intent(in) :: unit,prtvol
 character(len=4),optional,intent(in) :: mode_paral 

!Local variables-------------------------------
 integer :: unt,verb,nu,isym
 character(len=4) :: mode
 character(len=500) :: msg      
! ********************************************************************* 

 unt=std_out ; if (PRESENT(unit))       unt=unit
 verb=0      ; if (PRESENT(prtvol))     verb=prtvol 
 mode='COLL' ; if (PRESENT(mode_paral)) mode=mode_paral

 write(msg,'(a)')' Real(R)+Recip(G) space primitive vectors, cartesian coordinates (Bohr,Bohr^-1):'
 call wrtout(unt,msg,mode)
 do nu=1,3
  write(msg,'(1x,a,i1,a,3f11.7,2x,a,i1,a,3f11.7)')&
&  'R(',nu,')=',Cryst%rprimd(:,nu)+tol10,&
&  'G(',nu,')=',Cryst%gprimd(:,nu)+tol10 !tol10 is used to be consistent with metric.F90
  call wrtout(unt,msg,mode)
 end do

 write(msg,'(a,1p,e15.7,a)')&
& ' Unit cell volume ucvol=',Cryst%ucvol+tol10,' bohr^3'
 call wrtout(unt,msg,mode)

 write(msg,'(a,3es16.8,a)')&
& ' Angles (23,13,12)=',Cryst%angdeg(1:3),' degrees'
 call wrtout(unt,msg,mode)

 if (Cryst%timrev==1) then 
  write(msg,'(a)')' Time-reversal symmetry is not present '
 else if (Cryst%timrev==2) then 
  write(msg,'(a)')' Time-reversal symmetry is present '
 else 
  MSG_BUG('Wrong value for timrev') 
 end if
 call wrtout(unt,msg,mode)

 !if (Cryst%use_antiferro) then 
 ! write(msg,'(a)')' System has magnetic symmetries '
 ! call wrtout(unt,msg,mode)
 !end if

 call print_symmetries(Cryst%nsym,Cryst%symrel,Cryst%tnons,Cryst%symafm,unit=unt,mode_paral=mode)

end subroutine PrintCrystal
!!***

!!****f* m_crystal/print_symmetries
!! NAME
!! print_symmetries
!!
!! FUNCTION
!!
!! INPUTS
!!
!! OUTPUT
!!
!! PARENTS
!!      hdr_vs_dtset
!!
!! CHILDREN
!!      wrtout
!!
!! SOURCE
subroutine print_symmetries(nsym,symrel,tnons,symafm,unit,mode_paral)

 use defs_basis
 use defs_datatypes

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_01manage_mpi
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: nsym
 integer,optional,intent(in) :: unit
 character(len=4),optional,intent(in) :: mode_paral
!arrays
 integer,intent(in) :: symrel(3,3,nsym),symafm(nsym)
 real(dp),intent(in) :: tnons(3,nsym)

!Local variables-------------------------------
 integer :: unt,isym
 character(len=500) :: msg      
 character(len=4) :: mode
! *********************************************************************

 unt=std_out ; if (PRESENT(unit)) unt=unit
 mode='COLL' ; if (PRESENT(mode_paral)) mode=mode_paral

 write(msg,'(2a)')ch10,' Rotations                           Translations     Symafm '
 do isym=1,nsym
  write(msg,'(1x,3(3i3,1x),4x,3(f11.7,1x),6x,i2)')symrel(:,:,isym),tnons(:,isym),symafm(isym)
  call wrtout(unt,msg,mode)
 end do 

end subroutine print_symmetries 
!!***

!!****f* m_crystal/idx_spatial_inversion
!! NAME
!!  idx_spatial_inversion
!!
!! FUNCTION
!!  Return the index of the spatial inversion, 0 if not present
!!
!! INPUTS
!!
!! OUTPUT
!!
!! PARENTS
!!
!! CHILDREN
!!
!! SOURCE

function idx_spatial_inversion(Cryst) result(inv_idx)

 use defs_basis
 use defs_datatypes

 implicit none

!Arguments ------------------------------------
!scalars
 integer :: inv_idx
 type(Crystal_structure),intent(in) :: Cryst

!Local variables-------------------------------
!scalars
 integer :: isym
!arrays
 integer :: inversion(3,3)

! *************************************************************************

 inversion=RESHAPE((/-1,0,0,0,-1,0,0,0,-1/),(/3,3/))

 inv_idx=0
 do isym=1,Cryst%nsym
  if ( ALL(Cryst%symrel(:,:,isym)==inversion) ) then 
   inv_idx=isym
   RETURN
  end if
 end do

end function idx_spatial_inversion

END MODULE m_crystal
!!***
