!{\src2tex{textfont=tt}}
!!****m* ABINIT/m_coulombian
!! NAME
!!  m_coulombian
!!
!! FUNCTION
!!  This module contains the definition of the Coulombian_type as well
!!  as procedures to calculate the Coulombian interaction in reciprocal 
!!  space taking into account a possible cutoff in real space.
!!  Procedures to deal with the singularity for q-->0 are also provided.
!!
!! COPYRIGHT
!! Copyright (C) 1999-2008 ABINIT group (MG, FB, GMR, VO, LR, RWG)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~abinit/COPYING
!! or http://www.gnu.org/copyleft/gpl.txt .
!! For the initials of contributors, see ~abinit/doc/developers/contributors.txt .
!!
!!
!! INPUTS
!!
!! OUTPUT
!!
!! PARENTS
!!
!! CHILDREN
!!
!! SOURCE

#if defined HAVE_CONFIG_H
#include "config.h"
#endif

#include "abi_common.h"

MODULE m_coulombian

 use defs_basis
 use m_errors 
 use m_bz_mesh, only : get_BZ_item

 implicit none

 private 

 public :: & 
  setup_coulombian,    & ! Creation method. 
  plot_Vc,             & ! Plot vc in real and reciprocal space.
  print_coulombian,    & ! Report info on the object.
  cutoff_table,        & ! Create table defining wheter a FFT point lyes inside the cutoff region.
  cutoff_density,      & ! Redefine the density according to the cutoff mode.
  destroy_coulombian,  & ! Destruction method.
  cutoff_sphere,       & ! FT of the Coulombian truncated within a sphere.
  cutoff_cylinder,     & ! FT of the Coulombian truncated within a cylinder
  cutoff_surface,      & ! FT of the Coulombian truncated within a slab.
  cvc                    ! FT of the long ranged Coulombian interaction.

CONTAINS  !===========================================================
!!***

!!****f* m_coulombian/setup_coulombian
!! NAME
!! setup_coulombian
!!
!! FUNCTION
!! Perform general check and initialize the data type containing information on the cutoff technique
!!
!! INPUTS
!!  Qmesh<Bz_mesh_type>= Info on q-point sampling
!!  Kmesh<Bz_mesh_type>= Info on k-point sampling
!!  Gsph<Gvectors_type>= info of G sphere
!!   %gmet(3,3)= metric in reciprocal space
!!   %gprimd(3,3)=dimensional primitive translations for reciprocal space ($\textrm{bohr}^{-1}$)
!!   %gvec=G vectors
!!  ng=number of G-vectors to be used to describe the coulombian interaction
!!  ngfft(18)=information on the (fine) FFT grid used for the density.
!!
!! OUTPUT
!!  Vcp <type Coulombian_type> datatype gathering information on the coulombian cutoff technique
!!
!! SIDE EFFECTS
!!
!! NOTES
!! 
!!
!! PARENTS
!!      mrgscr,setup_screening,setup_sigma
!!
!! CHILDREN
!!      wrtout
!!
!! SOURCE

subroutine setup_coulombian(Dtset,Gsph,Qmesh,Kmesh,ng,rprimd,ngfft,MPI_enreg,Vcp)

 use defs_basis
 use defs_datatypes
 use m_gwdefs,   only : GW_TOLQ0
 use m_errors,   only : assert
 use m_io_tools, only : flush_unit
 use m_geometry, only : normv
#if defined FC_PGI6
 ! Buggy PGI6 doesnt like OPERATOR
 use m_numeric_tools 
#else
 use m_numeric_tools, only : arth, geop, imin_loc, llsfit_svd, l2norm, OPERATOR(.x.)
#endif

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_01manage_mpi
 use interfaces_12geometry
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: ng
 type(Bz_mesh_type),intent(in) :: Kmesh,Qmesh
 type(Dataset_type),intent(in) :: Dtset
 type(Gvectors_type),intent(in) :: Gsph
 type(MPI_type),intent(inout) :: MPI_enreg
 type(Coulombian_type),intent(out) :: Vcp
!arrays
 integer,intent(in) :: ngfft(18)
 real(dp),intent(in) :: rprimd(3,3)

!Local variables-------------------------------
!number of sampling points along each axis for the numerical integration
!scalars
 integer,parameter :: nqx=50
 integer :: nprocs,master,rank,spaceComm,ierr
 integer :: ig,ii,info,ipara,iq_bz,iq_ibz,iqg,iqx,iqy,iqz,istat,nfft,npar,npt
 integer :: opt_cylinder,opt_surface,test
 real(dp),parameter :: tolq0=1.d-3
 real(dp) :: bz_geometry_factor,bz_plane,check,chisq,dx,integ,q0_vol,q0_volsph
 real(dp) :: qbz_norm,step,ucvol
 logical,parameter :: use_faux=.FALSE.
 logical :: ltest
 character(len=500) :: msg
!arrays
 integer :: gamma_pt(3,1)
 integer,pointer :: gvec(:,:)
 real(dp) :: a1(3),a2(3),a3(3),b1(3),b2(3),b3(3),gmet(3,3),gprimd(3,3)
 real(dp) :: qbz_cart(3),qq(3),rmet(3,3)
 real(dp),allocatable :: cov(:,:),par(:),qfit(:,:),qpg(:),sigma(:),var(:)
 real(dp),allocatable :: vcfit(:,:),vcoul(:,:),xx(:),yy(:)

! *************************************************************************

 DBG_ENTER("COLL")
 !
 ! === Test if the first q-point is zero === 
 ! FIXME this wont work if nqptdm/=0
 !if (normv(Qmesh%ibz(:,1),gmet,'G')<GW_TOLQ0)) STOP 'setup_coulombian, non zero first point '

 call metric(gmet,gprimd,-1,rmet,rprimd,ucvol)
 gvec => Gsph%gvec(:,:) 
 !
 ! === Save dimension and other useful quantities in Vcp% ===
 Vcp%nfft      = ngfft(1)*ngfft(2)*ngfft(3) ! Number of points in FFT mesh
 Vcp%ng        = ng                         ! Number of G-vectors in the coulombian matrix elements
 Vcp%nqibz     = Qmesh%nibz                 ! Number of irred q-point 
 ! TODO this is not yet implemented 
 Vcp%nqsmall   = 1                           ! Number of small q-directions to deal with singularity and non Analytic behavior 
 Vcp%rcut      = Dtset%rcut                 ! Cutoff radius for cylinder
 Vcp%hcyl      = zero                       ! Length of finite cylinder (Rozzi"s method, default is Beigi)
 Vcp%ucvol     = ucvol                      ! Unit cell volume

 Vcp%rprimd    = rprimd(:,:)                ! Dimensional direct lattice
 Vcp%boxcenter = zero                       ! boxcenter at the moment is supposed to be at the origin
 !Vcp%boxcenter=Dtset%boxcenter     
 Vcp%vcutgeo   = Dtset%vcutgeo(:)           ! Info on the orientation and extension of the cutoff region
 Vcp%ngfft(:)  = ngfft(:)                   ! Info on FFT mesh
 !
 ! === Define geometry and cutoff radius (if used) ===
 Vcp%mode='NONE'
 if (Dtset%icutcoul==0) Vcp%mode='SPHERE'
 if (Dtset%icutcoul==1) Vcp%mode='CYLINDER'
 if (Dtset%icutcoul==2) Vcp%mode='SURFACE'
 if (Dtset%icutcoul==3) Vcp%mode='CRYSTAL'
 if (Dtset%icutcoul==4) Vcp%mode='ERF'
 if (Dtset%icutcoul==5) Vcp%mode='ERFC'
 !
 ! === Calculate Fourier coefficient of coulombian interaction ===
 ! * For the limit q-->0 we consider ng vectors due to a possible anisotropy in case of a cutoff interaction
 allocate(Vcp%vc_sqrt(ng,Vcp%nqibz))     ; Vcp%vc_sqrt  =czero
 !print*,'nqsmal',Qmesh%nsmall ; call flush_unit(std_out)
 allocate(Vcp%vcqs_sqrt(ng,Vcp%nqsmall),STAT=istat) ; Vcp%vcqs_sqrt=czero  !STAT required by buggy g95 if called by sigma
 allocate(vcoul(ng,Vcp%nqibz)) 

 a1=rprimd(:,1) ; b1=two_pi*gprimd(:,1)
 a2=rprimd(:,2) ; b2=two_pi*gprimd(:,2)
 a3=rprimd(:,3) ; b3=two_pi*gprimd(:,3)

 SELECT CASE (TRIM(Vcp%mode)) 

 CASE ('SPHERE')
  ! TODO check that L-d > R_c > d
  ! A non-positive value of rcut imposes
  ! to follow automatically the recipe of Spencer & Alavi, PRB 77, 193110 (2008).
  if (Vcp%rcut<tol12) then
   Vcp%rcut = (ucvol*Kmesh%nbz*3.d0/four_pi)**third
   write(msg,'(2a,2x,f8.4,a)')ch10,&
&   ' Using a calculated value for rcut = ',Vcp%rcut, ' to have the same volume as the BvK crystal '
   call wrtout(std_out,msg,'COLL') 
  end if

  Vcp%vcutgeo=zero
  call cutoff_sphere(Qmesh%nibz,Qmesh%ibz,ng,gvec,gmet,Vcp%rcut,vcoul)
  !Vcp%vc_sqrt=vcoul ; Vcp%vc_sqrt=SQRT(Vcp%vc_sqrt) 
  !deallocate(vcoul) ; allocate(vcoul(ng,Qmesh%nsmall))
  !call cutoff_sphere(Qmesh%nsmall,Qmesh%small,ng,gvec,gmet,Vcp%rcut,vcoul)
  !Vcp%vcqs_sqrt=vcoul ; Vcp%vcqs_sqrt=SQRT(Vcp%vcqs_sqrt) 
  !
  ! === Treat the limit q--> 0 ===
  ! * The small cube is approximated by a sphere, while vc(q=0)=2piR**2
  ! * if a single q-point is used, the expression for the volume is exact
  Vcp%i_sz=two_pi*Vcp%rcut**2
  call print_coulombian(Vcp) ; call print_coulombian(Vcp,unit=ab_out) 

 CASE ('CYLINDER')
  test=COUNT(ABS(Vcp%vcutgeo)>tol6) 
  call assert((test==1),'Wrong cutgeo for cylinder',__FILE__,__LINE__)
  ! === Beigi method is default i.e infinite cylinder of radius rcut ===
  opt_cylinder=1 ; Vcp%hcyl=zero ; Vcp%pdir(:)=0
  do ii=1,3
   check=Vcp%vcutgeo(ii)
   if (ABS(check)>tol6) then 
    Vcp%pdir(ii)=1
    if (check<zero) then 
     ! * In case it is possible to use Rozzi method with finite cylinder
     opt_cylinder=2 ; Vcp%hcyl=ABS(check)*SQRT(SUM(rprimd(:,ii)**2))
    end if
   end if
  end do
  test=COUNT(Vcp%pdir==1) 
  call assert((test==1),'Wrong pdir for cylinder',__FILE__,__LINE__)
  if (Vcp%pdir(3)/=1) STOP "not implemented yet"
  call cutoff_cylinder(Qmesh%nibz,Qmesh%ibz,ng,gvec,Vcp%rcut,Vcp%hcyl,Vcp%pdir,&
&  Vcp%boxcenter,rprimd,vcoul,opt_cylinder,MPI_enreg)
  !Vcp%vc_sqrt=vcoul ; Vcp%vc_sqrt=SQRT(Vcp%vc_sqrt) 
  !deallocate(vcoul) ; allocate(vcoul(ng,Qmesh%nsmall))
  !call cutoff_cylinder(Qmesh%nsmall,Qmesh%small,ng,gvec,Vcp%rcut,Vcp%hcyl,Vcp%pdir,&
  !&Vcp%boxcenter,rprimd,vcoul,opt_cylinder,MPI_enreg)
  !Vcp%vcqs_sqrt=vcoul ; Vcp%vcqs_sqrt=SQRT(Vcp%vcqs_sqrt) 
  !
  ! === Treat the limit q--> 0 ===
  if (opt_cylinder==1) then 
   npar=8 ; npt=100 ; gamma_pt=RESHAPE((/0,0,0/),(/3,1/))
   allocate(qfit(3,npt),vcfit(1,npt)) ! TODO for the moment only z-axis
   if (Qmesh%nibz==1) STOP
   qfit(:,:)=zero 
   step=half/(npt*(Qmesh%nibz-1))              ; qfit(3,:)=arth(tol6,step,npt)
   !step=(half/(Qmesh%nibz-1)/tol6)**(one/npt)  ; qfit(3,:)=geop(tol6,step,npt)
   call cutoff_cylinder(npt,qfit,1,gamma_pt,Vcp%rcut,Vcp%hcyl,Vcp%pdir,Vcp%boxcenter,rprimd,vcfit,opt_cylinder,MPI_enreg)
   allocate(xx(npt),yy(npt),sigma(npt),par(npar),var(npar),cov(npar,npar))
   do ii=1,npt ; xx(ii)=normv(qfit(:,ii),gmet,'G') ; end do
   sigma=one ; yy(:)=vcfit(1,:)
   !call llsfit_svd(xx,yy,sigma,npar,K0fit,chisq,par,var,cov,info)
   !do ii=1,npt
   ! write(99,*)xx(ii),yy(ii),DOT_PRODUCT(par,K0fit(xx(ii),npar))
   !end do
   bz_plane=l2norm(b1.x.b2)
   !integ=K0fit_int(xx(npt),par,npar)
   !write(*,*)' SVD fit : chi-square',chisq 
   !write(*,*)' fit-parameters : ',par
   !write(*,*)' variance ',var
   !write(*,*)' bz_plane ',bz_plane
   !write(*,*)' SCD integ ',integ
   ! Here Im assuming homogeneous mesh
   dx=(xx(2)-xx(1))
   integ=yy(2)*dx*3.0/2.0
   do ii=3,npt-2
    integ=integ+yy(ii)*dx
   end do
   integ=integ+yy(npt-1)*dx*3.0/2.0
   write(*,*)' simple integral',integ
   q0_volsph=(two_pi)**3/(Kmesh%nbz*ucvol) 
   q0_vol=bz_plane*two*xx(npt)
   write(*,*)' q0 sphere : ',q0_volsph,' q0_vol cyl ',q0_vol
   Vcp%i_sz=bz_plane*two*integ/q0_vol
   write(*,*)' spherical approximation ',four_pi*7.44*q0_volsph**(-two_thirds) 
   write(*,*)' Cylindrical cutoff value ',Vcp%i_sz
   !Vcp%i_sz=four_pi*7.44*q0_vol**(-two_thirds) 
   deallocate(xx,yy,sigma,par,var,cov)
  end if
  call print_coulombian(Vcp) ; call print_coulombian(Vcp,unit=ab_out)

 CASE ('SURFACE') 
  ! TODO should check that R=L_Z/2 and fill Pcv%rcut the surface must be along x-y
  test=COUNT(Vcp%vcutgeo/=zero) ; if (test/=2) STOP ' check cutgeo '
  !
  ! === Default is Beigi"s method ===
  opt_surface=1  ; Vcp%alpha(:)=zero 
  if (ANY(Vcp%vcutgeo<zero)) opt_surface=2
  Vcp%pdir(:)=zero
  do ii=1,3
   check=Vcp%vcutgeo(ii)
   if (ABS(test)>zero) then 
    ! * In case it is possible to use Rozzi method with a finite surface along x-y
    Vcp%pdir(ii)=1
    if (test<zero) then 
     Vcp%alpha(ii)=normv(check*rprimd(:,ii),rmet,'R')
    end if
   end if
  end do
  call cutoff_surface(Qmesh%nibz,Qmesh%ibz,ng,gvec,gprimd,gmet,Vcp%rcut,&
&  Vcp%boxcenter,Vcp%pdir,Vcp%alpha,vcoul,opt_surface)
  !Vcp%vc_sqrt=vcoul ; Vcp%vc_sqrt=SQRT(Vcp%vc_sqrt) 
  !deallocate(vcoul) ; allocate(vcoul(ng,Qmesh%nsmall))
  !call cutoff_surface(Qmesh%nsmall,Qmesh%small,ng,gvec,gprimd,gmet,Vcp%rcut,vcoul,opt_surface)
  !Vcp%vcqs_sqrt=vcoul ; Vcp%vcqs_sqrt=SQRT(Vcp%vcqs_sqrt) 
  !
  ! === Treat the limit q--> 0 TODO 
  call print_coulombian(Vcp) ; call print_coulombian(Vcp,unit=ab_out)

 CASE ('CRYSTAL')
  do iq_ibz=1,Qmesh%nibz
   call cvc(Qmesh%nibz,iq_ibz,Qmesh%ibz,ng,gvec,gprimd,vcoul(:,iq_ibz))
  end do
  vcoul=four_pi/(vcoul**2)
  !call clcqpg(ng,gvec,gprimd,Qmesh%ibz,Qmesh%nibz,vcoul)
  !WHERE (vcoul>tol12) vcoul=four_pi/(vcoul**2)
  !Vcp%vc_sqrt=vcoul ; Vcp%vc_sqrt=SQRT(Vcp%vc_sqrt)
  !deallocate(vcoul) ; allocate(vcoul(ng,Qmesh%nsmall))
  !call clcqpg(ng,gvec,gprimd,Qmesh%nsmall,Qmesh%small,vcoul)
  !Vcp%vcqs_sqrt=vcoul ; Vcp%vcqs_sqrt=SQRT(Vcp%vcqs_sqrt)
  !
  ! === Treat 1/q^2 singularity ===
  ! * We use the auxiliary function from PRB 75, 205126 (2007)
  q0_vol=(two_pi)**3/(Kmesh%nbz*ucvol) ; bz_geometry_factor=zero

if (use_faux) then
  ! Sorry Fabien, you increased nx and now this part takes a lot of time. 
  ! It might be useful to perform the integration using the routine quadrature
  ! so that we can control the accuracy of the integral and improve the numerical
  ! stability of the GW results. In my branch this part is commented out because
  ! I need to preserve the old numerical results for my developments. 
  do iq_bz=1,Qmesh%nbz
   qbz_cart(:)=Qmesh%bz(1,iq_bz)*b1(:)+Qmesh%bz(2,iq_bz)*b2(:)+Qmesh%bz(3,iq_bz)*b3(:)
   qbz_norm=SQRT(SUM(qbz_cart(:)**2))
   if (qbz_norm>tolq0) bz_geometry_factor=bz_geometry_factor-faux(Qmesh%bz(:,iq_bz))
  end do

  bz_geometry_factor = bz_geometry_factor + integratefaux()*Qmesh%nbz

  write(msg,'(2a,2x,f8.4)')ch10,&
&  ' integrate q->0 : numerical BZ geometry factor = ',bz_geometry_factor*q0_vol**(2./3.)
  call wrtout(std_out,msg,'COLL') !; call wrtout(ab_out,msg,'COLL')
  Vcp%i_sz=four_pi*bz_geometry_factor  ! Final result stored here

else 
  ! ££ MG: this is to restore the previous implementation, it will facilitate the merge
  ! Analytic integration of 4pi/q^2 over the volume element:
  ! $4pi/V \int_V d^3q 1/q^2 =4pi bz_geometric_factor V^(-2/3)$
  ! i_sz=4*pi*bz_geometry_factor*q0_vol**(-two_thirds) where q0_vol= V_BZ/N_k
  ! bz_geometry_factor: sphere=7.79, fcc=7.44, sc=6.188, bcc=6.946, wz=5.255
  ! (see gwa.pdf, appendix A.4)
  Vcp%i_sz=four_pi*7.44*q0_vol**(-two_thirds)
end if
  !call print_coulombian(Vcp) ; call print_coulombian(Vcp,unit=ab_out)

 CASE ('ERF') 
  ! using a modified long-range only Coulomb interaction thanks to the error function:
  ! Vc = erf(r/rcut)/r
  do iq_ibz=1,Qmesh%nibz
   call cvc(Qmesh%nibz,iq_ibz,Qmesh%ibz,ng,gvec,gprimd,vcoul(:,iq_ibz))
  end do
  !
  ! the Fourier transform of the error function reads
  vcoul=four_pi/(vcoul**2) * (  exp( -0.25d0 * (Vcp%rcut*vcoul)**2 ) )
  !
  ! === Treat 1/q^2 singularity === 
  ! * We use the auxiliary function from PRB 75, 205126 (2007)
  q0_vol=(two_pi)**3/(Kmesh%nbz*ucvol) ; bz_geometry_factor=zero
  do iq_bz=1,Qmesh%nbz
   qbz_cart(:)=Qmesh%bz(1,iq_bz)*b1(:)+Qmesh%bz(2,iq_bz)*b2(:)+Qmesh%bz(3,iq_bz)*b3(:)
   qbz_norm=SQRT(SUM(qbz_cart(:)**2))
   if (qbz_norm>tolq0) bz_geometry_factor=bz_geometry_factor-faux(Qmesh%bz(:,iq_bz))
  end do

  bz_geometry_factor = bz_geometry_factor + integratefaux()*Qmesh%nbz

  write(msg,'(2a,2x,f12.4)')ch10,&
&  ' integrate q->0 : numerical BZ geometry factor = ',bz_geometry_factor*q0_vol**(2./3.)
  call wrtout(std_out,msg,'COLL') !; call wrtout(ab_out,msg,'COLL')
  Vcp%i_sz=four_pi*bz_geometry_factor  ! Final result stored here
  !
  !call print_coulombian(Vcp) ; call print_coulombian(Vcp,unit=ab_out)
 CASE ('ERFC') 
  ! using a modified short-range only Coulomb interaction thanks to the complementar error function:
  ! Vc = [1-erf(r/rcut)]/r
  do iq_ibz=1,Qmesh%nibz
   call cvc(Qmesh%nibz,iq_ibz,Qmesh%ibz,ng,gvec,gprimd,vcoul(:,iq_ibz))
  end do
  !
  ! the Fourier transform of the error function reads
  vcoul=four_pi/(vcoul**2) * ( 1.d0 - exp( -0.25d0 * (Vcp%rcut*vcoul)**2 ) )
  !
  ! === Treat 1/q^2 singularity === 
  ! * There is NO singularity in this case.
  Vcp%i_sz=pi*Vcp%rcut**2 ! Final result stored here
  !
  !call print_coulombian(Vcp) ; call print_coulombian(Vcp,unit=ab_out)

 CASE DEFAULT
  write(msg,'(2a)')' Undefined cutoff mode ',TRIM(Vcp%mode)
  MSG_BUG(msg)
 END SELECT
 !
 ! === Store final results in complex array ===
 ! * Rozzi"s cutoff can give real negative values 
 Vcp%vc_sqrt=CMPLX(vcoul,zero) ; Vcp%vc_sqrt=SQRT(Vcp%vc_sqrt) 
 call plot_Vc(Vcp,Qmesh,Gsph,ng,vcoul,MPI_enreg)
 deallocate(vcoul)
 !
 ! === Create a table for FFT points inside the cutoff region ===
 if (TRIM(Vcp%mode)/='CRYSTAL') then
  allocate(Vcp%ctab(Vcp%nfft))
  call cutoff_table(Vcp)
 else 
  allocate(Vcp%ctab(1))
 end if

 DBG_EXIT("COLL")

contains !===============================================================

 real(dp) function integratefaux()

 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_lib01hidempi
!End of the abilint section

 implicit none

!Arguments ------------------------------------
 !MG here there is a bug in abilint since integratefaux is chaged to integrate, dont know why!
 !real(dp) :: integrate

!Local variables-------------------------------
  integer,parameter :: nref=3  ! area where the mesh is refined
  integer :: spaceComm,master,rank,nprocs,ierr
  integer :: iqx1,iqy1,iqz1,iqx2,iqy2,iqz2
  real(dp) :: qq1(3),qq2(3)

  call xcomm_init  (MPI_enreg,spaceComm)
  call xmaster_init(MPI_enreg,master)
  call xme_init    (MPI_enreg,rank  )
  call xproc_max(nprocs,ierr)

  integratefaux=zero
  do iqx1=1,nqx
   do iqy1=1,nqx
    do iqz1=1,nqx
     if(modulo(iqx1,nprocs)/=rank) cycle
     qq1(1)=DBLE(iqx1)/DBLE(nqx)-half
     qq1(2)=DBLE(iqy1)/DBLE(nqx)-half
     qq1(3)=DBLE(iqz1)/DBLE(nqx)-half
     !
     ! Refine the mesh for the point close to the origin
     if( abs(iqx1-nqx/2)<=nref .and. abs(iqy1-nqx/2)<=nref .and. abs(iqz1-nqx/2)<=nref ) then
      do iqx2=1,nqx
       do iqy2=1,nqx
        do iqz2=1,nqx
         qq2(1)=qq1(1) + (DBLE(iqx2)/DBLE(nqx)-half ) /DBLE(nqx)
         qq2(2)=qq1(2) + (DBLE(iqy2)/DBLE(nqx)-half ) /DBLE(nqx)
         qq2(3)=qq1(3) + (DBLE(iqz2)/DBLE(nqx)-half ) /DBLE(nqx)
         !
         ! Treat the remaining divergence in the origin as if it would be a spherical
         ! integration of 1/q^2
         if( iqx1/=nqx/2 .or. iqy1/=nqx/2 .or. iqz1/=nqx/2 .or. iqx2/=nqx/2 .or. iqy2/=nqx/2 .or. iqz2/=nqx/2 ) then
          integratefaux=integratefaux+ faux(qq2) /DBLE(nqx)**6
         else
          integratefaux=integratefaux&
&                 +7.7955* ( (two_pi)**3/ucvol/DBLE(nqx)**6 )**(-2./3.) /DBLE(nqx)**6
         endif
        enddo
       enddo
      enddo
     else
      integratefaux=integratefaux+faux(qq1)/DBLE(nqx)**3
     endif
    end do
   end do
  end do
  call xsum_mpi(integratefaux,spaceComm,ierr)

 end function integratefaux

 function faux(qq)


  real(dp),intent(in) :: qq(3)
  real(dp) :: faux

  faux= four*( dot_product(b1,b1) * SIN(two_pi*qq(1)/two)**2 &
&             +dot_product(b2,b2) * SIN(two_pi*qq(2)/two)**2 &
&             +dot_product(b3,b3) * SIN(two_pi*qq(3)/two)**2 &
&            )                                              &
&       +two*( dot_product(b1,b2) * SIN(two_pi*qq(1))*SIN(two_pi*qq(2)) &
&             +dot_product(b2,b3) * SIN(two_pi*qq(2))*SIN(two_pi*qq(3)) &
&             +dot_product(b3,b1) * SIN(two_pi*qq(3))*SIN(two_pi*qq(1)) &
&            )
  faux=(two_pi)**2/faux * exp( -0.25d0*Vcp%rcut**2* sum( ( qq(1)*b1(:)+qq(2)*b2(:)+qq(3)*b3(:) )**2 ) )
 end function faux

end subroutine setup_coulombian
!!***

!!****f* m_coulombian/plot_Vc
!! NAME
!! plot_Vc
!!
!! FUNCTION
!! Plot vccut(q,G) as a function of |q+G|, calculate also vc in real space.
!!
!! INPUTS
!!
!! OUTPUT
!!
!! PARENTS
!!      setup_coulombian
!!
!! CHILDREN
!!      wrtout
!!
!! SOURCE

subroutine plot_Vc(Vcp,Qmesh,Gsph,ng,vc,MPI_enreg)

 use defs_basis
 use defs_datatypes
 use m_io_tools, only : get_unit
 use m_geometry, only : normv

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_01manage_mpi
 use interfaces_lib01hidempi
 use interfaces_lib03numeric
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: ng
 type(Bz_mesh_type),intent(in) :: Qmesh
 type(Coulombian_type),intent(in) :: Vcp
 type(Gvectors_type),intent(in) :: Gsph
 type(MPI_type),intent(inout) :: MPI_enreg
!arrays
 real(dp),intent(in) :: vc(ng,Qmesh%nibz)

!Local variables-------------------------------
!scalars
 integer :: dimr,icount,idx_Sm1G,ierr,ig,igs,ii,iq_bz,iq_ibz,iqg,ir,isym,itim
 integer :: master,my_start,my_stop,nprocs,nqbz,nqibz,nr,ntasks,rank,spaceComm
 integer :: unt
 real(dp) :: arg,fact,l1,l2,l3,lmax,step,tmp,vcft,zmax
 character(len=500) :: msg
 character(len=fnlen) :: filnam
!arrays
 integer,allocatable :: insort(:)
 integer,pointer :: gvec(:,:)
 real(dp) :: b1(3),b2(3),b3(3),gmet(3,3),gprimd(3,3),qbz(3),qpgc(3)
 real(dp),allocatable :: kin(:),rr(:,:,:),vcr(:,:),vcr_cut(:,:)

!************************************************************************

 if (TRIM(Vcp%mode)=='CRYSTAL')  RETURN

 call xcomm_init  (MPI_enreg,spaceComm)
 call xme_init    (MPI_enreg,rank     )         
 call xmaster_init(MPI_enreg,master   ) 

 nqibz=Qmesh%nibz 
 nqbz =Qmesh%nbz

 gmet   =  Gsph%gmet(:,:) 
 gprimd =  Gsph%gprimd(:,:) 
 gvec   => Gsph%gvec(:,:)

 b1(:)=two_pi*gprimd(:,1)
 b2(:)=two_pi*gprimd(:,2)
 b3(:)=two_pi*gprimd(:,3)

 if (rank==master) then 
  allocate(insort(nqibz*ng),kin(nqibz*ng))
  iqg=1
  do iq_ibz=1,nqibz
   do ig=1,ng
    kin(iqg)=normv(Qmesh%ibz(:,iq_ibz)+gvec(:,ig),gmet,'g')
    insort(iqg)=iqg ; iqg=iqg+1
   end do
  end do
  call sort_dp(nqibz*ng,kin,insort,tol14)
  filnam='_VCoulFT_' ; call isfile(filnam,'new')
  unt=get_unit()
  open(unt,file=filnam,status='new',form='formatted')
  write(unt,'(a,i3,a,i6,a)')&
&  '#  |q+G|^2/2    q-points (',nqibz,')        gvec (',ng,' )  vc      vc_cutoff'
  do iqg=1,nqibz*ng
   iq_ibz=(insort(iqg)-1)/ng +1
   ig=(insort(iqg))-(iq_ibz-1)*ng
   write(unt,'(f12.6,2x,3f8.4,2x,3i6,2x,2es14.6)')&
&   kin(iqg),Qmesh%ibz(:,iq_ibz),gvec(:,ig),two_pi/kin(iqg)**2,vc(ig,iq_ibz)
  end do 
  close(unt)
  deallocate(insort,kin)
 end if

 nr=50 ; fact=one/(Vcp%ucvol*nqbz)
 ntasks=nqbz*ng
 call split_work(ntasks,my_start,my_stop)

 l1=SQRT(SUM(Vcp%rprimd(:,1)**2))
 l2=SQRT(SUM(Vcp%rprimd(:,2)**2))
 l3=SQRT(SUM(Vcp%rprimd(:,3)**2))
 lmax=MAX(l1,l2,l3) ; step=lmax/(nr-1)
 allocate(rr(3,nr,3),vcr(nr,3),vcr_cut(nr,3)) 
 !
 ! numb coding just to check cutoff implementation
 rr=zero
 do ii=1,3
  do ir=1,nr
   rr(ii,ir,ii)=(ir-1)*step
  end do
 end do

 vcr=zero ; vcr_cut=zero 
 do iq_bz=1,nqbz
  call get_BZ_item(Qmesh,iq_bz,qbz,iq_ibz,isym,itim)
  if (ABS(qbz(1))<0.01) qbz(1)=zero
  if (ABS(qbz(2))<0.01) qbz(2)=zero
  if (ABS(qbz(3))<0.01) qbz(3)=zero
  igs=1 ; if (ALL(qbz(:)==zero)) igs=2
  do ig=igs,ng
   icount=ig+(iq_bz-1)*ng ; if (icount<my_start.or.icount>my_stop) CYCLE
   idx_Sm1G=Gsph%rottbm1(ig,itim,isym) ! IS{^-1}G
   vcft=vc(idx_Sm1G,iq_ibz)
   qpgc(:)=qbz(:)+gvec(:,ig) ; qpgc(:)=b1(:)*qpgc(1)+b2(:)*qpgc(2)+b3(:)*qpgc(3)
   tmp=SQRT(DOT_PRODUCT(qpgc,qpgc)) ; tmp=tmp**2
   do ii=1,3
    do ir=1,nr
     arg=DOT_PRODUCT(rr(:,ir,ii),qpgc)
     vcr_cut(ir,ii)=vcr_cut(ir,ii)+vcft*COS(arg)
     vcr(ir,ii)=vcr(ir,ii)+four_pi/tmp*COS(arg)
    end do
   end do
  end do !ig
 end do !iq_ibz
 call leave_test(MPI_enreg)
 call xsum_master(vcr_cut,master,spaceComm,ierr)
 call xsum_master(vcr    ,master,spaceComm,ierr)
 if (rank==master) then 
  filnam='_VCoulR_' ; call isfile(filnam,'new')
  unt=get_unit()
  open(unt,file=filnam,status='new',form='formatted')
  do ir=1,nr
   write(unt,'(7es18.6)')(ir-1)*step,(fact*vcr(ir,ii),fact*vcr_cut(ir,ii),ii=1,3)
  end do
  close(unt)
 end if
 deallocate(rr,vcr)

end subroutine plot_Vc
!!***


!!****f* m_coulombian/print_coulombian
!! NAME
!! print_coulombian
!!
!! FUNCTION
!!  Print the content of a coulombian datatype
!!
!! INPUTS
!!  Vcp<Coulombian_type>=the datatype to be printed
!!  unit(optional)=the unit number for output 
!!  prtvol(optional)=verbosity level
!!  mode_paral=either "COLL" or "PERS"
!!
!! PARENTS
!!      setup_coulombian
!!
!! CHILDREN
!!      wrtout
!!
!! SOURCE

subroutine print_coulombian(Vcp,unit,prtvol,mode_paral)

 use defs_basis
 use defs_datatypes
 use m_numeric_tools, only : imin_loc

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_01manage_mpi
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in),optional :: prtvol,unit
 character(len=4),intent(in),optional :: mode_paral
 type(Coulombian_type),intent(in) :: Vcp

!Local variables-------------------------------
!scalars
 integer :: ii,unt,verbose
 character(len=100) :: fmt
 character(len=4) :: mode
 character(len=500) :: msg

! *************************************************************************

 unt=std_out ; if (PRESENT(unit))       unt=unit
 verbose=0   ; if (PRESENT(prtvol))     verbose=prtvol
 mode='COLL' ; if (PRESENT(mode_paral)) mode=mode_paral

 SELECT CASE (TRIM(Vcp%mode)) 

 CASE ('SPHERE')
  write(msg,'(5a,f10.4,3a,f10.2,3a,3f10.5,2a)')ch10,&
&  ' === Spherical cutoff === ',ch10,ch10,&
&  '  Cutoff radius ......... ',Vcp%rcut,' [Bohr] ',ch10,&
&  '  Volume of the sphere .. ',four_pi/three*Vcp%rcut**3,' [Bohr^3] '
!FB: This has no meaning here! &  '  Sphere centered at .... ',Vcp%boxcenter,' (r.l.u) ',ch10
!MG It might be useful if the system is not centered on the origin because in this case the 
!   matrix elements of the coulombian have to be multiplied by a phase depending on boxcenter.
!   I still have to decide if it is useful to code this possibility and which variable use to 
!   define the center (boxcenter comes is used in the tddft part). 
  call wrtout(unt,msg,mode) 

 CASE ('CYLINDER')
  ii=imin_loc(ABS(Vcp%pdir-1))
  write(msg,'(5a,f10.4,3a,i2,2a,3f10.2,a)')ch10,&
&  ' === Cylindrical cutoff === ',ch10,ch10,&
&  '  Cutoff radius .......... ',Vcp%rcut,' [Bohr] ',ch10,&
&  '  Axis parallel to dir.... ',ii,ch10,&
&  '  Passing through point .. ',Vcp%boxcenter,' (r.l.u) '
  call wrtout(unt,msg,mode) 
  write(msg,'(2a)')'  Infinite length  ....... ',ch10
  if (Vcp%hcyl/=zero) write(msg,'(a,f8.5,2a)')'  Finite length of ....... ',Vcp%hcyl,' [Bohr] ',ch10
  call wrtout(unt,msg,mode) 

 CASE ('SURFACE') 
  write(msg,'(5a,f10.4,3a,3f10.2,2a)')ch10,&
&  ' === Surface cutoff === ',ch10,ch10,&
&  '  Cutoff radius .................... ',Vcp%rcut,' [Bohr] ',ch10,&
&  '  Central plane passing through .... ',Vcp%boxcenter,' (r.l.u) ',ch10
  call wrtout(unt,msg,mode) 
  !write(msg,'(a)')'  Infinite length  .......'
  !if (Vcp%hcyl/=zero) write(msg,'(a,f8.5,a)')'  Finite length of .......',Vcp%hcyl,' [Bohr] '
  !call wrtout(unt,msg,mode) 

 CASE ('CRYSTAL')
  write(msg,'(3a)')ch10,&
&  ' setup_coulombian : cutoff-mode = ',TRIM(Vcp%mode)
  call wrtout(unt,msg,mode) 

 CASE ('ERF')
  write(msg,'(5a,f10.4,3a,f10.2,3a,3f10.5,2a)')ch10,&
&  ' === Error function cutoff === ',ch10,ch10,&
&  '  Cutoff radius ......... ',Vcp%rcut,' [Bohr] ',ch10
  call wrtout(unt,msg,mode)

 CASE ('ERFC')
  write(msg,'(5a,f10.4,3a,f10.2,3a,3f10.5,2a)')ch10,&
&  ' === Complement Error function cutoff === ',ch10,ch10,&
&  '  Cutoff radius ......... ',Vcp%rcut,' [Bohr] ',ch10
  call wrtout(unt,msg,mode)

 CASE DEFAULT
  write(msg,'(2a)')' Undefined cutoff mode ',TRIM(Vcp%mode)
  MSG_BUG(msg)
 END SELECT

 !TODO add additional information

end subroutine print_coulombian 
!!***

!!****f* m_coulombian/cutoff_table
!! NAME
!! cutoff_table
!!
!! FUNCTION
!!  Create a table for each point in the FFT grid: 0 if the point is outside the cutoff region, 1 otherwise 
!!  This table can be used to redefine the density in the isolated system (see cutoff_density)
!!
!! INPUTS
!!  rmet(3,3)=real space metric ($\textrm{bohr}^{2}$).
!!  Vcp<type Coulombian_type> 
!!   The following quantities are used :
!!   %ngfft(1:3)= FFT dimensions
!!   %nr= number of point in the FFT mesh
!!   %boxcenter=reduced coordinates of the center of the box  (input variable
!!  (FIXME kept from tddft.F90, should write something in doc) 
!!   %cutoffmode= string of characters defining the cutoff mode 
!!   %rcut= cutoff radius 
!!
!! OUTPUT
!!  Vcp%ctab(Vcp%nr)
!!  For each point of the FFT grid gives 
!!   1 if the point in inside the cutoff region
!!   0 otherwise
!!
!! SIDE EFFECTS
!!
!! NOTES
!!
!! PARENTS
!!      setup_coulombian
!!
!! CHILDREN
!!      wrtout
!!
!! SOURCE

subroutine cutoff_table(Vcp)

 use defs_basis
 use defs_datatypes
 use m_geometry, only : normv

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_12geometry
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 type(Coulombian_type),intent(inout) :: Vcp

!Local variables-------------------------------
!scalars
 integer :: ir,ix,iy,iz,ngfft1,ngfft2,ngfft3
 real(dp) :: ucvol
 character(len=500) :: msg
!arrays
 real(dp) :: gmet(3,3),gprimd(3,3),rmet(3,3),rr(3),rxy(3),rz(3)

!************************************************************************

 if (TRIM(Vcp%mode)/='CRYSTAL') RETURN

 call metric(gmet,gprimd,-1,rmet,Vcp%rprimd,ucvol)
 ngfft1=Vcp%ngfft(1) 
 ngfft2=Vcp%ngfft(2) 
 ngfft3=Vcp%ngfft(3)

 Vcp%ctab(:)=1
 do ix=0,ngfft1-1
  rr(1)=DBLE(ix)/ngfft1
  do iy=0,ngfft2-1
   rr(2)=DBLE(iy)/ngfft2
   do iz=0,ngfft3-1
    rr(3)=DBLE(iz)/ngfft3
    ir=1+ix+iy*ngfft1+iz*ngfft1*ngfft2
    rr(:)=rr(:)-Vcp%boxcenter(:)

    SELECT CASE (TRIM(Vcp%mode))
     CASE ('SPHERE')
      if (normv(rr,rmet,'R')>Vcp%rcut) Vcp%ctab(ir)=0
     CASE ('CYLINDER') ! Infinite cylinder
      rxy(1)=rr(1)
      rxy(2)=rr(2)
      rxy(3)=zero
      if (normv(rxy,rmet,'R')>Vcp%rcut) Vcp%ctab(ir)=0
     CASE ('SURFACE')
      rz(:)=zero
      rz(3)=rr(3)
      if (normv(rz,rmet,'R')>Vcp%rcut) Vcp%ctab(ir)=0
     CASE DEFAULT   
      write(msg,'(2a)')' Undefined cutoff mode ',TRIM(Vcp%mode)
      MSG_BUG(msg)
    END SELECT

   end do 
  end do 
 end do

end subroutine cutoff_table
!!***

!!****f* m_coulombian/cutoff_density
!! NAME
!! cutoff_density
!!
!! FUNCTION
!!  Modify density in case of calculations with Coulomb cutoff
!!
!! INPUTS
!!  ngfft(18)=information on the FFT grid 
!!  (at the moment only the first three elements are used and are checked
!!  against the values contained in the Coulombian_type datatype)
!!  nspden=input variable 
!!  Vcp <type Coulombian_type>, only ctab(ngfft(1)*ngfft(2)*ngfft(3)) is used: 
!!   For each point of the FFT grid gives 
!!   1 if the point in inside the cutoff region, 0 otherwise
!!  ucvol=unit cell volume 
!!
!! OUTPUT
!!  Only printing
!!
!! SIDE EFFECTS
!!
!! NOTES
!!
!! PARENTS
!!      screening,sigma
!!
!! CHILDREN
!!      wrtout
!!
!! SOURCE

subroutine cutoff_density(ngfft,nspden,nsppol,Vcp,rhor,MPI_enreg)

 use defs_basis
 use defs_datatypes

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_lib01hidempi
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: nspden,nsppol
 type(Coulombian_type),intent(in) :: Vcp
 type(MPI_type),intent(in) :: MPI_enreg
!arrays
 integer,intent(in) :: ngfft(18)
 real(dp),intent(in) :: rhor(ngfft(1)*ngfft(2)*ngfft(3),nspden)

!Local variables-------------------------------
!scalars
 integer :: ir,is,ix,iy,iz,master,ngfft1,ngfft2,ngfft3,rank
 real(dp),parameter :: EPS=1.d-2
 real(dp) :: fact
 character(len=500) :: msg
!arrays
 real(dp) :: nel(nspden),overflow(nspden),rr(3),rxy(3),rz(3)

! *************************************************************************
 
 ! === Do some check ===
 if (ANY(ngfft(1:3)/=Vcp%ngfft(1:3))) stop 'BUG in cutoff_density'
 if (Vcp%mode=='CRYSTAL') RETURN

 call xmaster_init(MPI_enreg,master   ) 
 call xme_init    (MPI_enreg,rank     )          

 ngfft1=ngfft(1) 
 ngfft2=ngfft(2) 
 ngfft3=ngfft(3)
 overflow(:)=zero ; nel(:)=zero

 do ix=0,ngfft1-1
  rr(1)=DBLE(ix)/ngfft1
  do iy=0,ngfft2-1
   rr(2)=DBLE(iy)/ngfft2
   do iz=0,ngfft3-1
    rr(3)=DBLE(iz)/ngfft3
    ir=1+ix+iy*ngfft1+iz*ngfft1*ngfft2

    nel(:)=nel(:)+rhor(ir,:)
    if (Vcp%ctab(ir)==0) then 
     overflow(:)=overflow+rhor(ir,:)
     !rhor(ir,:)=zero
    end if 

   end do 
  end do 
 end do

 nel(:)=nel(:)*Vcp%ucvol/(ngfft1*ngfft2*ngfft3)
 overflow(:)=overflow(:)*Vcp%ucvol/(ngfft1*ngfft2*ngfft3)

 if (rank==master) then
  write(*,*)' Number of electrons inside unit cell  = ',nel(1:nsppol)
  write(*,*)' Charge density outside cutoff region  =',overflow(1:nsppol)
  !
  ! If overflow is larger that few percent of the total charge warn or stop
  ! since one should increase the size of the supercell or the cutoff radius
  ! if (ANY(overflow(:)>EPS*nel(:)))  then 
  !  write(msg,'(4a,f8.5,3a)')ch10,&
  !&  ' cutoff_density : WARNING - ',ch10,&
  !&  '  More than ',eps,' % of the charge density is outside the cutoff region',ch10,&
  !&  '  You should try to increase the cutoff radius '
  !   call wrtout(std_out,msg,'COLL') 
  ! end if 
 end if

 !write(*,*)' restoring removed charge '
 ! Restore charge neutrality, spreading the overflow inside the cutoff region
 ! Using this approach if the charge is zero close to 
 ! the boundary then there should be no spurious effect
 !do is=1,nspden
 ! fact=nel(is)/(nel(is)-overflow(is))
 ! rhor(:,is)=rhor(:,is)*fact
 !end do

end subroutine cutoff_density
!!***

!!****f* m_coulombian/destroy_coulombian
!! NAME
!! destroy_coulombian
!!
!! FUNCTION
!!  Destroy a Coulombian_type type 
!!
!! SIDE EFFECTS
!!  Vcp<Coulombian_type>=the datatype to be destroyed
!!
!! PARENTS
!!      mrgscr,screening,sigma
!!
!! CHILDREN
!!      wrtout
!!
!! SOURCE

subroutine destroy_Coulombian(Vcp) 

 use defs_basis
 use defs_datatypes

 implicit none

!Arguments ------------------------------------
!scalars
 type(Coulombian_type),intent(inout) :: Vcp

!Local variables ------------------------------
!scalars
 integer :: istat
 character(len=500) :: msg

! *************************************************************************

 DBG_ENTER("COLL")

 if (associated(Vcp%ctab     ))  deallocate(Vcp%ctab,     STAT=istat)
 if (associated(Vcp%vc_sqrt  ))  deallocate(Vcp%vc_sqrt,  STAT=istat)
 if (associated(Vcp%vcqs_sqrt))  deallocate(Vcp%vcqs_sqrt,STAT=istat)

 DBG_EXIT("COLL")
 
end subroutine destroy_Coulombian
!!***

!!****f* m_coulombian/cutoff_sphere
!! NAME
!! cutoff_sphere
!!
!! FUNCTION
!!  Calculate the Fourier transform of the coulombian interaction with a spherical cutoff:  
!!   $ v_{cut}(G)= \frac{4\pi}{|q+G|^2} [ 1-cos(|q+G|*R_cut) ] $  (1)
!!
!! INPUTS
!!  gmet(3,3)=Metric in reciprocal space.
!!  gvec(3,ngvec)=G vectors in reduced coordinates.
!!  rcut=Cutoff radius of the sphere.
!!  ngvec=Number of G vectors
!!  nqpt=Number of q-points 
!!  qpt(3,nqpt)=q-points where the cutoff coulombian is required.
!!
!! OUTPUT
!!  vc_cut(ngvec,nqpt)=Fourier components of the effective coulombian interaction.
!!
!! NOTES
!!  For |q|<small and G=0 we use 2pi.R_cut^2, namely we consider the limit q-->0 of Eq. (1) 
!!
!! PARENTS
!!      setup_coulombian
!!
!! CHILDREN
!!      wrtout
!!
!! SOURCE

subroutine cutoff_sphere(nqpt,qpt,ngvec,gvec,gmet,rcut,vc_cut)

 use defs_basis
 use m_gwdefs,   only : GW_TOLQ0
 use m_errors,   only : assert
 use m_geometry, only : normv

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: ngvec,nqpt
 real(dp),intent(in) :: rcut
!arrays
 integer,intent(in) :: gvec(3,ngvec)
 real(dp),intent(in) :: gmet(3,3),qpt(3,nqpt)
 real(dp),intent(out) :: vc_cut(ngvec,nqpt)

!Local variables-------------------------------
!scalars
 integer :: ig,igs,iqpt
 real(dp) :: qpg
 logical :: ltest
 character(len=500) :: msg

!************************************************************************

 ltest=ALL(gvec(:,1)==0)
 call assert(ltest,'First G vector should be Gamma',&
& __FILE__,__LINE__)

 do iqpt=1,nqpt
  igs=1
  ! * For small q and G=0, use the limit q-->0.
  if (normv(qpt(:,iqpt),gmet,'G')<GW_TOLQ0) then  
   vc_cut(1,iqpt)=two_pi*rcut**2
   igs=2
  end if
  do ig=igs,ngvec
   qpg=normv(qpt(:,iqpt)+gvec(:,ig),gmet,'G')
   vc_cut(ig,iqpt)=four_pi*(one-COS(rcut*qpg))/qpg**2
  end do
 end do
 
end subroutine cutoff_sphere
!!***

!!****f* m_coulombian/cutoff_cylinder
!! NAME
!! cutoff_cylinder
!!
!! FUNCTION
!!  Calculate the Fourier components of an effective Coulombian interaction 
!!  zeroed outside a finite cylindrical region.
!!  Two methods are implemented: 
!!   method==1: The interaction in the (say) x-y plane is truncated outside the Wigner-Seitz 
!!              cell centered on the wire in the x-y plane. The interaction has infinite 
!!              extent along the z axis and the Fourier transform is singular only at the Gamma point. 
!!              Only orthorombic Bravais lattice are supported.
!!   method==2: The interaction is truncated outside a cylinder of radius rcut. The cylinder has finite 
!!              extent along z. No singularity occurs. 
!!
!! INPUTS
!!  boxcenter(3)= center of the wire in the x-y axis
!!  gvec(3,ng)=G vectors in reduced coordinates 
!!  ng=number of G vectors
!!  qpt(3,nq)= q-points 
!!  MPI_enreg= datatype containing information on parallelism
!!  nq=number of q-points 
!!  rprimd(3,3)=dimensional real space primitive translations (bohr)
!!   where: rprimd(i,j)=rprim(i,j)*acell(j)
!!  method=1 for Beigi approach (infinite cylinder with interaction truncated outside the W-S cell)
!!         2 for Rozzi method (finite cylinder)
!!
!! OUTPUT
!!  vccut(ng,nq)= Fourier components of the effective coulombian interaction 
!!
!! NOTES
!!
!! PARENTS
!!      setup_coulombian
!!
!! CHILDREN
!!      calck0,calck1,caljy0,caljy1,leave_test,metric,quadrature,split_work
!!      wrtout,xcomm_init,xme_init,xsum_mpi
!!
!! SOURCE

subroutine cutoff_cylinder(nq,qpt,ng,gvec,rcut,hcyl,pdir,boxcenter,rprimd,vccut,method,MPI_enreg)

 use defs_basis
 use m_gwdefs,        only : GW_TOLQ0
 use defs_datatypes
 use m_errors,        only : die
 use m_numeric_tools, only : quadrature
 use m_finite_cylinder

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_01manage_mpi
 use interfaces_12geometry
 use interfaces_lib01hidempi
 use interfaces_lib03numeric
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: ng,nq,method
 real(dp),intent(in) :: rcut,hcyl 
 type(MPI_type),intent(in) :: MPI_enreg
!arrays
 integer,intent(in) :: gvec(3,ng),pdir(3)
 real(dp),intent(in) :: boxcenter(3),qpt(3,nq),rprimd(3,3)
 real(dp),intent(out) :: vccut(ng,nq)

!Local variables-------------------------------
!scalars
 integer,parameter :: N0=1000
 integer :: icount,ig,igs,iq,spaceComm
 integer :: ntasks,rank,master,nprocs,ierr,my_start,my_stop
 real(dp) :: j0,j1,k0,k1,qpg2,qpg_xy,tmp
 real(dp) :: qpg_z,quad,rcut2,hcyl2,c1,c2,ucvol,SMALL
 character(len=500) :: msg
!arrays
 real(dp) :: qpg(3),b1(3),b2(3),b3(3),gmet(3,3),rmet(3,3),gprimd(3,3),qc(3),gcart(3)
 real(dp),allocatable :: qcart(:,:)
!************************************************************************

 call xcomm_init(MPI_enreg,spaceComm)  
 call xme_init  (MPI_enreg,rank     )           

 ! ===================================================
 ! === Setup for the quadrature of matrix elements ===
 ! ===================================================
 qopt_    =6         ! Quadrature method, see quadrature routine.
 ntrial_  =30        ! Max number of attempts.
 accuracy_=0.001     ! Fractional accuracy required.
 npts_    =6         ! Initial number of point (only for Gauss-Legendre method).
 SMALL    =GW_TOLQ0  ! Below this value (q+G)_i is treated as zero.
 rcut_    =rcut      ! Radial cutoff, used only if method==2
 hcyl_    =hcyl      ! Lenght of cylinder along z, only if method==2

 write(msg,'(3a,2(a,i5,a),a,f8.7)')ch10,&
& ' cutoff_cylinder: Info on the quadrature method : ',ch10,&
& '  Quadrature scheme      = ',qopt_,ch10,&
& '  Max number of attempts = ',ntrial_,ch10,&
& '  Fractional accuracy    = ',accuracy_
 call wrtout(std_out,msg,'COLL') 
 !
 ! === From reduced to Cartesian coordinates ===
 call metric(gmet,gprimd,-1,rmet,rprimd,ucvol)
 b1(:)=two_pi*gprimd(:,1)
 b2(:)=two_pi*gprimd(:,2)
 b3(:)=two_pi*gprimd(:,3)

 allocate(qcart(3,nq))
 do iq=1,nq
  qcart(:,iq)=b1(:)*qpt(1,iq)+b2(:)*qpt(2,iq)+b3(:)*qpt(3,iq)
 end do

 ntasks=nq*ng 
 call split_work(ntasks,my_start,my_stop)
 !
 ! ================================================
 ! === Different approaches according to method ===
 ! ================================================
 vccut(:,:)=zero 

 SELECT CASE (method)

 CASE (1)
  ! === Infinite cylinder, interaction is zeroed outside the Wigner-Seitz cell ===
  ! * Beigi"s expression holds only if the BZ is sampled only along z.
  write(msg,'(2(a,f8.4))')' cutoff_cylinder: Using Beigi''s Infinite cylinder '
  call wrtout(std_out,msg,'COLL') 
  if (ANY(qcart(1:2,:)>SMALL)) then 
   write(*,*)' qcart = ',qcart(:,:)
   write(msg,'(5a)')&
&   ' found q-points with non zero components in the X-Y plane.',ch10,&
&   ' This is not allowed, see Notes in cutoff_cylinder.F90.',ch10,&
&   ' ACTION: Modify the q-point sampling '
   MSG_ERROR(msg)
  end if 
  ! * Check if Bravais lattice is orthorombic and parallel to the Cartesian versors.
  !   In this case the intersection of the W-S cell with the x-y plane is a rectangle with -ha_<=x<=ha_ and -hb_<=y<=hb_ 
  if ( (ANY(ABS(rprimd(2:3,  1))>tol6)).or.&
&      (ANY(ABS(rprimd(1:3:2,2))>tol6)).or.&
&      (ANY(ABS(rprimd(1:2,  3))>tol6))    &
&    ) then
   write(msg,'(a)')' Bravais lattice should be orthorombic and parallel to the cartesian versors '
   MSG_ERROR(msg)
  end if

  ha_=half*SQRT(DOT_PRODUCT(rprimd(:,1),rprimd(:,1)))
  hb_=half*SQRT(DOT_PRODUCT(rprimd(:,2),rprimd(:,2)))
  r0_=MIN(ha_,hb_)/N0
  ! 
  ! === For each pair (q,G) evaluate the integral defining the cutoff coulombian  ===
  ! * Here the code assumes that all q-vectors are non zero and q_xy/=0.
  do iq=1,nq
   igs=1 
   ! * Skip singularity at Gamma, it will be treated "by hand" in csigme.
   !if (normv(qpt(:,iq),gmet,'G')<GW_TOLQ0) igs=2
   qc(:)=qcart(:,iq)
   !qc(1:2)=zero
   write(msg,'(2(a,i3))')' entering loop iq: ',iq,' with igs = ',igs
   call wrtout(std_out,msg,'COLL')

   do ig=igs,ng
    icount=ig+(iq-1)*ng ; if (icount<my_start.or.icount>my_stop) CYCLE

    gcart(:)=b1(:)*gvec(1,ig)+b2(:)*gvec(2,ig)+b3(:)*gvec(3,ig)
    qpg(:)=qc(:)+gcart(:)
    qpgx_=qpg(1) ; qpgy_=qpg(2) ; qpg_para_=ABS(qpg(3))
    !
    ! * Calculate $ 2\int_{WS} dxdy K_0{qpg_para_\rho) cos(x.qpg_x + y.qpg_y) $
    !   where WS is the Wigner-Seitz cell.
    tmp=zero
    ! Difficult part, integrate on a small cirle of radius r0 using spherical coordinates
    !call quadrature(K0cos_dth_r0,zero,r0_,qopt_,quad,ntrial_,accuracy_,npts_,ierr)
    !if (ierr/=0) ABI_DIE("Accuracy not reached")
    !write(*,'(i8,a,es14.6)')ig,' 1 ',quad
    !tmp=tmp+quad
    ! Add region with 0<=x<=r0 and y>=+-(SQRT(r0^2-x^2))since WS is rectangular
    !call quadrature(K0cos_dy_r0,zero,r0_,qopt_,quad,ntrial_,accuracy_,npts_,ierr)
    !if (ierr/=0) ABI_DIE("Accuracy not reached")
    !write(*,'(i8,a,es14.6)')ig,' 2 ',quad
    !tmp=tmp+quad
    ! Get the in integral in the rectangle with x>=r0, should be the easiest but sometimes has problems to converge
    !call quadrature(K0cos_dy,r0_,ha_,qopt_,quad,ntrial_,accuracy_,npts_,ierr)
    !if (ierr/=0) ABI_DIE("Accuracy not reached")
    !write(*,'(i8,a,es14.6)')ig,' 3 ',quad
    ! 
    ! === More stable method: midpoint integration with Romberg extrapolation ===
    call quadrature(K0cos_dy,zero,ha_,qopt_,quad,ntrial_,accuracy_,npts_,ierr)
    !write(*,'(i8,a,es14.6)')ig,' 3 ',quad
    if (ierr/=0) then 
     ABI_DIE("Accuracy not reached")
    end if
    ! === Store final result ===
    ! * Factor two comes from the replacement WS -> (1,4) quadrant thanks to symmetries of the integrad.
    tmp=tmp+quad
    vccut(ig,iq)=two*(tmp*two) 

   end do !ig
  end do !iq

 CASE (2) 
  ! === Finite cylinder of lenght hcyl, from Rozzi et al ===
  ! TODO add check on hcyl value that should be smaller that 1/deltaq 
  if (hcyl_<zero) then
   write(msg,'(a)')' Negative value for cylinder lenght '
   MSG_BUG(msg)
  end if

  if (ABS(hcyl_)>tol12) then 
   write(msg,'(2(a,f8.4))')' cutoff_cylinder: using finite cylinder of length= ',hcyl_,' rcut= ',rcut_
   call wrtout(std_out,msg,'COLL') 
   hcyl2=hcyl_**2 
   rcut2=rcut_**2

   do iq=1,nq
    write(msg,'(a,i3)')' entering loop iq: ',iq
    call wrtout(std_out,msg,'COLL')
    qc(:)=qcart(:,iq)
    do ig=1,ng
     ! === No singularity occurs in finite cylinder, thus start from 1 ===
     icount=ig+(iq-1)*ng ; if (icount<my_start.or.icount>my_stop) CYCLE
     gcart(:)=b1(:)*gvec(1,ig)+b2(:)*gvec(2,ig)+b3(:)*gvec(3,ig)
     qpg(:)=qc(:)+gcart(:)
     qpg_para_=ABS(qpg(3)) ; qpg_perp_=SQRT(qpg(1)**2+qpg(2)**2)

     if (qpg_perp_/=zero.and.qpg_para_/=zero) then 
      ! $ 4\pi\int_0^{R_c} d\rho\rho j_o(qpg_perp_.\rho)\int_0^hcyl dz\cos(qpg_para_*z)/sqrt(\rho^2+z^2) $ 
      call quadrature(F2,zero,rcut_,qopt_,quad,ntrial_,accuracy_,npts_,ierr)
      if (ierr/=0) then 
       ABI_DIE("Accuracy not reached")
      end if

      vccut(ig,iq)=four_pi*quad

     else if (qpg_perp_==zero.and.qpg_para_/=zero) then 
      ! $ \int_0^h sin(qpg_para_.z)/\sqrt(rcut^2+z^2)dz $
      call quadrature(F3,zero,hcyl_,qopt_,quad,ntrial_,accuracy_,npts_,ierr)
      if (ierr/=0) then 
       ABI_DIE("Accuracy not reached")
      end if

      c1=one/qpg_para_**2-COS(qpg_para_*hcyl_)/qpg_para_**2-hcyl_*SIN(qpg_para_*hcyl_)/qpg_para_
      c2=SIN(qpg_para_*hcyl_)*SQRT(hcyl2+rcut2)
      vccut(ig,iq)=four_pi*c1+four_pi*(c2-quad)/qpg_para_

     else if (qpg_perp_/=zero.and.qpg_para_==zero) then 
      ! $ 4pi\int_0^rcut d\rho \rho J_o(qpg_perp_.\rho) ln((h+\sqrt(h^2+\rho^2))/\rho) $
      call quadrature(F4,zero,rcut_,qopt_,quad,ntrial_,accuracy_,npts_,ierr)
      if (ierr/=0) then 
       ABI_DIE("Accuracy not reached")
      end if

      vccut(ig,iq)=four_pi*quad

     else if (qpg_perp_==zero.and.qpg_para_==zero) then 
      ! Use lim q+G --> 0
      vccut(ig,iq)=two_pi*(-hcyl2+hcyl_*SQRT(hcyl2+rcut2)+rcut2*LOG((hcyl_+SQRT(hcyl_+SQRT(hcyl2+rcut2)))/rcut_))

     else 
      MSG_BUG('You should not be here!')
     end if

    end do !ig
   end do !iq

  else 
   ! === Infinite cylinder ===
   write(msg,'(a)')' cutoff_cylinder : using Rozzi''s method with infinite cylinder '
   call wrtout(std_out,msg,'COLL') 
   do iq=1,nq
    write(msg,'(a,i3)')' entering loop iq: ',iq
    call wrtout(std_out,msg,'COLL')
    qc(:)=qcart(:,iq)
    do ig=1,ng
     icount=ig+(iq-1)*ng ; if (icount<my_start.or.icount>my_stop) CYCLE
     gcart(:)=b1(:)*gvec(1,ig)+b2(:)*gvec(2,ig)+b3(:)*gvec(3,ig)
     qpg(:)=qc(:)+gcart(:)
     qpg2  =DOT_PRODUCT(qpg,qpg)
     qpg_z =ABS(qpg(3)) ; qpg_xy=SQRT(qpg(1)**2+qpg(2)**2)
     if (qpg_z>SMALL) then 
      ! === Analytic expression ===
      call CALCK0(qpg_z *rcut_,k0,1)
      call CALJY1(qpg_xy*rcut_,j1,0)
      call CALJY0(qpg_xy*rcut_,j0,0)
      call CALCK1(qpg_z *rcut_,k1,1)
      vccut(iq,ig)=(four_pi/qpg2)*(one+rcut_*qpg_xy*j1*k0-qpg_z*rcut_*j0*k1) 
     else 
      if (qpg_xy>SMALL) then 
       ! === Integrate r*Jo(G_xy r)log(r) from 0 up to rcut_  ===
       call quadrature(F5,zero,rcut_,qopt_,quad,ntrial_,accuracy_,npts_,ierr)
       if (ierr/=0) then 
        ABI_DIE("Accuracy not reached")
       end if
       vccut(ig,iq)=-four_pi*quad
      else 
       ! === Analytic expression ===
       vccut(ig,iq)=-pi*rcut_**2*(two*LOG(rcut_)-one)
      end if 
     end if 
    end do !ig
   end do !iq 
  end if !finite/infinite

 CASE DEFAULT
  write(msg,'(a,i3)')' Wrong value for method: ',method 
  MSG_BUG(msg)
 END SELECT 
 !
 ! === Collect vccut on each node ===
 write(*,*)rank,' completed'
 call leave_test(MPI_enreg)
 call xsum_mpi(vccut(:,:),spaceComm,ierr)
 !write(*,*)MAXVAL(vccut),MINVAL(vccut)

 deallocate(qcart)

end subroutine cutoff_cylinder
!!***

!!****f* m_coulombian/cutoff_surface
!! NAME
!! cutoff_surface
!!
!! FUNCTION
!!  Calculate the Fourier components of an effective Coulombian interaction 
!!  within a slab of thickness 2*rcut which is symmetric with respect to the xy plane. 
!!  In this implementation rcut=L_z/2 where L_z is the periodicity along z 
!!
!! INPUTS
!!  gprimd(3,3)=Dimensional primitive translations in reciprocal space ($\textrm{bohr}^{-1}$).
!!  gvec(3,ng)=G vectors in reduced coordinates. 
!!  ng=Number of G vectors.
!!  qpt(3,nq)=q-points 
!!  nq=Number of q-points 
!!  gmet(3,3)=Metric in reciprocal space.
!!
!! OUTPUT
!!  vc_cut(ng,nq)=Fourier components of the effective coulombian interaction.
!!
!! NOTES
!!  The Fourier expression for an interaction truncated along the z-direction 
!!  (i.e non-zero only if |z|<R) is :
!!  
!!  vc(q.G) = 4pi/|q+G|^2 * [ 1 + e^{-((q+G)_xy)*R} * ( (q_z+G_z)/(q+G)_xy * sin((q_z+G_z)R) - 
!!   - cos((q_z+G_Z)R)) ]  (1)
!! 
!!  Equation (1) diverges when q_xy+G_xy --> 0 for any non zero q_z+G_z
!!  However if we choose R=L/2, where L defines the periodicity along z, 
!!  and we limit ourselves to consider q-points such as q_z==0, then 
!!  sin((q_z+G_z)R)=sin(G_Z 2pi/L)=0 for every G.  
!!  Under these assumptions we obtain
!!
!!  v(q,G) = 4pi/|q+G|^2} [ 1-e^{-(q+G)_xy*L/2}\cos((q_z+G_z)R) ]
!! 
!!  which is always finite when G_z /=0 while it diverges as 4piR/(q+G)_xy as (q+G)_xy -->0 
!!  but only in the x-y plane.
!!
!! PARENTS
!!      setup_coulombian
!!
!! CHILDREN
!!      wrtout
!!
!! SOURCE

subroutine cutoff_surface(nq,qpt,ng,gvec,gprimd,gmet,rcut,boxcenter,pdir,alpha,vc_cut,method)

 use defs_basis

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: method,ng,nq
 real(dp),intent(in) :: rcut
!arrays
 integer,intent(in) :: gvec(3,ng),pdir(3)
 real(dp),intent(in) :: alpha(3),boxcenter(3),gmet(3,3),gprimd(3,3),qpt(3,nq)
 real(dp),intent(out) :: vc_cut(ng,nq)

!Local variables-------------------------------
!scalars
 integer :: ig,igs,iq
 real(dp),parameter :: SMALL=tol6
 real(dp) :: ap1sqrt,qpg2,qpg_para,qpg_perp
 character(len=500) :: msg
!arrays
 real(dp) :: b1(3),b2(3),b3(3),gcart(3),qc(3),qpg(3)
 real(dp),allocatable :: qcart(:,:)

! *************************************************************************

 ! === From reduced to cartesian coordinates ===
 b1(:)=two_pi*gprimd(:,1)
 b2(:)=two_pi*gprimd(:,2)
 b3(:)=two_pi*gprimd(:,3)
 allocate(qcart(3,nq))
 do iq=1,nq
  qcart(:,iq) = b1*qpt(1,iq) + b2*qpt(2,iq) + b3*qpt(3,iq)
 end do
 !
 ! === Different approaches according to method ===
 vc_cut(:,:)=zero 

 SELECT CASE (method)

 CASE (1)
  ! === Beigi expression ===
  ! * q-points with non-zero component along the z-axis are not allowed if 
  !   the simplified Eq.1 for the Coulombian interaction is used.
  if (ANY(ABS(qcart(3,:))>SMALL)) then 
   write(*,*)qcart(:,:) 
   write(msg,'(5a)')&
&   ' Found q-points with non-zero component along non-periodic direction ',ch10,&
&   ' This is not allowed, see Notes in cutoff_surface.F90 ',ch10,&
&   ' ACTION : Modify the q-point sampling '
   MSG_ERROR(msg)
  end if 
  ! 
  ! === Calculate truncated coulombian interaction for a infinite surface ===
  ! * Here I suppose that all the input q-points are different from zero
  do iq=1,nq
   qc(:)=qcart(:,iq)
   do ig=1,ng
    gcart(:)=b1(:)*gvec(1,ig)+b2(:)*gvec(2,ig)+b3(:)*gvec(3,ig)
    qpg(:)=qc(:)+gcart(:)
    qpg2  =DOT_PRODUCT(qpg(:),qpg(:))
    qpg_para=SQRT(qpg(1)**2+qpg(2)**2) ; qpg_perp=qpg(3)
    ! if (abs(qpg_perp)<SMALL.and.qpg_para<SMALL) stop 'SMALL in cutoff_surface
    vc_cut(ig,iq)=four_pi/qpg2*(one-EXP(-qpg_para*rcut)*COS(qpg_perp*rcut))  
   end do 
  end do

 CASE (2)
  ! === Rozzi et al method ===
  !alpha=?? ; ap1sqrt=SQRT(one+alpha**2)
  ABI_DIE("working in progress")
  do iq=1,nq
   qc(:)=qcart(:,iq)
   do ig=1,ng
    gcart(:)=b1(:)*gvec(1,ig)+b2(:)*gvec(2,ig)+b3(:)*gvec(3,ig)
    qpg(:)=qc(:)+gcart(:)
    qpg2  =DOT_PRODUCT(qpg(:),qpg(:))
    qpg_para=SQRT(qpg(1)**2+qpg(2)**2) ; qpg_perp =qpg(3)
    if (qpg_para>SMALL) then 
     vc_cut(ig,iq)=four_pi/qpg2*(one+EXP(-qpg_para*rcut)*(qpg_perp/qpg_para*SIN(qpg_perp*rcut)-COS(qpg_perp*rcut))) 
    else 
     if (ABS(qpg_perp)>SMALL) then 
      vc_cut(ig,iq)=four_pi/qpg_perp**2*(one-COS(qpg_perp*rcut)-qpg_perp*rcut*SIN(qpg_perp*rcut)) !&
!&      + 8*rcut*SIN(qpg_perp*rcut)/qpg_perp*LOG((alpha+ap1sqrt)*(one+ap1sqrt)/alpha) ! contribution due to finite surface
     else 
      vc_cut(ig,iq)=-two_pi*rcut**2
     end if 
    end if 
   end do !ig 
  end do !iq

 CASE DEFAULT 
  write(msg,'(a,i3)')' Wrong value of method: ',method 
  ABI_DIE(msg)
 END SELECT

 deallocate(qcart)

end subroutine cutoff_surface
!!***

!!****f* m_coulombian/cvc
!! NAME
!! cvc
!!
!! FUNCTION
!! Set up table of lengths |q+G| for the Coulombian potential.
!!
!! INPUTS
!! gvec(3,npwvec)=Reduced coordinates of the G vectors.
!! gprimd(3,3)=Dimensional primitive translations for reciprocal space ($\textrm{bohr}^{-1}$)
!! iq=Index specifying the q point.
!! npwvec=Number of planewaves
!! nq=Number of q points.
!! q=Coordinates of q points.
!!
!! OUTPUT
!! qplusg(npwvec)=Norm of q+G vector
!!
!! PARENTS
!!      calc_ffm,cppm2par,cppm3par,cppm4par,eps1_tc,setup_coulombian
!!
!! CHILDREN
!!
!! SOURCE

subroutine cvc(nq,iq,q,npwvec,gvec,gprimd,qplusg)

 use defs_basis

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: iq,npwvec,nq
!arrays
 integer,intent(in) :: gvec(3,npwvec)
 real(dp),intent(in) :: gprimd(3,3),q(3,nq)
 real(dp),intent(out) :: qplusg(npwvec)

!Local variables ------------------------------
!scalars
 integer :: ig,ii
!arrays
 real(dp) :: gmet(3,3),gpq(3)

!************************************************************************

 ! Compute reciprocal space metrics
 do ii=1,3
  gmet(ii,:)=gprimd(1,ii)*gprimd(1,:)+&
&            gprimd(2,ii)*gprimd(2,:)+&
&            gprimd(3,ii)*gprimd(3,:)
 end do

 if (ALL(ABS(q(:,iq))<1.e-3)) then
  qplusg(1)=two_pi*SQRT(DOT_PRODUCT(q(:,iq),MATMUL(gmet,q(:,iq))))
  do ig=2,npwvec
   gpq(:)=gvec(:,ig)
   qplusg(ig)=two_pi*SQRT(DOT_PRODUCT(gpq,MATMUL(gmet,gpq)))
  end do
 else
  do ig=1,npwvec
   gpq(:)=gvec(:,ig)+q(:,iq)
   qplusg(ig)=two_pi*SQRT(DOT_PRODUCT(gpq,MATMUL(gmet,gpq)))
  end do
 end if

end subroutine cvc

END MODULE m_coulombian
!!***
