!{\src2tex{textfont=tt}}
!!****m* ABINIT/m_bz_mesh
!! NAME
!!  m_bz_mesh
!!
!! FUNCTION
!!  This module provides the definition of the BZ_mesh_type structure gathering information 
!!  on the sampling of the Brillouin zone. It also contains useful tools to operate on k-points.
!!
!! COPYRIGHT
!! Copyright (C) 2008 ABINIT group (MG)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~abinit/COPYING
!! or http://www.gnu.org/copyleft/gpl.txt .
!!
!! INPUTS
!!
!! OUTPUT
!!
!! PARENTS
!!
!! CHILDREN
!!
!! SOURCE

#if defined HAVE_CONFIG_H
#include "config.h"
#endif

#include "abi_common.h"

MODULE m_bz_mesh

 use defs_basis
 use m_errors 

 implicit none

 private  

 public ::              &
  InitKmesh,            &  ! Main creation method.
  NullifyBzMesh,        &  ! Nullify all pointers in BZ_mesh_type.
  destroy_bz_mesh_type, &  ! Free the structure.
  print_bz_mesh,        &  ! Printout of basic info on the object.
  get_BZ_item,          &  ! Get point in the  BZ as well as useful quantities.
  get_IBZ_item,         &  ! Get point in the IBZ as well as useful quantities. 
  get_BZ_diff,          &  ! Get the difference k1-k2 in the BZ (if any).
  isamek,               &  ! Check whether two points are equal within an umklapp G0.
  has_BZ_item,          &  ! Chesk if a point belongs to the BZ mesh.
  has_IBZ_item,         &  ! Check if a point is in the IBZ
  make_mesh,            &  ! Initialize the mesh starting from kptrlatt and shift (WARNING still under development)
  findk,                &  ! TODO to be removed
  identk                   ! Find BZ starting from the irreducible k-points.

CONTAINS  !===========================================================
!!***

!!****f* m_bz_mesh/InitKmesh
!! NAME
!! InitKmesh
!!
!! FUNCTION
!!  Initialize and construct a bz_mesh_type datatype
!!  gathering information on the mesh in the Brilloin zone. 
!!
!! INPUTS
!!  nkibz=number of irreducible k-points
!!  prtvol=verbosity level
!!  timrev=1 if time-reversal cannot be used, 2 otherwise
!!  kibz(3,nkibz)=irreducible k-points
!!  Cryst<Crystal_structure> = Info on unit cell and its symmetries
!!     %nsym=number of symmetry operations
!!     %symrec(3,3,nsym)=symmetry operations in reciprocal space
!!     %tnons(3,nsym)=fractional translations
!!
!! OUTPUT
!!  Kmesh<bz_mesh_type>=Datatype gathering information on the k point sampling.
!!
!! NOTES
!!
!! PARENTS
!!      gw_tools,joint_dos,mlwfovlp_qp,mrgscr,setup_screening,setup_sigma
!!
!! CHILDREN
!!      get_full_kgrid,get_tetra,matr3inv,pclock,smpbz,symkpt
!!
!! SOURCE

subroutine InitKmesh(nkibz,kibz,Cryst,Kmesh,prtvol)

 use defs_basis
 use defs_datatypes
 use m_io_tools, only : flush_unit

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_12geometry
!End of the abilint section

 implicit none

!Arguments ------------------------------------
! scalars
! arrays
!scalars
 integer,intent(in) :: nkibz 
 integer,optional,intent(in) :: prtvol
 type(BZ_mesh_type),intent(inout) :: Kmesh
 type(Crystal_structure),intent(in) :: Cryst
!arrays
 real(dp),intent(in) :: kibz(3,nkibz)

!Local variables-------------------------------
!scalars
 integer :: ierr,ik_bz,ik_ibz,isym,nkbz,nkbzX,nsym,timrev,my_prtvol
 real(dp) :: shift1,shift2,shift3,ucvol
 logical :: ltest,use_antiferro
 character(len=500) :: msg
!arrays
 integer,allocatable :: ktab(:),ktabi(:),ktabo(:),ktabr(:,:)
 integer,pointer :: symafm(:),symrec(:,:,:)
 real(dp) :: gmet(3,3),gprimd(3,3),rm1t(3),rmet(3,3)
 real(dp),allocatable :: kbz(:,:),kbz_wrap(:,:),wtk(:)
 real(dp),pointer :: tnons(:,:)

! *************************************************************************

 DBG_ENTER("COLL")

 my_prtvol=0 ; if (PRESENT(prtvol)) my_prtvol=prtvol

 call NullifyBzMesh(Kmesh)

 ! === Initial tests on input arguments ===
 ltest=(Cryst%timrev==1.or.Cryst%timrev==2)
 write(msg,'(a,i4)')'Wrong value for timrev= ',Cryst%timrev
 call assert(ltest,msg,__FILE__,__LINE__)

 ! === Get symmetries related stuff ==
 nsym   = Cryst%nsym
 timrev = Cryst%timrev
 use_antiferro = Cryst%use_antiferro
 symrec => Cryst%symrec
 symafm => Cryst%symafm
 tnons  => Cryst%tnons

 call metric(gmet,gprimd,-1,rmet,Cryst%rprimd,ucvol)
 Kmesh%gmet   = gmet
 Kmesh%gprimd = gprimd
 !
 ! === Find BZ from IBZ and fill tables ===
 nkbzX=nkibz*nsym*timrev ! Maximum possible number
 allocate(kbz(3,nkbzX),wtk(nkibz),ktab(nkbzX),ktabi(nkbzX),ktabo(nkbzX))

 call identk(kibz,nkibz,nkbzX,nsym,timrev,symrec,symafm,use_antiferro,kbz,ktab,ktabi,ktabo,nkbz,wtk,my_prtvol)
 !
 ! Wrap the BZ points in the interval ]-1/2,1/2]
 !allocate(kbz_wrap(3,nkbz))
 !do ik_bz=1,nkbz
 ! call canon9(kbz(1,ik_bz),kbz_wrap(1,ik_bz),shift1)
 ! call canon9(kbz(2,ik_bz),kbz_wrap(2,ik_bz),shift2)
 ! call canon9(kbz(3,ik_bz),kbz_wrap(3,ik_bz),shift3)
 !end do
 !
 ! ==== Create data structure to store information on k-points ====
 !
 ! === Dimensions ===
 Kmesh%nbz   =nkbz      ! Number of points in the full BZ
 Kmesh%nibz  =nkibz     ! Number of points in the IBZ
 Kmesh%nsym  =nsym      ! Number of operations
 Kmesh%timrev=timrev    ! 2 if time-reversal is used, 1 otherwise

 Kmesh%ntetra_irr=0     ! no tetrahedrons for the moment
 !
 ! === Arrays ===
 allocate(Kmesh%bz(3,nkbz))   ;  kmesh%bz(:,:)=  kbz(:,1:nkbz)  ! Red. coordinates of points in full BZ.
 allocate(Kmesh%ibz(3,nkibz)) ; kmesh%ibz(:,:)=kibz(:,1:nkibz)  ! Red. coordinates of points in IBZ.
 !allocate(Kmesh%bz(3,nkbz))   ; kmesh%bz(:,:)=kbz_wrap(:,1:nkbz)

 allocate(Kmesh%tab(nkbz))    ; Kmesh%tab(:)=ktab (1:nkbz)      ! Index of the irred. point in the array IBZ.
 allocate(Kmesh%tabi(nkbz))   ; Kmesh%tabi(:)=ktabi(1:nkbz)     !-1 if time reversal must be used to obtain this point,
                                                                ! 1 otherwise
 allocate(Kmesh%tabo(nkbz))   ; Kmesh%tabo(:)=ktabo(1:nkbz)     ! Symm. operation that rotates k_IBZ onto \pm k_BZ
                                                                ! (depending on tabi)
 allocate(Kmesh%wt(nkibz))    ; Kmesh%wt(:)=wtk(1:nkibz)        ! Weight for each k_IBZ

 allocate(Kmesh%rottbm1(nkbz,timrev,nsym))                      ! Index of rotated point (IS)^{-1} kbz
 allocate(Kmesh%rottb  (nkbz,timrev,nsym))                      ! Index of rotated point IS kbz where I is either the
                                                                ! identity or time-reversal

 call setup_k_rotation(nsym,symrec,timrev,nkbz,Kmesh%bz,Kmesh%rottb,Kmesh%rottbm1)

 allocate(Kmesh%tabp(nkbz)) ! Phase factors for non-symmorphic operations $e{-i2\pik_BZ.\tau}$
 do ik_bz=1,nkbz
  isym=Kmesh%tabo(ik_bz) ; ik_ibz=Kmesh%tab(ik_bz)
  rm1t=MATMUL(TRANSPOSE(symrec(:,:,isym)),tnons(:,isym))
  Kmesh%tabp(ik_bz)=EXP(-(0.,1.)*two_pi*DOT_PRODUCT(kibz(:,ik_ibz),rm1t))
 end do

 deallocate(kbz,wtk,ktab,ktabi,ktabo)
 !deallocate(kbz_wrap)

 call print_bz_mesh(Kmesh,prtvol=my_prtvol)

 DBG_EXIT("COLL")

end subroutine InitKmesh
!!***

!!****f* m_bz_mesh/NullifyBzMesh
!! NAME
!! NullifyBzMesh
!!
!! FUNCTION
!! Nullify the pointers in a BZ_mesh_type. [PRIVATE]
!!
!! INPUTS
!! Kmesh<BZ_mesh_type>= The datatype whose pointers have to be nullified.
!!
!! SIDE EFFECTS
!! All pointers set to null().
!!
!! PARENTS
!!      gw_tools,setup_kmesh,setup_qmesh
!!
!! CHILDREN
!!      get_full_kgrid,get_tetra,matr3inv,pclock,smpbz,symkpt
!!
!! SOURCE

subroutine NullifyBzMesh(Kmesh)

 use defs_basis
 use defs_datatypes

 implicit none

!Arguments ------------------------------------
!scalars
 type(BZ_mesh_type),intent(inout) :: Kmesh

! *********************************************************************

!BEGIN NEW
 nullify(Kmesh%shift     )
 nullify(Kmesh%tetra_full)
 nullify(Kmesh%tetra_mult)
 nullify(Kmesh%tetra_wrap)
!END NEW

!integer
 nullify(Kmesh%rottb  )
 nullify(Kmesh%rottbm1)
 nullify(Kmesh%tab    )
 nullify(Kmesh%tabi   )
 nullify(Kmesh%tabo   )

!real
 nullify(Kmesh%ibz    )
 nullify(Kmesh%bz     )
 nullify(Kmesh%wt     )

!complex
 nullify(Kmesh%tabp   )

end subroutine NullifyBzMesh
!!***

!!****f* m_bz_mesh/destroy_bz_mesh_type
!! NAME
!! destroy_bz_mesh_type
!!
!! FUNCTION
!! Deallocate all dynamics entities present in a BZ_mesh_type structure.
!!
!! INPUTS
!! Kmesh<BZ_mesh_type>=The datatype to be freed.
!!
!! SIDE EFFECTS
!! All allocated memory is released. 
!!
!! PARENTS
!!      gw_tools,joint_dos,mlwfovlp_qp,mrgscr,screening,sigma
!!
!! CHILDREN
!!      get_full_kgrid,get_tetra,matr3inv,pclock,smpbz,symkpt
!!
!! SOURCE

subroutine destroy_bz_mesh_type(Kmesh)

 use defs_basis
 use defs_datatypes

 implicit none

!Arguments ------------------------------------
!scalars
 type(BZ_mesh_type),intent(inout) :: Kmesh

!Local variables-------------------------------
!scalars
 integer :: istat

! *********************************************************************

 DBG_ENTER("COLL")

!BEGIN NEW
 if (associated(Kmesh%shift     )) deallocate(Kmesh%shift     )
 if (associated(Kmesh%tetra_full)) deallocate(Kmesh%tetra_full)
 if (associated(Kmesh%tetra_mult)) deallocate(Kmesh%tetra_mult)
 if (associated(Kmesh%tetra_wrap)) deallocate(Kmesh%tetra_wrap)
!END NEW

!integer
 if (associated(Kmesh%rottb  )) deallocate(Kmesh%rottb  )
 if (associated(Kmesh%rottbm1)) deallocate(Kmesh%rottbm1)
 if (associated(Kmesh%tab    )) deallocate(Kmesh%tab    )
 if (associated(Kmesh%tabi   )) deallocate(Kmesh%tabi   )
 if (associated(Kmesh%tabo   )) deallocate(Kmesh%tabo   )

!real
 if (associated(Kmesh%ibz    )) deallocate(Kmesh%ibz    )
 if (associated(Kmesh%bz     )) deallocate(Kmesh%bz     )
 if (associated(Kmesh%wt     )) deallocate(Kmesh%wt     )

!complex
 if (associated(Kmesh%tabp   )) deallocate(Kmesh%tabp   )

 DBG_EXIT("COLL")

end subroutine destroy_bz_mesh_type
!!***

!!****f* m_bz_mesh/print_bz_mesh
!! NAME
!! print_bz_mesh
!!
!! FUNCTION
!! Print the content of a bz_mesh_type datatype
!!
!! INPUTS
!! Kmesh<bz_mesh_type>=the datatype to be printed
!! unit[optional]=the unit number for output 
!! prtvol[optional]=verbosity level
!! mode_paral[optional]=either "COLL" or "PERS"
!!
!! OUTPUT
!!  Only printing.
!!
!! PARENTS
!!      setup_kmesh
!!
!! CHILDREN
!!      get_full_kgrid,get_tetra,matr3inv,pclock,smpbz,symkpt
!!
!! SOURCE

subroutine print_BZ_mesh(Kmesh,unit,prtvol,mode_paral)

 use defs_basis
 use defs_datatypes

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_01manage_mpi
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in),optional :: prtvol,unit
 character(len=4),intent(in),optional :: mode_paral
 type(BZ_mesh_type),intent(in) :: Kmesh

!Local variables-------------------------------
!scalars
 integer,parameter :: nmaxk=50
 integer :: ii,ik,unt,verbose
 character(len=100) :: fmt
 character(len=4) :: mode
 character(len=500) :: msg

! *************************************************************************

 unt    =std_out ; if (PRESENT(unit      )) unt    =unit
 verbose=0       ; if (PRESENT(prtvol    )) verbose=prtvol
 mode   ='COLL'  ; if (PRESENT(mode_paral)) mode   =mode_paral

 write(msg,'(2a,i5,3a)')ch10,&
& ' Number of points in the IBZ : ',Kmesh%nibz,ch10,&
& ' Reduced Coordinates and Weights : ',ch10
 call wrtout(unt,msg,mode)

 write(fmt,*)'(1x,i5,a,2x,3es16.8,3x,f11.5)'
 do ik=1,Kmesh%nibz
  write(msg,fmt) ik,') ',(Kmesh%ibz(ii,ik),ii=1,3),Kmesh%wt(ik)
  call wrtout(unt,msg,mode)
 end do

 select case (Kmesh%timrev)

 case (1)
  write(msg,'(2a,i2,3a,i5,a)')ch10,&
&  ' together with ',Kmesh%nsym,' symmetry operations (time-reversal not used) ',ch10,&
&  ' have yielded ',Kmesh%nbz,' k-points in the Brillouin Zone (BZ) :'

 case (2) 
  write(msg,'(2a,i2,3a,i5,a)')ch10,&
&  ' together with ',Kmesh%nsym,' symmetry operations and time-reversal ',ch10,&
&  ' have yielded ',Kmesh%nbz,' k-points in the Brillouin Zone (BZ) :'

 case default
  write(msg,'(a,i3)')' Wrong value for timrev = ',Kmesh%timrev
  MSG_BUG(msg)
 end select

 call wrtout(unt,msg,mode)

 if (verbose>0) then
  write(fmt,*)'(1x,i5,a,2x,3es16.8)'
  do ik=1,Kmesh%nbz
   if (verbose==1 .and. ik>nmaxk) then
    write(msg,'(a)')' prtvol=1, do not print more k-points.'
    call wrtout(unt,msg,mode) ; EXIT
   end if
   write(msg,fmt)ik,') ',(Kmesh%bz(ii,ik),ii=1,3)
   call wrtout(unt,msg,mode)
  end do
 end if
 !
 ! === Additional printing ===
 if (verbose>=10) then
  write(msg,'(2a)')ch10,&
&  '                  Irred point -->              Full-point           through:  Symrec  Time-Rev (1=No,-1=Yes) '
  call wrtout(unt,msg,mode)
  write(fmt,*)'(2x,i5,2x,2(3es16.8,2x),i3,2x,i2)'
  do ik=1,Kmesh%nbz
   write(msg,fmt)ik,Kmesh%bz(:,ik),Kmesh%ibz(:,Kmesh%tab(ik)),Kmesh%tabo(ik),Kmesh%tabi(ik)
   call wrtout(unt,msg,mode)
  end do
 end if

end subroutine print_BZ_mesh
!!***

!!****f* m_bz_mesh/setup_k_rotation
!! NAME
!! setup_k_rotation
!!
!! FUNCTION
!! Set up tables giving the correspondence btw a k-point and its rotated image. 
!!
!! INPUTS
!! kbz(3,nbz)=k-points in reduced coordinates.
!! timrev=2 if time-reversal can be used, 1 otherwise.
!! nsym=Number of symmetry operations
!! nbz=Number of k-points
!! symrec(3,3,nsym)=Symmetry operations in reciprocal space in reduced coordinates.
!!
!! OUTPUT
!! krottb(k,I,S)=Index of (IS) k in the array bz
!! krottbm1(k,I,S)=Index of IS^{-1} k
!!
!! PARENTS
!!      setup_kmesh
!!
!! CHILDREN
!!      get_full_kgrid,get_tetra,matr3inv,pclock,smpbz,symkpt
!!
!! SOURCE

subroutine setup_k_rotation(nsym,symrec,timrev,nbz,kbz,krottb,krottbm1)

 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: nbz,nsym,timrev
!arrays
 integer,intent(in) :: symrec(3,3,nsym)
 integer,intent(inout) :: krottb(nbz,timrev,nsym),krottbm1(nbz,timrev,nsym)
 real(dp),intent(in) :: kbz(3,nbz)

!Local variables ------------------------------
!scalars
 integer :: ik,ikp,isym,itim
 logical :: found,isok
 character(len=500) :: msg
!arrays
 integer :: G0(3)
 real(dp) :: kbase(3),krot(3)

!************************************************************************

 DBG_ENTER("COLL")
 !
 ! === Set up k-rotation tables ===
 ! * Use spatial inversion instead of time reversal whenever possible.
 isok=.TRUE.
 do ik=1,nbz
  kbase(:)=kbz(:,ik)
  do itim=1,timrev
   do isym=1,nsym
    krot(:)=(3-2*itim)*MATMUL(symrec(:,:,isym),kbase)
    found=.FALSE.
    do ikp=1,nbz
     if (isamek(krot,kbz(:,ikp),G0)) then
      found=.TRUE.
      krottb  (ik ,itim,isym)=ikp
      krottbm1(ikp,itim,isym)=ik
     end if
    end do

    if (.not.found) then
     isok=.FALSE.
     write(msg,'(2(a,i4),2x,2(3f12.6,2a),i3,a,i2)')&
&     ' Initial k-point ',ik,'/',nbz,kbase(:),ch10,&
&     ' Rotated k-point (not found) ',krot(:),ch10,&
      ' Through sym. operation ',isym,' and itim ',itim
     MSG_WARNING(msg)
    end if

   end do
  end do
 end do

 if (.not.isok) then
  MSG_ERROR('k-mesh not closed')
 end if

 DBG_EXIT("COLL")

end subroutine setup_k_rotation
!!***

!!****f* m_bz_mesh/get_BZ_item
!! NAME
!! get_BZ_item
!!
!! FUNCTION
!! Given the index of a point in the full BZ, report useful information.
!!
!! INPUTS
!! ikbz=The index of the required point in the BZ
!! Kmesh<BZ_mesh_type>=Datatype gathering information on the k point sampling. see defs_datatypes.F90
!!
!! OUTPUT
!! kbz(3)=the k-point in reduced coordinated
!! isym=index of the symmetry required to rotate ik_ibz onto ik_bz
!! itim=2 is time-reversal has to be used, 1 otherwise
!! ik_ibz=the index of corresponding point in the IBZ
!! ph_mkbzt=the phase factor for non-symmorphic operations 
!!  i.e e^{-i 2 \pi k_IBZ \cdot R{^-1}t}=e{-i 2\pi k_BZ cdot t}
!!
!! PARENTS
!!      cchi0,cchi0q0,csigme,joint_dos,paw_symcprj,setup_coulombian,solve_Dyson
!!      wf_info
!!
!! CHILDREN
!!      get_full_kgrid,get_tetra,matr3inv,pclock,smpbz,symkpt
!!
!! SOURCE

subroutine get_BZ_item(Kmesh,ik_bz,kbz,ik_ibz,isym,itim,ph_mkbzt)

 use defs_basis
 use defs_datatypes

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: ik_bz
 integer,intent(out) :: ik_ibz,isym,itim
 complex(gwpc),intent(out),optional :: ph_mkbzt
 type(BZ_mesh_type),intent(in) :: Kmesh
!arrays
 real(dp),intent(out) :: kbz(3)

!Local variables-------------------------------
!scalars
 character(len=500) :: msg

! *********************************************************************

 if (ik_bz>Kmesh%nbz.or.ik_bz<=0) then
  write(msg,'(a,i3)')' Wrong value for ik_bz: ',ik_bz
  MSG_BUG(msg)
 end if

 kbz(:)=Kmesh%bz(:,ik_bz)

 ik_ibz=Kmesh%tab(ik_bz)
 isym=Kmesh%tabo(ik_bz)
 itim=(3-Kmesh%tabi(ik_bz))/2
 if (PRESENT(ph_mkbzt)) ph_mkbzt=Kmesh%tabp(ik_bz)

end subroutine get_BZ_item
!!***

!!****f* m_bz_mesh/get_IBZ_item
!! NAME
!! get_IBZ_item
!!
!! FUNCTION
!! Report useful information on a k-point in the IBZ starting from its senquential index in %ibz. 
!!
!! INPUTS
!! ik_bz=The index of the required point in the IBZ
!! Kmesh<bz_mesh_type>=datatype gathering information on the k point sampling. see defs_datatypes.F90
!!
!! OUTPUT
!! kibz(3)=the k-point in reduced coordinated
!! wtk=the weight
!!
!! TODO
!!  Add mapping ibz2bz, ibz2star
!!
!! PARENTS
!!      paw_symcprj
!!
!! CHILDREN
!!      get_full_kgrid,get_tetra,matr3inv,pclock,smpbz,symkpt
!!
!! SOURCE

subroutine get_IBZ_item(Kmesh,ik_ibz,kibz,wtk)

 use defs_basis
 use defs_datatypes

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: ik_ibz
 real(dp),intent(out) :: wtk
 type(bz_mesh_type),intent(in) :: Kmesh
!arrays
 real(dp),intent(out) :: kibz(3)

!Local variables-------------------------------
!scalars
 character(len=500) :: msg

! *********************************************************************

 if (ik_ibz>Kmesh%nibz.or.ik_ibz<=0) then
  write(msg,'(a,i3)')' wrong value for ik_ibz: ',ik_ibz
  MSG_BUG(msg)
 end if

 kibz=Kmesh%ibz(:,ik_ibz)
 wtk =Kmesh%wt(ik_ibz)

end subroutine get_IBZ_item
!!***

!!****f* m_bz_mesh/get_BZ_diff
!! NAME
!! get_BZ_diff
!!
!! FUNCTION
!! Given two points k1 and k2 where k1 belongs to the BZ, check if the difference
!! k1-k2 still belongs to the BZ reporting useful quantities
!!
!! INPUTS
!!  Kmesh<bz_mesh_type>=datatype gathering information on the k-mesh
!!  k1(3)=the first k-points (supposed to be in the BZ)
!!  k2(3)=the second point
!!
!! OUTPUT
!!  idiff_bz=the idex of k1-k2 in the BZ
!!  G0(3)=the umklapp G0 vector required to bring k1-k2 back to the BZ
!!  nfound= the number of points in the BZ that are equal to k1-k2 (should be 1 if everything is OK)
!!
!! PARENTS
!!      cchi0,joint_dos
!!
!! CHILDREN
!!      get_full_kgrid,get_tetra,matr3inv,pclock,smpbz,symkpt
!!
!! SOURCE

subroutine get_BZ_diff(Kmesh,k1,k2,idiff_bz,G0,nfound)

 use defs_basis
 use defs_datatypes

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_01manage_mpi
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(out) :: idiff_bz,nfound
 type(bz_mesh_type),intent(in) :: Kmesh
!arrays
 integer,intent(out) :: G0(3)
 real(dp),intent(in) :: k1(3),k2(3)

!Local variables-------------------------------
!scalars
 integer :: ikp
 character(len=500) :: msg
!arrays
 integer :: umklp(3)
 real(dp) :: kdiff(3),ktrial(3)

! *********************************************************************

 if (.not.has_BZ_item(Kmesh,k1,ikp,umklp)) then
  write(msg,'(a,3f12.6)')' first point must be in BZ ',k1(:)
  MSG_ERROR(msg)
 end if

 kdiff(:)=k1 - k2
 nfound  =0 
 idiff_bz=0
 !
 ! === Find p such k1-k2=p+G0 where p in the BZ ===
 do ikp=1,Kmesh%nbz
  ktrial=Kmesh%bz(:,ikp)
  if (isamek(kdiff,ktrial,umklp)) then
   idiff_bz=ikp
   G0=umklp
   nfound=nfound+1
  end if
 end do
 !
 ! === Check if p has not found of found more than once ===
 ! * For extremely dense meshes, tol1q in defs_basis might be too large!
 if (nfound/=1) then
  if (nfound==0) then
   write(msg,'(a)')' k1-k2-G0 not found in BZ '
   MSG_WARNING(msg)
  else
   write(msg,'(a,i3)')' Multiple k1-k2-G0 found in BZ, nfound= ',nfound
   MSG_COMMENT(msg)
  end if
  write(msg,'(5a,3(a,3f12.6,a))')ch10,&
&  ' k1    = ',k1(:),ch10,&
&  ' k2    = ',k2(:),ch10,&
&  ' k1-k2 = ',kdiff(:),ch10
  call wrtout(std_out,msg,'COLL') !; call leave_new('COLL')
 end if

end subroutine get_BZ_diff
!!***

!!****f* m_bz_mesh/isamek
!! NAME
!! isamek
!!
!! FUNCTION
!! Test two k-points for equality. 
!! Return .TRUE. is they are equal within a reciprocal lattice vector G0.
!!
!! INPUTS
!!  k1(3),k2(3)=The two k points to be compared.
!!
!! OUTPUT
!! Return .TRUE. if they are the same within a RL vector,
!!        .FALSE. if they are different.
!! G0(3)=if .TRUE. G0(3) is the reciprocal lattice vector such as k1=k2+G0
!!
!! PARENTS
!!
!! CHILDREN
!!
!! SOURCE

logical function isamek(k1,k2,G0)

 use defs_basis
 use m_gwdefs, only : GW_TOLQ

 implicit none

!Arguments ------------------------------------
!arrays
 integer,intent(out) :: G0(3)
 real(dp),intent(in) :: k1(3),k2(3)

!Local variables-------------------------------
!scalars
 real(dp) :: f,x

! *************************************************************************

 ! === Statement function definition: f is zero only if x is integer ===
 f(x)=ABS(x-NINT(x))

 isamek=.FALSE. ; G0(:)=0
 if (f(k1(1)-k2(1))<GW_TOLQ) then
  if (f(k1(2)-k2(2))<GW_TOLQ) then
   if (f(k1(3)-k2(3))<GW_TOLQ) then
    isamek=.TRUE.
    G0(:)=NINT(k1(:)-k2(:))
   end if
  end if
 end if

end function isamek
!!***

!!****f* m_bz_mesh/has_BZ_item
!! NAME
!! has_BZ_item
!!
!! FUNCTION
!!  check if item belongs to the BZ  within a reciprocal lattice vector
!!  and return the index number and the reciprocal vector g0.
!!
!! INPUTS
!!  Kmesh<bz_mesh_type>=datatype gathering information on the k-mesh
!!  item(3)=the k-point to be checked
!!
!! OUTPUT
!!  .TRUE. if item is the BZ within a RL vector
!!  ikbz=Index of the k-point in the Kmesh%bz array
!!  G0(3)=Umklapp vector.
!!
!! PARENTS
!!
!! CHILDREN
!!
!! SOURCE

logical function has_BZ_item(Kmesh,item,ikbz,G0)

 use defs_basis
 use defs_datatypes

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
!End of the abilint section

 implicit none

!Arguments ------------------------------------
! scalars
! arrays
!scalars
 integer,intent(out) :: ikbz
 type(BZ_mesh_type),intent(in) :: Kmesh
!arrays
 integer,intent(out) :: G0(3)
 real(dp),intent(in) :: item(3)

!Local variables-------------------------------
!scalars
 integer :: ik_bz,yetfound
 character(len=500) :: msg

! *************************************************************************

 has_BZ_item=.FALSE. ; ikbz=0 ; G0(:)=0 ; yetfound=0
 do ik_bz=1,Kmesh%nbz
  if (isamek(item,Kmesh%bz(:,ik_bz),G0)) then
   has_BZ_item=.TRUE.
   ikbz=ik_bz
   yetfound=yetfound+1
   !EXIT
  end if
 end do

 if (yetfound/=0.and.yetfound/=1) then
  write(msg,'(a)')' multiple k-points found '
  MSG_BUG(msg)
 end if

end function has_BZ_item
!!***

!!****f* m_bz_mesh/has_IBZ_item
!! NAME
!! has_IBZ_item
!!
!! FUNCTION
!!  Check if item belongs to the IBZ within a reciprocal lattice vector
!!
!! INPUTS
!!  Kmesh<bz_mesh_type>=Datatype gathering information on the mesh in the BZ.
!!  item(3)=the k-point to be checked
!!
!! OUTPUT
!!  Return .TRUE. if item is the IBZ within a RL vector
!!
!! PARENTS
!!
!! CHILDREN
!!
!! SOURCE

logical function has_IBZ_item(Kmesh,item,ikibz,G0)

 use defs_basis
 use defs_datatypes

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
!End of the abilint section

 implicit none

!Arguments ------------------------------------
! scalars
! arrays
!scalars
 integer,intent(out) :: ikibz
 type(BZ_mesh_type),intent(in) :: Kmesh
!arrays
 integer,intent(out) :: G0(3)
 real(dp),intent(in) :: item(3)

!Local variables-------------------------------
!scalars
 integer :: ik_ibz,yetfound
 character(len=500) :: msg

! *************************************************************************

 has_IBZ_item=.FALSE. ; ikibz=0 ; G0(:)=0 ; yetfound=0
 do ik_ibz=1,Kmesh%nibz
  if (isamek(item,Kmesh%ibz(:,ik_ibz),G0)) then
   has_IBZ_item=.TRUE.
   ikibz=ik_ibz
   yetfound=yetfound+1
   !EXIT
  end if
 end do

 if (yetfound/=0.and.yetfound/=1) then
  write(msg,'(a)')' multiple k-points found '
  MSG_BUG(msg)
 end if

end function has_IBZ_item
!!***

!!****f* m_bz_mesh/make_mesh
!! NAME
!! make_mesh
!!
!! FUNCTION
!! Initialize the BZ_mesh_type starting from qptrlatt and qshft
!!
!! INPUTS
!! Cryst<Crystal_structure>=Info on the crystalline structure.
!! nqshft=Number of shifts.
!! qptrlatt=
!! qshft(3,nqshft)=Shifts
!!
!! OUTPUT
!! Kmesh
!!
!! PARENTS
!!      gw_tools
!!
!! CHILDREN
!!      get_full_kgrid,get_tetra,matr3inv,pclock,smpbz,symkpt
!!
!! SOURCE

subroutine make_mesh(Kmesh,Cryst,qptrlatt,nqshft,qshft)

 use defs_basis
 use defs_datatypes, only : BZ_mesh_type, Crystal_structure

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_11util
 use interfaces_13recipspace
!End of the abilint section

 implicit none

!Arguments -------------------------------
!scalars
 integer,intent(in) :: nqshft
 type(BZ_mesh_type),intent(out) :: Kmesh
 type(Crystal_structure),intent(in) :: Cryst
!arrays
 integer,intent(in) :: qptrlatt(3,3) 
 real(dp),intent(in) :: qshft(3,nqshft)

!Local variables -------------------------
!scalars
 integer :: max_nqpt,facbrv,brav,mtetra,iqibz,nqibz,nqbz
 integer :: nsym,timrev,option,iscf,kptopt,msym,ntetra_irr
 real(dp) :: tetra_vol
!arrays
 integer :: bravais(11),dsifkpt(3),vacuum(3)
 integer,pointer :: symrec(:,:,:),symrel(:,:,:),symafm(:)
 integer,allocatable :: ibz2bz(:),bz2ibz(:)
 integer,allocatable :: ngqpt(:,:),tetra_full(:,:,:),tetra_mult(:),tetra_wrap(:,:,:)
 real(dp) :: gmet(3,3),rprimd(3,3),qlatt(3,3),gprimd(3,3),rlatt(3,3)
 real(dp),pointer :: tnons(:,:)
 real(dp),allocatable :: qibz(:,:),qbz(:,:),qshft_loc(:,:)
 real(dp),allocatable :: wtq(:),wtq_folded(:)

! *************************************************************************

 ! Here we can select only particular symmetries if needed.
 ! Try to use getkgrid should be safer but it is difficult
 ! to encapsulate all the options.
 nsym=Cryst%nsym
 timrev=0 ; if (Cryst%timrev==2) timrev=1  !FIXME there an incompatibly between Cryst%timrev and symkpt
 rprimd = Cryst%rprimd
 gprimd = Cryst%gprimd
 gmet   = Cryst%gmet

 symrel => Cryst%symrel
 tnons  => Cryst%tnons
 symrec => Cryst%symrec
 symafm => Cryst%symafm

 if (nqshft==1) then
  allocate(qshft_loc(3,nqshft))
  qshft_loc=qshft
 else 
  ! try to reduce the qpoint grid to a single qshift, otherwise stop
  stop 'multiple shifts not implemented yet'
  iscf = 3
  msym=nsym
  kptopt=3
  dsifkpt(:)=1
  vacuum(:)=0
!  wtq(:) = one
!  call getkgrid(dsifkpt,std_out,iscf,kpt,kptopt,kptrlatt,kptrlen,&
!& msym,nkpt,nkpt_computed,nshiftk,nsym,rprimd,shiftk,symafm,&
!& symrel,tnons,vacuum,wtk)
 end if 

 max_nqpt = qptrlatt(1,1)*qptrlatt(2,2)*qptrlatt(3,3) &
&          +qptrlatt(1,2)*qptrlatt(2,3)*qptrlatt(3,1) &
&          +qptrlatt(1,3)*qptrlatt(2,1)*qptrlatt(3,2) &
&          -qptrlatt(1,2)*qptrlatt(2,1)*qptrlatt(3,3) &
&          -qptrlatt(1,3)*qptrlatt(2,2)*qptrlatt(3,1) &
&          -qptrlatt(1,1)*qptrlatt(2,3)*qptrlatt(3,2)

 allocate(qibz(3,max_nqpt),qbz(3,max_nqpt))

 !Save memory during the generation of the q-mesh in the full BZ  
 !Take into account the type of Bravais lattice
 !call symbrav to fill bravais
 !berryopt = 1
 !call symbrav(berryopt,bravais,iout,nsym,nptsym,ptsymrel,rmet,rprimd)

 option=1  ! just for printout
 brav=1
 facbrv=1
 !if (anaddb_dtset%brav==2) facbrv=2
 !if (anaddb_dtset%brav==3) facbrv=4

 call smpbz(brav,std_out,qptrlatt,max_nqpt,nqbz,nqshft,option,qshft_loc,qbz)
 ! 
 ! === Reduce the number of such points by symmetrization ===
 allocate(ibz2bz(nqbz),wtq(nqbz),wtq_folded(nqbz))
 wtq(:)=one/nqbz  ! Weights sum up to one
 option=1         ! Do not output

 call symkpt(gmet,ibz2bz,qbz,nqbz,nqibz,nsym,option,symrec,timrev,wtq,wtq_folded)

 ! Here I should initialize kmesh
 ! For the moment fill only the quantities I need for the DOS.
 Kmesh%nshift=nqshft
 Kmesh%gmet   = gmet
 Kmesh%gprimd = gprimd
 Kmesh%nsym   = nsym
 Kmesh%timrev =Cryst%timrev
 Kmesh%kptrlatt=qptrlatt
 allocate(Kmesh%shift(3,nqshft))
 Kmesh%shift=qshft_loc

 Kmesh%nbz    = nqbz
 Kmesh%nibz   = nqibz
 allocate(Kmesh%bz (3,nqbz ))
 allocate(Kmesh%ibz(3,nqibz))
 allocate(Kmesh%wt(nqibz))

 Kmesh%bz=qbz(3,nqbz)
 do iqibz=1,nqibz
  Kmesh%ibz(:,iqibz)=qbz(:,ibz2bz(iqibz))
  Kmesh%wt (iqibz)=wtq_folded(ibz2bz(iqibz))
  qibz(:,iqibz)=qbz(:,ibz2bz(iqibz))
 end do
 deallocate(ibz2bz,wtq,wtq_folded)

 ! === Tetrahedron method ===
 ! * convert kptrlatt to double and invert. 
 ! * qlatt here refer to the shortest qpt vectors
 rlatt(:,:)=qptrlatt(:,:)
 call matr3inv(rlatt,qlatt)
 !  
 ! === Make full kpoint grid and get equivalence to irred kpoints ===
 ! * Note: This routines badly scales wrt nqbz, 
 ! TODO should be rewritten, pass timrev and speed up by looping on shells.
 print*,'calling get_full_kgrid with ',nqbz
 call pclock(0)

 allocate(bz2ibz(nqbz))
 call get_full_kgrid(bz2ibz,qlatt,qibz,qbz,qptrlatt,nqibz,nqbz,nqshft,nsym,qshft_loc,symrel)

 print*,'after get_full_kgrid'
 call pclock(9999)
 !  
 ! === Get tetrahedra, ie indexes of the full q-points at their summits ===
 !   tetra_full(:,1,it) : the indices of the irreducible k-points
 !   tetra_full(:,2,it) : the indices of the full k-points
 !   tetra_wrap(:,:,it) : a flag to wrap q-points outside the IBZ (+-1) to get the irreducible tetrahedra
 !  * the number of equivalent tetrahedra is counted in tetra_mult and the inequivalent few (ntetra < mtetra) are 
 !   packed into the beginning of tetra_full
 mtetra=6*nqbz
 allocate(tetra_full(4,2,mtetra),tetra_wrap(3,4,mtetra),tetra_mult(mtetra))
   
 call get_tetra(bz2ibz,gprimd,qlatt,qbz,mtetra,nqbz,ntetra_irr,tetra_full,tetra_mult,tetra_wrap,tetra_vol)

 write(*,*)' Number of irreducible tetrahedrons = ',ntetra_irr
 deallocate(bz2ibz)

 ! === Fill in tetrahedron variables and arrays ===
 Kmesh%ntetra_irr=ntetra_irr
 Kmesh%tetra_vol =tetra_vol

 allocate(Kmesh%tetra_full(4,2,ntetra_irr))
 Kmesh%tetra_full=tetra_full(:,:,1:ntetra_irr)
 deallocate(tetra_full)

 allocate(Kmesh%tetra_wrap(3,4,ntetra_irr))
 Kmesh%tetra_wrap=tetra_wrap(:,:,1:ntetra_irr)
 deallocate(tetra_wrap)

 allocate(Kmesh%tetra_mult(ntetra_irr))
 Kmesh%tetra_mult=tetra_mult(1:ntetra_irr)
 deallocate(tetra_mult)

 deallocate(qshft_loc)

 write(*,*)'make Kmesh done'

end subroutine make_mesh
!!***

!!****f* m_bz_mesh/findk
!! NAME
!! findk
!!
!! FUNCTION
!! Check whether the k-point is in the set of the kbz
!!
!! COPYRIGHT
!! Copyright (C) 1999-2008 ABINIT group (GMR, VO, LR, RWG, MG)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~abinit/COPYING
!! or http://www.gnu.org/copyleft/gpl.txt .
!! For the initials of contributors, see ~abinit/doc/developers/contributors.txt .
!!
!! INPUTS
!!  kbz(3,nkbz)=coordinates of k points in the BZ
!!  nkcalc= number of k points for GW calculation (input variable)
!!  nkbz=number of k points in Brillouin zone
!!  xkcalc(3,nkcalc)= coordinates of the k points
!!  umklp_opt=0 if no umklapp vector is admitted, 1 otherwise
!!
!! OUTPUT
!!  kcalc=index of the k points inside kbz
!!
!! TODO 
!!  Should be removed and replaced by Kmesh methods
!!
!! PARENTS
!!      gw_tools,setup_sigma
!!
!! CHILDREN
!!      canon9,wrtout
!!
!! SOURCE

#if defined HAVE_CONFIG_H
#include "config.h"
#endif

subroutine findk(nkcalc,nkbz,xkcalc,kbz,kcalc,umklp_opt)

 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_01manage_mpi
 use interfaces_11util
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: nkbz,nkcalc,umklp_opt
!arrays
 integer,intent(out) :: kcalc(nkcalc)
 real(dp),intent(in) :: kbz(3,nkbz),xkcalc(3,nkcalc)

!Local variables-------------------------------
!scalars
 integer :: ik,jj
 real(dp) :: shift
 character(len=500) :: message
!arrays
 real(dp) :: dummy(3),ktest(3)

! *************************************************************************

 write(message,'(2a)')ch10,' findk : check if the k-points for sigma are in the set of BZ'
 call wrtout(06,message,'COLL')

 if (umklp_opt==0) then 
  do jj=1,nkcalc
   kcalc(jj)=0
   do ik=1,nkbz
    if (all(abs(xkcalc(:,jj)-kbz(:,ik))<1.e-3)) kcalc(jj)=ik
   end do
   if (kcalc(jj)==0) then
    write(message,'(4a,3(f6.3,1x),a)')ch10,' findk : ERROR -',&
&    ch10,' k-point ',xkcalc(:,jj),' not in the set of kbz'
    call wrtout(06,message,'COLL') ; call leave_new('COLL')
   end if
  end do
 else if (umklp_opt==1) then 
  do jj=1,nkcalc
   kcalc(jj)=0
   do ik=1,nkbz
    dummy=xkcalc(:,jj)-kbz(:,ik)
    call canon9(dummy(1),ktest(1),shift)
    call canon9(dummy(2),ktest(2),shift)
    call canon9(dummy(3),ktest(3),shift)
    if (all(abs(ktest)<1.e-3)) kcalc(jj)=ik
   end do
   if(kcalc(jj)==0) then
    write(message,'(4a,3(f6.3,1x),3a)')ch10,' findk : ERROR -',&
&    ch10,' k-point ',xkcalc(:,jj),' not in the set of kbz',&
&    ch10,' even though umklapp G0 vectors are allowed '
    call wrtout(06,message,'COLL') ; call leave_new('COLL')
   end if
  end do
 else 
  write(message,'(4a)')ch10,' findk : BUG-',ch10,' wrong value for uklp'
  call wrtout(06,message,'COLL') ; call leave_new('COLL')
 end if 

 write(message,'(2a)')' check ok !',ch10
 call wrtout(06,message,'COLL')

end subroutine findk
!!***

!!****f* m_bz_mesh/identk
!! NAME
!! identk
!!
!! FUNCTION
!! Identify k-points in the whole BZ starting from the IBZ. 
!! Generate also symmetry tables relating the BZ to the IBZ.
!!
!! INPUTS
!!  kibz(3,nkibz)=Coordinates of k-points in the IBZ.
!!  timrev=2 if time reversal symmetry can be used; 1 otherwise.
!!  nkibz=Number of k points in IBZ.
!!  nkbzmx=Maximum number of k points in BZ.
!!  nsym=Number of symmetry operations.
!!  symrec(3,3,nsym)=Symmetry operation matrices in reciprocal space.
!!  symafm(nsym)=(anti)ferromagnetic part of symmetry operations.
!!
!! OUTPUT
!!  kbz(3,nkbzmx)= k-points in whole BZ
!!  ktab(nkbzmx)= table giving for each k-point in the BZ (array kbz), 
!!   the corresponding irreducible point in the array (kibz)
!!   k_BZ= (IS) kIBZ where S is one of the symrec operations and I is the inversion or the identity
!!    where k_BZ = (IS) k_IBZ and S = \transpose R^{-1} 
!!  ktabi(nkbzmx)= for each k-point in the BZ defines whether inversion has to be 
!!   considered in the relation k_BZ=(IS) k_IBZ (1 => only S; -1 => -S)  
!!  ktabo(nkbzmx)= the symmetry operation S that takes k_IBZ to each k_BZ
!!  nkbz= no. of k-points in the whole BZ
!!  wtk(nkibz)= weight for each k-point in IBZ for symmetric quantities:
!!              no. of distinct ks in whole BZ/(timrev*nsym)
!!
!! NOTES
!!  The logic of the routine relies on the assumption that kibz really represent an irreducible set. 
!!  If symmetrical points are present in the input list, indeed, some the output weights will turn out to be zero.
!!  An initial check is done at the beginnig of the routine to trap this possible error.
!!
!! PARENTS
!!      rdm,setup_kmesh
!!
!! CHILDREN
!!      dosym,wrtout
!!
!! SOURCE

subroutine identk(kibz,nkibz,nkbzmx,nsym,timrev,symrec,symafm,use_antiferro,kbz,ktab,ktabi,ktabo,nkbz,wtk,prtvol)

 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_01manage_mpi
 use interfaces_15gw
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: nkbzmx,nkibz,nsym,timrev,prtvol
 integer,intent(out) :: nkbz
 logical,intent(in) :: use_antiferro
!arrays
 integer,intent(in) :: symafm(nsym),symrec(3,3,nsym)
 integer,intent(out) :: ktab(nkbzmx),ktabi(nkbzmx),ktabo(nkbzmx)
 real(dp),intent(in) :: kibz(3,nkibz)
 real(dp),intent(out) :: kbz(3,nkbzmx),wtk(nkibz)

!Local variables ------------------------------
!scalars
 integer :: div4,found,id,ii,ik1,ik2,ikbz,ikibz,ikp,iold,isym,itim,jj,res
 logical :: isirred
 character(len=100) :: fmt
 character(len=500) :: msg
!arrays
 integer :: G0(3)
 real(dp) :: knew(3),k1(3),k2(3)

! *************************************************************************

 DBG_ENTER("COLL")

 ! === Check whether kibz really forms an irreducible set ===
 isirred=.TRUE.
 do ik1=1,nkibz-1
  k1=kibz(:,ik1)
  do ik2=ik1+1,nkibz
   k2=kibz(:,ik2)

   do itim=1,timrev
    do isym=1,nsym
     if (use_antiferro.and.symafm(isym)==-1) CYCLE
     knew = (3-2*itim) * MATMUL(symrec(:,:,isym),k2)
     if (isamek(k1,knew,G0)) then
      isirred=.FALSE.
      write(msg,'(2(a,3f8.4),2(a,i2))')&
&      ' k1 = ',k1,' is symmetrical of k2 = ',k2,' through sym = ',isym,' itim = ',itim
      MSG_WARNING(msg)
     end if
    end do
   end do

  end do
 end do

 if (.not.isirred) then
  MSG_ERROR('input kibz is not irred.') 
 end if

 !
 ! === Loop over k-points in IBZ ===
 ! * Start with zero no. of k-points found.
 nkbz=0 
 do ikibz=1,nkibz
  wtk(ikibz) = zero

  ! === Loop over time-reversal I and symmetry operations S  ===
  ! * Use spatial inversion instead of time reversal whenever possible.
  do itim=1,timrev
   do isym=1,nsym

    if (use_antiferro.and.symafm(isym)==-1) CYCLE

    ! === Form IS k ===
    ! FIXME this is a temporary hacking to pass tests under gfortran, should use MATMUL
    call dosym(REAL(symrec(:,:,isym),dp),itim,kibz(:,ikibz),knew)
    !knew(:)=(3-2*itim)*MATMUL(symrec(:,:,isym),kibz(:,ikibz))

    ! * Check whether it has already been found (to within a RL vector).
    iold=0
    do ikbz=1,nkbz
     if (isamek(knew,kbz(:,ikbz),G0)) iold=iold+1
    end do
    ! * If not yet found add to kbz and increase the weight.
    if (iold==0) then
     nkbz=nkbz+1
     wtk(ikibz)=wtk(ikibz)+one
     if (nkbz>nkbzmx) then
      write(msg,'(a,i6,2a)')&
&      ' nkbzmx too small, nkbzmx = ',nkbzmx,ch10,&
&      ' ACTION : increase nkbzmx !'
      MSG_BUG(msg)
     end if
     kbz(:,nkbz)=knew(:)
     ktab(nkbz)=ikibz
     ktabo(nkbz)=isym
     ktabi(nkbz)=3-2*itim
    end if
   end do 
  end do 

 end do !ikibz

 ! * Weights are normalized to 1.
 wtk = wtk/SUM(wtk)

 ! ================================
 ! === Printout of the results ===
 ! ================================
 if (prtvol>0) then

  write(msg,'(2a,i3,2a,10x,2a)')ch10,&
&  ' number of k-points in the irreducible wedge (IBZ) ',nkibz,ch10,&
&  ' k-points [reciprocal lattice units]','weights',ch10
  call wrtout(std_out,msg,'COLL')
  write(fmt,*)'(i5,3f12.6,3x,f12.6)'
  do jj=1,nkibz
   write(msg,fmt) jj,(kibz(ii,jj),ii=1,3),wtk(jj)
   call wrtout(std_out,msg,'COLL')
  end do
  write(msg,'(2a,i2,3a,i5,a)')ch10,&
&  ' together with ',nsym,' symmetry operations and inversion',ch10,&
&  ' have yielded ',nkbz,' k-points in Brillouin Zone (BZ):'
  call wrtout(std_out,msg,'COLL')

  write(fmt,*)'(i5,2x,4(3f7.3,2x))'
  div4=nkbz/4 ; res=MOD(nkbz,4)
  do id=0,div4-1
   jj=4*id+1 
   write(msg,fmt)jj,((kbz(ii,ikp),ii=1,3),ikp=jj,jj+3)
   call wrtout(std_out,msg,'COLL')
  end do
  if (res/=0) then
   write(fmt,*)'(i5,2x,',res,'(3f7.3,2x),a)'
   write(msg,fmt)4*div4+1,((kbz(ii,ik1),ii=1,3),ik1=4*div4+1,nkbz),ch10
   call wrtout(std_out,msg,'COLL')
  end if

 end if !prtvol

 DBG_EXIT("COLL")

end subroutine identk

END MODULE m_bz_mesh
!!***
