!{\src2tex{textfont=tt}}
!!****f* ABINIT/make_transitions
!! NAME
!! make_transitions
!!
!! FUNCTION
!!  Calculate transition energies entering the espression for the irreducible polarizability 
!!
!! COPYRIGHT
!!  Copyright (C) 2007-2009 ABINIT group (MG)
!!  This file is distributed under the terms of the
!!  GNU General Public License, see ~abinit/COPYING
!!  or http://www.gnu.org/copyleft/gpl.txt .
!!
!! INPUTS
!!  nsspol=1 for spin unpolarized, 2 for spin polarized calculations
!!  nbnds=total number of bands
!!  kmesh<bz_mesh_type>=datatype gathering info on the k-mesh:
!!   | %nbz=number of k-points in the full BZ
!!   | %nibz=number of k-points in the IBZ
!!   | %tab(nkbz)=table giving for each k-point in the BZ, the corresponding irreducible point in the IBZ array
!!   | %bz(3,nkbz)=reduced coordinated of k-points
!!  mG0(3)=integer defining the number of shells in which the umklapp G0 vector has to be found
!!  TOL_DELTA_OCC=tolerance on the difference of the occupation numbers
!!  gw_energy(nbnds,kmesh%nkibz,nsppol)=quasi-particle energies energies 
!!  occ(nbnds,kmesh%nkibz,nsppol)=occupation numbers
!!  chi0alg=integer defining the method used to calculate chi0
!!   0 ==> calculate chi0 using the Adler-Wiser expression
!!   1 ==> use spectral method 
!!  timrev=if 2, time-reversal symmetry is considered; 1 otherwise
!!
!! OUTPUT
!!  
!!
!! SIDE EFFECTS
!!
!! NOTES
!! 
!!
!! PARENTS
!!      cchi0,cchi0q0
!!
!! CHILDREN
!!      abi_etsf_get_qp,bstruct_clean,bstruct_init,copy_bandstructure
!!      destroy_bz_mesh_type,destroy_sigma_results,destroycrystal
!!      destroywandata,findk,get_dos,hdr_check,hdr_clean,initkmesh,make_mesh
!!      make_path,makewannierhr,metric,nullifybzmesh,plotbands
!!      print_bandstructure,print_sigma_perturbative,printwandata,prompt
!!      readwandata,set2unit,wanmatinterpol,wannierinterpol,wrtout
!!
!! SOURCE

#if defined HAVE_CONFIG_H
#include "config.h"
#endif

subroutine make_transitions(chi0alg,nbnds,nbvw,nsppol,symchi,timrev,TOL_DELTA_OCC,zcut,&
& max_rest,min_rest,my_max_rest,my_min_rest,kmesh,ltg_q,mpi_enreg,mG0,gw_energy,occ,qpoint)

 use defs_basis
 use defs_datatypes
 use m_bz_mesh
 use m_io_tools, only : flush_unit

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_01manage_mpi
 use interfaces_lib01hidempi
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: chi0alg,nbnds,nbvw,nsppol,symchi,timrev
 real(dp),intent(in) :: TOL_DELTA_OCC,zcut
 real(dp),intent(out) :: max_rest,min_rest
 real(dp),intent(out) :: my_max_rest,my_min_rest
 type(bz_mesh_type),intent(in) :: kmesh
 type(little_group),intent(in) :: ltg_q
 type(MPI_type    ),intent(in) :: mpi_enreg
!arrays
 integer,intent(in) :: mG0(3)
 real(dp),intent(in) :: gw_energy(nbnds,kmesh%nibz,nsppol)
 real(dp),intent(in) :: occ(nbnds,kmesh%nibz,nsppol),qpoint(3)

!Local variables-------------------------------
!scalars
 integer :: ib1,ib2,ii,ikbz,ik_ibz,ikmq_bz,ikmq_ibz,is,nt,io,ntrans,my_ntrans
 integer :: ier,rank,spaceComm,nprocs,master,iloop
 real(dp) :: delta_ene,delta_occ,spin_fact
 character(len=500) :: msg
!arrays
 integer :: G0(3)
 real(dp) :: kmq(3)

!************************************************************************

#if defined DEBUG_MODE
 write(msg,'(a)')' make_transitions : enter'
 call wrtout(std_out,msg,'COLL') 
 call flush_unit(std_out)
#endif

 if (chi0alg<0 .or. chi0alg>=2) then 
  write(msg,'(4a,i3,a)')ch10,&
&  ' make_transitions : BUG ',ch10,&
&  ' chi0alg = ',chi0alg,' not allowed '
  call wrtout(std_out,msg,'COLL') ; call leave_new('COLL')
 end if 
 if (timrev/=1 .and. timrev/=2) then 
  write(msg,'(4a,i3,a)')ch10,&
&  ' make_transitions : BUG ',ch10,&
&  ' timrev = ',timrev,' not allowed'
  call wrtout(std_out,msg,'COLL') ; call leave_new('COLL')
 end if 
 !
 ! Initialize MPI quantities.
 call xcomm_init  (mpi_enreg,spaceComm) 
 call xmaster_init(mpi_enreg,master   )  
 call xme_init    (mpi_enreg,rank     )          
 call xproc_max(nprocs,ier)

 if (nprocs==1 .and. mpi_enreg%gwpara/=0) then 
  write(msg,'(4a,i4)')ch10,&
&  ' make_transitions : BUG : ',ch10,&
&  ' in a sequential run gwpara should be 0 while it is ',mpi_enreg%gwpara 
  call wrtout(std_out,msg,'COLL') ; call leave_new('COLL')
 end if
 !
 ! In the first loop calculate total number of transitions for this q-point
 ! as well min and max transition without taking into account distribution of bands. 
 ! In the second iteration calculate min and Max transition for this processor.
 !
 spin_fact=half ; if (nsppol==2) spin_fact=one
 my_max_rest=smallest_real ; my_min_rest=greatest_real
    max_rest=smallest_real ;    min_rest=greatest_real
 print*,"done"

 do iloop=1,2
  nt=0
 print*,"in loop",iloop
  do ikbz=1,kmesh%nbz

   ik_ibz=kmesh%tab(ikbz)
   kmq(:)=kmesh%bz(:,ikbz)-qpoint(:)

   if (symchi==1) then  
    if (ltg_q%ibzq(ikbz)/=1) cycle 
    ! This point does not belong to the IBZ defined by the little group
   end if 
   !
   ! Find kp=k-q-G0 and also G0 where kp is in the first BZ
   if (.not.has_BZ_item(Kmesh,kmq,ikmq_bz,g0)) then
    ! Stop as the weight 1.0/nkbz is wrong and should be changed in cchi0/cchi0q0
    write(msg,'(4a,2(2a,3f12.6),2a)')ch10,&
&    ' make_transitions : ERROR - ',ch10,&
&    ' kp  = k-q-G0 not found in the BZ mesh',ch10,&
&    ' k   = ',(kmesh%bz(ii,ikbz),ii=1,3),ch10,&
&    ' k-q = ',(kmq(ii),ii=1,3),ch10,&
&    ' weight in cchi0/cchi0q is wrong ' 
    call wrtout(std_out,msg,'COLL') ; call leave_new('COLL')
   end if 

   ikmq_ibz=kmesh%tab(ikmq_bz)
   do is=1,nsppol

    do ib1=1,nbnds           
     if (iloop==2 .and. mpi_enreg%gwpara==2) then
      if (mpi_enreg%proc_distrb(ik_ibz,ib1,is)/=rank) cycle
     end if

     do ib2=1,nbnds
      if (timrev==2 .and. ib1<ib2) cycle ! Thanks to time-reversal we gain a factor ~2.
      if (iloop==2 .and. mpi_enreg%gwpara==2) then
       if (mpi_enreg%proc_distrb(ik_ibz,ib2,is)/=rank) cycle
      end if

      ! Take care of "valence-valence" transitions in case of gwpara==2
      if (iloop==2 .and. mpi_enreg%gwpara==2) then
       if (ib1<=nbvw .and. ib2<=nbvw .and. rank/=master) cycle 
      end if

      delta_occ=spin_fact*(occ(ib1,ikmq_ibz,is)-occ(ib2,ik_ibz,is))
      delta_ene=gw_energy(ib1,ikmq_ibz,is)-gw_energy(ib2,ik_ibz,is)

      select CASE (chi0alg)
       CASE (0)  
        ! Adler-Wiser expression.
        ! Skip only if factor due to occupation number is smaller than TOL_DELTA_OCC
        if (abs(delta_occ) < abs(TOL_DELTA_OCC)) cycle
       CASE (1)
        ! Spectral method with time-reversal, only resonant transitions 
        ! This has to changed to include spectral method without time-reversal
        if (delta_ene < -abs(TOL_DELTA_OCC) .or. abs(delta_occ) < abs(TOL_DELTA_OCC)) cycle
      end select 
      !
      ! We have a new transition
      nt=nt+1

      if (iloop==1) then 
       max_rest=MAX(max_rest,zero,delta_ene)
       if (delta_ene>=-tol6) min_rest=MIN(min_rest,delta_ene)
      end if
      if (iloop==2) then 
       my_max_rest=MAX(my_max_rest,zero,delta_ene)
       if (delta_ene>=-tol6) my_min_rest=MIN(my_min_rest,delta_ene)
      end if

     end do 
    end do 
   end do
  end do
  if (iloop==1) ntrans=nt
  if (iloop==2) my_ntrans=nt
 end do !iloop

 !if (mpi_enreg%gwpara==2) then 
 ! call xmin_mpi(my_min_rest,min_rest,spaceComm,ier)
 ! call xmax_mpi(my_max_rest,max_rest,spaceComm,ier)
 !end if

 write(msg,'(2a,i9,2a,f8.3,3a,f8.3,a)')ch10,&
& ' Total number of transitions = ',ntrans,ch10,&
& ' min resonant     = ',min_rest*Ha_eV,' [eV] ',ch10,&
& ' Max resonant     = ',max_rest*Ha_eV,' [eV] '
 call wrtout(std_out,msg,'COLL')
 if (nprocs/=1) then 
  write(msg,'(2a,i9,2a,f8.3,3a,f8.3,a)')ch10,&
&  ' Total number of transitions for this processor= ',my_ntrans,ch10,&
&  ' min resonant     = ',my_min_rest*Ha_eV,' [eV] ',ch10,&
&  ' Max resonant     = ',my_max_rest*Ha_eV,' [eV] '
  call wrtout(std_out,msg,'PERS')
 end if

#if defined DEBUG_MODE
 write(msg,'(a)')' make_transitions : exit'
 call wrtout(std_out,msg,'COLL')
 call flush_unit(std_out)
#endif

end subroutine make_transitions 
!!***

!!****f* ABINIT/nullify_transitions
!! NAME
!! nullify_transitions
!!
!! FUNCTION
!!
!! INPUTS
!!
!! OUTPUT
!!
!! PARENTS
!!
!! CHILDREN
!!      abi_etsf_get_qp,bstruct_clean,bstruct_init,copy_bandstructure
!!      destroy_bz_mesh_type,destroy_sigma_results,destroycrystal
!!      destroywandata,findk,get_dos,hdr_check,hdr_clean,initkmesh,make_mesh
!!      make_path,makewannierhr,metric,nullifybzmesh,plotbands
!!      print_bandstructure,print_sigma_perturbative,printwandata,prompt
!!      readwandata,set2unit,wanmatinterpol,wannierinterpol,wrtout
!!
!! SOURCE
subroutine nullify_transitions(self)

  use defs_basis
  use defs_datatypes

  implicit none

  type(transitions_type),intent(out) :: self

  nullify(self%bands)
  nullify(self%distrb)
  nullify(self%ik_ibz)
  nullify(self%ikmq_ibz)
  nullify(self%ik_bz)
  nullify(self%ikmq_bz)    
  nullify(self%G0) 
  nullify(self%spin)      
  nullify(self%delta_occ)
  nullify(self%qpoint)
  nullify(self%delta_ene) 
  nullify(self%num_w)

end subroutine nullify_transitions 
!!***

!!****f* ABINIT/init_transitions
!! NAME
!! init_transitions
!!
!! FUNCTION
!!
!! INPUTS
!!
!! OUTPUT
!!
!! PARENTS
!!
!! CHILDREN
!!      abi_etsf_get_qp,bstruct_clean,bstruct_init,copy_bandstructure
!!      destroy_bz_mesh_type,destroy_sigma_results,destroycrystal
!!      destroywandata,findk,get_dos,hdr_check,hdr_clean,initkmesh,make_mesh
!!      make_path,makewannierhr,metric,nullifybzmesh,plotbands
!!      print_bandstructure,print_sigma_perturbative,printwandata,prompt
!!      readwandata,set2unit,wanmatinterpol,wannierinterpol,wrtout
!!
!! SOURCE
subroutine init_transitions(self,nkbz,nbnds,nbvw,nsppol,nomega,qpoint,ntrans)

   use defs_basis
   use defs_datatypes

   implicit none

   type(transitions_type),intent(out) :: self
   integer,intent(in) :: nkbz,nbnds,nbvw,nsppol,nomega
   integer,optional,intent(in) :: ntrans
   real(dp),intent(in) :: qpoint(3)
   integer :: nt

#if defined DEBUG_MODE
write(*,*)" init_transitions : enter"
#endif

   self%nbnds=nbnds
   self%nbvw=nbvw
   self%nomega=nomega
   self%nkbz=nkbz
   self%nsppol=nsppol

   self%ntrans=nkbz*nbnds**2*nsppol
   if (present(ntrans)) self%ntrans=ntrans


   nt=self%ntrans
   allocate(self%bands(2,nt)     )  ;  self%bands(:,:)    =0
   allocate(self%distrb(nt)      )  ;  self%bands(:,:)    =-999
   allocate(self%ik_ibz(nt)      )  ;  self%ik_ibz(:)     =0
   allocate(self%ikmq_ibz(nt)    )  ;  self%ikmq_ibz(:)   =0
   allocate(self%ik_bz(nt)       )  ;  self%ik_bz(:)      =0
   allocate(self%ikmq_bz(nt)     )  ;  self%ik_bz(:)      =0
   allocate(self%G0(3,nt)        )  ;  self%G0(:,:)       =0
   allocate(self%spin(2,nt)      )  ;  self%spin(:,:)     =0
   allocate(self%delta_occ(nt)   )  ;  self%delta_occ(:)  =zero
   allocate(self%qpoint(3)       )  ;  self%qpoint(:)     =qpoint(:)
   allocate(self%delta_ene(nt)   )  ;  self%delta_ene(:)  =czero
   allocate(self%num_w(nomega,nt))  ;  self%num_w(:,:)    =czero

#if defined DEBUG_MODE
write(*,*)" init_transitions : exit"
#endif

end subroutine init_transitions 
!!***

!!****f* ABINIT/destroy_transitions
!! NAME
!! destroy_transitions
!!
!! FUNCTION
!!
!! INPUTS
!!
!! OUTPUT
!!
!! PARENTS
!!
!! CHILDREN
!!      abi_etsf_get_qp,bstruct_clean,bstruct_init,copy_bandstructure
!!      destroy_bz_mesh_type,destroy_sigma_results,destroycrystal
!!      destroywandata,findk,get_dos,hdr_check,hdr_clean,initkmesh,make_mesh
!!      make_path,makewannierhr,metric,nullifybzmesh,plotbands
!!      print_bandstructure,print_sigma_perturbative,printwandata,prompt
!!      readwandata,set2unit,wanmatinterpol,wannierinterpol,wrtout
!!
!! SOURCE
subroutine destroy_transitions(self)

  use defs_basis
  use defs_datatypes

  implicit none

  type(transitions_type),intent(inout) :: self

   if (associated(self%bands    ))  deallocate(self%bands    )    
   if (associated(self%distrb   ))  deallocate(self%distrb   )    
   if (associated(self%ik_ibz   ))  deallocate(self%ik_ibz   )
   if (associated(self%ikmq_ibz ))  deallocate(self%ikmq_ibz )
   if (associated(self%ik_bz    ))  deallocate(self%ik_bz    )
   if (associated(self%ikmq_bz  ))  deallocate(self%ikmq_bz  ) 
   if (associated(self%G0       ))  deallocate(self%G0       )
   if (associated(self%spin     ))  deallocate(self%spin     ) 
   if (associated(self%delta_occ))  deallocate(self%delta_occ)  
   if (associated(self%qpoint   ))  deallocate(self%qpoint   )  
   if (associated(self%delta_ene))  deallocate(self%delta_ene)   
   if (associated(self%num_w    ))  deallocate(self%num_w    )

end subroutine destroy_transitions 
!!***

!!****f* ABINIT/copy_transitions
!! NAME
!! copy_transitions
!!
!! FUNCTION
!!
!! INPUTS
!!
!! OUTPUT
!!
!! PARENTS
!!
!! CHILDREN
!!      abi_etsf_get_qp,bstruct_clean,bstruct_init,copy_bandstructure
!!      destroy_bz_mesh_type,destroy_sigma_results,destroycrystal
!!      destroywandata,findk,get_dos,hdr_check,hdr_clean,initkmesh,make_mesh
!!      make_path,makewannierhr,metric,nullifybzmesh,plotbands
!!      print_bandstructure,print_sigma_perturbative,printwandata,prompt
!!      readwandata,set2unit,wanmatinterpol,wannierinterpol,wrtout
!!
!! SOURCE
subroutine copy_transitions(t_in,t_out)

  use defs_basis 
  use defs_datatypes

  implicit none

  type(transitions_type),intent(in) :: t_in
  type(transitions_type),intent(out) :: t_out

  !local
  integer :: nt,nw

  nt=min(t_in%ntrans,t_out%ntrans) 
  nw=min(t_in%nomega,t_out%nomega)

  if (t_in%ntrans/=t_out%ntrans) then 
   write(*,*)" Copying ",nt,"/",t_in%ntrans," transitions from input datatype"
  end if
  if (t_in%ntrans/=t_out%ntrans) then 
   write(*,*)" Copying ",nw,"/",t_in%nomega," frequencies from input datatype"
  end if 

  t_out%nbnds =t_in%nbnds
  t_out%nbvw  =t_in%nbvw
  t_out%nkbz  =t_in%nkbz
  t_out%nsppol=t_in%nsppol

  ! Use whit care!
  t_out%my_min_res=t_in%my_min_res
  t_out%my_max_res=t_in%my_max_res

  ! Copy a smaller number of transitions
  t_out%ntrans=nt
  t_out%nomega=nw

  t_out%bands(:,1:nt)      = t_in%bands(:,1:nt)   
  t_out%distrb(1:nt)       = t_in%distrb(1:nt)
  t_out%ik_ibz(1:nt)       = t_in%ik_ibz(1:nt)
  t_out%ikmq_ibz(1:nt)     = t_in%ikmq_ibz(1:nt)
  t_out%ik_bz(1:nt)        = t_in%ik_bz(1:nt)
  t_out%ikmq_bz(1:nt)      = t_in%ikmq_bz(1:nt) 
  t_out%G0(:,1:nt)         = t_in%G0(:,1:nt)       
  t_out%spin(:,1:nt)       = t_in%spin(:,1:nt)
  t_out%delta_occ(1:nt)    = t_in%delta_occ(1:nt)
  t_out%qpoint(:)          = t_in%qpoint(:)
  t_out%delta_ene(1:nt)    = t_in%delta_ene(1:nt)   
  t_out%num_w(1:nw,1:nt)   = t_in%num_w(1:nw,1:nt)

end subroutine copy_transitions
!!***

!!****f* ABINIT/print_transitions
!! NAME
!! print_transitions
!!
!! FUNCTION
!!
!! INPUTS
!!
!! OUTPUT
!!
!! PARENTS
!!
!! CHILDREN
!!      abi_etsf_get_qp,bstruct_clean,bstruct_init,copy_bandstructure
!!      destroy_bz_mesh_type,destroy_sigma_results,destroycrystal
!!      destroywandata,findk,get_dos,hdr_check,hdr_clean,initkmesh,make_mesh
!!      make_path,makewannierhr,metric,nullifybzmesh,plotbands
!!      print_bandstructure,print_sigma_perturbative,printwandata,prompt
!!      readwandata,set2unit,wanmatinterpol,wannierinterpol,wrtout
!!
!! SOURCE
subroutine print_transitions(self,unit,prtvol)

 use defs_basis
 use defs_datatypes

 implicit none
 
 integer,optional,intent(in) :: unit,prtvol
 type(transitions_type),intent(in) :: self

 integer :: unt,verbose

 unt=std_out
 if (present(unit)) unt=unit
 verbose=0
 if (present(prtvol)) verbose=prtvol

 write(unt,*)" q-point : ",self%qpoint(:)
 write(unt,*)" Total number of transitions : ",self%ntrans 

end subroutine print_transitions
!!***

!!****f* ABINIT/get_my_extrema
!! NAME
!! get_my_extrema
!!
!! FUNCTION
!!
!! INPUTS
!!
!! OUTPUT
!!
!! PARENTS
!!      gw_tools
!!
!! CHILDREN
!!      abi_etsf_get_qp,bstruct_clean,bstruct_init,copy_bandstructure
!!      destroy_bz_mesh_type,destroy_sigma_results,destroycrystal
!!      destroywandata,findk,get_dos,hdr_check,hdr_clean,initkmesh,make_mesh
!!      make_path,makewannierhr,metric,nullifybzmesh,plotbands
!!      print_bandstructure,print_sigma_perturbative,printwandata,prompt
!!      readwandata,set2unit,wanmatinterpol,wannierinterpol,wrtout
!!
!! SOURCE
subroutine get_my_extrema(self,my_min_res,my_max_res)

 use defs_basis
 use defs_datatypes

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_01manage_mpi
 use interfaces_lib01hidempi
!End of the abilint section

 implicit none

 type(transitions_type),intent(in) :: self
 real(dp),intent(out) :: my_min_res,my_max_res
 
 integer :: it,my_rank,my_ntrans
 real(dp),allocatable :: ene(:)

 if (all(self%distrb(:)==-999)) then 
  write(*,*)" get_my_extrema : ERROR -",ch10,&
&  " Either BUG while distributing or ",ch10,&
&  " number of processors exceeds number of transitions"
  call leave_new('COLL')
 end if

 call xme_whoiam(my_rank)
 my_ntrans=COUNT(self%distrb==my_rank)
 if (self%my_ntrans/=my_ntrans) stop "BUG in distribution"
 
 allocate(ene(my_ntrans)) ; ene=zero
 !do it=1,self%ntrans
 ! if (self%distrb(it)==my_rank) 
 ! ene(it)=self%delta_ene(it)
 !end do
 ene(:)= PACK(real(self%delta_ene), MASK=self%distrb==my_rank)

 my_min_res = MINVAL(ene, MASK=ene>=zero)
 my_max_res = MAXVAL(ene, MASK=ene>=zero)

end subroutine get_my_extrema 
!!***

!!****f* ABINIT/split_transitions
!! NAME
!! split_transitions
!!
!! FUNCTION
!!
!! INPUTS
!!
!! OUTPUT
!!
!! PARENTS
!!
!! CHILDREN
!!      abi_etsf_get_qp,bstruct_clean,bstruct_init,copy_bandstructure
!!      destroy_bz_mesh_type,destroy_sigma_results,destroycrystal
!!      destroywandata,findk,get_dos,hdr_check,hdr_clean,initkmesh,make_mesh
!!      make_path,makewannierhr,metric,nullifybzmesh,plotbands
!!      print_bandstructure,print_sigma_perturbative,printwandata,prompt
!!      readwandata,set2unit,wanmatinterpol,wannierinterpol,wrtout
!!
!! SOURCE
subroutine split_transitions(self,mpi_enreg)

 use defs_basis
 use defs_datatypes

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_01manage_mpi
 use interfaces_15gw, except_this_one => split_transitions
 use interfaces_lib01hidempi
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 type(transitions_type),intent(inout) :: self
 type(MPI_type),intent(in) :: mpi_enreg

!Local variables ------------------------------
!scalars
 integer :: ikibz,ib1,ib2,is,it
 integer :: nproc,ntrans,my_rank
 integer :: my_start,my_stop,my_ntrans,master
 real(dp) :: my_min_res,my_max_res
 character(len=500) :: msg
!arrays

 call xme_whoiam(my_rank)
 call xmaster_init(mpi_enreg,master) 
 nproc =mpi_enreg%nproc
 ntrans=self%ntrans

 self%distrb(:)=-999
 my_ntrans=0

 select case (mpi_enreg%gwpara) 

  case (0)
   !
   ! Sequential run
   !
   write(*,*)" split_transitions : Sequential run, nothing to do"
   self%distrb(:)=my_rank
   my_ntrans=self%ntrans

  case (1)
   !
   ! Parallelization over k-points, each node has the entire set of wavefunctions.
   ! Split the total number of transitions, the data access is optimized in make_transitions.
   !
   call split_work(ntrans,my_start,my_stop)

   self%distrb(my_start:my_stop)=my_rank
   my_ntrans=my_stop-my_start+1

  case (2)
   !
   ! Parallelism over band. Be careful because wavefunctions are separated into 
   ! "valence" and "conduction". "valence" means all the states with occupation smaller
   ! than a TOL_DELTA_OCC value. Remember that "conduction" states are distributed but "valence"
   ! states are on each node. Avoid multiple distribution in case of metallic systems.
   ! Moreover Im assumig that valence index is always on the left, this distribution 
   ! does not work if we dont use time-reversal.
   ! PResently only master takes care of the valence-valence trasitions in metals 

   if (self%nbvw<=0) stop "BUG while distributing"
   !
   do it=1,self%ntrans
    ib1 = self%bands(1,it)
    ib2 = self%bands(2,it)

    if (ib1<=self%nbvw .and. ib2<=self%nbvw) then 
     ! Each processor has these states, for the moment only master than we can optimize
     if (my_rank/=master) cycle
    end if

    if ( any(mpi_enreg%proc_distrb(:,ib1,:)==my_rank) .and. &
&        any(mpi_enreg%proc_distrb(:,ib2,:)==my_rank)       & 
&      ) then 
     self%distrb(it)=my_rank
     my_ntrans=my_ntrans+1
    end if 
   end do

!DEBUG this is just to keep in mind that I cannot parallelize memory over spin
   do ib1=1,size(mpi_enreg%proc_distrb, DIM=2)
    do is=1,size(mpi_enreg%proc_distrb, DIM=3)
     if (any(mpi_enreg%proc_distrb(:,ib1,is)/=mpi_enreg%proc_distrb(1,1,is))) then
      write(*,*)"BUG while distributing wavefunctions"
      call leave_new('COLL')
     end if 
    end do
   end do
!ENDDEBUG

  case DEFAULT
   write(msg,'(4a)')ch10,&
&   " split_transitions : BUG-",ch10,&
&   " called with wrong value of gwpara "
   call wrtout(std_out,msg,'COLL') ; call leave_new('COLL')
 end select

 self%my_ntrans=my_ntrans

 call get_my_extrema(self,my_min_res,my_max_res)
 self%my_min_res=my_min_res
 self%my_max_res=my_max_res

 write(msg,'(2a,i4,a,i5,a)')ch10,&
& " Processor ",my_rank," will treat ",my_ntrans," transitions "
 call wrtout(std_out,msg,'PERS')
 write(*,*)" min and Max resonant transitions = ",my_min_res,my_max_res

end subroutine split_transitions
!!***

!!****f* ABINIT/find_my_indeces
!! NAME
!! find_my_indeces
!!
!! FUNCTION
!!
!! INPUTS
!!
!! OUTPUT
!!
!! PARENTS
!!
!! CHILDREN
!!      abi_etsf_get_qp,bstruct_clean,bstruct_init,copy_bandstructure
!!      destroy_bz_mesh_type,destroy_sigma_results,destroycrystal
!!      destroywandata,findk,get_dos,hdr_check,hdr_clean,initkmesh,make_mesh
!!      make_path,makewannierhr,metric,nullifybzmesh,plotbands
!!      print_bandstructure,print_sigma_perturbative,printwandata,prompt
!!      readwandata,set2unit,wanmatinterpol,wannierinterpol,wrtout
!!
!! SOURCE
subroutine find_my_indeces(self,nomega,omega_mesh,my_w1,my_w2)

 use defs_basis
 use defs_datatypes

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_01manage_mpi
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 type(transitions_type),intent(in) :: self
 integer,intent(in) :: nomega
 integer,intent(out) :: my_w1,my_w2
!arrays 
 real(dp) :: omega_mesh(nomega)

!Local variables-------------------------------
!scalars
 integer :: io
 character(len=500) :: msg
!************************************************************************

 ! Note nomega-1, that is because omega_mesh encloses the 
 ! interval made of the possible resonant transitions
 ! This part is very sensitive to changes in setup_mesh
 my_w2=-999
 do io=1,nomega-1
  if (omega_mesh(io) > self%my_max_res) then 
   my_w2=io+1
   exit
  end if 
 end do

 my_w1=-999
 do io=nomega,1,-1
  if (omega_mesh(io)<= self%my_min_res) then ! Check metals
   my_w1=io
   exit
  end if 
 end do 

 if (my_w1==-999 .or. my_w2==-999) then 
  write(msg,"(a,2i4)")" ERROR in find_my_indeces ",my_w1,my_w2
  call wrtout(std_out,msg,'PERS') ; call leave_new('COLL')
 end if 

end subroutine find_my_indeces
!!***


!!****f* ABINIT/get_rhor
!! NAME
!! get_rhor
!!
!! FUNCTION
!!  Read the DEN file with name fname reporting the density on the real FFT mesh 
!!  specified through the input variable ngfft_asked. If the FFT mesh asked and that found 
!!  on file differ, perform a FFT interpolation and renormalize the density so that it 
!!  integrates to the correct number of electrons. If the two FFT meshes coincides 
!!  just report the array stored on file.
!!
!! COPYRIGHT
!!  Copyright (C) 2007-2009 ABINIT group (MG)
!!  This file is distributed under the terms of the
!!  GNU General Public License, see ~abinit/COPYING
!!  or http://www.gnu.org/copyleft/gpl.txt .
!!
!! INPUTS
!! fname=Name of the density file
!! accesswff=File format (Fortran, NETCDF-ETSF)
!! localrdwf=1 if density file is local to each node (FIXME not yet implemented)
!! ngfft_asked(18)=All info on the FFT mesh required.
!! paral_kgb=Flag related to the kpoint-band-fft parallelism (FIXME not implemented)
!! MPI_enreg<MPI_type>=Information about MPI parallelization
!!  
!! OUTPUT
!! rhor_out
!!
!! NOTES
!!  1) The renormalization of the charge is not done in case of PAW since the onsite
!!     contributions have to be added. This is left to the caller.
!!  2) No check is done on sensible dimensions (nsppol, nspden). 
!!     TODO Maybe I can pass the header?
!!  3) Wavelet case not treated.
!!  4) Complex densities not treated.
!!
!! PARENTS
!!      mrgscr
!!
!! CHILDREN
!!      abi_etsf_get_qp,bstruct_clean,bstruct_init,copy_bandstructure
!!      destroy_bz_mesh_type,destroy_sigma_results,destroycrystal
!!      destroywandata,findk,get_dos,hdr_check,hdr_clean,initkmesh,make_mesh
!!      make_path,makewannierhr,metric,nullifybzmesh,plotbands
!!      print_bandstructure,print_sigma_perturbative,printwandata,prompt
!!      readwandata,set2unit,wanmatinterpol,wannierinterpol,wrtout
!!
!! SOURCE

#if defined HAVE_CONFIG_H
#include "config.h"
#endif

subroutine get_rhor(fname,accesswff,localrdwf,nspden,nfft_asked,ngfft_asked,paral_kgb,MPI_enreg,rhor_out)

 use defs_basis
 use defs_datatypes
 use m_errors,   only : assert
 use m_io_tools, only : flush_unit, get_unit

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_01manage_mpi
 use interfaces_11util
 use interfaces_12geometry
 use interfaces_13io_mpi
 use interfaces_14iowfdenpot
 use interfaces_15gw, except_this_one => get_rhor
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: accesswff,localrdwf,paral_kgb,nfft_asked,nspden
 character(len=fnlen),intent(in) :: fname
 type(MPI_type),intent(inout) :: MPI_enreg
!arrays
 real(dp),intent(out) :: rhor_out(nfft_asked,nspden)
 integer,intent(in) :: ngfft_asked(18)

!Local variables-------------------------------
!scalars
 integer :: unt,ios,rdwr,fform,rdwrpaw,nfft_found,accessfil,i1f,i2f,i3f
 integer :: ispden,isppol,ikibz,nband_k,idx
 integer :: cplex,optin,optout
 real(dp) :: etotal,nelect,ratio,ucvol 
 logical :: interpolate,ltest,Iread,master_casts
 type(Hdr_type) :: Hdr
 type(Dataset_type) :: Dtset
 character(len=500) :: msg
!arrays
 integer :: ngfft_found(18)
 real(dp) :: gmet(3,3),gprimd(3,3),rmet(3,3)
 real(dp) :: rhogdum(1)
 real(dp),allocatable :: rhor_found(:,:)
 type(Pawrhoij_type),allocatable :: Pawrhoij(:)

! *************************************************************************

 accessfil=0
 if (accesswff==1) accessfil=4
 if (accesswff==2) accessfil=1
 if (accesswff==3) accessfil=3
 !TODO implement NETCDF as well as localrdwf=1

 ltest=(nfft_asked==PRODUCT(ngfft_asked(1:3)))
 call assert(ltest,'FFT parallelism not implemented',__FILE__,__LINE__)

 unt=get_unit()
 open(unit=unt,file=fname,form='unformatted',status='old',iostat=ios)
 if (ios/=0) then 
  write(msg,'(6a)')ch10,&
&  ' get_rhor: ERROR- ',ch10,  &
&  '  Opening file: ',TRIM(fname),' as old'
  call wrtout(std_out,msg,'COLL') 
  call leave_new('COLL')
 end if
 rdwr=1
 call hdr_io(fform,Hdr,rdwr,unt)
 call assert((fform==52),'fform/=52; '//TRIM(fname)//' is not a density file',&
& __FILE__,__LINE__)

 ngfft_found(1:3) =Hdr%ngfft(1:3)  
 ngfft_found(4)   =2*(ngfft_found(1)/2)+1 ! 4:18 are not used in ioarr 
 ngfft_found(5)   =2*(ngfft_found(2)/2)+1 ! but they are used in fourdp
 ngfft_found(6)   =ngfft_found(3)
 ngfft_found(7:18)=ngfft_asked(7:18)
 nfft_found=PRODUCT(Hdr%ngfft(1:3))    ! TODO complex densities are not treated.

 interpolate=ANY(ngfft_found(1:3)/=ngfft_asked(1:3))

 rdwr=1 ; rdwrpaw=Hdr%usepaw !; if(dtfil%ireadwf/=0) rdwrpaw=0
 allocate(Pawrhoij(Hdr%natom*Hdr%usepaw*rdwrpaw))

 !We hack a bit Dtset, wavelets are not treated.
 Dtset%nspden=Hdr%nspden ; Dtset%usewvl=0
 call assert((Hdr%nspden==nspden),'nspden differ',__FILE__,__LINE__)

 if (.not.interpolate) then
  call ioarr(accessfil,rhor_out,Dtset,etotal,fform,fname,Hdr,MPI_enreg,nfft_found,Pawrhoij,rdwr,rdwrpaw,ngfft_found)
 else 
  allocate(rhor_found(nfft_found,nspden))
  call ioarr(accessfil,rhor_found,Dtset,etotal,fform,fname,Hdr,MPI_enreg,nfft_found,pawrhoij,rdwr,rdwrpaw,ngfft_found)

  ! * Get number of electrons, used to renormalize rhor after the interpolation.
  ! * Cannot use znucl because we might have additional charge or alchemy.
  nelect = get_nelect_from_Hdr(Hdr)

  write(msg,'(a)')' get_rhor : FFT meshes differ, performing FFT interpolation. '
  call wrtout(std_out,msg,'COLL') 

  cplex =1 
  optin =0 ! input is taken from rhor
  optout=0 ! output is only in real space but I might add an option to return rhog_out

  call four_intepol(cplex,nspden,optin,optout,nfft_found,ngfft_found,nfft_asked,ngfft_asked,&
&  paral_kgb,MPI_enreg,rhor_found,rhor_out,rhogdum,rhogdum)

  deallocate(rhor_found)

  ! === Renormalize charge to avoid errors due to the interpolation ===
  ! * Do this only for NC since for PAW we should add the onsite contribution.
  if (Hdr%usepaw==0) then
   call metric(gmet,gprimd,-1,rmet,Hdr%rprimd,ucvol)
   ratio=nelect/( SUM(rhor_out(:,1))*ucvol/PRODUCT(ngfft_asked(1:3)) )
   rhor_out(:,:)=rhor_out(:,:)*ratio
   write(msg,'(a,f8.2,a,f8.4)')' Expected nelect: ',nelect,' renormalization: ',ratio
   call wrtout(std_out,msg,'COLL')
  end if

 end if ! Did ngfft_asked agree with ngfft_found?

 close(unt)

 ! === Free memory ===
 call hdr_clean(Hdr)
 call rhoij_free(Pawrhoij) ; deallocate(Pawrhoij)

end subroutine get_rhor
!!***

!!****f* ABINIT/four_intepol
!! NAME
!! four_intepol
!!
!! FUNCTION
!!  Perform a Fourier interpolation.
!!  Just a wrapper for transgrid, the table giving the correspondence between the coarse
!!  and the mesh FFT grids are constructed inside the routine. This allows to specify an 
!!  arbitrary FFT mesh to be used for the interpolation. Besides the routine works also in 
!!  case of NC calculations since it does not require Pawfgr.
!!
!! COPYRIGHT
!!  Copyright (C) 2007-2009 ABINIT group (MG)
!!  This file is distributed under the terms of the
!!  GNU General Public License, see ~abinit/COPYING
!!  or http://www.gnu.org/copyleft/gpl.txt .
!!
!! INPUTS
!! cplex=1 if rhor[f] is real, 2 if rhor[f] is complex
!! MPI_enreg<MPI_type>=Information about MPI parallelization
!! nspden=number of spin-density components
!! nfft_in =number of points in the input FFT box (WARNING no FFT parallelism)
!! nfft_out=number of points in the output FFT box
!! ngfft_in(18)=all needed information about 3D FFT, for the input grid
!! ngfft_out(18) =all needed information about 3D FFT, for the output grid
!! optin= 0: input density/potential is taken from rhor_in(:,nspden)
!!        1: input density/potential is taken from rhog_in(:)     (ispden=1)
!!                                             and rhor_in(:,2:4) (ispden=2,3,4)
!! optout= 0: output density/potential is given in r space in rhor_out(:,nspden)
!!         1: output density/potential is given in r space in rhor_out(:,nspden)
!!                                          and in g space in rhog_out(:)
!! ngfft_asked(18)=All info on the required FFT mesh.
!! paral_kgb=Flag related to the kpoint-band-fft parallelism (FIXME not implemented)
!!  
!! OUTPUT
!!  rhor_out(cplex*nfft_out,nspden)=output density/potential in r space on the required FFT mesh.
!!  if optout=1:
!!   rhog_out(2,nfftc)=Fourier transform of output density/potential on the coarse grid
!!
!! NOTES
!!
!! PARENTS
!!      gw_tools,rdqps
!!
!! CHILDREN
!!      abi_etsf_get_qp,bstruct_clean,bstruct_init,copy_bandstructure
!!      destroy_bz_mesh_type,destroy_sigma_results,destroycrystal
!!      destroywandata,findk,get_dos,hdr_check,hdr_clean,initkmesh,make_mesh
!!      make_path,makewannierhr,metric,nullifybzmesh,plotbands
!!      print_bandstructure,print_sigma_perturbative,printwandata,prompt
!!      readwandata,set2unit,wanmatinterpol,wannierinterpol,wrtout
!!
!! SOURCE

#if defined HAVE_CONFIG_H
#include "config.h"
#endif

subroutine four_intepol(cplex,nspden,optin,optout,nfft_in,ngfft_in,nfft_out,ngfft_out,&
& paral_kgb,MPI_enreg,rhor_in,rhor_out,rhog_in,rhog_out)

 use defs_basis
 use defs_datatypes
 use m_errors, only : assert

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_15paw
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: cplex,nspden,optin,optout
 integer,intent(in) :: nfft_in,nfft_out,paral_kgb
 type(MPI_type),intent(inout) :: MPI_enreg
!arrays
 integer,intent(in) :: ngfft_in(18),ngfft_out(18)
 real(dp),intent(inout) :: rhor_in(cplex*nfft_in,nspden)
 real(dp),intent(inout) :: rhog_in(2,nfft_in)
 real(dp),intent(out) :: rhor_out(cplex*nfft_out,nspden)
 real(dp),intent(out) :: rhog_out(2,nfft_out)

!Local variables ---------------------------------------
!scalars
 integer :: nfftf,nfftc,nfftc_tot,nfftf_tot,optgrid
 logical :: ltest
 type(Pawfgr_type) :: Pawfgr
!arrays
 integer :: ngfftc(18),ngfftf(18)

! *************************************************************************

 !=== FFT parallelism not implemented ===
 ltest=(paral_kgb==0)
 call assert(ltest,'paral_kgb/=0 not implemented',__FILE__,__LINE__)

 ! ================================
 ! === Which one is the coarse? ===
 ! ================================
 if (nfft_out>=nfft_in) then
  ! * From coarse to fine grid.
  ! * Take into account is meshes are the same.
  !   In this case we call transgrid anyway because of optout, optin.
  nfftf    =nfft_out 
  ngfftf(:)=ngfft_out(:)
  nfftf_tot =PRODUCT(ngfft_out(1:3))

  nfftc    =nfft_in 
  ngfftc(:)=ngfft_in(:)
  nfftc_tot =PRODUCT(ngfft_in (1:3))

  Pawfgr%usefinegrid=1 
  if (ALL(ngfft_in(1:3)==ngfft_out(1:3))) Pawfgr%usefinegrid=0 
  optgrid=1
 else
  ! * From fine towards coarse.
  nfftf    =nfft_in 
  ngfftf(:)=ngfft_in(:)
  nfftf_tot =PRODUCT(ngfft_in(1:3))

  nfftc    =nfft_out 
  ngfftc(:)=ngfft_out(:)
  nfftc_tot =PRODUCT(ngfft_out (1:3))
  Pawfgr%usefinegrid=1 
  optgrid=-1
 end if

 allocate(Pawfgr%coatofin(nfftc_tot),Pawfgr%fintocoa(nfftf_tot))
 call indgrid(Pawfgr%coatofin,Pawfgr%fintocoa,nfftc_tot,nfftf_tot,ngfftc,ngfftf)

 Pawfgr%mgfft   =MAXVAL (ngfftf(1:3))
 Pawfgr%nfft    =PRODUCT(ngfftf(1:3)) !no FFT parallelism!
 Pawfgr%ngfft(:)=ngfftf(:)

 Pawfgr%mgfftc   =MAXVAL (ngfftc(1:3))
 Pawfgr%nfftc    =PRODUCT(ngfftc(1:3)) !no FFT parallelism!
 Pawfgr%ngfftc(:)=ngfftc(:)

 if (optgrid==1) then 
  call transgrid(cplex,MPI_enreg,nspden,optgrid,optin,optout,paral_kgb,Pawfgr,rhog_in ,rhog_out,rhor_in ,rhor_out)
 else
  call transgrid(cplex,MPI_enreg,nspden,optgrid,optin,optout,paral_kgb,Pawfgr,rhog_out,rhog_in ,rhor_out,rhor_in)
 end if

 deallocate(Pawfgr%coatofin,Pawfgr%fintocoa)

end subroutine four_intepol
!!***

!!****f* ABINIT/my_GWannier
!! NAME
!! my_GWannier
!!
!! FUNCTION
!!
!! INPUTS
!!
!! OUTPUT
!!
!! PARENTS
!!      mrgscr
!!
!! CHILDREN
!!      abi_etsf_get_qp,bstruct_clean,bstruct_init,copy_bandstructure
!!      destroy_bz_mesh_type,destroy_sigma_results,destroycrystal
!!      destroywandata,findk,get_dos,hdr_check,hdr_clean,initkmesh,make_mesh
!!      make_path,makewannierhr,metric,nullifybzmesh,plotbands
!!      print_bandstructure,print_sigma_perturbative,printwandata,prompt
!!      readwandata,set2unit,wanmatinterpol,wannierinterpol,wrtout
!!
!! SOURCE
subroutine my_GWannier

 use defs_basis
 use defs_datatypes
 use m_crystal
 use m_bz_mesh
 use m_wannier2abinit
 use m_gwdefs,         only : GW_TOLQ, GW_TOLQ0
 use m_io_tools,       only : prompt, get_unit
 use m_errors,         only : assert
 use m_numeric_tools,  only : set2unit

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_01manage_mpi
 use interfaces_11util
 use interfaces_12geometry
 use interfaces_14iowfdenpot
 use interfaces_15gw, except_this_one => my_GWannier
!End of the abilint section

 implicit none

!Local variables-------------------------------
!scalars
 integer,parameter :: prtvol=0
 integer :: unt,accesswff,ndiv_small,mwan,ib,ii,nkbounds,nkintp,jj,info,isp,kk,idx
 integer :: ikbz,ikibz,iwan,io,restartpaw,restart,nkdense,nwan,ikcalc,itask
 integer :: bantot,isppol,iband,unt_sig,ikw,ikgw,spad,nband_k,method,ikpt
 real(dp) :: ucvol,spectot,ww,fixmom,broad,dosdeltae
 complex(dpc) :: zz,qp_ene
 logical :: ltest,use_afm,use_tr,ishermitian,witheader,found
 character(len=50) :: task
 character(len=500) :: msg,frmt
 character(len=fnlen) :: fname_wan,fname_bands,fname_qps,fname_spect,fname_dos
 type(BZ_mesh_type) :: Kmesh,Kmesh4dos,Kpath
 type(Crystal_structure) :: Cryst
 type(WannierData) :: WData
 type(Bandstructure_type) :: QP_BSt,KS_BSt,QPtmp,QP4Wan,KS_intp,QP_intp
 type(Sigma_results) :: Sr
 type(Hdr_type) :: Hdr
!arrays
 integer :: ngfft(18),kptrlatt(3,3),mp_dense(3)
 integer :: G0(3),kptrlatt4dos(3,3)
 integer,allocatable :: ndiv(:),kcalc2bz(:),wan2gw(:),gw2wan(:)
 integer,allocatable :: nband(:) 
 integer,pointer :: dummy(:),npwarr(:),istwfk(:)
 real(dp) :: gmet(3,3),gprimd(3,3),rmet(3,3),shift(3)
 real(dp) :: kwan(3),kgw(3)
 real(dp),allocatable :: kbounds(:,:),kdense(:,:),wtk(:)
 real(dp),pointer :: kintp(:,:)
 real(dp),allocatable :: matrix_out(:,:,:,:),matrix_in(:,:,:,:),A_test(:,:,:,:,:),Aw(:,:)
 real(dp),allocatable :: doccde(:),eigen(:),occfact(:)
 logical,allocatable :: bandselect(:,:,:)
 complex(dpc),allocatable :: Aw_in(:,:,:,:),Aw_out(:,:,:,:),A_fine(:,:,:,:)
 complex(dpc),allocatable :: ze0_in(:,:,:,:),ze0_out(:,:,:,:),zz_fine(:,:,:,:)
 complex(dpc),allocatable :: qpe_in(:,:,:,:),qpe_out(:,:,:,:),qpe_fine(:,:,:,:)

! *************************************************************************

 ! === Read WAN file containing the unitary transformation ===
 call prompt(' Enter name of the WAN file: ',fname_wan)

 accesswff=0
 call ReadWanData(WData,fname_wan,accesswff)

 call PrintWanData(WData,header='HELLO WAN',prtvol=1)

 call metric(gmet,gprimd,-1,rmet,WData%Hdr%rprimd,ucvol)
 
 ! === Construct the Hamiltonian in the Wannier representation ===
 call prompt(' Enter kptrlatt used during the Wannierization: ',kptrlatt)

 call MakeWannierHR(WData,kptrlatt)

 ! === Read GW results in NETCDF format ===
 call prompt(' Enter name of the QPS file: ',fname_qps)

 call abi_etsf_get_QP(Sr,KS_BSt,Hdr,Cryst,fname_qps)

 ! This is just to do a check, the file format is wrong!
 call hdr_check(1002,1002,Hdr,WData%Hdr,'COLL',restart,restartpaw)

 ! === Copy the KS bands to QP_Bst thus initializing the object ===
 ! * Apply GW corrections, this works only if full k-mesh
 call copy_bandstructure(KS_Bst,QP_BSt)
 do isppol=1,QP_BSt%nsppol
  do ikpt=1,QP_BSt%nkpt
   do iband=Sr%minbnd(ikpt),Sr%maxbnd(ikpt)
    QP_BSt%eig(iband,ikpt,isppol)=QP_BSt%eig(iband,ikpt,isppol)+REAL(Sr%degw(iband,ikpt,isppol))
   end do
  end do
 end do

 ! === Initialize the K-mesh (the same as that used of wavefunctions ====
 call InitKmesh(Hdr%nkpt,Hdr%kptns,Cryst,Kmesh,prtvol)

 ! === Find index of GW points in full BZ ===
 ! TODO the indexing has to be changed, everything should be defined in the IBZ or BZ
 allocate(kcalc2bz(Sr%nkcalc))
 call findk(Sr%nkcalc,Kmesh%nbz,Sr%xkcalc,Kmesh%bz,kcalc2bz,0)

 do isppol=1,Sr%nsppol
  do ikcalc=1,Sr%nkcalc
   ikibz=Kmesh%tab(kcalc2bz(ikcalc)) ! Irred k-point for GW
   do iband=Sr%minbnd(ikcalc),Sr%maxbnd(ikcalc) 
    witheader=.FALSE. ; if (iband==Sr%minbnd(ikcalc)) witheader=.TRUE.
    call print_Sigma_perturbative(Sr,ikibz,iband,isppol,witheader=witheader)
   end do
  end do
 end do

 ! === Extract GW bands for Wannier ===
 ! Here I am assuming that nwan is equal to the GW bands
 ! better integration will be done in the following.
 if (ANY(Sr%minbnd/=Sr%minbnd(1))) STOP 'GW bands are not constant'
 if (ANY(Sr%maxbnd/=Sr%maxbnd(1))) STOP 'GW bands are not constant'
 if (ANY(Sr%minbnd(:)/=1)) STOP 'GW bands should start at 1'
 !if (ANY(Sr%maxbnd(:)/=WData%mwan)) STOP 'GW bands should be equal to mwan'

 ii=Sr%minbnd(1) ; jj=Sr%maxbnd(1)
 if ((jj-ii+1)/=WData%mwan) STOP 'GW and Wannier bands do not agree'

 allocate(bandselect(QP_BSt%mband,QP_BSt%nkpt,QP_BSt%nsppol))
 bandselect=.FALSE. ; bandselect(ii:jj,:,:)=.TRUE.

 QPtmp = SelectBands(QP_BSt,bandselect=bandselect)
 deallocate(bandselect)

 call print_bandstructure(QPtmp)

 use_tr=(Cryst%timrev==2) 
 use_afm=Cryst%use_antiferro

 QP4Wan = ExpandBands(QPtmp,WData%nkpt,WData%Hdr%kptns,use_tr,use_afm,Cryst%nsym,Cryst%symrec,Cryst%symafm,info)
 if (info/=0) then
  write(msg,'(a)')' GW and Wannier mesh do not agree, check k-meshes'
  call wrtout(std_out,msg,'COLL') ; call leave_new('COLL')
 end if
 call bstruct_clean(QPtmp)

 ! === Start Wannier Interpolation ===
 call prompt(' Enter task, 1 for Bands, 2 for DOS: ',itask)
 task='BANDS' ; if (itask/=1) task='DOS'

 call NullifyBzMesh(Kpath)
 call NullifyBzMesh(Kmesh4dos)

 select case (task)

 case ('BANDS')
  call prompt(' Enter number of boundaries for path    : ',nkbounds)
  allocate(kbounds(3,nkbounds)) 
  call prompt(' Enter boundaries [r.l.u]               : ',kbounds)
  call prompt(' Enter the smallest number of divisions : ',ndiv_small)

  ! === Interpolate GW corrections ===
  ! * Make path in reciprocal space
  allocate(ndiv(nkbounds-1))
  call make_path(nkbounds,kbounds,gmet,'G',ndiv_small,ndiv,nkintp,kintp)
  deallocate(kbounds)

  allocate(wtk(nkintp))
  wtk=one/nkintp 

  Kpath%nibz=nkintp
  allocate(Kpath%ibz(3,Kpath%nibz),Kpath%wt(Kpath%nibz))
  Kpath%ibz=kintp
  Kpath%wt=one/Kpath%nibz

 ! === Interpolate KS bands first ===
 ! * All the data we need are already in WData
 call WannierInterpol(WData,Kpath,KS_intp)
 fname_bands='KS_bands'
 call PlotBands(KS_intp,gmet,fname_bands)

 case ('DOS')
  ! === Construct mesh in BZ, fold it back to IBZ and find weights ===
  call prompt(' Enter MP divisions for the dense k-mesh: ',mp_dense)
  call set2unit(kptrlatt4dos)
  kptrlatt4dos(1,1)=mp_dense(1)
  kptrlatt4dos(2,2)=mp_dense(2)
  kptrlatt4dos(3,3)=mp_dense(3)
  call prompt(' Enter shift for mesh (no more than one) ',shift)

  call make_mesh(Kmesh4dos,Cryst,kptrlatt4dos,1,shift)
  nkintp=Kmesh4dos%nibz 
  allocate(kintp(3,nkintp))
  kintp=Kmesh4dos%ibz 

  allocate(wtk(nkintp))
  wtk=Kmesh4dos%wt 

  ! === interpolate KS bands on Kmesh4dos === 
  call WannierInterpol(WData,Kmesh4dos,KS_intp)
  fname_dos='KS_tetra'//'_DOS' ; method=2
  broad=0.01/Ha_eV ; dosdeltae=0.001/Ha_eV
  call get_dos(KS_intp,Kmesh4dos,method,fname_dos,broad,dosdeltae)
  
 case default
  stop 'Wrong task'
 end select

 mwan=WData%mwan
 if (mwan/=QP4Wan%mband) stop 'mwan/=QP4Wan%mband'
 allocate(matrix_in (mwan,mwan,WData%nkpt,WData%nsppol))
 allocate(matrix_out(mwan,mwan,nkintp    ,WData%nsppol))
 matrix_in=czero
 do ib=1,WData%mwan
  matrix_in(ib,ib,:,:)=QP4Wan%eig(ib,:,:)
 end do

 ishermitian=.TRUE.
 call WanMatInterpol(WData,QP4Wan%nkpt,nkintp,kintp,ishermitian,QP4Wan%nsppol,mwan,matrix_in,matrix_out)

 ! Now matrix_out has the interpolated band structure.
 !do ii=1,nkintp
 ! write(54,'(8f8.4)')(matrix_out(ib,ib,ii,1),ib=1,mwan) 
 ! write(55,'(8f8.4)')(KS_intp%eig(ib,ii,1),ib=1,mwan) 
 !end do

 ! === Initialize new object containing the interpolated GW bands ===
 bantot=QP4Wan%nsppol*nkintp*mwan
 allocate(doccde(bantot),eigen(bantot),occfact(bantot))
 doccde(:)=zero ; eigen(:)=zero ; occfact(:)=zero 

 allocate(nband(nkintp*QP4Wan%nsppol))
 do isppol=1,QP4Wan%nsppol
  spad=(isppol-1)*nkintp
  nband(spad+1:spad+nkintp)=WData%nwan(isppol)
 end do
 allocate(dummy(nkintp)) ;  dummy=1
 istwfk => dummy
 npwarr => dummy

 call bstruct_init(bantot,QP_intp,QP4Wan%nelect,doccde,eigen,istwfk,kintp,nband,nkintp,npwarr,&
& QP4Wan%nsppol,QP4Wan%nspinor,QP4Wan%tphysel,QP4Wan%tsmear,QP4Wan%occopt,occfact,wtk)

 nullify(istwfk,npwarr)
 deallocate(dummy,nband,occfact,eigen,doccde)
 deallocate(wtk)

 ! === Put interpolated GW energies ===
 do isppol=1,QP_intp%nsppol
  do ikpt=1,QP_intp%nkpt
   nband_k=QP_intp%nband(ikpt+(isppol-1)*QP_intp%nkpt)
   do ib=1,nband_k
    QP_intp%eig(ib,ikpt,isppol)=matrix_out(ib,ib,ikpt,isppol)
   end do
  end do
 end do
 deallocate(matrix_out)
 deallocate(matrix_in )

 !this is dangerous
 !fixmom=99.99_dp
 !call update_occ(QP_intp,fixmom)

 if (task=='BANDS') then
  fname_bands='gw_bands'
  call PlotBands(QP_intp,gmet,fname_bands)
 else if (task=='DOS') then
  broad=0.01/Ha_eV ; dosdeltae=0.001/Ha_eV
  fname_dos='GW_gauss'//'_DOS' ; method=1 
  !call get_dos(QP_intp,Kmesh4dos,method,fname_dos,broad,dosdeltae)
  fname_dos='GW_tetra'//'_DOS' ; method=2
  call get_dos(QP_intp,Kmesh4dos,method,fname_dos,broad,dosdeltae)
 end if

 ! this does not work yet
 !fixmom=99.99_dp
 !call update_occ(KS_intp,fixmom)
 !call ReportGap(KS_intp,'Ks Gaps',unit=std_out)

 !call update_occ(QP_intp,fixmom)
 !call ReportGap(QP_intp,'QP Gaps',unit=std_out)
 

 ! =========================================
 ! === Interpolate the Spectral function ===
 ! =========================================
 if (Sr%nomega_r > 0) then

  write(*,*)' Sr% has spectral function results '
  stop 'uncomment me'
  call prompt(' Enter MP divisions for the dense k-mesh: ',mp_dense)

  nkdense=PRODUCT(mp_dense)
  allocate(kdense(3,nkdense))
  idx=0
  do kk=0,mp_dense(3)-1
   do jj=0,mp_dense(2)-1
    do ii=0,mp_dense(1)-1
     idx=idx+1
     kdense(1,idx)=DBLE(ii)/mp_dense(1)
     kdense(2,idx)=DBLE(jj)/mp_dense(2)
     kdense(3,idx)=DBLE(kk)/mp_dense(3)
    end do
   end do
  end do

  allocate(Aw_in (mwan,mwan,Kmesh%nbz,Sr%nsppol))
  allocate(Aw_out(mwan,mwan,nkdense,Sr%nsppol))
  allocate(A_fine(mwan,mwan,Sr%nsppol,Sr%nomega_r))
  allocate(A_test(mwan,mwan,Sr%nkcalc,Sr%nsppol,Sr%nomega_r))
  Aw_in =czero
  Aw_out=czero
  A_fine=czero
  
  unt_sig=1000

  do ikcalc=1,Sr%nkcalc
   ikibz=Kmesh%tab(kcalc2bz(ikcalc))
   write(unt_sig,'("# k = ",3f10.6)')Sr%xkcalc(:,ikcalc)
   write(unt_sig,'("# b = ",2i10)')Sr%minbnd(ikcalc),Sr%maxbnd(ikcalc)
   do isp=1,Sr%nsppol
    do io=1,Sr%nomega_r
     write(unt_sig,'(100(e11.5,2x))')&
&      REAL(Sr%omega_r(io))*Ha_eV,&
&      (REAL(Sr%sigxcme(ib,ikibz,io,isp))*Ha_eV,&
&      AIMAG(Sr%sigxcme(ib,ikibz,io,isp))*Ha_eV,&
&      one/pi*ABS(AIMAG(Sr%sigcme(ib,ikibz,io,isp)))&
&      /( (REAL(Sr%omega_r(io)-Sr%hhartree(ib,ib,ikibz,isp)-Sr%sigxcme(ib,ikibz,io,isp)))**2&
&        +(AIMAG(Sr%sigcme(ib,ikibz,io,isp)))**2) /Ha_eV,&
&      ib=Sr%minbnd(ikcalc),Sr%maxbnd(ikcalc))
    end do
   end do
  end do

  if (WData%nkpt/=Kmesh%nbz) STOP "nkpt differ"
  allocate(wan2gw(Kmesh%nbz))
  allocate(gw2wan(Kmesh%nbz))

  do ikw=1,WData%Hdr%nkpt
   kwan=WData%Hdr%kptns(:,ikw)
   found=.FALSE.
   do ikgw=1,Kmesh%nbz
    kgw=Kmesh%bz(:,ikgw)
    found=isamek(kwan,kgw,G0)
    if (found) then 
     wan2gw(ikw)=ikgw
     gw2wan(ikgw)=ikw
     EXIT
    end if
   end do
   if (.not.found) stop
  end do

  ! Interpolate spectral function by interpolating energies and zz separately.
  allocate(ze0_in (mwan,mwan,Kmesh%nbz,Sr%nsppol))
  allocate(ze0_out(mwan,mwan,nkdense,Sr%nsppol))
  ze0_in =czero
  ze0_out=czero

  allocate(qpe_in (mwan,mwan,Kmesh%nbz,Sr%nsppol))
  allocate(qpe_out(mwan,mwan,nkdense,Sr%nsppol))
  qpe_in =czero
  qpe_out=czero

  do isp=1,Sr%nsppol

   do ib=Sr%minbnd(1),Sr%maxbnd(1) ! here check that bands are constant
    iwan=ib-Sr%minbnd(1)+1
    do ikbz=1,WData%nkpt !here take care of the k-points, maybe I have to reorder
     ii=wan2gw(ikbz)
     ikibz=Kmesh%tab(ii)

     ze0_in(iwan,iwan,ikbz,isp) = Sr%ze0(ib,ikibz,isp)
     qpe_in(iwan,iwan,ikbz,isp) = Sr%egw(ib,ikibz,isp)
    end do !ikbz
   end do !ib
  end do !isp

  ishermitian=.FALSE.
  call WanMatInterpol(WData,QP4Wan%nkpt,nkdense,kdense,ishermitian,QP4Wan%nsppol,mwan,ze0_in,ze0_out)
  call WanMatInterpol(WData,QP4Wan%nkpt,nkdense,kdense,ishermitian,QP4Wan%nsppol,mwan,qpe_in,qpe_out)

  deallocate(ze0_in,qpe_in)

  allocate(Aw(Sr%nomega_r,Sr%nsppol))
  Aw=zero

  do isp=1,Sr%nsppol
   do io=1,Sr%nomega_r 
    ww=Sr%omega_r(io)
    do ikbz=1,nkdense
     do ib=1,mwan
      zz    =ze0_out(ib,ib,ikbz,isp)
      qp_ene=qpe_out(ib,ib,ikbz,isp)
      Aw(io,isp)=Aw(io,isp)+ make_spectral_diag(ww,qp_ene,zz)
     end do
    end do
   end do
  end do

  Aw=Aw/nkdense
  do io=1,Sr%nomega_r
   write(88,'(100(e11.5,2x))')Sr%omega_r(io)*Ha_eV,(Aw(io,isp)/Ha_eV,isp=1,Sr%nsppol)
  end do

  deallocate(Aw)
 end if

 ! === Free memory ===
 deallocate(kcalc2bz)
 deallocate(kintp)

 call DestroyWanData(WData)
 call destroy_Sigma_results(Sr)
 call DestroyCrystal(Cryst)
 call destroy_BZ_mesh_type(Kmesh)
 call destroy_BZ_mesh_type(Kmesh4dos)
 call destroy_BZ_mesh_type(Kpath)
 call bstruct_clean(KS_BSt) 
 call bstruct_clean(QP_BSt) 
 !£call bstruct_clean(QP_intp) 
 !£call bstruct_clean(KS_intp) 
 call hdr_clean(Hdr)

 STOP 'myGWannier OK'

end subroutine my_GWannier
!!***

!!****f* ABINIT/make_spectral_diag
!! NAME
!! make_spectral_diag
!!
!! FUNCTION
!!
!! INPUTS
!!
!! OUTPUT
!!
!! PARENTS
!!  Will be filled automatically by the parent script
!!
!! CHILDREN
!!  Will be filled automatically by the parent script
!!
!! SOURCE
function make_spectral_diag(omega,qp_ene,zz) result(spect_funct)

 use defs_basis
 use defs_datatypes, only : Sigma_results

 implicit none

!Arguments ------------------------------------
!scalars
 !integer,intent(in) :: iband,ikibz,isppol,iomega
 !type(Sigma_results),intent(in) :: Sr
 real(dp),intent(in) :: omega
 complex(dpc),intent(in) :: qp_ene,zz
!arrays
 real(dp) :: spect_funct

!Local variables-------------------------------
!scalars
 real(dp) :: re_z,im_z,re_qpe,im_qpe,ww

! *************************************************************************

 ww=omega
 re_z=REAL (zz)
 im_z=AIMAG (zz)
 re_qpe=REAL (qp_ene)
 im_qpe=AIMAG(qp_ene)

 spect_funct = piinv* ABS(im_qpe*re_qpe-(ww-re_qpe)*im_z)/ ((ww-re_qpe)**2 + im_qpe**2)
 
end function make_spectral_diag
!!***
