!{\src2tex{textfont=tt}}
!!****f* ABINIT/bstruct_init
!! NAME
!! bstruct_init
!!
!! FUNCTION
!! This subroutine initializes the bandstructure structured datatype
!!
!! COPYRIGHT
!! Copyright (C) 2002-2009 ABINIT group (XG, MG)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~abinit/COPYING
!! or http://www.gnu.org/copyleft/gpl.txt .
!! For the initials of contributors, see ~abinit/doc/developers/contributors.txt.
!!
!! INPUTS
!! bantot=total number of bands (=sum(nband(:))
!! doccde(bantot)=derivative of the occupation numbers with respect to the energy (Ha)
!! eig(bantot)=eigenvalues (hartree)
!! istwfk(nkpt)=parameter that describes the storage of wfs.
!! kptns(3,nkpt)=k points in terms of recip primitive translations
!! nband(nkpt*nsppol)=number of bands
!! nelect=Number of electrons.
!! nkpt=number of k points
!! npwarr(nkpt)=number of planewaves at each k point
!! nsppol=1 for unpolarized, 2 for spin-polarized
!! nspinor=Number of spinor components.
!! occopt=Occupation options (see input variable)
!! occ(bantot)=occupation numbers
!! tphysel=Physical temperature (input variable)
!! tsmear=Temperature of smearing.
!! wtk(nkpt)=weight assigned to each k point
!!
!! OUTPUT
!! bstruct<bstruct_type>=the bandstructure datatype
!!
!! SIDE EFFECTS
!!  %entropy and %fermie initialized to zero.
!!
!! TODO 
!!  It should become a module.
!!
!! PARENTS
!!      gstate,gw_tools,loper3,mlwfovlp_qp,newsp,nonlinear,respfn
!!      setup_screening,setup_sigma
!!
!! CHILDREN
!!      get_eneocc_vect,prteigrs
!!
!! SOURCE

#if defined HAVE_CONFIG_H
#include "config.h"
#endif

subroutine bstruct_init(bantot,bstruct,nelect,doccde,eig,istwfk,kptns,&
& nband,nkpt,npwarr,nsppol,nspinor,tphysel,tsmear,occopt,occ,wtk)

 use defs_basis
 use defs_datatypes

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_15gw, except_this_one => bstruct_init
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: bantot,nkpt,nsppol,nspinor,occopt
 real(dp),intent(in) :: nelect,tphysel,tsmear
 type(Bandstructure_type),intent(out) :: bstruct
!arrays
 integer,intent(in) :: istwfk(nkpt),nband(nkpt*nsppol),npwarr(nkpt)
 real(dp),intent(in) :: doccde(bantot),eig(bantot),kptns(3,nkpt),occ(bantot)
 real(dp),intent(in) :: wtk(nkpt)
! *************************************************************************

 ! Copy the scalars
 ! MG TODO here there is a inconsistency in the way occ are treated in the header
 ! (only the states used, bantot. are saved, and the way occ. and energies
 ! are passed to routines (mband,nkpt,nsppol). It might happen that bantot<mband*nktp*nsppol
 ! this should not lead to problems since arrays are passed by reference
 ! anyway the treatmen of these arryas have to be rationalized
 bstruct%bantot =bantot
 bstruct%mband  =MAXVAL(nband(1:nkpt*nsppol))
 bstruct%nkpt   =nkpt
 bstruct%nspinor=nspinor
 bstruct%nsppol =nsppol
 bstruct%occopt =occopt
 
 bstruct%entropy=zero  ! Initialize results
 bstruct%fermie =zero  ! Initialize results
 bstruct%nelect =nelect
 bstruct%tphysel=tphysel
 bstruct%tsmear =tsmear

 ! Allocate the components
 allocate(bstruct%nband(nkpt*nsppol))
 allocate(bstruct%istwfk(nkpt))
 allocate(bstruct%npwarr(nkpt))
 allocate(bstruct%kptns(3,nkpt))
 allocate(bstruct%eig   (bstruct%mband,nkpt,nsppol))
 allocate(bstruct%occ   (bstruct%mband,nkpt,nsppol))
 allocate(bstruct%doccde(bstruct%mband,nkpt,nsppol))
 allocate(bstruct%wtk(nkpt))

 ! Copy the arrays
 bstruct%nband(1:nkpt*nsppol)=nband(1:nkpt*nsppol)
 bstruct%istwfk(1:nkpt)      =istwfk(1:nkpt)
 bstruct%npwarr(1:nkpt)      =npwarr(1:nkpt)
 bstruct%kptns(1:3,1:nkpt)   =kptns(1:3,1:nkpt)

 ! In bstruct, energies and occupations are stored in a matrix (mband,nkpt,nsppol).
 ! put_eneocc_vect is used to reshape the values stored in vectorial form.
 call put_eneocc_vect(bstruct,'eig',   eig   ) 
 call put_eneocc_vect(bstruct,'occ',   occ   ) 
 call put_eneocc_vect(bstruct,'doccde',doccde) 

 bstruct%wtk(1:nkpt)=wtk(1:nkpt)

end subroutine bstruct_init
!!***

!!****f* ABINIT/bstruct_clean
!! NAME
!! bstruct_clean
!!
!! FUNCTION
!! This subroutine deallocates the components of the bandstructure structured datatype
!!
!! INPUTS
!!  bstruct<bandstructure_type>=The data type to be deallocated.
!!
!! OUTPUT
!!  Deallocate the dynamic arrays in the bandstructure type.
!!  (only deallocate)
!!
!! PARENTS
!!      gstate,gw_tools,loper3,mlwfovlp_qp,newsp,nonlinear,respfn,screening
!!      sigma
!!
!! CHILDREN
!!      get_eneocc_vect,prteigrs
!!
!! SOURCE

#if defined HAVE_CONFIG_H
#include "config.h"
#endif

subroutine bstruct_clean(bstruct)

 use defs_basis
 use defs_datatypes

 implicit none

!Arguments ------------------------------------
!scalars
 type(Bandstructure_type),intent(inout) :: bstruct
! *************************************************************************

!DEBUG
!write(6,*)' bstruct_clean : enter'
!stop
!ENDDEBUG

!Deallocate all components of bstruct
 deallocate(bstruct%istwfk)
 deallocate(bstruct%nband)
 deallocate(bstruct%npwarr)
 deallocate(bstruct%kptns)
 deallocate(bstruct%eig)
 deallocate(bstruct%occ)
 deallocate(bstruct%doccde)
 deallocate(bstruct%wtk)

end subroutine bstruct_clean
!!***

!!****f* ABINIT/copy_bandstructure
!! NAME
!!  copy_bandstructure
!!
!! FUNCTION
!! This subroutine performs a deep copy of a bandstructure datatype.
!! All the associated pointers in the input object will be copied preserving the shape.
!! If a pointer in BSt_in happens to be not associated, the corresponding
!! pointer in the copied object will be nullified.
!!
!! INPUTS
!!  BSt_in<bandstructure_type>=The data type to be copied.
!!
!! OUTPUT
!!  BSt_cp<bandstructure_type>=The copy.
!!
!! TODO 
!!  To be on the safe side one should nullify all pointers in the bandstructure_type 
!!  in the creation method. We have to follow F90 specifications and the initial status 
!!  of a pointer is not defined. This might lead to problem in deep_copy.
!!
!! PARENTS
!!      gw_tools,screening,sigma
!!
!! CHILDREN
!!      get_eneocc_vect,prteigrs
!!
!! SOURCE

subroutine copy_bandstructure(BSt_in,BSt_cp)

 use defs_basis
 use defs_datatypes
 use m_copy, only : deep_copy

 implicit none

!Arguments ------------------------------------
!scalars
 type(Bandstructure_type),intent(in)  :: BSt_in
 type(Bandstructure_type),intent(out) :: BSt_cp

! *********************************************************************

 ! === Copy scalars ===
 BSt_cp%bantot  = BSt_in%bantot  
 BSt_cp%mband   = BSt_in%mband
 BSt_cp%nkpt    = BSt_in%nkpt    
 BSt_cp%nspinor = BSt_in%nspinor 
 BSt_cp%nsppol  = BSt_in%nsppol  
 BSt_cp%occopt  = BSt_in%occopt  

 BSt_cp%entropy = BSt_in%entropy 
 BSt_cp%fermie  = BSt_in%fermie  
 BSt_cp%nelect  = BSt_in%nelect  
 BSt_cp%tphysel = BSt_in%tphysel 
 BSt_cp%tsmear  = BSt_in%tsmear  

 ! === Copy pointers ===
 call deep_copy( BSt_in%istwfk, BSt_cp%istwfk)
 call deep_copy( BSt_in%nband , BSt_cp%nband )     
 call deep_copy( BSt_in%npwarr, BSt_cp%npwarr)    

 call deep_copy( BSt_in%kptns , BSt_cp%kptns ) 
 call deep_copy( BSt_in%eig   , BSt_cp%eig   )  
 call deep_copy( BSt_in%occ   , BSt_cp%occ   )   
 call deep_copy( BSt_in%doccde, BSt_cp%doccde)   
 call deep_copy( BSt_in%wtk   , BSt_cp%wtk   )

end subroutine copy_bandstructure
!!***

!!****f* ABINIT/print_bandstructure
!! NAME
!! print_bandstructure
!!
!! FUNCTION
!! Print the content of the object.
!!
!! INPUTS
!!  BSt<bandstructure_type>The type containing the data.
!!  unit[optional]=unit number (std_out if None)
!!  header[optional]=title for info
!!  prtvol[optional]=Verbosity level (0 if None)
!!  mode_paral[optional]=Either 'COLL' or 'PERS' ('COLL' if None)
!!
!! OUTPUT
!!  Only writing
!!
!! PARENTS
!!      gw_tools
!!
!! CHILDREN
!!      get_eneocc_vect,prteigrs
!!
!! SOURCE

subroutine print_bandstructure(BSt,header,unit,prtvol,mode_paral)

 use defs_basis
 use defs_datatypes

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_01manage_mpi
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,optional,intent(in) :: prtvol,unit
 character(len=*),optional,intent(in) :: header
 character(len=4),optional,intent(in) :: mode_paral
 type(bandstructure_type),intent(in) :: BSt

!Local variables-------------------------------
 integer :: isppol,ikpt,iband,unt,my_prtvol,ii
 character(len=4) :: mode
 character(len=500) :: msg
! *************************************************************************

 unt=std_out ; if (PRESENT(unit      )) unt      =unit
 my_prtvol=0 ; if (PRESENT(prtvol    )) my_prtvol=prtvol
 mode='COLL' ; if (PRESENT(mode_paral)) mode     =mode_paral

 msg=' ==== Info on the bandstructure_type ==== '
 if (PRESENT(header)) msg=' ==== '//TRIM(ADJUSTL(header))//' ==== '
 call wrtout(unt,msg,mode)

 write(msg,'(5(a,i5,a))')&
& '  Number of spinorial components ...... ',BSt%nspinor,ch10,&
& '  Number of ind. spin polarizations ... ',BSt%nsppol,ch10,&
& '  Number of k-points in the IBZ ....... ',BSt%nkpt,ch10,&
& '  Maximum number of bands ............. ',BSt%mband,ch10,&
& '  Occupation option ................... ',BSt%occopt,ch10
 call wrtout(unt,msg,mode)

 write(msg,'(a,f14.2,a,4(a,f14.6,a))')&
& '  Number of valence electrons ......... ',BSt%nelect,ch10,&
& '  Fermi level  ........................ ',BSt%fermie,ch10,&
& '  Entropy ............................. ',BSt%entropy,ch10,&
& '  Tsmear value ........................ ',BSt%tsmear,ch10,&
& '  Tphysel value ....................... ',BSt%tphysel,ch10
 call wrtout(unt,msg,mode)

 if (my_prtvol>0) then

  if (BSt%nsppol==1)then
   write(msg,'(a,i4,a)')' New occ. numbers for occopt= ',BSt%occopt,' , spin-unpolarized case. '
   call wrtout(unt,msg,mode)
  end if
  do isppol=1,BSt%nsppol
   if (BSt%nsppol==2) then
    write(msg,'(a,i4,a,i2)')' New occ. numbers for occopt= ',BSt%occopt,' spin ',isppol
    call wrtout(unt,msg,mode)
   end if

   do ikpt=1,BSt%nkpt
    write(msg,'(2a,i4,a,3f12.6,a,f6.3)')ch10,&
 &   ' k-point number ',ikpt,') ',BSt%kptns(:,ikpt),'; ',BSt%wtk(ikpt)
    call wrtout(unt,msg,mode)
    do ii=1,BSt%nband(ikpt+(isppol-1)*BSt%nkpt)
     write(msg,'(3(f7.3,1x))')BSt%eig(ii,ikpt,isppol)*Ha_eV,BSt%occ(ii,ikpt,isppol),BSt%doccde(ii,ikpt,isppol)
     call wrtout(unt,msg,mode)
    end do

   end do !ikpt
  end do !isppol

  !TODO add additional info useful for debugging)
  !istwfk(:), nband(:)

 end if !my_prtvol

end subroutine print_bandstructure
!!***

!!****f* ABINIT/unpack_eneocc
!! NAME
!! unpack_eneocc
!!
!! FUNCTION
!!  Helper function to do a reshape of (energies|occupancies|derivate of occupancies)
!!  initially stored in a vector. Return a 3D array index by (iband,ikpt,isppol) 
!!
!! INPUTS
!!  nkpt=number of k-points
!!  nsppol=number of spin polarizations
!!  mband=Max number of bands over k-points (just to dimension the output)
!!  nbands(nkpt*nsppol)=Number of bands at eack k and spin
!!  bantot=Total number of bands
!!  vect(bantot)=The input values to reshape
!!
!! OUTPUT
!!  array3d(mband,nkpt,nsppol)=Arrays containing the values of vect. 
!!   Note that the first dimension is usually larger than the 
!!   number of bands really used for a particular k-point and spin.
!!
!! PARENTS
!!      bstruct_init
!!
!! CHILDREN
!!      get_eneocc_vect,prteigrs
!!
!! SOURCE

subroutine unpack_eneocc(nkpt,nsppol,mband,nband,bantot,vect,array3d)

 use defs_basis

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: nkpt,nsppol,mband,bantot
!arrays
 integer,intent(in) :: nband(nkpt*nsppol)
 real(dp),intent(in) :: vect(bantot)
 real(dp),intent(out) :: array3d(mband,nkpt,nsppol)

!Local variables-------------------------------
 integer :: isppol,ikpt,iband,idx
! *************************************************************************

 array3d(:,:,:)=zero
 idx=0
 ! elements in vect are packed in the first positions.
 do isppol=1,nsppol
  do ikpt=1,nkpt
   do iband=1,nband(ikpt+(isppol-1)*nkpt)
    idx=idx+1
    array3d(iband,ikpt,isppol)=vect(idx)
   end do
  end do
 end do

end subroutine unpack_eneocc
!!***

!!****f* ABINIT/pack_eneocc
!! NAME
!! pack_eneocc
!!
!! FUNCTION
!!  Helper function to do a reshape of (energies|occupancies|derivate of occupancies)
!!  initially stored in a 3D arrays returning a vector. 
!!
!! INPUTS
!!  nkpt=number of k-points
!!  nsppol=number of spin polarizations
!!  mband=Max number of bands over k-points (just to dimension the output)
!!  nbands(nkpt*nsppol)=Number of bands at eack k and spin
!!  bantot=Total number of bands
!!  array3d(mband,nkpt,nsppol)=Arrays containing the values to reshape.
!!
!! OUTPUT
!!  vect(bantot)=The input values stored in vector mode. Only the values really
!!   considered at each k-point and spin are copied.
!!
!! PARENTS
!!      bstruct_init
!!
!! CHILDREN
!!      get_eneocc_vect,prteigrs
!!
!! SOURCE

subroutine pack_eneocc(nkpt,nsppol,mband,nband,bantot,array3d,vect)

 use defs_basis

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: nkpt,nsppol,mband,bantot
!arrays
 integer,intent(in) :: nband(nkpt*nsppol)
 real(dp),intent(out) :: vect(bantot)
 real(dp),intent(in) :: array3d(mband,nkpt,nsppol)

!Local variables-------------------------------
 integer :: isppol,ikpt,iband,idx

! *************************************************************************

 vect(:)=zero
 idx=0
 do isppol=1,nsppol
  do ikpt=1,nkpt
   do iband=1,nband(ikpt+(isppol-1)*nkpt)
    idx=idx+1
    vect(idx)=array3d(iband,ikpt,isppol)
   end do
  end do
 end do

end subroutine pack_eneocc 
!!***

!!****f* ABINIT/get_eneocc_vect
!! NAME
!! get_eneocc_vect
!!
!! FUNCTION
!!  Retrieve energies or occupations from a Bandstructure structure accessing by name. 
!!  Results are reported in a vector to facilitate the interface with other abinit routines.
!!
!! INPUTS
!!  BSt<bandstructure_type>The type containing the data.
!!  arr_name=The name of the quantity to retrieve. Allowed values are
!!   == "eig"    == For the eigenvalues.
!!   == "occ"    == For the occupation numbers.
!!   == "doccde" == For the derivative of the occupancies wrt the energy.
!!
!! OUTPUT
!!  vect(BSt%bantot)=The values required.
!!
!! PARENTS
!!      bstruct_init,hdr_init,sigma
!!
!! CHILDREN
!!      get_eneocc_vect,prteigrs
!!
!! SOURCE

subroutine get_eneocc_vect(BSt,arr_name,vect)

 use defs_basis
 use defs_datatypes

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_01manage_mpi
 use interfaces_15gw, except_this_one => get_eneocc_vect
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 character(len=*),intent(in) :: arr_name
 type(Bandstructure_type),intent(in) :: BSt
 real(dp),intent(out) :: vect(BSt%bantot)

!Local variables-------------------------------
 integer :: nkpt,nsppol,mband,bantot
 character(len=500) :: msg
! *************************************************************************

 mband =BSt%mband 
 bantot=BSt%bantot
 nkpt  =BSt%nkpt
 nsppol=BSt%nsppol

 select case (arr_name)
  case ('occ')
   call pack_eneocc(nkpt,nsppol,mband,BSt%nband,bantot,BSt%occ,vect)
  case ('eig')
   call pack_eneocc(nkpt,nsppol,mband,BSt%nband,bantot,BSt%eig,vect)
  case ('doccde')
   call pack_eneocc(nkpt,nsppol,mband,BSt%nband,bantot,BSt%doccde,vect)
  case default
   write(msg,'(a)')' get_eneocc_vect : Wrong value of arr_name = '//TRIM(arr_name)
   call wrtout(ab_out,msg,'COLL') 
   call leave_new('COLL')
 end select 

end subroutine get_eneocc_vect
!!***

!!****f* ABINIT/put_eneocc_vect
!! NAME
!! put_eneocc_vect
!!
!! FUNCTION
!!  Update the energies or the occupations stored a Bandstructure structure. 
!!  The input values are stored in a vector according to the abinit convention
!!  In the data type, on the contrary,  we use 3D arrays (mband,nkpt,nsspol) 
!!  which are much easier to use inside loops.
!!
!! INPUTS
!!  vect(BSt%bantot)=The new values to be stored in the structure.
!!  arr_name=The name of the quantity to be saved (CASE insensitive). 
!!  Allowed values are
!!   == "eig"    == For the eigenvalues.
!!   == "occ"    == For the occupation numbers.
!!   == "doccde" == For the derivative of the occupancies wrt the energy.
!!
!! OUTPUT
!!  See SIDE EFFECTS
!!
!! SIDE EFFECTS
!!  BSt<bandstructure_type>=The object with updated values depending on the value of arr_name
!!
!! PARENTS
!!      bstruct_init
!!
!! CHILDREN
!!      get_eneocc_vect,prteigrs
!!
!! SOURCE

subroutine put_eneocc_vect(BSt,arr_name,vect) 

 use defs_basis
 use defs_datatypes
 use m_fstrings, only : tolower

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_01manage_mpi
 use interfaces_15gw, except_this_one => put_eneocc_vect
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 character(len=*),intent(in) :: arr_name
 type(Bandstructure_type),intent(inout) :: BSt
 real(dp),intent(in) :: vect(BSt%bantot)

!Local variables-------------------------------
 integer :: nkpt,nsppol,mband,bantot
 character(len=500) :: msg
! *************************************************************************

 mband =BSt%mband 
 bantot=BSt%bantot
 nkpt  =BSt%nkpt
 nsppol=BSt%nsppol

 select case (tolower(arr_name))
  case ('occ')
   call unpack_eneocc(nkpt,nsppol,mband,BSt%nband,bantot,vect,BSt%occ)
  case ('eig')
   call unpack_eneocc(nkpt,nsppol,mband,BSt%nband,bantot,vect,BSt%eig)
  case ('doccde')
   call unpack_eneocc(nkpt,nsppol,mband,BSt%nband,bantot,vect,BSt%doccde)
  case default
   write(msg,'(a)')' put_eneocc_vect : Wrong value of arr_name = '//TRIM(arr_name)
   call wrtout(ab_out,msg,'COLL') 
   call leave_new('COLL')
 end select 

end subroutine put_eneocc_vect
!!***

!!****f* ABINIT/get_bandenergy
!! NAME
!! get_bandenergy
!!
!! FUNCTION
!!  Return the band energy (weighted sum of occupied eigenvalues)
!!
!! INPUTS
!!
!! OUTPUT
!!
!! NOTES
!! TODO Likely this expression is not accurate since it is not variatonal
!!  One should use 
!!   band_energy = \int e N(e) de   for e<Ef , where N(e) is the e-DOS
!!
!! PARENTS
!!
!! CHILDREN
!!
!! SOURCE

function get_bandenergy(BSt) result(band_energy)

 use defs_basis
 use defs_datatypes

 implicit none

!Arguments ------------------------------------
!scalars
 type(Bandstructure_type),intent(in) :: BSt
 real(dp) :: band_energy

!Local variables-------------------------------
 integer :: isppol,ikibz,iband,nband_k
 real(dp) :: wtk
! *********************************************************************

 band_energy=zero
 do isppol=1,BSt%nsppol
  do ikibz=1,BSt%nkpt
   wtk=BSt%wtk(ikibz)
   nband_k=BSt%nband(ikibz+(isppol-1)*BSt%nkpt) 
   band_energy = band_energy + wtk*SUM( BSt%eig(1:nband_k,ikibz,isppol)*BSt%occ(1:nband_k,ikibz,isppol) )
  end do
 end do

end function get_bandenergy
!!***

!!****f* ABINIT/get_valence_idx
!! NAME
!!  get_valence_idx
!!
!! FUNCTION
!!  For each k-point and spin polarisation, report: 
!!   The index of the valence in case of Semiconductors.
!!   The index of the band at the Fermi energy+toldfe
!!
!! INPUTS
!!  BSt<bandstructure_type>=The object describing the band structure.
!!  tol_fermi[optional]
!!
!! OUTPUT
!!
!! PARENTS
!!
!! CHILDREN
!!
!! SOURCE

function get_valence_idx(BSt,tol_fermi) result(val_idx)

 use defs_basis
 use defs_datatypes

 implicit none

!Arguments ------------------------------------
!scalars
 real(dp),optional,intent(in) :: tol_fermi
 type(Bandstructure_type),intent(in) :: BSt
!arrays
 integer :: val_idx(BSt%nkpt,BSt%nsppol)

!Local variables-------------------------------
 integer :: iband,ikpt,isppol,idx,nband_k 
 real(dp) :: tol_
 character(len=500) :: msg

! *************************************************************************

 tol_=tol6 ; if (PRESENT(tol_fermi)) tol_=tol_fermi

 do isppol=1,BSt%nsppol
  do ikpt=1,BSt%nkpt
   nband_k=BSt%nband(ikpt+(isppol-1)*BSt%nkpt)

   idx=0
   do iband=1,nband_k
    if (BSt%eig(iband,ikpt,isppol) > BSt%fermie+ABS(tol_)) then
    idx=iband ; EXIT
    end if
   end do
   val_idx(ikpt,isppol)=idx-1
   if (idx==1) val_idx(ikpt,isppol)=idx
   if (idx==0) val_idx(ikpt,isppol)=nband_k

  end do
 end do

end function get_valence_idx
!!***

!!****f* ABINIT/apply_scissor
!! NAME
!!  apply_scissor
!!
!! FUNCTION
!!  Apply a scissor operator of amplitude scissor_energy.
!!
!! INPUTS
!!  scissor_energy=The energy shift
!!
!! OUTPUT
!!
!! SIDE EFFECT
!!  BSt<bandstructure_type>=The following quantities are modified:
!!   %eig(mband,nkpt,nsppol)=The band structure after the application of the scissor operator
!!   %fermi_energy
!!
!! PARENTS
!!      screening
!!
!! CHILDREN
!!      get_eneocc_vect,prteigrs
!!
!! SOURCE

subroutine apply_scissor(BSt,scissor_energy)

 use defs_basis
 use defs_datatypes

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_01manage_mpi
 use interfaces_15gw, except_this_one => apply_scissor
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 real(dp),intent(in) :: scissor_energy
 type(Bandstructure_type),intent(inout) :: BSt

!Local variables-------------------------------
 integer :: iband,ikpt,isppol,ival,nband_k 
 real(dp) :: fixmom_
 character(len=500) :: msg
!arrays
 integer :: val_idx(BSt%nkpt,BSt%nsppol)
! *************************************************************************

 ! === Get the valence band index for each k and spin ===
 val_idx(:,:) = get_valence_idx(BSt)

 do isppol=1,BSt%nsppol
  if (ANY(val_idx(:,isppol)/=val_idx(1,isppol))) then
   write(msg,'(4a,i2,a)')ch10,&
&   ' apply_scissor : COMMENT - ',ch10,&
&   '  Trying to apply a scissor operator on a metallic band structure for spin = ',isppol,&
&   '  I suppose you know what you are doing, continuing anyway! '
   call wrtout(std_out,msg,'COLL') 
   call wrtout(ab_out, msg,'COLL') 
   !Likely newocc will stop, unless the system is semimetallic ?
   !call leave_new('COLL')
  end if
 end do

 ! === Apply the scissor ===
 do isppol=1,BSt%nsppol
  do ikpt=1,BSt%nkpt
   nband_k=BSt%nband(ikpt+(isppol-1)*BSt%nkpt)
   ival=val_idx(ikpt,isppol)

   if (nband_k>=ival+1) then
    BSt%eig(ival+1:,ikpt,isppol) = BSt%eig(ival+1:,ikpt,isppol)+scissor_energy
   else 
    write(msg,'(5a,4(a,i4))')ch10,&
&    ' apply_scissor : COMMENT - ',ch10,&
&    '  Not enough bands to apply the scissor operator. ',ch10,&
&    '  spin = ',isppol,' ikpt = ',ikpt,' nband_k = ',nband_k,' but valence index = ',ival
    call wrtout(std_out,msg,'COLL') 
   end if

  end do
 end do

 ! === Recalculate the fermi level and occ. factors ===
 ! * For Semiconductors only the Fermi level is changed (in the middle of the new gap) 
 fixmom_=99.99_dp !; if (PRESENT(fixmom)) fixmom_=fixmom
 call update_occ(BSt,fixmom_)

end subroutine apply_scissor
!!***

!!****f* ABINIT/get_occupied
!! NAME
!!  get_occupied
!!
!! FUNCTION
!!  For each k-point and spin polarisation, report the band index
!!  after which the occupation numbers are less than tol_occ.
!!
!! INPUTS
!!  BSt<bandstructure_type>=The object describing the band structure.
!!  tol_occ[Optional]=Tollerance on the occupation factors.
!!
!! OUTPUT
!!
!! NOTES
!!  We assume that the occupation factors monotonically decrease as a function of energy.
!!  This is not always true for eavery smearing technique implemented in Abinit.
!!
!! PARENTS
!!
!! CHILDREN
!!
!! SOURCE

function get_occupied(BSt,tol_occ) result(occ_idx)

 use defs_basis
 use defs_datatypes

 implicit none

!Arguments ------------------------------------
!scalars
 real(dp),optional,intent(in) :: tol_occ
 type(Bandstructure_type),intent(in) :: BSt
!arrays
 integer :: occ_idx(BSt%nkpt,BSt%nsppol)

!Local variables-------------------------------
 integer :: iband,ikpt,isppol,idx,nband_k 
 real(dp) :: tol_
 character(len=500) :: msg

! *************************************************************************

 tol_=tol8 ; if (PRESENT(tol_occ)) tol_=tol_occ

 do isppol=1,BSt%nsppol
  do ikpt=1,BSt%nkpt
   nband_k=BSt%nband(ikpt+(isppol-1)*BSt%nkpt)

   idx=0
   do iband=1,nband_k
    if (BSt%occ(iband,ikpt,isppol)<ABS(tol_)) then
     idx=iband ; EXIT
    end if
   end do
   occ_idx(ikpt,isppol)=idx-1
   if (idx==1) occ_idx(ikpt,isppol)=idx
   if (idx==0) occ_idx(ikpt,isppol)=nband_k

  end do
 end do

end function get_occupied
!!***

!!****f* ABINIT/get_minmax
!! NAME
!!  get_minmax
!!
!! FUNCTION
!!  Report the min and max value over k-points and bands of (eig|occ|doccde) for each 
!!  spin. Cannot use F90 array syntax due to the internal storage used in abinit.
!!
!! INPUTS
!!  BSt<bandstructure_type>=The object describing the band structure.
!!  arr_name=The name of the array whose min and Max value has to be calculated.
!!   Possible values: 'occ', 'eig' 'doccde'
!!
!! OUTPUT
!! minmax(2,BSt%nsppol)=For each spin the min and max value of the quantity specified by "arr_name"
!!
!! NOTES
!!
!! PARENTS
!!
!! CHILDREN
!!
!! SOURCE

function get_minmax(BSt,arr_name) result(minmax)

 use defs_basis
 use defs_datatypes
 use m_fstrings, only : tolower

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_01manage_mpi
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 type(Bandstructure_type),intent(in) :: BSt
 character(len=*),intent(in) :: arr_name
!arrays
 real(dp) :: minmax(2,BSt%nsppol)

!Local variables-------------------------------
!scalars
 integer :: iband,ikpt,isppol,idx,nband_k 
 real(dp) :: datum
 character(len=500) :: msg
!arrays
 real(dp),pointer :: rdata(:,:,:)

! *************************************************************************

 select case (tolower(arr_name))
  case ('occ')
   rdata => BSt%occ
  case ('eig')
   rdata => BSt%eig
  case ('doccde')
   rdata => BSt%doccde
  case default
   write(msg,'(a)')' get_minmax : Wrong value of arr_name = '//TRIM(arr_name)
   call wrtout(std_out,msg,'COLL') 
   call leave_new('COLL')
 end select 

 minmax(1,:)=greatest_real
 minmax(2,:)=smallest_real
 
 do isppol=1,BSt%nsppol
  do ikpt=1,BSt%nkpt
   nband_k=BSt%nband(ikpt+(isppol-1)*BSt%nkpt)
   do iband=1,nband_k
    datum=rdata(iband,ikpt,isppol)
    minmax(1,isppol)=MIN(minmax(1,isppol),datum)
    minmax(2,isppol)=MAX(minmax(2,isppol),datum)
   end do
  end do
 end do

end function get_minmax
!!***

!!****f* ABINIT/get_FS
!! NAME
!!  get_FS
!!
!! FUNCTION
!!
!! INPUTS
!!  BSt<bandstructure_type>=The object describing the band structure.
!!   %nsppol
!!  ewin
!!
!! OUTPUT
!!  nFSkpt(nsppol)
!!  fs2ibz(:,:)
!!  fs_bands(:,:,:)
!!
!! PARENTS
!!
!! CHILDREN
!!      get_eneocc_vect,prteigrs
!!
!! SOURCE

subroutine get_FS(BSt,ewin,nFSkpt,fs2ibz,fs_bands)

 use defs_basis
 use defs_datatypes

 implicit none

!Arguments ------------------------------------
!scalars
 real(dp),intent(in) :: ewin
 type(Bandstructure_type),intent(in) :: BSt
 integer,intent(out) ::  nFSkpt(BSt%nsppol)
 integer,pointer :: fs2ibz(:,:),fs_bands(:,:,:)
!arrays

!Local variables-------------------------------
 integer :: iband,ikpt,ikfs,isppol,nband_k,nbnds_FS,i1,i2,nt
 real(dp) :: fermie,ene
 logical :: isinside
 character(len=500) :: msg
!arrays
 integer,allocatable :: FSkpt(:,:,:)

! *************************************************************************

 allocate(FSkpt(3,BSt%nkpt,BSt%nsppol))
 FSkpt=0 ; nFSkpt(:)=0

 fermie=BSt%fermie

 do isppol=1,BSt%nsppol
  do ikpt=1,BSt%nkpt
   nband_k=BSt%nband(ikpt+(isppol-1)*BSt%nkpt)

   i1=0 ; i2=0
   do iband=1,nband_k
    ene=BSt%eig(iband,ikpt,isppol)
    if (ene> fermie+ABS(ewin)) EXIT !iband
    if (ene>=fermie-ABS(ewin).and.ene<=fermie+ABS(ewin)) then
     i1=iband 
     i2=iband ; isinside=.TRUE.
     do while (isinside.and.i2<nband_k)
      i2=i2+1
      ene=BSt%eig(i2,ikpt,isppol)
      isinside=(ene<=fermie+ABS(ewin))
     end do
     EXIT !iband
    end if
   end do

   nbnds_FS=i2-i1 ; if (isinside) nbnds_FS=nbnds_FS+1
   if (nbnds_FS/=0) then 
    nFSkpt(isppol)=nFSkpt(isppol)+1 ; nt=nFSkpt(isppol)
    FSkpt(1,nt,isppol) = ikpt
    FSkpt(2,nt,isppol) = i1
    FSkpt(3,nt,isppol) = i2
   end if

  end do !ikpt
 end do !isppol

 nt=MAXVAL(nFSkpt)
 allocate(fs2ibz(nt,BSt%nsppol),fs_bands(2,nt,BSt%nsppol))
 fs2ibz=0 ; fs_bands=0

 do isppol=1,BSt%nsppol
  do ikfs=1,nFSkpt(isppol)
   fs2ibz  (  ikfs,isppol)=FSkpt(1,ikfs,isppol)
   fs_bands(1,ikfs,isppol)=FSkpt(2,ikfs,isppol)
   fs_bands(2,ikfs,isppol)=FSkpt(3,ikfs,isppol)
  end do
 end do

 deallocate(FSkpt)

end subroutine get_FS
!!***


!!****f* ABINIT/get_dos
!! NAME
!!  get_dos
!!
!! FUNCTION
!!  Calculate the electronic density of states.
!!  Ideally the routine should report an object DOS that can be written on file using a method dump_dos 
!!
!! INPUTS
!!  BSt<Bandstructure_type>=The object describing the band structure.
!!  Kmesh<BZ_mesh_type>=Strcuture defining the BZ sampling.
!!  method
!!  fildos
!!  broad
!!  dosdeltae
!!
!! OUTPUT
!!  For the moment results are printed on an external file.
!!
!! PARENTS
!!      gw_tools
!!
!! CHILDREN
!!      get_eneocc_vect,prteigrs
!!
!! SOURCE

subroutine get_dos(BSt,Kmesh,method,fildos,broad,dosdeltae)

 use defs_basis
 use defs_datatypes
 use m_errors,       only : assert
 use m_io_tools,     only : get_unit
 use m_bz_mesh,      only : isamek

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_01manage_mpi
 use interfaces_14occeig
 use interfaces_15gw, except_this_one => get_dos
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: method
 real(dp),intent(in) :: dosdeltae,broad
 character(len=fnlen),intent(in) :: fildos
 type(Bandstructure_type),intent(in)  :: BSt
 type(BZ_mesh_type),intent(in)  :: Kmesh

!Local variables-------------------------------
!scalars
 integer,parameter :: option4dos=2
 integer :: unitdos,isppol,iband,ikpt
 integer :: mtetra,ntetra,nene,iene
 real(dp),parameter :: buffer=0.01_dp
 real(dp) :: nelect_out,entropy_out,max_occ,rcvol,enex
 real(dp) :: enemax,enemin,deltaene,integral_DOS
 character(len=500) :: msg
 logical :: ltest
!arrays
 integer,pointer :: tetra_mult(:),tetra_full(:,:,:)
 integer :: G0(3)
 real(dp) :: minmax(2,BSt%nsppol)
 real(dp) :: gprimd(3,3),k1(3),k2(3)
 real(dp),allocatable :: eigen(:),doccde(:),occ(:)
 real(dp),allocatable :: dtweightde(:,:),integ_dos(:,:)
 real(dp),allocatable :: partial_dos(:,:)
 real(dp),allocatable :: tmp_eigen(:),total_dos(:)
 real(dp),allocatable :: total_dos_m(:,:),total_dos_paw1(:,:)
 real(dp),allocatable :: total_dos_pawt1(:,:),total_integ_dos(:)
 real(dp),allocatable :: total_integ_dos_m(:,:),tweight(:,:)

! *********************************************************************

#if defined DEBUG_MODE
 write(msg,'(a)')' get_dos : enter '
 call wrtout(std_out,msg,'COLL')
#endif

 select case (method)

 case (1) 
  ! === DOS with broadening ===
  ! The unit is closed in getnel, not so elegant!
  unitdos=get_unit()
  open(unit=unitdos,file=fildos,status='unknown',form='formatted')

  ! To be consistent with the interface of getnel 
  max_occ=two/(BSt%nspinor*BSt%nsppol)  ! Will not work in the fixed moment case ????? MG should check why
  allocate(doccde(BSt%mband*BSt%nkpt*BSt%nsppol))  
  allocate(eigen (BSt%mband*BSt%nkpt*BSt%nsppol))  
  allocate(occ   (BSt%mband*BSt%nkpt*BSt%nsppol))  
  doccde(:)=zero
  call get_eneocc_vect(BSt,'eig',eigen)
  call get_eneocc_vect(BSt,'occ',occ  )

  deltaene=dosdeltae ; if (ABS(deltaene)<tol10) deltaene=0.001_dp

  ! Use Gaussian smearing. TODO should be input
  call getnel(doccde,deltaene,eigen,entropy_out,BSt%fermie,max_occ,BSt%mband,BSt%nband,&
&  nelect_out,BSt%nkpt,BSt%nsppol,occ,3,option4dos,BSt%tphysel,broad,unitdos,BSt%wtk)

  !if (nelect_out/=BSt%nelect) STOP
  deallocate(doccde,eigen,occ)

 case (2)
  ! === Tetrahedron Method ===

  ltest=(BSt%nkpt>=2)
  call assert(ltest,'At least 2 points to use tetrahedrons',__FILE__,__LINE__)
  ltest=(Kmesh%nshift==1)
  call assert(ltest,'For tetrahedrons, nshift must be 1',__FILE__,__LINE__)
  ltest=ALL(BSt%nband==BSt%nband(1))
  call assert(ltest,'For tetrahedrons, nband(:) must be constant',__FILE__,__LINE__)

  gprimd = Kmesh%gprimd
  rcvol = abs (gprimd(1,1)*(gprimd(2,2)*gprimd(3,3)-gprimd(3,2)*gprimd(2,3)) &
&  -gprimd(2,1)*(gprimd(1,2)*gprimd(3,3)-gprimd(3,2)*gprimd(1,3)) &
&  +gprimd(3,1)*(gprimd(1,2)*gprimd(2,3)-gprimd(2,2)*gprimd(1,3)))

  ! Choose the lower and upper energies and Extend the range to a nicer value
  minmax=get_minmax(BSt,'eig')
  enemin=MINVAL(minmax(1,:))-buffer
  enemax=MAXVAL(minmax(2,:))+buffer
  enemax=0.1_dp*CEILING(enemax*ten)
  enemin=0.1_dp*FLOOR(enemin*ten)

  !Choose the energy increment
  deltaene=dosdeltae
  if (ABS(deltaene)<tol10) deltaene=0.00005_dp ! Higher resolution possible (and wanted) for tetrahedron
  nene=NINT((enemax-enemin)/deltaene)+1

  ! === kpoints must be the same and with the same ordering ===
  ltest=(BSt%nkpt==Kmesh%nibz)
  call assert(ltest,'Mismatch in number of k-points',__FILE__,__LINE__)
  ltest=.TRUE.
  do ikpt=1,BSt%nkpt
   k1=BSt%kptns(:,ikpt)
   k2=Kmesh%ibz(:,ikpt)
   ltest=(ltest.and.isamek(k1,k2,G0))
  end do
  call assert(ltest,'k-points in Kmesh and BSt are not equivalent.',__FILE__,__LINE__)

  ltest=(associated(Kmesh%tetra_mult).and.associated(Kmesh%tetra_full))
  call assert(ltest,'tetra_full or tetra_mult not associated.',__FILE__,__LINE__)

  ! Here ndosfraction has been removed, should provide a low 
  ! level routine to perform integrations with tetrahedrons of matrix elements provided by the user

  mtetra=Kmesh%ntetra_irr
  ntetra=Kmesh%ntetra_irr
  tetra_mult => Kmesh%tetra_mult
  tetra_full => Kmesh%tetra_full

  max_occ=two/(BSt%nspinor*BSt%nsppol)  ! Will not work in the fixed moment case ????? MG should check why
  allocate(tweight(BSt%nkpt,nene),dtweightde(BSt%nkpt,nene))
  allocate(total_dos(nene),total_integ_dos(nene))

  unitdos=get_unit()
  open(unit=unitdos,file=fildos,status='unknown',form='formatted')

  ! For each spin and band, interpolate over kpoints, 
  ! calculate integration weights and DOS contribution.
  do isppol=1,BSt%nsppol

   total_dos(:)=zero
   total_integ_dos(:)= zero

   if (BSt%nsppol==2) then
    if (isppol==1) write(msg,'(a,16x,a)')  '#','Spin-up DOS'
    if (isppol==2) write(msg,'(a,16x,a)')  '#','Spin-dn DOS'
    call wrtout(unitdos,msg,'COLL')
   end if

   allocate(tmp_eigen(BSt%nkpt))

   do iband=1,BSt%nband(1)

    ! For each band get its contribution
    tmp_eigen(:)=Bst%eig(iband,:,isppol)
    !  
    ! === Calculate integration weights at each irred kpoint ===
    ! * Blochl et al PRB 49 16223 ===
    call get_tetra_weight(tmp_eigen,enemin,enemax,max_occ,mtetra,nene,Bst%nkpt,ntetra,rcvol,&
&    tetra_full,tetra_mult,tweight,dtweightde,Kmesh%tetra_vol)

    ! === Calculate DOS and integrated DOS projected with the input dos_fractions ===
!    call get_dos_1band (dos_fractions(:,iband,isppol,:),enemin,enemax,&
!&    integ_dos(:,:,iband),nene,Bst%nkpt,ndosfraction,partial_dos(:,:,iband),tweight,dtweightde)

    do iene=1,nene
     do ikpt=1,Bst%nkpt
      total_dos(iene)       = total_dos(iene) + dtweightde(ikpt,iene) 
      total_integ_dos(iene) = total_integ_dos(iene) + tweight(ikpt,iene) 
     end do
    end do
   
   end do ! iband
   deallocate(tmp_eigen)

   ! === Write DOS values ===
   write(msg,'(a)')'#index  energy(Ha)     DOS  integrated DOS'
   call wrtout(unitdos,msg,'COLL')

   enex=enemin
   do iene=1,nene
    ! Print the data for this energy. Note the upper limit, to be
    ! consistent with the format. The use of "E" format is not adequate,
    ! for portability of the self-testing procedure.
    ! write(msg, '(i5,f9.4,f14.6)' ) iene-1,enex,total_dos(iene,:)
    write(msg,'(i6,f11.5,2f10.4)')iene-1,enex,min(total_dos(iene),9999.9999_dp),total_integ_dos(iene)
    call wrtout(unitdos,msg,'COLL')
    enex=enex+deltaene
   end do

   !integral_DOS=deltaene*SUM(total_dos(iene,:))
   integral_DOS=total_integ_dos(nene)
   write(msg,'(a,es16.8)')' tetrahedron : integrate to',integral_DOS
   call wrtout(std_out,msg,'COLL')

  end do !isppol

  close(unitdos)

  ! === Free memory ===
  deallocate(total_dos,total_integ_dos)
  deallocate(tweight,dtweightde)

 case default
  write(msg,'(4a)')ch10,&
&  ' get_dos : BUG - ',ch10,&
&  ' Wrong value for method '
  call wrtout(ab_out,msg,'COLL') 
  call leave_new('COLL')
 end select

#if defined DEBUG_MODE
 write(msg,'(a)')' get_dos : ended'
 call wrtout(std_out,msg,'COLL')
#endif

end subroutine get_dos
!!***

!!****f* ABINIT/update_occ
!! NAME
!! update_occ
!!
!! FUNCTION
!! Calculate new occupation numbers, the Fermi level and the Max occupied band index 
!! for each spin channel starting from the the knowledge of eigenvalues.
!!
!! INPUTS
!!  fixmom=if differ from -99.99d0, fix the magnetic moment (in Bohr magneton)
!!  stmbias=
!!  prtvol=Verbosity level (0 for lowest level)
!!  BSt<Bandstructure_type>=Info on the band structure, the smearing technique and the physical temperature used.
!!
!! OUTPUT
!!  see also SIDE EFFECTS.
!!
!! SIDE EFFECTS
!!  === For metallic occupation the following quantites are recalculated ===
!!   %fermie=the new Fermi energy
!!   %entropy=the new entropy associated with the smearing.
!!   %occ(mband,nkpt,nsppol)=occupation numbers
!!   %doccde(mband,nkpt,nsppol)=derivative of occupancies wrt the energy for each band and k point
!!  === In case of semiconductors ===
!!   All the quantitities in BSt are left unchanged with the exception of:
!!   %fermie=Redefined so that it is in the middle of the gap
!!   %entropy=Set to zero
!!
!! PARENTS
!!      bstruct_init,screening,sigma
!!
!! CHILDREN
!!      get_eneocc_vect,prteigrs
!!
!! SOURCE

#if defined HAVE_CONFIG_H
#include "config.h"
#endif

subroutine update_occ(BSt,fixmom,stmbias,prtvol)

 use defs_basis
 use defs_datatypes
 use m_errors, only : assert

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_01manage_mpi
 use interfaces_14occeig
 use interfaces_15gw, except_this_one => update_occ
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 type(Bandstructure_type),intent(inout) :: BSt
 integer,optional,intent(in) :: prtvol
 real(dp),intent(in) :: fixmom
 real(dp),optional,intent(in) :: stmbias
!arrays

!Local variables-------------------------------
!scalars
 integer :: iband,mband,ikibz,nkpt,isppol,nsppol,prtvol_local,nband_k
 real(dp) :: entropy,fermie,stmbias_local,ndiff,cbot,vtop,maxocc
 logical :: ltest
 character(len=100) :: fmt
 character(len=500) :: msg
!arrays
 real(dp) :: nelect_spin(BSt%nsppol),condbottom(BSt%nsppol),valencetop(BSt%nsppol)
 real(dp),allocatable :: doccdet(:),occt(:),eigent(:)

! *************************************************************************

 prtvol_local =0    ; if (PRESENT(prtvol )) prtvol_local =prtvol
 stmbias_local=zero ; if (PRESENT(stmbias)) stmbias_local=stmbias

 ! === Compute number of electrons for each spin channel ===
 nelect_spin(:)=zero 
 do isppol=1,BSt%nsppol
  do ikibz=1,BSt%nkpt
   nband_k=BSt%nband(ikibz+(isppol-1)*BSt%nkpt)
   nelect_spin(isppol)= nelect_spin(isppol) + BSt%wtk(ikibz)*SUM(BSt%occ(1:nband_k,ikibz,isppol))
  end do
 end do

 ndiff=BSt%nelect-SUM(nelect_spin)
 if (prtvol_local>0) then
  write(msg,'(2a,f6.2,2a,f6.4)')ch10,&
&  ' total number of electrons = ',SUM(nelect_spin),ch10,&
&  ' input and calculated no. of electrons differ by ',ndiff 
  call wrtout(std_out,msg,'COLL')
 end if
 ltest=(ABS(ndiff)<5.d-2*BSt%nelect) 
 call assert(ltest,'Too large difference in no. of electrons',__FILE__,__LINE__)

 if (BSt%occopt>=3.and.BSt%occopt<=7) then
  ! === Compute new occupation numbers if occupation is metallic ===
  write(msg,'(a,f9.5)')' metallic scheme, calling newocc with fixmom = ',fixmom
  call wrtout(std_out,msg,'COLL')

  ! The following to be consistent with the interface of newocc.
  mband  = BSt%mband
  nkpt   = BSt%nkpt
  nsppol = BSt%nsppol

  allocate(eigent(mband*nkpt*nsppol))
  call get_eneocc_vect(BSt,'eig',eigent)

  allocate(occt(mband*nkpt*nsppol),doccdet(mband*nkpt*nsppol))

  call newocc(doccdet,eigent,entropy,fermie,fixmom,mband,BSt%nband,&
&  BSt%nelect,BSt%nkpt,BSt%nspinor,BSt%nsppol,occt,BSt%occopt, &
&  prtvol_local,stmbias_local,BSt%tphysel,BSt%tsmear,BSt%wtk)

  ! === Save output in BSt% === 
  BSt%entropy = entropy
  BSt%fermie  = fermie
  call put_eneocc_vect(BSt,'occ'   ,occt   ) 
  call put_eneocc_vect(BSt,'doccde',doccdet) 
  deallocate(eigent,occt,doccdet)

 else 
  ! === Semiconductor or Insulator ===
  maxocc=two/(BSt%nsppol*BSt%nspinor)

  ! * Calculate the valence index for each spin channel.
  do isppol=1,BSt%nsppol
   valencetop(isppol)= smallest_real
   condbottom(isppol)= greatest_real

   do ikibz=1,BSt%nkpt
    nband_k=BSt%nband(ikibz+(isppol-1)*BSt%nkpt) 
    do iband=1,nband_k
     if (BSt%occ(iband,ikibz,isppol)/maxocc>one-tol6 .and. valencetop(isppol)<BSt%eig(iband,ikibz,isppol)) then 
      valencetop(isppol)=BSt%eig(iband,ikibz,isppol)
     end if
     if (BSt%occ(iband,ikibz,isppol)/maxocc<tol6 .and. condbottom(isppol)>BSt%eig(iband,ikibz,isppol)) then 
      condbottom(isppol)=BSt%eig(iband,ikibz,isppol)
     end if
    end do
   end do 

  end do 
  vtop=MAXVAL(valencetop)
  cbot=MINVAL(condbottom)
  write(msg,'(a,f6.2,2a,f6.2)')&
&  ' top of valence       [eV] ',vtop*Ha_eV,ch10,&
&  ' bottom of conduction [eV] ',cbot*Ha_eV
  call wrtout(std_out,msg,'COLL')
  if (BSt%nsppol==2) then 
   if (ABS(vtop-MINVAL(valencetop))>tol6) then 
    write(msg,'(a,i2)')' top of valence is spin ',MAXLOC(valencetop)
    call wrtout(std_out,msg,'COLL')
   end if
   if (ABS(cbot-MAXVAL(condbottom))>tol6) then 
    write(msg,'(a,i2)')' bottom of conduction is spin ',MINLOC(condbottom)
    call wrtout(std_out,msg,'COLL')
   end if
  end if

  ! === Save output === 
  ! Here I dont know if it is better to be consistent with the abinit convention i.e fermi=vtop
  BSt%entropy=zero
  BSt%fermie=(vtop+cbot)/2 
  if (ABS(cbot-vtop)<1.d-4) BSt%fermie=vtop ! To avoid error on the last digit FIXME is it really needed
 end if

 write(msg,'(a,f6.2,a)')' Fermi energy         [eV] ',BSt%fermie*Ha_eV,ch10
 call wrtout(std_out,msg,'COLL')

end subroutine update_occ
!!***

!!****f* ABINIT/ReportGap
!! NAME
!! ReportGap
!!
!! FUNCTION
!!
!! INPUTS
!!  BSt<Bandstructure_type>=Info on the band structure, the smearing technique and the physical temperature used.
!!  header[Optional]
!!  kmask[Optional]
!!  unit[Optional]
!!  mode_paral[Optional]
!!
!! OUTPUT
!!  Only writing.
!!
!! SIDE EFFECTS
!!
!! PARENTS
!!      sigma
!!
!! CHILDREN
!!      get_eneocc_vect,prteigrs
!!
!! SOURCE

subroutine ReportGap(BSt,header,kmask,unit,mode_paral)

 use defs_basis
 use defs_datatypes
 use m_numeric_tools, only : imin_loc, imax_loc

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_01manage_mpi
 use interfaces_15gw, except_this_one => ReportGap
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in),optional :: unit
 character(len=4),intent(in),optional :: mode_paral
 character(len=*),intent(in),optional :: header
 type(Bandstructure_type),intent(in)  :: BSt

!arrays
 logical,optional,intent(in) ::  kmask(BSt%nkpt)

!Local variables-------------------------------
!scalars
 integer :: ikibz,nband_k,iband,isppol,ikopt,ivk,ick,ivb,icb,unt,first
 real(dp),parameter :: tol_fermi=tol6
 real(dp) :: ediff,gap,optical_gap
 logical :: ismetal
 character(len=4) :: mode
 character(len=500) :: msg
!arrays
 integer :: val_idx(BSt%nkpt,BSt%nsppol)
 real(dp) :: top_valence(BSt%nkpt),bot_conduct(BSt%nkpt) 
 logical :: my_kmask(BSt%nkpt)
! *********************************************************************

 unt =std_out ; if (PRESENT(unit      )) unt =unit
 mode='COLL'  ; if (PRESENT(mode_paral)) mode=mode_paral

 my_kmask=.TRUE.
 if (PRESENT(kmask)) my_kmask=kmask

 val_idx(:,:) = get_valence_idx(BSt,tol_fermi)
 first=0

 do isppol=1,BSt%nsppol

  ! No output if system i metallic
  ismetal=ANY(val_idx(:,isppol)/=val_idx(1,isppol)) 
  if (ismetal) CYCLE
  first=first+1
  if (first==1) then
   msg=ch10
   if (PRESENT(header)) msg=ch10//' === '//TRIM(ADJUSTL(header))//' === '
   call wrtout(unt,msg,mode) 
  end if

  ivb=val_idx(1,isppol)
  icb=ivb+1

  do ikibz=1,BSt%nkpt
   if (.not.my_kmask(ikibz)) CYCLE
   nband_k=BSt%nband(ikibz+(isppol-1)*BSt%nkpt)
   top_valence(ikibz)=BSt%eig(ivb,ikibz,isppol)
   if (icb>nband_k) GOTO 10 ! Only occupied states are present, no output!
   bot_conduct(ikibz)=BSt%eig(icb,ikibz,isppol)
  end do

  ! === Get minimum of the optical Gap ===
  ikopt= imin_loc(bot_conduct-top_valence,MASK=my_kmask)
  optical_gap=bot_conduct(ikopt)-top_valence(ikopt)

  ! === Get fundamental Gap ===
  ick = imin_loc(bot_conduct,MASK=my_kmask)
  ivk = imax_loc(top_valence,MASK=my_kmask)
  gap = BSt%eig(icb,ick,isppol)-BSt%eig(ivb,ivk,isppol)

  write(msg,'(a,i2,a,2(a,f8.4,a,3f8.4,a),33x,a,3f8.4)')&
&  '  >>>> For spin ',isppol,ch10,&
&  '   Minimum optical gap = ',optical_gap*Ha_eV,' [eV], located at k-point      : ',BSt%kptns(:,ikopt),ch10,&
&  '   Fundamental gap     = ',gap*Ha_eV,        ' [eV], Top of valence bands at : ',BSt%kptns(:,ivk),ch10,  &
&                                                '       Bottom of conduction at : ',BSt%kptns(:,ick)
  call wrtout(unt,msg,mode) 

 end do !isppol

 return

10 write(msg,'(a)')' ReportGap : WARNING - not enough states to calculate the band gap.'
   call wrtout(unt,msg,mode) 

end subroutine ReportGap
!!***

!!****f* ABINIT/WannierInterpol
!! NAME
!! WannierInterpol
!!
!! FUNCTION
!!
!!
!! Copyright (C) 2007 Jonathan Yates, Arash Mostofi,          
!!  Young-Su Lee, Nicola Marzari, Ivo Souza, David Vanderbilt 
!! Copyright (C) 2008-2009 ABINIT group (MG)
!! This file is distributed under the terms of the GNU        
!! General Public License. See the file `LICENSE' in          
!! the root directory of the present distribution, or         
!! http://www.gnu.org/copyleft/gpl.txt .                      
!!                                                            
!! INPUTS
!!  BSt<Bandstructure_type>=Info on the band structure
!!
!! OUTPUT
!!
!! SIDE EFFECTS
!!
!! PARENTS
!!      gw_tools
!!
!! CHILDREN
!!      get_eneocc_vect,prteigrs
!!
!! SOURCE

subroutine WannierInterpol(WData,Kmesh,BSt)

 use defs_basis
 use defs_datatypes
 use m_fstrings,       only : toupper
 use m_errors,         only : assert

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_12geometry
 use interfaces_lib03numeric
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 type(WannierData),intent(inout) :: WData
 type(Bandstructure_type),intent(out)  :: BSt
 type(BZ_mesh_type),intent(in) :: Kmesh
!arrays

!Local variables-------------------------------
!scalars
 integer :: ikpt,nwan,nrpts,nrpts_cut,isp,prtvol
 integer :: ierr,iR,ii,jj,info,nfound,spad
 real(dp) :: rdotk,ABSTOL,ucvol
 complex(dpc),parameter :: cmplx_i=(0.0_dp,1.0_dp)
 complex(dpc) :: fact
 character(len=500) :: msg

!arrays
 integer,pointer :: irvec(:,:),ndegen(:)
 integer,allocatable :: iwork(:),ifail(:)
 real(dp) :: gmet(3,3),gprimd(3,3),rmet(3,3)
 real(dp),allocatable :: eig_interp(:,:),bands_proj(:,:),rwork(:)     
 complex(dpc),pointer :: ham_r(:,:,:)
 complex(dpc),allocatable  :: ham_r_cut(:,:,:)
 complex(dpc),allocatable  :: ham_pack(:)
 complex(dpc),allocatable  :: ham_kprm(:,:)
 complex(dpc),allocatable  :: U_int(:,:)
 complex(dpc),allocatable  :: cwork(:)     
 !integer, allocatable :: irvec_cut(:,:)
 !integer              :: irvec_max(3)
 !integer              :: nrpts_cut
! *********************************************************************

 prtvol=0
 call metric(gmet,gprimd,-1,rmet,WData%Hdr%rprimd,ucvol)

 ! === Bst will contain only the interpolated bands ===
 !FIXME We hack a bit Bst but this has to be done in a much cleaner way to have a working object
 Bst%nkpt=Kmesh%nibz 
 allocate(Bst%kptns(3,Bst%nkpt),Bst%wtk(Bst%nkpt))
 Bst%kptns=Kmesh%ibz 
 Bst%wtk  =Kmesh%wt

 Bst%nsppol=WData%nsppol 
 Bst%mband=MAXVAL(WData%nwan)

 !FIXME this has to be initialized correctly
 Bst%nspinor=1

 allocate(Bst%nband(Bst%nkpt*Bst%nsppol)) 
 do isp=1,Bst%nsppol
  spad=(isp-1)*Bst%nkpt
  Bst%nband(1+spad:Bst%nkpt+spad)=WData%nwan(isp)
 end do

 allocate(Bst%eig(Bst%mband,Bst%nkpt,Bst%nsppol))

 ! =======================================
 ! === Start the Wannier interpolation ===
 ! =======================================
 if (.not.associated(WData%irvec).or..not.associated(WData%ndegen)) then
  stop 'irvec and ndegen are not associated'
 end if
 nrpts  =  WData%nrpts
 irvec  => WData%irvec
 ndegen => WData%ndegen

 do isp=1,WData%nsppol

  ! Pay attention here: point the Wannier functions for this spin
  nwan=WData%nwan(isp) 
  ham_r => WData%hamWR(1:nwan,1:nwan,1:WData%nrpts,isp)

  ! TODO add cut case at this level !if (cut) then ...

  allocate(eig_interp(nwan,BSt%nkpt))
  eig_interp=zero 
  !allocate(bands_proj(nwan,BSt%nkpt),stat=ierr)
  !bands_proj=zero

  allocate(ham_pack((nwan*(nwan+1))/2),stat=ierr)
  allocate(ham_kprm(nwan,nwan),stat=ierr)
  allocate(U_int(nwan,nwan),stat=ierr)
  allocate(cwork(2*nwan),rwork(7*nwan),iwork(5*nwan),ifail(nwan))
  !
  ! Cut H matrix in real-space
  !if (index(bands_plot_mode,'cut').ne.0)  call plot_cut_hr()

  ! === Interpolate the Hamiltonian at each kpoint ===
  do ikpt=1,BSt%nkpt

   ham_kprm(:,:)=czero
   !if (index(bands_plot_mode,'s-k').ne.0) then
    do iR=1,nrpts
     rdotk=two_pi*DOT_PRODUCT(BSt%kptns(:,ikpt),irvec(:,iR))
     fact=EXP(cmplx_i*rdotk)/REAL(ndegen(iR),dp)
     ham_kprm=ham_kprm+fact*ham_r(:,:,iR)
    end do
   !TODO
   !else if (index(bands_plot_mode,'cut').ne.0) then
   ! do iR=1,nrpts_cut
   !  rdotk=two_pi*DOT_PRODUCT(BSt%kptns(:,ikpt),irvec_cut(:,iR))
   !  !!$[aam] check divide by ndegen?
   !  fact=EXP(cmplx_i*rdotk)
   !  ham_kprm=ham_kprm+fact*ham_r_cut(:,:,iR)
   ! end do
   !end if

   ! === Diagonalise H_k (->basis of eigenstates) ===
   do jj=1,nwan
    do ii=1,jj
     ham_pack(ii+((jj-1)*jj)/2)=ham_kprm(ii,jj)
    end do
   end do

   ABSTOL=-one !; ABSTOL= 2*DLAMCH('S')

   call ZHPEVX('V','A','U',nwan,ham_pack,zero,zero,0,0,ABSTOL,&
    nfound,eig_interp(:,ikpt),U_int,nwan,cwork,rwork,iwork,ifail,info)

   if (info<0) then
    write(std_out,'(a,i3,a)') 'THE ',-info, ' ARGUMENT OF ZHPEVX HAD AN ILLEGAL VALUE'
   end if
   if (info>0) then
    write(std_out,'(i3,a)')info,' EIGENVECTORS FAILED TO CONVERGE'
   end if

   ! Compute projection onto WF if requested
   !if (num_bands_project>0) then
   ! do loop_w=1,nwan
   !  do loop_p=1,nwan 
   !   if (any(bands_plot_project==loop_p)) then
   !    bands_proj(loop_w,ikpt)=bands_proj(loop_w,ikpt)+abs(U_int(loop_p,loop_w))**2
   !   end if 
   !  end do
   ! end do
   !end if

   ! * eig_interp contains the interpolated nwan bands for this k-points
   ! TODO here be careful since first index might refer to an arbitrary band!
   BSt%eig(1:nwan,ikpt,isp)=eig_interp(:,ikpt)
   !write(97,'(7f8.4)')BSt%kptns(:,ikpt),eig_interp(1:nwan,ikpt)
  end do !ikpt

  ! Interpolation Finished.
  deallocate(eig_interp)
  deallocate(ham_pack)
  deallocate(ham_kprm)
  deallocate(U_int)
  deallocate(cwork,rwork,iwork,ifail)

  !if (allocated(ham_r_cut)) deallocate(ham_r_cut,stat=ierr)
  !if (allocated(irvec_cut)) deallocate(irvec_cut,stat=ierr)
 end do !isp

end subroutine WannierInterpol
!!***


!!****f* ABINIT/PlotBands
!! NAME
!! PlotBands
!!
!! FUNCTION
!! Plots the interpolated band structure in Xmgrace format  
!! Based on plot_interpolate_xmgrace, routine of the Wannier90 code.
!!
!! INPUTS
!!  BSt<bandstructure_type>The type containing the data.
!!  fname=The name of the file.
!!
!! Copyright (C) 2007 Jonathan Yates, Arash Mostofi,          
!!  Young-Su Lee, Nicola Marzari, Ivo Souza, David Vanderbilt 
!! Copyright (C) 2008-2009 ABINIT group (MG)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~abinit/COPYING
!! or http://www.gnu.org/copyleft/gpl.txt .
!! For the initials of contributors, see ~abinit/doc/developers/contributors.txt.
!!
!! OUTPUT
!!  Only writing
!!
!! PARENTS
!!      gw_tools
!!
!! CHILDREN
!!      get_eneocc_vect,prteigrs
!!
!! SOURCE

subroutine PlotBands(BSt,gmet,fname,bands_label)

 use defs_basis
 use defs_datatypes
 use m_io_tools, only : get_unit
 use m_geometry, only : normv

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_15gw, except_this_one => PlotBands
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 character(len=*),intent(in) :: fname
 type(Bandstructure_type),intent(in) :: BSt
!arrays
 real(dp),intent(in) :: gmet(3,3)
 character(len=1),optional,intent(in) :: bands_label(BSt%nkpt)

!Local variables-------------------------------
 integer :: ikpt,ikpath,ib,xunit,nspec_k
 real(dp) :: emin,emax
 character (len=9) :: cdate, ctime
 character(len=fnlen) :: xname
!arrays 
 real(dp) :: minmax_ene(2,BSt%nsppol),kdiff(3)
 real(dp) :: xval(BSt%nkpt)
 character(len=10),allocatable  :: ctemp(:),xlabel(:)
! *************************************************************************

 !call io_date(cdate, ctime)

 ! === Axis labels ===
 ! TODO doesnt work yet
#if 0
 allocate(ctemp(BSt%nkpt))

 ! Switch any G to Gamma
 do ikpt=1,BSt%nkpt 
  if (bands_label(i)/='0') then
   ctemp(ikpt)=bands_label(ikpt)
   if (ctemp(ikpt)=='G') ctemp(ikpt)='\xG\0'
  end if
 end do

 nspec_k=COUNT(ctemp/='0')
 allocate(xlabel(nspec_k))

 xlabel(1)=' '//ctemp(1)//' '
 !do i=2,num_paths
 ! if(ctemp(2*(i-1))/=ctemp(2*(i-1)+1)) then
 !  xlabel(i)=ctemp(2*(i-1))//'/'//ctemp(2*(i-1)+1)
 ! else
 !  xlabel(i)=ctemp(2*(i-1))
 ! end if
 !end do
 !xlabel(num_spts)=ctemp(bands_num_spec_points)
 deallocate(ctemp)
#endif

 ! Get min and Max energy in Ev
 minmax_ene = get_minmax(BSt,'eig')*Ha_eV
 emin = MINVAL(minmax_ene(1,:)) - one
 emax = MAXVAL(minmax_ene(2,:)) + one

 !TODO gmet shoul be stored in Bst, or at least gprimd
 xval(1)=zero
 do ikpath=2,Bst%nkpt
  kdiff=Bst%kptns(:,ikpath)-Bst%kptns(:,ikpath-1)
  xval(ikpath) = xval(ikpath-1)+ normv(kdiff,gmet,'G')
 end do

 ! Xmgrace format
 xunit=get_unit() 
 xname=TRIM(fname)//'_band.agr'
 open(xunit,file=xname,form='formatted')

 write(xunit,'(a)') '# Grace project file                      '
 write(xunit,'(a)') '# written using Abinit www.abinit.org     '
 write(xunit,'(a)') '@version 50113                            '
 write(xunit,'(a)') '@page size 792, 612                       '
 write(xunit,'(a)') '@page scroll 5%                           '
 write(xunit,'(a)') '@page inout 5%                            '
 write(xunit,'(a)') '@link page off                            '

 !write(xunit,'(a)') '@timestamp def "'//cdate//' at '//ctime//'" ' 
 write(xunit,'(a)') '@with g0'                                  
 write(xunit,'(a)') '@    world xmin 0.00'
 !write(xunit,'(a,f10.5)') '@    world xmax ',xval(Bst%nkpt)
 write(xunit,'(a,i5)') '@    world xmax ',Bst%nkpt-1
 write(xunit,'(a,f10.5)') '@    world ymin ',emin
 write(xunit,'(a,f10.5)') '@    world ymax ',emax
 write(xunit,'(a)') '@default linewidth 1.5'
 write(xunit,'(a)') '@    xaxis  tick on'
 write(xunit,'(a)') '@    xaxis  tick major 1'
 write(xunit,'(a)') '@    xaxis  tick major color 1'
 write(xunit,'(a)') '@    xaxis  tick major linestyle 3'
 write(xunit,'(a)') '@    xaxis  tick major grid on'
 write(xunit,'(a)') '@    xaxis  tick spec type both'
 !write(xunit,'(a,i0)') '@    xaxis  tick spec ',1+bands_num_spec_points/2
 write(xunit,'(a)') '@    xaxis  tick major 0, 0'
 !do i=1,bands_num_spec_points/2
 ! write(xunit,'(a,i0,a,a)') '@    xaxis  ticklabel ',i-1,',', '"'//trim(adjustl(xlabel(i)))//'"'
 ! write(xunit,'(a,i0,a,f10.5)') '@    xaxis  tick major ',i,' , ',sum(kpath_len(1:i))
 !end do
 !write(xunit,'(a,i0,a)') '@    xaxis  ticklabel ',bands_num_spec_points/2 &
 ! ,',"'//trim(adjustl(xlabel(1+bands_num_spec_points/2)))//'"'
 write(xunit,'(a)') '@    xaxis  ticklabel char size 1.500000'
 write(xunit,'(a)') '@    yaxis  tick major 10'
 write(xunit,'(a)') '@    yaxis  label "Band Energy (eV)"'
 write(xunit,'(a)') '@    yaxis  label char size 1.500000'
 write(xunit,'(a)') '@    yaxis  ticklabel char size 1.500000'

 !here I suppose no dependency on k and spin
 do ib=1,BSt%mband !num_wann
  write(xunit,'(a,i4,a)') '@    s',ib-1,' line color 1'
 end do

 do ib=1,BSt%mband !num_wann
  write(xunit,'(a,i4)') '@target G0.S',ib-1
  write(xunit,'(a)') '@type xy'
  do ikpt=1,BSt%nkpt
   !write(xunit,'(2e16.8)')xval(ikpt),BSt%eig(ib,ikpt,1)*Ha_eV !FIXME doesnt work for nsppol=2
   write(xunit,'(i5,e16.8)')ikpt-1,BSt%eig(ib,ikpt,1)*Ha_eV !FIXME doesnt work for nsppol=2
  end do
  write(xunit,'(a)')'&'
 end do

 close(xunit)

end subroutine PlotBands 
!!***


!!****f* ABINIT/SelectBands
!! NAME
!! SelectBands
!!
!! FUNCTION
!!
!! INPUTS
!!  BSt<Bandstructure_type>=Info on the band structure 
!!
!! OUTPUT
!!
!! SIDE EFFECTS
!!
!! PARENTS
!!
!! CHILDREN
!!
!! SOURCE

function SelectBands(BSt,bandselect,kselect,spinselect) result(NewBSt)

 use defs_basis
 use defs_datatypes

 implicit none

!Arguments ------------------------------------
!scalars
 type(Bandstructure_type),intent(in)  :: BSt 
 logical,optional,intent(in) :: bandselect(BSt%mband,BSt%nkpt,BSt%nsppol)
 logical,optional,intent(in) :: kselect(BSt%nkpt)
 logical,optional,intent(in) :: spinselect(BSt%nsppol)
 type(Bandstructure_type) :: NewBSt

!Local variables-------------------------------
!scalars
 integer :: ikpt,nband_k,iband,isppol,new_nband_k,idx,isp_new,ik_new,ib_new
 character(len=500) :: msg
!arrays
 logical :: band_in(BSt%mband,BSt%nkpt,BSt%nsppol)
 logical :: kpt_in(BSt%nkpt)
 logical :: spin_in(BSt%nsppol)
! *********************************************************************

 band_in=.TRUE. ; if (PRESENT(bandselect)) band_in=bandselect
 kpt_in =.TRUE. ; if (PRESENT(kselect   )) kpt_in =kselect
 spin_in=.TRUE. ; if (PRESENT(spinselect)) spin_in=spinselect

 NewBSt%nkpt   =COUNT(kpt_in)
 NewBSt%nspinor=BSt%nspinor      
 NewBSt%nsppol =COUNT(spin_in)  
 NewBSt%occopt =BSt%occopt        

 NewBSt%entropy = BSt%entropy 
 NewBSt%fermie  = BSt%fermie  
 NewBSt%nelect  = BSt%nelect  
 NewBSt%tphysel = BSt%tphysel 
 NewBSt%tsmear  = BSt%tsmear  

 allocate(NewBSt%nband(NewBSt%nkpt*NewBSt%nsppol))
 idx=0
 do isppol=1,BSt%nsppol
  if (.not.spin_in(isppol)) CYCLE
  do ikpt=1,BSt%nkpt
   if (.not.kpt_in(ikpt)) CYCLE
   nband_k=BSt%nband(ikpt+(isppol-1)*BSt%nkpt)
   idx=idx+1
   new_nband_k=COUNT(band_in(1:nband_k,ikpt,isppol))
   NewBSt%nband(idx)=new_nband_k
  end do
 end do

 NewBSt%mband =MAXVAL(NewBSt%nband)
 NewBSt%bantot=SUM   (NewBSt%nband)

 allocate(NewBSt%istwfk(NewBSt%nkpt))      
 allocate(NewBSt%npwarr(NewBSt%nkpt))
 allocate(NewBSt%kptns(3,NewBSt%nkpt))  
 allocate(NewBSt%eig   (NewBSt%mband,NewBSt%nkpt,NewBSt%nsppol))
 allocate(NewBSt%occ   (NewBSt%mband,NewBSt%nkpt,NewBSt%nsppol))
 allocate(NewBSt%doccde(NewBSt%mband,NewBSt%nkpt,NewBSt%nsppol))    
 allocate(NewBSt%wtk(NewBSt%nkpt))

 ! === Copy arrays that depend only on k ===
 idx=0
 do ikpt=1,BSt%nkpt
  if (.not.kpt_in(ikpt)) CYCLE
  idx=idx+1
  NewBSt%istwfk(idx) =BSt%istwfk(ikpt)      
  NewBSt%npwarr(idx) =BSt%npwarr(ikpt)      
  NewBSt%kptns(:,idx)=BSt%kptns(:,ikpt)    
  NewBSt%wtk(idx)    =BSt%wtk(ikpt)        
 end do

 ! * Renormalize weights
 NewBSt%wtk=NewBSt%wtk/SUM(NewBSt%wtk)

 ! === Copy arrays that depend on b-k-s ===
 isp_new=0
 do isppol=1,BSt%nsppol
  if (.not.spin_in(isppol)) CYCLE
  isp_new=isp_new+1

  ik_new=0
  do ikpt=1,BSt%nkpt
   if (.not.kpt_in(ikpt)) CYCLE
   ik_new=ik_new+1

   nband_k=BSt%nband(ikpt+(isppol-1)*BSt%nkpt)
   ib_new=0
   do iband=1,nband_k
    if (.not.band_in(iband,ikpt,isppol)) CYCLE
    ib_new=ib_new+1
    NewBSt%eig   (ib_new,ik_new,isp_new)=BSt%eig   (iband,ikpt,isppol)        
    NewBSt%occ   (ib_new,ik_new,isp_new)=BSt%occ   (iband,ikpt,isppol)        
    NewBSt%doccde(ib_new,ik_new,isp_new)=BSt%doccde(iband,ikpt,isppol)     
   end do !iband
  end do !ikpt
 end do !isppol

end function SelectBands
!!***

!!****f* ABINIT/ExpandBands
!! NAME
!! ExpandBands
!!
!! FUNCTION
!!
!! INPUTS
!!  BSt<Bandstructure_type>=Info on the band structure 
!!
!! OUTPUT
!!
!! SIDE EFFECTS
!!  In info/=0, some of the k-points in klist have no corresponding image in Bst_in%kptns
!!
!! PARENTS
!!
!! CHILDREN
!!
!! SOURCE

function ExpandBands(BSt_in,nklist,klist,use_tr,use_afm,nsym,symrec,symafm,info) result(BSt_out)

 use defs_basis
 use defs_datatypes
 use m_bz_mesh,     only: isamek

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_01manage_mpi
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: nsym,nklist
 integer,intent(out) :: info
 logical,intent(in) :: use_tr,use_afm
 type(Bandstructure_type),intent(in) :: BSt_in 
 type(Bandstructure_type) :: BSt_out
!arrays
 real(dp),intent(in) :: klist(3,nklist)
 integer,intent(in) :: symrec(3,3,nsym)
 integer,intent(in) :: symafm(nsym)

!Local variables-------------------------------
!scalars
 integer :: iklist,itim,timrev,isym,ikibz,nkfound,ii,jj
 integer :: ikpt,nband_k,iband,isppol,ikdx
 character(len=500) :: msg
 logical :: found
!arrays
 integer :: G0(3)
 real(dp) :: krot(3)
 integer,allocatable :: klist2ibz(:)
! *********************************************************************

 do ii=1,nklist-1
  do jj=ii+1,nklist
   if (isamek(klist(:,ii),klist(:,jj),G0)) then
    write(msg,'(2a,i4,a,i4,5a)')ch10,&
&    ' Points ',ii,' and ',jj,' in klist are equivalent ',ch10,&
&    ' This is not allowed in the present implementation.',ch10,&
&    ' Change the input values to avoid duplicates. '
    call wrtout(ab_out,msg,'COLL') 
    call leave_new('COLL')
   end if
  end do
 end do

 info=0
 timrev=1 ; if (use_tr) timrev=2
 allocate(klist2ibz(nklist)) ; klist2ibz=0

 do iklist=1,nklist
  found=.FALSE.

ibzl: do ikibz=1,BSt_in%nkpt
       do itim=1,timrev
        do isym=1,nsym
         if (use_afm.and.symafm(isym)==-1) CYCLE
         ! * Form IS k ===
         krot(:)=(3-2*itim)*MATMUL(symrec(:,:,isym),BSt_in%kptns(:,ikibz))

         ! * Check whether it is equal to klist(:,ilist) within a RL vector.
         ! FIXME see notes below related to this function
         if (isamek(krot,klist(:,iklist),G0)) then 
          found=.TRUE.
          klist2ibz(iklist)=ikibz
          exit ibzl
         end if

        end do !isym 
       end do !itim
      end do ibzl

      if (.not.found) info=info+1
 end do !iklist

 nkfound=COUNT(klist2ibz/=0)

 Bst_out%nkpt   =nkfound
 Bst_out%nspinor=BSt_in%nspinor      
 Bst_out%nsppol =Bst_in%nsppol
 Bst_out%occopt =Bst_in%occopt        

 Bst_out%entropy = Bst_in%entropy 
 Bst_out%fermie  = Bst_in%fermie  
 Bst_out%nelect  = Bst_in%nelect  
 Bst_out%tphysel = Bst_in%tphysel 
 Bst_out%tsmear  = Bst_in%tsmear  

 allocate(Bst_out%nband(Bst_out%nkpt*Bst_out%nsppol))
 ikdx=0
 do isppol=1,Bst_out%nsppol
  do iklist=1,nklist
   ikibz=klist2ibz(iklist)
   if (ikibz==0) CYCLE
   ikdx=ikdx+1
   nband_k=Bst_in%nband(ikibz+(isppol-1)*Bst_in%nkpt)
   Bst_out%nband(ikdx)=nband_k
  end do
 end do

 Bst_out%mband =MAXVAL(Bst_out%nband)
 Bst_out%bantot=SUM   (Bst_out%nband)

 allocate(Bst_out%istwfk(Bst_out%nkpt))      
 allocate(Bst_out%nband(Bst_out%nkpt*Bst_out%nsppol))
 allocate(Bst_out%npwarr(Bst_out%nkpt))
 allocate(Bst_out%kptns(3,Bst_out%nkpt))  
 allocate(Bst_out%eig   (Bst_out%mband,Bst_out%nkpt,Bst_out%nsppol))
 allocate(Bst_out%occ   (Bst_out%mband,Bst_out%nkpt,Bst_out%nsppol))
 allocate(Bst_out%doccde(Bst_out%mband,Bst_out%nkpt,Bst_out%nsppol))    
 allocate(Bst_out%wtk(Bst_out%nkpt))


 ! === Copy arrays that depend only on k ===
 ikdx=0
 do iklist=1,nklist
  ikibz=klist2ibz(iklist)
  if (ikibz==0) CYCLE
  ikdx=ikdx+1
  Bst_out%istwfk(ikdx) =Bst_in%istwfk(ikibz)      
  Bst_out%npwarr(ikdx) =Bst_in%npwarr(ikibz)      
  Bst_out%kptns(:,ikdx)=klist(:,iklist) !Use klist, not Bst_in%kptns
  Bst_out%wtk(ikdx)    =Bst_in%wtk(ikibz)        
 end do

 ! * Renormalize weights
 Bst_out%wtk=Bst_out%wtk/SUM(Bst_out%wtk)

 ! === Copy arrays that depend on b-k-s ===
 do isppol=1,Bst_in%nsppol
  ikdx=0
  do iklist=1,nklist
   ikibz=klist2ibz(iklist)
   if (ikibz==0) CYCLE
   ikdx=ikdx+1
   Bst_out%eig   (:,ikdx,isppol)=Bst_in%eig   (:,ikibz,isppol)        
   Bst_out%occ   (:,ikdx,isppol)=Bst_in%occ   (:,ikibz,isppol)        
   Bst_out%doccde(:,ikdx,isppol)=Bst_in%doccde(:,ikibz,isppol)     
  end do !ikpt
 end do !isppol

 deallocate(klist2ibz)

end function ExpandBands
!!***

!!****f* ABINIT/wrap_prteigrs
!! NAME
!! wrap_prteigrs
!!
!! FUNCTION
!!
!! INPUTS
!!
!! OUTPUT
!!
!! PARENTS
!!
!! CHILDREN
!!      get_eneocc_vect,prteigrs
!!
!! SOURCE
! FIXME do not use this routine is still under development
subroutine wrap_prteigrs(Bst,kptopt,iscf,nnsclo_now,tolwfr,prteig,enunit,filnam,unit,prtvol)

 use defs_basis
 use defs_datatypes

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_15common
 use interfaces_15gw, except_this_one => wrap_prteigrs
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: enunit,iscf,kptopt,nnsclo_now,prteig
 integer,optional,intent(in) :: unit,prtvol
 real(dp),intent(in) :: tolwfr
 character(len=fnlen),intent(in) :: filnam
 type(Bandstructure_type),intent(in) :: Bst

!Local variables-------------------------------
!scalars
 integer :: my_unit,my_prtvol,option
 real(dp)  :: vxcavg
!arrays
 real(dp),allocatable :: occ_vect(:),eig_vect(:),resid(:)
! *************************************************************************

 my_prtvol=0       ; if (PRESENT(prtvol)) my_prtvol=prtvol
 my_unit  =std_out ; if (PRESENT(unit  )) my_unit  =unit

 allocate(occ_vect(Bst%bantot))
 allocate(eig_vect(Bst%bantot))
 call get_eneocc_vect(BSt,'occ',occ_vect) 
 call get_eneocc_vect(BSt,'eig',eig_vect) 

 allocate(resid(BSt%mband*BSt%nkpt*BSt%nsppol))
 resid=zero
 vxcavg=greatest_real !this is just to remind me that this part is not implemented

 option=3
 call prteigrs(eig_vect,enunit,Bst%fermie,filnam,my_unit,iscf,Bst%kptns,kptopt,Bst%mband,Bst%nband,&
&  Bst%nkpt,nnsclo_now,Bst%nsppol,occ_vect,Bst%occopt,option,prteig,my_prtvol,resid,tolwfr,vxcavg,Bst%wtk)

 deallocate(resid)
 deallocate(occ_vect,eig_vect)

end subroutine wrap_prteigrs
!!***
