!{\src2tex{textfont=tt}}
!!****f* ABINIT/wrpawps
!! NAME
!! wrpawps
!!
!! FUNCTION
!! Write a PAW pseudopotential file formatted for Abinit
!!
!! COPYRIGHT
!! Copyright (C) 1998-2007 ABINIT group (FJ, MT)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~ABINIT/Infos/copyright
!! or http://www.gnu.org/copyleft/gpl.txt .
!! For the initials of contributors, see ~ABINIT/Infos/contributors .
!!
!! INPUTS
!!  filename= output file name for Abinit
!!  funit= output unit number
!!  loggrd
!!    %meshsz=mesh size for the logarithmic grid
!!    %uselog=TRUE if data are transfered on a log. grid before being written
!!    %log_step=logarithmic step for the logarithmic grid
!!    %rad_step=radial step for the logarithmic grid
!!  pawarray
!!    %shapefunc(wav_meshsz)= Normalized shape function
!!    %shpnrm(l_size)= Moments of shape function for each l
!!  pawps
!!    %coreden4pr2(core_meshsz)= Core density multiplied by 4Pi.r2
!!    %tcoreden4pr2(core_meshsz)= Pseudized core density multiplied by 4Pi.r2
!!    %tvaleden4pr2(vale_meshsz)= Pseudized valence density multiplied by 4Pi.r2
!!    %dij0(lmn2_size)= Part of the Dij term calculated in the psp part
!!    %phi(wav_meshsz,basis_size)= PAW atomic wavefunctions
!!                                on the radial grid
!!    %tphi(wav_meshsz,basis_size)= PAW atomic pseudo-wavefunctions
!!                                 on the radial grid
!!    %tproj(prj_msz_max,basis_size)= PAW projectors
!!                                 on the radial grid
!!    %rhoij0= Atomic initialization of rhoij
!!    %vbare(sph_meshsz)= bare local potential (part of VH(tnzc))
!!    %vhtnzc(core_meshsz)= Hartree potential of the ps-density
!!                           of the nucleus + core electrons
!!  pawrad= radial grid definitions
!!  pawrso
!!    %userso=TRUE if REAL Space Optimization is required
!!  pshead
!!    %atomic_charge= Total atomic charge
!!    %basis_size= Number of elements for the paw nl basis
!!    %core_charge= Core charge
!!    %core_meshsz= Dimension of radial mesh for core density
!!    %creatorid= ID of psp generator (here creatorID=1 !)
!!    %hat_meshsz= Dimension of radial mesh for shape function
!!    %lambda= Lambda in gaussian type g(r)
!!    %lmax= Maximum value of l
!!    %lmn_size= Number of elements for the paw basis
!!    %l_size= Max. value of l+1 leading to a non zero Gaunt coeffs
!!    %mesh_type=  Flag defining ther radial grid type
!!    %orbitals(basis_size)= Quantum number l for each basis function
!!    %prj_meshsz= Dimension of radial mesh for tproj
!!    %pspcod= Psp code number for Abinit (here PAW->pspcod=7 !)
!!    %pspxc_abinit= Abinit s code number for the exchange-correlation
!!    %rad_step= Step corresponding to radial mesh
!!    %rc_proj= Sphere radius for tproj
!!    %rc_hat= radius for shape function
!!    %rc_sph= Default PAW sphere radius
!!    %shape_type= Shape function type
!!    %sigma= Sigma for gaussian type g(r)
!!    %sph_meshsz= Dimension of radial mesh corresponding to PAW spheres
!!    %title= Title for pseudopotential
!!    %vale_meshsz= Dimension of radial mesh for pseudo valence density (0if not present)
!!    %vloc_meshsz= Dimension of radial mesh for vloc=vhtnzc
!!    %sph_meshsz= Dimension of radial mesh for phi, tphi ...
!!    %vlocopt= Option for Vloc
!!    %wav_meshsz= Dimension of radial mesh for Phi, tPhi, ...
!!
!! NOTES
!!  File format of formatted PAW psp input for Abinit:
!!  --------------------------------------------------
!!  (1) title (character) line
!!  (2) znucl, zion, pspdat
!!  (3) pspcod, pspxc, lmax, lloc, mmax, r2well
!!  (4) psp_version, creatorID
!!  (5) basis_size, lmn_size
!!  (6) orbitals (for l=1 to basis_size)
!!  (7) number_of_meshes
!!  For imsh=1 to number_of_meshes
!!      (8)  mesh_index, mesh_type ,mesh_size, rad_step[, log_step]
!!  (9) r_cut(SPH)
!!  (10) shape_type, r_shape[, shapefunction arguments]
!!  For iln=1 to basis_size
!!      (11) comment(character)
!!      (12) radial mesh index for phi
!!      (13) phi(r) (for ir=1 to phi_meshsz)
!!  For iln=1 to basis_size
!!      (14) comment(character)
!!      (15) radial mesh index for tphi
!!      (16) tphi(r) (for ir=1 to phi_mesh_size)
!!  For iln=1 to basis_size
!!      (17) comment(character)
!!      (18) radial mesh index for tproj
!!      (19) tproj(r) (for ir=1 to proj_mesh_size)
!!  (20) comment(character)
!!  (21) radial mesh index for core_density
!!  (22) core_density (for ir=1 to phi_mesh_size)
!!  (23) comment(character)
!!  (24) radial mesh index for tcore_density
!!  (25) tcore_density (for ir=1 to phi_mesh_size)
!!  (26) comment(character)
!!  (27) Dij0 (for ij=1 to lmn_size*(lmn_size+1)/2)
!!  (28) comment(character)
!!  (29) Rhoij0 (for ij=1 to lmn_size*(lmn_size+1)/2)
!!  (30) comment(character)
!!  (31) radial mesh index for Vloc, format of Vloc (0=Vbare, 1=VH(tnzc))
!!  (32) Vloc(r) (for ir=1 to vloc_mesh_size)
!!  ===== Following lines only if shape_type=-1 =====
!!  For il=1 to 2*max(orbitals)+1
!!      (33) comment(character)
!!      (34) radial mesh index for shapefunc
!!      (35) shapefunc(r)*gnorm(l)*r**l (for ir=1 to phi_meshsz)
!!  --------------------------------------------------
!!
!! PARENTS
!!      atompaw2abinit
!!
!! CHILDREN
!!      date_and_time,deducer0,splinefit
!!
!! SOURCE

 subroutine wrpawps(fname,loggrd,pawarray,pawps,pawrad,pawrso,pshead,funit)

 use defs_basis
 use defs_pawps

 implicit none

!Arguments ---------------------------------------------
 integer :: funit
 character*(fnlen)   :: fname
 type(loggrd_type)   :: loggrd
 type(pawarray_type) :: pawarray
 type(pawps_type)    :: pawps
 type(pawrad_type)   :: pawrad
 type(pawrso_type)   :: pawrso
 type(pshead_type)   :: pshead

!Local variables ---------------------------------------
 integer, parameter :: nmesh_max=5
 integer :: coremeshsz,ib,icoremesh,ii,il,ilmn,iprjmesh,ir
 integer :: ivalemesh,ivlocmesh,iwavmesh,jlmn,mtyp,nmesh
 integer :: prjmeshsz,valemeshsz,vlocmeshsz,wavmeshsz
 real(dp) :: rstep,lstep
 character*8 :: strdate
 integer :: meshsz(nmesh_max),meshtp(nmesh_max)
 real(dp) :: radstp(nmesh_max),logstp(nmesh_max)
 real(dp), allocatable :: ffit(:),ntmp(:),rr_log(:),shpf(:)

!-------------------------------------------------------

!Meshes definitions (if necessary define a logarithmic radial grid)
!=== Use of an auxilliary log grid
 if (loggrd%uselog) then
  loggrd%rad_step=(pawrad%rad(pshead%wav_meshsz)*(1.d0-1.d-12))*exp(-loggrd%log_step*dble(loggrd%meshsz-2))
  mtyp=3;rstep=loggrd%rad_step;lstep=loggrd%log_step
  wavmeshsz=loggrd%meshsz
  prjmeshsz=pshead%prj_meshsz
  coremeshsz=int((1.d0+1.d-12)*log(pshead%rad_step*dble(pshead%core_meshsz-1)/loggrd%rad_step)/loggrd%log_step)+2
  if (pshead%vale_meshsz>0) then
   valemeshsz=int((1.d0+1.d-12)*log(pshead%rad_step*dble(pshead%vale_meshsz-1)/loggrd%rad_step)/loggrd%log_step)+2
  else
   valemeshsz=0
  end if
  if (pshead%vlocopt==0) then
   vlocmeshsz=int((1.d0+1.d-12)*log(pshead%rad_step*dble(pshead%sph_meshsz-1)/loggrd%rad_step)/loggrd%log_step)+2
  else
   vlocmeshsz=int((1.d0+1.d-12)*log(pshead%rad_step*dble(pshead%vloc_meshsz-1)/loggrd%rad_step)/loggrd%log_step)+2
  endif
  allocate(rr_log(max(wavmeshsz,coremeshsz,valemeshsz,vlocmeshsz)));rr_log(1)=zero
  do ir=2,max(wavmeshsz,coremeshsz,valemeshsz,vlocmeshsz)
   rr_log(ir)=loggrd%rad_step*exp(loggrd%log_step*dble(ir-2))
  enddo
  write(6,'(3(/,a),2(/,a,g10.4),4(/,a,i4))') &
&    'Info:',&
&    '  All quantities (except nl projectors) are transfered',&
&    '  into a logarithmic grid (r(i)=A*exp[B(i-2)])...',&
&    '  Log. grid parameters: rad_step=',loggrd%rad_step,&
&    '                        log_step=',loggrd%log_step,&
&    '                        Size       =',wavmeshsz,&
&    '                        Size (core)=',coremeshsz,&
&    '                        Size (vale)=',valemeshsz,&
&    '                        Size (Vloc)=',vlocmeshsz
 else
!=== No use of an auxilliary log grid
  mtyp=pshead%mesh_type;rstep=pshead%rad_step;lstep=pshead%log_step
  wavmeshsz=min(pshead%sph_meshsz+5,pshead%wav_meshsz)
  prjmeshsz=pshead%prj_meshsz
  coremeshsz=pshead%core_meshsz
  valemeshsz=pshead%vale_meshsz
  if (pshead%vlocopt==0)then
   vlocmeshsz=pshead%sph_meshsz
  else
   vlocmeshsz=pshead%vloc_meshsz
  endif
 endif
!=== Build mesh definitions
 nmesh=1;iwavmesh=1
 meshtp(1)=mtyp;meshsz(1)=wavmeshsz;radstp(1)=rstep;logstp(1)=lstep
 if (loggrd%uselog.or.wavmeshsz/=prjmeshsz) then
  nmesh=nmesh+1;iprjmesh=nmesh;meshsz(nmesh)=pshead%prj_meshsz
  meshtp(nmesh)=pshead%mesh_type;radstp(nmesh)=pshead%rad_step;logstp(nmesh)=pshead%log_step 
 else
  iprjmesh=iwavmesh
 endif  
 if (wavmeshsz/=coremeshsz) then
  if (prjmeshsz/=coremeshsz) then
   nmesh=nmesh+1;icoremesh=nmesh;meshsz(nmesh)=coremeshsz
   meshtp(nmesh)=mtyp;radstp(nmesh)=rstep;logstp(nmesh)=lstep
  else
   icoremesh=iprjmesh
  endif
 else
  icoremesh=iwavmesh
 endif
 if (wavmeshsz/=vlocmeshsz) then
  if(prjmeshsz/=vlocmeshsz) then
   if(coremeshsz/=vlocmeshsz) then
    nmesh=nmesh+1;ivlocmesh=nmesh;meshsz(nmesh)=vlocmeshsz
    meshtp(nmesh)=mtyp;radstp(nmesh)=rstep;logstp(nmesh)=lstep
   else
    ivlocmesh=icoremesh
   endif
  else
   ivlocmesh=iprjmesh
  endif
 else
  ivlocmesh=iwavmesh
 endif
 if (wavmeshsz/=valemeshsz) then
  if(prjmeshsz/=valemeshsz) then
   if(coremeshsz/=valemeshsz) then
    if(vlocmeshsz/=valemeshsz) then
     nmesh=nmesh+1;ivalemesh=nmesh;meshsz(nmesh)=valemeshsz
     meshtp(nmesh)=mtyp;radstp(nmesh)=rstep;logstp(nmesh)=lstep
    else
     ivalemesh=ivlocmesh
    endif
   else
    ivalemesh=icoremesh
   endif
  else
   ivalemesh=iprjmesh
  endif
 else
  ivalemesh=iwavmesh
 endif

!Open the file for writing
 open(unit=funit,file=trim(fname),form='formatted',status='unknown')

!Write the header
 call date_and_time(strdate)
 write(funit,'(a)') trim(pshead%title)
 write(funit,'(1x,f7.3,1x,f7.3,1x,a,14x,a)') &
&      pshead%atomic_charge,&
&      pshead%atomic_charge-pshead%core_charge,&
&      trim(strdate),&
&      " : zatom,zion,pspdat"
 write (funit,'(3(1x,i2)," 0 ",i5," 0.",19x,a)') &
&      pshead%pspcod,&
&      pshead%pspxc_abinit,&
&      pshead%lmax,&
&      wavmeshsz,&
&      " : pspcod,pspxc,lmax,lloc,mmax,r2well"
 write (funit,'(1x,"paw4",1x,i4,29x,a)') &
&       pshead%creatorid,&
&       " : pspfmt,creatorID"
 write (funit,'(2(1x,i2),33x,a)') &
&       pshead%basis_size,&
&       pshead%lmn_size,&
&       " : basis_size,lmn_size"
 do ib=1,pshead%basis_size
  write (funit,'(1x,i1)',ADVANCE='NO') pshead%orbitals(ib)
 enddo
 if (pshead%basis_size<20) then
  do ib=pshead%basis_size+1,20
   write (funit,'(a)',ADVANCE='NO') '  '
  enddo
 endif
 write (funit,'(a)') ": orbitals"
 write (funit,'(1x,i1,37x,a)') &
&       nmesh," : number_of_meshes"
 do ii=1,nmesh
  if (meshtp(ii)==1) then
   write (funit,'(1x,i1,1x,i1,1x,i4,1x,es18.12,15x,a,i1,a)') &
&         ii,meshtp(ii),meshsz(ii),radstp(ii),&
&         " : mesh ",ii,", type,size,rad_step[,log_step]"
  else
   write (funit,'(1x,i1,1x,i1,1x,i4,1x,es16.10,1x,es16.10,a,i1,a)') &
&         ii,meshtp(ii),meshsz(ii),radstp(ii),logstp(ii),&
&         " : mesh ",ii,", type,size,rad_step[,log_step]"
  endif
 enddo
 write (funit,'(1x,f13.10,25x,a)') &
&       pshead%rc_sph,&
&      " : r_cut(PAW)"
 if (pshead%shape_type==1) then
  write (funit,'(1x,i1,1x," 0.",i2,1x,es20.12,10x,a)') &
&         pshead%shape_type,pshead%lambda,pshead%sigma,&
&        " : shape_type,rshape,lambda,sigma"
 else
  if (pshead%hat_meshsz==pshead%sph_meshsz) then
   write (funit,'(1x,i1," 0.",34x,a)') &
&         pshead%shape_type,&
&         " : shape_type,rshape"
  else
   write (funit,'(1x,i1,1x,f13.10,20x,a)') &
&         pshead%shape_type,pshead%rc_hat,&
&         " : shape_type,rshape"
  endif
 endif

!Write W-functions
 do ib=1,pshead%basis_size
  write(funit,'(a,i1,a)') "===== PHI ",ib,&
&      " =====   [phi(r)=PHI(r)/r*Ylm(th,ph)]"
  write(funit,'(i2,a)') iwavmesh,"  : radial mesh index"
  if (loggrd%uselog) then
   allocate(ffit(wavmeshsz))
   call splinefit(pshead%wav_meshsz,pshead%rad_step,pawps%phi(:,ib),&
&                 wavmeshsz,rr_log(1:wavmeshsz),ffit)
   write(funit,'(3(1x,es23.16))') ffit
   deallocate(ffit)
  else
   write(funit,'(3(1x,es23.16))') (pawps%phi(ir,ib),ir=1,wavmeshsz)
  endif
 enddo

!Write pseudo W-functions
 do ib=1,pshead%basis_size
  write(funit,'(a,i1,a)') "===== TPHI ",ib,&
&      " =====   [tphi(r)=TPHI(r)/r*Ylm(th,ph)]"
  write(funit,'(i2,a)') iwavmesh,"  : radial mesh index"
  if (loggrd%uselog) then
   allocate(ffit(wavmeshsz))
   call splinefit(pshead%wav_meshsz,pshead%rad_step,pawps%tphi(:,ib),&
&                 wavmeshsz,rr_log(1:wavmeshsz),ffit)
   write(funit,'(3(1x,es23.16))') ffit
   deallocate(ffit)
  else
   write(funit,'(3(1x,es23.16))') (pawps%tphi(ir,ib),ir=1,wavmeshsz)
  endif
 enddo

!Write projectors
 do ib=1,pshead%basis_size
  write(funit,'(a,i1,a)') "===== TPROJECTOR ",ib,&
&      " =====   [tp(r)=TPROJECTOR(r)/r*Ylm(th,ph)]"
  write(funit,'(i2,a)') iprjmesh,"  : radial mesh index"
  write(funit,'(3(1x,es23.16))') (pawps%tproj(ir,ib),ir=1,pshead%prj_meshsz)
 enddo

!Write core density
 write(funit,'(a)') "===== CORE_DENSITY ====="
 write(funit,'(i2,a)') icoremesh,"  : radial mesh index"
 allocate(ntmp(pshead%core_meshsz))
 ntmp(2:pshead%core_meshsz)=pawps%coreden4pr2(2:pshead%core_meshsz) &
&                          /(four_pi*pawrad%rad(2:pshead%core_meshsz)**2)
 call deducer0(ntmp,pawrad)
 if (loggrd%uselog) then
  allocate(ffit(coremeshsz))
  call splinefit(pshead%core_meshsz,pshead%rad_step,ntmp,coremeshsz,rr_log(1:coremeshsz),ffit)
  write(funit,'(3(1x,es23.16))') ffit
  deallocate(ffit)
 else
  write(funit,'(3(1x,es23.16))') (ntmp(ir),ir=1,coremeshsz)
 endif
 deallocate(ntmp)

!Write pseudo core density
 write(funit,'(a)') "===== PSEUDO_CORE_DENSITY ====="
 write(funit,'(i2,a)') icoremesh,"  : radial mesh index"
 allocate(ntmp(pshead%core_meshsz))
 ntmp(2:pshead%core_meshsz)=pawps%tcoreden4pr2(2:pshead%core_meshsz) &
&                          /(four_pi*pawrad%rad(2:pshead%core_meshsz)**2)
 call deducer0(ntmp,pawrad)
 if (loggrd%uselog) then
  allocate(ffit(coremeshsz))
  call splinefit(pshead%core_meshsz,pshead%rad_step,ntmp,coremeshsz,rr_log(1:coremeshsz),ffit)
  write(funit,'(3(1x,es23.16))') ffit
  deallocate(ffit)
 else
  write(funit,'(3(1x,es23.16))') (ntmp(ir),ir=1,coremeshsz)
 endif
 deallocate(ntmp)

!Write Dij0 and Rhoij0
 write(funit,'(a)') "===== Dij0 ====="
 ii=0
 do jlmn=1,pshead%lmn_size
  write(funit,'(100(1x,es23.16))') (pawps%dij0(ii+ilmn),ilmn=1,jlmn)
  ii=ii+jlmn
 enddo
 write(funit,'(a)') "===== Rhoij0 ====="
 ii=0
 do jlmn=1,pshead%lmn_size
  write(funit,'(100(1x,es23.16))') (pawps%rhoij0(ii+ilmn),ilmn=1,jlmn)
  ii=ii+jlmn
 enddo

!Write Vloc
 if (pshead%vlocopt==0) write(funit,'(a)') "===== Vbare (Vloc(r)) ====="
 if (pshead%vlocopt==1) write(funit,'(a)') "===== VHntZC (Vloc(r)) ====="
 write(funit,'(i2,1x,i2,a)') ivlocmesh,pshead%vlocopt,"  : radial mesh index, Vloc format (0=Vbare, 1=VH(tnzc))"
 if (loggrd%uselog) then
  allocate(ffit(vlocmeshsz))
  if (pshead%vlocopt==0) then
   call splinefit(pshead%sph_meshsz,pshead%rad_step,pawps%vbare,&
&                 vlocmeshsz,rr_log(1:vlocmeshsz),ffit)
  else
   call splinefit(pshead%vloc_meshsz,pshead%rad_step,pawps%vhtnzc,&
&                 vlocmeshsz,rr_log(1:vlocmeshsz),ffit)
  endif
  write(funit,'(3(1x,es23.16))') ffit
  deallocate(ffit)
 else
  if (pshead%vlocopt==0) then
   write(funit,'(3(1x,es23.16))') (pawps%vbare(ir),ir=1,vlocmeshsz)
  else
   write(funit,'(3(1x,es23.16))') (pawps%vhtnzc(ir),ir=1,vlocmeshsz)
  endif
 endif

!Write (eventually) shape functions
 if (pshead%shape_type==-1) then
  allocate(shpf(wavmeshsz))
  do il=1,pshead%l_size
   write(funit,'(a,i1,a)') "===== SHAPEF (l=",il-1,") ====="
   write(funit,'(a)') " 1  : radial mesh index"
   if (loggrd%uselog) then
    allocate(ffit(wavmeshsz))
    call splinefit(pshead%wav_meshsz,pshead%rad_step,pawarray%shapefunc(:),&
&                  wavmeshsz,rr_log(1:wavmeshsz),ffit)
    if (il==1) then
     shpf(1:wavmeshsz)=ffit(1:wavmeshsz)
    else
     do ir=1,wavmeshsz
      shpf(ir)=ffit(ir)*pawarray%shpnrm(il)*rr_log(ir)**(il-1)
     enddo
    endif
    deallocate(ffit)
   else
    if (il==1) then
     shpf(1:wavmeshsz)=pawarray%shapefunc(1:wavmeshsz)
    else
     do ir=1,wavmeshsz
      shpf(ir)=pawarray%shapefunc(ir)*pawarray%shpnrm(il)*pawrad%rad(ir)**(il-1)
     enddo
    endif
   endif
   write(funit,'(3(1x,es23.16))') (shpf(ir),ir=1,wavmeshsz)
  enddo
  deallocate(shpf)
 endif

!Write pseudo core density
 write(funit,'(a)') "===== PSEUDO_VALENCE_DENSITY ====="
 write(funit,'(i2,a)') ivalemesh,"  : radial mesh index"
 allocate(ntmp(pshead%vale_meshsz))
 ntmp(2:pshead%vale_meshsz)=pawps%tvaleden4pr2(2:pshead%vale_meshsz) &
&                          /(four_pi*pawrad%rad(2:pshead%vale_meshsz)**2)
 call deducer0(ntmp,pawrad)
 if (loggrd%uselog) then
  allocate(ffit(valemeshsz))
  call splinefit(pshead%vale_meshsz,pshead%rad_step,ntmp,valemeshsz,rr_log(1:valemeshsz),ffit)
  write(funit,'(3(1x,es23.16))') ffit
  deallocate(ffit)
 else
  write(funit,'(3(1x,es23.16))') (ntmp(ir),ir=1,valemeshsz)
 endif
 deallocate(ntmp)

!Close the file and end
 close(funit)
 if (loggrd%uselog) deallocate(rr_log)

 end subroutine
!!***
