!{\src2tex{textfont=tt}}
!!****f* ABINIT/rdpawps2
!! NAME
!! rdpawps2
!!
!! FUNCTION
!! Read the body of a PAW pseudopotential file generated by AtomPAW
!! (the header must have been read before)
!!
!! COPYRIGHT
!! Copyright (C) 1998-2007 ABINIT group (FJ, MT)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~ABINIT/Infos/copyright
!! or http://www.gnu.org/copyleft/gpl.txt .
!! For the initials of contributors, see ~ABINIT/Infos/contributors .
!!
!! INPUTS
!!  filename= input file name (atomicdata)
!!  funit= input unit number
!!  pshead
!!    %basis_size= Number of elements for the paw nl basis
!!    %core_meshsz= Dimension of radial mesh for core density
!!    %lmn_size= Number of elements for the paw basis
!!    %orbitals(basis_size)= Quantum number l for each basis function
!!    %shape_type= Shape function type
!!    %sph_meshsz= Dimension of radial mesh corresponding to PAW spheres
!!    %vale_meshsz= Dimension of radial mesh for pseudo valence
!!    %vloc_meshsz= Dimension of radial mesh for vloc=htnzc
!!    %vlocopt= Option for Vloc
!!    %wav_meshsz= Dimension of radial mesh for phi, tphi ...
!!  pawarray
!!    %indlmn(6,lmn_size)= Gives l,m,n,lm,ln,s for i=lmn
!!  pawrad= radial grid definitions
!!
!! OUTPUT
!!  pawarray
!!    %kij(lmn2_size)= Kinetic overlap operator
!!  pawps
!!    %coreden4pr2(core_meshsz)= Core density multiplied by 4Pi.r2
!!    %tcoreden4pr2(core_meshsz)= Pseudized core density multiplied by 4Pi.r2
!!    %tvaleden4pr2(vale_meshsz)= Pseudized valence density multiplied by 4Pi.r2
!!    %phi(wav_meshsz,basis_size)= PAW atomic wavefunctions on the radial grid
!!    %tphi(wav_meshsz,basis_size)= PAW atomic pseudo-wavefunctions on the radial grid
!!    %tproj(prj_msz_max,basis_size)= PAW projectors on the radial grid
!!    %vbare(sph_meshsz)= bare local potential (part of VH(tnzc))
!!
!! SIDE EFFECTS
!!  pshead
!!    %lambda= Lambda in gaussian type g(r)
!!    %sigma= Sigma for gaussian type g(r)
!!
!! PARENTS
!!      atompaw2abinit
!!
!! CHILDREN
!!
!! SOURCE

 subroutine rdpawps2(filename,pawarray,pawps,pawrad,pshead,funit)

 use defs_basis
 use defs_pawps

 implicit none

!Arguments ---------------------------------------------
 integer :: funit
 character*(fnlen) :: filename
 type(pawarray_type) :: pawarray
 type(pawrad_type)  :: pawrad
 type(pawps_type)    :: pawps
 type(pshead_type)   :: pshead

!Local variables ---------------------------------------
 integer :: ib,ilmn,ilm,iln,ir1,ir2,ir,jlmn,jlm,jln,j0lmn,klmn,nkij
 logical :: found,endfile,corefound,tcorefound,tvalefound,kijfound,shapeffound,vbarefound
 logical :: vionfound,usevbare
 integer,allocatable :: indkij(:,:)
 logical,allocatable :: phifound(:),tphifound(:),tprojfound(:)
 real(dp) :: sf1,sf2,r1,r2
 real(dp), allocatable :: kij(:),shapef(:),tmp(:)
 character*(fnlen) :: line,readline

!functions
 character*(20) :: stripcomment
 integer :: mkint

! *********************************************************************

! ==================================================
! ====== 1-READ  ATOMICDATA FILE FROM AtomPAW ======
! ==================================================

!Init some flags
 allocate(phifound(pshead%basis_size));phifound=.false.
 allocate(tphifound(pshead%basis_size));tphifound=.false.
 allocate(tprojfound(pshead%basis_size));tprojfound=.false.
 shapeffound=.true.;if (pshead%shape_type==1) shapeffound=.false.
 corefound=.false.;tcorefound=.false.;tvalefound=.false.
 vbarefound=.false.;vionfound=.false.
 kijfound=.false.
 usevbare=(pshead%vlocopt==0)
 
!Open the atomicdata file for reading
 open(unit=funit,file=filename,form='formatted',status='old')

!Start a reading loop
 endfile=.false.
 do while (.not.endfile)
   read(funit,'(a)',err=10,end=10) readline
   line=adjustl(readline);goto 20
10 line="";endfile=.true.
20 continue

!--Read WAVE FUNCTIONS PHI
  if (line(1:4)=='PHI ') then
   read(unit=line(5:fnlen),fmt=*) ib
   if (ib>pshead%basis_size) then
    write(6,'(a)') "Error in rdpawps2: PHI(ib) for ib>basis_sz !"
    close(funit);stop
   endif
   read(funit,*) (pawps%phi(ir,ib),ir=1,pshead%wav_meshsz)
   phifound(ib)=.true.
  endif

!--Read PSEUDO WAVE FUNCTIONS TPHI
  if (line(1:5)=='TPHI ') then
   read(unit=line(6:fnlen),fmt=*) ib
   if (ib>pshead%basis_size) then
    write(6,'(a)') "Error in rdpawps2: TPHI(ib) for ib>basis_sz !"
    close(funit);stop
   endif
   read(funit,*) (pawps%tphi(ir,ib),ir=1,pshead%wav_meshsz)
   tphifound(ib)=.true.
  endif

!--Read PROJECTORS TPROJ
  if (line(1:10)=='TPROJECTOR') then
   read(unit=line(11:fnlen),fmt=*) ib
   if (ib>pshead%basis_size) then
    write(6,'(a)') "Error in rdpawps2: TPROJ(ib) for ib>basis_sz !"
    close(funit);stop
   endif
   pawps%tproj(:,ib)=zero
   read(funit,*) (pawps%tproj(ir,ib),ir=1,pshead%wav_meshsz)
   tprojfound(ib)=.true.
  endif

!--Read core density CORE_DENSTY
  if (line(1:12)=='CORE_DENSITY') then
   read(funit,*) (pawps%coreden4pr2(ir),ir=1,pshead%wav_meshsz)
   corefound=.true.
  endif

!--Read pseudized core density CORETAIL_DENSITY
  if (line(1:16)=='CORETAIL_DENSITY') then
   read(funit,*) (pawps%tcoreden4pr2(ir),ir=1,pshead%core_meshsz)
   tcorefound=.true.
  endif

!--Read pseudized valence density PSEUDO_VALENCE_DENSITY
  if (line(1:22)=='PSEUDO_VALENCE_DENSITY') then
   if (pshead%vale_meshsz>0) then
    read(funit,*) (pawps%tvaleden4pr2(ir),ir=1,pshead%vale_meshsz)
   end if
   tvalefound=.true.
  endif

!--Read Vbare potential VLOCFUN     --no longer needed
  if (line(1:7)=='VLOCFUN') then
   if (pshead%vlocopt==0) then
    allocate(tmp(pshead%wav_meshsz))
    read(funit,*) (tmp(ir),ir=1,pshead%wav_meshsz)
    pawps%vbare(1:pshead%sph_meshsz)=tmp(1:pshead%sph_meshsz)*0.5d0
    deallocate(tmp)
   end if
   vbarefound=.true.
  endif

!--Read local ionic potential VLOCION -- ionic local potential with correct
!       unscreening of vxc   (vxc(tilde_core + tilde_n + hat_hat_n))
  if (line(1:7)=='VLOCION') then
   if (pshead%vlocopt==1) then
    read(funit,*) (pawps%vhtnzc(ir),ir=1,pshead%vloc_meshsz)
    pawps%vhtnzc=pawps%vhtnzc*0.5d0
   end if
   vionfound=.true.
  endif

!--Read Kinetic term KINETIC_ENERGY_MATRIX
  if (line(1:21)=='KINETIC_ENERGY_MATRIX') then
   allocate(indkij(pshead%basis_size,pshead%basis_size));indkij(:,:)=0;nkij=0
   do jln=1,pshead%basis_size
    do iln=jln,pshead%basis_size
     if (pshead%orbitals(iln)==pshead%orbitals(jln)) then
      nkij=nkij+1;indkij(iln,jln)=nkij;indkij(jln,iln)=nkij
     endif
    enddo
   enddo
   allocate(kij(nkij));read(funit,*) kij(:)
   pawarray%kij=zero
   do jlmn=1,pshead%lmn_size
    j0lmn=jlmn*(jlmn-1)/2
    jlm=pawarray%indlmn(4,jlmn);jln=pawarray%indlmn(5,jlmn)
    do ilmn=1,jlmn
     klmn=j0lmn+ilmn
     ilm=pawarray%indlmn(4,ilmn);iln=pawarray%indlmn(5,ilmn)
     if (ilm==jlm) pawarray%kij(klmn)=half*kij(indkij(iln,jln))
    enddo
   enddo
   deallocate(indkij,kij);kijfound=.true.
  endif

!End of reading loop
 enddo

!Close the atomicdata file
 close(funit)

!Test flags: is anything OK ?
 do ib=1,pshead%basis_size
  if (.not.phifound(ib)) &
&  write(6,'(a,i2,a)') "Error in rdpawps2: PHI(",ib,") not found !"
  if (.not.tphifound(ib)) &
&  write(6,'(a,i2,a)') "Error in rdpawps2: TPHI(",ib,") not found !"
  if (.not.tprojfound(ib)) &
&  write(6,'(a,i2,a)') "Error in rdpawps2: TPROJ(",ib,") not found !"
 enddo
 if (.not.corefound) &
&  write(6,'(a,i2,a)') "Error in rdpawps2: CORE DENSITY not found !"
 if (.not.tcorefound) &
&  write(6,'(a,i2,a)') "Error in rdpawps2: CORETAIL DENSITY not found !"
 if ((pshead%vale_meshsz>0).and.(.not.tvalefound)) &
&  write(6,'(a,i2,a)') "Error in rdpawps2: PSEUDO VALENCE DENSITY not found !"
 if (.not.kijfound) &
&  write(6,'(a,i2,a)') "Error in rdpawps2: KINETIC ENERGY MATRIX not found !"
! if (.not.shapeffound) &
!&  write(6,'(a,i2,a)') "Error in rdpawps2: SHAPE FUNC not found !"
 if (usevbare.and.(.not.vbarefound)) &
&  write(6,'(a,i2,a)') "Error in rdpawps2: VLOC FUNC not found !"
 if ((.not.usevbare).and.(.not.vionfound)) &
&  write(6,'(a,i2,a)') "Error in rdpawps2: VLOC ION not found !"
 found=(usevbare.and.vbarefound).or.((.not.usevbare).and.vionfound)
 found=found.and.corefound.and.tcorefound.and.tvalefound.and.kijfound
 do ib=1,pshead%basis_size
  found=found.and.phifound(ib).and.tphifound(ib).and.tprojfound(ib)
 enddo
 if (.not.found) stop
 deallocate(phifound,tphifound,tprojfound)

!===========================================================
!Copy pseudized core density to core density outside spheres
 if (pshead%core_meshsz>pshead%wav_meshsz) &
&  pawps%coreden4pr2(pshead%wav_meshsz+1:pshead%core_meshsz) = &
&  pawps%tcoreden4pr2(pshead%wav_meshsz+1:pshead%core_meshsz)

!===========================================================
!Test pseudo valence density
 if (pshead%vale_meshsz>0) then
  write(6,'(/,a,a,f5.2,a,a,g11.4)') 'Info:',&
&   '  At r=',pawrad%rad(pshead%vale_meshsz),' a.u.,',&
&   ' Pseudo_valence_density= ', pawps%tvaleden4pr2(pshead%vale_meshsz) &
&                               /pawrad%rad(pshead%vale_meshsz)**2/(4*pi)
  write(6,'(a)') '  This quantity must be as small as possible.'
endif

!===========================================================
!Test potential
 if (pshead%vlocopt==1) then
  write(6,'(/,a,a,f5.2,a,a,g11.4)') 'Info:',&
&   '  At r_vloc=',pawrad%rad(pshead%vloc_meshsz),' a.u.,',&
&   ' VHartree(ntild(Zv+Zc))= -Zv/r + ', pawps%vhtnzc(pshead%vloc_meshsz) &
&    +(pshead%atomic_charge-pshead%core_charge)/pawrad%rad(pshead%vloc_meshsz)
  write(6,'(a)') '  This quantity must be as small as possible.'
endif

 end subroutine
!!***
