!{\src2tex{textfont=tt}}
!!****f* ABINIT/rdpawps1
!! NAME
!! rdpawps1
!!
!! FUNCTION
!! Read the header of a PAW pseudopotential file generated by AtomPAW
!!
!! COPYRIGHT
!! Copyright (C) 1998-2007 ABINIT group (FJ, MT)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~ABINIT/Infos/copyright
!! or http://www.gnu.org/copyleft/gpl.txt .
!! For the initials of contributors, see ~ABINIT/Infos/contributors .
!!
!! INPUTS
!!  filename(3)= input file names (atomicdata, density, potential)
!!  unit(3)= input unit numbers
!!
!! OUTPUT
!!  pshead
!!    %atomic_charge= Total atomic charge
!!    %basis_size= Number of elements for the paw nl basis
!!    %core_charge= Core charge
!!    %core_meshsz= Dimension of radial mesh for core density
!!    %lambda= Lambda in gaussian type g(r)
!!    %lmn_size= Number of elements for the paw basis
!!    %mesh_type= type of radial grid (regular or log)
!!    %orbitals(basis_size)= Quantum number l for each basis function
!!    %pspxc_atom= AtomPaw s code number for the exchange-correlation
!!    %rad_step= Step corresponding to radial mesh
!!    %rc_hat= radius for shape function
!!    %rc_sph= Default PAW sphere radius
!!    %shape_type= Shape function type
!!    %sigma= Sigma for gaussian type g(r)
!!    %title= Title for pseudopotential
!!    %vale_meshsz= Dimension of radial mesh for pseudo valence density
!!    %vloc_meshsz= Dimension of radial mesh for vloc=vhtnzc
!!    %wav_meshsz= Dimension of radial mesh for phi, tphi ...
!!  version= version number for AtomPAW2Abinit
!!
!! PARENTS
!!      atompaw2abinit
!!
!! CHILDREN
!!
!! SOURCE

 subroutine rdpawps1(filename,pshead,funit,version)

 use defs_basis
 use defs_pawps

 implicit none

!Arguments ---------------------------------------------
 integer :: funit
 character*(fnlen) :: filename
 character*5 :: version
 type(pshead_type) :: pshead

!Local variables ---------------------------------------
 integer :: ib,ii
 real(dp) :: rstep,lstep
 logical :: endfile
 character*(fnlen) :: line,readline,strg,word

! *************************************************************************

!Inits
 pshead%mesh_type=1
 pshead%rc_sph=-1.d0
 pshead%rc_hat=-1.d0
 pshead%basis_size=0
 pshead%core_meshsz=0
 pshead%vale_meshsz=0
 pshead%vloc_meshsz=0

!Open the atomicdata file for reading
 open(unit=funit,file=filename,form='formatted',status='old')

!Start a reading loop
 endfile=.false.
 do while (.not.endfile)
   read(funit,'(a)',err=10,end=10) readline
   line=adjustl(readline);goto 20
10 line="";endfile=.true.
20 continue

!--Read TITLE
  if (line(1:8)=='ATOMTYPE') then
   read(unit=line(9:fnlen),fmt=*) strg
   write(unit=pshead%title,fmt='(4a)') &
&   "Paw atomic data for element ",trim(ADJUSTL(strg)),&
&   " - Generated by AtomPAW + AtomPAW2Abinit v",version
  endif

!--Read BASIS SIZE
  if (line(1:10)=='BASIS_SIZE') then
   read(unit=line(11:fnlen),fmt=*) pshead%basis_size
   if (pshead%basis_size>20) then
    write(6,'(a)') "Error in rdpawps1: basis size too large (>20) !"
    close(funit);stop
   endif
  endif

!--Read ORBITALS
  if (line(1:8)=='ORBITALS') then
   read(funit,'(a)',err=11,end=11) readline
   line=adjustl(readline);goto 21
11 write(6,'(a)') "Error in rdpawps1: error reading orbitals !"
   close(funit);stop
21 read(unit=line(1:fnlen),fmt=*) &
&      (pshead%orbitals(ib),ib=1,pshead%basis_size)
  endif

!--Read INITOCC  -- added by NAWH 1/23/06
  if (line(1:7)=='INITOCC') then
   read(funit,*) (pshead%occ(ib),ib=1,pshead%basis_size)
   read(funit,'(a)') readline     ! read 'END'
  endif

!--Read RC
  if (line(1:2)=='RC') then
   read(unit=line(3:fnlen),fmt=*) pshead%rc_sph
  endif

!--Read MESH SIZE
  if (line(1:9)=='MESH_SIZE') then
   read(unit=line(10:fnlen),fmt=*) pshead%wav_meshsz
  endif

!--Read CORETAIL_POINTS
  if (line(1:15)=='CORETAIL_POINTS') then
   read(unit=line(16:fnlen),fmt=*) pshead%core_meshsz
  endif

!--Read # OF POINTS FOR PSEUDO_VALENCE_DENSITY
  if (line(1:22)=='PSEUDO_VALENCE_DENSITY') then
   read(unit=line(23:fnlen),fmt=*) pshead%vale_meshsz
  endif

!--Read MESH STEP
  if (line(1:9)=='MESH_STEP') &
&    read(unit=line(10:fnlen),fmt=*) rstep

!--Read LOG_GRID
  if (line(1:9)=='LOG_GRID') then
   read(unit=line(10:fnlen),fmt=*) lstep
   pshead%mesh_type=2
  end if

!--Read # OF POINTS FOR VLOCION
  if (line(1:7)=='VLOCION') then
   read(unit=line(8:fnlen),fmt=*) pshead%vloc_meshsz
  endif

!--Read ATOMIC CHARGE
  if (line(1:13)=='ATOMIC_CHARGE') &
&    read(unit=line(14:fnlen),fmt=*) pshead%atomic_charge

!--Read CORE CHARGE
  if (line(1:11)=='CORE_CHARGE') &
&    read(unit=line(12:fnlen),fmt=*) pshead%core_charge

!--Read EXCHANGE-CORRELATION TYPE
  if (line(1:10)=='ATOMXCTYPE') then
   read(unit=line(11:fnlen),fmt=*) strg
   if (trim(ADJUSTL(strg))=="LDA-PW") then
    pshead%pspxc_abinit=7
   elseif (trim(ADJUSTL(strg))=="GGA-PBE") then
    pshead%pspxc_abinit=11
   else
    write(6,'(a)') "Error in rdpawps1: unknown XC type !"
    close(funit);stop
   endif
  endif

!--Read SHAPE TYPE
  if (line(1:10)=='SHAPE_TYPE') then
   strg=trim(adjustl(line(11:fnlen)))
   if (strg(1:8)=="gaussian") then
    pshead%shape_type=1;pshead%lambda=2
    word=strg(9:fnlen)
    if (word/="") then
      read(unit=word,fmt=*) pshead%sigma
    else
      write(6,*) 'warning in rdpawps1:  cannot read Gauss sigma -- default used'
      pshead%sigma=pshead%rc_sph/sqrt(log(1.d4))
    endif
   elseif (strg(1:6)=="bessel") then
    pshead%shape_type=3;pshead%lambda=0;pshead%sigma=0.d0
    word=strg(7:fnlen)
    if (word/="") then
      read(unit=word,fmt=*) pshead%rc_hat
    else
      pshead%rc_hat=pshead%rc_sph
    endif
   elseif (strg(1:5)=="sinc2") then
    pshead%shape_type=2;pshead%lambda=0;pshead%sigma=0.d0
    word=strg(6:fnlen)
    if (word/="") then
      read(unit=word,fmt=*) pshead%rc_hat
    else
      pshead%rc_hat=pshead%rc_sph
    endif
   else
    write(6,'(a)') "Error in rdpawps1: unknown shape type !"
    close(funit);stop
   endif
  endif

!End of reading loop
 enddo

!Close the atomicdata file
 close(funit)

!Tests of consistency
 if (pshead%core_meshsz<pshead%wav_meshsz) then
  write(6,'(a)') "Error in rdpawps1:"
  write(6,'(a)') "   Mesh size for tcore density (CORETAIL_POINTS)"
  write(6,'(a)') "   must be greater or equal than MESH_SIZE !"
  stop
 endif
 if (pshead%mesh_type==1) then
  pshead%rad_step=rstep
  pshead%log_step=zero
  if (pshead%rc_sph>pshead%rad_step*dble(pshead%wav_meshsz-1)+tol8) then
   write(6,'(a)') "Error in rdpawps1:"
   write(6,'(a)') "   Radius for PAW spheres (RC)"
   write(6,'(a)') "   must be less (or equal) than R(MESH_SIZE) !"
   stop
  endif
 else
  pshead%rad_step=lstep
  pshead%log_step=rstep
  if (pshead%rc_sph>pshead%rad_step*(exp(pshead%log_step*dble(pshead%wav_meshsz-1))-1.d0)+tol8) then
   write(6,'(a)') "Error in rdpawps1:"
   write(6,'(a)') "   Radius for PAW spheres (RC)"
   write(6,'(a)') "   must be less (or equal) than  R(MESH_SIZE) !"
   stop
  endif
 endif

!Here must compute lmn_size
 pshead%lmn_size=0
 do ib=1,pshead%basis_size
  pshead%lmn_size=pshead%lmn_size+2*pshead%orbitals(ib)+1
 enddo

 end subroutine
!!***
