!{\src2tex{textfont=tt}}
!!****f* ABINIT/defs_pawps
!! NAME
!! defs_pawps
!!
!! FUNCTION
!! This module contains the following structured data types:
!!   pawps_type   : Specific paw variables
!!   pshead_type  : Abinit psp file header variables
!!   pawrad_type  : Grid definitions
!!   pawarray_type: Various arrays used in paw atomic data generation
!!
!! COPYRIGHT
!! Copyright (C) 1998-2007 ABINIT group (FJ, MT)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~ABINIT/Infos/copyright
!! or http://www.gnu.org/copyleft/gpl.txt .
!! For the initials of contributors, see ~ABINIT/Infos/contributors .
!!
!! NOTES
!!
!! The paw wavefunctions tphi and phi, and the projectors tprojector radial
!! parts are defined as: f(r_vector)=f(r)/r* Ylm(r_vector)
!!
!!
!! SOURCE

 module defs_pawps

 use defs_basis

 implicit none


!Structures
!**********************************************************************

 type pawps_type

! ===== Real (double precision) arrays

  real(dp), pointer :: coreden4pr2(:)
   ! coreden4pr2(core_meshsz)
   ! Gives the core density multiplied by 4Pi.r2

  real(dp), pointer :: tcoreden4pr2(:)
   ! tcoreden4pr2(core_meshsz)
   ! Gives the pseudized core density multiplied by 4Pi.r2

  real(dp), pointer :: tvaleden4pr2(:)
   ! tvaleden4pr2(core_meshsz)
   ! Gives the pseudized core density multiplied by 4Pi.r2 (up to r(vale_meshsz))
  
  real(dp), pointer :: phi(:,:)
   ! phi(sph_meshsz,basis_size)
   ! Gives, on the radial grid, the PAW atomic wavefunctions

  real(dp), pointer :: tphi(:,:)
   ! tphi(sph_meshsz,basis_size)
   ! Gives, on the radial grid, the PAW atomic pseudowavefunctions

  real(dp), pointer :: tproj(:,:)
   ! tproj(prj_msz_max,basis_size)
   ! Gives, on the radial grid, the PAW projectors for each basis function

  real(dp), pointer :: dij0(:)
   ! dij0(lmn2_size)
   ! Part of the Dij term (non-local operator) completely
   !  calculated in the pseudopotential part

  real(dp), pointer :: rhoij0(:)
   ! rhoij0(lmn2_size)
   ! Atomic initialization of rhoij

  real(dp), pointer :: vbare(:)
   ! vbare(sph_meshsz)
   ! Gives the "bare" local potential

  real(dp), pointer :: vhtnzc(:)
   ! vhtnzc(core_meshsz)
   ! Gives the Hartree potential of the pseudo density
   !  of the nucleus + core electrons of the atom

 end type pawps_type

! =====================================================================

 type pshead_type

! ===== Integer scalars

  integer :: basis_size
   ! Number of elements for the paw nl basis

  integer :: core_meshsz
   ! Dimension of radial mesh for core density
   ! Must be greater or equal than sph_meshsz

  integer :: creatorid
   ! ID of psp generator (here creatorID=1 !)

  integer :: hat_meshsz
   ! Dimension of radial mesh for hat density

  integer :: l_size
   ! Maximum value of l+1 leading to a non zero Gaunt coefficient

  integer :: lambda
   ! Lambda in gaussian type g(r)

  integer :: lmax
   ! Maximum value of l

  integer :: lmn_size
   ! Number of elements for the paw basis

  integer :: lmn2_size
   ! lmn2_size=lmn_size*(lmn_size+1)/2

  integer :: max_meshsz
   ! max. dimension for radial meshes

  integer :: mesh_type
   ! Flag: 1 if all meshes are regular grids
   !       2 if all meshes are log. grids

  integer :: prj_meshsz
   ! Dimension of radial mesh for tproj

  integer :: prj_msz_max
   ! Maximal dimension of radial mesh for tproj

  integer :: pspcod
   ! Psp code number for Abinit (here PAW->pspcod=7 !)

  integer :: pspxc_abinit
   ! Abinit s code number for the exchange-correlation functionnal

  integer :: shape_type
   ! Shape function type
   ! shape_type=1 ; g(r)=sin
   ! shape_type=2 ; g(r)=exp
   ! shape_type=2 ; g(r)=bessel
   ! shape_type=-1; g(r)=tabulated

  integer :: sph_meshsz
   ! Dimension of radial mesh corresponding to PAW radius

  integer :: vale_meshsz
   ! Dimension of radial mesh for pseudo valence density
  
  integer :: vloc_meshsz
   ! Dimension of radial mesh for vloc=vhtnzc
   ! For r>r(vloc_meshsz), we assume vloz=-Z/r

  integer :: vlocopt
   ! Option for the format of Vloc
   !  0: Vloc is Vbare (Blochl s formulation)
   !  1: Vloc is VH(tnzc) (Kresse s formulation)

  integer :: wav_meshsz
   ! Dimension of radial mesh for wave functions (phi, tphi)

! ===== Integer arrays

   integer, pointer :: orbitals(:)
   ! orbitals(basis_size)
   ! Gives the quantum number l for each basis function

! ===== Real (double precision) scalars

  real(dp) :: atomic_charge
   ! Total atomic charge

  real(dp) :: core_charge
   ! Core charge

  real(dp) :: log_step
   ! Step corresponding to exponential term in logarithimc radial mesh

  real(dp) :: rad_step
   ! Step corresponding to radial mesh

  real(dp) :: rc_hat
   ! Radius used in sinus shape function

  real(dp) :: rc_sph
   ! Default PAW sphere radius

  real(dp) :: sigma
   ! Sigma for gaussian type g(r)

! ===== Real (double precision) arrays

   real(dp), pointer :: occ(:)
   ! occ(basis_size)
   ! Gives the occupation for each basis function

! ===== Strings

   character*(fnlen) exctype_atompaw
   ! AtomPaw s expression of exchange-correlation functionnal

   character*(fnlen) :: title
   ! Title for pseudopotential

   character*(fnlen), pointer :: stateid(:)
   ! orbid(basis_size)
   ! Gives the "state id" for each basis function (from XML file)


 end type pshead_type

! =====================================================================

 type pawrad_type

! ===== Integer scalars

  integer :: islog
   ! 0 if the radial grid is a linear grid
   ! 1 if the radial grid is a logarithmic grid

  integer :: meshsz
   ! Dimension of max. radial mesh (max. of all mesh sizes)

! ===== Real (double precision) scalars

  real(dp) :: rstep
   ! Step corresponding to radial mesh

  real(dp) :: lstep
   ! Step corresponding to exponential term (logarithmic mesh)

! ===== Real (double precision) arrays

  real(dp), pointer :: rad(:)
   ! rad(max_meshsz)
   ! Coordinates of all the points of the radial grid

  real(dp), pointer :: radfact(:)
   ! radfact(max_meshsz)
   ! radfact(:)= factor used to compute radial integrals on generalized grid

 end type pawrad_type

! =====================================================================

 type pawarray_type

! ===== Integer arrays

   integer, pointer :: indlmn(:,:)
   ! indlmn(6,lmn_size)
   ! Gives l,m,n,lm,ln,s for i=lmn

! ===== Real (double precision) arrays

  real(dp), pointer :: hatden4pr2(:)
   ! hatden4pr2(sph_meshsz)
   ! Gives the compensation density multiplied by 4Pi.r2 (following Abinit s definition)

  real(dp), pointer :: kij(:)
   ! kij(lmn2_size)
   ! Kinetic overlap operator

  real(dp), pointer :: shapefunc(:)
   ! shapefunc(sph_meshsz)
   ! Gives the normalized shape function of the atom used for
   ! the compensation density

  real(dp), pointer :: shpnrm(:)
   ! shpnrm(l_size)
   ! Gives the moment of shapefunction for each l

  real(dp), pointer :: tvaleden4pr2(:)
   ! tvaleden4pr2(sph_meshsz)
   ! Gives the pseudized valence density multiplied by 4Pi.r2 (only part inside spheres)

 end type pawarray_type

! =====================================================================

 type pawrso_type

! ===== Logical

 logical :: userso
  ! TRUE if Real Space Optimization is required

! ===== Real (double precision)

 real(dp) :: ecut
  ! Real Space Optimization parameter: plane wave cutoff = 1/2 Gmax**2

 real(dp) :: gfact
  ! Real Space Optimization parameter: Gamma/Gmax ratio

 real(dp) :: werror
  ! Real Space Optimization parameter: max. error W_l allowed

 end type pawrso_type

! =====================================================================

 type loggrd_type

! ===== Integer

 integer :: meshsz
  ! Mesh size for the logarithmic grid

! ===== Logical

 logical :: uselog
  ! TRUE if data are transfered on a log. grid before being written

! ===== Real (double precision)

 real(dp) :: log_step
  ! Logarithmic step for the logarithmic grid

 real(dp) :: rad_step
  ! Radial step for the logarithmic grid

 end type loggrd_type

!**********************************************************************

 end module defs_pawps
!!***
